#-----------------------------------------------------
#
# Copyright (c) 2014 by cisco Systems, Inc.
# All rights reserved.
#-----------------------------------------------------
__author__ = 'havishwa'

import logging
import os

from ..utils.utils import Utils
from ..api.systeminfo import SystemInfo
from ..api.diagnostic import PlatformDiagnostic
from ..api.systemhealthinfo import SystemHealthInfo
from ..cartridge.cartridge import Cartridge, CartridgeManager
from ..utils.infraexceptions import *
from ..utils.commandwrappers import *
from ..utils.docker_command_utils import docker_command_dict
import psutil
log = logging.getLogger("runtime.hosting")

class PlatformCapabilities(object):
    __singleton = None # the one, true Singleton
    MGMT_API_VERSION = "2.0"
    MIN_APP_MANIFEST_VERSION = "1.0"
    DISK_STATUS_NA = -1

    def __new__(cls, *args, **kwargs):
        # Check to see if a __singleton exists already for this class
        # Compare class types instead of just looking for None so
        # that subclasses will create their own __singleton objects
        if cls != type(cls.__singleton):
        #if not cls.__singleton:
            cls.__singleton = super(PlatformCapabilities, cls).__new__(cls, *args, **kwargs)
        return cls.__singleton

    def __init__(self):
        '''
        Donot sub-class this class. If you do, because of singleton code above, init
        would be called multiple times
        '''
        self._total_cpu_units = 100
        self._available_cpu_units = 0
        self._total_memory_mb = 0
        self._available_memory_mb = 0
        self._total_storage_mb = 0
        self._available_storage_mb = 0
        self._total_logvol_mb = 0
        self._available_logvol_mb = 0

        self._platform_product_id = None
        self._total_vcpu_count = psutil.cpu_count()
        self._cpuset = "0"
        self._apphosting_cpu_shares = 1024
        self._aufs_supported = False
        self._overlayfs_supported = False
        self._ipv6_supported = False

        self._supported_host_mount_paths = []
        self._supported_docker_volume_types = []
        self._supported_cartridge_handlers = []
        self._supported_app_types = []
        self._remote_docker_supported = False
        self._iox_release = "1.0"
        self._supported_platform_features = []
        self._supported_profile_types = []
        self._third_party_apps = True

        self._supported_network_modes = []
        self._supported_app_schema_list = []

        self._compute_node_name = SystemInfo.get_hostname()
        self._compute_node_id = SystemInfo.get_systemid()
        self._supported_app_validation = None
        self._persist_dir = None
        self._max_vcpu_per_app = 1
        self._persistent_partition = None
        self._supported_compression_types = ["gz"]
        self._supported_rootfs_types = ["img", "ext2", "iso"]
        self._black_listed_docker_run_options = []
        self._supported_docker_runtime_options = []
        self._supported_merged_docker_runtime_options = []
        self._load_platform_info()
        self._repo_dir = Utils.getSystemConfigValue("controller", "repo")
        self._repo_partition = self.get_app_disk_space_repo(self._repo_dir)

        datastore_section = Utils.getSystemConfigSection("datastore")
        if datastore_section is not None and datastore_section.get("enabled", "no") == "yes":
            self._datastore_supported = True
        else:
            self._datastore_supported = False

        visualization_section = Utils.getSystemConfigSection("visualization")
        if visualization_section is not None and visualization_section.get("enabled", "no") == "yes":
            self._visualization_supported = True
        else:
            self._visualization_supported = False

        self._privileged_mode_supported = SystemInfo.is_app_privileged_mode_supported()



    def _load_platform_info(self):
        platform_capability_file = Utils.getPlatformCapabilities()
        self._platform_product_id= Utils.getPlatformProductID()
        log.debug("Reading platform capabilities file: %s" % platform_capability_file)
        log.debug("Current platform product ID : %s" % self._platform_product_id)
        pc = Utils.read_yaml_file(platform_capability_file)
        current_platform_limits_file = None
        if pc:
            current_platform_limits_file = Utils.get_platform_limit_filename(self._platform_product_id)

        if current_platform_limits_file:
            cpl = os.path.join(Utils.getCapabilitiesFolder(), current_platform_limits_file)
            log.debug("Will read platform limits file : %s" % cpl)
        else:
            log.error("No matching limits file found for product id : %s" % self._platform_product_id)
            cpl = os.path.join(Utils.getCapabilitiesFolder(), "default_capabilities.yaml")
            log.info("Will read default platform limits file : %s" % cpl)
    
        if os.path.isfile(cpl):
            platform_limits = Utils.read_yaml_file(cpl)
            try:
                self._total_cpu_units = int(platform_limits["global"]["cpu_units"])
                self._available_cpu_units = self._total_cpu_units

                self._total_memory_mb = int(platform_limits["global"]["memory"])
                self._available_memory_mb = self._total_memory_mb

                self._total_storage_mb = int(platform_limits["global"]["disk"])
                self._available_storage_mb = self._total_storage_mb

                if "logical_volume" in platform_limits["global"]:
                    self._total_logvol_mb = int(platform_limits["global"]["logical_volume"])
                    self._available_logvol_mb = self._total_logvol_mb

                if "vcpu" in platform_limits["global"]:
                    self._total_vcpu_count = int(platform_limits["global"]["vcpu"])

                if "cpuset" in platform_limits["global"]:
                    self._cpuset = int(platform_limits["global"]["cpuset"])

                if "max_vcpu_per_app" in platform_limits["global"]:
                    self._max_vcpu_per_app = int(platform_limits["global"]["max_vcpu_per_app"])
                else:
                    self._max_vcpu_per_app = self._total_vcpu_count

                self._supported_app_types = platform_limits["global"]["supported_app_types"]
                if "supported_app_pkg_security" in platform_limits["global"]:
                    self._supported_app_validation = platform_limits["global"]["supported_app_pkg_security"]

                if "third_party_apps" in platform_limits["global"]:
                    self._third_party_apps = platform_limits["global"]["third_party_apps"]

                if "iox_release" in platform_limits["global"]:
                    self._iox_release = platform_limits["global"]["iox_release"]

                if "supported_union_fs_types" in platform_limits["global"]:
                    self._supported_union_fs = platform_limits["global"]["supported_union_fs_types"]
                else:
                    self._supported_union_fs = None

                if self._supported_union_fs and "aufs" in self._supported_union_fs:
                    self._aufs_supported = True

                if self._supported_union_fs and "overlay" in self._supported_union_fs:
                    self._overlayfs_supported = True
                self._ipv6_supported = platform_limits["global"].get("ipv6_supported", False)
                self._remote_docker_supported = platform_limits["global"].get("remote_docker_supported", False)
                if "supported_cartridge_handlers" in platform_limits["global"]:
                    self._supported_cartridge_handlers = platform_limits["global"]["supported_cartridge_handlers"]
                else:
                    self._supported_cartridge_handlers = ["mountable"]

                if "supported_profile_types" in platform_limits["global"]:
                    self._supported_profile_types = platform_limits["global"]["supported_profile_types"]
                else:
                    self._supported_profile_types = "default"

                if "cpu_shares" in platform_limits["global"]:
                    self._apphosting_cpu_shares = platform_limits["global"]["cpu_shares"]
                else:
                    self._apphosting_cpu_shares = 1024

                self._supported_app_schema_list = platform_limits["global"]["supported_app_schemas"]

                if "supported_network_modes" in platform_limits["global"]:
                    self._supported_network_modes = platform_limits["global"]["supported_network_modes"]

                if "supported_host_mount_paths" in platform_limits["global"]:
                    self._supported_host_mount_paths = platform_limits["global"]["supported_host_mount_paths"]
                    for path in self._supported_host_mount_paths:
                        if not os.path.exists(path):
                            os.makedirs(path)
                if "supported_docker_volume_types" in platform_limits["global"]:
                    self._supported_docker_volume_types = platform_limits["global"]["supported_docker_volume_types"]

                if "supported_compression_types" in platform_limits["global"]:
                    self._supported_compression_types = platform_limits["global"]["supported_compression_types"]

                if "supported_rootfs_types" in platform_limits["global"]:
                    self._supported_rootfs_types = platform_limits["global"]["supported_rootfs_types"]
                if "black_listed_docker_run_options" in platform_limits["global"]:
                    self._black_listed_docker_run_options = platform_limits["global"]["black_listed_docker_run_options"]
                self._supported_docker_runtime_options = docker_command_dict.keys()

                log.debug("Platform product id : %s" % self._platform_product_id)
                log.debug("Total CPU units: %s" % str(self._total_cpu_units))
                log.debug("Total Memory in MB : %s" % str(self._total_memory_mb))
                log.debug("Total storage in MB : %s" % str(self._total_storage_mb))
                log.debug("Total logical volume in MB : %s" % str(self._total_logvol_mb))
                log.debug("Total number of availabe VCPUs : %s" % str(self._total_vcpu_count))
                log.debug("Max vcpu's that can be specified per App : %s" % str(self._max_vcpu_per_app))
                log.debug("Supported app types : %s" % str(self._supported_app_types))
                log.debug("Third party apps supported : %s" % str(self._third_party_apps))
                log.debug("Supported application package validation: %s" % str(self._supported_app_validation))
                log.debug("Aufs Supported : %s" % str(self._aufs_supported))
                log.debug("Supported cartridge handlers: %s" % str(self._supported_cartridge_handlers))
                log.debug("Supported profile types: %s" % str(self._supported_profile_types))
                log.debug("App hosting CPU Shares: %s" % str(self._apphosting_cpu_shares))
                log.debug("Aufs Supported : %s" % str(self._aufs_supported))
                log.debug("IPv6 Supported : %s" % str(self._ipv6_supported))
                log.debug("Remote docker engine access supported: %s" % str(self._remote_docker_supported))
                log.debug("Host mount paths Supported : %s" % str(self._supported_host_mount_paths))
                log.debug("Docker volume types supported: %s" % str(self._supported_docker_volume_types))
                log.debug("Compression types Supported : %s" % str(self._supported_compression_types))
                log.debug("Rootfs types Supported: %s " % str(self._supported_rootfs_types))

            except Exception as ex:
                log.exception("Unable to retrieve platform capability info from %s" % cpl)
        else:
            log.error("Platform limits file %s not found! Capabilities info will be incorrect!" % cpl)

    def check_runtime_resource_availability(self, app_cpu_units, app_memory):
        log.debug("check runtime resources, requested by App, cpu: %s, memory: %s", app_cpu_units, app_memory)
        if self._available_cpu_units < app_cpu_units:
            raise ResourceLimitError("Platform does not have enough cpu resource, available cpu %s" % str(self._available_cpu_units))
        if self._available_memory_mb < app_memory:
            raise ResourceLimitError("Platform does not have enough memory , available memory %s" % str(self._available_memory_mb))

        return True

    def set_persistent_dir(self, persist_dir):
        self._persist_dir = persist_dir
        self._persistent_partition = self.get_app_disk_space_repo(self._persist_dir)

    def check_disk_status(self):
        disk_status_script = Utils.getSystemConfigValue("platform", "disk_status_script", default="")
        if not disk_status_script:
            log.debug("Disk status script not configured")
            return self.DISK_STATUS_NA

        if disk_status_script.startswith('/'):
            # Try absolute path first
            self.disk_status_script_path = disk_status_script
            if not os.path.exists(self.disk_status_script_path):
                log.debug("Specified disk_status script %s does not exist. Will look in scripts dir.", self.disk_status_script_path)
                # Try relative to scripts dir
                self.disk_status_script_path = Utils.getScriptsFolder() + disk_status_script
                if not os.path.exists(self.disk_status_script_path):
                    log.debug("Disk status script %s does not exist.", self.disk_status_script_path)
                    return self.DISK_STATUS_NA
        else:
            # Try relative to scripts dir
            self.disk_status_script_path = os.path.join(Utils.getScriptsFolder(), disk_status_script)
            if not os.path.exists(self.disk_status_script_path):
                log.debug("Disk status script %s does not exist", self.disk_status_script_path)
                return self.DISK_STATUS_NA

        log.info("Calling disk status script: %s" % self.disk_status_script_path) 
        cmd = [self.disk_status_script_path]
        output, rcode = call_script(cmd)
        log.info("Executed %s: returncode: %s, message: %s", self.disk_status_script_path, rcode, output)
        return rcode

        

        
    def check_disk_availability(self, app_disk):
        log.debug("check disk availability, disk requested by app: %s", app_disk)
        if self._available_storage_mb < app_disk:
            raise ResourceLimitError("platform does not have enough storage, available storage %s" % str(self._available_storage_mb))
        if self._persist_dir:
            free_avl_space = Utils.get_free_disk_space(self._persist_dir)
            log.debug("Required: %s. Actual Disk space available: %s" % (app_disk, str(free_avl_space)))
            if free_avl_space < app_disk:
                raise ResourceLimitError("Required: %s. Platform does not have enough storage, available storage %s" % (app_disk, str(free_avl_space)))
        return True

    def check_volgrp_size(self):
        return vgs()

    def check_logvol_availability(self, app_lv):
        avail_vg = self.check_volgrp_size()
        log.debug("logical vol available: %dMB; required by app: %dMB", avail_vg, app_lv)
        if avail_vg <= app_lv:
            log.error("Platform does not have enough logical volume space, available: %dMB" % avail_vg)
            return False
        return True

    def reduce_available_runtime_resources(self, app_cpu, app_mem):
        log.debug("Before reducing runtime resources cpu %s mem %s", str(self._available_cpu_units), str(self._available_memory_mb))
        log.debug("Reducing available resources. Requested : CPU=%s shares, memory=%s MB" % (str(app_cpu),
                                                                                                         str(app_mem)))


        self._available_cpu_units -= int(app_cpu)
        self._available_memory_mb -= int(app_mem)

        log.debug(
            "Available resources : CPU=%s shares, memory=%s MB, disk=%s MB" % (str(self._available_cpu_units),
                                                                               str(self._available_memory_mb),
                                                                               str(self._available_storage_mb)))
    def reduce_available_disk_resource(self, app_disk):
        log.debug("Available disk %s ", str(self._available_storage_mb))
        log.debug("Reducing App disk resource %s MB", app_disk)
        self._available_storage_mb -= int(app_disk)
        log.debug("Available disk %s ", str(self._available_storage_mb))

    def reduce_available_serial_resources(self, dev_id):
        if dev_id in self._serial_ports:
            self.available_serial_ports.remove(dev_id)

    def increase_available_serial_resources(self, dev_id):
        if dev_id in self._serial_ports:
            self.available_serial_ports.append(dev_id)

    def increase_available_runtime_resources(self, app_cpu, app_mem):
        log.debug("Increasing available resources. Will add : CPU=%s shares, memory=%s MB" % (str(app_cpu),
                                                                                                          str(app_mem)))

        self._available_cpu_units += int(app_cpu)
        self._available_memory_mb += int(app_mem)

        log.debug(
            "Available resources : CPU=%s shares, memory=%s MB, disk=%s MB" % (str(self._available_cpu_units),
                                                                               str(self._available_memory_mb),
                                                                               str(self._available_storage_mb)))

    def get_mount_device_disk_name(self, partitiondisk):
        # return the device name for a particular data partition
        for list in psutil.disk_partitions():
            if list.mountpoint == partitiondisk:
                return list.device

    def get_app_disk_space_repo(self, datalocation):
        # return the mounted disk partition from the given data location
        repo = datalocation
        count = 1
        leng = len(repo)
        for i in range(1, leng + 1):
            if (repo[leng - i] == '/'):
                repo = repo[:-count]
                if i == leng:
                    repo = "/"
                if Utils.ismount_exists(repo):
                    break
                count = 1
            else:
                count = count + 1
        return repo

    def get_apps_disk_usage(self):
        log.debug("Getting the apps disk space usage!")
        repo = self._repo_dir
        apps_disk_usage = []
        total_size = 0
        from appfw.runtime.hostingmgmt import HostingManager
        hm = HostingManager.get_instance()
        app_mgmt = hm.get_service("app-management")
        if app_mgmt:
            for conn_info in app_mgmt.connectorInfoMap.values():
                data = {}
                data["id"] = conn_info.app_id
                repo_path = conn_info.app_repo_path
                data["application_payload_size"] = int(round(float(Utils.get_dir_size(repo_path)) / (1024 * 1024))) # Convert to MB
                data["unit"] = "MB"
                data["data_disk"] = 0
                if app_mgmt.resource_manager._app_disks.get(conn_info.app_id):
                    data["data_disk"] = app_mgmt.resource_manager._app_disks.get(conn_info.app_id).get("size", 0)
                total_size += data["application_payload_size"] + data["data_disk"]
                apps_disk_usage.append(data)
        return total_size, apps_disk_usage

    def get_app_data_disk_details(self):
        log.debug("getting the app data disk usage!")
        apps_data_disk_usage = []
        from appfw.runtime.hostingmgmt import HostingManager
        hm = HostingManager.get_instance()
        app_mgmt = hm.get_service("app-management")
        from appfw.profiles.app_profile import AppProfileManager
        pr_manager = None
        if app_mgmt:
            pr_manager = AppProfileManager.getInstance()   
            for conn_info in app_mgmt.connectorInfoMap.values():
                data = self.populate_disk_detail(conn_info.app_id)
                """
                data = {}
                data["id"] = conn_info.app_id
                data_disk = None
                data["asked"] = 0
                data["actual"] = 0
                data["unit"] = "MB"
                if conn_info.app_id in app_mgmt.resource_manager._app_disks:
                    data["asked"] = app_mgmt.resource_manager._app_disks[conn_info.app_id].get("size", 0)
                if self._persist_dir:
                    if os.path.isfile(os.path.join(self._persist_dir, conn_info.app_id + app_mgmt.resource_manager.DATA_EXT2_SUFFIX)):
                        data_disk = os.path.join(self._persist_dir, conn_info.app_id + app_mgmt.resource_manager.DATA_EXT2_SUFFIX)
                    elif os.path.isfile(os.path.join(self._persist_dir, conn_info.app_id + app_mgmt.resource_manager.DATA_EXT4_SUFFIX)):
                        data_disk = os.path.join(self._persist_dir, conn_info.app_id + app_mgmt.resource_manager.DATA_EXT4_SUFFIX)
                if data_disk:
                    data["actual"] = int(round((os.path.getsize(data_disk))/ float(1024 * 1024)))
                    mounted_to = Utils.disk_mounted_to(data_disk)
                    if mounted_to:
                        utilized_disk = int(round((Utils.get_dir_size(mounted_to))/ float(1024 * 1024)))
                        data["utilized"] = utilized_disk
                        data["available"] = data["asked"] - utilized_disk
                """
                apps_data_disk_usage.append(data)
        if pr_manager:
            for profile in pr_manager.app_profiles.values():
                data = self.populate_disk_detail(profile.profile_id+"@pr")
                apps_data_disk_usage.append(data)
        return apps_data_disk_usage

    def populate_disk_detail(self, app_id):
        """
        Poulates the data disk usage of app_id passed
        Returns poulated data dictionary with asked, axtual, available, utilized details
        """ 
        from appfw.runtime.hostingmgmt import HostingManager
        hm = HostingManager.get_instance()
        app_mgmt = hm.get_service("app-management")
        data = {}
        data["id"] = app_id
        data_disk = None
        data["asked"] = 0
        data["actual"] = 0
        data["unit"] = "MB"
        if app_id in app_mgmt.resource_manager._app_disks:
            data["asked"] = app_mgmt.resource_manager._app_disks[app_id].get("size", 0)
        if self._persist_dir:
            if os.path.isfile(os.path.join(self._persist_dir, app_id + app_mgmt.resource_manager.DATA_EXT2_SUFFIX)):
                data_disk = os.path.join(self._persist_dir, app_id + app_mgmt.resource_manager.DATA_EXT2_SUFFIX)
            elif os.path.isfile(os.path.join(self._persist_dir, app_id + app_mgmt.resource_manager.DATA_EXT4_SUFFIX)):
                data_disk = os.path.join(self._persist_dir, app_id + app_mgmt.resource_manager.DATA_EXT4_SUFFIX)
        if data_disk:
            data["actual"] = int(round((os.path.getsize(data_disk))/ float(1024 * 1024)))
            mounted_to = Utils.disk_mounted_to(data_disk)
            if mounted_to:
                utilized_disk = int(round((Utils.get_dir_size(mounted_to))/ float(1024 * 1024)))
                data["utilized"] = utilized_disk
                data["available"] = data["asked"] - utilized_disk
        return data


    def get_layers_disk_usage(self):
        log.debug("Getting the layers disk space usage!")
        layer_disk_usages = []
        total_size = 0
        from appfw.runtime.hostingmgmt import HostingManager
        hm = HostingManager.get_instance()
        layer_reg_serv = hm.get_service("layer_reg_service")
        if layer_reg_serv:
            repo_path = layer_reg_serv.repo
            for layer_info in layer_reg_serv.get():
                data_layer = {}
                data_layer["id"] = layer_info.get("layer_id")
                data_layer["unit"] = "MB"
                data_layer["used_by"] = []
                data_layer["layer_payload_size"] = 0
                if os.path.isdir(os.path.join(repo_path, layer_info.get("layer_id"))):
                    data_layer["layer_payload_size"] = int(round(float(Utils.get_dir_size(os.path.join(repo_path, layer_info.get("layer_id")))) / (1024 * 1024))) # Convert to MB
                    data_layer["used_by"] = layer_info.get("used_by", [])
                total_size += data_layer["layer_payload_size"]
                layer_disk_usages.append(data_layer)
        return total_size, layer_disk_usages

    def get_cartridges_disk_usage(self):
        log.debug("Getting the cartridges disk space usage!")
        cartridges_disk_usages = []
        total_size = 0
        from appfw.runtime.hostingmgmt import HostingManager
        hm = HostingManager.get_instance()
        cartridge_manager = hm.get_service("cartridge-management")
        if cartridge_manager:
            repo_path = cartridge_manager._cartridge_root
            for cart_id in cartridge_manager._cartridges.keys():
                data_layer = {}
                data_layer["id"] = cart_id
                data_layer["unit"] = "MB"
                data_layer["cartridge_payload_size"] = 0
                data_layer["used_by"] = []
                for car_prov in cartridge_manager.get(cart_id).__getattribute__("provides_info"):
                    data_layer["used_by"] += car_prov.get("used_by", [])
                if os.path.isdir(os.path.join(repo_path, cart_id)):
                    data_layer["cartridge_payload_size"] = int(round(float(Utils.get_dir_size(os.path.join(repo_path, cart_id))) / (1024 * 1024))) # Convert to MB
                total_size += data_layer["cartridge_payload_size"]
                cartridges_disk_usages.append(data_layer)
        return total_size, cartridges_disk_usages

    def increase_available_disk_resource(self, app_disk):
        log.debug("Increasing available disk resource.Will add %s MB", app_disk)
        self._available_storage_mb += int(app_disk)
        log.debug("Available disk resource %s", str(self._available_storage_mb))

    @property
    def total_cpu_units(self):
        return self._total_cpu_units

    @property
    def available_cpu_units(self):
        return self._available_cpu_units

    @available_cpu_units.setter
    def available_cpu_units(self, value):
        self._available_cpu_units = value

    @property
    def total_memory(self):
        return self._total_memory_mb

    @property
    def available_memory(self):
        return self._available_memory_mb

    @available_memory.setter
    def available_memory(self, value):
        self._available_memory_mb = value

    @property
    def supported_app_types(self):
        return self._supported_app_types

    @property
    def iox_release(self):
        return self._iox_release

    @property
    def supported_app_validation(self):
        return self._supported_app_validation

    @property
    def supported_remote_docker_access(self):
        return self._remote_docker_supported

    @property
    def app_signature_validation_enabled(self):
        if self.supported_app_validation is None:
            return False
        elif "app_signature" in self.supported_app_validation:
            return True
        else:
            return False

    @property
    def trust_anchor_management_enabled(self):
        if self.supported_app_validation is None:
            return False
        elif "manage_trust_anchor" in self.supported_app_validation:
            return True
        else:
            return False
    @property
    def supported_profile_types(self):
        return self._supported_profile_types

    @property
    def supported_network_modes(self):
        return self._supported_network_modes

    @property
    def apphosting_cpu_shares(self):
        return self._apphosting_cpu_shares

    @property
    def aufs_supported(self):
        return self._aufs_supported

    @property
    def overlayfs_supported(self):
        return self._overlayfs_supported

    @property
    def supported_union_fs(self):
        return self._supported_union_fs

    @property
    def ipv6_supported(self):
        return self._ipv6_supported

    @property
    def supported_app_schemas(self):
        return self._supported_app_schema_list

    @property
    def supported_cartridge_handlers(self):
        return self._supported_cartridge_handlers

    @property
    def host_mount_paths(self):
        return self._supported_host_mount_paths

    @property
    def docker_volume_types(self):
        return self._supported_docker_volume_types

    @property
    def supported_compression_types(self):
        return self._supported_compression_types

    @supported_compression_types.setter
    def supported_compression_types(self, value):
        self._supported_compression_types = value

    @property
    def supported_rootfs_types(self):
        return self._supported_rootfs_types

    @property
    def total_storage(self):
        return self._total_storage_mb

    @property
    def available_storage(self):
        return self._available_storage_mb

    @property
    def total_logvol(self):
        return self._total_logvol_mb

    def available_logvol(self):
        return self.check_volgrp_size()

    @property
    def total_virtual_cores(self):
        return self._total_vcpu_count

    @property
    def max_vcpu_per_app(self):
        return self._max_vcpu_per_app

    @property
    def cpuset(self):
        return self._cpuset

    @property
    def black_listed_docker_run_options(self):
        return self._black_listed_docker_run_options

    @property
    def supported_docker_runtime_options(self):
        return self._supported_docker_runtime_options

    def get_merged_docker_runtime_options(self):
        if not self._supported_merged_docker_runtime_options:
            import copy
            supported_options = copy.deepcopy(self.supported_docker_runtime_options)
            for black_listed_option in self.black_listed_docker_run_options:
                if black_listed_option.lstrip("-") in supported_options:
                    supported_options.remove(black_listed_option.lstrip("-"))
            self._supported_merged_docker_runtime_options = supported_options
        return self._supported_merged_docker_runtime_options


    def get_services_status(self):
        services_status={}
        from appfw.runtime.hostingmgmt import HostingManager
        hm = HostingManager.get_instance()

        dm = hm.get_service("device-management")
        if dm:
            services_status["device-service"] = True
        else:
            services_status["device-service"] = False
        
        nc = hm.get_service("network-management")
        if nc:
            if nc.get_network_status() == True:
                services_status["network-service"] = True
            else:
                services_status["network-service"] = False
        else:
            services_status["network-service"] = False
        
        mon= hm.get_service("monitoring-service")
        if mon:
            services_status["monitoring-service"] = True
        else:
            services_status["monitoring-service"] = False

        app_mgmt = hm.get_service("app-management")
        if app_mgmt :
            services_status["application-management"] = True
        else:
            services_status["application-management"] = False

        sec_mgmt = hm.get_service("security-management")
        if sec_mgmt :
            services_status["security-service"] = True
        else:
            services_status["security-service"] = False

        cs = hm.get_service("console-management")
        if cs :
            services_status["console-service"] = True
        else:
            services_status["console-service"] = False

        scp = hm.get_service("scp-management")
        if scp :
            services_status["scp-service"] = True
        else:
            services_status["scp-service"] = False

        broker = hm.get_service("broker-service")
        if broker :
            services_status["broker-service"] = True
        else:
            services_status["broker-service"] = False

        oauth = hm.get_service("oauth-service")
        if oauth :
            services_status["oauth-service"] = True
        else:
            services_status["oauth-service"] = False

        life_hooks = hm.get_service("lifecycle-hooks")
        if life_hooks :
            services_status["lifecycle-hooks-service"] = True
        else:
            services_status["lifecycle-hooks-service"] = False

        tm = hm.get_service("task-management")
        if tm :
            services_status["task-service"] = True
        else:
            services_status["task-service"] = False

        bist = hm.get_service("bist-service")
        if bist :
            services_status["bist-service"] = True
        else:
            services_status["bist-service"] = False

        ps = hm.get_service("push-service")
        if ps :
            services_status["push-service"] = True
        else:
            services_status["push-service"] = False

        hs = hm.get_service("hasync-service")
        if hs :
            services_status["hasync-service"] = True
        else:
            services_status["hasync-service"] = False

        autocli = hm.get_service("autoconfigcli-management")
        if autocli :
            services_status["autoconfigcli-service"] = True
        else:
            services_status["autoconfigcli-service"] = False

        smartlic = hm.get_service("smartlicense")
        if smartlic :
            services_status["smartlicense-service"] = True
        else:
            services_status["smartlicense-service"] = False

        layer_reg = hm.get_service("layer_reg_service")
        if layer_reg :
            services_status["layer_reg-service"] = True
        else:
            services_status["layer_reg-service"] = False

        dstore = hm.get_service("datastore_service")
        if dstore :
            services_status["datastore-service"] = True
        else:
            services_status["datastore-service"] = False

        return services_status

    def get_platform_capability(self, detailed=False):
        #get the installed cartridges
        from appfw.runtime.hostingmgmt import HostingManager
        from appfw.runtime.hostingmgmt import HostingManager
        from resourcemanager import ResourceManager

        rsmgr = ResourceManager.getInstance()
        rsmgr.update_platform_resource()
        hm = HostingManager.get_instance()
        dm = hm.get_service("device-management")

        cm = CartridgeManager.getInstance()
        cart_list = cm.list()
        capability = dict()
        capability["mgmt_api_version"] = self.MGMT_API_VERSION
        capability["min_app_manifest_version"] = self.MIN_APP_MANIFEST_VERSION


        if self._platform_product_id == "default":
            self._platform_product_id = Utils.getPlatformProductID()
        capability["product_id"] = self._platform_product_id
        capability["compute_nodes"] = []

        native_compute_node = dict()
        native_compute_node["name"] = self._compute_node_name
        native_compute_node["id"] = self._compute_node_id

        from appfw.runtime.hostingmgmt import HostingManager
        native_compute_node["installed_services"] = []

        app_mgr = hm.get_service("app-management")
        nc = hm.get_service("network-management")

        features = []
        if app_mgr:
            services = app_mgr.list(is_service=True)
            features = app_mgr.get_supported_features()
            if hm.get_service("smartlicense") is not None:
                features.append("smartlicense")
                
            if PlatformDiagnostic.is_platform_diagnostic_supported():
                features.append("platform_diagnostic")
                
            native_compute_node["supported_features"] = features

            for service in services:
                service_info = {}
                service_info["id"] = service.id
                service_info["provides"]= service.provides
                service_info["status"]= service.state
                native_compute_node["installed_services"].append(service_info)
            if app_mgr.platform_services_enabled:
                native_compute_node["platform_services"] = app_mgr.get_platform_services()
        log.debug("Checking for CAF enabled services")
        native_compute_node["caf_services"] = []
        caf_services = Utils.get_caf_services()
        for service in caf_services:
            native_compute_node["caf_services"].append(service)

        native_compute_node["resources"] = dict()

        native_compute_node["resources"]["visualization"] = self._visualization_supported
        native_compute_node["resources"]["privileged_mode_supported"] = self._privileged_mode_supported
        native_compute_node["resources"]["datastore"] = self._datastore_supported
        native_compute_node["resources"]["disk_info"] = dict()
        total_disk, available_disk, utilized_disk = Utils.get_disk_space_details(self._repo_partition)
        native_compute_node["resources"]["disk_info"]["total_disk_provided"] = total_disk
        native_compute_node["resources"]["disk_info"]["disk_available"] = available_disk
        native_compute_node["resources"]["disk_info"]["disk_utilized"] = utilized_disk
        native_compute_node["resources"]["disk_info"]["disk_partition"] = self.get_mount_device_disk_name(self._repo_partition)
        native_compute_node["resources"]["disk_info"]["unit"] = "MB"
        if detailed:
            total_apps_disk, apps_disk_usage = self.get_apps_disk_usage()
            native_compute_node["resources"]["disk_info"]["apps_disk_usage"] = {
                "apps_disk_usage":apps_disk_usage,
                "total": total_apps_disk,
                "unit": "MB"
            }
            total_layers_disk, layers_disk_usage = self.get_layers_disk_usage()
            native_compute_node["resources"]["disk_info"]["layers_disk_usage"] = {
                "layers_disk_usage": layers_disk_usage,
                "total": total_layers_disk,
                "unit": "MB"
            }
            total_cartridges_disk, cartridges_disk_usage = self.get_cartridges_disk_usage()
            native_compute_node["resources"]["disk_info"]["cartridges_disk_usage"] = {
                "cartridges_disk_usage": cartridges_disk_usage,
                "total": total_cartridges_disk,
                "unit": "MB"
            }

        native_compute_node["resources"]["cpu"] = dict()
        native_compute_node["resources"]["cpu"]["info"] = SystemInfo.get_cpu_info()
        native_compute_node["resources"]["cpu"]["vcpu_count"] = self._total_vcpu_count
        native_compute_node["resources"]["cpu"]["total"] = self._total_cpu_units
        native_compute_node["resources"]["cpu"]["available"] = self._available_cpu_units
        native_compute_node["resources"]["cpu"]["max_vcpu_per_app"] = self._max_vcpu_per_app
        native_compute_node["resources"]["cpu"]["unit"] = "UNITS"

        native_compute_node["resources"]["memory"] = dict()
        native_compute_node["resources"]["memory"]["total"] = self._total_memory_mb
        native_compute_node["resources"]["memory"]["available"] = self._available_memory_mb
        native_compute_node["resources"]["memory"]["unit"] = "MB"

        native_compute_node["resources"]["storage"] = dict()
        native_compute_node["resources"]["storage"]["total"] = self._total_storage_mb
        native_compute_node["resources"]["storage"]["data_disk_details"] = self.get_app_data_disk_details()
        available_storage = self._available_storage_mb

        if self._persist_dir:
            free_avl_space = Utils.get_free_disk_space(self._persist_dir)
            log.debug("Persistent dir: %s Free space: %s" % (self._persist_dir, free_avl_space))
            if free_avl_space < available_storage:
                available_storage = free_avl_space

        native_compute_node["resources"]["storage"]["available"] = available_storage
        enable_lvm = Utils.getSystemConfigValue("controller", "enable_lvm", default=False, parse_as="bool")
        native_compute_node["resources"]["enable_lvm"] = enable_lvm 
        if enable_lvm:
            native_compute_node["resources"]["logvol"] = dict()
            native_compute_node["resources"]["logvol"]["total"] = self._total_logvol_mb
            native_compute_node["resources"]["logvol"]["available"] = self.available_logvol()

        native_compute_node["resources"]["disk_storage"] = dict()
        native_compute_node["resources"]["disk_storage"]["app_disk"] = dict()
        native_compute_node["resources"]["disk_storage"]["app_disk"]["device"] = self.get_mount_device_disk_name(self._repo_partition)
        native_compute_node["resources"]["disk_storage"]["app_disk"]["available"] = Utils.get_free_disk_space(self._repo_partition)
        if (self._persistent_partition != self._repo_partition):
            native_compute_node["resources"]["disk_storage"]["data_disk"] = dict()
            native_compute_node["resources"]["disk_storage"]["data_disk"]["device"] = self.get_mount_device_disk_name(self._persistent_partition)
            native_compute_node["resources"]["disk_storage"]["data_disk"]["available"] = Utils.get_free_disk_space(self._persistent_partition)
        disk_status = self.check_disk_status()
        native_compute_node["resources"]["disk_status"]=disk_status
        if nc:
            native_compute_node["resources"]["host_mode"] = nc.host_mode
            native_compute_node["resources"]["networks"] = nc.list_networks()
            native_compute_node["resources"]["default_network"] = nc.get_default_network()
        device_list = dm.list_devices();
        native_compute_node["resources"]["devices"] = device_list
        if self.supported_app_validation is not None:
            if app_mgr:
                native_compute_node["supported_features"].extend(self.supported_app_validation)
            native_compute_node["resources"]["app_signing"] = dict()
            from ..app_package.pkgsign import PackageSigning
            pkgsign = PackageSigning.getInstance()
            native_compute_node["resources"]["app_signing"]["check_package_signature"] = pkgsign.appsign_enabled
            if "manage_trust_anchor" in self.supported_app_validation:
                native_compute_node["resources"]["app_signing"]["trust_anchor_checksum"] = pkgsign.calculate_sha1sum()

        native_compute_node["third_party_apps"] = {}
        native_compute_node["third_party_apps"]["allowed"] = self._third_party_apps
        native_compute_node["third_party_apps"]["status"] = "OK"
        if not self._third_party_apps:
            native_compute_node["third_party_apps"]["status"] = "Storage device not available for hosting third party apps."
        native_compute_node["third_party_apps"]["current_storage"] = os.path.dirname(self._repo_dir)

        native_compute_node["installed_cartridges"] = cart_list
        native_compute_node["supported_app_types"] = self._supported_app_types
        native_compute_node["iox_release"] = self._iox_release
        native_compute_node["ipv6_supported"] = self._ipv6_supported
        native_compute_node["remote_docker_supported"] = self._remote_docker_supported
        native_compute_node["supported_profile_types"] = self._supported_profile_types
        native_compute_node["supported_app_schemas"] = self._supported_app_schema_list
        native_compute_node["apphosting_cpu_shares"] = self._apphosting_cpu_shares
        native_compute_node["caf_status"] = SystemHealthInfo.get_iox_health()
        native_compute_node["caf_services_status"] = self.get_services_status()
        native_compute_node["aufs_supported"] = self._aufs_supported
        native_compute_node["overlayfs_supported"] = self._overlayfs_supported
        native_compute_node["supported_features"] = features
        native_compute_node["supported_host_mount_paths"] = self._supported_host_mount_paths
        native_compute_node["supported_docker_volume_types"] = self._supported_docker_volume_types
        native_compute_node["supported_compression_types"] = self._supported_compression_types
        native_compute_node["black_listed_docker_runtime_options"] = self._black_listed_docker_run_options
        native_compute_node["docker_runtime_supported_options"] = self.supported_docker_runtime_options
        Secure_Storage_section = Utils.getSystemConfigSection("secure_storage_server")
        if Secure_Storage_section is not None and Secure_Storage_section.get("enabled", "no") == "yes":
            native_compute_node["supported_features"].append("secure_storage")
        push_notifications = Utils.get_push_notifications_cap()
        is_push_supported = (push_notifications and True) or False
        if is_push_supported:
            native_compute_node["push_notifications"] = push_notifications
        capability["compute_nodes"].append(native_compute_node)

        #update the system-capabilities supported by platform
        sc = hm.get_service("security-management")
        if sc:
            caps = sc.get_platform_syscap()
            if caps:
                capability.update(caps)

        return capability

    def get_json_repr(self):
        import json
        return json.dumps(self.get_platform_capability())

    @classmethod
    def getInstance(cls):
        '''
        Returns a singleton instance of the class
        '''
        if not cls.__singleton:
            cls.__singleton = PlatformCapabilities()
        return cls.__singleton

'''
if "__main__" == __name__:
    import json
    pc = PlatformCapabilities()
    print json.dumps(pc.get_platform_capability(), indent=2)
    '''
