'''
Copyright (c) 2014 by Cisco Systems, Inc.
All rights reserved.

@author: havishwa
'''

import logging
import os
import shutil
from stager import StagingResponse
from ..utils.utils import Utils
from ..utils import utils
from ..utils.infraexceptions import StagingError
from appfw.utils.commandwrappers import *


log = logging.getLogger("runtime.hosting")

class LXCStager(object):
    """
    Stager for docker type application container
    """
    __singleton = None
    
    def __new__(cls, *args, **kwargs):
        # subclasses will create their own __singleton objects
        if cls != type(cls.__singleton):
            cls.__singleton = super(LXCStager, cls).__new__(cls, *args, **kwargs)
        return cls.__singleton

    @classmethod
    def getInstance(cls, *args):
        '''
        Returns a singleton instance of the class
        '''
        if not cls.__singleton:
            cls.__singleton = LXCStager(*args)
        return cls.__singleton

    def __init__(self, config, languageRuntimes=None):
        self._config = config
        self._staging_location = self._config.get("lxc-container", "lxc_staging_directory")
        if not os.path.isdir(self._staging_location):
            os.makedirs(self._staging_location)
        
    def stageConnector(self, stagingReq):
        """
        LXC stager simply moves the extracted contents to the corresponding 
        stager directory. It avoids copying 
        """
        log.debug("Begin staging for lxc app. AppID: %s" % stagingReq.getConnectorId())
        staging_location = os.path.join(self._staging_location, stagingReq.getConnectorId())
        if not os.path.isdir(staging_location):
            os.makedirs(staging_location)


        # The artifacts.tar.gz envelope is already extracted by the pkg manager.
        # The extracted path should have app.ext2 (that contains the rootfs of the actual app)
        # Create the ext2 image for staging

        pkg = stagingReq.getPackageObject()
        appmanifest = stagingReq.getMetadata()

        # Find out what the app developer has specified in the rootfs section
        rootfs = appmanifest.startup.get("rootfs")

        # Check for the existence of app.ext2

        app_ext2_file = Utils.normalize_artifact_path(rootfs, pkg.dest_dir)
        #app_ext2_file = os.path.join(pkg.dest_dir, rootfs)
        log.debug("Checking to see if app ext2 file exists: %s", app_ext2_file)
        if not os.path.isfile(app_ext2_file):
            raise StagingError("Application ext2 disk is missing! : %s" % app_ext2_file)

        log.info("App ext2 %s found", app_ext2_file)
        mnt_location = ""

        try:

            # If app's rootfs disk is found, mount and do basic format and rootfs checks

            # If CAF crashes for some reason, but the app's container is still running,
            # the disk will be already mounted.
            # In such scenario, skip all the pre-tests and let the container manager reconcile states

            if Utils.is_disk_mounted(app_ext2_file):
                log.warning("It appears that disk %s is already mounted!", app_ext2_file)
                log.warning("Stager will proceed without attempting to mount for basic sanity checks..")

            else:
                log.debug("Doing basic sanity checks on rootfs disk..")
                mnt_location = os.path.join(staging_location, "mnt")
                if not os.path.isdir(mnt_location):
                    os.makedirs(mnt_location)

                if Utils.ismount_exists(mnt_location) :
                    out, rc = umount("-l",  mnt_location)
                    if rc != 0:
                        log.error("Unmounting failed for an mount point %s. ret code: %s error: %s"
                                    % (mnt_location, rc, str(out)))
                        raise StagingError("Unmounting failed for an mount point %s. ret code: %s error: %s"
                                    % (mnt_location, rc, str(out)))

                # Get security mount options for transient operations. For
                # SELinux provides a context that CAF can do operations on
                # without having to be unconfined but before resources
                # including security attributes have been allocated to
                # an app.
                from appfw.runtime.hostingmgmt import HostingManager
                hm = HostingManager.get_instance()
                sc = hm.get_service("security-management")
                lsm_mnt_string = sc.get_passive_mount_string()

                # Mount the app.ext2 and do basic rootfs checks
                if not Utils.ismount_exists(mnt_location):
                    log.debug("Mounting application as ext2 at:%s" % mnt_location)
                    if lsm_mnt_string:
                        out, rc = mountext2('-o', lsm_mnt_string, app_ext2_file, mnt_location)
                    else:
                        out, rc = mountext2(app_ext2_file, mnt_location)
                    if rc != 0:
                        log.error("Error in Mounting  app ext2 image: %s", str(out))
                        raise StagingError("Error in mounting app ext2 image: %s" % str(out))

                    log.debug("Mounted app ext2 directory for container %s at %s" %
                            (stagingReq.getConnectorId(), mnt_location))

                rfsdirs = os.listdir(mnt_location)

                # Verify that common/expected directories exist
                es = set(['bin', 'lib', 'etc'])
                if len(es.intersection(set(rfsdirs))) != len(es):
                    log.error("Expected directories %s not found at %s", es, mnt_location)
                    raise StagingError("Rootfs seems to be invalid. Missing mandatory directories %s" % es)

                log.info("Verified that %s contains mandatory rootfs directories" % app_ext2_file)



            app_config_file_name = Utils.find_app_config_filename(pkg.dest_dir)
            if app_config_file_name:
                app_config_file_name = os.path.join(pkg.dest_dir, app_config_file_name)

            response = dict()
            response["staged-package-path"] = pkg.dest_dir
            response["app-config-file-name"] = app_config_file_name
            sr = StagingResponse(stagingReq.getConnectorId(),
                                    stagingReq.getPackagePath(),
                                    staging_location,
                                    response)
            log.debug("Returning staging response : %s" % str(sr))
            return sr

        finally:
            if Utils.ismount_exists(mnt_location) :
                out, rc = umount("-l",  mnt_location)
                if rc != 0:
                    log.error("Unmounting failed for an mount point %s. ret code: %s error: %s"
                                % (mnt_location, rc, str(out)))
                    raise StagingError("Unmounting failed for an mount point %s. ret code: %s error: %s"
                                % (mnt_location, rc, str(out)))
            if os.path.isdir(mnt_location):
                shutil.rmtree(mnt_location, ignore_errors=True)

    def finish(self, connector, metadata, metadataFile, connectorArchiveFile, stagingResp):
        """
        Finish staging after connector deploys
        """
        if connectorArchiveFile:
            if os.path.exists(connectorArchiveFile):
                os.remove(connectorArchiveFile) 
        log.debug("Staging complete.")
