__author__ = 'hvishwanath'

import logging
import tempfile
import os
import falcon
import json
import re

from jsonencoder import JSONEncoder
from apiservice import ResourceRoute, APIService
from common import AuthenticatedResource, make_response, make_error_response
from ..utils.cafevent import CAFEvent
from ..utils.infraexceptions import *

log = logging.getLogger("runtime.api.resources")

jsonencoder = JSONEncoder()

@ResourceRoute("/platform/networks", endpoint="networks")
class NetworksResource(AuthenticatedResource):

    def on_get(self, request, response):
        network_manager = APIService.instance.network_manager
        networkInfo = network_manager.list_networks()
        out = jsonencoder.encode(networkInfo)
        response.body = out
        response.status = falcon.HTTP_200
        response.set_headers({'Content-Type': "application/json",
                              'Cache-Control': "no-cache"})


@ResourceRoute("/platform/macregistry", endpoint="macregistry")
class MacRegistryResource(AuthenticatedResource):

    def on_get(self, request, response):
        network_manager = APIService.instance.network_manager
        out = network_manager.get_mac_registry()
        response.body = jsonencoder.encode(out)
        response.status = falcon.HTTP_200
        response.set_headers({'Content-Type': "application/json",
                              'Cache-Control': "no-cache"})


@ResourceRoute("/platform/portregistry", endpoint="portregistry")
class MacRegistryResource(AuthenticatedResource):

    def on_get(self, request, response):
        network_manager = APIService.instance.network_manager
        out = network_manager.get_port_registry()
        response.body = jsonencoder.encode(out)
        response.status = falcon.HTTP_200
        response.set_headers({'Content-Type': "application/json",
                              'Cache-Control': "no-cache"})


@ResourceRoute("/platform/networks/{network_id}", endpoint="network")
class NetworkResource(AuthenticatedResource):

    def on_get(self, request, response, network_id):
        network_manager = APIService.instance.network_manager
        network = network_manager.get_network(network_id)
        if network is None:
            response = make_error_response(response,
                                            "The network : %s, does not exist" % network_id,
                                            "The network : %s, does not exist" % network_id,
                                            falcon.HTTP_404)
            return

        response.status = falcon.HTTP_200
        response.body = jsonencoder.encode(network.serialize())
        response.set_headers({'Content-Type': "application/json",
                              'Cache-Control': "no-cache"})


@ResourceRoute("/platform/network_config", endpoint="network_config")
class NetworkConfigResource(AuthenticatedResource):

    def on_get(self, request, response):
        network_manager = APIService.instance.network_manager
        config = network_manager.config
        response.status = falcon.HTTP_200
        response.body = jsonencoder.encode(config)
        response.set_headers({'Content-Type': "application/json",
                              'Cache-Control': "no-cache"})

    def on_put(self, request, response):
        network_manager = APIService.instance.network_manager

        try:
            config = json.load(request.stream)
            log.debug("Request to update network controller's config to %s", config)
            network_manager.config = config

            config = network_manager.config
            response.status = falcon.HTTP_200
            response.body = jsonencoder.encode(config)
            response.set_headers({'Content-Type': "application/json",
                                  'Cache-Control': "no-cache"})
        except ValueError as ve:
            log.exception("Value error in incoming payload")

            response = make_error_response(response,
                      "Invalid request payload : %s" % str(ve),
                      "Invalid request payload",
                      falcon.HTTP_400)

        except Exception as ex:
            log.exception("Error setting network config")
            if isinstance(ex, C3Exception):
                response = make_error_response(response, ex.message,
                                               "Error setting network config",
                                               falcon.HTTP_500,
                                               ex.errorcode)
            else:
                response = make_error_response(response,
                          "Error setting network config: %s" % str(ex),
                          "Error setting network config",
                          falcon.HTTP_500)

        return

@ResourceRoute("/platform/networks/default_network", endpoint="default_network")
class DefaultNetworkResource(AuthenticatedResource):

    def on_get(self, request, response):
        net_docker = request.get_param("docker")
        if isinstance(net_docker, basestring):
            if net_docker.lower() == "false":
                net_docker = False
            elif net_docker.lower() == "true":
                net_docker = True
            else:
                net_docker = False
        else:
            net_docker = False
        network_manager = APIService.instance.network_manager
        dn = dict()
        dn["default_network"] = network_manager.get_default_network(net_docker)

        out = jsonencoder.encode(dn)
        response.body = out
        response.status = falcon.HTTP_200
        response.set_headers({'Content-Type': "application/json",
                              'Cache-Control': "no-cache"})


    def on_put(self, request, response):
        network_manager = APIService.instance.network_manager

        try:
            dn = json.load(request.stream)
            default_network = dn["default_network"]
            log.debug("Request to set default network to %s", default_network)
            network_manager.set_default_network(default_network)

            dn = dict()
            dn["default_network"] = network_manager.get_default_network()

            out = jsonencoder.encode(dn)
            response.body = out
            response.status = falcon.HTTP_200
            response.set_headers({'Content-Type': "application/json",
                                  'Cache-Control': "no-cache"})

        except ValueError as ve:
            log.exception("Value error in incoming payload")

            response = make_error_response(response,
                      "Invalid request payload : %s" % str(ve),
                      "Invalid request payload",
                      falcon.HTTP_400)

        except Exception as ex:
            log.exception("Error setting default network")
            if isinstance(ex, C3Exception):
                response = make_error_response(response, ex.message,
                                               "Error setting default network: %s" % str(ex),
                                               falcon.HTTP_500,
                                               ex.errorcode)
            else:
                response = make_error_response(response,
                          "Error setting default network : %s" % str(ex),
                          "Error setting default network",
                          falcon.HTTP_500)

        return



@ResourceRoute("/platform/networks/hosting_bridges", endpoint="hb")
class BridgesResource(AuthenticatedResource):

    def on_get(self, request, response):
        network_manager = APIService.instance.network_manager
        if network_manager is None:
            log.error("Network manager is disabled")
            response = make_error_response(response,
                                           "Network manager is disabled. Check latest IOx events and errors for troubleshooting.",
                                           "Network manager is disabled. Check latest IOx events and errors for troubleshooting.",
                                           falcon.HTTP_503)
            return
        networkInfo = network_manager.list_hosting_bridges()
        out = jsonencoder.encode(networkInfo)
        response.body = out
        response.status = falcon.HTTP_200
        response.set_headers({'Content-Type': "application/json",
                              'Cache-Control': "no-cache"})
    
    # Create hosting bridge    
    def on_post(self, request, response):
        network_manager = APIService.instance.network_manager
        
        try:
            bridge_config = json.load(request.stream)                             
            bridge_id = network_manager.add_hosting_bridge(bridge_config)
            bridge_url = request.uri + "/" + bridge_id
            header = {"Content-Location": bridge_url}
            response = make_response(response, "", falcon.HTTP_201, header)
            log.info("Created bridge network: %s" % bridge_url)
        except ValueError as ve:
            log.exception("Value error in incoming payload")

            response = make_error_response(response,
                      "Invalid request payload : %s" % str(ve),
                      "Invalid request payload",
                      falcon.HTTP_400)

        except Exception as ex:
            log.exception("Error creating the hosting bridge.")
            if isinstance(ex, C3Exception):
                response = make_error_response(response, ex.message,
                                               "Error creating network: %s" % str(ex),
                                               falcon.HTTP_500,
                                               ex.errorcode)
            else:
                response = make_error_response(response,
                          "Error creating network : %s" % str(ex),
                          "Error creating network",
                          falcon.HTTP_500)


@ResourceRoute("/platform/networks/hosting_bridges/{bridge_id}", endpoint="bridge")
class BridgeResource(AuthenticatedResource):

    def on_get(self, request, response, bridge_id):
        network_manager = APIService.instance.network_manager
        network = network_manager.get_hosting_bridge(bridge_id)
        if network is None:
            response = make_error_response(response,
                                            "The bridge : %s, does not exist" % bridge_id,
                                            "The bridge : %s, does not exist" % bridge_id,
                                            falcon.HTTP_404)
            return

        response.status = falcon.HTTP_200
        response.body = jsonencoder.encode(network.serialize())
        response.set_headers({'Content-Type': "application/json",
                              'Cache-Control': "no-cache"})
                              
    # Update hosting bridge
    def on_put(self, request, response, bridge_id):
        network_manager = APIService.instance.network_manager
        
        try:
            bridge_config = json.load(request.stream)
            network_manager.update_hosting_bridge(bridge_id, bridge_config)
            response = make_response(response, "", falcon.HTTP_200)
            
        except NetworkNotExistingError as ex:
            log.exception("Hosting bridge %s doesn't exist." % bridge_id)
            response = make_error_response(response, ex.message,
                                           "Error updating network: %s" % str(ex),
                                           falcon.HTTP_404,
                                           ex.errorcode)
        
        except ValueError as ve:
            log.exception("Value error in incoming payload")

            response = make_error_response(response,
                      "Invalid request payload : %s" % str(ve),
                      "Invalid request payload",
                      falcon.HTTP_400)

        except Exception as ex:
            log.exception("Error updating the hosting bridge.")
            if isinstance(ex, C3Exception):
                response = make_error_response(response, ex.message,
                                               "Error updating network: %s" % str(ex),
                                               falcon.HTTP_500,
                                               ex.errorcode)
            else:
                response = make_error_response(response,
                          "Error updating network : %s" % str(ex),
                          "Error updating network",
                          falcon.HTTP_500)
                
        
    # Delete hosting bridge
    def on_delete(self, request, response, bridge_id):
        network_manager = APIService.instance.network_manager
        
        try:
            network_manager.delete_hosting_bridge(bridge_id)
            response = make_response(response, "", falcon.HTTP_200)
            log.info("Deleted hosting bridge %s" % bridge_id) 
        except NetworkNotExistingError as ex:
            log.exception("Hosting bridge %s doesn't exist." % bridge_id)
            response = make_error_response(response, ex.message,
                                           "Error updating network: %s" % str(ex),
                                           falcon.HTTP_404,
                                           ex.errorcode)
        
        except Exception as ex:
            log.exception("Error deleting network")
            if isinstance(ex, C3Exception):
                response = make_error_response(response, ex.message,
                                               "Error deleting network: %s" % str(ex),
                                               falcon.HTTP_500,
                                               ex.errorcode)
            else:
                response = make_error_response(response,
                          "Error deleting network : %s" % str(ex),
                          "Error deleting network",
                          falcon.HTTP_500)
