#-----------------------------------------------------
#
# Copyright (c) 2012-2016 by cisco Systems, Inc.
# All rights reserved.
#-----------------------------------------------------

__all__ = ["PushService"]

import Queue
import logging
import ssl
from   appfw.utils.cafevent import CAFEvent
from   appfw.utils.utils import Utils
from ..taskmgmt.taskmanager import TaskManager
import urlparse
import json
import yaml
from appfw.api.jsonencoder import JSONEncoder
from ConfigParser import RawConfigParser
from ..utils.infraexceptions import NetworkConnectionError, InvalidConfigError
from appfw.runtime.caf_abstractservice import CAFAbstractService
import time
import os
CALLHOME_CONFIG_FILE_NAME = "callhome_config.json"
jsonencoder = JSONEncoder()
log = logging.getLogger("push_service")
taskmgr = TaskManager.getInstance()
from appfw.runtime.stats import StatsCollector
from datetime import datetime


def retry(max_retries=2, timeout=2):
    def wraps(func):
        request_exceptions = (
                Exception
        )
        def inner(*args, **kwargs):
            for i in range(max_retries):
                try:
                    result = func(*args, **kwargs)
                except request_exceptions:
                    time.sleep(timeout)
                    continue
                else:
                    return result
            else:
                raise NetworkConnectionError
        return inner
    return wraps

class PushService(CAFAbstractService):

    __singleton = None # the one, true Singleton

    def __new__(cls, *args, **kwargs):
        # Check to see if a __singleton exists already for this class
        # Compare class types instead of just looking for None so
        # that subclasses will create their own __singleton objects
        if cls != type(cls.__singleton):
        #if not cls.__singleton:
            cls.__singleton = super(PushService, cls).__new__(cls, *args, **kwargs)
        return cls.__singleton

    def __init__(self, params, push_repo_folder, event_subscriber, event_unsubscriber, client=None):
        self.name = params.name
        self._config = params.config
        self._config_file = params.config_file
        self.event_queue = Queue.Queue()
        self.call_home_params = {}
        self.server_url_list = []
        self.token_id = None
        self.token = None
        self.client = client
        self.taskmgr = TaskManager.getInstance()
        self.app_metrics_collect_freq = 60
        self.app_metrics_publish_freq = 90
        self.call_home_url = None
        self.publish_queue = Queue.Queue(maxsize=20)
        #self._config = config
        self._store_path = push_repo_folder
        self.event_subscriber = event_subscriber
        self.event_unsubscriber = event_unsubscriber
        self._protocols = ["websocket", "http"]
        self._task_list = []
        #self.heartbeat_freq = int(self._config.get("heartbeat_freq", 10))
        #self.time_out = int(self._config.get("timeout", 20))
        self.is_call_home_set = False
        self._running = False
        self.http_server = None



        log.debug("Initialization of push service")

    @property
    def is_enabled(self):
        if isinstance(self._config.get("enabled"), basestring):
            return RawConfigParser._boolean_states.get(self._config.get("enabled", 'false'), False)
        return self._config.get("enabled", False)

    @property
    def heartbeat_freq(self):
        return int(self._config.get("heartbeat_freq", 10))

    @property
    def time_out(self):
        return int(self._config.get("timeout", 20))

    @property
    def is_running(self):
        return self._running

    @property
    def supported_client_protocols(self):
        return self._protocols

    def get_config(self):
        conf = {}
        conf["enabled"] = self.is_enabled
        conf["heartbeat_freq"] = self.heartbeat_freq
        conf["timeout"] = self.time_out
        conf["protocols"] = self.supported_client_protocols
        return conf

    def subscribe_events(self):
        log.debug("Calling event subscriber in push service")
        self.event_subscriber(self.event_handler)

    def unsubscribe_events(self):
        log.debug("Calling event subscriber in push service")
        self.event_unsubscriber(self.event_handler)

    def event_handler(self, caf_event):
        log.debug("Sending event %s to notification server", caf_event)
        if self.is_call_home_set:
            try:
                if StatsCollector.getInstance().enabled:
                    restapi_registry_push = StatsCollector.getInstance().get_statsregistry("PUSHNOT", "event")
                    restapi_registry_push.gauge("last_event").set_value(str(caf_event))
                    if caf_event.event_nature == "metrics":
                        restapi_registry_push.counter("metrics_cnt").inc()
                    else:
                        restapi_registry_push.counter("unsoliciated_cnt").inc()
                    restapi_registry_push.gauge("last_event_time").set_value(str(datetime.now()))
                    restapi_registry_push.counter("total_events").inc()
                self.push_event(caf_event)
            except Exception as ex:
                if StatsCollector.getInstance().enabled:
                    restapi_registry_push = StatsCollector.getInstance().get_statsregistry("PUSHNOT", "event")
                    restapi_registry_push.gauge("last_error_time").set_value(str(datetime.now()))
                    restapi_registry_push.counter("total_error").inc()
                    restapi_registry_push.gauge("last_error").set_value(str(ex))
                log.exception("Exception %s while sending message to the server", str(ex))
                self.on_server_connection_error(str(ex))

    def set_config(self, config):
        if self.validate_config(config):
            try:
                if self.is_running:
                    self.stop()
            except Exception as ex:
                log.exception("Push service stop failed, with reason: %s"%str(ex))
                raise Exception("Push service stop failed, with reason: %s"%str(ex))
            self._update_config(config)
            try:
                if self.is_enabled:
                    self.start()
                else:
                    log.debug("Push service is disabled as part of new config update!")
            except Exception as ex:
                log.exception("Error while setting up the Push service with new config %s, cause: %s"%(config, str(ex)))
                self.stop()
                raise Exception("Error while setting up the Push service with new config %s, cause: %s"%(config, str(ex)))
        else:
            log.error("Given config %s is invalid!"%config)
            raise InvalidConfigError("Given config %s is invalid!"%config)

    def validate_config(self, config):
        log.debug("Validating the given config %s"%config)
        allowed_keys = self._config.keys()
        for key in config.keys():
            if key not in allowed_keys:
                log.debug("Invalid key %s, has been found in new config"%key)
                return False
        return True

    def _update_config(self, config):
        self._config.update(config)
        self._save_data()

    def _save_data(self):
        """
        Save config file to disk. Default location is repo/running_config/.push_service. Will be in yaml format
        :return:
        """
        with open(self._config_file, "w", 0) as f:
            yaml.safe_dump(self._config, f, default_flow_style=False)
            log.debug("Saved monitoring configuration to %s", self._config_file)

    @retry(max_retries=3, timeout=5)
    def push_event(self, caf_event):
        if self.client:
            try:
                ret_val = self.client.send_message(jsonencoder.encode(caf_event.serialize()))
                if not ret_val:
                    log.warning("Client is not connected, unable to send events")
                    raise Exception("Client is not connected, cannot push events")
                    #return ret_val
            except Exception as ex:
                log.exception("Exception while sending events in JSON format to the server %s", str(ex))
                raise ex
        if self.http_server:
            try:
                ret_val = self.push_event_https_server(self.http_server, self.shared_key, caf_event.serialize())
                if not ret_val:
                    log.warning("Client is not connected, unable to send events")
                    raise Exception("Client is not connected, cannot push events")
            except Exception as ex:
                log.exception("Exception while sending events in JSON format to the server %s", str(ex))
                raise ex
        if not self.client and not self.http_server:
            raise ValueError("There is no client connection details are provided")

    def _construct_callhome_config_file(self, config):
        is_config_available = (config and True) or False
        if not is_config_available:
            raise ValueError("Call home Config empty!!")
        file_name = os.path.join(self._store_path, CALLHOME_CONFIG_FILE_NAME)
        log.debug("writing call home config into  %s", file_name)
        if os.path.exists(self._store_path):
            file(file_name, "w").write(json.dumps(config))
            log.debug("call home config written to the disk")
        else:
            raise Exception("Repo for push service not available")

    def start(self):
        if self.is_enabled:
            log.debug("Starting push service")
            self.subscribe_events()
            try:
                task_id = taskmgr.queue_task(self.get_push_config, args=())
                self.put_push_task(task_id)
                self._running = True
            except Exception as ex:
                log.exception("Posting task returned exception %s", str(ex))

    def stop(self, forceful=True):
        """If forceful, terminate dispatcher thread immediately.
        Else, wait on queue so that all events are serviced and then exit"""
        if self.is_enabled:
            log.debug("Stopping Push Service")
            self.unsubscribe_events()
            self.delete_all_push_tasks()
            if not forceful:
                log.debug("Waiting for event queue to be completely serviced")
                if self.publish_queue:
                    self.publish_queue.join()
            if self.client:
                try:
                    self.client.close(timeout=20)
                except Exception as ex:
                    log.exception("Exception while closing the connection %s", str(ex))
                self.client = None
            self._running = False

    @classmethod
    def getInstance(cls, *args):
        '''
        Returns a singleton instance of the class
        '''
        if cls.__singleton == None:
            cls.__singleton = PushService(*args)

        return cls.__singleton

    def _clear_queue(self):
        q = self.publish_queue
        while not q.empty():
            q.get()
            q.task_done()

    def put_push_task(self, id):
        if id:
            if id not in self._task_list:
                log.debug("Inserting %s into push service list", id)
                self._task_list.append(id)
        else:
            log.debug("task id is None")

    def get_push_task(self, id):
        if id in self._task_list:
            return True
        return False

    def delete_push_task(self, id):
        if id in self._task_list:
            log.debug("Deleting id %s from the push list", id)
            self._task_list.remove(id)
            return True
        return False

    def delete_all_push_tasks(self):
        for task_id in self._task_list:
            taskmgr.revoke(task_id)
            #self._task_list.remove(task_id)
            log.debug("Removing push task %s", task_id)
        if self._task_list:
            del self._task_list[:]


    def set_callhome_params(self, callhome_payload):
        try:
            self.clear_callhome_params()
        except Exception as ex:
            log.exception("Exception in clearing call home config before setting up with new config %s", str(ex))
        log.debug("Setting call home params %s", callhome_payload)
        self.call_home_params = callhome_payload
        self.call_home_url = callhome_payload.get("call_home_url", None)
        if not self.call_home_url:
            raise ValueError("Call url not provided in the body")
        self.token_id = callhome_payload.get("token", None)
        if not self.token_id:
            raise ValueError("Token ID not provided in the body")
        try:
            self._construct_callhome_config_file(callhome_payload)
            self.is_call_home_set = True
            task_id = taskmgr.queue_task(self.get_push_config, args=(self.call_home_url, self.token_id, ))
            self.put_push_task(task_id)
        except Exception as ex:
            log.exception("func to get push config from the server raised exception %s", str(ex))
        return self.call_home_url

    def clear_callhome_params(self):
        if self.client:
            try:
                self.client.close(timeout=20)
            except Exception as ex:
                log.exception("Exception in closing client connection %s", str(ex))
            self.client = None
        self.call_home_url = None
        self.call_home_params = {}
        self.token_id = None
        self.is_call_home_set = False
        if self.publish_queue:
            self.publish_queue.queue.clear()
        self.http_server = None
        try:
            log.debug("push service task list %s", self._task_list)
            self.delete_all_push_tasks()
        except Exception as ex:
            log.exception("Exception in revoking tasks %s", str(ex))
        callhome_config_file = os.path.join(self._store_path, CALLHOME_CONFIG_FILE_NAME)
        if os.path.exists(callhome_config_file):
            log.debug("Deleting call home config file %s", callhome_config_file)
            os.remove(callhome_config_file)

    def on_server_message(self, msg):
        log.debug("Received %s message from notification server", msg)

    def on_server_connection_error(self, error_msg):
        log.debug("Error in client connection to the server , %s", error_msg)
        if self.client:
            try:
                self.client.close()
            except Exception as ex:
                log.exception("Exception in closing client socket %s", str(ex))
        self.client = None
        log.debug("Revoking the tasks queued")
        try:
            self.delete_all_push_tasks()
        except Exception as ex:
            log.exception("Exception in revoking tasks on server error connection %s", str(ex))
        log.debug("Since all the url's are unreachable , try getting new set of params from callhome")
        task_id = taskmgr.queue_task(self.get_push_config, interval=5, args=(self.call_home_url, self.token_id, ))
        self.put_push_task(task_id)



    def get_push_config(self, callhome_url=None, token_id=None):
        if callhome_url is None:
            file_name = os.path.join(self._store_path, CALLHOME_CONFIG_FILE_NAME)
            if not os.path.exists(file_name):
                log.warning("Call home Config not Set, cannot proceed!!")
                return False
            else:
                data = file(file_name, "r").read()
                log.debug("Reading call home config %s", file_name)
                callhome_config = json.loads(data)
                callhome_url = callhome_config.get("call_home_url", None)
                token_id = callhome_config.get("token", None)

        if callhome_url is not None:
            self.call_home_url = callhome_url
        else:
            return False

        if token_id is not None:
            self.token_id = token_id
        else:
            return False

        header = {"X-Token-Id": self.token_id}

        self.is_call_home_set = True

        try:
            #ret_val = self.connect_to_callhome_url(callhome_url, header, self.on_server_connection_error)
            ret_val = self.connect_to_callhome_url(callhome_url, header)
            return ret_val
        except Exception as ex:
            log.exception("Error %s while connecting to call home url %s", str(ex), callhome_url)
            self.on_server_connection_error(str(ex))
            return False

    @retry(max_retries=2, timeout=2)
    def connect_to_callhome_url(self, callhome_url, header, on_connection_error=None):
        conn = None
        try:
            conn = Utils.create_http_connection(callhome_url, timeout=self.time_out)
            url = urlparse.urlparse(callhome_url)
            path = url.path
            log.debug("Using Url path %s for notification server for getting IP address", callhome_url)
            conn.request("GET", path, None, header)
            if conn:
                resp = conn.getresponse()
                if resp.status == 200:
                    body = json.loads(resp.read(), "utf-8")
                    log.debug("%s response", body)
                    is_http_server = body.get("http_server", False)
                    if not is_http_server:
                        del self.server_url_list[:]
                        primary_destination = body.get("primary_destination", None)
                        if primary_destination:
                            self.server_url_list.append(primary_destination)
                        log.debug("%s primary destination", body["primary_destination"])
                        secondary_destination = body.get("secondary_destination", None)
                        if secondary_destination:
                            log.debug("Secondary destination %s", secondary_destination)
                            self.server_url_list.append(secondary_destination)
                        if not self.server_url_list:
                            log.debug("No contact url's provided by the server")
                            return False
                    else:
                        self.http_server = body.get("http_server")
                        if not self.http_server:
                            raise ValueError("There is no HTTP server is defined!")
                    # self.server_url_list.append(body.get("primary_destination", None))
                    # self.server_url_list.append(body.get("secondary_destination", None))
                    self.shared_key = body.get("shared_key", None)
                    if not self.shared_key:
                        self.http_server = None
                        self.client = None
                        raise ValueError("Shared key is not defined!")
                    self.app_metrics_collect_freq = body.get("app_metrics_collect_freq", 60)
                    self.app_metrics_publish_freq= body.get("app_metrics_publish_freq", 90)
                    log.debug("Got success response from call home url")
                    log.debug("Metrics collect frequency %s publish frequency %s", self.app_metrics_collect_freq, self.app_metrics_publish_freq)
                    # task_id = taskmgr.queue_task(self.setup_client_connection, args=(self.shared_key, self.server_url_list,))
                    # self.put_push_task(task_id)
                    if not is_http_server:
                        self.setup_client_connection(self.shared_key, self.server_url_list)
                    else:
                        self.create_push_service_tasks(self.app_metrics_collect_freq, self.app_metrics_publish_freq)
                        if StatsCollector.getInstance().enabled:
                            restapi_registry_push = StatsCollector.getInstance().get_statsregistry("PUSHNOT", "event")
                            restapi_registry_push.gauge("app_metrics_collect_freq").set_value(
                                self.app_metrics_collect_freq)
                            restapi_registry_push.gauge("app_metrics_publish_freq").set_value(
                                self.app_metrics_publish_freq)
                    return True
                else:
                    log.warning("Got %s response from call home", resp.status)
                    return False
        except Exception as ex:
            log.exception("Exception in connecting to call home url %s", str(ex))
            if on_connection_error:
                on_connection_error(str(ex))
            raise
        finally:
            if conn:
                conn.close()
        return False

    def setup_client_connection(self, shared_key, server_url_list):
        try:
            token = Utils.get_aes_encrypted_key(shared_key)
            log.debug("Got success response from call home , setting up connection with server")
            header = [('X-Token-Id', token)]
            from websockclientservice import WebSockClientService
            self.client = WebSockClientService(server_url_list, self.on_server_message, self.on_server_connection_error, token=token, headers=header, heartbeat_freq=self.heartbeat_freq)
            if self.client:
                # task_id = taskmgr.queue_task(self.create_push_service_tasks, args=(self.app_metrics_collect_freq, self.app_metrics_publish_freq,))
                # self.put_push_task(task_id)
                self.create_push_service_tasks(self.app_metrics_collect_freq, self.app_metrics_publish_freq)
                if StatsCollector.getInstance().enabled:
                    restapi_registry_push = StatsCollector.getInstance().get_statsregistry("PUSHNOT", "event")
                    restapi_registry_push.gauge("app_metrics_collect_freq").set_value(self.app_metrics_collect_freq)
                    restapi_registry_push.gauge("app_metrics_publish_freq").set_value(self.app_metrics_publish_freq)
                return self.client
            else:
                log.warning("Unable to setup client connection with %s", server_url_list)
                return False
        except Exception as ex:
            log.exception("Error in setting up connection with server %s", str(ex))
            return False

    def push_event_https_server(self, server, shared_key, event):
        """
        This will setup HTTP connection
        :param server:
        :param shared_key:
        :return:
        """
        token = Utils.get_aes_encrypted_key(shared_key)
        conn = Utils.create_http_connection(server, timeout=self.time_out, ssl_context=ssl._create_unverified_context())
        header = {'X-Token-Id': token}
        try:
            if conn:
                conn.request("POST", server, jsonencoder.encode(event), header)
                resp = conn.getresponse()
                if resp.status == 200:
                    log.debug("Pushing the event is successfull!")
                    return True
                else:
                    log.error("Error while pushing the connection to HTTP server: %s"%server)
                    raise Exception("Error while pushing the connection to HTTP server: %s"%server)
            else:
                raise Exception("Unable to setup connection with the server: %s"%server)
        except Exception as ex:
            log.exception("Error while pushing event to http server : %s, cause: %s"%(server, str(ex)))
            raise Exception("Error while pushing event to http server : %s, cause: %s"%(server, str(ex)))
        finally:
            if conn:
                conn.close()

    def create_push_service_tasks(self, app_metrics_collect_freq=30,app_metrics_publish_freq=90):
        log.debug("Setting periodic tasks for collecting at %s interval and publishing at %s interval", app_metrics_collect_freq, app_metrics_publish_freq)
        try:
            task_id = taskmgr.queue_task(self.collect_app_metrics, interval=int(app_metrics_collect_freq), is_periodic=True, is_eager=True, args=())
            log.debug("collect app metrics task name %s", task_id)
            self.put_push_task(task_id)
            task_id = taskmgr.queue_task(self.publish_app_metrics, interval=int(app_metrics_publish_freq), is_periodic=True, is_eager=True, args=())
            log.debug("publish app metrics task name %s", task_id)
            self.put_push_task(task_id)
        except Exception as ex:
            log.exception("Error creating tasks for push service %s", str(ex))
            return False
        return True

    def collect_app_metrics(self):
        from ..runtime.runtime import RuntimeService
        runtime = RuntimeService.getInstance()
        controller = runtime._runtime_context.get_service("app-management")
        log.debug("Collect app metrics event triggered")
        if controller:
            try:
                app_metrics_data = controller.get_app_metrics()
            except Exception as ex:
                log.exception("Caught exception %s while invoking get_app_metrics", str(ex))
                return False
        else:
            return False
        app_resources = app_metrics_data.get("resource_usage", None)
        is_metrics_available = True
        if not app_resources:
            is_metrics_available = False
        if is_metrics_available:
            log.debug("Collect App Metrics %s", app_metrics_data)
            try:
                self.publish_queue.put(app_metrics_data)
            except Queue.Full as ex:
                log.debug("Invoking publish metrics since Queue is full %s", str(ex))
                self.publish_app_metrics()
            except Exception as ex:
                log.exception("Exception while putting data in queue %s", ex)
                return False
        else:
            log.debug("App metrics not available, since no apps are installed")
            return False
        return True

    def publish_data_callback(self, msg):
        log.debug("Publish data callback return val %s", msg)

    def publish_app_metrics(self):
        publish_data = list()
        data = None
        log.debug("Event triggered for publishing app metrics")
        while not self.publish_queue.empty():
            try:
                data = self.publish_queue.get()
            except Exception as ex:
                log.exception("Exception while retrieving data from queue %s", ex)
            """
            Either we can send the json data for every get,
            or append all the data and then send together as a single entity
            """
            if data:
                publish_data.append(data)
                self.publish_queue.task_done()
        #Check if list is empty, if empty do not send any notifications
        if publish_data:
            publish_metrics_event = CAFEvent(event_type=CAFEvent.TYPE_CAF_PUBLISH_APP_METRICS, event_message="publishing app metrics", event_nature=CAFEvent.METRICS, payload=publish_data)
            log.debug("CAF Event %s ", publish_metrics_event)
            log.debug("payload %s", publish_data)
            from ..runtime.runtime import RuntimeService
            ns = RuntimeService.getInstance()._runtime_context.get_service("notification-service")
            if ns:
                ns.post_event(publish_metrics_event)
                log.debug("Posting publish metrics event to notification service")
        else:
            log.debug("Queue is empty, App metrics not collected and posted to the queue")
            return False
        return True

    def get_callhome_config(self):
        return self.call_home_params


if __name__ == "__main__":
    import logging
    import time
    #Some unit tests
    print "Starting push Service.."
    ps = PushService.getInstance()
    ps.start()


