'''
@author: rgopidi
'''
import logging
import os
import tempfile
import threading
import json
from caf_abstractservice import CAFAbstractService
from appfw.utils.infraexceptions import DatastoreLimitError

log = logging.getLogger("runtime.hosting")
SECRET="secret"
VALUE="value"
TOKEN="token"

class DatastoreService(CAFAbstractService):
    """
    A simple service for apps to store and
    retrieve data. Supported operaitons
     - put (key, value, secret), secret is optional
     - get (key, secret)
     - delete(key, token)
    """
    __singleton = None #the one, true singleton
    __singleton_init_done = False

    def __new__(cls, *args, **kwargs):
        # Check to see if a __singleton exists already for this class
        # Compare class types instead of just looking for None so
        # that subclasses will create their own __singleton objects
        if cls != type(cls.__singleton):
            cls.__singleton = super(DatastoreService, cls).__new__(cls, *args, **kwargs)
        return cls.__singleton

    def __init__(self,  params, datastore_num_keys=64, datastore_value_max_len=1024):
        if not self.__singleton_init_done:
            self._store = {}
            self._config_file = params.config_file
            self._config = params.config
            self.name = params.name
            self._lock = threading.Lock()
            self.__singleton_init_done = True
            self._isRunning = False
            self._max_len = datastore_value_max_len
            self._num_keys = datastore_num_keys

    def set_config(self, config):
        self._config = config

    def get_config(self):
        return self._config

    def validate_config(self, config):
        return True

    def is_running(self):
        return self._isRunning

    def start(self):
        self._isRunning = True
        return True

    def stop(self):
        self._isRunning = False
        self._cleanupDatastore()

    def put(self, key, value, token, secret=None):

        if key not in self._store and len(self._store) >= self._num_keys:
            raise DatastoreLimitError('IOx data store limit %s exceeded' %self._num_keys)

        entry = {}
        self._lock.acquire()
        if key in self._store:
            entry = self._store[key]
            if entry is not None:
                if entry[TOKEN] is not None and entry[TOKEN] != token:
                    log.debug("tokens dont match for key %s, token %s, entry[TOKEN] %s" %(key, token, entry[TOKEN]))
                    self._lock.release()
                    return False
            else:
                entry = {}

        if isinstance(value, basestring):
            if len(value) > self._max_len:
                log.debug("value %s for key %s is greater than the size limit. Trimming it to len %s" %(value, key, self._max_len))
                value = value[:self._max_len]
        entry[VALUE] = value
        entry[TOKEN]= token
        entry[SECRET] = secret
        self._store[key] = entry
        self._lock.release()
        return True

    def get(self, key, secret=None):
        self._lock.acquire()
        if key in self._store:
            entry = self._store[key]
        else:
            entry = None
        self._lock.release()

        if entry is None:
            log.debug("no entry found for key %s" %key)
            return None

        if entry[SECRET] is not None:
            if secret is None or secret != entry[SECRET]:
                log.debug("Secrets don't match for key %s secret %s entry[SECRET]%s"%(key, secret, entry[SECRET]))
                return None
        return entry[VALUE]

    def get_internal(self, key):
        '''
        An internal routine to retrieve keys
        with out specifying the secret
        '''
        self._lock.acquire()
        if key in self._store:
            entry = self._store[key]
        else:
            entry = None
        self._lock.release()

        if entry is None:
            return None
        else:
            return enty[VALUE]

    def delete(self, key, token):
        ret = False
        self._lock.acquire()
        if key in self._store:
            entry = self._store[key]
            if entry is not None and entry[TOKEN] == token:
                del self._store[key]
                ret = True
            elif entry is not None:
                log.debug("tokens dont match for key %s token %s entry[TOKEN] %s" %(key, token, entry[TOKEN]))

        self._lock.release()

        return ret

    def _cleanupDatastore(self):
        self._lock.acquire()
        self._store.clear()
        self._lock.release()

    def delete_keys_of_token(self, token):
        #TODO make it more efficient interms of execution time
        self._lock.acquire()
        for key, entry in self._store.items():
            if entry[TOKEN] == token:
                del self._store[key]
        self._lock.release()

    @classmethod
    def getInstance(cls, *args):
        if cls.__singleton == None:
            cls.__singleton = DatastoreService(*args)
        return cls.__singleton
