'''
Created on Sep 29, 2012

@author: rnethi
'''

import logging
from ..utils.infraexceptions import *
from ..hosting.apptypes import AppType


log = logging.getLogger("runtime")

class HostingRegistry(object):
    """
    Executor is responsible for providing a connector execution environment. 
    It interacts with the connector hosting infra i.e., container management 
    on behalf of controller and orchestrates the creation of container, deploying 
    the connector package and starting/stopping the containers.
    """
    _containerMgr = None
    __singleton = None # the one, true Singleton

    def __new__(cls, *args, **kwargs):
        # Check to see if a __singleton exists already for this class
        # Compare class types instead of just looking for None so
        # that subclasses will create their own __singleton objects
        if cls != type(cls.__singleton):
        #if not cls.__singleton:
            cls.__singleton = super(HostingRegistry, cls).__new__(cls, *args, **kwargs)
        return cls.__singleton

    def __init__(self, config, language_runtimes, runtime_context, supported_apptypes=[]):
        '''
        Donot sub-class this class. If you do, because of singleton code above, init
        would be called multiple times
        '''
        self._config = config
        self._language_runtimes = language_runtimes
        self._runtime_context = runtime_context
        self._supported_apptypes = supported_apptypes
        self._hosting_map = None
        # Mapping between apptype -> (stager, container)
        self._app_registry = dict()
        self._service_registry = dict()
        # class registry. So that we maintain created instances of stagers and containers
        self._class_registry = dict()

        # Create app registry
        self._construct_app_registry()
        # Create service registry
        self._construct_service_registry()

    def _get_cls_from_string(self, s):
        if s == "docker-container":
            from ..hosting.dockercontainer import DockerContainerManager
            return DockerContainerManager

        if s == "lxc-container":
            from ..hosting.libvirt.lxc.lxccontainer import LXCContainerManager
            return LXCContainerManager

        if s == "kvm-container":
            from ..hosting.libvirt.kvm.kvmcontainer import KVMContainerManager
            return KVMContainerManager

        if s == "vm-container":
            from ..hosting.vman import VmanContainerManager
            return VmanContainerManager

        if s == "process-container":
            from ..hosting.process import ProcessContainerManager
            return ProcessContainerManager

        if s == "lxc-stager":
            from ..staging.lxcstager import LXCStager
            return LXCStager

        if s == "kvm-stager":
            from ..staging.kvmstager import KVMStager
            return KVMStager

        if s == "docker-stager":
            from ..staging.dockerstager import DockerStager
            return DockerStager

        if s == "vm-stager":
            from ..staging.vmstager import VmStager
            return VmStager

        if s == "paas-stager":
            from ..staging.stager import Stager
            return Stager

        if s == "native-docker-stager":
            from ..staging.nativedockerstager import NativeDockerStager
            return NativeDockerStager

        return None


    def _get_mapping_from_config(self, apptype, section):
        """Returns (Stager, ContainerManager) from entry specified in config file"""
        if self._config.has_section(section):
            if self._config.has_option(section, apptype):
                l = self._config.get(section, apptype)
                if l:
                    s, c = l.split(",")
                    s = s.lstrip().rstrip()
                    c = c.lstrip().rstrip()
                    # Return appropriate Stager, ContainerManager classes
                    stgr = self._get_cls_from_string(s)
                    cmgr = self._get_cls_from_string(c)
                    return stgr, cmgr
        return None

    def _get_mapping_str_from_config(self, apptype, section):
        """Returns (Stager, ContainerManager) from entry specified in config file"""
        if self._config.has_section(section):
            if self._config.has_option(section, apptype):
                l = self._config.get(section, apptype)
                if l:
                    s, c = l.split(",")
                    s = s.lstrip().rstrip()
                    c = c.lstrip().rstrip()
                    return s, c
        return None

    def _get_mapping_from_app_registry_str(self, apptype, is_service=False):
        if is_service:
            return self.getServiceStager(apptype).__class__.__name__,self.getServiceContainerManager(apptype).__class__.__name__
        else:
            return self.getAppStager(apptype).__class__.__name__, self.getAppContainerManager(apptype).__class__.__name__


    def _get_mapping_from_code(self, apptype):
        if apptype == AppType.VM:
            stgr = self._get_cls_from_string("vm-stager")
            cmgr = self._get_cls_from_string("vm-container")

        elif apptype == AppType.DOCKER:
            stgr = self._get_cls_from_string("docker-stager")
            cmgr = self._get_cls_from_string("docker-container")

        elif apptype == AppType.PAAS:
            stgr = self._get_cls_from_string("paas-stager")
            cmgr = self._get_cls_from_string("process-container")

        elif apptype == AppType.LXC:
            stgr = self._get_cls_from_string("lxc-stager")
            cmgr = self._get_cls_from_string("lxc-container")

        else:
            return None

        return stgr, cmgr

    def _construct_app_registry(self):
        log.debug("Constructing hosting map for app")

        # Construct the hosting registry
        for apptype in self._supported_apptypes:

            # First check if there is a mapping specified in config file
            sctuple = self._get_mapping_from_config(apptype, "app-hosting-map")
            if sctuple:
                log.debug("Mapping entry found for apptype %s", apptype)
            else:
                log.debug("No Mapping entry found for apptype %s", apptype)
                log.debug("Will find out from internal implementation")
                sctuple = self._get_mapping_from_code(apptype)

            if sctuple is None:
                log.error("No Stager/Container implementation found for apptype %s", apptype)
                continue

            stgr, cmgr = sctuple
            if stgr and cmgr:
                # Add an entry to registry
                self._app_registry[apptype] = (
                        stgr.getInstance(self._config, self._language_runtimes),
                        cmgr.getInstance(self._config, self._language_runtimes, self._runtime_context))
            else:
                log.error("Invalid configuration for apptype %s: %s. Skipping it..", apptype, sctuple)

        log.info("Created HostingRegistry for app: \n%s", str(self._app_registry))

    def _construct_service_registry(self):
        log.debug("Constructing hosting map for service")

        # Construct the hosting registry
        for apptype in self._supported_apptypes:

            # First check if there is a mapping specified in config file
            sctuple = self._get_mapping_from_config(apptype, "service-hosting-map")
            if sctuple:
                log.debug("Mapping entry found for apptype %s", apptype)
            else:
                log.debug("No Mapping entry found for apptype %s", apptype)
                log.debug("Will find out from internal implementation")
                sctuple = self._get_mapping_from_code(apptype)

            if sctuple is None:
                log.error("No Stager/Container implementation found for apptype %s", apptype)
                continue

            stgr, cmgr = sctuple
            if stgr and cmgr:
                # Add an entry to registry
                self._service_registry[apptype] = (
                     stgr.getInstance(self._config, self._language_runtimes),
                     cmgr.getInstance(self._config, self._language_runtimes, self._runtime_context))
            else:
                log.error("Invalid configuration for apptype %s: %s. Skipping it..", apptype, sctuple)

        log.info("Created HostingRegistry for service: \n%s", str(self._service_registry))

    def setRegistry(self, appType, stgr, cmgr):

        if stgr and cmgr:
            self._app_registry[appType] = (
                stgr.getInstance(self._config, self._language_runtimes),
                cmgr.getInstance(self._config, self._language_runtimes,
                                 self._runtime_context))
        else:
            log.error("Invalid Request to set stager %s, container type %s \
            dynamically", stgr, cmgr)
        log.info("Set HostingRegistry dynamically : \
            \n%s", str(self._app_registry))

    def listContainerManagers(self):
        """
        Return a list of available and instantiated container managers
        """
        rval = set() 
        for apptype in self._app_registry:
            rval.add(self._app_registry[apptype][1])
        for apptype in self._service_registry:
            rval.add(self._service_registry[apptype][1])
        return rval


    def getAppContainerManager(self, apptype):
        """
        Returns a app container manager instance for a given app type
        """
        if apptype in self._app_registry:
            return self._app_registry[apptype][1]

        raise AppTypeNotSupportedError("App type : %s, is not supported!" % apptype)

    def getServiceContainerManager(self, apptype):
        """
        Returns a service container manager instance for a given app type
        """
        if apptype in self._service_registry:
            return self._service_registry[apptype][1]

        raise AppTypeNotSupportedError("Service type : %s, is not supported!" % apptype)

    def getAppStager(self, apptype):
        """
        Returns a stager instance for a given app type
        """
        if apptype in self._app_registry:
            return self._app_registry[apptype][0]

        raise NotImplementedError("Following app type not supported: %s. Please uninstall the current instance" % apptype)

    def getServiceStager(self, apptype):
        """
        Returns a stager instance for a given app type
        """
        if apptype in self._service_registry:
            return self._service_registry[apptype][0]

        raise NotImplementedError("Following app type not supported: %s. Please uninstall the current instance" % apptype)

    def getAppContainerManager_str(self, apptype):
        """
        Returns a app container manager string for a given app type
        """
        stager, comtainermgr = self._get_mapping_from_app_registry_str(apptype)
        return comtainermgr

    def getServiceContainerManager_str(self, apptype):
        """
        Returns a service container manager string for a given app type
        """
        stager, comtainermgr = self._get_mapping_from_app_registry_str(apptype, is_service=True)
        return comtainermgr
