#! /bin/ash
#
# Available PD hooks and inputs for docker type application lifecycle
# Environment variables defined in app yaml are also accessible from this PD script
#
# Return codes:
# Only least significant 8 bits of the return code is retained (POSIX convention)
# eg. return code -1 is interpreted as 255
# Non-zero return codes < 127 are reported as errors in CAF log, but CAF operation won't terminate
# Return codes >=128 are considered fatal and will cause CAF to terminate the lifecycle operation
#
export PATH=/bin:/sbin:/usr/bin:/var/wireless:/usr/sbin:/opt/cisco/bin
CAF_REPO=/storage/software/caf/work/repo

final_firmware_status_check() {
    FILE=/storage/iot_status_$2.txt
    if [[ -f "$FILE" ]]; then
        status=$(awk '{for (I=1;I<=NF;I++) if ($I == "firmware") \
            {print $(I+1)};} ; {for (I=1;I<=NF;I++) \
            if ($I == "status") {print $(I+1)};}' $FILE)
        status=$(echo $status | tr " " "\n")
        i=1
        for j in $status; do
            if [[ "$i" == 1 ]]; then
                firmware_on_chip=$j
                let i=i+1
            else
                firmware_status=$j
            fi
        done
        if [ "$firmware_on_chip" == "$1" ] && [ "$firmware_status" == "success" ]; then
            return 1
        else
            return 0
        fi
    else
        return 0
    fi
}

is_valid_alias() {
    if [ "$1" == "ble" ] || [ "$1" == "thread" ] || [ "$1" == "zigbee" ] ; then
        return 0
    else
        return 1
    fi
}

firmware_download_process() {
    alias=`grep "alias*:" $app_repo_dir/package.yaml | awk -F ": " '{ print $2 }' \
           | sed 's/^[ \t]*//;s/[ \t]*$//'`
    if [ -z "$alias" ] ; then
        return 0
    elif is_valid_alias $alias ; then
        serial_device=`python /opt/lxc/scripts/iox_app_get_serial_device.py $alias`
        if [ "$serial_device" == "False" ] ; then
            echo "Requested serial device does not exist"
            return 255
        else
            process="iox_$app_id"
            #If BLE management is enabled, disable BLE management from CMX and sendout the reason code to wlc.
            ble_management_device=`wcp_read ble_drv.show_ble_summary | grep "^Active BLE" | awk -F ": " '{ print $2 }'`
            device=${serial_device#"/dev/"}
            alias=${alias#"external_"}
            /usr/bin/cli_scripts/iot_fw_dnld.sh $alias $device $process
            if [ "$serial_device" == "$ble_management_device" ] ; then
                if [ "$alias" == "zigbee" ] ; then
                    final_firmware_status_check $alias $device
                    if [ "$?" -eq 1 ]; then
                        echo "Notify eWLC that IOx app is using ZigBee Firmware"
                        wcp_write ble_drv.admin_cfg_cmd 8
                    fi
                elif [ "$alias" == "ble" ] ; then
                    final_firmware_status_check $alias $device
                    if [ "$?" -eq 1 ]; then
                        echo "Notify eWLC that IOx app is using BLE Firmware"
                        wcp_write ble_drv.admin_cfg_cmd 9
                    fi
                fi
            fi
            if [ "$1" == "pre-start" ] ; then
                if final_firmware_status_check $alias $device ; then
                    echo -n $alias | tr [a-z] [A-Z] ; echo " firmware download failed"
                    exit 255
                fi
            fi
        fi
    else
        echo "Incorrect alias value"
        return 255
    fi
    return 0
}

cleanup_process() {
    alias=`grep "alias*:" $app_repo_dir/package.yaml | awk -F ": " '{ print $2 }' \
           | sed 's/^[ \t]*//;s/[ \t]*$//'`
    if [ -z "$alias" ] ; then
        exit 0
    elif is_valid_alias $alias ; then
        serial_device=`python /opt/lxc/scripts/iox_app_get_serial_device.py $alias`
        if [ "$serial_device" == "False" ] ; then
            exit 0
        else
            filename="/storage/iot_status_${serial_device#"/dev/"}.txt"
            echo "deleting file $filename"
            if [ -f $filename ]; then
                rm $filename
            fi
            ble_management_device=`wcp_read ble_drv.show_ble_summary | grep "^Active BLE" | awk -F ": " '{ print $2 }'`
            echo $ble_management_device
            if [ "$serial_device" == "$ble_management_device" ] ; then
                wcp_write ble_drv.admin_cfg_cmd 5
            fi
        fi
    fi
}

handle_post_activate() {
    alias=""
    firmware_download_process "post-activate"
    ret=$?
    if [ ! -z $alias ] && is_valid_alias $alias;
    then
        send_ipc grpc iox_app_status $appid ACTIVATED $alias
    else
        send_ipc grpc iox_app_status $appid ACTIVATED
    fi
    exit $ret
}

handle_pre_deploy() {
    exit 0
}

handle_pre_activate() {
    exit 0
}

handle_post_deploy() {
    send_ipc grpc iox_app_status $appid DEPLOYED
    firmware_download_process "post-deploy"
    exit $?
}

handle_pre_start() {
    firmware_download_process "pre-start"
    exit $?
}

handle_post_start() {
    send_ipc grpc iox_app_status $appid RUNNING
    exit 0
}

handle_pre_stop() {
    exit 0
}

handle_post_stop() {
    send_ipc grpc iox_app_status $appid STOPPED
    exit 0
}

handle_pre_deactivate() {
    exit 0
}

handle_post_deactivate() {
    send_ipc grpc iox_app_status $appid DEACTIVATED
    cleanup_process
    exit 0
}

handle_pre_uninstall() {
    exit 0
}

handle_post_uninstall() {
    send_ipc grpc iox_app_status $appid UNDEPLOYED
    exit 0
}

appid=$2
case "$1" in
    pre-deploy)
        # $2: <app_id>
        # $3: <app_repo_dir>
        # Call function here to handle PD pre-deploy action
        handle_pre_deploy
        ;;
    post-deploy)
        # $2: <app_id>
        # $3: <app_repo_dir>
        # $4: <app_extracted_dir>
        # Call function here to handle PD post-deploy action
        app_id=$2
        app_repo_dir=$3
        handle_post_deploy
        ;;
    pre-activate)
        # $2: <app_id>
        # Call function here to handle PD pre-activate action
        handle_pre_activate
        ;;
    post-activate)
        # $2: <app_id>
        # Call function here to handle PD post-activate action
        app_id=$2
        app_repo_dir="$CAF_REPO/$2"
        shift 3
        /opt/lxc/scripts/iox-app-port-setup.py activate $app_id "$*"
        handle_post_activate
        ;;
    pre-start)
        # $2: <app_id>
        # Call function here to handle PD pre-start action
        app_id=$2
        app_repo_dir="$CAF_REPO/$2"
        handle_pre_start
        ;;
    post-start)
        # $2: <app_id>
        # Call function here to handle PD post-start action
        handle_post_start
        ;;
    pre-stop)
        # $2: <app_id>
        # Call function here to handle PD pre-stop action
        handle_pre_stop
        ;;
    post-stop)
        # $2: <app_id>
        # Call function here to handle PD post-stop action
        handle_post_stop
        ;;
    pre-deactivate)
        # $2: <app_id>
        # $3: <rootfs_dir>
        # Call function here to handle PD pre-deactivate action
        handle_pre_deactivate
        ;;
    post-deactivate)
        # $2: <app_id>
        # Call function here to handle PD post-deactivate action
        app_id=$2
        app_repo_dir="$CAF_REPO/$2"
        /opt/lxc/scripts/iox-app-port-setup.py deactivate $app_id
        handle_post_deactivate
        ;;
    pre-uninstall)
        # $2: <app_id>
        # Call function here to handle PD pre-uninstall action
        handle_pre_uninstall
        ;;
    post-uninstall)
        # $2: <app_id>
        # Call function here to handle PD post-uninstall action
        handle_post_uninstall
        ;;
    *)
        # Do nothing
        exit 0
	    ;;
esac
exit 0
