__author__ = 'suressan'

import logging
import os
import time
import signal
import shutil
import threading
from appfw.utils.commandwrappers import *
from ..utils.infraexceptions import MandatoryDataMissingError
from appfw.api.systeminfo import SystemInfo

log = logging.getLogger("runtime")
'''
Helper class to import, verify, regenerate and retrieve ssl
certificate/key pair.
'''


class CertHelper(object):
    __singleton = None

    def __new__(cls, *args, **kwargs):
        # Check to see if a __singleton exists already for this class
        # Compare class types instead of just looking for None so
        # that subclasses will create their own __singleton objects
        if cls != type(cls.__singleton):
            # if not cls.__singleton:
            cls.__singleton = super(CertHelper, cls).__new__(
                cls, *args, **kwargs)
        return cls.__singleton

    def __init__(self, config=None):
        if not config:
            from apiservice import APIService
            self.config = APIService.instance.config
        else:
            self.config = config
        self._lock = threading.Lock()
        self.sslcertificate = self.config.get("api", "sslcertificate")
        self.sslPrivateKey = self.config.get("api", "sslprivatekey")
        # Update the ssl cirt path if secure storage is enabled
        if SystemInfo.is_secure_storage_supported(
        ) and SystemInfo.is_secure_storage_server_up_and_available():
            encrypted_dir = self.config.get("secure_storage_server",
                                            "encrypted_dir")
            self.sslcertificate = os.path.join(encrypted_dir, "certs",
                                               os.path.basename(
                                                   self.sslcertificate))
            self.sslPrivateKey = os.path.join(encrypted_dir, "certs",
                                              os.path.basename(
                                                  self.sslPrivateKey))
        self.ssldir = os.path.dirname(self.sslcertificate)
        self.sslconf = os.path.join(self.ssldir, "openssl.cnf")

    def generate_default_cert(self):
        openssl_cnf = """[req]
        distinguished_name = req_distinguished_name

        [ req_distinguished_name ]
        # Variable name   Prompt string
        #----------------------   ----------------------------------
        unstructuredName = unstructured Name"""

        os.remove(self.sslcertificate)
        with open(self.sslconf, 'w', 0) as f:
            f.write(openssl_cnf)

        out, rc = openssl("req", "-config", self.sslconf, "-x509", "-days",
                          "3650", "-new", "-key", self.sslPrivateKey, "-out",
                          self.sslcertificate, "-subj",
                          "/unstructuredName=0.0.0.0")
        if rc != 0:
            raise Exception(
                "Failed to generate default self signed certificate : %s" %
                out)

    def generate_default_priv_key(self):
        os.remove(self.sslPrivateKey)
        # Generate default private key
        out, rc = openssl("genrsa", "-out", self.sslPrivateKey, "2048")
        if rc != 0:
            raise Exception(
                "Failed to generate default private key : %s" % out)

    def update_ssl_cert(self, cert):
        os.remove(self.sslcertificate)
        shutil.move(cert, self.sslcertificate)

    def update_ssl_key(self, key):
        os.remove(self.sslPrivateKey)
        shutil.move(key, self.sslPrivateKey)

    def update_ssl_crypto(self, cert, key):
        try:
            self._lock.acquire()

            if cert == "":
                raise MandatoryDataMissingError("Missing certificate content")

            self.update_ssl_cert(cert)

            if key == "":
                pass
            else:
                self.update_ssl_key(key)

        finally:
            self._lock.release()

    def setup_default_crypto(self):
        try:
            self._lock.acquire()

            self.generate_default_priv_key()
            self.generate_default_cert()
        finally:
            self._lock.release()

    def stop_caf(self):
        def send_stopsig_caf():
            try:
                log.debug("CAF will be shutdown in 10 seconds")
                time.sleep(10)
                os.kill(os.getpid(), signal.SIGINT)
            except Exception as ex:
                log.exception("Failed to stop caf - %s", str(ex))

        t = threading.Thread(target=send_stopsig_caf)
        t.setDaemon(True)
        t.start()

    def verifyCertificate(self, cert):
        out, rc = openssl("x509", "-in", cert, "-text", "-noout")
        return out, rc

    def getSubject(self, cert):
        out, rc = openssl("x509", "-in", cert, "-subject", "-noout")
        if rc != 0:
            log.error("Error in getting subject of cert: %s" % out)
            raise Exception("Error in getting subject of cert: %s" % out) 
        else:
            log.debug("Certificate Subject name output: %s" % out)
            subject = out.split("subject=")[1].strip()
            log.debug("Certificate Subject name: %s" % subject)
            return subject 

    def getIssuer(self, cert):
        out, rc = openssl("x509", "-in", cert, "-issuer", "-noout")
        if rc != 0:
            log.error("Error in getting issuer of cert: %s" % out)
            raise Exception("Error in getting issuer of cert: %s" % out)
        else:
            log.debug("Certificate Issuer name output: %s" % out)
            issuer = out.split("issuer=")[1].strip()
            log.debug("Certificate Issuer name: %s" % issuer)
            return issuer 

    def verifyKey(self, key):
        # verify the format of privatekey.pem
        out, rc = openssl("rsa", "-in", key, "-check")
        return out, rc

    def isSelfSignedCert(self, cert):
        """
        Returns True if the cert is self signed otherwise False
        """
        subject = self.getSubject(cert)
        issuer = self.getIssuer(cert)
        if subject == issuer:
            return True
        return False

    def verifyCertKeyMatch(self, cert, key):
        try:
            certOut, rc = openssl("x509", "-noout", "-modulus", "-in", cert)
            if rc != 0:
                raise Exception(
                    "Failed to generate md5 of ssl certificate : %s" % keyOut)

            if key == "":
                key = self.sslPrivateKey

            keyOut, rc = openssl("rsa", "-noout", "-modulus", "-in", key)
            if rc != 0:
                raise Exception(
                    "Failed to generate md5 of ssl private key : %s" % keyOut)

            if certOut != keyOut:
                return False

            return True

        except Exception as ex:
            log.exception("Failed to generate md5 digest of ssl certificate and"
                      "private key.")
            raise ex

    def getCertificate(self):
        cert = ""
        if os.path.exists(self.sslcertificate):
            try:
                with open(self.sslcertificate) as file_hdlr:
                    cert = file_hdlr.read()
                return cert
            except IOError:
                log.exception("Failed to read ssl certificate at %s.",
                          self.sslcertificate)
                raise IOError
        return cert

    @classmethod
    def getInstance(cls, *args):
        '''
        Returns a singleton instance of the class
        '''
        if not cls.__singleton:
            cls.__singleton = CertHelper(*args)
        return cls.__singleton
