__author__ = 'utandon'

import logging
import tempfile
import os
import falcon
import json
import re

from jsonencoder import JSONEncoder
from apiservice import ResourceRoute, APIService
from common import AuthenticatedResource, make_response, make_error_response
from ..utils.cafevent import CAFEvent
from ..utils.infraexceptions import *

log = logging.getLogger("runtime.api.resources")

jsonencoder = JSONEncoder()

@ResourceRoute("/platform/devices", endpoint="devices")
class DevicesResource(AuthenticatedResource):

    def on_get(self, request, response):
        device_manager = APIService.instance.device_manager
        deviceInfo = device_manager.list_devices()
        out = jsonencoder.encode(deviceInfo)
        response.body = out
        response.status = falcon.HTTP_200
        response.set_headers({'Content-Type': "application/json",
                              'Cache-Control': "no-cache"})

@ResourceRoute("/platform/devices/{device_type}", endpoint="type")
class DevicesResource(AuthenticatedResource):

    def on_get(self, request, response, device_type):
        device_manager = APIService.instance.device_manager
        deviceInfo = device_manager.list_devices(device_type)
        out = jsonencoder.encode(deviceInfo)
        response.body = out
        response.status = falcon.HTTP_200
        response.set_headers({'Content-Type': "application/json",
                              'Cache-Control': "no-cache"})


@ResourceRoute("/platform/devices/{device_type}/{device_name}", endpoint="device_name")
class DeviceResource(AuthenticatedResource):

    def on_get(self, request, response, device_type, device_name):
        
        device_manager = APIService.instance.device_manager
        device = device_manager.get_device_by_name(device_type, device_name)
        if device is None:
            response = make_error_response(response,
                                            "The device of type %s: %s, does not exist" % (device_type, device_name),
                                            "The device of type %s: %s, does not exist" % (device_type, device_name),
                                            falcon.HTTP_404)
            return

        response.status = falcon.HTTP_200
        response.body = jsonencoder.encode(device.serialize())
        response.set_headers({'Content-Type': "application/json",
                              'Cache-Control': "no-cache"})


@ResourceRoute("/platform/device_config", endpoint="device_config")
class DeviceConfigResource(AuthenticatedResource):

    def on_get(self, request, response):
        device_manager = APIService.instance.device_manager
        config = device_manager.config
        response.status = falcon.HTTP_200
        response.body = jsonencoder.encode(config)
        response.set_headers({'Content-Type': "application/json",
                              'Cache-Control': "no-cache"})
