__author__ = 'hvishwanath'

import logging
import json
import os
import io
import shutil
import tempfile
import libvirt
import tarfile
import glob
import time
import subprocess
import cStringIO
import copy
import signal
import psutil
from lxcxmlutils import LXCLibvirtXMLGenerator
from ..libvirtcontainer import LibvirtContainer, LibvirtContainerManager
from appfw.hosting.container_utils import ContainerUtils
from appfw.hosting.apptypes import AppType
from appfw.hosting.filemgmt import FileMgmt
from appfw.utils.infraexceptions import *
from appfw.utils.filegen import filegen
from appfw.utils.utils import Utils, APP_CONFIG_NAME_2, USER_EXTRACTED_DIR, LAYER_CONTENTS_DIR, LAYER_ARCHIVE_FILE
from appfw.utils.docker_utils import DockerUtils, DOCKER_MANIFEST_FILE_NAME, DOCKER_LAYERS_MANIFEST_FILE
from appfw.utils.commandwrappers import *
from appfw.cartridge.cartridge import *
from appfw.runtime.platformcapabilities import PlatformCapabilities
from appfw.paas_utils.rfs_composer import RFSComposer
from appfw.paas_utils.rootaufs_composer import AUFS_Composer
from appfw.paas_utils import *
from appfw.api.systeminfo import SystemInfo
from appfw.runtime.hostingmgmt import HostingManager
from appfw.utils.pipecommand_timeout import PipeCommand

log = logging.getLogger("runtime.hosting")
#Defaul rw layer size for docker style app is 10 MB
DEFAULT_RW_LAYER_SIZE = 10
DEFAULT_RW_LAYER_TO_CODE_RATIO = 10
DOCKER_RW_LAYER_ROOT = "ulayer"
DOCKER_WORK_DIR = "workdir"
LIBVIRT_VERSION_SUPPORT_DOKCER_COMMANDS=3008000
DOCKER_RW_LAYER_ROOTFS_TYPE = ("img","ext2","ext4")

class LXCContainer(LibvirtContainer):
    """
    Base class for container instances. An instance of of type AbstractContainer represents
    a single logical container that could host a single connector
    """

    def __init__(self, appid, domain_xml, libvirt_client, apptype, connection_str,
                 resources={}, dst_rootfs=None,app_ext2_src=None,
                 data_ext_src=None, cartridge_list=None, network_info=None, libvirt_network_list=None,
                 device_list=None, app_env={}, reconcile=False, use_ext4=False,
                 sec_attr=None, non_composed_rootfs=False, os_mode=False, mount_points = []):
        self.mem_limit_mb = 0
        self.cpu_shares = 0
        self.disk = 0
        self._resources = resources
        self._get_resource_requirements()
        self._dst_rootfs=dst_rootfs
        self._app_ext2_src=app_ext2_src
        self._data_ext_src=data_ext_src
        self._use_ext4=use_ext4
        self._cartridge_list=cartridge_list
        self._app_env = app_env
        self._sec_attr = sec_attr
        self._mount_points = mount_points

        self._seclabel = False

        if self.sec_attr and self.sec_attr.get("lsm", False) and self.sec_attr.get("lsm").get("enabled", False):
            self._seclabel = True

        self._os_mode = os_mode

        if device_list and not reconcile:
            self._hm = HostingManager.get_instance()
            self._dm = self._hm.get_service("device-management")
            for dev in device_list:
                self._dm.app_setup_hook(apptype, dev.get("type"), dev.get("device-id"))

        LibvirtContainer.__init__(self, appid, domain_xml, "lxc", apptype, connection_str, libvirt_client, network_info=network_info,
                                  libvirt_network_list=libvirt_network_list, device_list=device_list, reconcile=reconcile,
                                  non_composed_rootfs=non_composed_rootfs)

    def __str__(self):
        return "LXC"

    @property
    def app_resources(self):
        return self._resources

    @property
    def dst_rootfs(self):
        return self._dst_rootfs

    @property
    def app_ext2_src(self):
        return self._app_ext2_src

    @property
    def data_ext_src(self):
        return self._data_ext_src

    @property
    def cartridge_list(self):
        return self._cartridge_list

    @property
    def app_env(self):
        return self._app_env

    @property
    def sec_attr(self):
        return self._sec_attr

    @property
    def seclabel(self):
        return self._seclabel

    @property
    def mount_points(self):
        return self._mount_points

    @mount_points.setter
    def set_mount_points(self, mnt_pt):
        if isinstance(mnt_pt, list):
            self._mount_points.extend(mnt_pt)
        elif mnt_pt:
            self._mount_points.append(mnt_pt)

    def __repr__(self):
        return "LXC Container : Name: %s, UUID: %s" % (self.domain.name(), self.domain.UUIDString())

    def __str__(self):
        return self.__repr__()

    def _get_resource_requirements(self):
        self.mem_limit_mb = self._resources.get("memory", 10)
        self.cpu_shares = self._resources.get("cpu", 10)
        self.disk = self._resources.get("disk", 10)

        log.debug("Requested resources : Memory : %s, CPU Shares : %s disk space : %s" % (
            self.mem_limit_mb, self.cpu_shares, self.disk))

    def start(self):
        super(LXCContainer, self).start()
        # Need to setup the network for the docker style app
        if not self._os_mode and self._apptype == AppType.DOCKER:
            self.handle_docker_networking("setup")

    def terminate(self):
        super(LXCContainer, self).terminate()

    def stop(self, graceful=True):
        if not self._os_mode and self._apptype == AppType.DOCKER:
            self.handle_docker_networking("teardown")
        super(LXCContainer, self).stop(graceful=graceful)

    def handle_docker_networking(self, action):
        """
        @param action:
        @return:
        """
        log.debug("handling the docker networking : action : %s"%action)
        sec_attr = None
        hm = HostingManager.get_instance()
        sc = hm.get_service("security-management")
        if sc:
            sec_attr = sc.get_app_security_config(self.appid)
        hooks_dir = Utils.getPDHooksFolder()
        nw_setup_script = Utils.getSystemConfigValue("docker-container", "nw_setup_script", "docker_network_udhcpc.sh")
        nw_setup_script = os.path.join(hooks_dir, nw_setup_script)
        log.debug("Docker network setup script is %s"%nw_setup_script)
        dhcp_client_custom_script = Utils.getSystemConfigValue("docker-container", "dhcp_client_custom_script", "udhcpc_custom.sh")
        dhcp_client_custom_script = os.path.join(hooks_dir, dhcp_client_custom_script)
        log.debug("DHCP client custom script is %s"%dhcp_client_custom_script)
        network_teardown_script = Utils.getSystemConfigValue("docker-container", "dhcp_client_teardown_script", "")
        if network_teardown_script and network_teardown_script != "":
            network_teardown_script = os.path.join(hooks_dir, network_teardown_script)
        else:
            network_teardown_script = ""
        repo_dir = Utils.getSystemConfigValue("controller", "repo", "")
        if os.path.isdir(os.path.join(repo_dir, self.app_id)):
            pid_dir = os.path.join(repo_dir, self.app_id, ".dhcp_repo")
            if not os.path.isdir(pid_dir):
                log.debug("DHCP client client pid repo %s is not there, so creating it!"%pid_dir)
                os.mkdir(pid_dir)
            else:
                log.debug("DHCP client client pid repo %s is already there!"%pid_dir)
        else:
            log.error("Repo for the app is not defined!")
            raise Exception("Repo for the app is not defined!")
        repodir = LXCContainerManager.getInstance().rootPath
        ipv6_supported = LXCContainerManager.getInstance().ipv6_supported
        container_dir = os.path.join(repodir, self.app_id)
        target_rootfs_dir = os.path.join(container_dir, "rootfs_mnt")
        etc_dir = os.path.join(target_rootfs_dir, "etc")
        if not os.path.isdir(etc_dir):
            os.mkdir(etc_dir)
        r_conf = os.path.join(target_rootfs_dir, etc_dir, "resolv.conf")
        if os.path.islink(r_conf):
            log.info("Resolve conf file %s is found in rootfs, so removing it"%r_conf)
            os.remove(r_conf)
        netns_dir = Utils.getSystemConfigValue("docker-container", "netns_dir", "/var/run/netns")
        if not os.path.isdir(netns_dir):
            log.debug("NETNS dir %s, is not found so creating it"%netns_dir)
            os.mkdir(netns_dir)
        if action == "setup":
            self._docker_teardown_network(network_teardown_script, pid_dir, netns_dir, r_conf, dhcp_client_custom_script, ipv6_supported)
            try:
                self._docker_setup_network(nw_setup_script, pid_dir, netns_dir, r_conf, dhcp_client_custom_script, ipv6_supported, sec_attr)
            except Exception as e:
                if self.isRunning():
                    self.stop()
                raise e
        elif action == "teardown":
            try:
                self._docker_teardown_network(network_teardown_script, pid_dir, netns_dir, r_conf, dhcp_client_custom_script, ipv6_supported)
            except Exception as e:
                raise e
        else:
            raise Exception("Wrong action %s is provided"%action)

    def _docker_setup_network(self, script, pid_repo, netns_dir, r_conf, dhcp_client_script=None, ipv6_supported=False, sec_attr=None):
        """
        Will run the network setup script provided which will spawn dhcp client process foe the interfces,
         and the PID's of them are stored in pid_repo given.
        """
        log.debug("Setting up the network for docker style apps!")
        if not os.path.isfile(script):
            log.error("Given docker network script %s is not a valid file"%script)
            raise Exception("Given docker network script %s is not a valid file"%script)
        con_pid = self.get_container_pid()
        if con_pid is None:
            log.error("Container doesn't seem to have a pid!?")
            raise Exception("Container doesn't seem to have a pid!?")
        src = "/proc/" + con_pid + "/ns/net"
        link = os.path.join(netns_dir, self.app_id)
        if os.path.islink(link):
            log.debug("Apps syslink %s, is found in netns dir, so removing it"%link)
            os.remove(link)
        if os.path.isfile(link):
            log.debug("Libvirt is already created the Network name-space for the app, So CAF will be skipping it!")
        else:
            log.debug("Apps network name space symlink %s to %s is going to get created"%(src, link))
            os.symlink(src, link)
        need_dhcp = False
        ip_families = ["ipv4"]
        if ipv6_supported:
            log.debug("Platform supports IPv6 so, will be running dhcp client to get IPv6 address also!")
            ip_families.append("ipv6")
        log.debug("Libvirt network list info: %s" % str(self._libvirt_network_list))
        for values in self._libvirt_network_list:
            intf = values.get('interface_name')
            if values.get('mode') == 'static':
                log.debug("Interface %s is asking for static ip, so not running dhcp"%intf)
                continue
            for ip_fam in ip_families:
                ip_cfg = values.get(ip_fam)
                if ip_cfg:
                    if ip_cfg.get('mode') == 'static':
                        continue
                    if ip_cfg.get('disabled', False) == True:
                        log.debug("IP: %s is disabled" % ip_fam)
                        continue
                cmd = [script]
                cmd.append("-d")
                cmd.append(self.app_id)
                cmd.append("-p")
                cmd.append(pid_repo)
                cmd.append("-i")
                cmd.append(intf)
                cmd.append("-r")
                cmd.append(r_conf)
                # check if dhcp client id is specified in environment variables
                if ip_fam == "ipv6":
                    env_label = "CAF_APP_DHCP_CLIENT_ID_%s_V6" % intf.replace(":", "_")
                else:
                    env_label = "CAF_APP_DHCP_CLIENT_ID_%s" % intf.replace(":", "_")
                if env_label in self.app_env:
                    client_id = self.app_env.get(env_label)
                    cmd.append("-c")
                    cmd.append(client_id)
                if dhcp_client_script is not None and os.path.isfile(dhcp_client_script):
                    cmd.append("-s")
                    cmd.append(dhcp_client_script)
                cmd.append("-f")
                cmd.append(ip_fam)
                cmd.append("-pid")
                cmd.append(con_pid)
                if ipv6_supported:
                    mac = values["mac_address"]
                    cmd.append("-m")
                    cmd.append(mac)
                if sec_attr:
                    sec_label = sec_attr.get("label", "")
                    if sec_label and sec_label != "NA" and sec_label != "_":
                        cmd.append("-l")
                        cmd.append(sec_label)
                try:
                    log.debug("Executing script : %s" % " ".join(cmd))
                    rval = subprocess.check_output(" ".join(cmd), stderr=subprocess.STDOUT, shell=True)
                    output, rcode = rval, 0
                except subprocess.CalledProcessError as c:
                    output, rcode = c.output, c.returncode
                except Exception as ex:
                    log.exception("Error executing script : %s" % " ".join(cmd))
                    output, rcode = str(ex), -1
                if rcode == 0:
                    log.debug("DHCP client for interface %s_%s is successfully triggered" % (intf, ip_fam))
                elif rcode == 3 and not os.path.exists(os.path.join("/proc", str(con_pid))):
                    log.error("App got crashed, before CAF can setup the networking. Cause: %s"%output)
                    raise Exception("App got crashed, before CAF can setup the networking!")
                else:
                    log.error("Error wile triggering DHCP client for interface %s: cause %s" % (intf, output))
                need_dhcp = True
        # if all interfaces asks for static ip then, no need of this softlink
        if not need_dhcp:
            if os.path.islink(link):
                os.remove(link)

    def _docker_teardown_network(self, network_teardown_script, pid_repo, netns_dir, r_conf, dhcp_client_script=None, ipv6_supported=False):
        """
        Will run the provided network tear down script by passing the arguments as domain id and pid_repo+pid_file.

        """
        log.debug("Tearingdown the docker apps network")
        run_teardown_script = False
        if os.path.isfile(network_teardown_script):
            log.debug("Given docker network teardown script is :%s"%network_teardown_script)
            run_teardown_script = True
        else:
            log.debug("There is no docker networking teardown script is provided!")

        log.debug("Libvirt Network list: %s" % self._libvirt_network_list)
        for nw in self._libvirt_network_list:
            intf = nw['interface_name']
            for ip_fam in 'ipv4', 'ipv6':
                if run_teardown_script:
                    ip_cfg = nw.get(ip_fam)
                    if ip_cfg:
                        if ip_cfg.get('mode') == 'static':
                            continue
                        if ip_cfg.get('disabled', False) == True:
                            log.debug("IP: %s is disabled" % ip_fam)
                            continue
                    cmd = [network_teardown_script]
                    cmd.append("-d")
                    cmd.append(self.app_id)
                    cmd.append("-p")
                    cmd.append(pid_repo)
                    cmd.append("-i")
                    cmd.append(intf)
                    cmd.append("-r")
                    cmd.append(r_conf)
                    if dhcp_client_script:
                        cmd.append("-s")
                        cmd.append(dhcp_client_script)
                    cmd.append("-f")
                    cmd.append(ip_fam)
                    output, rcode, err = "", "", ""
                    try:
                        log.debug("Executing script : %s" % " ".join(cmd))
                        out, err, rv = PipeCommand(cmd).run(capture=True, timeout=5, sig=signal.SIGKILL)
                        output, rcode, error = out, rv, err
                    except subprocess.CalledProcessError as c:
                        log.error("Error while running the docker network teardown script. Return code: %s, output:%s"%(c.returncode, c.output))
                        output, rcode, error = c.output, c.returncode, err
                    except Exception as ex:
                        log.error("Error while running the docker network teardown script. Return code: %s, output:%s"%(-1, ex.message))
                        output, rcode, error = str(ex), -1, ex.message
                    log.debug("Result after executing the docker network teardown script: Output: %s, ReturnCode: %s"%(output, rcode))

                if os.path.isfile(os.path.join(pid_repo, intf+ip_fam)):
                    pid_file = os.path.join(pid_repo, intf+ip_fam)
                    log.debug("DHCP client PID file %s exists"%pid_file)
                    with open(pid_file) as f:
                        dhcp_pid = f.read().strip()
                        if not kill(dhcp_pid, 2):
                            log.error("Error while killing the pid %s"%dhcp_pid)
                            raise Exception("Error while killing the pid %s"%dhcp_pid)
        link = os.path.join(netns_dir, self.app_id)
        if os.path.islink(link):
            log.debug("Apps syslink %s, is found in netns dir, so removing it"%link)
            os.remove(link)



    def get_app_cpu_allocated(self):
        """Return allocated cpu for the app in percentage"""
        # How do we get it from XML?
        return self.cpu_shares


    def get_app_memory_allocated(self):
        """Return allocated memory for the app in MB"""
        # How do we get it from XML?
        return self.mem_limit_mb


    def get_app_disk_allocated(self):
        """Return allocated disk for the app in MB"""
        return self.disk

    def get_app_ipaddress_info(self):
        """
        Returns the ipaddress  for the container using virsh lxc-enter-namespace'
        Parses ifconfig output:
        eth0      Link encap:Ethernet  HWaddr 00:0C:29:48:08:8A  
          inet addr:192.168.255.157  Bcast:192.168.255.255  Mask:255.255.255.0
          inet6 addr: fe80::20c:29ff:fe48:88a/64 Scope:Link
          UP BROADCAST RUNNING MULTICAST  MTU:1500  Metric:1
          RX packets:96292 errors:0 dropped:0 overruns:0 frame:0
          TX packets:68113 errors:0 dropped:0 overruns:0 carrier:0
          collisions:0 txqueuelen:1000 
          RX bytes:63760035 (60.8 MiB)  TX bytes:8819236 (8.4 MiB)
        """
        import re
        network_info = copy.deepcopy(self.network_info)
        log.debug("Going to populate ipaddresses for %s" , network_info) 
        buf = None
        data = None

        if self.isRunning():
            try:
                pdhook = LXCContainerManager.getInstance().PDHook
                datadir = os.path.join(self.getContainerRoot(), self.appid)
                log.debug("Calling app hook: type:%s hook:%s id:%s rootfs:%s data_dir:%s" % 
                                (self._container_type, 
                                         pdhook.PRE_FETCH_IFCONFIG, 
                                         self.appid,
                                         self.dst_rootfs,
                                         datadir))
                data, rc = pdhook.call_app_lifecycle_hook(self._container_type, 
                                         pdhook.PRE_FETCH_IFCONFIG, 
                                         {},
                                         self.appid,
                                         self.dst_rootfs,
                                         datadir)
                if rc != 10:  #10 is the success value for ifconfig  
                    if which("nsenter") is not None:
                        con_pid = self.get_container_pid()
                        if con_pid is None:
                            log.error("Container doesn't seem to have a pid!?")
                            raise Exception("Container doesn't seem to have a pid!?")
                        data = self._get_nsenter_ifconfig(con_pid)
                        #even though nsenter command is there, might be the case like attributes not there
                        #if data is None:
                        #     data = self._get_lxc_namespace_ifconfig()
                    else:
                        data = self._get_lxc_namespace_ifconfig()
                if data is None:
                    return {}
                log.debug("Container %s ifconfig: %s" % (self.app_id, data))
                for paragraph in data.split("\n\n"):
                    if paragraph is None:
                        continue
                    try:
                        interface_name = paragraph.split()[0]
                    except IndexError:
                        continue

                    if "lo" in interface_name:
                        continue

                    re_ipv4 = re.search(r'inet addr:(\S+)', paragraph)
                    re_ipv6 = re.findall(r'inet6 addr: (\S+)', paragraph)
                    re_mac = re.search(r'HWaddr (\S+)', paragraph)

                    ipv4 = None
                    ipv6 = None
                    mac = None

                    if re_ipv4:
                        ipv4 = re_ipv4.group(1)

                    if re_ipv6:
                        ipv6 = re_ipv6

                    if re_mac:
                        mac = re_mac.group(1)
                    
                    d = network_info.get(interface_name)
                    if d is None:
                        network_info[interface_name] = {}
                        d = network_info.get(interface_name)

                    d["ipv4"] = ipv4
                    d["ipv6"] = ipv6
                    d["mac"] =  mac
                    
            except Exception as e:
                log.exception("get_app_ipaddress_info failed: %s" % str(e))
                if data is not None:
                    log.error("Result:%s" % data)
                return {}
            finally:
                if buf is not None:
                    buf.close()
        return network_info

    def get_parent_pid(self):
        if not self._domain_run_xml:
            self._domain_run_xml = self.domain.XMLDesc()
        data = self._domain_run_xml
            
        import xml.etree.ElementTree as ET
        root = ET.fromstring(data)
        attr = root.attrib
        parent_pid = attr.get('id')
        return parent_pid

    def get_container_pid(self):
        """
        Get the dumpxml info of the container and parse it through for the pid, once the pid is found,
         return it. If not return None
        """
        log.debug("Getting the pid for the container %s"%self.domain.name())
        pid = None
        #Here we can expect libvirt error in case of any failure occures.
        parent_pid = self.get_parent_pid()
        if parent_pid is not None:
            if not psutil.pid_exists(int(parent_pid)):
                log.info("Parent PID %s doesn't exist, trying to get it again" % str(parent_pid))
                try:
                    self.domain = self._libvirt_client.lookupByName(self.app_id)
                    self._domain_run_xml = self.domain.XMLDesc()
                except Exception as e:
                    log.error("Failed to get container domain %s" , str(e))
                    return pid
                parent_pid = self.get_parent_pid()
                if parent_pid is None:
                    return pid

            if which("pgrep") is not None:
                data, rc = pgrep('-P',  parent_pid)
                if rc != 0:
                    log.error("pgrep: to get container init pid failed: %s" % str(data))
                    return pid
                pid = data.split()[0]
            else:
                #Find the child PID(which is containers init PID)
                children_file_path = os.path.join("/proc", parent_pid, "task", parent_pid, "children")
                if os.path.isfile(children_file_path):
                    with open(children_file_path, "r") as f:
                        children_pids = f.read()
                        children_pids = children_pids.split(" ")
                        pid = children_pids[0]
                else:
                    children_pids = Utils.get_child_processes(int(parent_pid))
                    if children_pids:
                        pid = str(children_pids[0])
                    return pid
        return pid

    def _get_nsenter_ifconfig(self, con_pid):
        """
        Execute the commands :
            nsenter -t <PID> -n /sbin/ifconfig
            nsenter -t <PID> -n /bin/ifconfig
        and return the result if any one is successfull.
        If neither got successfull return None
        """
        data, rc = nsenter('-t',  con_pid, '--net', '/sbin/ifconfig')
        if rc != 0:
            log.error("nsenter: /sbin/ifconfig, get app ipaddress info failed: %s" % str(data))
            #Retry again with /bin path
            log.info("nsenter: Retrying again with /bin/ifconfig path")
            data, rc = nsenter('-t',  con_pid, '-n', '/bin/ifconfig')
            if rc != 0:
                log.error("nsenter:did not got app ipaddress info : %s" % str(data))
                return None
        return data

    def _get_lxc_namespace_ifconfig(self):
        """
        Execute the commands :
            virsh -c lxc:/// lxc-enter-namespace <name> --noseclabel /sbin/ifconfig
            virsh -c lxc:/// lxc-enter-namespace <name> --noseclabel /bin/ifconfigg
        and return the result if any one is successfull.
        If neither got successfull return None
        """

        if self.seclabel:
            data, rc = virsh('-c',  'lxc:///', 'lxc-enter-namespace', self.domain.name(), '/sbin/ifconfig')
        else:
            data, rc = virsh('-c',  'lxc:///', 'lxc-enter-namespace', self.domain.name(), '--noseclabel', '/sbin/ifconfig')

        if rc != 0:
            log.error("Virsh : get_app_ipaddress_info failed: %s" % str(data))
            #Retry again with /bin path
            log.info("Virsh: Retrying again with /bin/ifconfig path")
            if self.seclabel:
                data, rc = virsh('-c',  'lxc:///', 'lxc-enter-namespace', self.domain.name(), '/bin/ifconfig')
            else:
                data, rc = virsh('-c',  'lxc:///', 'lxc-enter-namespace', self.domain.name(), '--noseclabel', '/bin/ifconfig')
            if rc != 0:
                log.error("get_app_ipaddress_info failed: %s" % str(data))
                return None
        return data

    def get_app_network_usage(self):
        """ 
        Returns total bytes recieved and transferred
        Gets Network information from inside a container
        Reads /proc/net/dev to get the network stats
        Sample contents of /proc/net/dev are:
        Inter-|   Receive                                                |  Transmit
        face |bytes    packets errs drop fifo frame compressed multicast|bytes    packets errs drop fifo colls carrier compressed
        sit0:       0       0    0    0    0     0          0         0        0       0    0    0    0     0       0          0
        lo:   67232     928    0    0    0     0          0         0    67232     928    0    0    0     0       0          0
        ...
        """
        log.debug("get_app_network called for container %s" % self.getId())
        total_bytes = 0
        buf = None
        data = None

        if self.isRunning():
            try:
                con_pid = self.get_container_pid()
                if con_pid is None:
                    log.error("Container doesn't seem to have a pid!?")
                    raise Exception("Container doesn't seem to have a pid!?")
                net_stat_file = ("/proc/%s/net/dev" % con_pid)
                if os.path.exists(net_stat_file):
                    data, rc = cat(net_stat_file)
                    if rc != 0:
                        log.error("cat: get_app_network_usage failed: %s" % str(data))
                        return None
                elif which("nsenter") is not None:
                    data, rc = nsenter('-t',  con_pid, '--net', '/bin/cat', '/proc/net/dev')
                    if rc != 0:
                        log.error("nsenter: get_app_network_usage failed: %s" % str(data))
                        return None
                else:
                    if self.seclabel:
                        data, rc = virsh('-c',  'lxc:///', 'lxc-enter-namespace', self.domain.name(), '/bin/cat', '/proc/net/dev')
                    else:
                        data, rc = virsh('-c',  'lxc:///', 'lxc-enter-namespace', self.domain.name(), '--noseclabel', '/bin/cat', '/proc/net/dev')

                    if rc != 0:
                        log.error("get_app_network_usage failed: %s" % str(data))
                        return None
                if data is None:
                    return None
                buf = cStringIO.StringIO(data)
                log.debug("Contents of /proc/net/dev inside container %s:%s\n", self.app_id, data)
                total_bytes = Utils.parse_proc_dev_net(buf)
                return total_bytes
            except Exception as e:
                log.error("get_app_network_usage failed: %s" % str(e))
                if data is not None:
                    log.error("/proc/net/dev:%s" % data)
                return None
            finally:
                if buf is not None:
                    buf.close()
        else:
            return None

    def get_app_disk_usage(self):
        """
        Gets Disk information regarding data used from inside a container in MBs
        """
        datadir = os.path.join(self.getContainerRoot(), self.appid)
        log.debug("Checking size of directory %s", datadir)
        data_disk_storage =  Utils.get_dir_size(datadir)
        log.debug("App %s disk data usgae %s" % (self.app_id, str(data_disk_storage / float(1024 * 1024))))
        return int(round((data_disk_storage)/ float(1024 * 1024)))

    def getContainerRoot(self):
        """
        Returns the absolute path to the container's root directory
        """
        data_root = LXCContainerManager.getInstance().get_container_data_root_path()
        return data_root

    def getPersistentDataPath(self):
        return self._resources.get("persistent_data_target", LXCContainerManager.getInstance().get_data_mount_point())

    def getContainerLogDir(self):
        data_root = self.getContainerRoot()
        logDirName = LXCContainerManager.getInstance().get_container_logDir()
        rfs_type = LXCContainerManager.getInstance().get_container_rfs_type()
        if rfs_type is RFSComposer.TYPE_AUFS and self.apptype == AppType.PAAS:
            data_mount = self.getPersistentDataPath()
            logDirPath = os.path.join(data_root, self.appid, data_mount.lstrip("/"), logDirName)
        elif rfs_type is RFSComposer.TYPE_OVERLAY and self.apptype == AppType.PAAS:
            data_mount = self.getPersistentDataPath()
            logDirPath = os.path.join(data_root, self.appid, DOCKER_RW_LAYER_ROOT, data_mount.lstrip("/"), logDirName)
        else:
            logDirPath = os.path.join(data_root, self.appid, logDirName)
        return logDirPath

    def getContainerCoreDir(self):
        coreDir= LXCContainerManager.getInstance().get_container_coreDir()
        app_coredir = os.path.join(coreDir, self.appid)
        return app_coredir

    def getContainerDriverLogDir(self):
        logDirPath = LXCContainerManager.getInstance().get_container_driver_logDir()
        return logDirPath

    def getContainerIoxLayerStore(self):
        ioxLayerPath = LXCContainerManager.getInstance().iox_layer_store
        return ioxLayerPath

    def _get_logfile_path(self, logfilename):
        """
        Get absolute path for a given log file name
        """
        logsDir = self.getContainerLogDir()
        if os.path.isabs(logfilename):
            return logfilename
        else:
            if logfilename.startswith(self.LIBVIRT_DRIVER_LOG_PREFIX):
                return self._get_driver_logfile_path(logfilename[len(self.LIBVIRT_DRIVER_LOG_PREFIX):])
            else:
                return os.path.join(logsDir, logfilename)

    def _get_driver_logfile_path(self, logfileName):
        driverLogsDir = self.getContainerDriverLogDir()
        if os.path.isabs(logfileName):
            return logfileName
        else:
            return os.path.join(driverLogsDir, logfileName)

    def getLogPath(self, filename):
        """If the passed filename is a valid log file (i,e monitored)
        then return full path. Else return None"""

        filePath = self._get_logfile_path(filename)
        if os.path.isfile(filePath) and filePath in self._monitoredLogFiles:
            log.debug("Log path for %s is %s" % (filename, filePath))
            return filePath

        log.error("Requested file %s is not a valid log file" % filename)
        return None


    def preserve_container_data(self, archive_path, preserve_file_list=[], preserve_app_config=False):
        # Currently we preserve entire container root while upgrade.
        data_root = self.getContainerRoot()
        tmpdir = LXCContainerManager.getInstance().get_tmpdir()
        container_data_arch, preserved_files_records = Utils.preserve_container_data(data_root, self.appid, archive_path,
               preserve_file_list, preserve_app_config, tmpdir)
        return container_data_arch, preserved_files_records

    def restore_container_data(self, container_data_arch):
        data_root = self.getContainerRoot()
        Utils.restore_container_data(data_root, container_data_arch)

    def execute_health_script(self, app_health_script, timeout=5):
        """
        Exceutes the health script within container
        """

        if self.apptype == AppType.PAAS:
            #For paas type apps health script cannot be the absolute path
            appdir = LXCContainerManager.getInstance().appdir
            if not app_health_script.startswith("/"):
                app_health_script = os.path.join(appdir,"app",app_health_script)

        if which("nsenter") is not None:
            #nsenter -t pid -m -u -i -n -p -U -S 0 -G 0 -r -w -Z app_health_script
            pid = self.get_container_pid()
            cmdargs = ['-t', pid, '-m', '-u', '-i', '-n', '-p']
            if self.sec_attr:
                model = None
                if self.sec_attr.get("lsm") and self.sec_attr.get("lsm").get("enabled"):
                    model = self.sec_attr.get("lsm").get("model")
                if self.sec_attr.get("userns") and self.sec_attr.get("userns").get("enabled") and model != "smack":
                    cmdargs.append('-U')
                if model == "selinux":
                    cmdargs.append('-Z')
            cmdargs.extend(['-S 0', '-G 0', '-r', '-w'])
            cmdargs.append(app_health_script)
            health_cmd = "nsenter " + " ".join(cmdargs)
        else:
            if self.seclabel:
                health_cmd= "virsh -c lxc:/// lxc-enter-namespace %s %s" % (self.appid, app_health_script)
            else:
                health_cmd= "virsh -c lxc:/// lxc-enter-namespace --noseclabel %s %s" % (self.appid, app_health_script)
        log.debug("health_cmd = %s" % health_cmd)
        health_out, err, rv = PipeCommand(health_cmd.split()).run( capture=True, timeout=timeout)    
        log.debug("Executed health scripts: %s rv:%s output:%s error:%s" % (app_health_script, rv, health_out, err))
        return health_out, err, rv

    def add_static_dns_entry(self, dns_entries=[]):
        """
        Adds the static DNS entry to the containers etc/resolv.conf file
        :param dns_entries: Static DNS entries needs to added to the resolv.conf file
        """
        try:
            if dns_entries:
                log.debug("Static DNS entries needs to be present %s, if not will add them"%dns_entries)
                # Check for entries are present in the file or not
                if self.seclabel:
                    out_put, rcode = virsh("-c", "lxc:///", "lxc-enter-namespace", self.app_id, "/bin/cat", "/etc/resolv.conf")
                else:
                    out_put, rcode = virsh("-c", "lxc:///", "lxc-enter-namespace", self.app_id, "--noseclabel", "/bin/cat", "/etc/resolv.conf")
                if rcode != 0:
                    log.error("Error while executing the command for grepping static DNS entries. Output: %s, Return Code: %s"%(out_put, rcode))
                else:
                    #Parse the contents of the resolv.conf file
                    entries = []
                    for line in out_put.splitlines():
                        line = line.split('#', 1)[0]
                        line = line.strip()
                        if 'nameserver' in line:
                            if len(line.split()) >= 2:
                                entries.append(line.split()[1])
                    entries_needs_added = list(set(dns_entries) - set(entries))
                    log.debug("Going to add the static DNS entries %s in to the container"%entries_needs_added)
                    if entries_needs_added:
                        data = ""
                        for entry in entries_needs_added:
                            data = data + "nameserver %s\n"%entry
                        if self.seclabel:
                            cmd = "virsh -c lxc:/// lxc-enter-namespace %s /bin/sh -c '/bin/echo -e \"%s\" >> /etc/resolv.conf'"%(self.app_id, data)
                        else:
                            cmd = "virsh -c lxc:/// lxc-enter-namespace %s --noseclabel /bin/sh -c '/bin/echo -e \"%s\" >> /etc/resolv.conf'"%(self.app_id, data)
                        try:
                            out_put, err, rcode = PipeCommand(cmd).run(capture=True, timeout=5, shell=True)
                        except Exception as ex:
                            log.exception("Error while adding static DNS entries: Cause: %s "%str(ex))
                            err = str(ex)
                            rcode = -1
                        if rcode == 0:
                            log.info("Successfully added the static entries: %s"%entries_needs_added)
                            return True
                        else:
                            log.error("Error while adding static DNS entries: Cause: %s"%err)
                    else:
                        log.debug("All static DNS entries are already present")
                        return True
        except Exception as ex:
            log.exception("Error while adding static DNS entries: Cause: %s"%str(ex))
        return False


class LXCContainerManager(LibvirtContainerManager):

    __singleton = None # the one, true Singleton

    DATA_EXT2_SUFFIX = "_data.ext2"    
    DATA_EXT4_SUFFIX = "_data.ext4"    
    def __new__(cls, *args, **kwargs):
        # Check to see if a __singleton exists already for this class
        # Compare class types instead of just looking for None so
        # that subclasses will create their own __singleton objects
        if cls != type(cls.__singleton):
        #if not cls.__singleton:
            cls.__singleton = super(LXCContainerManager, cls).__new__(cls, *args, **kwargs)
        return cls.__singleton

    @classmethod
    def getInstance(cls, *args):
        '''
        Returns a singleton instance of the class
        '''
        if not cls.__singleton:
            cls.__singleton = LXCContainerManager(*args)
        return cls.__singleton

    def __init__(self, config, languageRuntimes, runtime_context=None):
        log.debug("LXC CONTAINER MGR INIT")
        root_path = config.get("lxc-container", "root")
        data_root = config.get("lxc-container", "data_volume_root")
        data_mount_point = config.get("lxc-container", "data_mount_point")
        self._emulator = config.get("lxc-container", "emulator")
        log_dir = config.get("app-settings", "appLogsDir")

        self.appdir = "appdir"
        if config.has_option("app-settings", "appDirName"):
            self.appdir = config.get("app-settings", "appDirName")

        appdata_dir = "appdata"
        if config.has_option("app-settings", "appDataDir"):
            appdata_dir = config.get("app-settings", "appDataDir")

        if config.has_option("lxc-container", "app_image_path"):
            self.app_image_path = config.get("lxc-container", "app_image_path")
        else:
            self.app_image_path = root_path

        if config.has_option("lxc-container", "paas_copy_rootfs"):
            self.paas_copy_rootfs = config.getboolean("lxc-container", "paas_copy_rootfs")
        else:
            self.paas_copy_rootfs = False

        corefile_dir="/tmp/cores"
        if config.has_option("corefiles", "coreFilesDir"):
            corefile_dir = config.get("corefiles", "coreFilesDir")
        src_corefile_dir=corefile_dir
        if config.has_option("corefiles", "src_core_dir"):
            src_corefile_dir = config.get("corefiles", "src_core_dir")
        self._mnt_corefile_dir = corefile_dir

        if not os.path.isdir(self.app_image_path):
            os.makedirs(self.app_image_path)

        if config.has_option("overlayFs", "iox_layer_store"):
            self._iox_layer_store = config.get("overlayFs", "iox_layer_store")
        else:
            self._iox_layer_store = data_root

        if config.has_option("lxc-container", "preferred_union_fs"):
            self._preferred_union_fs = config.get("lxc-container", "preferred_union_fs")
        else:
            self._preferred_union_fs = None

        if not os.path.isdir(self._iox_layer_store):
            os.makedirs(self._iox_layer_store)

        pc = PlatformCapabilities.getInstance()
        self.aufs_support = pc.aufs_supported
        self.overlay_supported = pc.overlayfs_supported
        self.ipv6_supported = pc.ipv6_supported

        self._supported_rootfs_types = pc.supported_rootfs_types

        if self.paas_copy_rootfs:
            self.rfs_type = RFSComposer.TYPE_COPY
        elif self.overlay_supported:
            self.rfs_type = RFSComposer.TYPE_OVERLAY
        elif self.aufs_support:
            self.rfs_type = RFSComposer.TYPE_AUFS
        else:
            self.rfs_type = RFSComposer.TYPE_MOUNT

        if pc.supported_union_fs is not None and self._preferred_union_fs is not None:
            log.debug("Preferred union fs %s and supported fs %s", self._preferred_union_fs, pc.supported_union_fs)
            if self._preferred_union_fs in pc.supported_union_fs:
                if self._preferred_union_fs == "overlay":
                    self.rfs_type = RFSComposer.TYPE_OVERLAY
                elif self._preferred_union_fs == "aufs":
                    self.rfs_type = RFSComposer.TYPE_AUFS
                else:
                    log.info("Invalid preferred union fs provided, goahead with default %s rfs", self.rfs_type)
            else:
                log.info("Preferred union Fs type not available in supported platform capabilities rfs type %s", self.rfs_type)

        if config.has_option("overlayFs", "rw_layer_size"):
            self.rw_layer_size = config.get("overlayFs", "rw_layer_size")
        else:
            self.rw_layer_size = DEFAULT_RW_LAYER_SIZE

        if config.has_option("overlayFs", "max_rw_layer_size"):
            self.max_rw_layer_size = config.get("overlayFs", "max_rw_layer_size")
        else:
            self.max_rw_layer_size = DEFAULT_RW_LAYER_SIZE

        if config.has_option("overlayFs", "min_rw_layer_size"):
            self.min_rw_layer_size = config.get("overlayFs", "min_rw_layer_size")
        else:
            self.min_rw_layer_size = 1

        if config.has_option("overlayFs", "rw_layer_to_code_size_ratio"):
            self.rw_layer_to_code_size_ratio = config.get("overlayFs", "rw_layer_to_code_size_ratio")
        else:
            self.rw_layer_to_code_size_ratio =  DEFAULT_RW_LAYER_TO_CODE_RATIO

        if config.has_option("controller", "persistent_store"):
            self.persistent_store = config.get("controller", "persistent_store")
        else:
            self.persistent_store = data_root

        self._use_ext4 = False
        if config.has_option("controller", "use_ext4"):
            self._use_ext4 = config.getboolean("controller", "use_ext4")
        
        if config.has_option("lxc-container", "app_mount_point"):
            self.app_mount_point = config.get("lxc-container", "app_mount_point")
        else:
            self.app_mount_point = "/app"
        if config.has_option("lxc-container", "cartridge_permission"):
            self.cartridge_permission = config.get("lxc-container", "cartridge_permission")
        else:
            self.cartridge_permission = "ro"
        if not os.path.isdir(self.persistent_store):
            os.makedirs(self.persistent_store)

        # Read tmp location from system config, default to /tmp if not specified
        self.tmpUploadDir = '/tmp'
        if config.has_option("controller", "upload_dir"):
            self.tmpUploadDir = config.get("controller", "upload_dir")

        if not os.path.exists(self.tmpUploadDir):
            os.makedirs(self.tmpUploadDir)

        self._app_root = config.get("lxc-container", "app_root")
        self._cartridge_mount_point = config.get("lxc-container", "cartridge_mount_point")
        if not os.path.isdir(root_path):
            os.makedirs(root_path)

        self._connection_str = config.get("lxc-container", "connection_str", "lxc:///")

        self._lxcLogDirName = "/var/log/libvirt/lxc"
        if config.has_option("logging", "lxcDriverLogPath"):
            self._lxcLogDirName = config.get("logging", "lxcDriverLogPath")

        self.usb_storage_container_mount = "/mnt/usbdrive"
        if config.has_option("lxc-container", "usb_storage_container_mount"):
            self.usb_storage_container_mount = config.get("lxc-container", "usb_storage_container_mount")

        self.usb_storage_host_mount = "/mnt/host/usbstorage"
        if config.has_option("lxc-container", "usb_storage_host_mount"):
            self.usb_storage_host_mount = config.get("lxc-container", "usb_storage_host_mount")

        self.dirMount_blacklist = "bin, boot, dev, etc, opt, proc, root, run, sbin, sys, usr, var"
        if config.has_option("lxc-container", "DirectoryMount_blacklist"):
            self.dirMount_blacklist = config.get("lxc-container", "DirectoryMount_blacklist")

        LibvirtContainerManager.__init__(self,  config, languageRuntimes, "lxc", 
            runtime_context=runtime_context, root_path=root_path,
            data_root=data_root, data_mount_point=data_mount_point, 
            log_dir=log_dir, appdata_dir=appdata_dir, core_dir = src_corefile_dir)

    @property
    def get_tmpdir(self):
        return self.tmpUploadDir

    @property
    def iox_layer_store(self):
        return self._iox_layer_store

    def get_aufs_support(self):
        return self.aufs_support

    def get_overlayFs_support(self):
        return self.overlay_supported

    def get_copy_rootfs_support(self):
        return self.paas_copy_rootfs

    def get_data_mount_point(self):
        return self.data_mount_point

    def get_container_rfs_type(self):
        return self.rfs_type

    def set_container_rfs_type(self, type):
        self.rfs_type = type

    def supportsAppType(self, apptype):
        if apptype == AppType.LXC:
            return True
        if apptype == AppType.PAAS:
            return True
        if apptype == AppType.DOCKER:
            return True
        return False

    def _create_container_data_mount_point(self, containerId, rootfsdir):
        target = os.path.join(rootfsdir, self.data_mount_point.lstrip("/"))
        if not os.path.isdir(target):
            os.makedirs(target)
        log.debug("Created target data directory for container %s at %s" % (containerId, target))
        return target

    def _create_data_dir(self, data_ext, containerId, size=10, security_str=None):
        #Create the file of the specified size
        #directory to mount ext2 file
        log.debug("create data dir, data_ext %s, security_str %s", data_ext, security_str)
        mtpnt = os.path.join(self.data_root, containerId)
        if not os.path.isdir(mtpnt):
            os.makedirs(mtpnt)

        #cmdargs = "-t ext2 -o loop,rw"
        if self._use_ext4:
            cmdargs = ["-t", "ext4", "-o"]
        else:
            cmdargs = ["-t", "ext2", "-o"]
        rw = "loop,rw,noatime"
        if security_str:
            #cmdargs += "," + security_str
            rw = rw + "," + security_str
        cmdargs.append(rw)

        cmdargs.extend([data_ext, mtpnt])
        if not Utils.ismount_exists(mtpnt):
            out, rc = mount(cmdargs)
            if rc != 0:
                log.error("Error in creating data directory: %s", str(out))
                raise AppInstallationError("Error in creating data directory: %s", str(out))

        log.debug("Created data directory for container %s at %s" % (containerId, mtpnt))     
        return mtpnt

    def _evaluate_rw_layer_size(self, app_artifacts_size, rw_layer_code_size_ratio, min_size, max_size, app_container_size=None):
        '''
        This function calculates the size of rw layer for docker Apps
        Inputs:
        app_artifacts_size: Cumulative size of all the App layers
        rw_layer_to_code_ratio: % of app artifacts size that can be allocated
        min_size: minimuim size recommended on a given platform
        max_size: maximuim size recommended on a given platform
        app_container_size: Value provided by admin via activation payload.This will have precedence provided it does
        not exceed the max value configured
        '''

        log.debug("Evaluate rw layer size, app size %s , rw layer to code ratio %s min size %s max size %s app container size %s", app_artifacts_size,rw_layer_code_size_ratio,min_size,max_size,app_container_size)
        min_size = float(min_size)
        max_size = float(max_size)
        rw_layer_code_size_ratio = float(rw_layer_code_size_ratio)
        app_artifacts_size = float(app_artifacts_size)

        if app_container_size is not None:
            app_container_size = float(app_container_size)
            if app_container_size >= min_size and app_container_size <= max_size:
                return app_container_size
            elif app_container_size > max_size:
                log.info("Overriding the rw layer by max size %s since requested value %s exceeds the platform config", max_size, app_container_size)
                return max_size
            elif app_container_size < min_size:
                return min_size
            else:
                return DEFAULT_RW_LAYER_SIZE

        rw_layer_size = int(round(rw_layer_code_size_ratio * app_artifacts_size)/100)
        if rw_layer_size <= min_size:
            return min_size
        elif rw_layer_size >= max_size:
            return max_size
        else:
            return rw_layer_size

    def _create_rw_layer_dir(self, rw_disk, containerId, size=10, security_str=None):
        log.debug("create rw layer dir, ext FS  %s, security_str %s", rw_disk, security_str)
        mtpnt = os.path.join(self.iox_layer_store, containerId)
        if not os.path.isdir(mtpnt):
            os.makedirs(mtpnt)

        if self._use_ext4:
            cmdargs = ["-t", "ext4", "-o"]
        else:
            cmdargs = ["-t", "ext2", "-o"]
        rw = "loop,rw,noatime"
        if security_str:
            rw = rw + "," + security_str
        cmdargs.append(rw)

        cmdargs.extend([rw_disk, mtpnt])
        if not Utils.ismount_exists(mtpnt):
            out, rc = mount(cmdargs)
            if rc != 0:
                log.error("Error in creating rw layer directory: %s", str(out))
                raise AppInstallationError("Error in creating rw layer directory: %s", str(out))

        log.debug("Created rw layer directory for container %s at %s" % (containerId, mtpnt))
        return mtpnt

    def get_security_attributes(self, appid, resources):
        """
        Get the applicable security attribute
        """
        #Security labels and attributes
        app_privileged = resources.get("privileged", False)
        if  resources.get("use_host_mode", False) == True:
            sec_attr = None
        else:
            sec_attr = self.setup_app_security(appid, app_privileged)

        security_str = ""
        if sec_attr and sec_attr.get("lsm"):
            security_str = sec_attr.get("lsm").get("mnt_string", "")
        return sec_attr, security_str


    def get_mount_security_attributes(self, sec_attr):
        """
        Get security attributes for read only mount
        
        :param sec_attr:
        :param security_str:
        :return:
        """

        sec_str = ""

        # Apply container security (if enabled)
        if sec_attr:
            security_str = ""
            # LSM
            if sec_attr.get("lsm") and sec_attr.get("lsm").get("enabled"):
                label = sec_attr.get("lsm").get("label")
                security_str = sec_attr.get("lsm").get("mnt_string", "")
                sec_str += "," + security_str
            # USERNS
            if sec_attr.get("userns") and sec_attr.get("userns").get(
                "enabled"):
                uid_target = sec_attr["userns"]["uidtarget"]
                gid_target = sec_attr["userns"]["gidtarget"]
                sec_str += "," + "uid=" + str(uid_target) + ",gid=" + str(
                    gid_target)

        return sec_str

    def set_container_rootfs_security(self, file_list, sec_attr, scrub=False):
        """
        Apply security attributes to rootfs components of a
        container
        """
        # Do folders recursively first then individual files and folders that
        # do not require entire contents to be changed.
        try:
            completed = {
                "recursive" : [],
                "non-recursive" : []
            }
            if file_list.get("recursive", False):
                for directory in file_list["recursive"]:
                    Utils.apply_rootfs_attributes(sec_attr, directory, True, scrub)
                    completed["recursive"].append(directory)
            if file_list.get("non-recursive", False):
                for directory in file_list["non-recursive"]:
                    Utils.apply_rootfs_attributes(sec_attr, directory, False, scrub)
                    completed["non-recursive"].append(directory)
        except Exception as e:
            # If scrub/reverse is enabled, then we've already failed once
            # and we probably shouldn't attempt to alter the rootfs
            # again and just report the error.
            if scrub:
                log.exception("Failed to undo container security operation.")
                pass
            log.exception("There was a problem applying container security attributes to rootfs.")
            for directory in completed["non-recursive"]:
                Utils.apply_rootfs_attributes(sec_attr, directory, True, True)
            for directory in completed["recursive"]:
                Utils.apply_rootfs_attributes(sec_attr, directory, False, True)
            raise


    def host_mount_paths(self, resources, appid):
        host_paths_mapping = []
        if resources and "host_mounts" in resources:
            for idx, host_mount in enumerate(resources["host_mounts"]):
                if host_mount.get("target_mount", None) and host_mount.get("host_mount_path", None):
                    #dst_mnt_path = host_mnt_prefix + str(idx)
                    dst_mnt_path = host_mount["target_mount"]
                    #if not os.path.exists(dst_mnt_path) :
                    #    os.makedirs(dst_mnt_path)
                    if os.path.isdir(host_mount["host_mount_path"]):
                        if "readonly" in host_mount and host_mount["readonly"]:
                            #Do not create appid dir if it is readonly mount point
                            pass
                        else:
                            write_host_point = os.path.join(host_mount["host_mount_path"], appid)
                            if not os.path.exists(write_host_point) :
                                os.makedirs(write_host_point)
                    host_paths_mapping.append({"src_path" : host_mount["host_mount_path"], "dst_path": dst_mnt_path, "readonly": host_mount["readonly"]}) 
        return host_paths_mapping

    def provision_log_config_appdata(self, prov_data_dir, appcfgfile, full_datamount_perm=False):
        try:
            #Provision log directory inside new rootfs
            log_dirpath = os.path.join(prov_data_dir, self._logDirName)
            if not os.path.exists(log_dirpath) :
                os.makedirs(log_dirpath)
            if full_datamount_perm:
                os.chmod(log_dirpath, 0o777)
            #Provision appdata directory inside new rootfs
            appdata_dirpath = os.path.join(prov_data_dir, self._appdata_dir)
            if not os.path.exists(appdata_dirpath) :
                os.makedirs(appdata_dirpath)
            if full_datamount_perm:
                os.chmod(appdata_dirpath, 0o777)
            log.debug("Provisioned appdata dir at %s" % appdata_dirpath)

            if os.path.isfile(appcfgfile):
                # copy the config file to mountpoint
                existing_cfg = os.path.join(prov_data_dir, os.path.basename(appcfgfile))
                if not os.path.isfile(existing_cfg):
                    shutil.copy(appcfgfile, prov_data_dir)
                    log.debug("Provisioned the appconfig file at : %s", prov_data_dir)
        except Exception as ex:
            log.exception("Failed to provision direcories under data dir:%s error:%s" % (prov_data_dir, str(ex)))
            raise ex

    def mount_usb(self, dev_list, containerRequest, target_rootfs_dir, apptype, sec_attr):
        log.debug("dev_list Device  List: %s", dev_list)

        mounted_storage_drive = False
        mnt_point = None
        src_mnt_dir = None
        usb_mountpoints = []
        env={}
        for dev in dev_list:
            if dev and dev.get("type") == "usbdev" and dev.get("function") == "storage":
                try:
                    mnt_point = dev.get("mount-point")

                    if not mnt_point:
                        mnt_point = self.usb_storage_container_mount
                    else:
                        self._check_valid_mountpoint(mnt_point, containerRequest.containerId, apptype)

                    hm = HostingManager.get_instance()
                    dm = hm.get_service("device-management")
                    device = dm.get_device(dev.get("type"), dev.get("device-id"))

                    #Mount the usb Mass Storage and pass the right attributes for libvirt xml.
                    #use the right sec attributes for mounting, if one is available and security is also enabled
                    out, rc = self.mount_usb_storage(containerRequest.containerId, target_rootfs_dir, device, sec_attr,
                                                    self.usb_storage_host_mount, mnt_point)

                    if rc == 0:
                        log.debug("Mount storage details: %s, %s", device.storage_params["mount_point"],
                                 mnt_point)

                        src_mnt_dir = device.storage_params["mount_point"]

                        # check "device-directory-to-mount" to mount specific directory if it exists
                        dev_dir = dev.get("device-directory-to-mount")
                        if dev_dir is not None:
                            full_mnt_path = device.storage_params["mount_point"] + "/" + dev_dir
                            if os.path.exists(full_mnt_path):
                                src_mnt_dir = full_mnt_path

                        # creating the /usb/mntdrive in rootFS
                        usb_path = os.path.normpath(target_rootfs_dir + mnt_point)
                        log.debug("Mount storage exact: %s, %s", target_rootfs_dir,
                                  usb_path)
                        if not os.path.exists(usb_path):
                            os.makedirs(usb_path)
                        mounted_storage_drive = True
                        usb_mountpoints.append({"src_dir": src_mnt_dir, "dst_dir": mnt_point})

                        # here device-id will give u'/dev/bus/usb/001/026', dev_name is needed for mountpoint label
                        env[dev.get("label")] = mnt_point

                    elif rc != 0 and dev.get("mandatory"):
                        log.error("mount_usb_storage failed. ret code: %s error: %s"
                                  % (rc, str(out)))
                        self._remove_container_data(containerRequest.containerId, apptype)
                        raise AppInstallationError("Error while creating container : %s", str(out))
                except Exception as ex:
                    log.exception("Failed to mount usb storage device:%s" % str(ex))
                    raise ex
            else:
                env[dev.get("label")] = dev.get("device-id")

        return mounted_storage_drive, usb_mountpoints, env

    def provision_env_vars(self, containerRequest, appmanifest, dev_envs, network_info, host_envs=None):
        # Create a wrapper script that starts the container.
        persistent_data_target = appmanifest.resources.get("persistent_data_target", self.data_mount_point)
        env = {
            "CAF_APP_PERSISTENT_DIR": persistent_data_target,
            "CAF_APP_LOG_DIR": os.path.join(persistent_data_target, self._logDirName),
            "CAF_APP_APPDATA_DIR": os.path.join(persistent_data_target, self._appdata_dir),
            "CAF_APP_CONFIG_DIR": os.path.join(persistent_data_target),
            "CAF_APP_USERNAME": "root",
            "CAF_APP_CORE_DIR": self._mnt_corefile_dir,
            "CAF_SYSTEM_UUID": SystemInfo.get_system_uuid(),
            "CAF_SYSTEM_PRODUCT_ID": SystemInfo.get_productid(),
            "CAF_SYSTEM_SERIAL_ID": SystemInfo.get_systemid(),
            "CAF_SYSTEM_NAME": SystemInfo.get_hostname()
        }

        if containerRequest.appconfigname:
            env["CAF_APP_CONFIG_FILE"] = os.path.join(persistent_data_target, os.path.basename(containerRequest.appconfigname))

        # export device id as label required
        env.update(dev_envs)

        conenv = containerRequest.appmanifest.app_env
        if conenv:
            for option, value in conenv.items():
                env[option] = value
        #Updating with dependent service co-ordinates
        env.update(containerRequest.dep_service_coordinates_map)
        env.update(containerRequest.container_resources_allocated_map)

        if (SystemInfo.is_secure_storage_supported() == True):
            env["CAF_SS_PORT"] = SystemInfo.get_secure_storage_port()
            ss_ipv4, ss_ipv6 = SystemInfo.get_secure_storage_ip_addr()
            if ss_ipv4 is not None:
                env["CAF_SS_IP_ADDR"] = ss_ipv4

            if ss_ipv6 is not None:
                env["CAF_SS_IPV6_ADDR"] = ss_ipv6

        if "device-info" in appmanifest.resources:
            device_info_list = appmanifest.resources["device-info"]
            log.debug("Device info list %s", device_info_list)
            env.update(Utils.create_device_info_env_var(device_info_list=device_info_list))

        if "oauth" in appmanifest.resources or "access-control" in appmanifest.resources:
            oauth_env = ContainerUtils.get_oauth_env(containerRequest.containerId, network_info, appmanifest.resources, containerRequest.oauth_default_scopes)
            env.update(oauth_env)

        if "visualization" in appmanifest.resources or "datastore" in appmanifest.resources:
            iox_env = ContainerUtils.get_iox_datastore_env(containerRequest.containerId)
            env.update(iox_env)
        
        if host_envs:
            env.update(host_envs)

        return env

    def create_env_file(self, env, prov_data_dir):
        """
        Create env var file for exporting inside container
        Return wrapper script
        """

        wrapper_script = "#!/bin/sh\n"
        wrapper_script = wrapper_script + "\n".join(["export %s=%s" % (k, v) for (k, v) in env.iteritems()])

        envfile = os.path.join(prov_data_dir, ".env")
        content = "#!/bin/sh\n"
        content = content + "\n".join(["export %s=%s" % (k, v) for (k, v) in env.iteritems()])

        with file(envfile, "w") as f:
            f.write(content)

        log.debug("Provisioned the environment variables to file: %s", envfile)
        return wrapper_script

    def _create_paas_startup_script(self, target_rootfs_dir, containerRequest, wrapper_script, appstart):
        """
        Creates the startup script for paas applications"
        """
        # DHCP is run as part of the rootfs initialization.
        # It should get ip addresses to all the interfaces created within the container.
        watchdog_log = os.path.join("$CAF_APP_LOG_DIR", "watchDog.log")
        wrapper_script += '\n START=$(date +%s);'
        wrapper_script += '\n echo "APP START TIME:$START" >> %s \n' % (watchdog_log)
        if containerRequest.enable_debug:
            wrapper_script += '\n ulimit -c unlimited'

        wrapper_script += '\n exec %s & \n'% appstart
        wrapper_script += '\n echo "App %s started with PID : $!" >> %s \n' % (containerRequest.containerId, watchdog_log)
        wrapper_script += '\n echo "Monitoring this process now" >> %s \n' % (watchdog_log)
        wrapper_script += '\n wait $!'
        wrapper_script += '\n echo "App %s completed with exit code: $?" >> %s \n' % (containerRequest.containerId,watchdog_log)
        wrapper_script += '\n END=$(date +%s);'
        wrapper_script += '\n echo "APP END TIME:$END" >> %s \n' % (watchdog_log)
        wrapper_script += '\n diff=$(($END-$START))'
        wrapper_script += '\n echo "Time taken by App : $(($diff / 60)) minutes and $(($diff % 60)) seconds." '+'>> %s \n' %(watchdog_log)
        if not containerRequest.enable_debug:
            wrapper_script += '\n echo "Stopping the container" >> %s \n' % (watchdog_log)
            wrapper_script += '\n telinit 0 \n'
        log.debug("Container Start Wrapper script : %s", wrapper_script)

        app_startup = os.path.join(target_rootfs_dir, "bin/startcontainer.sh")
        startup = os.path.join(target_rootfs_dir, "etc/init.d/startapp.sh")
        file(app_startup, "w").write(wrapper_script)
        os.chmod(app_startup, 0777)
        startup_script = '#!/bin/sh\n'
        startup_script += 'exec /bin/startcontainer.sh & \n'
        file(startup, "w").write(startup_script)
        os.chmod(startup, 0777)
        #Creating symbolic link in /etc/rc5.d/s100startapp.sh --> /etc/init.d/startapp.sh
        #in order to make the app start as part of rootfs initialization.
        symlink = os.path.join(target_rootfs_dir, "etc/rc5.d/S100startapp.sh")
        if not os.path.isfile(symlink):
            os.symlink("../init.d/startapp.sh", symlink)
            os.chmod(symlink, 0777)


    def _handle_paas_app(self, containerRequest, binds=None):
        appmanifest = containerRequest.appmanifest
        runtime = appmanifest.runtime
        cartridge_list = []
        if containerRequest.cartridge_list:
            cartridge_list= containerRequest.cartridge_list
        resources = containerRequest.appmanifest.resources
        if (cartridge_list is None or len(cartridge_list) == 0) and runtime != 'generic-linux':
            raise AppConfigurationError("Runtime %s:%s is unsupported!" %
                                        (appmanifest.runtime, appmanifest.runtime_version))

        # Extract the base rootfs
        rootfs_cart=None
        cart_layerspec_list=[]
        for c in cartridge_list:
            if c.type == "baserootfs":
                rootfs_cart = c
            cart_layerspec_list.append((c, self.cartridge_permission))
                
        if rootfs_cart is None:
            log.error("Base rootfs not found")
            raise AppConfigurationError("Base rootfs not found")

        rootfs = rootfs_cart.get_location()

        log.debug("using rootfs: %s" , rootfs)

        mem = resources.get("memory", None)
        if mem is None:
            log.error("Missing memory in the resources for container")
            raise ValueError("Missing memory in the resources for container")

        # Libvirt needs this in KB
        mem = int(mem) * 1024
        cpu = resources.get("cpu")
        if cpu is None:
            log.error("Missing cpu in the resources for container")
            raise ValueError("Missing cpu in the resources for container")

        vcpu = resources.get('vcpu', 1)

        startup = appmanifest.startup.get("target")
        
        chardev_list = resources.get('chardev', None)
        log.debug("chardev_list %s", chardev_list)

        # Create repo for this container
        container_dir = os.path.join(self.rootPath, containerRequest.containerId)

        if not os.path.isdir(container_dir):
            os.makedirs(container_dir)

        target_rootfs_dir = os.path.join(container_dir, "rootfs_mnt")
        if os.path.exists(target_rootfs_dir):
            log.debug("target rootfs already created")

        app_mount  = os.path.join(container_dir, "app_mount")
        if not os.path.isdir(app_mount):
            os.makedirs(app_mount)

        app_ext2 = containerRequest.connectorArchivePath
        try:
            #Security labels and attributes
            sec_attr, security_str = self.get_security_attributes(containerRequest.containerId, resources)

            #Mount app
            if not Utils.ismount_exists(app_mount):
                out, rc = mountext2(app_ext2, app_mount)
                if rc != 0:
                    log.error("Error in mounting staged  app ext2 image: %s", str(out))
                    self._remove_container_data(containerRequest.containerId, AppType.PAAS)
                    raise AppInstallationError("Error in creating app ext2 image: %s", str(out))
                log.debug("Mounted app directory for container %s at %s" %
                        (containerRequest.containerId, app_mount))

            disk = resources.get("disk")
            datadir = self._create_data_dir(containerRequest.disk_ext, containerRequest.containerId, disk, security_str)

            rfsComposer = RFSComposer.get_rfs_composer(self.rfs_type)

            persistent_data_target = appmanifest.resources.get("persistent_data_target", self.data_mount_point)
            data_specs = {"src":datadir, "dst":persistent_data_target, "perm":"rw", "disk_file":containerRequest.disk_ext}
            app_specs = {"src":app_mount, "dst":self.app_mount_point, "perm":"ro", "disk_file":app_ext2}
            compose_rootfs_dict = rfsComposer.compose_rootfs(
                                    target_rootfs_dir = target_rootfs_dir,
                                    app_specs=app_specs,
                                    data_specs=data_specs,
                                    cartridge_specs_list=cart_layerspec_list,
                                    dst_cartridge_dir=self._cartridge_mount_point,
                                    sec_info=sec_attr,
                                    target_rootfs_disk=os.path.dirname(os.path.abspath(app_ext2)))
        except Exception as e:
            log.exception("Error while composing rootfs %s" % str(e))
            self._remove_container_data(containerRequest.containerId, AppType.PAAS)
            raise AppInstallationError("Error while creating container %s" % e)
        
        try:
            # Provision app config file if it exists
            appcfgfile = containerRequest.appconfigname

            if "data_dir" in compose_rootfs_dict:
                prov_data_dir = compose_rootfs_dict["data_dir"]
                self.provision_log_config_appdata(prov_data_dir, appcfgfile)

            if "app_dir" in compose_rootfs_dict:
                appstart = os.path.join(compose_rootfs_dict["app_dir"], containerRequest.startCommand)
            else:
                appstart = os.path.join("/", containerRequest.startCommand)

            if "upperdir" in compose_rootfs_dict and compose_rootfs_dict["upperdir"] is not None:
                upperdir = compose_rootfs_dict["upperdir"]
            else:
                upperdir = data_specs["src"]
            #Provision core dir
            app_coredir = self._create_core_dir(containerRequest.containerId, self._core_dir)

            app_core_dir = os.path.join(target_rootfs_dir, "tmp", "cores")
            if not os.path.exists(app_core_dir):
                os.makedirs(app_core_dir)


            #Copy configuration files from rootfs cartridges --
            #Due to behavior of user namespaces and Smack, shared resources cannot be altered or copied-up
            #on demand by container. For now, we assume a list of the most common files and locations
            #that may need changes and copy to data partition.
            #Not required for COPY based rootfs

            if self.rfs_type is not RFSComposer.TYPE_COPY:
                rootfs_cfg_files = [ "etc/shadow", \
                                     "etc/securetty", \
                                     "etc/hosts", \
                                     "etc/hostname", \
                                     "etc/passwd", \
                                     "etc/ssh/sshd_config", \
                                     "sbin/halt.sysvinit" ]

                rootfs_cfg_fldrs = [ "etc", \
                                     "etc/init.d", \
                                     "etc/ssh", \
                                     "home", \
                                     "home/root", \
                                     "sbin", \
                                     "var", \
                                     "var/volatile", \
                                     "var/volatile/tmp", \
                                     "opt", \
                                     "tmp", \
                                     "mnt", \
                                     "usr" ]

                for fldr in rootfs_cfg_fldrs:
                    if not os.path.exists(os.path.join(upperdir,fldr)):
                        os.makedirs(os.path.join(upperdir,fldr))
                for files in rootfs_cfg_files:
                    #Only copy if it doesn't exist already on data and source exists on rootfs
                    if not os.path.exists(os.path.join(upperdir,files)) and \
                           os.path.exists(os.path.join(rootfs,files)):
                        shutil.copy(os.path.join(rootfs,files),os.path.join(upperdir,files))
                    
            out, rc = self.PDHook.call_app_lifecycle_hook(self._container_type, 
                                         self.PDHook.PRE_NETWORK_ACTIVATE, {},
                                         containerRequest.containerId, 
                                         target_rootfs_dir, datadir,
                                         str(appmanifest.network),
                                         str(resources))

            log.debug("Parsing app's network requirements..")
            network_info, libvirt_network_list = self.setup_app_network(appmanifest, containerRequest.containerId)

            # Setup /etc/network/interfaces file
            content = self.get_network_interfaces_content(appid=containerRequest.containerId,
                                                          source_network_list=libvirt_network_list)

            self._add_dns_entry(target_rootfs_dir, libvirt_network_list)

            enf = os.path.join(target_rootfs_dir, "etc/network/interfaces")
            with file(enf, "w") as f:
                f.write(content)
                log.debug("Provisioned /etc/network/interfaces file with %s", content)

            # Setup Device requirements 
            libvirt_device_list, dev_list = self._get_device_list(appmanifest.devices)
            log.debug("Device  List: %s", libvirt_device_list)

            # libvirt filesystem mounting list
            mount_list = []

            rc, dev_envs = self._checking_mountable_devices(containerRequest.containerId,
                                AppType.PAAS, sec_attr, dev_list, mount_list, target_rootfs_dir)

            #Provision host mount points
            host_mount_list = self.host_mount_paths(resources, containerRequest.containerId)
            log.debug("host mounts: %s " % (host_mount_list))


            env = self.provision_env_vars(containerRequest, appmanifest, dev_envs, network_info)
            log.debug("App env variable list %s", env)

            # Create a wrapper script that starts the container.
            wrapper_script = self.create_env_file(env, prov_data_dir)



            cartridge_mount_dict = compose_rootfs_dict.get("cartridge_mount_dict", {}) 
            for cartridge in cartridge_list:
                if cartridge.id in cartridge_mount_dict:
                    cart_mnt_detail = cartridge_mount_dict[cartridge.id]
                    cartridge_mount = cart_mnt_detail["dst_dir"]
                    cartridge_obj = cart_mnt_detail["cartridge"]
                    wrapper_script = self._add_cartridge_env(wrapper_script, cartridge_obj, cartridge_mount)
                    mount_list.append(cart_mnt_detail) 
        
                else:
                    wrapper_script = self._add_cartridge_env(wrapper_script, cartridge, "/")
            if "extra_mount_list"  in compose_rootfs_dict:
                mount_list.extend(compose_rootfs_dict["extra_mount_list"])    

            #Append the core dir location which needs to be mounted
            mount_list.append({ 
                            "src_dir": app_coredir,
                            "dst_dir": self._mnt_corefile_dir,
                            "perm" : "rw"})

            host_src_list = []
            for host_mount in host_mount_list:
                perm="rw"
                if "readonly" in host_mount and host_mount["readonly"]:
                    perm="r"
                    host_src_list.append(host_mount["src_path"])
                else: 
                    host_src_list.append(os.path.join(host_mount["src_path"], containerRequest.containerId))
                dst_mnt_path =  os.path.join(target_rootfs_dir,host_mount["dst_path"]) 
                if not os.path.exists(dst_mnt_path) :
                    os.makedirs(dst_mnt_path)
                mount_list.append({"src_dir" : host_mount["src_path"], "dst_dir": dst_mnt_path, "perm": perm})
                
            #Create the startup script for paas application
            self._create_paas_startup_script(target_rootfs_dir, containerRequest, wrapper_script, appstart)
    
            out, rc = self.PDHook.call_app_lifecycle_hook(self._container_type,
                                         self.PDHook.PRE_SECURITY_ACTIVATE, {},
                                         containerRequest.containerId,
                                         target_rootfs_dir, datadir, str(resources))

            # Apply security attributes to container's rootfs and related
            # components since proper ownership is required for container to work
            # when security is enabled.
            if sec_attr:
                try:
                    # Components of rootfs for PaaS app
                    file_list = {
                        "recursive": [prov_data_dir],
                        "non-recursive": [app_coredir]
                    }
                    if host_src_list:
                         file_list["recursive"].extend(host_src_list)
                    # Append flat disk mount point if not using layered mount
                    # For unionfs styled mounts, append upper rw layer and
                    # PaaS specific app mount.
                    if self.rfs_type is RFSComposer.TYPE_AUFS or \
                       self.rfs_type is RFSComposer.TYPE_OVERLAY:
                        file_list["recursive"].extend((upperdir, app_mount))
                    else:
                        file_list["recursive"].append(target_rootfs_dir)

                    self.set_container_rootfs_security(file_list, sec_attr)

                    # Since we are modifying the underlying file system after
                    # the union file system has been created, some attributes will
                    # be out of sync and will require a remount to resync. This issue
                    # is only seen on OverlayFS.
                    if self.rfs_type is RFSComposer.TYPE_OVERLAY:
                        rfsComposer.remove_rootfs(target_rootfs_dir)
                        rfsComposer.compose_rootfs(
                                            target_rootfs_dir = target_rootfs_dir,
                                            app_specs=app_specs,
                                            data_specs=data_specs,
                                            cartridge_specs_list=cart_layerspec_list,
                                            dst_cartridge_dir=self._cartridge_mount_point,
                                            sec_info=sec_attr,
                                            target_rootfs_disk=os.path.dirname(os.path.abspath(app_ext2)))
                    # Mark security complete
                    self.set_app_security_complete(containerRequest.containerId)
                except Exception as e:
                    log.exception("Could not apply security on container's rootfs.")
                    raise

            '''
            LibVirt requires in /name format, and by default appends suffix .partition to cgroup name
            '''
            cgroup_parent_name = "/" + containerRequest.cgroup_parent

            syscap = None
            if hasattr(appmanifest, "app_system_capabilities"):
                syscap = appmanifest.app_system_capabilities
            features = self.get_custom_features(containerRequest.containerId, syscap)

            ramfs = appmanifest.resources.get("ramfs", None)

            domainxml = LXCLibvirtXMLGenerator.generate_libvirt_lxc_xml(appid=containerRequest.containerId,
                                                                        memory=mem,
                                                                        startup="/sbin/init",
                                                                        cpu=cpu,
                                                                        cgroup_parent=cgroup_parent_name,
                                                                        rootfs=target_rootfs_dir,
                                                                        vcpu=vcpu,
                                                                        emulator=self._emulator,
                                                                        mount_list=mount_list,
                                                                        source_network_list=libvirt_network_list,
                                                                        device_list = dev_list,
                                                                        sec_attr = sec_attr,
                                                                        chardev_list = chardev_list,
                                                                        features = features,
                                                                        dhcp_filters=self.get_dhcp_filters(),
                                                                        ramfs=ramfs
                                                                        )
            log.debug("Creating container with domainXML : %s" % domainxml)
        except Exception as e:
            msg =("Error while creating domain wrapper script %s for container" % str(e))
            self._handle_container_exception(containerRequest, AppType.PAAS, msg)

        try:
            app_ext2 = containerRequest.connectorArchivePath
            if containerRequest.use_ext4:
                data_ext_fn = containerRequest.containerId + LXCContainerManager.DATA_EXT4_SUFFIX
            else:
                data_ext_fn = containerRequest.containerId + LXCContainerManager.DATA_EXT2_SUFFIX
            data_ext = os.path.join(self.persistent_store, data_ext_fn) 
            d = LXCContainer(containerRequest.containerId, domainxml,
                    self._libvirt_client,
                    AppType.PAAS, 
                    self._connection_str,
                    resources, dst_rootfs = target_rootfs_dir,
                    app_ext2_src=app_ext2, data_ext_src=data_ext, 
                    cartridge_list=cartridge_list, network_info=network_info, 
                    libvirt_network_list=libvirt_network_list,
                   device_list=dev_list, app_env=env, use_ext4=containerRequest.use_ext4, sec_attr=sec_attr)
            self._containers[containerRequest.containerId] = d

            return d
        except Exception as e:
            msg =("Error while creating domain %s "% str(e))
            self._handle_container_exception(containerRequest, AppType.PAAS, msg)


    def _add_cartridge_env(self, wrapper_script, cartridge, cartridge_mount):
        # The wrapper script sets up the right environment for app execution
        # by passing required env variables and starting prerequisite services
        ccmds=[]
        if cartridge is not None:
            ccmds = cartridge.get_commands(cartridge_mount)
        wrapper_script = wrapper_script + "\n" + "\n".join(ccmds)
        return wrapper_script

    def _add_dns_entry(self, rootfs_dir, network_list):
        """
        If app asked for static network, then this will add the DNS entries to the resolv.conf file
        """
        log.debug("Verifying for dns if static ip is requied for any network interface")
        static_required = False
        is_nameserver_added = False

        for network in network_list:
            if network.get('mode') == 'static' or (network.get('ipv4') and network['ipv4'].get('mode') == 'static') or (network.get('ipv6') and network['ipv6'].get('mode') == 'static') :
                static_required = True
        if not static_required:
            log.debug("No interface required static ip  so no dns entry to be added")
            return None
                
        log.debug("Trying to add the dns entry to the app, if there is a static mode asked")
        resolv_path = os.path.join(rootfs_dir, "etc")
        if not os.path.exists(resolv_path):
            os.mkdir(resolv_path)
        resolv_conf = os.path.join(rootfs_dir, 'etc/resolv.conf')
        if os.path.islink(resolv_conf):
            os.remove(resolv_conf)
        with open(resolv_conf, "w") as f:
            for network in network_list:
                if network.get('mode') == 'static' or (network.get('ipv4') and network['ipv4'].get('mode') == 'static') or (network.get('ipv6') and network['ipv6'].get('mode') == 'static') :
                    dns_list = []
                    if network.get('ipv4') and network['ipv4'].get('dns'):
                        if isinstance(network['ipv4'].get('dns'), list):
                            dns_list.extend(network['ipv4'].get('dns'))
                        else:
                            dns_list.append(network['ipv4'].get('dns'))
                    if network.get('ipv6') and network['ipv6'].get('dns'):
                        if isinstance(network['ipv6'].get('dns'), list):
                            dns_list.extend(network['ipv6'].get('dns'))
                        else:
                            dns_list.append(network['ipv6'].get('dns'))
                    for dns in dns_list:
                        f.write('nameserver ' + dns + "\n")
                    if dns_list:
                        is_nameserver_added = True

        if is_nameserver_added:
            return resolv_path
        return None

    def _get_wrapper_script(self, wrapper_script, appstart, cartridge, cartridge_mount):
        # The wrapper script sets up the right environment for app execution
        # by passing required env variables and starting prerequisite services
        ccmds=[]
        if cartridge is not None:
                ccmds = cartridge.get_commands(cartridge_mount)
        ccmds.append("exec %s" % appstart)
        wrapper_script = wrapper_script + "\n" + "\n".join(ccmds)
        log.debug("Container Start Wrapper script : %s", wrapper_script)
        return wrapper_script

    def _mount_storage_drive(self, appid, dev, sec_attr, host_dir):

        """
        The storage device has to be mounted in the host at the mount point
        and in the container rootfs at container_mount point (/mnt/host/usbstorage).
        The corresponding name has to be exposed in the CAF environment variables with the label name given in the app.yaml

        If the host device is already mounted, then unmount the earlier device and mount at a new specified location.

        :param appid:
        :param dev:
        :param sec_attr:
        :param host_dir:
        :return:
        """

        rc = -1
        out = "failed mount_storage_drive ret code"
        devname = ""

        if dev:

            if dev.type == "usbdev":
                devname = dev.storage_params["device_name"]

            elif dev.type == "storage":
                if not dev.dev_partitions:
                    log.error("No Partitions in the stroage device type: %s error: %s"
                              % (rc, str(out)))
                    return out,rc
                devname = dev.dev_partitions[0]

            host_mount = host_dir + devname[devname.rfind("/"):]

            # if mount exist, then unmount it and remount with new path
            if Utils.ismount_exists(host_mount):
                # ErrorCheck: the usb should always be free at this point,
                out, rc = umount("-l", host_mount)
                if rc != 0:
                    log.error("Unmount failed host_mount ret code: %s error: %s"
                              % (rc, str(out)))
                    return out, rc

            if not os.path.exists(host_mount):
                os.makedirs(host_mount)
            else:
                l = os.listdir(host_mount)
                if l:
                    log.error(out)
                    out = "Mounting directory is not empty %s" %host_mount
                    return out, rc

            cmdargs = ["-t", "vfat", "-o"]
            rw = "rw"

            sec_str = self.get_mount_security_attributes(sec_attr)

            # cmdargs += rw,sec_str " " + /dev/sdc1" " + /mnt/host/usbstorage
            cmdargs.extend([rw + sec_str, devname, host_mount])

            # mount -t vfat -o rw,uid=uidtarget,gid=gidtarget,smackfsdef=label,smackfsroot=label, src dest
            log.debug("Mounting with args %s, %s", cmdargs, rw)
            out, rc = mount(cmdargs)

            if rc == 0:
                # Cleanup the mount point on deactivation.
                if dev.type == "usbdev":
                    dev.storage_params["mount_point"] = host_mount

                elif dev.type == "storage":
                    dev.mounted_partitions.append(host_mount)

                dev.available = False
                dev.used_by = appid

        return out, rc

    def _check_valid_mountpoint(self, mountpt, appid, apptype):

        ret = False
        mntpt = mountpt.split("/")
        if mntpt[0] == '':
            mntpt = mntpt[1:]

        log.debug("Mount points: %s, blklist : %s",str(mntpt), self.dirMount_blacklist)
        if mntpt[0] in ("".join(self.dirMount_blacklist.split())).split(","):
            ret = True
            log.error("Not a valid Mount Point. error")
            self._remove_container_data(appid, apptype)
            msg = "MountPoint specified is not valid/Mountable"
            raise AppInstallationError("Error while creating container %s" % str(msg))
        return ret

    def _remove_host_mounts(self, resources, appid) :
        if resources and "host_mounts" in resources:
            for host_mount in resources["host_mounts"]:
                if host_mount.get("host_mount_path", None):
                    write_host_point = os.path.join(host_mount["host_mount_path"], appid)
                    if os.path.exists(write_host_point) :
                        shutil.rmtree(write_host_point, ignore_errors=True)

    def _checking_mountable_devices(self, appID, apptype, sec_attr, dev_list, mount_list, target_rootfs_dir):

        log.debug("dev_list Device  List: %s", dev_list)

        usb_dev_type = "usbdev"
        storage_type = "storage"
        err_msg = ""
        rc = 0
        env = {}

        for app_dev in dev_list:
            label = app_dev.get("label", None)

            if not((app_dev.get("type") == usb_dev_type) and
                           app_dev.get("function")== storage_type) and \
                (app_dev.get("type") != storage_type):
                log.debug("Device type asked %s is not of type usbdev or "
                          "storage device for app %s"%(app_dev.get("type"),appID))

                if label and app_dev.get("device-name", None):
                    if app_dev.get("type") == usb_dev_type:
                        # In case of USB serial device, expose device node
                        # /dev/ttyUSB via env param
                        env[label] = app_dev.get("device-name")
                    else:
                        env[label] = app_dev.get("device-id")

                #we are interested in only mountable storage devices,
                # will pass other device types.
                continue

            # Verifying the device from the CAF.
            hm = HostingManager.get_instance()
            dm = hm.get_service("device-management")
            device = dm.get_device(app_dev.get("type"), app_dev.get("device-id"), app_dev.get("device-name"))
            
            log.debug("Device name selected: %s", app_dev.get("device-name"))

            if not device or not device.available:
                err_msg = "Device type asked %s is not available for %s"%(app_dev.get("type"), appID)
                raise AppInstallationError(err_msg)

            mnt_point = None
            if device.type == usb_dev_type and device.is_storage :
                if device.storage_params["fstype"] != "vfat":
                    err_msg = "FileSystem is not of compatible type vfat  for device %s for app %s" % (app_dev.get("device-id"), appID)
                    raise AppInstallationError(err_msg)
                else:
                    mnt_point = app_dev.get("mount-point",None)

            else:
                if device.type != storage_type:
                    err_msg = "App %s selected device %s is not of storage type" % (appID,app_dev.get("device-id") )
                    raise AppInstallationError(err_msg)
                else:
                    mnt_point = device.mount_point

            if not mnt_point:
                mnt_point = self.usb_storage_container_mount
            else:
                self._check_valid_mountpoint(mnt_point, appID, apptype)

            container_mounted_dir = self.usb_storage_host_mount

            # Mount the usb Mass Storage with attributes along with Security settings
            out, rc = self._mount_storage_drive(appID,device, sec_attr, container_mounted_dir)


            if rc != 0 and app_dev.get("mandatory",None):
                log.error("Storage drive Mounting failed. ret code: %s error: %s"
                                          % (rc, str(out)))

                self._remove_container_data(appID, apptype)
                err_msg = "Error while creating container,cant assign dev %s : %s" % (app_dev.get("type"), str(out))
                raise AppInstallationError(err_msg)

            src_mnt_dir = None

            if device.type == usb_dev_type:
                src_mnt_dir = device.storage_params.get("mount_point",None)
            else: # device.type == storage_type:,
                src_mnt_dir = device.mounted_partitions[0]  #supporting only 1st partition now


            #"device-directory-to-mount" field to mount specific dir in storage drive
            dev_dir = app_dev.get("device-directory-to-mount")
            if dev_dir is not None:
                #if the device directory field doesnt exist,
                # then mount from the USB root drive
                host_full_mnt_path = src_mnt_dir + "/" + dev_dir
                if os.path.exists(host_full_mnt_path):
                    src_mnt_dir = host_full_mnt_path

            log.debug("Mount storage details: %s, %s", src_mnt_dir, mnt_point)

            mount_list.append({"src_dir": src_mnt_dir, "dst_dir": mnt_point, "perm": "rw"})


            # creating the target RootFS dir for bind mounting
            target_dir = os.path.normpath(target_rootfs_dir +
                                              mnt_point)
            if not os.path.exists(target_dir):
                os.makedirs(target_dir)

            if label and mnt_point:
                env[label] = mnt_point

            log.debug("Successful in Mount storage device %s for app %s"%(app_dev,
                                                                      appID))

        return rc, env


    def _handle_lxc_app(self, containerRequest, binds=None):
        appmanifest = containerRequest.appmanifest
        resources = containerRequest.appmanifest.resources
        mem = resources.get('memory')
        if mem is None:
            mem = 10

        # Libvirt needs this in KB
        mem = int(mem) * 1024
        cpu = resources.get('cpu')
        vcpu = resources.get('vcpu', 1)
        startup = appmanifest.startup.get("target")

        app_ext2 = Utils.normalize_artifact_path(appmanifest.startup.get("rootfs"),
                                                 containerRequest.connectorArchivePath)

        chardev_list = resources.get('chardev', None)
        log.debug("chardev_list %s", chardev_list)

        # Create repo for this container
        container_dir = os.path.join(self.rootPath, containerRequest.containerId)

        if not os.path.isdir(self.rootPath):
            os.makedirs(self.rootPath)

        if not os.path.isfile(app_ext2):
            log.error("App ext2 disk not found! : %s" % app_ext2)
            self._remove_container_data(containerRequest.containerId, AppType.LXC)
            raise ValueError("App ext2 disk not found : %s!" % app_ext2)

        try:
            # Get the security label if applicable
            sec_attr, security_str = self.get_security_attributes(containerRequest.containerId, resources)
 
            # For lxc style applications, we will simply mount app.ext2 to "/"
            # And data disk to "/data"
            disk = resources.get("disk")
            datadir = self._create_data_dir(
                containerRequest.disk_ext,
                containerRequest.containerId,
                disk,
                security_str)

            target_rootfs_dir = os.path.join(container_dir, "rootfs_mnt")

            if not Utils.ismount_exists(target_rootfs_dir) :
                if not os.path.exists(target_rootfs_dir):
                    os.makedirs(target_rootfs_dir)

                log.debug("Mounting rootfs. ext2: %s, mountpoint: %s", app_ext2, target_rootfs_dir)
                #cmdargs = "-t ext2 -o loop,rw"
                cmdargs = ["-t", "ext2", "-o"]
                rw = "loop,rw,noatime"
                if security_str:
                   #cmdargs += "," + security_str
                    rw = rw + "," + security_str
                cmdargs.append(rw)

                #cmdargs += " " + app_ext2 + " " + target_rootfs_dir
                cmdargs.extend([app_ext2, target_rootfs_dir])

                out, rc = mount(cmdargs)

                if rc != 0:
                    log.error("Rootfs mount failed. ret code: %s error: %s"
                                % (rc, str(out)))
                    self._remove_container_data(containerRequest.containerId, AppType.LXC)
                    raise AppInstallationError("Error while creating container : %s", str(out))

            # Data disk has to be mounted at data_mount_point in the target rootfs
            prov_data_dir = datadir

            appcfgfile = containerRequest.appconfigname
            #Provision log config and data directory inside data
            self.provision_log_config_appdata(prov_data_dir, appcfgfile)



            #Provision core dir
            app_coredir = self._create_core_dir(containerRequest.containerId, self._core_dir)

            #Provision host mount points
            host_mount_list = self.host_mount_paths(resources, containerRequest.containerId)
            log.debug("host mounts: %s host " % (host_mount_list))
            persistent_data_target = resources.get("persistent_data_target", self.data_mount_point)
            mount_list = []
            mount_list.append({"src_dir" : datadir, "dst_dir": persistent_data_target, "perm": "rw"})
            mount_list.append({"src_dir" : app_coredir, "dst_dir": self._mnt_corefile_dir, "perm": "rw"})

            host_src_list = []
            for host_mount in host_mount_list:
                perm="rw"
                if "readonly" in host_mount and host_mount["readonly"]:
                    perm="r"
                    host_src_list.append(host_mount["src_path"])
                else: 
                    host_src_list.append(os.path.join(host_mount["src_path"], containerRequest.containerId))
                mount_list.append({"src_dir" : host_mount["src_path"], "dst_dir": host_mount["dst_path"], "perm": perm})

            conenv = containerRequest.appmanifest.app_env

            out, rc = self.PDHook.call_app_lifecycle_hook(self._container_type,
                                         self.PDHook.PRE_SECURITY_ACTIVATE,
                                         conenv,
                                         containerRequest.containerId,
                                         target_rootfs_dir, datadir, str(resources))

            out, rc = self.PDHook.call_app_lifecycle_hook(self._container_type,
                                         self.PDHook.PRE_NETWORK_ACTIVATE,
                                         conenv,
                                         containerRequest.containerId,
                                         target_rootfs_dir, datadir,
                                         str(appmanifest.network),
                                         str(resources))

            # Understand the app's network needs
            log.debug("Parsing app's network requirements..")
            network_info, libvirt_network_list = self.setup_app_network(appmanifest, containerRequest.containerId)
            self._add_dns_entry(target_rootfs_dir, libvirt_network_list)

            # App is expected to bring the right /etc/network/interfaces file (or equivalent)

            # Apply security attributes to container's rootfs and related
            # components since proper ownership is required for container to work
            # when security is enabled.
            if sec_attr and not sec_attr["applied"]:
                try:
                    # Components of rootfs for standard LXC app
                    non_rec_list = [app_coredir]
                    rec_list =  [target_rootfs_dir, prov_data_dir]
                    if host_src_list:
                        rec_list.extend(host_src_list)
                    file_list = {
                        "recursive": rec_list,
                        "non-recursive": non_rec_list
                    }
                    self.set_container_rootfs_security(file_list, sec_attr)

                    # Mark security complete
                    self.set_app_security_complete(containerRequest.containerId)
                except Exception as e:
                    log.exception("Could not apply security on container's rootfs.")
                    raise


            # Setup Device requirements
            libvirt_device_list, dev_list = self._get_device_list(appmanifest.devices)
            log.debug("Libvirt Device  List: %s", libvirt_device_list)


            rc, dev_envs = self._checking_mountable_devices(containerRequest.containerId,
                                                AppType.LXC,sec_attr,dev_list,mount_list,target_rootfs_dir)

            filesystem = None
            if 'filesystem' in appmanifest.resources:
                filesystem = appmanifest.resources['filesystem']

            ramfs = appmanifest.resources.get('ramfs', None)

            syscap = None
            if hasattr(appmanifest, "app_system_capabilities"):
                syscap = appmanifest.app_system_capabilities
            features = self.get_custom_features(containerRequest.containerId, syscap)

            cgroup_parent_name = "/" + containerRequest.cgroup_parent
            domainxml = LXCLibvirtXMLGenerator.generate_libvirt_lxc_xml(
                appid=containerRequest.containerId,
                memory=mem,
                cpu=cpu,
                cgroup_parent=cgroup_parent_name,
                rootfs=target_rootfs_dir,
                startup=startup,
                vcpu=vcpu,
                emulator=self._emulator,
                mount_list=mount_list,
                source_network_list=libvirt_network_list,
                device_list=dev_list,
                sec_attr = sec_attr,
                chardev_list = chardev_list,
                features = features,
                dhcp_filters=self.get_dhcp_filters(),
                filesystem = filesystem,
                ramfs= ramfs
            )

            log.debug("Creating container with domainXML : %s" % domainxml)

            # Provision the environment variables
            env = self.provision_env_vars(containerRequest, appmanifest, dev_envs, network_info)
            log.debug("App env variable list %s", env)

            # Create a wrapper script that starts the container.
            wrapper_script = self.create_env_file(env, prov_data_dir)

        except Exception as e:
            msg =("Error while creating domain wrapper script %s "% str(e))
            self._handle_container_exception(containerRequest, AppType.LXC, msg)

        try:
            if containerRequest.use_ext4:
                data_ext_fn = containerRequest.containerId + LXCContainerManager.DATA_EXT4_SUFFIX
            else:
                data_ext_fn = containerRequest.containerId + LXCContainerManager.DATA_EXT2_SUFFIX
            data_ext = os.path.join(self.data_root, data_ext_fn)


            d = LXCContainer(containerRequest.containerId, domainxml,
                             self._libvirt_client, AppType.LXC, self._connection_str,
                             resources,
                             dst_rootfs=target_rootfs_dir,
                             data_ext_src=data_ext,
                             cartridge_list=[],
                             network_info=network_info,
                             libvirt_network_list=libvirt_network_list,
                             device_list=dev_list,
                             app_env=env, sec_attr=sec_attr)
            self._containers[containerRequest.containerId] = d

            return d
        except Exception as e:
            msg = ("Error while creating domain %s " % str(e))
            self._handle_container_exception(containerRequest, AppType.LXC, msg)


    def _handle_dns_entries(self, target_rootfs, sec_attr):
        """
        Handles creating resolv.conf for DNS entries in the container filesystem.
        
        :param target_rootfs: target rootfs path
        :param sec_attr: Security attributes for labelling
        :return: -
        """
        etc_dir = os.path.join(target_rootfs, "etc")
        if not os.path.isdir(etc_dir):
            os.mkdir(etc_dir)
        r_conf = os.path.join(etc_dir, "resolv.conf")
        if os.path.islink(r_conf):
            log.info("Resolve conf file %s is found in rootfs, so removing it" % r_conf)
            os.remove(r_conf)
        if not os.path.isfile(r_conf):
            with open(r_conf, "w", 0) as f:
                pass
        try:
            if sec_attr:
                self.set_container_rootfs_security({"non-recursive": [r_conf]},
                                                   sec_attr)
        except Exception as e:
            log.exception("Could not apply security on container's rootfs.")


    def _handle_remove_emulation_layers(self, target_rootfs_dir):
        # Remove emulation layers from the mounted rootfs
        iox_dir = os.path.join(target_rootfs_dir, ".iox")
        if os.path.exists(iox_dir):
            shutil.rmtree(iox_dir)
            bin_sh = os.path.join(target_rootfs_dir, "bin/sh")
            if os.path.islink(bin_sh) and os.readlink(bin_sh).endswith((".iox/sh.shim")):
                if os.path.exists(os.path.join(target_rootfs_dir, "bin/busybox.nosuid")):
                    os.symlink("/bin/busybox.nosuid", bin_sh)
                elif os.path.exists(os.path.join(target_rootfs_dir, "bin/dash")):
                    os.symlink("/bin/dash", bin_sh)
                elif os.path.exists(os.path.join(target_rootfs_dir, "bin/tcsh")):
                    os.symlink("/bin/tcsh", bin_sh)
                elif os.path.exists(os.path.join(target_rootfs_dir, "bin/bash")):
                    os.symlink("/bin/bash", bin_sh)
                else:
                    log.debug("creating a sym link /bin/sh -> /bin/busybox")
                    os.symlink("/bin/busybox", bin_sh)

    def _handle_docker_flatrootfs(self, containerRequest, target_rootfs,
                                  repo_rootfs, app_rootfs, security_str):
        """
        Handle application with flat rootfs without any layering.
        Mounts the rootfs with the required label.
        
        Raises MandatoryFileMissingError, AppInstallationError 
        
        :param containerRequest: container request object
        :param target_rootfs: target rootfs path
        :param repo_rootfs: repo path
        :param app_rootfs: app rootfs data path
        :param security_str: security attributes
        :return: out and rc, of the mounted file system
        """
        apptype = AppType.LXC
        # this section handles docker apps with layers already merged into an ext2 file
        app_ext = os.path.join(repo_rootfs, app_rootfs)

        if not os.path.isfile(app_ext):
            msg = "App ext disk not found! : %s" % app_ext
            log.error(msg)
            self._remove_container_data(containerRequest.containerId, apptype)
            raise MandatoryFileMissingError(msg)

        log.debug("Mounting rootfs. ext: %s, mountpoint: %s", app_ext,
                  target_rootfs)

        cmdargs = [ "-o"]
        rw = "loop,rw,noatime"
        if security_str:
            # cmdargs += "," + security_str
            rw = rw + "," + security_str
        cmdargs.append(rw)
        cmdargs.extend([app_ext, target_rootfs])
        out, rc = mount(cmdargs)

        if rc != 0:
            log.error("Rootfs mount failed. ret code: %s error: %s" % (rc, str(out)))
            self._remove_container_data(containerRequest.containerId, apptype)
            raise AppInstallationError("Error while creating container : %s", str(out))


    def _evaluateandcreate_rw_layers(self, containerRequest, total_app_layers_size, security_str):
        """
        Handle creating rw layer for the Docker layering 
        
        Raises AppInstallationError 
        
        :param containerRequest: object
        :param total_app_layers_size: application layer size in total
        :param security_str: security attribute for labelling
        :return: upper_rw_layer -> created upper layer
        """

        from appfw.runtime.resourcemanager import ResourceManager
        rsmgr = ResourceManager.getInstance()
        try:
            container_size = self._get_app_container_size(containerRequest)
            self.rw_layer_size = self._evaluate_rw_layer_size(total_app_layers_size,
                                                              self.rw_layer_to_code_size_ratio,
                                                              self.min_rw_layer_size,
                                                              self.max_rw_layer_size,
                                                              app_container_size=container_size)
            log.debug("Final upper rw layer size is %s, app requested for %s",
                      self.rw_layer_size, container_size)

            rw_disk = rsmgr.create_rw_layer_disk(self.iox_layer_store,
                                                 containerRequest.containerId,
                                                 size=self.rw_layer_size)
            upper_rw_layer = self._create_rw_layer_dir(rw_disk, containerRequest.containerId,
                                                       size=self.rw_layer_size,
                                                       security_str=security_str)
            return upper_rw_layer

        except Exception as ex:
            log.error("Failed to create rw layer for docker App %s", str(ex))
            raise AppInstallationError("Error while creating container : %s", str(ex))



    def _create_working_dirs(self, base_dir):
        """
        Creates workdir and upper layer directories on top of base directories
        :param base_dir:
        :return:
        """
        upperdir = os.path.join(base_dir, DOCKER_RW_LAYER_ROOT)
        if not os.path.exists(upperdir):
            os.mkdir(upperdir)

        workdir = os.path.join(base_dir, DOCKER_WORK_DIR)
        if not os.path.exists(workdir):
            os.mkdir(workdir)

        # Add core dump folder to upper layer
        app_core_dir = os.path.join(upperdir, "tmp", "cores")
        if not os.path.exists(app_core_dir):
            os.makedirs(app_core_dir)


        return upperdir,workdir


    def _get_app_container_size(self, containerRequest):
        resources = containerRequest.appmanifest.resources
        app_container_size = resources.get("container-size", None)

        return app_container_size

    def _handle_docker_manifestlayers_(self, containerRequest, target_rootfs,
                               repo_rootfs, app_rootfs, layernames):

        manifest = os.path.join(repo_rootfs, DOCKER_MANIFEST_FILE_NAME)
        layer_manifest = os.path.join(repo_rootfs, DOCKER_LAYERS_MANIFEST_FILE)
        total_app_layers_size = 0

        if os.path.isfile(manifest):
            with open(manifest, "r") as f:
                layers = json.loads(f.read())
            for layer in layers:
                if not os.path.isdir(os.path.join(repo_rootfs, layer)):
                    log.info("Layer path %s is not found in repo" % os.path.join(repo_rootfs, layer))
                    raise KeyError("Layer path %s is not found in repo" % os.path.join(repo_rootfs, layer))
                layer_dir = os.path.join(repo_rootfs, layer)
                total_app_layers_size = total_app_layers_size + \
                                        Utils.get_diskusage(layer_dir)

                layernames.append(os.path.join(repo_rootfs, layer))
        elif os.path.isfile(layer_manifest):
            with open(layer_manifest, "r") as f:
                layers = json.loads(f.read())
            for layer in layers:
                layer_dir = os.path.join(repo_rootfs, layer)
                if os.path.islink(layer_dir):
                    layer_dir = os.readlink(layer_dir)
                layer_cont_dir = os.path.join(layer_dir, LAYER_CONTENTS_DIR)
                if not os.path.isdir(layer_dir) or not os.path.isdir(layer_cont_dir):
                    log.info("Layer path %s is not found in repo" % layer_dir)
                    raise KeyError("Layer path %s is not found in repo" % layer_dir)
                if os.path.exists(os.path.join(layer_cont_dir, LAYER_ARCHIVE_FILE)) and tarfile.is_tarfile(os.path.join(layer_cont_dir, LAYER_ARCHIVE_FILE)):
                    out_put, ret = tar_extractall(os.path.join(layer_cont_dir, LAYER_ARCHIVE_FILE), layer_cont_dir)
                    if ret != 0:
                        log.debug("Extraction of tar file %s using tar binary failed. Reason: %s ret_code: %s" % (
                            os.path.join(layer_cont_dir, LAYER_ARCHIVE_FILE), out_put, ret))
                        with tarfile.open(os.path.join(layer_cont_dir, LAYER_ARCHIVE_FILE), ignore_zeros=True) as tar:
                            Utils.check_for_absolutepaths(tar)
                            tar.extractall(layer_cont_dir)
                    os.remove(os.path.join(layer_cont_dir, LAYER_ARCHIVE_FILE))
                layer_dir = os.path.join(repo_rootfs, layer)
                total_app_layers_size = total_app_layers_size + \
                                        Utils.get_diskusage(layer_dir)
                layernames.append(layer_cont_dir)
        else:
            log.info("There is no manifest file found")
            raise KeyError("There is no manifest file found")

        log.debug("Total app size by calculating size of layers is %s",
                                                total_app_layers_size)

        return total_app_layers_size


    def _handle_docker_layering(self, containerRequest, target_rootfs,
                                repo_rootfs, app_rootfs, layernames,
                                security_str, sec_attr):

        apptype = AppType.DOCKER
        container_dir = os.path.join(self.rootPath, containerRequest.containerId)
        total_app_layers_size = self._handle_docker_manifestlayers_(containerRequest,
                                                                    target_rootfs,
                                                                    repo_rootfs,
                                                                    app_rootfs,
                                                                    layernames)

        if self.rfs_type != RFSComposer.TYPE_COPY:
            upper_rw_layer = self._evaluateandcreate_rw_layers(containerRequest,
                                                               total_app_layers_size,
                                                               security_str)
            upperdir, workdir = self._create_working_dirs(upper_rw_layer)

            # Append the topmost rw layer that we created
            layernames.append(upperdir)
            containerRequest.upperdir = upperdir
            containerRequest.workdir = workdir

        log.debug("Mounting rootfs to mountpoint: %s" % target_rootfs)

        # Get Rootfs composition type (flat, aufs, overlay)
        log.debug("Using ROOTFS COMPOSER of type %s" % self.rfs_type)
        rfsComposer = RFSComposer.get_rfs_composer(self.rfs_type)

        dst_datadir = "/data"

        # Security attributes for layers (including ulayer)
        #
        # TODO: Consolidate security labeling efforts in one area and
        # append new files using the union mount once it's available.
        # We really should not touch any of the lower layers once the
        # union mount has been created or we may get mismatching
        # attributes and other strange behavior.
        if sec_attr and not sec_attr["applied"]:
            # Send entire list of layers to be processed
            log.debug("Applying security attributes to layers %s, "
                      "before performing the union mount." % (layernames))
            try:

                file_list = [workdir]
                file_list.extend(layernames)
                self.set_container_rootfs_security({"recursive": file_list},
                                                   sec_attr)
            except Exception as e:
                log.exception("Could not apply security on container's rootfs.")
                raise

        if self.rfs_type == RFSComposer.TYPE_COPY:
            total_app_size = (total_app_layers_size + self.rw_layer_size) *1024*1024# bytes
            rfsComposer.docker_compose_rootfs(layernames, target_rootfs,
                                              workdir=None, sec_info=sec_attr,
                                              total_size=total_app_size,
                                              target_dir=container_dir)

        else:
            # Create the union mount
            out, rc = rfsComposer.docker_compose_rootfs(layernames, target_rootfs,
                                                        workdir=containerRequest.workdir,
                                                        sec_info=sec_attr)
            if rc != 0:
                msg = "Error while creating container. Rootfs mount failed. " \
                      "ret code: %s error: %s" % (rc, str(out))
                self._handle_container_exception(containerRequest, apptype, msg)

        self._handle_dns_entries(target_rootfs, sec_attr)
        self._handle_remove_emulation_layers(target_rootfs)


    def _handle_docker_readonly_rootfs(self, containerRequest, target_rootfs,
                                       repo_rootfs, app_rootfs, mount_points,
                                       security_str, sec_attr):


        apptype = AppType.DOCKER
        container_dir = os.path.join(self.rootPath, containerRequest.containerId)

        #read only mount dir
        ro_mount_dir = os.path.join(container_dir, "romnt")
        if not os.path.exists(ro_mount_dir ):
            os.mkdir(ro_mount_dir)
        else:
            if Utils.ismount_exists(ro_mount_dir):
                log.error("Error!! mount point shouldnt exist here before app is activated")
                out, rc = umount("-l", ro_mount_dir)
                if rc != 0:
                    msg = ("Unmounting failed for data dir. ret code: %s error: %s"
                              % (rc, str(out)))
                    log.error(msg)
                    raise C3Exception(msg)

        app_ro = os.path.join(repo_rootfs, app_rootfs)
        if not os.path.isfile(app_ro):
            msg = ("App img disk not found! : %s" % app_ro)
            log.error(msg)
            self._remove_container_data(containerRequest.containerId, apptype)
            raise MandatoryFileMissingError(msg)

        log.debug("Mounting rootfs readonly: %s, mountpoint: %s",app_ro,
                                                    ro_mount_dir)

        cmdargs = ["-o"]
        options = "loop,ro" + self.get_mount_security_attributes(sec_attr)

        cmdargs.extend([options, app_ro, ro_mount_dir, "nofsck"])
        out, rc = mount(cmdargs)

        if rc != 0:
            msg = "Error while creating container. Rootfs mount failed. ret code: " \
                                        "%s error: %s" % (rc, str(out))
            log.error(msg)
            self._remove_container_data(containerRequest.containerId, apptype)
            raise AppInstallationError(msg)

        mount_points.append(ro_mount_dir)

        layernames = []
        #Dont include RO layer to have security labelling, as it creates
        # COW files in upper layer (Security labels are passed during mount point),
        #only include other layers which needs security labelling explicitly
        seclayernames = []

        layernames.append(ro_mount_dir)
        total_app_layers_size = Utils.get_diskusage(ro_mount_dir)

        log.debug("Total app size by calculating size of layers is %s",
                                                total_app_layers_size)

        if self.rfs_type != RFSComposer.TYPE_COPY:
            upper_rw_layer = self._evaluateandcreate_rw_layers(containerRequest,
                                                               total_app_layers_size,
                                                               security_str)
            upperdir, workdir = self._create_working_dirs(upper_rw_layer)

            # Append the topmost rw layer that we created
            layernames.append(upperdir)
            seclayernames.extend([upperdir,workdir])
            containerRequest.upperdir = upperdir
            containerRequest.workdir = workdir


        if sec_attr :
            # Send entire list of layers to be processed
            log.debug("Applying security attributes to layers %s, "
                      "before performing the union mount." % (seclayernames))
            try:
                self.set_container_rootfs_security({"recursive": seclayernames},
                                                   sec_attr)
            except Exception as e:
                log.exception("Could not apply security on container's rootfs.")
                raise

        log.debug("Using ROOTFS COMPOSER of type %s" % self.rfs_type)
        rfsComposer = RFSComposer.get_rfs_composer(self.rfs_type)

        if self.rfs_type == RFSComposer.TYPE_COPY:
            total_app_size = (total_app_layers_size + self.rw_layer_size) *1024*1024# bytes

            log.debug("Total app size by adding rwlayer is %s",
                      total_app_layers_size)
            rfsComposer.docker_compose_rootfs(layernames, target_rootfs,
                                              workdir=None, sec_info=sec_attr,
                                              total_size=total_app_size,
                                              target_dir=container_dir)

        else:
            # Create the union mount
            out, rc = rfsComposer.docker_compose_rootfs(layernames, target_rootfs,
                                                        workdir=containerRequest.workdir,
                                                        sec_info=sec_attr)
            if rc != 0:
                msg = "Error while creating container. Rootfs mount failed. " \
                      "ret code: %s error: %s" % (rc, str(out))
                self._handle_container_exception(containerRequest, apptype, msg)


        self._handle_dns_entries(target_rootfs, sec_attr)

    def _handle_container_exception(self, containerRequest, apptype, msg):


        resources = containerRequest.appmanifest.resources

        log.exception(msg)
        self._remove_container_data(containerRequest.containerId, apptype)
        self._remove_host_mounts(resources, containerRequest.containerId)
        raise AppInstallationError(msg)

    def _remove_container_mounts(self, appid):

        container = self._containers.get(appid, None)

        if container is not None:
            for mntpt in container.mount_points:
                if Utils.ismount_exists(mntpt):
                    out, rc = umount("-l", mntpt)
                    if rc != 0:
                        msg = ("Unmounting failed for data dir. ret code: %s error: %s"
                                  % (rc, str(out)))
                        log.error(msg)
                        raise C3Exception(msg)
                if os.path.exists(mntpt):
                    shutil.rmtree(mntpt, ignore_errors=True)



    def _get_rootfs_type(self, containerRequest):
        appmanifest = containerRequest.appmanifest
        app_rootfs = appmanifest.startup.get("rootfs")

        rootfstype = None

        if app_rootfs.endswith(".tar"):
            rootfstype = "layers"
        elif app_rootfs.endswith(DOCKER_RW_LAYER_ROOTFS_TYPE):
            rootfstype = "rwimage"
        elif app_rootfs.endswith(".iso"):
            rootfstype = "roimage"
        else:
            raise AppInstallationError("Error:No proper rootfs type specified : %s", str(ex))

        return rootfstype


    def _handle_docker_app(self, containerRequest, binds=None):

        appmanifest = containerRequest.appmanifest
        app_rootfs = appmanifest.startup.get("rootfs")

        apptype = AppType.DOCKER
        resources = containerRequest.appmanifest.resources
        mem = resources.get('memory',10)

        mem = int(mem) * 1024  # For Libvirtin KB
        vcpu = resources.get('vcpu', 1)
        cpu = resources.get('cpu')
        app_container_size = resources.get("container-size", None)
        extractarchive = os.path.join(self._config.get("controller", "repo"),
                                      containerRequest.containerId, USER_EXTRACTED_DIR)

        # Create repo for this container
        container_dir = os.path.join(self.rootPath, containerRequest.containerId)
        upperdir=None
        mount_points = []

        if not os.path.isdir(self.rootPath):
            os.makedirs(self.rootPath)

        try:
            target_rootfs_dir = os.path.join(container_dir, "rootfs_mnt")
            # Get the security label if applicable and apply
            sec_attr, security_str = self.get_security_attributes(
                                        containerRequest.containerId, resources)

            disk = resources.get("disk")
            datadir = self._create_data_dir(containerRequest.disk_ext, containerRequest.containerId,
                                            disk,security_str)
            full_datamount_perm = False
            if appmanifest.startup.get("user", ""):
                full_datamount_perm = True
                os.chmod(datadir, 0o777)
            repo_rootfs_dir = os.path.join(extractarchive, "rootfs")

            total_app_layers_size = 0
            layernames = []

            if not Utils.ismount_exists(target_rootfs_dir):
                if not os.path.exists(target_rootfs_dir):
                    os.makedirs(target_rootfs_dir)
            else:
                raise AppInstallationError("Error!Mount exists for targetroot: %s", str(target_rootfs_dir))

            rootfstype  = self._get_rootfs_type(containerRequest)
            non_composed_rootfs = False
            target_rootfs_labelling = True
            if rootfstype == "layers" :

                self._handle_docker_layering(containerRequest,target_rootfs_dir,
                                             repo_rootfs_dir, app_rootfs,
                                             layernames, security_str, sec_attr)

            elif rootfstype == "rwimage":

                apptype = AppType.LXC
                non_composed_rootfs = True
                upperdir = target_rootfs_dir
                containerRequest.upperdir = upperdir

                self._handle_docker_flatrootfs(containerRequest, target_rootfs_dir,
                                               repo_rootfs_dir, app_rootfs, security_str)

            elif rootfstype == "roimage":
                self._handle_docker_readonly_rootfs(containerRequest,
                                                    target_rootfs_dir,
                                                    repo_rootfs_dir, app_rootfs,
                                                    mount_points,
                                                    security_str, sec_attr)
                # From this point onwards any file write have to be done to target_rootfs_dir
                # instead of upperdir
                upperdir = target_rootfs_dir
                target_rootfs_labelling = False



            prov_data_dir = datadir
            appcfgfile = containerRequest.appconfigname
            #Provision log config and data directory inside data
            self.provision_log_config_appdata(prov_data_dir, appcfgfile, full_datamount_perm)

            # Provision core dir
            app_coredir = self._create_core_dir(containerRequest.containerId, self._core_dir)
            persistent_data_target = resources.get("persistent_data_target", self.data_mount_point)
            mount_list = []
            mount_list.append({"src_dir" : datadir, "dst_dir": persistent_data_target, "perm": "rw"})
            mount_list.append({"src_dir": app_coredir, "dst_dir": self._core_dir, "perm": "rw"})

            #Provision host mount points
            host_mount_list = self.host_mount_paths(resources, containerRequest.containerId)
            log.debug("host mounts: %s" % (host_mount_list))


            host_src_list = []
            for host_mount in host_mount_list:
                perm="r"
                if "readonly" in host_mount and host_mount["readonly"]:
                    host_src_list.append(host_mount["src_path"])
                else: 
                    perm="rw"
                    host_src_list.append(os.path.join(host_mount["src_path"], containerRequest.containerId))
                mount_list.append({"src_dir" : host_mount["src_path"], "dst_dir": host_mount["dst_path"], "perm": perm})

            log.debug("Datadir is %s", datadir)
            log.debug("Data_mount_point is %s", persistent_data_target)
            log.debug("App corddir is %s", app_coredir)
            log.debug("core dir is %s", self._core_dir)

            conenv = containerRequest.appmanifest.app_env

            #Security Attributes have to be passed to any scripts which needs to create files inside container
            sec_str = self.get_mount_security_attributes(sec_attr)
            out, rc = self.PDHook.call_app_lifecycle_hook(self._container_type,
                                            self.PDHook.PRE_SECURITY_ACTIVATE,
                                            conenv,
                                            containerRequest.containerId,
                                            upperdir, datadir, sec_str,
                                            str(resources))

            out, rc = self.PDHook.call_app_lifecycle_hook(self._container_type,
                                              self.PDHook.PRE_NETWORK_ACTIVATE,
                                              conenv,
                                              containerRequest.containerId,
                                              upperdir, datadir, sec_str,
                                              str(appmanifest.network),
                                              str(resources))

            # Understand the app's network needs
            log.debug("Parsing app's network requirements..")
            network_info, libvirt_network_list = self.setup_app_network(appmanifest,
                                                                        containerRequest.containerId)
            added_dns_entry = self._add_dns_entry(target_rootfs_dir, libvirt_network_list)
            if added_dns_entry:
                host_src_list.append(added_dns_entry)
            # Setup Device requirements
            libvirt_device_list, dev_list = self._get_device_list(appmanifest.devices)
            watchdog_log = os.path.join(persistent_data_target, self._logDirName, "watchDog.log")
            log.debug("Libvirt Device  List: %s", libvirt_device_list)
            #interfaces=" ".join(network_info.keys())
            interfaces = ""
            ipv6_req = ""
            for intf in network_info.keys():
                ipv4_disabled=False
                ipv6_disabled=False
                for lw_net in libvirt_network_list:
                    if intf == lw_net['interface_name']:
                        if lw_net.get('ipv4'): 
                            ipv4_disabled = lw_net['ipv4'].get('disabled', False)

                        if lw_net.get('ipv6'): 
                            ipv6_disabled = lw_net['ipv6'].get('disabled', False)
                        
                        break

                log.debug("ipv4_disabled: %s ipv6_disabled=%s ipv6_supported=%s" % (ipv4_disabled, ipv6_disabled, self.ipv6_supported))
                if ((ipv4_disabled is True) and ((ipv6_disabled is True) or (self.ipv6_supported is False))):
                    log.info("ip is disabled for interface:%s" % intf)
                else: 
                    interfaces += intf + " "
                    if network_info[intf].get("ipv6_required"):
                        ipv6_req += 'true '
                    else:
                        ipv6_req += 'false '
            
                
            interfaces = interfaces.strip()
            ipv6_req = ipv6_req.strip()

            if self.ipv6_supported:
                ipv6_supp = 'true'
            else:
                ipv6_supp = 'false'

            rc, dev_envs = self._checking_mountable_devices(containerRequest.containerId, AppType.DOCKER,
                                                sec_attr, dev_list, mount_list, target_rootfs_dir)

            # Provision the environment variables
            env = self.provision_env_vars(containerRequest, appmanifest, dev_envs, network_info)
            log.debug("App env variable list %s", env)

            self.create_env_file(env, prov_data_dir)

            env_content = ""
            env_content = env_content + "\n".join(["export %s=%s" % (k, v) for (k, v) in env.iteritems()])

            # Check if OS mode is enabled for docker containers
            os_mode = appmanifest.startup.get("os-mode", False)
            #All docker wrapper script related binaries and wrapper script itself will be kept here
            iox_binaries_dir = os.path.join(target_rootfs_dir, ".iox")
            if not os.path.isdir(iox_binaries_dir):
                os.mkdir(iox_binaries_dir)
            startup = "/.iox/startcontainer.sh"
            app_target = ""
            user = ""
            group = ""
            currentdir = ""
            args = None
            if os_mode:
                log.debug("OS mode is enabled")
                startup = appmanifest.startup.get("target")
            else:
                startup_section = appmanifest.startup
                user = startup_section.get("user", "")
                group = startup_section.get("group", "")
                currentdir = startup_section.get("workdir", "")
                app_target = containerRequest.target
                args = containerRequest.args
            log.debug("Setting startup to: %s" % startup)
            docker_command_support_in_script = True
            if libvirt.getVersion() >= LIBVIRT_VERSION_SUPPORT_DOKCER_COMMANDS:
                log.info("Libvirt version %s suppoprts docker commands as part of XML tags, so that startup container script will remain intact!")
                docker_command_support_in_script = False
            else:
                log.info("Libvirt version %s doesn't suppoprts docker commands as part of XML tags, so that startup container script will be cahnged to support!")
            # Will return the wrapper script for the container
            app_startup = os.path.join(iox_binaries_dir, "startcontainer.sh")
            if os_mode:
                log.debug("Skipping creation of startup script %s when OS mode is enabled", app_startup)
            else:
                runtime_parameters = [containerRequest.containerId, interfaces, app_target, watchdog_log,
                            containerRequest.enable_debug, env_content, target_rootfs_dir, iox_binaries_dir, ipv6_req, ipv6_supp]
                if docker_command_support_in_script:
                    runtime_parameters.extend([user, group, currentdir, args])

                # Copy usersetup static binary to the app's rootfs which was built using non cisco 
                # Docker image Ex. Alpine. This provides the USER instruction support for non-cisco docker image
                #caf_home_dir = Utils.getRuntimeSourceFolder()
                #iox_home_dir = os.path.split(caf_home_dir)[0] # Get iox home dir
                #static_usersetup_bin = os.path.join(iox_home_dir, "app_tools", "usersetup")
                #app_usersetup = os.path.join(target_rootfs_dir, "bin", "usersetup")
                #if not os.path.exists(app_usersetup):
                #    if os.path.exists(static_usersetup_bin) and os.path.exists(os.path.dirname(app_usersetup)):
                #        shutil.copy2(static_usersetup_bin, app_usersetup)
                wrapper_script = DockerUtils.get_container_wrapper_script(*runtime_parameters)

                with open(app_startup, "w") as f:
                    f.write(wrapper_script)
                os.chmod(app_startup, 0777)
                log.debug("Created startup script %s with content: \n%s", app_startup, wrapper_script)

            # Apply security attributes to container's rootfs and related
            # components since proper ownership is required for container to work
            # when security is enabled.
            if sec_attr:
                try:
                    # in case of union fs based rootfs, chown workdir and upperdir
                    if not non_composed_rootfs:
                        self.set_container_rootfs_security({"recursive": [containerRequest.upperdir,
                                                                          containerRequest.workdir]},
                                                        sec_attr)
                    self.set_container_rootfs_security({"non-recursive": [app_coredir]},
                                                    sec_attr)
                    if not sec_attr["applied"]:
                        # Components of rootfs for Docker-styled app
                        file_list = {
                            "recursive": [prov_data_dir]
                        }
                        if host_src_list:
                            file_list["recursive"].extend(host_src_list)

                        # Append flat disk mount point if not using layered mount
                        # For unionfs styled mounts, append the work directory.
                        if non_composed_rootfs:
                            file_list["recursive"].append(target_rootfs_dir)
                        self.set_container_rootfs_security(file_list, sec_attr)

                         # Mark security complete
                        self.set_app_security_complete(containerRequest.containerId)

                except Exception as e:
                    log.exception("Could not apply security on container's rootfs.")
                    # Rollback file system changes on lower layers since that was an earlier operation
                    # if this was the first iteration
                    if not sec_attr["applied"]:
                        self.set_container_rootfs_security({"recursive": layernames},
                                                            sec_attr,
                                                            True)
                    raise

            #Applying permissions to /tmp dir
            os.chmod(os.path.join(target_rootfs_dir,"tmp"), 0777)

            syscap = None
            if hasattr(appmanifest, "app_system_capabilities"):
                syscap = appmanifest.app_system_capabilities
            features = self.get_custom_features(containerRequest.containerId, syscap)

            cgroup_parent_name = "/" + containerRequest.cgroup_parent
            if not os.path.isdir(os.path.join(self._config.get("controller", "repo"), containerRequest.containerId, '.udhcpc_pids')):
                log.debug("Creating pid repo %s for UDHCPC"%os.path.join(self._config.get("controller", "repo"), containerRequest.containerId, '.udhcpc_pids'))
                os.mkdir(os.path.join(self._config.get("controller", "repo"), containerRequest.containerId, '.udhcpc_pids'))

            ramfs = appmanifest.resources.get("ramfs", None)

            libvirt_xml_kwargs = {
                "appid":containerRequest.containerId,
                "memory":mem,
                "cpu":cpu,
                "cgroup_parent":cgroup_parent_name,
                "rootfs":target_rootfs_dir,
                "vcpu":vcpu,
                "startup":startup,
                "emulator":self._emulator,
                "mount_list":mount_list,
                "source_network_list":libvirt_network_list,
                "device_list":dev_list,
                "sec_attr":sec_attr,
                "features" :features,
                "dhcp_filters":self.get_dhcp_filters(),
                "ramfs":ramfs
            }
            if not docker_command_support_in_script:
                libvirt_xml_kwargs["user"] = user
                libvirt_xml_kwargs["group"] = group
                libvirt_xml_kwargs["currentdir"] = currentdir
                libvirt_xml_kwargs["args"] = args

            domainxml = LXCLibvirtXMLGenerator.generate_libvirt_lxc_xml(**libvirt_xml_kwargs)
            log.debug("Creating container with domainXML : %s" % domainxml)


        except Exception as e:
            msg = ("Error while creating domain wrapper script %s" % str(e))
            self._handle_container_exception(containerRequest, apptype, msg)


        try:
            if containerRequest.use_ext4:
                data_ext_fn = containerRequest.containerId + LXCContainerManager.DATA_EXT4_SUFFIX
            else:
                data_ext_fn = containerRequest.containerId + LXCContainerManager.DATA_EXT2_SUFFIX
            data_ext = os.path.join(self.data_root, data_ext_fn) 

            d = LXCContainer(containerRequest.containerId, domainxml,
                             self._libvirt_client, AppType.DOCKER, self._connection_str,
                             resources,
                             dst_rootfs=target_rootfs_dir,
                             data_ext_src=data_ext,
                             cartridge_list=[],
                             network_info=network_info,
                             libvirt_network_list=libvirt_network_list,
                             device_list=dev_list,
                             app_env=env,
                             sec_attr=sec_attr,
                             non_composed_rootfs=non_composed_rootfs,
                             os_mode=os_mode,
                             mount_points = mount_points)
            self._containers[containerRequest.containerId] = d

            return d
        except Exception as e:
            msg = ("Error while creating domain %s" % str(e))
            self._handle_container_exception(containerRequest, apptype, msg)


    def create(self, containerRequest):
        """
        Load the image into local repo. If successful, create a container
        """
        appmanifest = containerRequest.appmanifest
        apptype = appmanifest.apptype
        resources = appmanifest.resources
        persistent_data_target = resources.get("persistent_data_target", self.data_mount_point)
        binds = {os.path.join(self.data_root, containerRequest.containerId) : persistent_data_target}
        if apptype not in [AppType.LXC, AppType.PAAS, AppType.DOCKER]:
            raise AppInstallationError("Unsupported app type %s" % apptype)
        d = None
        try:
            d = self._libvirt_client.lookupByName(containerRequest.containerId)
        except Exception:
            pass

        # Check if OS mode is enabled for docker containers
        os_mode = appmanifest.startup.get("os-mode")

        runtime_uuid = Utils.get_app_uuid_from_repo(containerRequest.containerId)

        container_dir = os.path.join(self.rootPath, containerRequest.containerId)
        target_rootfs_dir = os.path.join(container_dir, "rootfs_mnt")

        if containerRequest.use_ext4:
            data_ext_fn = containerRequest.containerId + LXCContainerManager.DATA_EXT4_SUFFIX
        else:
            data_ext_fn = containerRequest.containerId + LXCContainerManager.DATA_EXT2_SUFFIX

        data_ext = os.path.join(self.data_root, data_ext_fn) 
        data_dir = os.path.join(self.data_root, containerRequest.containerId, persistent_data_target.lstrip("/"))
        host_data_mount = os.path.join(self.data_root, containerRequest.containerId)
        mount_points = [target_rootfs_dir, host_data_mount]
        # Check if the container artifacts already exist and if we need to reconcile
        reconcile = False

        if (d is not None and runtime_uuid == d.UUIDString()):
            # Container is still defined in libvirt
            log.info("Container %s is defined in libvirt domain", containerRequest.containerId)
            log.info("Checking to see if mount points required by the container exists...")
            reconcile = True
            for m in mount_points:
                if not Utils.ismount_exists(m):
                    reconcile = False
                    log.info("Required mount point %s doesn't exist. Cannot reconcile!", m)
                    break

            log.info("Container domain %s and all required mount points %s exists..Will reconcile..", containerRequest.containerId, mount_points)

        if not reconcile:
            if apptype == AppType.LXC:
                return self._handle_lxc_app(containerRequest, binds)
            elif apptype == AppType.PAAS:
                return self._handle_paas_app(containerRequest, binds)
            elif apptype == AppType.DOCKER:
                return self._handle_docker_app(containerRequest, binds)
        else:
            try:
                log.debug("Reconciling..")
                env_dict = {}
                conenv = containerRequest.appmanifest.app_env
                if conenv:
                    for option, value in conenv.items():
                        env_dict[option] = value
                env_file = os.path.join(data_dir, ".env")
                if os.path.exists(env_file):
                    env_dict = Utils.parse_envfile(env_file)
                # for lxc and docker style apps there is a slight directory changes
                elif os.path.isfile(os.path.join(self.data_root, containerRequest.containerId, ".env")):
                    env_dict = Utils.parse_envfile(os.path.join(self.data_root, containerRequest.containerId, ".env"))
                network_info, libvirt_nw_list = self.setup_app_network(containerRequest.appmanifest, containerRequest.containerId, True)
                libvirt_device_list, dev_list = self._get_device_list(appmanifest.devices, reconcile=True)
                con = LXCContainer(containerRequest.containerId, d.XMLDesc(),
                                 self._libvirt_client, apptype, self._connection_str,
                                 containerRequest.appmanifest.resources,
                                 dst_rootfs=target_rootfs_dir,
                                 data_ext_src=data_ext,
                                 cartridge_list=containerRequest.cartridge_list,
                                 network_info=network_info,
                                 libvirt_network_list=libvirt_nw_list,
                                 device_list=dev_list,
                                 app_env=env_dict,
                                 os_mode=os_mode,
                                 reconcile=True)
                self._containers[containerRequest.containerId] = con
                log.debug("Successfully reconciled the container")
                return con
            except Exception as ex:
                log.exception("Error while reconciling the container %s, cause %s"%(containerRequest.containerId, ex.message))
                raise Exception("Error while reconciling the container %s, cause %s"%(containerRequest.containerId, ex.message))

    def get_container_data_root_path(self):
        return self.data_root

    def get_container_logDir(self):
        return self._logDirName

    def get_container_coreDir(self):
        return self._core_dir

    def get_container_driver_logDir(self):
        return self._lxcLogDirName

    def get_persistent_datadir_target(self, containerId):
        container = self._containers.get(containerId)
        return container._resources.get("persistent_data_target", self.data_mount_point.lstrip("/"))

    def _remove_container_rw_layer(self, containerId, apptype):
        from appfw.runtime.resourcemanager import ResourceManager
        rsmgr = ResourceManager.getInstance()
        if apptype is AppType.DOCKER:
            if Utils.ismount_exists(os.path.join(self._iox_layer_store, containerId)):
                out, rc = umount("-l",  os.path.join(self._iox_layer_store, containerId))
                if rc != 0:
                    log.error("Unmounting failed for data dir. ret code: %s error: %s"
                                    % (rc, str(out)))
                    raise C3Exception("Unmounting failed for an data dir. ret code: %s error: %s"
                                    % (rc, str(out)))
            if os.path.exists(os.path.join(self._iox_layer_store, containerId)):
                shutil.rmtree(os.path.join(self._iox_layer_store, containerId), ignore_errors=True)
            rsmgr.remove_rw_layer_disk(self.iox_layer_store, containerId)
        elif apptype is AppType.PAAS and self.rfs_type is RFSComposer.TYPE_OVERLAY:
            cshared = os.path.join(self.data_root, containerId, DOCKER_RW_LAYER_ROOT)
            data_dir = os.path.join(self.data_root, containerId, DOCKER_RW_LAYER_ROOT, self.get_persistent_datadir_target(containerId))
            try:
                exclude_dirs = []
                exclude_dirs.append(data_dir)
                Utils.empty_dir(cshared, exclude_dirs=exclude_dirs)
                os.system("sync")
            except Exception as ex:
                log.exception("Exception %s while cleaning rw layer for app", str(ex))
        else:
            return


    def _remove_container_data(self, containerid, apptype):
        # Remote the container repo directory
        container_dir = os.path.join(self.rootPath, containerid)
        target_rootfs_dir = os.path.join(container_dir, "rootfs_mnt")

        if apptype == AppType.PAAS:
            log.info("Going to remove rootfs and app for paas style app")

            if os.path.exists(target_rootfs_dir):
                if Utils.ismount_exists(target_rootfs_dir):
                    rfsComposer = RFSComposer.get_rfs_composer(self.rfs_type)
                    rfsComposer.remove_rootfs(target_rootfs_dir)

            app_mnt_dir = os.path.join(container_dir, "app_mount")
            if os.path.exists(app_mnt_dir):
                if Utils.ismount_exists(app_mnt_dir):
                    #Remove app mount and ext2 app image
                    out, rc = umount("-l",  app_mnt_dir)
                    if rc != 0:
                        log.error("Unmounting failed for an app. ret code: %s error: %s"
                                        % (rc, str(out)))
            if self.rfs_type is RFSComposer.TYPE_OVERLAY:
                self._remove_container_rw_layer(containerid, AppType.PAAS)

        elif apptype == AppType.LXC:
            if Utils.ismount_exists(target_rootfs_dir):
                log.debug("Unmounting rootfs mountpoint %s" % target_rootfs_dir)
                out, rc = umount("-l", target_rootfs_dir)
                if rc != 0:
                    log.error("Unmount failed ret code: %s error: %s"
                                        % (rc, str(out)))
                    raise C3Exception("Unmount failed ret code: %s error: %s"
                                        % (rc, str(out)))

            log.debug("Deleting dir %s" % target_rootfs_dir)
            if (os.path.exists(target_rootfs_dir)):
                shutil.rmtree(target_rootfs_dir, ignore_errors=True)

        elif apptype == AppType.DOCKER:
            log.info("Going to remove rootfs and app for docker style app")
            if os.path.exists(target_rootfs_dir):
                if Utils.ismount_exists(target_rootfs_dir):
                    rfsComposer = RFSComposer.get_rfs_composer(self.rfs_type)
                    rfsComposer.remove_rootfs(target_rootfs_dir)
            self._remove_container_rw_layer(containerid, AppType.DOCKER)
            self._remove_container_mounts(containerid)

        else:
            raise ValueError("Unsupported app type %s" % apptype)

        if Utils.ismount_exists(self.usb_storage_host_mount):
            log.debug("Unmounting usb_storage_host_mount mountpoint %s" % self.usb_storage_host_mount)
            out, rc = umount("-l", self.usb_storage_host_mount)
            if rc != 0:
                log.error("Unmount failed ret code: %s error: %s"
                          % (rc, str(out)))
                raise C3Exception("Unmount failed ret code: %s error: %s"
                                  % (rc, str(out)))
        log.info("Going to remove rootfs and app for docker style app")
        shutil.rmtree(container_dir, ignore_errors=True)
        log.debug("Deleted container data at : %s", container_dir)

        # Unmount data directory created for this container
        log.debug("unmounting the persistent data dir for container %s , data root %s", containerid, self.data_root)
        if Utils.ismount_exists(os.path.join(self.data_root, containerid)):
            out, rc = umount("-l",  os.path.join(self.data_root, containerid))
            if rc != 0:
                log.error("Unmounting failed for data dir. ret code: %s error: %s"
                                % (rc, str(out)))
                raise C3Exception("Unmounting failed for an data dir. ret code: %s error: %s"
                                % (rc, str(out)))
        shutil.rmtree(os.path.join(self.data_root, containerid), ignore_errors=True)
        #Remove core dir for an app
        shutil.rmtree(os.path.join(self._core_dir, containerid), ignore_errors=True)

    def get_app_config_path(self, containerId):
        container = self._containers.get(containerId)
        if container is not None:
            apptype = container.apptype

            cshared = os.path.join(self.data_root, containerId)

            # If it is PaaS and AUFS mounted, the actual directory will be inside the data disk
            if self.rfs_type is RFSComposer.TYPE_AUFS and apptype == AppType.PAAS:
                cshared = os.path.join(cshared, self.get_persistent_datadir_target(containerId))

            if self.rfs_type is RFSComposer.TYPE_OVERLAY and apptype == AppType.PAAS:
                cshared = os.path.join(cshared, DOCKER_RW_LAYER_ROOT, self.get_persistent_datadir_target(containerId))

            cfgfile = os.path.join(cshared, Utils.find_app_config_filename(cshared))
            if os.path.isfile(cfgfile):
                return cfgfile
        return ""

    def get_app_config(self, containerId):
        cfgfile = self.get_app_config_path(containerId)
        if os.path.isfile(cfgfile):
            data = file(cfgfile, "r").read()
            return data
        else:
            return None

    def set_app_config(self, containerId, content):
        container = self._containers[containerId]
        apptype = container.apptype

        cshared = os.path.join(self.data_root, containerId)

        if self.rfs_type is RFSComposer.TYPE_AUFS and apptype == AppType.PAAS:
            cshared = os.path.join(cshared, self.get_persistent_datadir_target(containerId))

        if self.rfs_type is RFSComposer.TYPE_OVERLAY and apptype == AppType.PAAS:
            cshared = os.path.join(cshared, DOCKER_RW_LAYER_ROOT, self.get_persistent_datadir_target(containerId))

        cfgfile = os.path.join(cshared, Utils.find_app_config_filename(cshared))
        if os.path.isfile(cfgfile):
            with open(cfgfile, "w") as f:
                f.write(content)
            log.debug("Appconfig file content updated!")
        else:
            log.debug("No appConfig file has found so creating the new one with name %s" % APP_CONFIG_NAME_2)
            cfgfile = os.path.join(cshared, APP_CONFIG_NAME_2)
            with open(cfgfile, "w") as f:
                f.write(content)
            hm = HostingManager.get_instance()
            sc = hm.get_service("security-management")
            if sc is None:
                log.exception("Security-Management subsystem is not available")
                raise Exception("Security-Management subsystem is not available")
            sec_attr = sc.get_app_security_config(containerId)
            if sec_attr:
                Utils.apply_rootfs_attributes(sec_attr, cfgfile)
            log.debug("New app config file %s created and contents are updates" % APP_CONFIG_NAME_2)


    def add_data_file(self, containerId, data_file_name, data_file):
        container = self._containers[containerId]
        apptype = container.apptype
        cshared = os.path.join(self.data_root, containerId)
        
        if data_file_name is None or data_file_name == "":
            log.error("Invalid file name: %s" % data_file_name)
            raise ValueError("Invalid file name: %s" % data_file_name)

        if self.rfs_type is RFSComposer.TYPE_AUFS and apptype == AppType.PAAS:
            cshared = os.path.join(cshared, self.get_persistent_datadir_target(containerId))

        if self.rfs_type is RFSComposer.TYPE_OVERLAY and apptype == AppType.PAAS:
            cshared = os.path.join(cshared, DOCKER_RW_LAYER_ROOT, self.get_persistent_datadir_target(containerId))
        #Append appdata
        cshared = os.path.join(cshared, self._appdata_dir)
        
        dstfilepath = os.path.join(cshared, data_file_name.lstrip("/"))
        FileMgmt.add_data_file(data_file, dstfilepath, data_file_name)
        hm = HostingManager.get_instance()
        sc = hm.get_service("security-management")
        if sc is None:
            log.exception("Security-Management subsystem is not available")
            raise Exception("Security-Management subsystem is not available")
        sec_attr = sc.get_app_security_config(containerId)
        if sec_attr:
            #Apply the smack labels to all the folders under appdata again
            #as there can newly added subfolders
            Utils.apply_rootfs_attributes(sec_attr, cshared, True)

    def get_data_file(self, containerId, data_file_name):
        """
        Reads the content of data_file_name
        if it is directory:
            Returns dictionary with key as dirlist and Value is the list of file dict 
            e.g data["dirlist] = [ 
                name: "a.txt", type: "file", path: "appdata/a.txt", size: 100, last_modified_time: 1-Jan-15),
                name: "dir1/", type: "dir", path: "appdata/dir1", size: 0, last_modified_time: 2-Feb-15
            ]
        if data_file_name is regular file:
            Returns the generator function that reads data  contents of data_file_name 
        """

        container = self._containers[containerId]
        apptype = container.apptype
        cshared = os.path.join(self.data_root, containerId)

        if data_file_name is None :
            log.error("Invalid file name: %s" % data_file_name)
            raise ValueError("Invalid file name: %s" % data_file_name)

        if self.rfs_type is RFSComposer.TYPE_AUFS and apptype == AppType.PAAS:
            cshared = os.path.join(cshared, self.get_persistent_datadir_target(containerId))

        if self.rfs_type is RFSComposer.TYPE_OVERLAY and apptype == AppType.PAAS:
            cshared = os.path.join(cshared, DOCKER_RW_LAYER_ROOT, self.get_persistent_datadir_target(containerId))

        #Append appdata
        cshared = os.path.join(cshared, self._appdata_dir)

        dstfilepath = os.path.join(cshared, data_file_name.lstrip("/"))

        return FileMgmt.get_data_file(dstfilepath, data_file_name)

    def delete_data_file(self, containerId, data_file_name):
        """
        Delete the data_file_name from app appdata directory
        """
        container = self._containers[containerId]
        apptype = container.apptype
        cshared = os.path.join(self.data_root, containerId)

        if data_file_name is None or data_file_name == "" :
            log.error("Invalid file name: %s" % data_file_name)
            raise ValueError("Invalid file name: %s" % data_file_name)

        if self.rfs_type is RFSComposer.TYPE_AUFS and apptype == AppType.PAAS:
            cshared = os.path.join(cshared, self.get_persistent_datadir_target(containerId))

        if self.rfs_type is RFSComposer.TYPE_OVERLAY and apptype == AppType.PAAS:
            cshared = os.path.join(cshared, DOCKER_RW_LAYER_ROOT, self.get_persistent_datadir_target(containerId))

        #Append appdata
        cshared = os.path.join(cshared, self._appdata_dir)

        dstfilepath = os.path.join(cshared, data_file_name.lstrip("/"))

        if dstfilepath == cshared:
            #Deleting the root appdata not allowed
            log.error("Cannot delete the root %s" % data_file_name);
            raise ValueError("Cannot delete the root %s" % data_file_name)
            
        FileMgmt.delete_data_file(dstfilepath, data_file_name)

    def get_app_console(self, appid):
        container = self._containers[appid]

        if container.apptype == AppType.DOCKER and not container._os_mode:
            # Check to see if console is enabled
            cs = self._hm.get_service("console-management")
            if cs is None or not cs.is_enabled or not cs.is_running:
                log.debug("Console access is disabled")
                raise ConsoleDisabledError("Console access is disabled")
            d = self._containers.get(appid, None)

            if d:
                # Construct the command to be used to get into "console" via shell execution in namespace
                pid = d.get_container_pid()
                hm = HostingManager.get_instance()
                sc = hm.get_service("security-management")
                if sc is None:
                    log.exception("Security-Management subsystem is not available")
                    raise Exception("Security-Management subsystem is not available")
                sec_attr = sc.get_app_security_config(appid)
                noseclabel = "--noseclabel"

                if sec_attr and sec_attr.get("lsm", False) and sec_attr.get(
                                  "lsm").get("enabled", False):
                    noseclabel = ""

                exec_command = "/home/" + cs.user_name + "/.bin/virsh -c lxc:/// lxc-enter-namespace " + appid + " " + noseclabel + " /bin/sh"

                log.debug("Console command: %s" % exec_command)
                pubkey, privkey, cmdentry = cs.setup_app_console(appid, exec_command)

                # Make a dict with relevant information
                rval = dict()
                rval["private_key"] = privkey
                rval["user_name"] = cs.user_name
                return rval

            return None
        else:
            return super(LXCContainerManager,self).get_app_console(appid)

    def get_app_session(self, appid):
        container = self._containers[appid]

        # Check to see if console is enabled
        cs = self._hm.get_service("console-management")
        if cs is None or not cs.is_enabled or not cs.is_running:
            log.error("Console service is disabled")
            raise ConsoleDisabledError("Console service is disabled")
        d = self._containers.get(appid, None)

        if d:
            # Construct the command to be used to get session via shell execution in namespace
            pid = d.get_container_pid()
            hm = HostingManager.get_instance()
            sc = hm.get_service("security-management")
            if sc is None:
                log.exception("Security-Management subsystem is not available")
                raise Exception("Security-Management subsystem is not available")
            sec_attr = sc.get_app_security_config(appid)
            noseclabel = "--noseclabel"

            if sec_attr and sec_attr.get("lsm", False) and sec_attr.get(
                              "lsm").get("enabled", False):
                noseclabel = ""

            exec_command = "/home/" + cs.user_name + "/.bin/virsh -c lxc:/// lxc-enter-namespace " + appid + " " + noseclabel + " /bin/sh"

            log.debug("Console command: %s" % exec_command)
            session_id = appid + "_session"
            pubkey, privkey, cmdentry = cs.setup_app_console(session_id, exec_command)

            # Make a dict with relevant information
            rval = dict()
            rval["private_key"] = privkey
            rval["user_name"] = cs.user_name
            return rval

        return None

