import logging
import re
import cStringIO as StringIO
from gzip import GzipFile
import zlib
import bz2


log = logging.getLogger("runtime.api")


class CompressionMiddleware(object):

    def gzip_buffer(self, string, compression_level=6):
        """gzips a string."""
        zbuf = StringIO.StringIO()
        f = GzipFile(filename=None, mode='wb',
                compresslevel=compression_level, fileobj=zbuf)
        f.write(string)
        f.close()
        return zbuf.getvalue()

    def zlib_buffer(self, string, compression_level=6):
        """
        Zlib compression a string
        """
        compressed = zlib.compress(string, compression_level)
        return compressed

    def bzip_buffer(self, string, compression_level=6):
        """
        Bzip compression a string
        """
        compressed = bz2.compress(string, compression_level)
        return compressed

    def client_accepts_gzip(self, environ):
        """Checks whether the client accepts gzipped output."""
        re_accepts_gzip = re.compile(r'\bgzip\b')
        accept_header = environ.get('HTTP_ACCEPT_ENCODING', '')
        return re_accepts_gzip.search(accept_header)

    def client_accepts_deflate(self, environ):
        """Checks whether the client accepts defalte output."""
        re_accepts_defalte = re.compile(r'\bdeflate\b')
        accept_header = environ.get('HTTP_ACCEPT_ENCODING', '')
        return re_accepts_defalte.search(accept_header)

    def client_accepts_bzip(self, environ):
        """Checks whether the client accepts defalte output."""
        re_accepts_bzip = re.compile(r'\bbzip\b')
        accept_header = environ.get('HTTP_ACCEPT_ENCODING', '')
        return re_accepts_bzip.search(accept_header)

    def process_response(self, request, response, resource, req_succeeded):
        if response.body and response.stream is None:
            if response.get_header('content-encoding') is None:
                body = response.body
                buflen = len(body)
                if not buflen <= 200:
                    if self.client_accepts_gzip(request.env):
                        log.debug("GZIP compressing the response body as client accepts gzip compressed data")
                        gzipped_response = self.gzip_buffer(body)
                        # Last check: only send the response if it's actually shorter
                        if not len(gzipped_response) >= buflen:
                            response.set_header('Content-Encoding', 'gzip')
                            response.body = gzipped_response
                            response.set_header('Content-Length', str(len(gzipped_response)))
                    elif self.client_accepts_deflate(request.env):
                        log.debug("DEFLATE compressing the response body as client accepts defalte compressed data")
                        zlib_response = self.zlib_buffer(body)
                        # Last check: only send the response if it's actually shorter
                        if not len(zlib_response) >= buflen:
                            response.set_header('Content-Encoding', 'deflate')
                            response.body = zlib_response
                            response.set_header('Content-Length', str(len(zlib_response)))
                    elif self.client_accepts_bzip(request.env):
                        log.debug("BZIP compressing the response body as client accepts bzip compressed data")
                        bzip_response = self.bzip_buffer(body)
                        # Last check: only send the response if it's actually shorter
                        if not len(bzip_response) >= buflen:
                            response.set_header('Content-Encoding', 'bzip')
                            response.body = bzip_response
                            response.set_header('Content-Length', str(len(bzip_response)))
