'''

@author: havishwa

Copyright (c) 2014-2015 by Cisco Systems, Inc.
All rights reserved.
'''


from jsonencoder import JSONEncoder
from apiservice import ResourceRoute, APIService
from common import AuthenticatedResource, make_response, make_error_response
from ..utils.cafevent import CAFEvent
from ..utils.infraexceptions import *
from ..utils.utils import Utils
from ..cartridge.cartridge import Cartridge, CartridgeManager

import logging
import tempfile
import os
import falcon
import json
import re
import tarfile

log = logging.getLogger("runtime.api.resources")

jsonencoder = JSONEncoder()

@ResourceRoute("/cartridges", endpoint="cartridges")
class CartridgesResource(AuthenticatedResource):

    def on_post(self, request, response):
        '''
        Accepts a cartridge archive and deploys it
        '''

        cartridge = None

        # Read upload location from system config, default to /tmp if not specified
        tmpUploadDir = '/tmp'
        if APIService.instance.config.has_option("controller", "upload_dir"):
            tmpUploadDir = APIService.instance.config.get("controller", "upload_dir")

        if not os.path.exists(tmpUploadDir):
            os.makedirs(tmpUploadDir)

        # Do not assume file to be zip file, it can be .zip or .ova
        fd, filePath = tempfile.mkstemp("", "tmpArchive", tmpUploadDir)
        with os.fdopen(fd, "wb") as f:
            while True:
                chunk = request.stream.read(4096)
                if not chunk:
                    break
                f.write(chunk)

        f.close()

        log.debug("Downloaded the file at:%s" % filePath)

        try:
            #cm.add(uniqueCartridgeId, filePath)
            cm = CartridgeManager.getInstance()
            if cm.is_paas_app_supported():
                cart_id = cm.add(filePath)
        except Exception as ex:
            log.exception("Error in cartridge installation: (%s)", str(ex))
            if isinstance(ex, C3Exception):
                response = make_error_response(response, ex.message,
                                               "Error during cartridge installation",
                                               falcon.HTTP_500,
                                               ex.errorcode)
            else:
                response = make_error_response(response,
                                                "Error during cartridge installation: %s" % str(ex),
                                                "Error during cartridge installation",
                                                falcon.HTTP_500)
            return
        finally:
            os.remove(filePath) if os.path.exists(filePath) else None

        #cartridgeURL = request.uri + "/" + uniqueCartridgeId
        cartridgeURL = request.uri + "/" + cart_id

        # Post a deployed event
        ns = APIService.instance.hosting_manager.get_service("notification-service")
        if ns:
            log.debug("Posting 'deployed' event on %s" % str(cart_id))
            event_message = "Cartridge: %s deployed using: %s" % (str(cart_id), CAFEvent.SOURCE_RESTAPI)

            ns.post_event(CAFEvent(cart_id,
                                   CAFEvent.TYPE_DEPLOYED,
                                   CAFEvent.SOURCE_RESTAPI,
                                   event_message=event_message))

        response.status = falcon.HTTP_201
        response.body = "Successfully deployed cartridge"
        response.set_header("Content-Location", cartridgeURL)

    def on_get(self, request, response):
        cm = CartridgeManager.getInstance()
        out = jsonencoder.encode(cm.list())
        response.body = out
        response.status = falcon.HTTP_200
        response.set_headers({'Content-Type': "application/json",
                              'Cache-Control': "no-cache"})



@ResourceRoute("/cartridges/{cartridge_id}", endpoint="cartridge")
class CartridgeResource(AuthenticatedResource):

    def on_get(self, request, response, cartridge_id):
        cm = CartridgeManager.getInstance()
        c = cm.get(cartridge_id)
        if c is None:
            response = make_error_response(response,
                                            "The cartridge : %s, does not exist" % cartridge_id,
                                            "The cartridge : %s, does not exist" % cartridge_id,
                                            falcon.HTTP_404)
            return

        response.status = falcon.HTTP_200
        ser_obj = c.serialize()
        log.debug("Cartiridge Serialized Object: %s" % str(ser_obj))
        response.body = jsonencoder.encode(ser_obj)
        response.set_headers({'Content-Type': "application/json",
                              'Cache-Control': "no-cache"})

    def on_put(self, request, response, cartridge_id):
        cm = CartridgeManager.getInstance()
        c = cm.get(cartridge_id)
        if c is None:
            response = make_error_response(response,
                                            "The cartridge : %s, does not exist" % cartridge_id,
                                            "The cartridge : %s, does not exist" % cartridge_id,
                                            falcon.HTTP_404)
            return

        # Read upload location from system config, default to /tmp if not specified
        tmpUploadDir = '/tmp'
        if APIService.instance.config.has_option("controller", "upload_dir"):
            tmpUploadDir = APIService.instance.config.get("controller", "upload_dir")

        if not os.path.exists(tmpUploadDir):
            os.makedirs(tmpUploadDir)

        # Check condition for file to be TAR file
        fd, filePath = tempfile.mkstemp("", "tmpCartArchive", tmpUploadDir)
        with os.fdopen(fd, "wb") as f:
            while True:
                chunk = request.stream.read(4096)
                if not chunk:
                    break
                f.write(chunk)

        log.debug("Downloaded the file at:%s" % filePath)
        try:
            if cm.is_paas_app_supported():
                cm.upgrade(cartridge_id, filePath)
        except Exception as ex:
            log.exception("Error while upgrading cartridge: %s - %s" % (cartridge_id, str(ex)))
            if isinstance(ex, C3Exception):
                response = make_error_response(response, ex.message,
                                                "Error while upgrading cartridge",
                                                falcon.HTTP_500,
                                                ex.errorcode)
            else:
                response = make_error_response(response,
                                                "Error while upgrading cartridge: %s - %s" % (cartridge_id, str(ex)),
                                                "Error while upgrading cartridge",
                                                falcon.HTTP_500)
            return

        finally:
            os.remove(filePath) if os.path.exists(filePath) else None


        response.status = falcon.HTTP_200
        response.body = "Upgrade successful."

    def on_delete(self, request, response, cartridge_id):
        '''
        delete the cartridge
        '''
        cm = CartridgeManager.getInstance()
        exists = cm.exists(cartridge_id)
        if not exists:
            response = make_error_response(response,
                                           "The cartridge : %s, does not exist" % cartridge_id,
                                           "The cartridge : %s, does not exist" % cartridge_id,
                                           falcon.HTTP_404)
            return
        try:
            cm.remove(cartridge_id)
        except CartridgeDependError as ex:
            log.exception("Error in cartridge deletion: (%s)", str(ex))
            response = make_error_response(response,
                                           "Error removing cartridge : %s" % str(ex),
                                           "Error removing cartridge : %s" % cartridge_id,
                                           falcon.HTTP_500)
            return
        except Exception as ex:
            log.exception("Error in cartridge deletion: (%s)", str(ex))
            response = make_error_response(response,
                                           "Error removing cartridge : %s" % str(ex),
                                           "Error removing cartridge : %s" % cartridge_id,
                                           falcon.HTTP_500)
            return

        # Post undeployed event
        ns = APIService.instance.hosting_manager.get_service("notification-service")
        if ns:
            log.debug("Posting 'undeployed' event on %s" % str(cartridge_id))
            event_message = "Cartridge : %s undeployed using: %s" % (str(cartridge_id), CAFEvent.SOURCE_RESTAPI)
            ns.post_event(CAFEvent(cartridge_id,
                                   CAFEvent.TYPE_UNDEPLOYED,
                                   CAFEvent.SOURCE_RESTAPI,
                                   event_message = event_message))

        response.status = falcon.HTTP_200
        response.body = ""
        return
