'''
Created on Nov 27, 2012

@author: samagraw

Copyright (c) 2012-2013 by Cisco Systems, Inc.
All rights reserved.
'''

import logging
import os
import re
import ConfigParser, shutil
import tempfile
import time
import urllib

from apiservice import APIService
from   ..utils.utils import Utils
from ..utils.infraexceptions import FilePathError, MandatoryFileMissingError
from appfw.utils.commandwrappers import grep_pipe

log = logging.getLogger("runtime")

class LogsInfo(object):

    config_object = APIService.instance.config

    logs_path = "/var/log"
    if config_object.has_option("logging", "log_location"):
        logs_path = config_object.get("logging", "log_location")

    log.debug("Logs are located at : %s" % logs_path)

    query_string = "/admin/download/logs"

    log_cfg = getattr(config_object, "log_config_file", None)
    log_configparser = getattr(config_object, "log_configparser", None)
    ext_log_cfg = getattr(config_object, "ext_log_config_file", None)
    if log_cfg is None:
        # Auto deduce based on caf folder structure
        log.debug("Inferring location of logconfig file based on relative folder path")
        MODULE_DIR = os.path.dirname(os.path.realpath(__file__))
        appfw_dir, api_tok = os.path.split(MODULE_DIR)
        caf_src_dir, appfw_tok = os.path.split(appfw_dir)
        caf_dir, src_tok = os.path.split(caf_src_dir)
        log_cfgdefa= os.path.join(caf_dir,  "config/log-config.ini")

    log.debug("Log config file is at : %s" % log_cfg)
    if ext_log_cfg is not None:
        log.debug("Extra log config file is at : %s" % ext_log_cfg)
    categoryList = []
    #Propagating log categories defined in logconfig file
    for key, val in logging.Logger.manager.loggerDict.iteritems():
        propagate = val.__dict__.get("propagate")
        if propagate is not None and propagate == 0:
            categoryList.append(key)
    log_level_lookup = {'NOTSET':0, 'DEBUG':10, 'INFO':20, \
                        'WARNING':30, 'ERROR':40, 'CRITICAL':50}
    categoryToQualnameMap = {
        'rest_apis':'runtime.api',
        'connector_management':'runtime.hosting',
        'system_information':'runtime',
        'other':'root'}

    platform_logs = "auth.log,dmesg,libvirtd.log,syslog"
    logging_sec = Utils.getSystemConfigSection("logging")
    if logging_sec is not None:
        platform_logs = logging_sec.get("platform_logs", platform_logs)

    #split the string into tokens
    platform_logs = platform_logs.split(",")

    #remove whitespace
    for indx, file_name in enumerate(platform_logs):
        platform_logs[indx] = file_name.strip()
    
    log.debug("platform log files list %s" %platform_logs)
    #Get contents from log file
    @classmethod
    def get_logs(cls, file_name):
        #if file_name.startswith('/'):
        pattern = re.compile("^[^(\/.)][a-zA-Z0-9-/_]+([.][a-zA-Z0-9-/_]+)*$")
        if pattern.match(file_name) is None:
            log.exception("Absolute file path %s is not allowed", file_name)
            raise FilePathError("Absolute file path %s is not allowed", file_name)

        log.debug("Get contents of the log file %s", file_name)
        logs = ''

        #Get contents of host log file
        log_file_path = os.path.join(LogsInfo.logs_path, file_name)
        if os.path.exists(log_file_path):
            try:
                with open(log_file_path) as file_hdlr:
                    logs = file_hdlr.read()                               
                return logs
            except IOError:
                log.exception("Exception while reading log file: %s", file_name)
                raise IOError
        else:
            #Get contents of container's log file
            connectorId = file_name.split("-", 1)[0]
            containerLogFile = file_name.split("-", 1)[1]

            return APIService.instance.app_manager.get_logfile_contents(connectorId, containerLogFile)

    @classmethod
    def get_taillog(cls, file_name, lines=10):
        """
        Will tail the contents of a given file and return the same.
        """
        log.debug("Tail logging the file: %s, with number of lines: %s "%(file_name, lines))
        lines = int(lines)
        pattern = re.compile("^[^(\/.)][a-zA-Z0-9-/_]+([.][a-zA-Z0-9-/_]+)*$")
        if pattern.match(file_name) is None:
            log.exception("Absolute file path %s is not allowed", file_name)
            raise FilePathError("Absolute file path %s is not allowed", file_name)
        log_file = os.path.join(LogsInfo.logs_path, file_name)
        resp = {}
        log_contents = ""
        resp['log_file'] = file_name
        resp['log_lines'] = log_contents
        log.debug("Returning response for tail log : %s" % str(resp))
        try:
            if not os.path.isfile(log_file):
                if len(file_name.split("-", 1)) == 2:
                    connectorId = file_name.split("-", 1)[0]
                    containerLogFile = file_name.split("-", 1)[1]
                    if APIService.instance.app_manager:
                        log_contents = APIService.instance.app_manager.get_logfile_lines(connectorId, containerLogFile, lines)
                    else:
                        log.debug("App manager is down!")
                else:
                    raise MandatoryFileMissingError("Given file %s is not found/invalid!"%file_name)
            else:
                if Utils.is_textfile(log_file):
                    log_contents = Utils.tail(file(log_file), lines)
                else:
                    raise IOError("Requested file %s is not a valid log file" % file_name)
            resp['log_lines'] = log_contents
        except Exception as e:
            log.exception("Error while reading the file %s. Cause: %s"%(file_name, e.message))
            raise e
        return resp

    #Populate list of log files
    @classmethod
    def addToLogsList(cls, fileName, size, timeStamp, prefix='', tag ='', err_count='', relative_path=None):
        encoded_filename = urllib.quote_plus(fileName)
        if relative_path:
            fileName = relative_path
        return ({"filename" : fileName, "size_bytes" : size,
                 "timestamp" : timeStamp,
                 "tag" : tag,
                 "errorcount": err_count,
                 "download_link" : LogsInfo.query_string + '?filename=' + prefix + encoded_filename})

    #Check if the file is a platform log file
    @classmethod
    def is_platform_log_file(cls, fileName):
        for f in LogsInfo.platform_logs:
            if fileName.startswith(f):
                return True
        return False

    #Get list of log files
    @classmethod
    def get_logs_list(cls, url):
        log.debug("Get list of log files under %s" %LogsInfo.logs_path)
        logs_list = []

        for dirpath, dirs, files in os.walk(LogsInfo.logs_path):
            relpath = ''
            if dirpath != LogsInfo.logs_path:
                relpath = os.path.relpath(dirpath, LogsInfo.logs_path)
                if not relpath.endswith(os.sep):
                    relpath += os.sep

            for file_name in files:
                file = os.path.join(dirpath, file_name)
                tag = ''
                search_keywords="ERROR|CRITICAL"
                grep_options="-Eci"
                if "caf" in file_name[0:3]:
                    tag = 'caf logs'
                    search_keywords=":ERROR|:CRITICAL"
                    #don't need case insensitivity for caf logs
                    grep_options="-Ec"

                elif LogsInfo.is_platform_log_file(file_name):
                    tag = 'platform logs'

                err_count = '0'
                try:
                    if os.path.getsize(file) > 0:
                        rval, rc = grep_pipe(grep_options, search_keywords, file)
                        if rc == 0:
                            err_count = str(rval)
                        elif rc > 1 or rc < 0:
                            err_count = '?'
                            #rc == 1 implies no lines were selected.
                            log.error("Unable to grep for errors in %s, %s" %(file, rval))
                except Exception as ex:
                    err_count = '?'
                    log.exception("Error while getting size of file: %s, %s" %(file, str(ex)))
                relative_path = relpath + urllib.quote_plus(file_name)
                logs_list.append(LogsInfo.addToLogsList(file_name,
                                os.path.getsize(file),
                                time.ctime(os.path.getmtime(file)),
                                relpath,
                                tag,
                                err_count,
                                relative_path))

        log.debug("Returned List of logs files: %s", logs_list)
        return logs_list

    #Get list of log files for one connector
    @classmethod
    def get_connector_logs_list(cls, connector_id):
        log.debug("Get list of connector log files")
        logs_list = []

        # Get container log list
        app_manager = APIService.instance.app_manager
        logs = app_manager.get_logfile_list(connector_id)
        for key in logs:
            logs_list.append(LogsInfo.addToLogsList(key[0], key[1], key[2], connector_id+'-'))
            
        log.debug("Returned List of connector %s logs files: %s", connector_id, logs_list)
        return logs_list

    #Delete the connector log file specified
    @classmethod
    def delete_connector_log_files(cls, connector_id, filename=None):
        """
        Will delete the app logs specified, if not then it will delete the entire log directory
        of app mouted.
        """
        log.debug("Delete the log file %s, of the connector %s"%(filename, connector_id))
        app_manager = APIService.instance.app_manager
        app_manager.delete_app_logfiles(connector_id, filename)
        log.info("Log file %s successfully got deleted" % filename)

    #Get list of log files for one connector
    @classmethod
    def get_tail_of_logfile(cls, connector_id, filename, lines=10):
        log.debug("Getting tail of %s:%s" % (connector_id, filename))

        # Get container log list
        logLines = APIService.instance.app_manager.get_logfile_lines(connector_id, filename, lines)
        resp = {}
        resp['connector_id'] = connector_id
        resp['log_file'] = filename
        resp['log_lines'] = logLines
            
        log.debug("Returning response for tail log : %s" % str(resp))
        return resp

    #Returns Default (factory settings) log-levels for caf
    @classmethod
    def getDefaultLogLevel(cls):
        log.debug("Get default log-level i.e. factory settings for caf")
        element ={}
        config = ConfigParser.SafeConfigParser()
        config.read(Utils.getDefaultLogConfigIniPath())

        for category, qualname in LogsInfo.categoryToQualnameMap.iteritems():
            logLevel = config.get("logger_"+str(qualname), "level")
            log.debug("Default log level for %s is %s", qualname, logLevel)
            element[category] = logLevel.lower()
        for category in cls.categoryList:
            if category not in LogsInfo.categoryToQualnameMap.values():
                logLevel = config.get("logger_"+str(category), "level")
                element[category] = logLevel.lower()

        return element

    #Returns Runtime log-levels for caf
    @classmethod
    def getRuntimeLogLevel(cls):
        log.debug("Get current log-level for caf")
        element ={}
        reverse_look_up = dict(zip(LogsInfo.log_level_lookup.values(), \
                                   LogsInfo.log_level_lookup.keys()))

        for category, qualname in LogsInfo.categoryToQualnameMap.iteritems():
            getLog = logging.getLogger(qualname)
            element[category] = reverse_look_up[getLog.getEffectiveLevel()].lower()

        for category in cls.categoryList:
             if category not in LogsInfo.categoryToQualnameMap.values():
                getLog = logging.getLogger(category)
                element[category] = reverse_look_up[getLog.getEffectiveLevel()].lower()

        return element

    #Updates running time log-level for all categories
    @classmethod
    def setRuntimeLogLevel(cls, qualname, logLevel):
        log.debug("Setting current log-level for %s to %s", qualname, logLevel)
        setLog = logging.getLogger(qualname)
        setLog.setLevel(LogsInfo.log_level_lookup[logLevel])
   
    #Updates caf log-level for all categories
    @classmethod
    def setLogLevel(cls, data):
        log.debug("Setting log-level for caf")
        config = LogsInfo.log_configparser
        try:
            outfile = None
            for category, level in data.iteritems():
                qualname = LogsInfo.categoryToQualnameMap.get(category)
                if qualname is None:
                    if category not in cls.categoryList:
                        continue
                    else:
                        qualname = category
                LogsInfo.setRuntimeLogLevel(qualname, level.upper())
                
                #Also, update config to update log-config.ini file
                config.set('logger_'+str(qualname), "level", level.upper())

            # Now, update the log-config file incase caf or vm restarts
            # Create a copy to be safe, and them copy it later to original file
            outfp, outfile = tempfile.mkstemp(".ini", "log", None, False)
            with os.fdopen(outfp, "w", 0) as f:
                config.write(f)
            try:
                shutil.move(outfile, LogsInfo.log_cfg)
            except Exception as ex:
                log.exception("Error while updating the log config file: %s, Cause: %s"%(LogsInfo.log_cfg, ex.message))

            # save a copy to extra log config if specified on system-config.ini
            if LogsInfo.ext_log_cfg is not None:
                extconfig = ConfigParser.RawConfigParser()
                if os.path.isfile(LogsInfo.ext_log_cfg):
                    extconfig.read(LogsInfo.ext_log_cfg)

                for category, level in data.iteritems():
                    qualname = LogsInfo.categoryToQualnameMap.get(category)
                    if qualname is None:
                        if category not in cls.categoryList:
                            continue
                        else:
                            qualname = category
                    if not extconfig.has_section('logger_'+str(qualname)):
                        extconfig.add_section('logger_'+str(qualname))
                    extconfig.set('logger_'+str(qualname), "level", level.upper())

                log.info("write to extra log config %s" % LogsInfo.ext_log_cfg)
                try:
                    with open(LogsInfo.ext_log_cfg, 'w', 0) as f:
                        extconfig.write(f)
                except Exception as ex:
                    log.exception("Error while updating the ext log config file: %s, Cause: %s" % (LogsInfo.ext_log_cfg, ex.message))

            log.info("Update for caf log-level successful")
            return True           
        except KeyError:
            if outfile and os.path.isfile(outfile):
                os.remove(outfile)
            log.error("Exception while updating caf log-level")
            return False 
