#-----------------------------------------------------
#
# Copyright (c) 2012-2014 by cisco Systems, Inc.
# All rights reserved.
#-----------------------------------------------------
__author__ = 'havishwa'

import threading
import select
import os
import errno
import fcntl
import logging

log = logging.getLogger("overrides")

def Event(*args, **kwargs):
    return _Event(*args, **kwargs)

def set_non_blocking(fd):
    flags = fcntl.fcntl(fd, fcntl.F_GETFL) | os.O_NONBLOCK
    fcntl.fcntl(fd, fcntl.F_SETFL, flags)


class _Event():
    """
    Python's default threading.Event() uses threading.Condition() underneath.
    The condition.wait() method implementation in Python 2.7 results in semi
    busy waiting, which may hog cpu in resource constrained environments.

    http://stackoverflow.com/questions/9709022/better-solution-for-python-threading-event-semi-busy-waiting
    http://stackoverflow.com/questions/22146351/arbitrary-sleeping-in-threadings-wait-with-timeout?rq=1

    Wherever there is a need to use Event object between two threads for signalling
    purpose, this class can be used.
    This uses os.pipe and select() call for signalling.

    NOTE: 1. This implementation doesn't provide support for notify() and notify_all() methods
          2. This object should not be used for synchronization between multiple threads
    """

    def __init__(self):
        self.__flag = False
        self.__lock = threading.RLock()
        self.__pipe = os.pipe()
        for fd in self.__pipe:
            set_non_blocking(fd)

    def __enter__(self):
        return self.__lock.__enter__()

    def __exit__(self, *args):
        return self.__lock.__exit__(*args)

    def isSet(self):
        return self.__flag

    is_set = isSet

    def __wakeup(self):
        """
        Wake up the event by writing to the PIPE
        """
        try:
            os.write(self.__pipe[1], b'.')
            #log.debug("Wrote to pipe..")
        except IOError as e:
            if e.errno not in [errno.EAGAIN, errno.EINTR]:
                raise

    def __sleep(self, timeout=None):
        """\
        Sleep until PIPE is readable or we timeout.
        A readable PIPE means a signal occurred.
        """
        try:
            ready = select.select([self.__pipe[0]], [], [], timeout)
            if not ready[0]:
                return
            while os.read(self.__pipe[0], 1):
                # Empty the pipe
                pass
            #log.debug("Read from pipe..")
        except select.error as e:
            if e.args[0] not in [errno.EAGAIN, errno.EINTR]:
                raise
        except OSError as e:
            if e.errno not in [errno.EAGAIN, errno.EINTR]:
                raise
        except KeyboardInterrupt:
            pass

    def set(self):
        self.__flag = True
        #log.debug("Flag set to true")
        self.__wakeup()

    def clear(self):
        self.__flag = False
        #log.debug("Flag set to false")

    def wait(self, timeout=None):
        if not self.__flag:
            #log.debug("Waiting for %s seconds" % str(timeout))
            self.__sleep(timeout)
        return self.__flag

    def __del__(self):
        if self.__pipe:
            for fd in self.__pipe:
                os.close(fd)

    # notify, notify_all, notifyAll just does a set
    notify = set
    notify_all = set
    notifyAll = set

'''
if "__main__" == __name__:
    import threading
    import time
    import random

    print "PID : %s" % str(os.getpid())

    def set_event(ce, sleeptime=None):
        if sleeptime:
            time.sleep(sleeptime)
        print "%s is setting event %s " % (threading.currentThread(), str(hex(id(ce))))
        ce.set()
        print "%s: ce.__flag: %s" % (threading.currentThread(), ce.is_set())

    ce = Event()
    assert(ce.is_set() == False)

    print "Starting a thread to set event in 10 seconds"
    # Start a thread that would set the event in 10 seconds
    t1 = threading.Thread(name="Thread1", target=set_event, args=(ce, 10))
    t1.setDaemon(True)
    t1.start()

    print "Waiting on the event %s for 20 seconds " % hex(id(ce))
    st = time.time()
    ce.wait(20)
    et = time.time()
    print "ce.is_set : %s" % str(ce.is_set())
    print "%s: ce.__flag: %s" % (threading.currentThread(), ce.is_set())
    assert(ce.is_set() == True)
    assert((et-st) < 20)
    print "Event set by the thread in %s seconds" % (et-st)

    ce.clear()
    print "Waiting on the event for 30 seconds. Check CPU usage"
    ce.wait(30)
    assert(ce.is_set() == False)
    print "Event is not set after 30 seconds"

    print "Starting a thread to set event in 20 seconds"
    # Start a thread that would set the event in 10 seconds
    t1 = threading.Thread(name="Thread1", target=set_event, args=(ce, 10))
    t1.setDaemon(True)
    t1.start()
    print "Waiting infinitely on the event. Check CPU."
    st = time.time()
    ce.wait()
    et = time.time()
    assert(ce.is_set() == True)
    print "Event set by the thread in %s seconds" % (et-st)
    '''
