'''
Created on Feb 12, 2017
Copyright (c) 2012-2017 by Cisco Systems, Inc.
All rights reserved.
'''

import sys
import logging
import os
from ConfigParser import RawConfigParser

#from appfw.runtime.caf_abstractservice import CAFAbstractService
#from ..utils.utils import Utils
import json
import psutil

log = logging.getLogger("runtime.bist")
PLATFORM_HWID_FILE="/etc/platform/hwid"
PLATFORM_PRODUCT_ID_FILE="/etc/platform/product_id"
CAF_PORT="8443"
NETWORK_CONFIG_FILE_NAME = "network_config.yaml"
CAF_EXTERNAL_DEPENDENCY_FILE="caf_external_dependencies.yaml"
CAF_REQUIREMENTS_FILE="caf-requirements.txt"
log = logging.getLogger("bist")

'''
This Python implements unix like "which" utility to locate the binary
'''
def execute_which(cmd, mode=os.F_OK | os.X_OK, path=None):
    """Given a command, mode, and a PATH string, return the path which
    conforms to the given mode on the PATH, or None if there is no such
    file.

    `mode` defaults to os.F_OK | os.X_OK. `path` defaults to the result
    of os.environ.get("PATH"), or can be overridden with a custom search
    path.

    """
    # Check that a given file can be accessed with the correct mode.
    # Additionally check that `file` is not a directory, as on Windows
    # directories pass the os.access check.
    def _access_check(fn, mode):
        return (os.path.exists(fn) and os.access(fn, mode)
                and not os.path.isdir(fn))

    # If we're given a path with a directory part, look it up directly rather
    # than referring to PATH directories. This includes checking relative to the
    # current directory, e.g. ./script
    if os.path.dirname(cmd):
        if _access_check(cmd, mode):
            return cmd
        return None

    if path is None:
        path = os.environ.get("PATH", os.defpath)
    if not path:
        return None
    path = path.split(os.pathsep)

    if sys.platform == "win32":
        # The current directory takes precedence on Windows.
        if not os.curdir in path:
            path.insert(0, os.curdir)

        # PATHEXT is necessary to check on Windows.
        pathext = os.environ.get("PATHEXT", "").split(os.pathsep)
        # See if the given file matches any of the expected path extensions.
        # This will allow us to short circuit when given "python.exe".
        # If it does match, only test that one, otherwise we have to try
        # others.
        if any(cmd.lower().endswith(ext.lower()) for ext in pathext):
            files = [cmd]
        else:
            files = [cmd + ext for ext in pathext]
    else:
        # On other platforms you don't have things like PATHEXT to tell you
        # what file suffixes are executable, so just pass on cmd as-is.
        files = [cmd]

    seen = set()
    for dir in path:
        normdir = os.path.normcase(dir)
        if not normdir in seen:
            seen.add(normdir)
            for thefile in files:
                name = os.path.join(dir, thefile)
                if _access_check(name, mode):
                    return name
    return None
'''
Utility function to read the yaml file
'''
def read_yaml_file(file_name):
    import yaml
    parsed_yaml = None
    if os.path.isfile(file_name):
        with open(file_name, 'r') as fp:
            parsed_yaml = yaml.safe_load(fp)

    return parsed_yaml

'''
Utility function to get the system config file required by CAF
'''
def getSystemConfigPath():
    '''
    Returns scripts directory. Derives it as relative to the
    runtime source directory.
    '''
    p = getConfigFolder()
    p = os.path.join(p, "system-config.ini")
    log.debug("System config path: %s", p)
    return p


def getRuntimeSourceFolder():
    '''
    Returns runtime source directory.
    If CAF_HOME env variable is set, the same value is returned.
    Else, the location of caf directory is inferred relative to
    the location of the current module.
    '''

    caf_home = os.getenv("CAF_HOME")
    if caf_home:
        log.debug("CAF_HOME is set to : %s" % caf_home)
    else:
        p = os.path.dirname(os.path.abspath(__file__))
        p = os.path.split(p)[0]    #get appfw
        p = os.path.split(p)[0]    #get src directory
        caf_home = os.path.split(p)[0]    #get caf directory

    log.debug("Runtime source folder is at: %s" % caf_home)
    return caf_home

def getConfigFolder():
    '''
    Returns path to config directory that contains
    system-config, log-config etc.,
    '''
    p = getRuntimeSourceFolder()
    c = os.path.join(p, "config")
    log.debug("Config folder is at: %s", c)
    return c


def getNetworkConfigFile():
    '''
    Returns network configuration file
    '''
    p = getConfigFolder()
    c = os.path.join(p, NETWORK_CONFIG_FILE_NAME)
    return c

def getCafDependenciesFile():
    '''
    :return: caf external dependency file name
    '''
    p = getConfigFolder()
    c = os.path.join(p, CAF_EXTERNAL_DEPENDENCY_FILE)
    return c

def getCafRequirementsFile():
    '''

    :return: returns caf requirements file
    '''
    p = getConfigFolder()
    c = os.path.join(p, CAF_REQUIREMENTS_FILE)
    return c

def getSystemConfigValue(cfg, section, key, default=None, parse_as="str"):
    '''
    Returns system config object
    '''
    # cfg = ConfigParser.SafeConfigParser()
    # cfg.read(cls.getSystemConfigPath())
    if cfg.has_option(section, key):
        if parse_as == "str":
            ret = cfg.get(section, key)
        elif parse_as == "int":
            ret = cfg.getint(section, key)
        elif parse_as == "float":
            ret = cfg.getfloat(section, key)
        elif parse_as == "bool":
            ret = cfg.getboolean(section, key)
        elif parse_as == "json":
            ret = json.loads(cfg.get(section, key))
    else:
        if default is not None:
            ret = default
        else:
            # Let NoOptionError be raised providing proper error context
            cfg.get(section, key)

    return ret

from socket import AF_INET, AF_INET6, inet_ntop
from ctypes import (
    Structure, Union, POINTER,
    pointer, get_errno, cast,
    c_ushort, c_byte, c_void_p, c_char_p, c_uint, c_int, c_uint16, c_uint32
)
import ctypes.util
import ctypes

class struct_sockaddr(Structure):
    _fields_ = [
        ('sa_family', c_ushort),
        ('sa_data', c_byte * 14),]

class struct_sockaddr_in(Structure):
    _fields_ = [
        ('sin_family', c_ushort),
        ('sin_port', c_uint16),
        ('sin_addr', c_byte * 4)]

class struct_sockaddr_in6(Structure):
    _fields_ = [
        ('sin6_family', c_ushort),
        ('sin6_port', c_uint16),
        ('sin6_flowinfo', c_uint32),
        ('sin6_addr', c_byte * 16),
        ('sin6_scope_id', c_uint32)]

class union_ifa_ifu(Union):
    _fields_ = [
        ('ifu_broadaddr', POINTER(struct_sockaddr)),
        ('ifu_dstaddr', POINTER(struct_sockaddr)),]

class struct_ifaddrs(Structure):
    pass
struct_ifaddrs._fields_ = [
    ('ifa_next', POINTER(struct_ifaddrs)),
    ('ifa_name', c_char_p),
    ('ifa_flags', c_uint),
    ('ifa_addr', POINTER(struct_sockaddr)),
    ('ifa_netmask', POINTER(struct_sockaddr)),
    ('ifa_ifu', union_ifa_ifu),
    ('ifa_data', c_void_p),]

libc = ctypes.CDLL(ctypes.util.find_library('c'))

def ifap_iter(ifap):
    ifa = ifap.contents
    while True:
        yield ifa
        if not ifa.ifa_next:
            break
        ifa = ifa.ifa_next.contents

def getfamaddr(sa):
    family = sa.sa_family
    addr = None
    if family == AF_INET:
        sa = cast(pointer(sa), POINTER(struct_sockaddr_in)).contents
        addr = inet_ntop(family, sa.sin_addr)
    elif family == AF_INET6:
        sa = cast(pointer(sa), POINTER(struct_sockaddr_in6)).contents
        addr = inet_ntop(family, sa.sin6_addr)
    return family, addr

'''
Class for listing available network interfaces on the host
'''
class NetworkInterface(object):
    def __init__(self, name):
        self.name = name
        self.index = libc.if_nametoindex(name)
        self.addresses = {}

    def __str__(self):
         return "%s [index=%d, IPv4=%s, IPv6=%s]" % (
             self.name, self.index,
             self.addresses.get(AF_INET),
             self.addresses.get(AF_INET6))

'''
API for fetching the list of available network interfaces
'''
def get_network_interfaces():
    ifap = POINTER(struct_ifaddrs)()
    result = libc.getifaddrs(pointer(ifap))
    if result != 0:
        raise OSError(get_errno())
    del result
    try:
        retval = {}
        for ifa in ifap_iter(ifap):
            name = ifa.ifa_name
            i = retval.get(name)
            if not i:
                i = retval[name] = NetworkInterface(name)
            family, addr = getfamaddr(ifa.ifa_addr.contents)
            if addr:
                i.addresses[family] = addr
        return retval.values()
        #return retval

    finally:
        libc.freeifaddrs(ifap)

import platform,re
import subprocess

'''
API to ping a given hostname with timeout
'''
def execute_ping(hostname,timeout):
    log.debug("Executing ping for %s timeout %s", hostname, timeout)
    if platform.system() == "Windows":
        command="ping "+hostname+" -n 1 -w "+str(timeout*1000)
    else:
        command="ping -i "+str(timeout)+" -c 1 " + hostname
    try:
        proccess = subprocess.Popen(command, stdout=subprocess.PIPE)
        matches=re.match('.*time=([0-9]+)ms.*', proccess.stdout.read(),re.DOTALL)
        if matches:
            return matches.group(1)
        else:
            return False
    except Exception as ex:
        return False

'''
This class parsers the required configs for verifying the dependencies that CAF has on the platform
Provides API's for providing detailed report on system dependency, python dependencies
API for Verifying the system config and network config provided for CAF on a given platform
This designed in such a way so that it can be invoked via CAF rest API's as well as can be independently
executed as python file.It has minimal or no dependencies on CAF source code
'''
class BuiltInSelfTestModule(object):
    __singleton = None
    def __new__(cls, *args, **kwargs):
        # Check to see if a __singleton exists already for this class
        # Compare class types instead of just looking for None so
        # that subclasses will create their own __singleton objects
        if cls != type(cls.__singleton):
            cls.__singleton = super(BuiltInSelfTestModule, cls).__new__(cls, *args, **kwargs)
        return cls.__singleton

    def __init__(self, params=None, dependency_file=None, caf_requirements_file=None, system_config=None , network_config=None, device_config_file=None):
        if params:
            self.config = params.config
            self._config_file = params.config_file
            self.name = params.name
        else:
            self.config = dict(enabled=True)

        self._dependency_file= dependency_file
        self._caf_requirements_file=caf_requirements_file
        self._system_config=system_config
        self._network_confige=network_config
        self._device_config_file=device_config_file
        self._python_imports = {}
        #self._python_mappings = {}
        self._mandatory_dependencies = {}
        self._optional_dependencies = {}
        self._external_dependencies = {}
        self._load_config()

    def _load_config(self):
        import re
        if self._dependency_file is None:
            raise ValueError("External dependency yaml file is required to proceed")
        if self._system_config is None:
            raise ValueError("System config not provided")
        self._external_dependencies = read_yaml_file(self._dependency_file)
        self._mandatory_dependencies = self._external_dependencies.get("mandatory")
        self._optional_dependencies = self._external_dependencies.get("optional")
        #self._python_mappings = self._external_dependencies.get("python_dependencies")
        if self._caf_requirements_file is None:
            raise ValueError("CAF requirements file is required to proceed")
        with open(self._caf_requirements_file, "r") as cache:
            # read file into a list of lines
            lines = cache.readlines()
            # loop through lines
            for line in lines:
                # skip lines starting with "--".
                if not line.startswith("#"):
                    # replace random amount of spaces (\s) with tab (\t),
                    # strip the trailing return (\n), split into list using
                    # "\t" as the split pattern
                    line = re.sub("\s\s+", "\t", line).strip().split("==")
                    # use first item in list for the key, join remaining list items
                    # with ", " for the value.
                    self._python_imports[line[0]] = line[1]
        if self._system_config:
            if self._system_config.has_option("platform", "pid_file"):
                self.platform_pid_file = self._system_config.get("platform", "pid_file")
            if self._system_config.has_option("platform", "hwid_file"):
                self.platform_hwid_file = self._system_config.get("platform", "hwid_file")


    @property
    def is_enabled(self):
        if isinstance(self.config.get("enabled"), basestring):
            return RawConfigParser._boolean_states.get(self.config.get("enabled", 'false'), False)
        return self.config.get("enabled", False)

    @property
    def mandatory_dependencies(self):
        return self._mandatory_dependencies

    @property
    def optional_dependencies(self):
        return self._optional_dependencies

    @property
    def external_dependencies(self):
        return self._external_dependencies

    @property
    def caf_python_imports(self):
        return self._python_imports

    @property
    def caf_system_config(self):
        return self._system_config

    '''
    Implementing API's defined by Abstract service
    '''
    def set_config(self, config):
        pass

    def get_config(self):
        return self.config

    def _save_data(self):
        with open(self._config_file, "w", 0) as f:
            yaml.safe_dump(self.config, f, default_flow_style=False)
            log.debug("Saved BIST configuration to %s", self._config_file)

    def get_config(self):
        conf = {}
        conf['enabled'] = self.is_enabled
        return conf

    def validate_config(self, config):
        log.debug("Validating the given config %s"%config)
        return True

    def _update_config(self, config):
        self.config.update(config)
        self._save_data()

    def start(self):
        pass

    def stop(self):
        pass

    def verify_external_dependencies(self, command_list=None):
        '''
        verifies and provides JSON object having details on linux system dependencies
        '''
        import subprocess
        external_dependencies_report = {}
        if command_list is None:
            command_list = self.mandatory_dependencies
        log.debug("Dependencies list %s", command_list)
        for command, version in command_list.iteritems():
            retval = execute_which(command)
            if retval:
                try:
                    log.debug("Executing command : %s" % " ".join(version))
                    rval = None
                    if version:
                        version = version[0]
                        v = version.split(' ')
                        cmd_arg0 = v[0]
                        cmd_arg1 = v[1]
                        cmd = []
                        cmd.append(retval)
                        cmd.append(cmd_arg1)
                        rval = subprocess.check_output(cmd, stderr=subprocess.STDOUT)
                    str = "Command exists and and output of provided command is: " + rval
                    external_dependencies_report[command] = str
                    log.debug("External dependencies report %s", external_dependencies_report)
                except subprocess.CalledProcessError as c:
                    log.exception("Called Exception %s %s", c.output, c.returncode)
                    str = "Command" + command + "exists but failed to execute , ret code: " + c.returncode
                    external_dependencies_report[command] = str
                except Exception as ex:
                    log.exception("Error executing command : %s" % " ".join(version))
                    reportStr = "Command" + command + "exists but failed to execute , execption"
                    external_dependencies_report[command] = reportStr
            else:
                log.warning("Command does not exist %s", command)
                str = "Command does not exist " + command
                external_dependencies_report[command] = str

        #print external_dependencies_report
        log.debug("BIST Caf external dependencies report %s", external_dependencies_report)
        return external_dependencies_report


    def verify_caf_python_imports(self, mod_list=None):
        '''
        verifies and provides JSON object having details on python dependencies
        '''
        import pkg_resources
        log.debug("CAF python dependencies %s", mod_list)
        successful_imports = []
        unsuccessful_imports = []
        if mod_list is None:
            mod_list = self._python_imports
        for mod in mod_list.iterkeys():
            try:
                pkg_resources.get_distribution(mod)
                mod_version = pkg_resources.get_distribution(mod).version
                # py_dep_list = self._python_mappings.get(mod)
                # py_mod = py_dep_list[0]
                # if py_mod:
                #     mod_version = "1.0"
                #     mod_imported = False
                #     if py_mod in sys.modules:
                #         mod_imported = True
                #     ret_val = __import__(py_mod)
                #     # if hasattr(py_mod, '__version__'):
                #     #     mod_version = py_mod.__version__
                #     mod_version = pkg_resources.get_distribution(mod).version
                #
                #     if not mod_imported:
                #         if ret_val:
                #             del ret_val
                #         #sys.modules.pop(py_mod)

                str = "Import successful for " + mod + " , version info: " + mod_version
                successful_imports.append(str)
                # else:
                #     str = "Not found mapping for " + mod
                #     unsuccessful_imports.append(str)
            except ImportError:
                error_str = "Unable to import dependency for " + mod
                unsuccessful_imports.append(error_str)
                log.warning("Unable to import %s", mod)
            except Exception as ex:
                error_str = "Unable to import dependency for " + mod
                unsuccessful_imports.append(error_str)
                log.exception("Unable to import %s", mod)

        caf_import_report = {}
        caf_import_report["successful_imports"] = successful_imports
        caf_import_report["unsuccessful_imports"] = unsuccessful_imports
        log.debug("BIST CAf python dependency report %s", caf_import_report)
        return caf_import_report

    def _verify_platform_dependencies(self, command_list=None):
        '''
        verifies and provides JSON object having platform dependencies specified in system config file of CAF
        '''
        if not self.platform_hwid_file:
            hwid_file = PLATFORM_HWID_FILE
        else:
            hwid_file = self.platform_hwid_file

        if not self.platform_pid_file:
            product_id_file = PLATFORM_PRODUCT_ID_FILE
        else:
            product_id_file = self.platform_pid_file
        platform_dependency_report = {}
        if os.path.exists(hwid_file) and os.path.getsize(hwid_file) > 0:
            with open(hwid_file, 'r') as f:
                hwid = f.readline()
            str = "platform hwid id file " + hwid_file + " exists and hwid is "  + hwid
            platform_dependency_report['hwid'] = str
        else:
            log.warning("Platform hwid file is empty or does not exist")
            platform_dependency_report['hwid'] = "Hwid file does not exist or is empty"
        if os.path.exists(product_id_file) and os.path.getsize(product_id_file) > 0:
            with open(product_id_file, 'r') as f:
                product_id = f.readline()
            str = "platform hwid id file " +  product_id_file + " exists and product is "  + product_id
            platform_dependency_report['product_id'] = str
        else:
            log.warning("Platform hwid file is empty or does not exist")
            platform_dependency_report['product_id'] = "product_id file does not exist or is empty"
        log.debug("BIST Platform dependency report %s", platform_dependency_report)
        return platform_dependency_report


    def verify_caf_system_config(self, config=None):
        '''
        verifies and provides JSON object having details of system config file for CAF
        '''
        system_config_report ={}
        system_config_report['platform_dependent_report'] = self._verify_platform_dependencies(config)
        if self._system_config.has_option("api", "port"):
            caf_port = int(self._system_config.get("api", "port"))
        else:
            caf_port = "8443"
        system_config_report['is_port_open'] = self.check_is_port_open(caf_port)
        repo_folder = getSystemConfigValue(self._system_config, "controller", "repo", "/etc")
        work_folder = os.path.dirname(repo_folder)
        if not os.path.exists(work_folder):
            try:
                os.makedirs(work_folder)
                system_config_report['is_work_dir_writable'] = self.is_writable(work_folder)
                os.rmdir(work_folder)
            except OSError:
                system_config_report['is_work_dir_writable'] = False
        else:
            system_config_report['is_work_dir_writable'] = self.is_writable(work_folder)
        log.debug("BIST: CAF System config report %s", system_config_report)
        return system_config_report

    def check_bridge_status(self, bridge_id):
        '''
        Checks if bridge has been setup required by CAF and specified in network_config
        '''
        from socket import socket, AF_INET, SOCK_DGRAM
        import fcntl , struct, sys
        SIOCGIFFLAGS = 0x8913
        null256 = '\0'*256

        # Create a socket so we have a handle to query
        s = socket(AF_INET, SOCK_DGRAM)

        # Call ioctl(  ) to get the flags for the given interface
        result = fcntl.ioctl(s.fileno(  ), SIOCGIFFLAGS, bridge_id + null256)

        # Extract the interface's flags from the return value
        flags, = struct.unpack('H', result[16:18])

        # Check "UP" bit and print a message
        up = flags & 1
        print ('DOWN', 'UP')[up]
        if up:
            return True
        else:
            return False

    def verify_caf_network_config(self, network_config=None, check_connectivity=None):
        '''
        Verifies and provides report for networking dependencies specified in network config file
        '''
        import yaml
        caf_network_status = {}
        network_info = []
        for ni in get_network_interfaces():
            network_info.append(str(ni))
        caf_network_status['list_interfaces'] = network_info
        if network_config is None:
            network_config_file = getNetworkConfigFile()
            network_config = yaml.safe_load(file(network_config_file))
        hbs = network_config.get('hosting_bridges')
        for i, bridge_id in enumerate(hbs):
            caf_network_status['bridge_status'] = dict(bridge_id=bridge_id, status=self.check_bridge_status(bridge_id))
        log.debug("BIST: Caf network status report %s", caf_network_status)
        log.debug("BIST: check connectivity param %s", check_connectivity)
        if check_connectivity:
            retval = execute_ping(check_connectivity, timeout=2)
            caf_network_status["check_connectivity"] = {check_connectivity:retval}

        return caf_network_status


    def verify_caf_device_config(self):
        pass

    def check_is_port_open(self, port):
        from contextlib import closing
        import socket
        host = "127.0.0.1"
        with closing(socket.socket(socket.AF_INET, socket.SOCK_STREAM)) as sock:
            sock.settimeout(2)
            if sock.connect_ex((host, port)) == 0:
                log.info("Port %s is open", port)
                return True
            else:
                log.warning("Port %s is not open", port)
                return False


    def is_writable(self, path):
        import tempfile
        try:
            testfile = tempfile.TemporaryFile(dir = path)
            testfile.close()
        except OSError as e:
            return False
        except Exception as ex:
            return False
        return True

    def collect_system_status(self):
        log.debug("BIST: Collecting system status report for CAF")
        from handlesystemstatus import CollectSystemStatus
        sys_stat = CollectSystemStatus()
        system_report = sys_stat.collect_system_status()
        return system_report

    @classmethod
    def getInstance(cls, *args):
        '''
        Returns a singleton instance of the class
        '''
        if not cls.__singleton:
            cls.__singleton = BuiltInSelfTestModule(*args)
        return cls.__singleton

if __name__ == '__main__' and __package__ is None:
    import sys
    import yaml
    import logging
    #sys.path.insert(0, '../utils/')
    import ConfigParser
    import argparse

    logging.basicConfig(
         level=logging.DEBUG,
         datefmt='%H:%M:%S'
    )

    # set up logging to console
    console = logging.StreamHandler()
    console.setLevel(logging.DEBUG)
    # set a format which is simpler for console use
    # formatter = logging.Formatter('%(name)-12s: %(levelname)-8s %(message)s')
    # console.setFormatter(formatter)
    # add the handler to the root logger
    logging.getLogger().addHandler(console)

    parser = argparse.ArgumentParser(description="BIST program")
    parser.add_argument("-sys_config", "--system_config_file", dest="system_config_file",help="provide system config file with abs path", required=True)
    parser.add_argument("-caf_req", "--caf_req_file", dest="caf_requirements",help="provide caf requirements file with abs path", required=True)
    parser.add_argument("-net_config", "--net_config_file", dest="network_config_file", help="provide network config file with abs path", required=True)
    parser.add_argument("-caf_dep", "--caf_dep_file", dest="dependency_file", help="provide dependency file with abs path", required=True)
    parser.add_argument("--verify_system_config", action="store_true", help="verify caf system config")
    parser.add_argument("--verify_system_dep", action="store_true", help="verify caf system dependencies")
    parser.add_argument("--verify_network_config", action="store_true", help="verify caf network config")
    parser.add_argument("--verify_system_status", action="store_true", help="verify caf system status")
    parser.add_argument("--verify_python_dependency", action="store_true", help="verify caf python dependencies")
    parser.add_argument("--verify_all", action="store_true", help="verify caf python dependencies")

    args = parser.parse_args()
    print args
    system_config_file = args.system_config_file
    network_config_file = args.network_config_file
    caf_requirements = args.caf_requirements
    dependency_file = args.dependency_file
    if dependency_file is None:
        dependency_file = "/home/rammohan/caf_external_dependencies.yaml"
    if caf_requirements is None:
        caf_requirements = "/home/rammohan/caf-requirements.txt"
    if network_config_file is not None:
        network_config_file = getNetworkConfigFile()
    network_config = yaml.safe_load(file(network_config_file))
    if system_config_file is not None:
        system_config_file = "/home/rammohan/system-config.ini"
    appConfig = ConfigParser.SafeConfigParser()
    appConfig.read(system_config_file)
    appConfig.system_config_file = system_config_file

    caf_depends = BuiltInSelfTestModule(dependency_file=dependency_file, caf_requirements_file=caf_requirements, system_config=appConfig, network_config=network_config)
    print "Printing verify options "
    if args.verify_system_dep or args.verify_all:
        system_dependency_report = caf_depends.verify_external_dependencies(caf_depends.mandatory_dependencies)
        print "Printing CAF System dependency report======"
        print json.dumps(system_dependency_report, indent=2)
    if args.verify_python_dependency or args.verify_all:
        caf_python_dep_report = caf_depends.verify_caf_python_imports(caf_depends.caf_python_imports)
        print "CAF python dependencies report============"
        print json.dumps(caf_python_dep_report, indent=2)
    if args.verify_network_config or args.verify_all:
        network_dep_report = caf_depends.verify_caf_network_config()
        print "CAF Network dependency report============"
        print json.dumps(network_dep_report, indent=2)
    if args.verify_system_config or args.verify_all:
        system_config_report = caf_depends.verify_caf_system_config()
        print "System Config verification report========="
        print json.dumps(system_config_report, indent=2)
    if args.verify_system_status or args.verify_all:
        system_status = caf_depends.collect_system_status()
        print "System status where CAF is running========"
        print json.dumps(system_status, indent=2)
