'''
Created on Jan 10, 2019

@author: varshven

Copyright (c) 2012 by Cisco Systems, Inc.
All rights reserved.
'''

import logging
import falcon
import json
import os
import shutil
import tempfile
import time
import tarfile
import yaml
import subprocess
import sys
import datetime
import hashlib
import xml.etree.ElementTree as ET
from xml.dom import minidom
from collections import defaultdict
from ..utils.utils import Utils, PACKAGE_MANIFEST_NAME, APP_METADATA_FILE_NAME, APP_DESCRIPTOR_NAME
from ..utils.commandwrappers import *
from ..utils.infraexceptions import *
from appfw.runtime.caf_abstractservice import CAFAbstractService
import re

APP_DESCRIPTOR_NAME = "package.yaml"
PACKAGE_MANIFEST_NAME = "package.mf"
APP_METADATA_FILE_NAME = ".package.metadata"
log = logging.getLogger("runtime.hosting")
QEMU_DISK_EXT           = ".qcow2"
CPU_UNITS_FACTOR        = 2.5
APP_TYPE_VM             = "vm"
DFLT_NUM_VCPU           = "1"
DFLT_SOCKS_PER_CORE     = "1"
DFLT_CPU_CORES          = "1"
OVF_SPEC_V1             = "1.0"
OVF_RES_CPU             = "3"
OVF_RES_MEMORY          = "4"
OVF_RES_NW              = "10"
OVF_RES_SERIAL_PORT     = "21"
OVF_RES_USB             = "23"
OVF_RES_DISK_DRIVE      = "17"
OVF_RES_IDE_CNTRLR      = "5"
OVF_RES_SCSI_CNTRLR     = "6"
OVF_RES_SATA_CNTRLR     = "20"
ETH_INTF_NAME           = "eth"
SERIAL_INTF_NAME        = "HOST_SERIAL_DEV"
USB_INTF_NAME           = "HOST_USB_DEV"
VM_APP_DESCRIPTOR_TEMPLATE = """

descriptor-schema-version: "2.7"

info:
  name: IOx VM App
  description: "IOx VM app converted from OVA"
  version: "1.0"
  author-link: "http://www.cisco.com"
  author-name: "Cisco Systems"

app:
  type: vm
  cpuarch: x86_64
  resources:
    profile: custom
    cpu: 2500
    memory: 64
    disk: 2
    graphics:
        vnc: true
  # Specify runtime and startup
  startup:
    ostype: linux
    qemu-guest-agent: False
"""

class VMPackageTools(object):
    def __init__(self):
        self.vmapp_dir = ""
        self.final_pkg_path = ""
        self.tmpUploadDir = ""
        self.ovaextractedDir = ""
        self.ovf_descriptor = {}

    def extractOVA(self, ova_path):
        try:
            with tarfile.open(ova_path, 'r', errors='ignore') as tar:
                Utils.check_for_absolutepaths(tar)
                tar_extractall(ova_path, self.ovaextractedDir)
        except Exception as cause:
            log.exception("Unable to extract %s to %s Error:%s" % (ova_path, self.ovaextractedDir, str(cause)))
            raise Exception("Unable to extract %s to %s Error:%s" % (ova_path, self.ovaextractedDir, str(cause)))


    def setupTempDirs(self):
        # Read upload location from system config, default to /tmp if not specified
        tmpUploadDir = Utils.getSystemConfigValue('controller', 'upload_dir', '/tmp')
        self.tmpUploadDir = tmpUploadDir
        self.vmapp_dir = tempfile.mkdtemp("", "vmapp_", tmpUploadDir)
        log.debug("Temporary VM app directory created at: %s" % (self.vmapp_dir))
        self.ovaextractedDir = tempfile.mkdtemp("", "ovaextr_", self.vmapp_dir)
        log.debug("OVA contents will be extracted at: %s" % (self.ovaextractedDir))

    def moveFiletoAppdir(self, incoming_file_path, incoming_file):
        log.debug("Moving file from %s to %s " % (incoming_file_path, self.vmapp_dir))
        try:
            shutil.move(incoming_file_path , os.path.join(self.vmapp_dir, incoming_file))
        except Exception as ex:
            log.error("Unable to move file %s, %s" % (incoming_file, str(ex)))
            raise Exception(ex)

    def ConvertToMegaHertz(self, unit, quantity):
        size = int(quantity)
        if (unit == "hertz * 10^3") or (unit == "KiloHertz") or (unit == "KHz"):
            size = size * 1000
        elif (unit == "hertz * 10^6") or (unit == "MegaHertz") or (unit == "MHz"):
            return size
        elif (unit == "hertz * 10^9") or (unit == "GigaHertz") or (unit == "GHz"):
            size = size * 1000 * 1000 * 1000
        elif (unit == "hertz * 10^12") or (unit == "TeraHertz") or (unit == "THz"):
            size = size * 1000 * 1000 * 1000 * 1000
        else:
            raise Exception("Given CPU units is not handled")
        return (size / (1000 * 1000))


    def ConvertToMegaBytes(self, unit, quantity):
        size = int(quantity)
        if (unit == "byte * 2^10") or (unit == "KiloBytes") or (unit == "KB"):
            size = size * 1024
        elif (unit == "byte * 2^20") or (unit == "MegaBytes") or (unit == "MB"):
            return size
        elif (unit == "byte * 2^30") or (unit == "GigaBytes") or (unit == "GB"):
            size = size * 1024 * 1024 * 1024
        elif (unit == "byte * 2^40") or (unit == "TeraBytes") or (unit == "TB"):
            size = size * 1024 * 1024 * 1024 * 1024
        elif (unit == "bytes") or (unit == "Bytes") or (unit == "B"):
            size = size
        else:
            raise Exception("Given Memory bytes units is not handled")
        return (size / (1024 * 1024))


    def extractOVFContent(self):
        ovfcount, extracted_text, ovffiles = 0, "", []
        files_list = os.listdir(self.ovaextractedDir)
        for files in files_list:
            if files.endswith("ovf"):
                ovfcount += 1
                ovffiles.append(files)
        if ovfcount != 1:
            log.error("Unexpected number of OVF files detected in the archive")
            raise MandatoryFileMissingError("Unexpected number of OVF files in the archive")
        try:
            with open(os.path.join(self.ovaextractedDir, ovffiles[0]), "r") as f:
                extracted_text = f.read()
                log.debug("Extracted ovf text is %s" % extracted_text)
        except Exception as ex:
            log.error("Failed to read content from OVF file, exception: %s" % str(ex))
            raise MandatoryDataMissingError("Failed to read content from OVF file, exception: %s" % str(ex))
        return extracted_text

    def parseProductSection(self, product_element):
        prod_dict = {}
        log.debug("Parsing product section of the Ovf descriptor")
        try:
            for prod_item in product_element.items():
                if 'Product' in prod_item[0]:
                    prod_dict['Name'] = prod_item[1]
                elif 'Info' in prod_item[0]:
                    prod_dict['Desc'] = prod_item[1]
                elif 'Version' in prod_item[0]:
                    prod_dict['Version'] = prod_item[1]
                elif 'VendorUrl' in prod_item[0]:
                    prod_dict['VendorUrl'] = prod_item[1]
                elif 'Vendor' in prod_item[0]:
                    prod_dict['Vendor'] = prod_item[1]
                else:
                    pass
            if prod_dict:
                if prod_dict.get("Version") is None:
                    prod_dict['Version'] = "1.0"
                if prod_dict.get("VendorUrl") is None:
                    prod_dict['VendorUrl'] = "http://www.cisco.com"
                if prod_dict.get("Vendor") is None:
                    prod_dict['Vendor'] = "Cisco Systems"
                self.ovf_descriptor['VirtualSystem']['Product'].append(prod_dict)
        except Exception as ex:
            log.error("Error while parsing Product section")
            raise Exception(ex)
        return

    def parseVirtualHardwareSection(self, virthw_section):
        virthw = list(virthw_section)
        self.ovf_descriptor['VirtualSystem']['VirtHW'] = {}
        self.ovf_descriptor['VirtualSystem']['VirtHW']['Item'] = []
        self.ovf_descriptor['VirtualSystem']['VirtHW']['StorageItem'] = []
        self.ovf_descriptor['VirtualSystem']['VirtHW']['PortItem'] = []
        log.debug("Parsing virtual hardware section of the Ovf descriptor")
        for virthw_child in virthw:
            if 'System' in virthw_child.tag:
                sys_child = list(virthw_child)
                try:
                    for each_sys_child in sys_child:
                        if each_sys_child.tag.find("VirtualSystemType") != -1:
                            self.ovf_descriptor['VirtualSystem']['VirtHW']['Type'] = each_sys_child.text
                        else:
                            pass
                except Exception as ex:
                    log.error("Error in assigning virtual hardware type")
                    raise Exception(ex)
            elif 'Item' in virthw_child.tag:
                tmp_dict = {}
                item_children = list(virthw_child)
                try:
                    for item_child in item_children:
                        if item_child.tag.find("ResourceType") != -1:
                            tmp_dict['Type'] = item_child.text
                        elif item_child.tag.find("ElementName") != -1:
                            tmp_dict['Name'] = item_child.text
                        elif item_child.tag.find("Description") != -1:
                            tmp_dict['Desc'] = item_child.text
                        elif item_child.tag.find("InstanceID") != -1:
                            tmp_dict['Id'] = item_child.text
                        elif item_child.tag.find("Parent") != -1:
                            tmp_dict['ParentId'] = item_child.text
                        elif item_child.tag.find("AllocationUnits") != -1:
                            tmp_dict['Unit'] = item_child.text
                        elif item_child.tag.find("Reservation") != -1:
                            tmp_dict['Resv'] = item_child.text
                        elif item_child.tag.find("VirtualQuantity") != -1:
                            tmp_dict['Qty'] = item_child.text
                        elif item_child.tag.find("ProcessorArchitecture") != -1:
                            tmp_dict['CpuArch'] = item_child.text
                        elif item_child.tag.find("CoresPerSocket") != -1:
                            tmp_dict['CpuCores'] = item_child.text
                        else:
                            pass
                    self.ovf_descriptor['VirtualSystem']['VirtHW']['Item'].append(tmp_dict)
                except Exception as ex:
                    log.error("Error in assigning virtual hardware Items")
                    raise Exception(ex)
            elif 'StorageItem' in virthw_child.tag:
                tmp_dict = {}
                storage_item_children = list(virthw_child)
                try:
                    for item_child in storage_item_children:
                        if item_child.tag.find("ResourceType") != -1:
                            tmp_dict['Type'] = item_child.text
                        elif item_child.tag.find("Parent") != -1:
                            tmp_dict['ParentId'] = item_child.text
                        else:
                            pass
                    self.ovf_descriptor['VirtualSystem']['VirtHW']['StorageItem'].append(tmp_dict)
                except Exception as ex:
                    log.error("Error in assigning virtual hardware storage items")
                    raise Exception(ex)
            elif 'EthernetPortItem' in virthw_child.tag:
                tmp_dict = {}
                port_item_children = list(virthw_child)
                try:
                    for item_child in port_item_children:
                        if item_child.tag.find("ResourceType") != -1:
                            tmp_dict['Type'] = item_child.text
                        elif item_child.tag.find("InstanceID") != -1:
                            tmp_dict['Id'] = item_child.text
                        else:
                            pass
                    self.ovf_descriptor['VirtualSystem']['VirtHW']['PortItem'].append(tmp_dict)
                except Exception as ex:
                    log.error("Error in assigning virtual port Items")
                    raise Exception(ex)
            else:
                pass
        return

    def parseBootOrderSection(self, boot_order):
        boot_dict = {}
        log.debug("Parsing boot order section of the Ovf descriptor")
        try:
            for boot_item in boot_order.items():
                if 'instanceId' in boot_item[0]:
                    boot_dict['Id'] = boot_item[1]
                elif 'type' in boot_item[0]:
                    boot_dict['Type'] = boot_item[1]
            self.ovf_descriptor['VirtualSystem']['BootSection'].append(boot_dict)
        except Exception as ex:
            log.error("Error occurred while parsing Boot order section %s" % str(ex))
            raise Exception(ex)
        return

    def parseMachineSection(self, machine):
        machine_children = list(machine)
        self.ovf_descriptor['VirtualSystem']['VboxSection'] = {}
        log.debug("Parsing VboxSection of the Ovf descriptor")
        # First look for Hardware under Machine, then Hardware has three children of our concern: Boot (Order), DeviceFilters (USB), UART (Serial port)
        for hw in machine_children:
            if hw.tag.find("Hardware") != -1:
                hw_children = list(hw)
                log.debug("allocating possible dictionaries/lists for hardware section")
                self.ovf_descriptor['VirtualSystem']['VboxSection']['Hardware'] = {}
                self.ovf_descriptor['VirtualSystem']['VboxSection']['Hardware']['Boot'] = {}
                self.ovf_descriptor['VirtualSystem']['VboxSection']['Hardware']['Usb'] = {}
                self.ovf_descriptor['VirtualSystem']['VboxSection']['Hardware']['Serial'] = {}
                for hw_child in hw_children:
                    if hw_child.tag.find("Boot") != -1:
                        try:
                            boot_children = list(hw_child)
                            for boot_child in boot_children:
                                if boot_child.tag.find("Order") != -1:
                                    boot_dict = {}
                                    for boot_item in boot_child.items():
                                        if 'position' in boot_item[0]:
                                            boot_dict['Pos'] = boot_item[1]
                                        elif 'device' in boot_item[0]:
                                            boot_dict['Dev'] = boot_item[1]
                                        else:
                                            pass
                                    if 'Order' not in self.ovf_descriptor['VirtualSystem']['VboxSection']['Hardware']['Boot']:
                                        self.ovf_descriptor['VirtualSystem']['VboxSection']['Hardware']['Boot']['Order'] = []
                                    self.ovf_descriptor['VirtualSystem']['VboxSection']['Hardware']['Boot']['Order'].append(boot_dict)
                                else:
                                    pass
                        except Exception as ex:
                            log.error("Error occurred while parsing boot section %s" % str(ex))
                            raise Exception(ex)
                    elif hw_child.tag.find("USB") != -1:
                        try:
                            usb_children = list(hw_child)
                            for usb_child in usb_children:
                                if 'DeviceFilters' in usb_child.tag:
                                    devices = list(usb_child)
                                    for dev in devices:
                                        if dev.tag.find("DeviceFilter") != -1:
                                            device_dict = {}
                                            for dev_item in dev.items():
                                                if 'name' in dev_item[0]:
                                                    device_dict['Name'] = dev_item[1]
                                                elif 'active' in dev_item[0]:
                                                    device_dict['Active'] = dev_item[1]
                                            if 'Device' not in self.ovf_descriptor['VirtualSystem']['VboxSection']['Hardware']['Usb']:
                                                self.ovf_descriptor['VirtualSystem']['VboxSection']['Hardware']['Usb']['Device'] = []
                                            self.ovf_descriptor['VirtualSystem']['VboxSection']['Hardware']['Usb']['Device'].append(device_dict)
                                        else:
                                            pass
                                else:
                                    pass
                        except Exception as ex:
                            log.error("Error occurred while parsing USB section %s" % str(ex))
                            raise Exception(ex)
                    elif hw_child.tag.find("UART") != -1:
                        try:
                            serial_children = list(hw_child)
                            for serial in serial_children:
                                if serial.tag.find("Port") != -1:
                                    port_dict = {}
                                    for port_item in serial.items():
                                        if 'slot' in port_item[0]:
                                            port_dict['Slot'] = port_item[1]
                                        elif 'enabled' in port_item[0]:
                                            port_dict['Enable'] = port_item[1]
                                        else:
                                            pass
                                    if 'Port' not in self.ovf_descriptor['VirtualSystem']['VboxSection']['Hardware']['Serial']:
                                        self.ovf_descriptor['VirtualSystem']['VboxSection']['Hardware']['Serial']['Port'] = []
                                    self.ovf_descriptor['VirtualSystem']['VboxSection']['Hardware']['Serial']['Port'].append(port_dict)
                                else:
                                    pass
                        except Exception as ex:
                            log.error("Error occurred while parsing UART section %s" % str(ex))
                            raise Exception(ex)
                    else:
                        pass
            else:
                pass
        return

    def parseDiskSection(self, disk_section):
        self.ovf_descriptor['Disks']['Disk'] = []
        log.debug("Parsing disk section of the Ovf descriptor")
        try:
            for DiskElements in disk_section.iter():
                if DiskElements.tag.find("Disk") != -1:
                    tmp_dict = {}
                    for diskelement in DiskElements.items():
                        if 'capacityAllocationUnits' in diskelement[0]:
                            tmp_dict['Units'] = diskelement[1]
                        elif 'capacity' in diskelement[0]:
                            tmp_dict['Capacity'] = diskelement[1]
                        elif 'populatedSize' in diskelement[0]:
                            tmp_dict['Size'] = diskelement[1]
                        else:
                            pass
                    if tmp_dict:
                        self.ovf_descriptor['Disks']['Disk'].append(tmp_dict)
                else:
                    pass
        except Exception as ex:
            log.error("Error occurred while parsing disk section %s" % str(ex))
            raise Exception(ex)

    def parseVirtualSystem(self, virtsys_section):
        ovf_id = ""
        try:
            self.ovf_descriptor['VirtualSystem']['BootSection'] = []
            self.ovf_descriptor['VirtualSystem']['Product'] = []
        except Exception as ex:
            log.error("Error in assigning bootsection and product lists")
            raise Exception(ex)
        try:
            for vs_item in virtsys_section.items():
                if 'id' in vs_item[0]:
                    self.ovf_descriptor['VirtualSystem']['Id'] = vs_item[1]
                    ovf_id = self.ovf_descriptor['VirtualSystem']['Id']
        except Exception as ex:
            log.error("Error in parsing virtual system section items")
            raise Exception(ex)

        VirtSysChildren = list(virtsys_section)
        for vs_child in VirtSysChildren:
            if 'Name'in vs_child.tag:
                self.ovf_descriptor['VirtualSystem']['Name'] = ''.join(vs_child.itertext())
            elif 'Info' in vs_child.tag:
                self.ovf_descriptor['VirtualSystem']['Info'] = ''.join(vs_child.itertext())
            elif 'AnnotationSection' in vs_child.tag: #requires an extra level of check AnnotationSection > Annotation
                annot = list(vs_child)
                for ann in annot:
                    if ann.tag.find('Annotation') != -1:
                        self.ovf_descriptor['VirtualSystem']['Annotation'] = ''.join(ann.itertext())
            # Product section
            elif 'ProductSection' in vs_child.tag:
                self.parseProductSection(vs_child)

            #Virtual Hardware Sectiom
            elif 'VirtualHardwareSection' in vs_child.tag:
                self.parseVirtualHardwareSection(vs_child)

            #Boot Order section
            elif 'BootOrderSection' in vs_child.tag:
                self.parseBootOrderSection(vs_child)

            #Machine section
            elif 'Machine' in vs_child.tag:
                self.parseMachineSection(vs_child)

            else:
                #other sections are not of concern
                pass
        return ovf_id

    def parseOVF(self, ovf_text):
        ovf_id = ""
        self.ovf_descriptor['Disks'] = {}
        self.ovf_descriptor['VirtualSystem'] = {}
        try:
            root = ET.fromstring(ovf_text)
        except Exception as ex:
            log.error("Could not get the root element of the ovf descriptor")
            raise Exception(ex)
        main_children = list(root)
        for child in  main_children:
            # Disk Section has a section called Disk and there could be multiple of those Disk items
            if child.tag.find("DiskSection") != -1:
                self.parseDiskSection(child)
            # Virtual System section has sub sections and our concerns are: Product, Virtual Hardware, Boot Order and Machine
            elif child.tag.find("VirtualSystem") != -1:
                ovf_id = self.parseVirtualSystem(child)
            else:
                #other sections are not needed
                pass
        if not ovf_id:
            log.debug("Setting default Id for VM App pkg")
            ovf_id = "vm"
            self.ovf_descriptor['VirtualSystem']['Id'] = ovf_id
        return ovf_id

    def setDiskResources(self, pkg_descriptor):
        try:
            if "Disk" in self.ovf_descriptor["Disks"] and len(self.ovf_descriptor["Disks"]["Disk"]) > 0:
                capacity = 0
                disks = self.ovf_descriptor["Disks"]["Disk"]
                for i in range(len(disks)):
                    if 'Units' not in disks[i]:
                        disks[i]['Units'] = "Bytes"
                    capacity += self.ConvertToMegaBytes(disks[i]['Units'], disks[i]['Capacity'])
                pkg_descriptor["app"]["resources"]["disk"] = capacity
        except Exception as ex:
            log.error("Error occurred in setting disk resources in package.yaml")
            raise Exception(ex)
        return

    def setCpuAttributes(self, cpuitem, pkg_descriptor):
        try:
            if 'CpuArch' in cpuitem:
                pkg_descriptor['app']['cpuarch'] = cpuitem['CpuArch']
            if 'Resv' in cpuitem:
                cpu_freq = self.ConvertToMegaHertz(cpuitem['Unit'], cpuitem['Resv'])
                cpu_units = float(cpu_freq) * CPU_UNITS_FACTOR
                pkg_descriptor['app']['resources']['cpu'] = str(int(cpu_units))
            if 'Qty' in cpuitem:
                pkg_descriptor['app']['resources']['vcpu'] = cpuitem['Qty']
                vcpu = pkg_descriptor['app']['resources']['vcpu']
            if 'CpuCores' in cpuitem:
                pkg_descriptor['app']['resources']['cpu-topology'] = {}
                pkg_descriptor['app']['resources']['cpu-topology']['cores'] = cpuitem['CpuCores']
                pkg_descriptor['app']['resources']['cpu-topology']['sockets-per-core'] = str(int(vcpu) / int(cpuitem['CpuCores']))
        except Exception as ex:
            log.error("Error occurred in setting cpu resources in package.yaml")
            raise Exception(ex)

    def setMemoryAttributes(self, memitem, pkg_descriptor):
        try:
            capacity = self.ConvertToMegaBytes(memitem['Unit'], memitem['Qty'])
            pkg_descriptor['app']['resources']['memory'] = capacity
        except Exception as ex:
            log.error("Error occurred in setting memory resources in package.yaml")
            raise Exception(ex)
        return

    def setNetworkAttributes(self, pkg_descriptor):
        try:
            if 'network' not in pkg_descriptor['app']['resources']:
                pkg_descriptor['app']['resources']['network'] = []
            networks = pkg_descriptor['app']['resources']['network']
            eachInterface = {}
            eachInterface['interface-name'] = ETH_INTF_NAME + str(len(networks))
            networks.append(eachInterface)
        except Exception as ex:
            log.error("Error occurred in setting network section in package.yaml")
            raise Exception(ex)
        return

    def setSerialDevices(self, deviceitem, pkg_descriptor):
        try:
            pkg_descriptor['app']['resources']['devices'] = []
            device = {}
            device['type'] = "serial"
            if 'Name' in deviceitem:
                device['label'] = deviceitem['Name']
            else:
                device['label'] = SERIAL_INTF_NAME + str(len(devices))
            device['usage'] = deviceitem['Desc']
            pkg_descriptor['app']['resources']['devices'].append(device)
        except Exception as ex:
            log.error("Error occurred in setting serial device section in package.yaml")
            raise Exception(ex)
        return

    def setUSBDevices(self, pkg_descriptor):
        try:
            if len(self.ovf_descriptor[':']['VboxSection']['Hardware']['Usb']['Device']) != 0:
                devices = self.ovf_descriptor['VirtualSystem']['VboxSection']['Hardware']['Usb']['Device']
                if 'devices' not in pkg_descriptor['app']['resources']:
                    pkg_descriptor['app']['resources']['devices'] = []

                for i in range(len(devices)):
                    device = {}
                    device['type'] = "usbdev"
                    if 'Name' in devices[i]:
                        device['label'] = devices[i]['Name']
                    else:
                        device['label'] = USB_INTF_NAME + str(len(devices))
                    pkg_descriptor['app']['resources']['devices'].append(device)
            else:
                pass
        except Exception as ex:
            log.error("Error occurred in setting usb device section in package.yaml")
            raise Exception(ex)

    def setSerialPortDevices(self, pkg_descriptor):
        try:
            if 'VboxSection' in self.ovf_descriptor['VirtualSystem'] and 'Hardware' in self.ovf_descriptor['VirtualSystem']['VboxSection'] and len(self.ovf_descriptor['VirtualSystem']['VboxSection']['Hardware']['Serial']['Port']) > 0:
                if 'devices' not in pkg_descriptor['app']['resources']:
                    pkg_descriptor['app']['resources']['devices'] = []

                devices = self.ovf_descriptor['VirtualSystem']['VboxSection']['Hardware']['Serial']['Port']
                currdevices = pkg_descriptor['app']['resources']['devices']
                for i in range(len(devices)):
                    device = {}
                    device["type"] = "serial"
                    device["label"] = SERIAL_INTF_NAME + str(len(currdevices))
                    currdevices.append(device)
        except Exception as ex:
            log.error("Error occurred in setting serial port section in package yaml")
            raise Exception(ex)

    def setDiskinPackageYaml(self, pkg_descriptor):
        try:
            ParentInstId, ScsiInstId, SataInstId, IdeInstId = "", "", "", ""
            disk = {}
            if "VirtHW" in self.ovf_descriptor['VirtualSystem'] and 'Item' in self.ovf_descriptor['VirtualSystem']['VirtHW']:
                for i in range(len(self.ovf_descriptor['VirtualSystem']['VirtHW']['Item'])):
                    resType = self.ovf_descriptor['VirtualSystem']['VirtHW']['Item'][i]['Type']
                    # currItem = self.ovf_descriptor['VirtualSystem']['VirtHW']['Item'][i]
                    if resType == OVF_RES_DISK_DRIVE:
                        ParentInstId = self.ovf_descriptor['VirtualSystem']['VirtHW']['Item'][i]['ParentId']
                    elif resType == OVF_RES_SCSI_CNTRLR:
                        ScsiInstId = self.ovf_descriptor['VirtualSystem']['VirtHW']['Item'][i]['Id']
                    elif resType == OVF_RES_SATA_CNTRLR:
                        SataInstId = self.ovf_descriptor['VirtualSystem']['VirtHW']['Item'][i]['Id']
                    elif resType == OVF_RES_IDE_CNTRLR:
                        IdeInstId = self.ovf_descriptor['VirtualSystem']['VirtHW']['Item'][i]['Id']
                    else:
                        continue

                if self.ovf_descriptor['Version'] != OVF_SPEC_V1:
                    if 'StorageItem' in self.ovf_descriptor['VirtualSystem']['VirtHW']:
                        for i in range(len(self.ovf_descriptor['VirtualSystem']['VirtHW']['StorageItem'])):
                            resType = self.ovf_descriptor['VirtualSystem']['VirtHW']['StorageItem'][i]['Type']
                            if resType == OVF_RES_DISK_DRIVE:
                                ParentInstId = self.ovf_descriptor['VirtualSystem']['VirtHW']['StorageItem'][0]['ParentId']
                                break
                    else:
                        pass
                else:
                    pass

                if ParentInstId == ScsiInstId or ParentInstId == SataInstId:
                    disk["target-dev"] = "sda"
                elif ParentInstId == IdeInstId:
                    disk["target-dev"] = "hda"
                else:
                    pass
            
            disk["file"] = self.ovf_descriptor["VirtualSystem"]["Id"] + QEMU_DISK_EXT
            pkg_descriptor["app"]["startup"]["disks"] = []
            pkg_descriptor["app"]["startup"]["disks"].append(disk)
        except Exception as ex:
            log.error("Error occurred in setting disk section in package.yaml")
            raise Exception(ex)

    def generateYaml(self, package_name, qemu_present, ostype):
            """
            This method will generate the package.yaml inside the dest dir and return the path of it for given ova.
            In case of any failures this method will return 'None'
            :param package_name:
            :param qemu_present:
            :return:
            """
            log.debug("Generating app descriptor file from given ova!")
            try:
                pkg_descriptor = yaml.load(VM_APP_DESCRIPTOR_TEMPLATE)
            except Exception as ex:
                log.error("Error occurred while loading default package descriptor")
                raise Exception(ex)
            bootFromDisk = False
            try:
                if 'BootSection' in self.ovf_descriptor['VirtualSystem'] and len(self.ovf_descriptor['VirtualSystem']['BootSection']) > 0:
                    bootsect = self.ovf_descriptor['VirtualSystem']['BootSection']
                    for i in range(len(bootsect)):
                        if bootsect[i].Type == "disk":
                            bootFromDisk = True
                            break
                    if not bootFromDisk:
                        log.error("Unsupported boot order from OVF file", "Could not find hard disk in boot order section")
                        raise InvalidConfigError("Unsupported boot order from OVF file")
                elif 'VboxSection' in self.ovf_descriptor['VirtualSystem'] and 'Hardware' in self.ovf_descriptor['VirtualSystem']['VboxSection'] and 'Boot' in self.ovf_descriptor['VirtualSystem']['VboxSection']['Hardware'] and 'Order' in self.ovf_descriptor['VirtualSystem']['VboxSection']['Hardware']['Boot']:
                    boot_order = self.ovf_descriptor['VirtualSystem']['VboxSection']['Hardware']['Boot']['Order']
                    for i in range(len(boot_order)):
                        if boot_order[i].Dev == "HardDisk":
                            bootFromDisk = True
                            break
                    if not bootFromDisk:
                        log.error("Unsupported boot order from OVF file", "Could not find hard disk in boot order section")
                        raise InvalidConfigError("Unsupported boot order from OVF file")
            except Exception as ex:
                log.error("Error occurred in retrieving components from ovf")
                raise Exception(ex)

            # OVF generated by latest vmware ovftool 4.3.0 does not encode OVF spec version number.
            # But it supports ovf spec upto 1.0, So marking here Version to 1.0 if OVF file does not have OVF version.
            if 'Version' not in self.ovf_descriptor:
                self.ovf_descriptor['Version'] = OVF_SPEC_V1

            #info section
            #If user provides package_name, then it takes priority over the ovf attribute
            try:
                if package_name != "":
                    pkg_descriptor["info"]["name"] = package_name
                else:
                    if "Id" in self.ovf_descriptor["VirtualSystem"]:
                        pkg_descriptor["info"]["name"] = self.ovf_descriptor["VirtualSystem"]["Id"]
            except Exception as ex:
                log.error("Could not set package descriptor name %s" % str(ex))
                raise Exception(ex)

            try:
                if "Annotation" in self.ovf_descriptor["VirtualSystem"]:
                    pkg_descriptor["info"]["description"] = self.ovf_descriptor["VirtualSystem"]["Annotation"]
                elif "Info" in self.ovf_descriptor["VirtualSystem"]:
                    pkg_descriptor["info"]["description"] = self.ovf_descriptor["VirtualSystem"]["Info"]
            except Exception as ex:
                log.error("Could not set Description in package.yaml %s" % str(ex))
                raise Exception(ex)

            try:
                if "Product" in self.ovf_descriptor["VirtualSystem"] and len(self.ovf_descriptor["VirtualSystem"]["Product"]) > 0:
                    log.debug("product info - %s " % self.ovf_descriptor["VirtualSystem"]["Product"]) 
                    pkg_descriptor["info"]["version"] = self.ovf_descriptor["VirtualSystem"]["Product"][0].Version
                    pkg_descriptor["info"]["author-name"] = self.ovf_descriptor["VirtualSystem"]["Product"][0].Vendor
                    pkg_descriptor["info"]["author-link"] = self.ovf_descriptor["VirtualSystem"]["Product"][0].VendorUrl
            except Exception as ex:
                log.error("Could not set author or version attributes %s" % str(ex))
                raise Exception(ex)

            pkg_descriptor["app"]["resources"]["disk"] = int("10")
            if "VirtHW" in self.ovf_descriptor['VirtualSystem']:
                for i in range(len(self.ovf_descriptor['VirtualSystem']['VirtHW']['Item'])):
                    resType = self.ovf_descriptor['VirtualSystem']['VirtHW']['Item'][i]['Type']
                    currItem = self.ovf_descriptor['VirtualSystem']['VirtHW']['Item'][i]
                    if resType == OVF_RES_CPU:
                        self.setCpuAttributes(currItem, pkg_descriptor)
                    elif resType == OVF_RES_MEMORY:
                        self.setMemoryAttributes(currItem, pkg_descriptor)
                    elif resType == OVF_RES_NW:
                        self.setNetworkAttributes(pkg_descriptor)


            self.setDiskinPackageYaml(pkg_descriptor)
            pkg_descriptor["app"]["startup"]["qemu-guest-agent"] = qemu_present
            pkg_descriptor["app"]["startup"]["ostype"] = ostype
            return pkg_descriptor

    def getIOxPkgDescriptor(self, ova_path, package_name, qemu_present, ostype):
        ovfid = ""
        #Check for ova_path if or not it exists, since this is the first method called
        if not ova_path:
            log.error("Invalid OVA path or OVA doesn't exist: %s" % ova_path)
            raise MandatoryFileMissingError("Invalid OVA path or OVA doesn't exist: %s" % ova_path)

        if not os.path.exists(ova_path) or not os.path.isfile(ova_path):
            log.error("Invalid OVA path or OVA doesn't exist: %s" % ova_path)
            raise MandatoryFileMissingError("Invalid OVA path or OVA doesn't exist: %s" % ova_path)

        #Extract OVF content to a hashmap
        ovfcontent = self.extractOVFContent()
        log.debug("ovf content - %s" % ovfcontent)
        try:
            ovfid = self.parseOVF(ovfcontent)
        except Exception as ex:
            log.error("Failed to parse OVF file, exception: %s" % ex)
            raise Exception(ex)

        #Convert hashmap to yaml
        pkg_descriptor = self.generateYaml(package_name, qemu_present, ostype)

        if pkg_descriptor:
            Utils.write_yaml_file(APP_DESCRIPTOR_NAME, pkg_descriptor)
        else:
            log.error("Failed to generate package yaml, exception: %s" % ex)
            return None

        #move yaml to the temporary directory
        self.moveFiletoAppdir(os.path.join(APP_DESCRIPTOR_NAME), APP_DESCRIPTOR_NAME)
        #Save the path where your final package should go
        self.final_pkg_path = os.path.dirname(ova_path)
        return ovfid

    def runQemuCommand(self, package_name, ovfId):
        ret = subprocess.call(["which","qemu-img"])
        if ret:
            msg = "qemu-img command does not exist on this platforms"
            log.error("%s" % msg)
            raise Exception(msg)
            return -1

        out_qcow_path = os.path.join(self.ovaextractedDir, ovfId + QEMU_DISK_EXT)
        vmdk_file_list = []
        #Find all vmdk files in the ova and append them to vmdk_list
        log.debug("Looking for vmdk files in %s" % (self.ovaextractedDir))
        try:
            vmdk_file_list = subprocess.check_output(["find", self.ovaextractedDir, "-type", "f", "-name", "*\.vmdk"]).decode('utf8').split('\n')
            # Remove the last empty element that occurs because there's a newline characte at the end of output of find command.
            vmdk_file_list = vmdk_file_list[:-1]
            log.debug("vmdk files list - %s" % vmdk_file_list)
        except Exception as ex:
            log.error("Failed to find vmdk files within OVA or execute find command: %s" % ex)
            raise Exception(ex)
            return -1

        if len(vmdk_file_list) < 1:
            log.error("A vmdk file does not exist in the ova_path: %s" % ova_path)
            return -1

        if len(vmdk_file_list) > 1:
            msg = "Unsupported ova type found. More than one vmdk disks found in ova file. Multiple disks are not supported"
            log.error("%s" % msg)
            raise Exception(msg)
            return -1
        
        try:
            iso_file_list = subprocess.check_output(["find", self.ovaextractedDir, "-type", "f", "-name", "*\.iso"]).decode('utf8').split('\n')
            iso_file_list = iso_file_list[:-1]
            log.debug("ISO files list - %s" % iso_file_list)
            if len(iso_file_list):
                msg = "Unsupported ova type found. ISO disk file based ova is not supported"
                log.error("%s" % msg)
                raise Exception(msg)
                return -1
        except Exception as ex:
            log.error("Failed to execute find command: %s" % ex)
            raise Exception(ex)
            return -1
        
        out, retcode = qemu_img("convert", "-c", "-O", "qcow2", vmdk_file_list[0], out_qcow_path)
        if retcode:
            msg = "Error while converting vmdk to qcow2 file - %s" % out
            log.error("%s" % msg)
            raise Exception(msg)

        self.moveFiletoAppdir(os.path.abspath(out_qcow_path), ovfId+".qcow2")

        if os.path.isdir(self.ovaextractedDir):
            if subprocess.call(["rm", "-rf", self.ovaextractedDir]):
                msg = "Failed to clean up ova extracted directory"
                log.error("%s" % msg)
                raise Exception(msg)

        return None

    def writePackageMetaFile(self):
        """
            Forms the map for package metadata, converts to JSON and writes to file
        """
        metadata_map = defaultdict(str)
        packageInfo = defaultdict(str)
        packageInfo["builtOn"] = datetime.datetime.now().strftime("%Y-%m-%d %H:%M:%S")
        packageInfo["builtBy"] = "caf"
        try:
            iox_version = Utils.getIOxVersion()
        except Exception as ex:
            log.error("Error while getting iox version: %s" % ex)
            raise Exception(ex)
        packageInfo["buildHost"] = iox_version["platform_version_number"]
        packageInfo["buildDirectory"] = os.getcwd()
        packageInfo["cafVersion"] = iox_version["caf_version_number"]
        metadata_map["packageInfo"] = packageInfo
        try:
            with open(APP_METADATA_FILE_NAME, "w") as f:
                f.write(json.dumps(metadata_map))
        except Exception as ex:
            log.error("Failed to write contents to package metadata file: %s" % ex)
            raise Exception(ex)
        self.moveFiletoAppdir(os.path.join(APP_METADATA_FILE_NAME), APP_METADATA_FILE_NAME)
        return

    def writePackageManifest(self):
        """
            Write SHA256 sums of package metadata file, package yaml and qcow2 into package.mf
        """
        files_list = os.listdir(self.vmapp_dir)
        # doesn't this file list include tmp extracted dir as well ?
        log.debug("files list to generete sha256 - %s" % files_list)
        text = ""
        try:
            for file_name in files_list:
                sha256_sum = Utils.sha256_file(os.path.abspath(self.vmapp_dir) + "/" + file_name)
                text += "SHA256(" + str(file_name) +")= " + sha256_sum + "\n"
        except Exception as ex:
            log.error("SHA256 for files couldn't be calculated: %s" % ex)
            raise Exception(ex)
        try:
            with open(PACKAGE_MANIFEST_NAME, "w") as f:
                f.write(text)
        except Exception as ex:
            log.error("Failed to write contents to package manifest file: %s" % ex)
            raise Exception(ex)
        self.moveFiletoAppdir(os.path.join(PACKAGE_MANIFEST_NAME), PACKAGE_MANIFEST_NAME)
        return

    def packageContentstoApp(self, package_name):
        ret = subprocess.call(["which","tar"])
        if ret:
            log.error("tar command does not exist on this platform")
            return ret

        pkg_name = package_name + ".tar"
        final_pkg = os.path.join(self.final_pkg_path, pkg_name)
        try:
            cwd = os.getcwd()
            os.chdir(self.vmapp_dir)
        except Exception as ex:
            log.error("Could not change directory to tar contents %s" % self.vmapp_dir)
            raise Exception(ex)

        files = [f for f in os.listdir(".")]
        out, retcode = tar_create(final_pkg, files)
        if retcode:
            msg = "Could not create tar archive of the package contents - %s" % out
            log.error("%s" % msg)
            raise Exception(msg)

        try:
            os.chdir(cwd)
        except Exception as ex:
            log.error("Could not move back to original directory %s" % cwd)
            raise Exception(ex)
            return -1
        log.debug("Packaged contents to IOx app")
        return None

    def cleanUpContents(self, ovaCleanPath = ""):
        if os.path.isdir(self.vmapp_dir):
            if subprocess.call(["rm", "-rf", self.vmapp_dir]):
                msg = "Failed to cleanup ova staging directory"
                log.error("%s" % msg)
                raise Exception(msg)
        if ovaCleanPath:
            if subprocess.call(["rm", "-rf", ovaCleanPath]):
                msg = "Failed to cleanup ova artifacts"
                log.error("%s" % msg)
                raise Exception(msg)
        return None

    # For now, calling all three phases using this function
    def convertOVAtoIoxPackage(self, ova_path, package_name, qemu_present, auto_deploy, controller, uri_str, cleanup_ova, ostype):
        resp_body = {}
        #set the tmpupload directory and create tmp directory for this vm extraction
        self.setupTempDirs()
        #Check if there's twice the disk space, one to extract the ova on to tmp, one to copy tar to usb path
        log.info("size of ova is %s" % str(os.stat(ova_path).st_size))
        log.debug("Checking if there's enough disk space...")
        ova_size = os.stat(ova_path).st_size
        space_avail = 0

        try:
            space_avail = Utils.get_free_disk_space(self.tmpUploadDir)
            space_avail = space_avail*1024*1024
            log.debug("Available disk space: %s Required:%s" % (space_avail, ova_size))
            if space_avail < (2 * ova_size):
                log.error("Not enough space to extract the OVA and convert to qcow2, needed %d, got %d"%(2 * ova_size, space_avail))
                raise Exception("Not enough space to extract the OVA and convert to qcow2, needed %d, got %d"%(2 * ova_size, space_avail))

            dest_dir = os.path.dirname(ova_path)
            space_avail = Utils.get_free_disk_space(dest_dir)
            space_avail = space_avail*1024*1024
            log.debug("Available disk space: %s Required:%s" % (space_avail, ova_size))
            if space_avail < ova_size:
                log.error("Not enough space in %s to copy the IOx package, needed %d, got %d"%(dest_dir, ova_size, space_avail))
                raise Exception("Not enough space in %s to copy the IOx package, needed %d, got %d"%(dest_dir, ova_size, space_avail))

            self.extractOVA(ova_path)

            ovfid = self.getIOxPkgDescriptor(ova_path, package_name, qemu_present, ostype)
            log.info("First stage success: Generated package yaml successfully")
            bodystr = "Generated package metadata descriptor successfully"
            resp_body["status_code"] = falcon.HTTP_200
            resp_body["message"] = bodystr
            resp_body["headers"] = {}
            data = json.dumps(resp_body)
            yield Utils.prepareDataForChunkedEncoding(data)

            if self.runQemuCommand(package_name, ovfid):
                raise Exception("Failure in running Qemu command to convert from vmdk to qcow format")
            log.info("Second stage success: Converted vmdk to qcow")
            bodystr = "Converted vmdk to qcow successfully"
            resp_body["status_code"] = falcon.HTTP_200
            resp_body["message"] = bodystr
            resp_body["headers"] = {}
            data = json.dumps(resp_body)
            yield Utils.prepareDataForChunkedEncoding(data)

            self.writePackageMetaFile()
            self.writePackageManifest()
            log.info("Third stage success: Package metadata generated")
            bodystr = "Generated package manifest file successfully"
            resp_body["status_code"] = falcon.HTTP_200
            resp_body["message"] = bodystr
            resp_body["headers"] = {}
            data = json.dumps(resp_body)
            yield Utils.prepareDataForChunkedEncoding(data)

            if self.packageContentstoApp(package_name):
                raise Exception("Packaging of app returned errors. Either tar utility is not present or could not package contents")
            log.info("Fourth stage success: Packaged ova contents to IOx package")
            bodystr = "Generated IOx package successfully"
            resp_body["status_code"] = falcon.HTTP_200
            resp_body["message"] = bodystr
            resp_body["headers"] = {}
            data = json.dumps(resp_body)
            yield Utils.prepareDataForChunkedEncoding(data)

            #Success case: conversion was done. Add the converted vm to the conversion map
            vm_mgr = VMToolsManager.getInstance()
            try:
                ova_conv_config = {}
                iox_pkg_full_path = str(self.final_pkg_path) + "/" + package_name + ".tar"
                ova_conv_config["target_pkg"] = iox_pkg_full_path
                ova_conv_config["qemu_present"] = qemu_present
                ova_conv_config["auto_deploy"] = auto_deploy
                ova_conv_config["ostype"] = ostype

                vm_mgr.vmconversions.append({ova_path: ova_conv_config})
            except Exception as ex:
                log.error("Could not add this conversion to the list of vm conversions")
                raise Exception(ex)

            if auto_deploy:
                warn_messages = []
                if controller is None:
                    log.error("Controller object is None, so will be not be able to proceed with installing the app")
                    raise ServiceDepenendencyError("Controller object is None, so will be not be able to install the app")

                deploy_warn_messages = controller.install_app(package_name, iox_pkg_full_path, self.vmapp_dir, cleanup_ova)
                warn_messages.extend(deploy_warn_messages)
                connectorURL = uri_str+ "/" + package_name
                resp_body["status_code"] = falcon.HTTP_201
                resp_body["message"] = "Successfully deployed"
                if len(warn_messages) > 0 :
                    resp_body["message"] += "\n With warnings : \n %s" % warn_messages
                    log.debug("response body %s" % resp_body["body"])
                resp_body["headers"] = {"Content-Location":connectorURL}
                log.info("App: %s deployed successfully" % str(package_name))
                data = json.dumps(resp_body)
                yield Utils.prepareDataForChunkedEncoding(data)

        except Exception as ex:
            log.exception("Exception while converting the OVA to IOx compatible application %s", str(ex))
            resp_body["status_code"] = falcon.HTTP_500
            resp_body["headers"] = {}
            resp_body["message"] = str(ex)
            data = json.dumps(resp_body)
            #In case any exception happens, clean up everything since this is atomic operation
            yield Utils.prepareDataForChunkedEncoding(data)

        finally:
            log.debug("vm conversion finally block invoked")
            log.debug("cleanup_ova - %s, ova_path= %s" % (cleanup_ova, ova_path))
            if cleanup_ova:
                self.cleanUpContents(ova_path)
            else:
                self.cleanUpContents()

        if resp_body["status_code"] != falcon.HTTP_200 and resp_body["status_code"] != falcon.HTTP_201:
            log.error("Error occurred for the conversion of OVA, last response status code:%s, resp body:%s", resp_body.get('status_code'), resp_body.get('message'))
            return

class VMToolsManager(CAFAbstractService):
    __singleton = None # the one, true Singleton

    def __new__(cls, *args, **kwargs):
        # Check to see if a __singleton exists already for this class
        # Compare class types instead of just looking for None so
        # that subclasses will create their own __singleton objects
        if cls != type(cls.__singleton):
            cls.__singleton = super(VMToolsManager, cls).__new__(cls, *args, **kwargs)
        return cls.__singleton

    @classmethod
    def getInstance(cls, *args):
        '''
        Returns a singleton instance of the class
        '''
        if not cls.__singleton:
            cls.__singleton = VMToolsManager(*args)
        return cls.__singleton

    def __init__(self):
        import collections
        self.vmconversions = collections.deque(maxlen=10)
