'''
Created on Apr 02, 2015

@author: utandon

Copyright (c) 2015-2016 by Cisco Systems, Inc.
All rights reserved.
'''
import os
import shutil
import logging
from appfw.utils.utils import Utils
from appfw.utils.infraexceptions import *
log = logging.getLogger("rfs_composer")



class Mount_Composer(object):    
    __singleton = None # the one, true Singleton

    def __new__(cls, *args, **kwargs):
        # Check to see if a __singleton exists already for this class
        # Compare class types instead of just looking for None so
        # that subclasses will create their own __singleton objects
        if cls != type(cls.__singleton):
            cls.__singleton = super(Mount_Composer, cls).__new__(cls, *args, **kwargs)
        return cls.__singleton

    @classmethod
    def getInstance(cls, *args):
        '''
        Returns a singleton instance of the class
        '''
        if not cls.__singleton:
            cls.__singleton = Mount_Composer(*args)
        return cls.__singleton

    """
    Default PD utils class that provides common functions
    """
    def compose_rootfs(cls, target_rootfs_dir, 
               app_specs, data_specs, 
               cartridge_specs_list, dst_cartridge_dir=None, sec_info=None,target_rootfs_disk=None):
        """
        This implementation composes rootfs for non aufs system
        target_rootfs_dir : output rootfs dir 
        data_specs: dictionary of 
            src: src data dir to be mounted 
            dst: destination data directory inside container
            perm: ro/rw for read only or read write mount
        app_specs: dictionary of
            src: src app dir to be mounted 
            dst: destination app directory inside container
            perm: ro/rw for read only or read write mount
        cartridge_specs_list: list of cartridge objects and layer permissions
        dst_cartridge_dir: cartridge dir inside container

        Returns the dictionary of
        "data_dir" # Persistence data directory  
        "app_dir" # target app location inside container
        "cartridge_mount_dict" # Dictionary of cartridge id and their mount details
        "extra_mount_list" # Extra mount which may be needed apart from cartridges e.g app and data
        """
        log.debug("Composing rootfs using mounts")

        rootfs=None
        for c_spec in cartridge_specs_list:
            cartridge = c_spec[0]
            perm = c_spec[1]
            if cartridge.type == "baserootfs":
                rootfs = cartridge.get_location()
                break 
        if rootfs is None:
            log.error("No rootfs cound in cartridge list")
            raise ValueError("No rootfs cound in cartridge list")

        log.debug("Copying rootfs from %s to %s" % (rootfs, target_rootfs_dir))
        shutil.copytree(rootfs, target_rootfs_dir, symlinks=True)

        extra_mount_list = []
        cart_mount_dict = {}

        rootfs=None
        for c_spec in cartridge_specs_list:
            cartridge = c_spec[0]
            perm = c_spec[1]
            if cartridge.type == "baserootfs":
                continue
            if "mountable" in cartridge.handleas:
                src_cartridge_dir = cartridge.get_location()
                cartridge_mount = os.path.join(dst_cartridge_dir, cartridge.id)
                dcmount = os.path.join(target_rootfs_dir, cartridge_mount.lstrip("/"))
                if not os.path.isdir(dcmount):
                    os.makedirs(dcmount)
                cart_mount_dict[cartridge.id] = { 
                                "cartridge": cartridge,
                                "src_dir": src_cartridge_dir,
                                "dst_dir": cartridge_mount,
                                "perm": perm }
            else:
                log.error("Cartridge %s: is of type %s not supported ignoring" %
                                (cartridge.id, cartridge.type))

        #create dst mount point for data
        if not "src" in data_specs:
            log.error("Source data directory not specified")
            raise ValueError("Source data directory not specified")

        if not "dst" in data_specs:
            log.error("Destination data directory not specified")
            raise ValueError("Destination data directory not specified")

        perm="rw"
        if "perm" in data_specs:
            perm = data_specs["perm"]
            

        src_datadir = data_specs["src"]
        dst_datadir = data_specs["dst"]
        dcmount = os.path.join(target_rootfs_dir, dst_datadir.lstrip("/"))
        if not os.path.isdir(dcmount):
            os.makedirs(dcmount)
        extra_mount_list.append({ 
                            "src_dir": src_datadir,
                            "dst_dir": dst_datadir,
                            "perm" : perm})

        #create dst mount point for app
        if not "src" in app_specs:
            log.error("Source app directory not specified")
            raise ValueError("Source app directory not specified")
        src_appdir = app_specs["src"]

        dst_appdir = "/app"
        if "dst" in app_specs:
            dst_appdir = app_specs["dst"]

        perm="ro"
        if "perm" in app_specs:
            perm = app_specs["perm"]

        prov_appdir = os.path.join(target_rootfs_dir, dst_appdir.lstrip("/"))
        if not os.path.exists(prov_appdir):
            os.mkdir(prov_appdir)
        extra_mount_list.append({ 
                            "src_dir": src_appdir,
                            "dst_dir": dst_appdir,
                            "perm" : perm})
        return_list = {}
        return_list["data_dir"] = src_datadir # Persistence data directory  
        return_list["app_dir"] = "/app" # target app location inside container
        return_list["cartridge_mount_dict"] = cart_mount_dict # cartridge to be mounted 
        return_list["extra_mount_list"] = extra_mount_list #extra mounts needed 
        return return_list

    def remove_rootfs(self, target_rootfs):
        log.debug("Deleting dir %s" % target_rootfs)
        if (os.path.exists(target_rootfs)):
            shutil.rmtree(target_rootfs, ignore_errors=True)
