'''
Created on Nov 27, 2012

@author: samagraw

Copyright (c) 2012-2018 by Cisco Systems, Inc.
All rights reserved.
'''

import logging
import psutil
import subprocess
import platform
import os
from   ..utils.utils import Utils
import platform
from   appfw.hosting.apptypes import AppType

log = logging.getLogger("runtime")

class SystemInfo(object):

    cpuinfo_path = "/proc/cpuinfo"
    arp_path = "/proc/net/arp"
    dns_path = "/etc/resolv.conf"
    ntp_config = "/etc/ntp.conf"
    systemid_file = Utils.getPlatformHwIDFile()
    productid_file = Utils.getPlatformProductIDFile()
    system_name_file = Utils.getSystemNameFile()
    repo_dir = Utils.getSystemConfigValue("controller", "repo", "/software/caf")
    UUID_FILE = "iox_uuid"

    system_id = ""
    product_id = ""
    UUID = ""
    exclude_interfaces = Utils.getExcludeInterfacesDetails()

    # Get storage information
    @classmethod
    def get_storage_info(cls):
        log.debug("Get storage information for the host")
        storage_list = []
        for part in psutil.disk_partitions(all=False):
            usage = psutil.disk_usage(part.mountpoint)
            storage_list.append({"name" : part.device,
                                 "mount" : part.mountpoint,
                                 "filesystem" : part.fstype,
                                 "size" : usage.total,
                                 })
        return storage_list

    #Get Interface information      
    @classmethod
    def get_interfaces_info(cls):
        log.debug("Getting list of interfaces on the host.")
        ifconfig = subprocess.check_output("ifconfig", shell=False)
        ifaces = ifconfig.split('\n\n')
        # return all non-veth interfaces
        return [SystemInfo._extractIfaceInfo(iface) 
            for iface in ifaces if iface is not '' and iface not in cls.exclude_interfaces]


    # Returns dictionary of values for an interface
    @classmethod
    def _extractIfaceInfo(cls, iface):
        name = type = mtu = ipv4 = mask = ''
        for line in iface.split('\n'):
            if 'encap' in line:
                name = line.split()[0]
                type = line.split('Link encap:')[1].split()[0]
            if 'MTU' in line:
                mtu = line.split('MTU:')[1].split()[0]
            if 'inet addr' in line:
                ipv4 = line.split('inet addr:')[1].split()[0]
                mask = line.split('Mask:')[1]
        return {'name':name, 'type':type, 'ipv4_address':ipv4, 'ipv4_netmask':mask, 'mtu':int(mtu)}


    # Get route information
    @classmethod
    def get_route_info(cls):
        log.debug("Get route information for the host")
        route_list = []
        route = subprocess.check_output(['route', '-n'], shell=False)
        lines = [line.strip() for line in route.split("\n") \
                 if line.strip() != '']
        for line in lines:
            dest = line.split()[0]
            if dest == 'Kernel':
                continue
            if dest == 'Destination':
                continue
            gateway = line.split()[1]
            genmask = line.split()[2]
            flags = line.split()[3]
            metric = line.split()[4]
            iface = line.split()[7]
            route_list.append({"destination" : dest,
                               "gateway" : gateway,
                               "genmask" : genmask,
                               "flags" : flags,
                               "metric" : int(metric),
                               "iface" : iface})
        return route_list

    #Get ARP ifnormation
    @classmethod
    def get_arp_info(cls):
        log.debug("Get arp information for the host")
        arp_list = []
        arp_addr = ''
        arp_hw_addr = ''
        arp_inf = ''
        
        try:
            with open(SystemInfo.arp_path) as file_hdlr:
                lines = file_hdlr.readlines()
            for line in lines:
                arp_addr = line.split()[0]
                arp_hw_addr = line.split()[3]
                arp_inf = line.split()[5]
                if arp_addr == 'IP':
                    continue
                arp_list.append({"address" : arp_addr,
                                "hwaddress" : arp_hw_addr,
                                "iface" : arp_inf
                                })
            return arp_list
        except IOError:
            return ''

    #Get DNS information
    @classmethod
    def get_dns_info(cls):
        log.debug("Get dns information for the host")
        nameserver_list = []
        dns_search = ''
        dns_domain = ''

        try:
            with open(SystemInfo.dns_path) as file_hdlr:
                lines = file_hdlr.readlines()
            for line in lines:
                if 'search' in line:
                    dns_search = line.split()[1]
                if 'domain' in line:
                    dns_domain = line.split()[1]
                if 'nameserver' in line:
                    nameserver_list.append(line.split()[1])

        except IOError:
            log.debug("Error reading DNS file");

        dns_info = {"search" : dns_search,
                    "domain" : dns_domain,
                    "nameservers" : nameserver_list
                    }
        return dns_info

    #Get CPU information
    @classmethod
    def get_cpu_info(cls):
        log.debug("Get cpu information for the host")
        
        try:
            cpu_family = 0
            cpu_model = 0
            cpu_model_name = ""
            cpu_stepping = 0
            cpu_frequency = 0
            cpu_core = Utils.getSystemConfigValue("platform", "cpu_core", "")
            with open(SystemInfo.cpuinfo_path) as file_hdlr:
                lines = file_hdlr.readlines()
            for line in lines:
                if 'family' in line:
                    cpu_family = line.split(':')[1].strip()
                if 'model' in line and 'name' not in line:
                    cpu_model = line.split(':')[1].strip()
                if 'model name' in line:
                    cpu_model_name = line.split(':')[1].strip()
                if 'stepping' in line: 
                    cpu_stepping = line.split(':')[1].strip()
                if 'cpu MHz' in line:
                    cpu_frequency = line.split(':')[1].strip()
                if cpu_core == "":
                    if 'cpu' == line.split(':')[0].strip():
                        cpu_core = line.split(':')[1].strip()
            if "armv6l" in platform.platform():
                cpu_cores = 1
            else:
                try:
                    import multiprocessing
                    cpu_cores = multiprocessing.cpu_count()
                    #cpu_cores = subprocess.check_output("nproc", shell=False)
                except Exception as ex:
                    log.error("Error getting number of cores: %s. Assuming it to be 1" % str(ex))
                    cpu_cores = 1
        
            cpu_arch = subprocess.check_output(["uname", "-m"], shell=False)
            cpu_arch = cpu_arch.strip()
            log.debug("Cpu architecture %s" % cpu_arch) 
            
            cpu_info = {"family" : int(cpu_family),
                        "model" : cpu_model,
                        "model_name" : cpu_model_name,
                        "stepping" : int(cpu_stepping),
                        "frequency" : float(cpu_frequency),
                        "number_cores" : int(cpu_cores),
                        "cpu_arch" : cpu_arch 
                       } 
            if cpu_core:
                cpu_info["cpu_core"] = cpu_core
            return cpu_info
        except IOError:
            return ''

    #Get memory information
    @classmethod
    def get_memory_info(cls):
        log.debug("Get memory information for the host") 
        if "armv6l" in platform.platform():
            total_mem = psutil.total_virtmem()
            swap_total = 0
        else:
            mem = psutil.virtual_memory()
            total_mem = mem.total
            swap = psutil.swap_memory()
            swap_total = swap.total

    
        mem_info = {"size" : int(total_mem), 
                    "swap" : int(swap_total)
                   }
        return mem_info

    #Get c3 version
    @classmethod
    def get_version(cls):
        return Utils.getIOxVersion()

    #Get hostname
    @classmethod
    def get_hostname(cls):
        log.debug("Getting hostname")
        hostname = None
        # we're not caching the value since it can be changed out-of-band

        # First, let us look at the environment variable HOSTNAME, if present, pick it up from there.
        hostname = os.getenv("HOSTNAME", None)
        if hostname != None:
            log.debug("Obtained hostname from environment variable HOSTNAME : %s", hostname)
            return hostname

        # If the platform has a system_name_file, read from there.
        if os.path.isfile(SystemInfo.system_name_file):
            try:
                with open(SystemInfo.system_name_file) as file_hdlr:
                    hostname = file_hdlr.read()
                    log.debug("Obtained hostname from file %s, hostname : %s", SystemInfo.system_name_file, hostname)
            except:
                log.error("Error reading system name from %s" % SystemInfo.system_name_file)

        # If we couldn't read from the system_name_file, get it from the platform
        if hostname is None:
            log.debug("Getting hostname from platform.uname")
            hostname = platform.uname()[1]

        return hostname

    #Get c3 ntp server setting
    @classmethod
    def get_ntp_server(cls):
        log.debug("Get ntp server setting for the host")
        try:
            with open(SystemInfo.ntp_config) as file_hdlr:
                lines = file_hdlr.readlines()
            server = []
            for line in lines:
                if line.startswith("server"):
                    server.append(line.split(' ')[1].strip())
            return server
        except IOError:
            return ''

    @classmethod
    def get_cpuserial_arm(cls):
        # Extract serial from cpuinfo file
        cpuserial = "0000000000000000"
        try:
            f = open('/proc/cpuinfo','r')
            for line in f:
                if line[0:6]=='Serial':
                    cpuserial = line[10:26]
            f.close()
        except:
            cpuserial = "ERROR000000000"

        return cpuserial

    #Return the UDI format <Product_id>:<Serial_id>
    @classmethod
    def get_system_udi(cls):
        product_id = Utils.getPlatformProductID()
        serial_id = SystemInfo.get_systemid()
        udi = product_id + ":" + serial_id
        return udi

    # Get product ID
    @classmethod
    def get_productid(cls):
        product_id = Utils.getPlatformProductID()
        log.debug("Get product ID for the host %s", product_id)
        return product_id

    @classmethod
    def get_system_uuid(cls, appid=None):
        log.debug("Get system UUID")
        if appid:
            uuid_file = os.path.join(SystemInfo.repo_dir, appid, SystemInfo.UUID_FILE)
        else:
            uuid_file = os.path.join(SystemInfo.repo_dir, SystemInfo.UUID_FILE)
        if os.path.isfile(uuid_file):
            uuid = file(uuid_file, "r").readline()
            SystemInfo.UUID = uuid.lstrip().rstrip()
            log.debug("Obtained UUID : %s" % SystemInfo.UUID)
        else:
            import uuid
            iox_uuid = str(uuid.uuid4()).lstrip().rstrip()
            try:
                file(uuid_file, "w").write(iox_uuid)
            except Exception as e:
                log.error("Exception while writing in to the uuid file %s", str(e))
            SystemInfo.UUID = iox_uuid
            log.debug("Generated UUID as a uuid4 : %s" % SystemInfo.UUID)
        return SystemInfo.UUID

    # Get system ID
    @classmethod
    def get_systemid(cls):
        log.debug("Get system ID for the host")

        # If already stored in the attribute, return
        if SystemInfo.system_id:
            return SystemInfo.system_id


        if os.path.isfile(SystemInfo.systemid_file):
            try:
                with open(SystemInfo.systemid_file) as file_hdlr:
                    SystemInfo.system_id = file_hdlr.read().strip()
            except:
                log.error("Error reading system ID from %s" % SystemInfo.systemid_file)

        # return if successful
        if SystemInfo.system_id:
            return SystemInfo.system_id


        # Else, continue with alternate logic
        # If the platform is an ARM based host, use /proc/cpuinfo to get serial for systemid
        if "armv6l" in platform.platform():
            SystemInfo.system_id = cls.get_cpuserial_arm()
        

        # Platform may have /usr/sbin/dmidecode. Extract systemid from there
        elif os.path.isfile("/usr/sbin/dmidecode"):
            outpt = subprocess.check_output(['/usr/sbin/dmidecode',  '-t',  '1'], shell=False)
            lines = [line.strip() for line in outpt.split("\n") if line.strip() != '']
            for line in lines:
                if line.split()[0] == 'UUID:':
                    SystemInfo.system_id = line.split()[1]
                    log.debug("Obtained System ID from DMI: %s" % SystemInfo.system_id)
                    break
        else:
            # There is a no usr/sbin/dmidecode
            # If this is the first time, generate uuid.uuid4() and store it into /etc/iox_sys_id
            # If the file is already available, read it from there.
            SystemInfo.system_id = SystemInfo.get_system_uuid()

        # return ID or empty line if was not able to get the ID
        return SystemInfo.system_id
        
    @classmethod
    def get_vpg_interfaces(cls):
        log.debug("Get vpg interfaces that can be enslaved.")
        nw_attrs = SystemInfo.get_network_manager_attrs()
        if nw_attrs.get("network_creation_api"):
            return Utils.getlines_from_file(nw_attrs["vpg_inf_file"])
        else:
            return []
            
    @classmethod
    def get_network_manager_attrs(cls):
        log.debug("Get network manager attributes.")
        
        from apiservice import APIService
        network_manager = APIService.instance.network_manager
        try:
            return {"suggested_nat_range": network_manager.suggested_nat_range,
                    "mirror_mode_supported": network_manager.mirror_mode_supported,
                    "network_creation_api": network_manager.network_creation_api,
                    "vpg_inf_file": network_manager.vpg_inf_file}
        except Exception as ex:
            log.error("Could not get network manager attributes: %s", str(ex))
            return {}

    #Get system information
    @classmethod
    def get_system_info(cls, brief=False):
        log.debug("Get system information for the host")
        sys_info = {
                    "version" : SystemInfo.get_version(),
                    "hostname" : SystemInfo.get_hostname(),
                    "system_id" : SystemInfo.get_systemid(),
                    "product_id" : SystemInfo.get_productid(),
                    "udi" : SystemInfo.get_system_udi(),
                    "uuid" : SystemInfo.get_system_uuid(),
                    "server_info": SystemInfo.get_server_info(),
                    "libvirt_version": Utils.get_libvirt_version(),
                    "docker_version": Utils.get_docker_version()
                    }
        if not brief:
            sys_info.update({"cpu" : SystemInfo.get_cpu_info(),
                        "memory" : SystemInfo.get_memory_info(),
                        "storage" : SystemInfo.get_storage_info(),
                        "interfaces" : SystemInfo.get_interfaces_info(),
                        "ipv4_routing" : SystemInfo.get_route_info(),
                        "arp_cache" : SystemInfo.get_arp_info(),
                        "dns_resolver" : SystemInfo.get_dns_info(),
                        "ntp_server" : SystemInfo.get_ntp_server()
                        })
        if not brief:
            vpg_interfaces = SystemInfo.get_vpg_interfaces()
            if vpg_interfaces != []:
                sys_info["vpg_interfaces"] = vpg_interfaces

                nw_attrs = SystemInfo.get_network_manager_attrs()
                sys_info["suggested_nat_range"] = nw_attrs.get("suggested_nat_range")
                sys_info["mirror_mode_supported"] = nw_attrs.get("mirror_mode_supported")

        from appfw.runtime.hostingmgmt import HostingManager
        hm = HostingManager.get_instance()
        app_manager = hm.get_service("app-management")
        if app_manager:
            sys_info["corrupted_appids"] = app_manager.corrupted_appids

        ssd_wear_ratio, ssd_fail_msg = Utils.get_ssd_wear_ratio() 
        if ssd_wear_ratio != None:
            sys_info["ssd_status"] = ssd_wear_ratio
            sys_info["ssd_status_error_reason"] = ssd_fail_msg

        log.debug("System Information for the host: %s", sys_info)
        return sys_info

    @classmethod                                                                                   
    def get_secure_storage_port(cls):
        port = Utils.getSystemConfigValue("secure_storage_server", "port")      
        log.debug("secure storage port = %s", port);                            
        return port                                                             
                                                                                
    @classmethod                                                                
    def get_secure_storage_ip_addr(cls, app_type=AppType.DOCKER):                                        
        # Get default bridge
        from appfw.runtime.hostingmgmt import HostingManager
        hm = HostingManager.get_instance()
        network_manager = hm.get_service("network-management")
        if not network_manager:
            return 
        ss_net = network_manager.get_secure_storage_network(app_type)
        log.debug("Got Secure Storage network - ss_net: %s", ss_net)
        if ss_net:
            ipv4host = ss_net.gateway_ip
            ipv6host = None
        else:
            default_bridge = network_manager.get_default_bridge()
            host_bridge = network_manager.get_hosting_bridge(default_bridge)
            #Read IPV4 address. 
            ipv4host = host_bridge.get_bridge_ipv4_address()
            #Read IPV6 address 
            ipv6host = host_bridge.get_bridge_ipv6_address()

        log.debug("Secure Storage ipv4 addr = %s, ipv6 = %s", ipv4host, ipv6host)
        return (ipv4host, ipv6host)

    @classmethod
    def is_secure_storage_supported(cls):
        '''
        Check the 'enabled' variable in secure storage section of system_config.ini
        '''
        Secure_Storage_section = Utils.getSystemConfigSection("secure_storage_server")
        if Secure_Storage_section is not None and Secure_Storage_section.get("enabled", "no") == "yes":
            return True
        else:
            return False
    @classmethod
    def is_secure_storage_server_up_and_available(cls):
        '''
        Check the sentinel which indicates secure storage server is UP and available
        '''
        Secure_Storage_section = Utils.getSystemConfigSection("secure_storage_server")
        if Secure_Storage_section is not None:
            ss_available_sentinel_file = Secure_Storage_section.get("available_sentinel", "")
            ss_alive_sentinel_file = Secure_Storage_section.get("alive_sentinel", "")
            return (os.path.isfile(ss_available_sentinel_file) and os.path.isfile(ss_alive_sentinel_file))
        else:
            return False

    @classmethod
    def is_caf_recovery_mode_triggered(cls):
        '''
        Check the sentinel file which indicates CAF to start up in recovery mode
        '''
        recovery_mode_section = Utils.getSystemConfigSection("recovery_mode")
        if recovery_mode_section is not None:
            caf_recovery_sentinel_file = recovery_mode_section.get("sentinel_path", "")
            return os.path.isfile(caf_recovery_sentinel_file)
        else:
            return False

    @classmethod
    def caf_recovery_mode_reason(cls):
        '''
        Return the reason for starting CAF in recovery mode.
        '''
        recovery_mode_section = Utils.getSystemConfigSection("recovery_mode")
        if recovery_mode_section is not None:
            caf_recovery_sentinel_file = recovery_mode_section.get("sentinel_path", "")
            try:
                with open (caf_recovery_sentinel_file) as f:
                    return f.read()
            except Exception as ex:
                log.exception("Failed to read caf recovery sentinel file = %s", str(ex))
                return ""
        else:
            return ""

    @classmethod
    def is_app_privileged_mode_supported(cls):
        return Utils.getSystemConfigValue("app-settings", "privileged_mode_supported", False, "bool")

    @classmethod
    def get_server_info(cls):
        if not Utils.getSystemConfigValue("api", "fcgi_socket_bind_address", False):
            address = Utils.getSystemConfigValue("api", "secondary_server_address", "::") if Utils.getSystemConfigValue("api", "enable_secondary_server", False, "bool") else Utils.getSystemConfigValue("api", "address", "::")
            port = Utils.getSystemConfigValue("api", "secondary_server_port", "::") if Utils.getSystemConfigValue("api", "enable_secondary_server", False, "bool") else Utils.getSystemConfigValue("api", "port", "::")
            protocol = "HTTPS" if Utils.getSystemConfigValue("api", "use_ssl", False, "bool") else "HTTP"
            return {
                "address": address,
                "port": port,
                "protocol": protocol
            }
        else:
            return {}
