#-----------------------------------------------------
#
# Copyright (c) 2012-2013 by cisco Systems, Inc.
# All rights reserved.
#-----------------------------------------------------

'''
@author: rnethi
'''

import logging
import sys
import os
import io
import json
import time
import copy
import datetime
from collections import deque
import collections
import threading
# Use the implementation in appfw.overrides


from   ..hosting.apptypes import AppType
from   ..hosting.state import State, Inputs
from   threading import Lock, RLock
from   ..utils.utils import Utils, ConcurrentAccessException, APP_METADATA_FILE_NAME, USER_EXTRACTED_DIR, PACKAGE_MANIFEST_NAME, APP_RESOURCES_FILE, APP_PACKAGE_CONFIG_SIZE_LIMIT
from   ..utils.utils import CARTRIDGE_LANG_PREFIX, DEFAULT_APP_PERSISTENT_DATA_DISK_SIZE_MB
from   ..utils.infraexceptions import *
from resourcemanager import ResourceManager
from ..utils.commandwrappers import *

log = logging.getLogger("runtime.hosting")

USER_CONNECTOR_ARCHIVE = "connector_archive"
STATIC_DNS_MAX_RETRIES = 5
class ConnectorWrapper(object):

    _connectorWrapper_rlock = RLock()

    _to_serialize = ("id", "name", "description", "version", "author", "startup", "image_name", "image_tag", 
                     "authorLink", "packageMetaData", "state", "appgroup", "toolkitServicesUsed", "appType", 
                     "debugMode", "is_service", "appCustomOptions", "host_mode", "networkInfo", 
                     "dependsOn", "provides", "resources", "env", "is_autoinstalled", "auto_start", "auto_remove", "auto_deactivate",
                     "ttycon", "ttyaux", "ttytra", "ttylog",
                     "container_type", "packageManifest", "last_state_change_time", "app_checksums",
                     "app_health", "ioxv_metadata", "post_upgrade_script_output",
                     "reconcile_failure", "reconcile_attempted", "dynamic_app_resources", 
                     "package_config", "platform_service","system_capabilities", "app_payload_size",
                     "svc_access_security","svc_security_schemas", "used_by", "verified_signature_model")
    _detailed = ("app_metrics",)
  
    def __init__(self, connector, container, is_autoinstalled=False, verified_signature_model=None):
        self._container = container
        self._connector = connector
        self._archiveFile = None
        self._state = State.DEPLOYED if self._connector.metadata else State.CREATED
        self._data_disk = None
        self._is_autoinstalled = is_autoinstalled
        self._verified_signature_model = verified_signature_model
        self._debug_mode = False
        self._auto_start = False
        self._last_started_time = 0
        self._last_state_change_time = 0
        self._reconcile_failure = False
        self._reconcile_attempted = False
        self._restart_attempts = 0
        self.is_network_assigned = False
        self._health_status = 0
        self._post_upgrade_status = 0
        self._health_poll_cnt = 1
        self._last_health_probe_output = ""
        self._last_health_probe_err = ""
        self._health_probe_fail_cnt = 0
        self._post_upgrade_output = ""
        self._post_upgrade_err = ""
        self._last_monitor_time=int(time.time())
        self._rootfs_read_bytes = 0
        self._rootfs_write_bytes = 0
        self._data_read_bytes = 0
        self._data_write_bytes = 0
        self._read_threshold_exceeded = False
        self._write_threshold_exceeded = False
        self._old_nw_info = {}
        self.desired_state = ""
        self._platform_service = False
        self._ioxv_max_num_val = self.getVisualizationConfig("persist_num_app_data", 64)
        self._ioxv_len_limits = {}
        self._ioxv_len_limits["value"] = self.getVisualizationConfig("data_len", 1024)
        self._ioxv_max_num_datapoints = self.getVisualizationConfig("max_num_datapoints", 8)
        self._ioxv_len_limits["label"] = self.getVisualizationConfig("label_len", 32)
        self._ioxv_datapoints = None
        self._ioxv_metadata = collections.OrderedDict()
        self._lock = threading.Lock()
        self._system_capabilities = None
        self._available_scopes = []
        self._requested_scopes = []
        self._app_payload_size = 0
        self._static_dns_entries = []
        self._static_dns_tries_on_failure = STATIC_DNS_MAX_RETRIES
        self._static_dns_entries_added = False
        self._autoupgrade_state = None
        self._dependent_volumes = []
        self._dependent_host_mounts = []
        repo_folder = Utils.getSystemConfigValue("controller", "repo", "/etc")
        work_folder = os.path.dirname(repo_folder)
        self.storage_repo = os.path.join(work_folder, "storage")
        self.storage_deps_file = ".storagedeps"
        self.storage_deps_fullpath = os.path.join(self.storage_repo, self.storage_deps_file)
        self.storage_deps = {}
        self._used_by = []
        self._auto_remove = False
        self._auto_deactivate = False
        self._ttycon = ""
        self._ttyaux = ""
        self._ttytra = ""
        self._ttylog = ""

    def save_storage_deps(self, vols, host_mounts):
        self._load_storage_deps()

        if not os.path.isdir(self.storage_repo):
            os.makedirs(self.storage_repo)

        if not self.storage_deps.get(self._connector.id):
            self.storage_deps[self._connector.id] = {}
            self.storage_deps[self._connector.id]["host_mounts"] = []
            self.storage_deps[self._connector.id]["volumes"] = []

        updated_host_mounts = list(set().union(host_mounts, self.storage_deps[self._connector.id]["host_mounts"]))
        log.debug("updated_host_mounts = %s" % (updated_host_mounts))
        self.storage_deps[self._connector.id]["host_mounts"] = updated_host_mounts
        log.debug("new storage deps - host_mounts = %s" % self.storage_deps[self._connector.id]["host_mounts"])

        updated_volumes = list(set().union(vols, self.storage_deps[self._connector.id]["volumes"]))
        log.debug("updated volumes = %s" % (updated_volumes))
        self.storage_deps[self._connector.id]["volumes"] = updated_volumes
        log.debug("new storage deps - volumes = %s" % self.storage_deps[self._connector.id]["volumes"])
        file(self.storage_deps_fullpath, "w").write(json.dumps(self.storage_deps))
        log.debug("updated storage config file - %s" % json.dumps(self.storage_deps))

    def _load_storage_deps(self):
        try:
            if os.path.isfile(self.storage_deps_fullpath):
                self.storage_deps = json.load(file(self.storage_deps_fullpath, "r"))
                log.debug("Loaded apps storage dependencies from %s", self.storage_deps_fullpath)
        except Exception as ex:
            msg = "Error loading storage dependencies from %s:" % (self.storage_deps_fullpath)
            if isinstance(ex, ValueError):
                log.error("%s due to invalid JSON format" % msg)
            else:
                log.exception("%s - %s" % (msg, str(ex)))
            self.storage_deps = {}

    def get_storage_deps(self):
        self.load_storage_deps()
        return self.storage_deps.get(self._connector.id, {})

    def get_all_apps_storage_deps(self):
        self._load_storage_deps()
        return self.storage_deps

    def remove_app_storage_deps(self):
        self._load_storage_deps()
        self.storage_deps.pop(self._connector.id, None)
        file(self.storage_deps_fullpath, "w").write(json.dumps(self.storage_deps))

    def getVisualizationConfig (self, key, default):
        vis_section = Utils.getSystemConfigSection("visualization")
        if vis_section is None:
            return default
        else:
            return Utils.getSystemConfigValue('visualization', key, default, 'int')

    def get_ioxv_datapoints(self, req_from, no_of_values):
        if self._ioxv_datapoints is None:
            return None
        self._lock.acquire()
        timestamp = time.time()
        ret_dp = dict()
        try:
            for datapoint in self._ioxv_datapoints:
                ret_dp[datapoint] = {}
                #ret_dp[datapoint]["lastUpdated"] = self._ioxv_datapoints[datapoint]["lastUpdated"]
                ret_dp[datapoint]["lastUpdated"] = timestamp
                if no_of_values != -1:
                    data_list = deque(maxlen=no_of_values)
                else:
                    data_list = []
                for item in self._ioxv_datapoints[datapoint]["data"]:
                    if item["timestamp"] >= req_from:
                        log.debug("datapoint %s value %s" %(datapoint, item["value"]))
                        data_list.append(item)

                #if data_list is an instance of deque
                #then it needs to be converted to list for deserialization
                #purposes
                ret_dp[datapoint]["data"] = list(data_list)
        finally:
            self._lock.release()
        return ret_dp

    def append_ioxv_datapoints(self, val):
        '''
        structure of val
        {
            datapoint : [
                {
                    "value" :
                    "label" :
                    "timestamp" :
                },

            ],
        }
        '''

        if val is None or self._ioxv_datapoints is None:
            return
        self._lock.acquire()
        import time
        timestamp = time.time()
        try:
            for datapoint in val:
                if self._ioxv_datapoints.has_key(datapoint):
                    for item in val[datapoint]:
                        self._ioxv_datapoints[datapoint]["lastUpdated"] = timestamp
                        for key in item.keys():
                            #ignoring all the other entries in data for now
                            if key == "value" or key == "label":
                                if isinstance(item[key], basestring) and len(item[key]) > self._ioxv_len_limits[key]:
                                        log.debug("trimming the datapoint %s value to len %s"%(datapoint, self._ioxv_len_limits[key]))
                                        item[key] = item[key][:self._ioxv_len_limits[key]]
                            else:
                                del(item[key])

                        #setting the timestamp to the time when data was received
                        item["timestamp"] = timestamp
                        log.debug("datapoint %s value %s" %(datapoint, item["value"]))

                        self._ioxv_datapoints[datapoint]["data"].append(item)
                else:
                    raise InvalidDataPointError('Datapoint %s not found in metadata' %datapoint)

        finally:
            self._lock.release()

    @property
    def autoupgrade_state(self):
        return self._autoupgrade_state

    @autoupgrade_state.setter
    def autoupgrade_state(self, val):
        self._autoupgrade_state = val

    @property
    def ioxv_metadata(self):
        return self._ioxv_metadata

    @ioxv_metadata.setter
    def ioxv_metadata(self, val):
        if 'datapoints' in val:
            if len(val['datapoints']) > self._ioxv_max_num_datapoints:
                raise IOXVDataPointLimitError("Number of datapoints %s is more than the IOX limit %s" %(len(val['datapoints'], self._ioxv_max_num_datapoints)))

        for key, value in sorted(val.items()):
            self._ioxv_metadata[key] = value

        self._ioxv_metadata["persist_num_datapoint_values"] = self._ioxv_max_num_val
        log.debug("metdata %s" %self._ioxv_metadata)
        if 'datapoints' in self._ioxv_metadata:
            self._lock.acquire()
            try:
                if self._ioxv_datapoints is None:
                    self._ioxv_datapoints = dict()
                for datapoint in self._ioxv_metadata['datapoints']:
                    if datapoint not in self._ioxv_datapoints:
                        self._ioxv_datapoints[datapoint] = {}
                        self._ioxv_datapoints[datapoint]["lastUpdated"] = 0
                        self._ioxv_datapoints[datapoint]["data"] = deque(maxlen=self._ioxv_max_num_val)
            finally:
                self._lock.release()

    @property
    def app_repo_path(self):
        return self._connector.getPath()

    @property
    def last_state_change_time(self):
        return self._last_state_change_time

    @property
    def last_started_time(self):
        return self._last_started_time

    @last_started_time.setter
    def last_started_time(self, val):
        self._last_started_time = val

    @property
    def restart_attempts(self):
        return self._restart_attempts

    @restart_attempts.setter
    def restart_attempts(self, val):
        self._restart_attempts = val

    @property
    def reconcile_failure(self):
        return self._reconcile_failure

    def set_reconcile_failure(self, val):
        self._reconcile_failure = val

    @property
    def reconcile_attempted(self):
        return self._connector.reconcile_attempted

    @property
    def container_type(self):
        if self._container:
            return str(self._container)
        return "ABSTRACT"

    @property
    def auto_start(self):
        return self._auto_start

    @property
    def auto_remove(self):
        return self._auto_remove

    @auto_remove.setter
    def auto_remove(self, val):
        self._auto_remove = val

    @property
    def ttycon(self):
        return self._ttycon

    @ttycon.setter
    def ttycon(self, val):
        self._ttycon = val

    @property
    def ttyaux(self):
        return self._ttyaux

    @ttyaux.setter
    def ttyaux(self, val):
        self._ttyaux = val

    @property
    def ttytra(self):
        return self._ttytra

    @ttytra.setter
    def ttytra(self, val):
        self._ttytra = val

    @property
    def ttylog(self):
        return self._ttylog

    @ttylog.setter
    def ttylog(self, val):
        self._ttylog = val

    @auto_start.setter
    def auto_start(self, v):
        self._auto_start = v

    @property
    def auto_deactivate(self):
        return self._auto_deactivate

    @auto_deactivate.setter
    def auto_deactivate(self, val):
        self._auto_deactivate = val

    @property
    def id(self):
        return self._connector.id

    @property
    def used_by(self):
        return self._used_by

    def add_used_by(self, appid):
        self._used_by.append(appid)


    def remove_used_by(self, appid):
        if appid in self._used_by:
            log.debug("Removing %s from used_by list of dependent apps" % appid)
            self._used_by.remove(appid)
        else:
            log.debug("Appid %s is not in used_by apps dependent list" % appid)

    @property
    def manifestfile(self):
        return self._connector.metadata.manifestfile if self._connector.metadata else ""

    @property
    def name(self):
        return self._connector.metadata.name if self._connector.metadata else self.id

    @property
    def description(self):
        return self._connector.metadata.description if self._connector.metadata else ""

    @property
    def version(self):
        return self._connector.metadata.version if self._connector.metadata else ""

    @property
    def author(self):
        return self._connector.metadata.author if self._connector.metadata else ""

    @property
    def startup(self):
        return self._connector.metadata.startup if self._connector.metadata.startup else ""

    @property
    def monitor(self):
        return self._connector.metadata.monitor if self._connector.metadata else ""

    @property
    def image_name(self):
        return self._connector.image_name if self._connector.image_name else ""

    @property
    def image_tag(self):
        return self._connector.image_tag if self._connector.image_tag else ""

    @property
    def post_upgrade(self):
        return self._connector.metadata.post_upgrade if self._connector.metadata else ""

    @property
    def packageMetaData(self):
        data = {}
        metadata_file = os.path.join(self._connector.getPath(), USER_EXTRACTED_DIR, APP_METADATA_FILE_NAME)
        if os.path.isfile(metadata_file):
            with open(metadata_file) as f:
                try:
                    data = json.load(f)
                except ValueError as e:
                    log.info("Given metadata file '%s' is having invalid json data"%APP_METADATA_FILE_NAME)
        return data

    @property
    def authorLink(self):
        return self._connector.metadata.authorLink if self._connector.metadata else ""

    @property
    def toolkitServicesUsed(self):
        return self._connector.metadata.toolkitServicesUsed if self._connector.metadata else ""
        
    @property
    def upgradePreserveFilesList(self):
        return self._connector.metadata.upgradePreserveFilesList if self._connector.metadata else ""
    
    @property
    def appCustomOptions(self):
        return self._connector.runtimeConfig.appCustomOptions

    @property
    def appgroup(self):
        return self._connector.runtimeConfig.appgroup

    @property
    def provides(self):
        return self._connector.metadata.provides if self._connector.metadata else []


    #################################
    @property
    def appType(self):
        return self._connector.metadata.apptype if self._connector.metadata else ""

    @property
    def debugMode(self):
        return self._debug_mode

    @debugMode.setter
    def debugMode(self, value):
        self._debug_mode = value

    @property
    def runtime(self):
        return self._connector.metadata.runtime if self._connector.metadata else ""
       
    @property
    def runtime_version(self):
        return self._connector.metadata.runtime_version if self._connector.metadata else ""
       
    @property
    def is_service(self):
        return self._connector.is_service

    @property
    def platform_service(self):
        from hostingmgmt import HostingManager
        app_manager = HostingManager.get_instance().get_service("app-management")
        if app_manager:
            if app_manager.platform_services_enabled:
                return app_manager.platform_services.is_platform_service(self.app_id)
        return self._platform_service

    @property
    def host_mode(self):
        if not self._connector.metadata:
            return False
        app_resources = self._connector.metadata.resources
        return app_resources.get("use_host_mode", False)

    @property
    def dependsOn(self):
        dep_on = {}
        if self._connector.metadata is None:
            return dep_on
        if self._connector.metadata.dependsOn:
            dep_on = self._connector.metadata.dependsOn
        
        if self.runtime:
            cart_runtime = CARTRIDGE_LANG_PREFIX + self.runtime
            if "cartridges" in dep_on:
                found=False
                for cart in dep_on["cartridges"]:
                    if cart_runtime == cart["id"] and self.runtime_version == cart["version"] :
                        log.debug("Runtime id %s and version %s already exists in cartridge dependency" % (cart_runtime, self.runtime_version))
                        found=True
                        break

                if not found:    
                    dep_on["cartridges"].append({"id": cart_runtime, "version": str(self.runtime_version)})
            else:
                dep_on["cartridges"] = [{"id": cart_runtime, "version": str(self.runtime_version)}]
        return dep_on 

    @property
    def packageManifest(self):
        manifest_file = os.path.join(self._connector.getPath(), USER_EXTRACTED_DIR, PACKAGE_MANIFEST_NAME)
        manifest_contents = []
        if os.path.exists(manifest_file):
            try:
                import io
                with io.open(manifest_file, "r", encoding='utf8') as f:
                    manifest_contents = f.readlines()
            except Exception as ex:
                log.exception("Failed to read manifest file %s : Error: %s" % (manifest_file, str(ex)))
        return manifest_contents

    @property
    def dynamic_app_resources(self):
        """
        This will return the contents of a app_resources.json file,
            where CAF will be storing activation payload contents.
        """
        dynamicresources = {}
        if self._connector:
            app_resource_file = os.path.join(self._connector.getPath(), APP_RESOURCES_FILE)
            if os.path.isfile(app_resource_file):
                try:
                    with open(app_resource_file, "r") as f:
                        dynamicresources = json.load(f)

                    if dynamicresources["resources"].get("graphics") and \
                        dynamicresources["resources"]["graphics"].get("vnc-password"):
                        dynamicresources["resources"]["graphics"]["vnc-password"] = "*****"
                except Exception as ex:
                    log.exception("Error while loading the App resources file: %s, Cause: %s"%(app_resource_file, ex.message))
        return dynamicresources

    @property
    def package_config(self):
        """
        Will return the JSON object with metadata about the config file along with contents of the file.
        """
        metadata = {
            "last_modified": "",
            "size": 0,
            "contents_parsed": False
        }
        contents = ""
        if not self._container:
            log.debug("There is no container got created for the app %s, so CAF will not parse the config file"%self.app_id)
        else:
            try:
                from hostingmgmt import HostingManager
                app_manager = HostingManager.get_instance().get_service("app-management")
                config_path = app_manager.get_app_config_path(self.app_id)
                if os.path.isfile(config_path):
                    stat_info = os.stat(config_path)
                    size = stat_info.st_size
                    metadata["size"] = size
                    metadata["last_modified"] = time.strftime('%Y-%m-%d %H:%M:%S', time.localtime(stat_info.st_mtime))
                    if size > APP_PACKAGE_CONFIG_SIZE_LIMIT:
                        metadata["contents_parsed"] = False
                    else:
                        with open(config_path, "r") as f:
                            contents = f.read()
                        metadata["contents_parsed"] = True
                else:
                    metadata["contents_parsed"] = True
            except InvalidStateTransitionError as ex:
                log.info("Invalid state transistion error occurred while reading app config file: Messaeg: %s"%ex.message)
            except Exception as ex:
                log.exception("Error while loading the package config file, Cause: %s"%(ex.message))
        return {
            "metadata": metadata,
            "contents": contents
            }


    # @property
    # def dependsOnServices(self):
    #     return self._connector.metadata.dependsOnServices

    @property
    def dependsOnPackages(self):
        return self._connector.metadata.dependsOnPackages if self._connector.metadata else None

    @property
    def dependsOnCartridges(self):
        return self._connector.metadata.dependsOnCartridges if self._connector.metadata else None

    @property
    def isVisualizationEnabled(self):
        return self._connector.metadata.resources.get("visualization") if self._connector.metadata else False

    @property
    def isDatastoreEnabled(self):
        return self._connector.metadata.resources.get("datastore") if self._connector.metadata else False
       
    @property
    def resources(self):
        if  not self._connector.metadata: 
            log.error("Application meta data is missing")
            return {}
        app_resources = self._connector.metadata.resources.copy()
        rsmgr = ResourceManager.getInstance()
        app_cpu = app_resources.get('cpu')
        if app_cpu is not None:
            if self.state == State.DEPLOYED:
                app_resources['cpu'] = int(app_cpu)
            else:
                app_resources['cpu'] = rsmgr.construct_cpu_units_from_shares(app_cpu)
        else:
            app_resources['cpu'] = 'Not Available'

        app_mem = app_resources.get('memory')

        if app_mem is None:
            app_resources['memory'] = 'Not Available'
        else:
            app_resources['memory'] = int(app_mem)
            
        app_vcpu = app_resources.get('vcpu')
        if app_vcpu:
            app_resources['vcpu'] = int(app_vcpu)
            
        if self.container:
            app_resources["disk"] = rsmgr.get_disk_allocated(self.app_id)
        else:
            if not app_resources.get("disk"):
                app_resources["disk"] = DEFAULT_APP_PERSISTENT_DATA_DISK_SIZE_MB

        graphics =  app_resources.get("graphics")
        if graphics:
            if self._container:
                vnc = graphics.get('vnc')
                if vnc and isinstance(vnc,dict):
                    log.debug("vnc object - %s" % vnc)
                    if vnc["autoport"] == "yes":
                        vnc["port"] = self._container.get_vnc_port()
                    vnc["password"] = "*****"
            graphics["vnc-password"] = "*****"

        app_resources["ttycon"] = self.ttycon
        app_resources["ttyaux"] = self.ttyaux
        app_resources["ttytra"] = self.ttytra
        app_resources["ttylog"] = self.ttylog

        return app_resources

    @property
    def env(self):
        if self._container:
            return self._container.app_env
        else:
            return {}

    @property
    def app_version(self):
        return self._connector.metadata.app_version if self._connector.metadata else ""

    @property
    def dependentCartridges(self):
        cart_list = []
        if self._container.cartridge_list:
            for cart in  self._container.cartridge_list:
                cart_list.append(cart.id)
        return cart_list
    
    @property
    def app_dir(self):
        return self._container.getContainerRoot()
    
    @property
    def app_id(self):
        return self._connector.metadata.app_id if self._connector.metadata else ""
        
    @property
    def app_name(self):
        return self._connector.metadata.app_name if self._connector.metadata else ""
            
    @property
    def app_machine_name(self):
        return self._connector.metadata.app_machine_namee if self._connector.metadata else ""

    @property
    def app_start(self):
        return self._connector.metadata.app_start if self._connector.metadata else ""

    @property
    def app_description(self):
        return self._connector.metadata.app_description if self._connector.metadata else ""

    @property
    def app_binary(self):
        return self._connector.metadata.app_binary if self._connector.metadata else ""

    @property
    def app_params(self):
        return self._connector.metadata.app_params if self._connector.metadata else ""

    @property
    def app_vendor(self):
        return self._connector.metadata.app_vendor if self._connector.metadata else ""

    @property
    def app_max_wait_time(self):
        return self._connector.metadata.app_max_wait_time if self._connector.metadata else ""

    @property
    def app_supports_log(self):
        return self._connector.metadata.app_supports_log if self._connector.metadata else ""

    @property
    def app_supports_status(self):
        return self._connector.metadata.app_supports_status if self._connector.metadata else ""

    @property
    def app_supports_statistics(self):
        return self._connector.metadata.app_supports_statistics if self._connector.metadata else ""

    @property
    def app_heartbeat_interval(self):
        return self._connector.metadata.app_heartbeat_interval if self._connector.metadata else ""


    @property
    def app_checksums(self):
        app_csums={}
        app_csums["app_state"] = self.state
        if self._connector:
            app_resource_file = os.path.join(self._connector.getPath(), APP_RESOURCES_FILE)
            if os.path.exists(app_resource_file):
                sha256_app_resources = Utils.sha256_file(app_resource_file)
                if sha256_app_resources:
                    app_csums["app_resource"] = sha256_app_resources
        if self._container:
            from hostingmgmt import HostingManager
            app_manager = HostingManager.get_instance().get_service("app-management")
            app_config_contents = app_manager.get_app_config(self._connector.id)
            if app_config_contents:
                sha256_app_config = Utils.sha256_data(app_config_contents)
                if sha256_app_config:
                    app_csums["app_config"] = sha256_app_config
        return app_csums
       
        
    ###################################################3
        
    @property
    def state(self):
        #This is a temporary workaround to get the real status from container
        #without container telling us if it stopped, we will never know
        state = self._state
        if self._container:
            try:
                if self._container.isRunning() and self._container.hasFailures():
                    state = State.FAILED
                    log.critical("connector %s state: %s",self._connector.id, state)
                elif self.container.isRunning():
                    state = State.RUNNING
                elif state == State.DEPLOYED or state == State.ACTIVATED:
                    pass
                else:
                    state = State.STOPPED
                # TODO: The state should not be set here, this method should really
                #       only be retrieving the state, not setting it as well.
                #       (i.e. this method should be idempotent)
                #if self._state != state:
                #    self.setConnectorState(state)
            except:
                log.exception("Failed to get state of connector %s", self._connector.id) 
                state = State.FAILED
                log.critical("connector %s state: %s",self._connector.id, state)
        log.debug("connector %s state: %s", self._connector.id, state)
        return state

    def get_internal_status(self):
        return self._state

    @Utils.synchronized_nonblocking(_connectorWrapper_rlock, timeout=5)
    def get_internal_status_with_lock(self):
        return self._state

    @Utils.synchronized_nonblocking(_connectorWrapper_rlock, timeout=5)
    def get_desired_status_with_lock(self):
        return self.desired_state

    @property
    def connector(self):
        return self._connector
    
    @property
    def container(self):
        return self._container

    @property
    def archiveFile(self):
        return self._archiveFile

    @property
    def is_autoinstalled(self):
        return self._is_autoinstalled
        
    @property
    def health_poll_cnt(self):
        return self._health_poll_cnt

    @health_poll_cnt.setter
    def health_poll_cnt(self, value):
        self._health_poll_cnt = value

    @property
    def health_probe_fail_cnt(self):
        return self._health_probe_fail_cnt

    @health_probe_fail_cnt.setter
    def health_probe_fail_cnt(self, value):
        self._health_probe_fail_cnt = value

    @property
    def app_health(self):
        app_health ={}
        app_health["health_status"]=self.health_status    
        app_health["last_health_probe_output"]=self.last_health_probe_output
        app_health["last_health_probe_err"]=self.last_health_probe_err
        return app_health

    @property
    def post_upgrade_script_output(self):
        post_upgrade_script_output ={}
        post_upgrade_script_output["post_upgrade_status"]=self.post_upgrade_status    
        post_upgrade_script_output["post_upgrade_output"]=self.post_upgrade_output
        post_upgrade_script_output["post_upgrade_err"]=self.post_upgrade_err
        return post_upgrade_script_output

    @property
    def last_monitor_time(self):
        return self._last_monitor_time

    @last_monitor_time.setter
    def last_monitor_time(self, value):
        self._last_monitor_time = value

    @property
    def read_threshold_exceeded(self):
        return self._read_threshold_exceeded

    @read_threshold_exceeded.setter
    def read_threshold_exceeded(self, value):
        self._read_threshold_exceeded = value

    @property
    def write_threshold_exceeded(self):
        return self._write_threshold_exceeded

    @write_threshold_exceeded.setter
    def write_threshold_exceeded(self, value):
        self._write_threshold_exceeded = value

    @property
    def rootfs_read_bytes(self):
        return self._rootfs_read_bytes

    @rootfs_read_bytes.setter
    def rootfs_read_bytes(self, value):
        self._rootfs_read_bytes = value

    @property
    def rootfs_write_bytes(self):
        return self._rootfs_write_bytes

    @rootfs_write_bytes.setter
    def rootfs_write_bytes(self, value):
        self._rootfs_write_bytes = value

    @property
    def data_read_bytes(self):
        return self._data_read_bytes

    @data_read_bytes.setter
    def data_read_bytes(self, value):
        self._data_read_bytes = value

    @property
    def data_write_bytes(self):
        return self._data_write_bytes

    @data_write_bytes.setter
    def data_write_bytes(self, value):
        self._data_write_bytes = value

    @property
    def health_status(self):
        return self._health_status

    @health_status.setter
    def health_status(self, value):
        self._health_status = value
        
    @property
    def post_upgrade_status(self):
        return self._post_upgrade_status

    @post_upgrade_status.setter
    def post_upgrade_status(self, value):
        self._post_upgrade_status = value

    @property
    def last_health_probe_output(self):
        return self._last_health_probe_output

    @last_health_probe_output.setter
    def last_health_probe_output(self, value):
        self._last_health_probe_output = value
        
    @property
    def post_upgrade_output(self):
        return self._post_upgrade_output

    @post_upgrade_output.setter
    def post_upgrade_output(self, value):
        self._post_upgrade_output = value

    @property
    def last_health_probe_err(self):
        return self._last_health_probe_err

    @last_health_probe_err.setter
    def last_health_probe_err(self, value):
        self._last_health_probe_err = value
        
    @property
    def post_upgrade_err(self):
        return self._post_upgrade_err

    @post_upgrade_err.setter
    def post_upgrade_err(self, value):
        self._post_upgrade_err = value
        
    @property
    def PDHook(self):
        from hostingmgmt import HostingManager
        return HostingManager.get_instance().get_service("lifecycle-hooks")

    @archiveFile.setter
    def archiveFile(self, value):
        self._archiveFile = value

    @container.setter
    def container(self, value):
        self._container = value

    @connector.setter
    def connector(self, value):
        self._connector = value

    @property
    def system_capabilities(self):
        return self._system_capabilities

    @system_capabilities.setter
    def system_capabilities(self, value):
        self._system_capabilities = value

    @property
    def app_metrics(self):
        return self.getAppMetrics()

    @property
    def app_payload_size(self):
        try:
            if not self._app_payload_size:
                self._app_payload_size = int(round(float(Utils.get_dir_size(self.app_repo_path)) / (1024 * 1024)))
                size = self._app_payload_size
            else:
                size = self._app_payload_size
            return {
                    "size": size,
                    "unit": "MB"
                     }
        except Exception as ex:
            log.exception("Error while calculating the application payload size. id: %s, cause: %s"%(self.app_id, ex.message))
            return {
                "size": 0,
                "unit": "MB"
            }

    @property
    def svc_access_security(self):
        return self._connector.svc_access_security

    @property
    def svc_security_schemas(self):
        return self._connector.svc_security_schemas

    @property
    def verified_signature_model(self):
        return self._verified_signature_model

    def getPackage(self):
        """
        Returns the connector archive information as a tuple containing
        the following information.Throws an exception if connectorId is not valid
        (generator, name, mimeType, encoding)
        generator - Generator that returns archive content
        name - name of the archive 
        mimeType - mime type of the archive
        encoding - content encoding if available, otherwise None for platform default 
        """
        def archiveGenerator():
            archivePath = os.path.join(self._connector.getPath(), USER_CONNECTOR_ARCHIVE)        
            with io.open(archivePath, 'br') as f:
                chunk = f.read()
                yield chunk
        if self.appType == AppType.VM: #Not supported for VM type app
            return (None, None, None, None)
        return (archiveGenerator, USER_CONNECTOR_ARCHIVE, "application/zip", None)
    
    def copyPackageTo(self, f):
        """
        Copies the connector package to specified file. file argument can be
        either absolute file path or a file object opened in "w+b" mode
        """
        close  = False
        if isinstance(f, str):
            f = open(f, 'w+b')
            close = True
        
        with io.open(os.path.join(self._connector.getPath(), USER_CONNECTOR_ARCHIVE), 'br') as source:
            chunk = source.read()
            f.write(chunk)
        if close:
            f.close()

    @Utils.synchronized_blocking(_connectorWrapper_rlock)
    def activateConnector(self, resources={}, save_state=True):
        '''
        Activate Connector
        save_state: Persist the state
        '''
        self.setConnectorState(State.ACTIVATED, set_desired_state=True, save_state=save_state)

    @Utils.synchronized_blocking(_connectorWrapper_rlock)
    def deactivateConnector(self, preserveConnectorState = False, save_state=True):
        '''
        Activate Connector
        save_state: Persist the state
        '''
        if not preserveConnectorState:
            self.setConnectorState(State.DEPLOYED, True, set_desired_state=True, save_state=save_state)
        else:
            log.debug("deactivateConnector: preserving connector state")
        self.restart_attempts = 0


    @Utils.synchronized_nonblocking(_connectorWrapper_rlock, timeout=5)
    def startConnector(self, save_state=True):
        '''
        Starts the connector in container.
        save_state: Persist the state
        '''
        try:
            self.PDHook.call_app_lifecycle_hook(self.appType, 
                                         self.PDHook.PRE_START, self.env,
                                         self._connector.id, "", self._connector.metadata.resources)
                                         
            if self._container:
                if (not self._container.isRunning()):
                    # Delete OAuth Tokens issued previously
                    from hostingmgmt import HostingManager
                    oauthService = HostingManager.get_instance().get_service("oauth-service")
                    if oauthService:
                        oauthService.delete_tokens_for_app(self.connector.id)
                    self._container.start()

                else:
                    log.warning("Connector %s is already running. Ignoring "
                                "start request", self._connector.id )
                #import time
                #time.sleep(10)
                self.setConnectorState(State.RUNNING, set_desired_state=True)
                self.last_started_time = time.time()
                self._old_nw_info = {}
                self._static_dns_entries_added = False
                self._static_dns_tries_on_failure = STATIC_DNS_MAX_RETRIES
                log.info("Connector '%s' successfully started." 
                        % self._connector.id)
                        
                self.PDHook.call_app_lifecycle_hook(self.appType, 
                                         self.PDHook.POST_START, self.env,
                                         self._connector.id, "", self._connector.metadata.resources)

                # After app start, make sure host bridges retain the correct mtu
                from hostingmgmt import HostingManager
                nwc = HostingManager.get_instance().get_service("network-management")
                if nwc:
                    nwc.set_mtu(self.networkInfo)
                        
                return 0
            else:
                log.error("Connector %s is missing container. Cannot start"
                          %  self._connector.id )
                self.setConnectorState(State.STOPPED, set_desired_state=True, save_state=save_state)
                return -1
        except Exception as e:
            log.error("Exception while starting the connector:%s" % self._connector.id)
            log.debug("Traceback:",  exc_info=True)
            raise AppStartError("Error while starting the app: %s, Cause: %s"%(self._connector.id, str(e)))
    
    @Utils.synchronized_nonblocking(_connectorWrapper_rlock, timeout=5)
    def stopConnector(self, preserveConnectorState=False, graceful=True, force_container_stop=False, save_state=True):
        try: 
            self.PDHook.call_app_lifecycle_hook(self.appType, 
                                         self.PDHook.PRE_STOP, self.env,
                                         self._connector.id, "", self._connector.metadata.resources)
            if self._container:
                if self._container.isRunning() or self.get_internal_status() == State.RUNNING:
                    if self._container.isRunning() or force_container_stop:
                        self._container.stop(graceful=graceful)

                if preserveConnectorState :
                    log.info("Preserving Connector '%s''s state as '%s'" % (self._connector.id, self._state))
                else:
                    #if forceful:
                    #TODO AC5 if we dont set this , dependent connector may throw error since this connector is not stopped
                    #Make sure if forceful is False , to invoke terminate Connector API
                    #Also while invoking terminate we check if its running, if it gets stopped state will not be set
                    #import time
                    #time.sleep(10)
                    self.setConnectorState(State.STOPPED, set_desired_state=True, save_state=save_state)
                log.info("Connector '%s' successfully stopped." % self._connector.id)
                
                self.PDHook.call_app_lifecycle_hook(self.appType, 
                                         self.PDHook.POST_STOP, self.env,
                                         self._connector.id, "", self._connector.metadata.resources)
                return 0
            return -1
        except:
            log.error("Exception while stoping the connector:%s" % self._connector.id)
            log.debug("Traceback:",  exc_info=True)
            raise

    @Utils.synchronized_nonblocking(_connectorWrapper_rlock, timeout=5)
    def terminateConnector(self, preserveConnectorState=False):
        try:
            if self._container:
                if self._container.isRunning() or self.get_internal_status() == State.RUNNING:
                    if self._container.isRunning():
                        self._container.terminate()
                if preserveConnectorState :
                    log.info("Preserving Connector '%s''s state as '%s'" % (self._connector.id, self._state))
                else:
                    self.setConnectorState(State.STOPPED, set_desired_state=True)
                return 0
            return -1
        except:
            log.error("Exception while terminating the connector:%s" % self._connector.id)
            log.debug("Traceback:",  exc_info=True)
            #AC5 Dont raise the exception
            #raise


    def setAppCustomOptions(self, app_custom_options):
        if app_custom_options is not None:
            self._connector.runtimeConfig.appCustomOptions = app_custom_options
    
    def setAppGroup(self, app_group=False):
        self._connector.runtimeConfig.appgroup = app_group


    @Utils.synchronized_nonblocking(_connectorWrapper_rlock, timeout=5)
    def setConnectorState(self, state, force_write=False, set_desired_state=False, save_state=True):
        if force_write == False and self._state == state: return

        log.info("Connector '%s' state change: %s-->%s" 
                    % (self._connector.id, self._state, state))
        if state == State.STOPPED:
            self.is_network_assigned = False
        self._state = state
        if save_state:
            self._connector.runtimeConfig.runtimeStatus = state
        self._connector.runtimeConfig.app_uuid = self.get_app_uuid() 
        self._last_state_change_time = datetime.datetime.now().strftime("%Y-%m-%d %H:%M:%S")
        if set_desired_state:
            self.desired_state = state

    def getProcessInfo(self):
        if self._container is None:
            raise Exception("Container does not exist. State %s", self.state)
        return self._container.getProcessInfo()

    def get_app_uuid(self):
        """
        Return unique identifier for the app. What this unique ID is container dependent.
        For libvirt based container implementations, this can be the domains UUID, for process container, it can be the process ID.
        """
        if self._container is None:
            return ""

        app_uuid = self._container.get_app_uuid()
        if app_uuid == None:
            app_uuid = ""

        return app_uuid

    def executeCustomScript(self, scriptName, args=None):
        if self._container is None:
            raise Exception("Container does not exist. State %s", self.state)
        r = self._container.executeCustomScript(scriptName, args)

        if None == r:
            raise Exception("Script %s is not supported by the app %s" % (
                scriptName, self._connector.id
            ))

        return r

    def getConnectorLogsList(self):
        if self._container is None:
            raise Exception("Container does not exist. State %s", self.state)
        return self._container.getLogsList()     

    def deleteConnectorLogFiles(self, filename=None):
        if self._container is None:
            log.error("Container does not exist. State %s", self.state)
            raise Exception("Container does not exist. State %s", self.state)
        return self._container.deleteLogfiles(filename)

    def getConnectorCoreList(self):
        if self._container is None:
            raise Exception("Container does not exist. State %s", self.state)
        return self._container.getCoreList()     

    def getConnectorCoreFile(self, corefile):
        if self._container is None:
            raise Exception("Container does not exist. State %s", self.state)
        return self._container.getCoreFile(corefile)     

    def getAppMetrics(self):
        appMetrics = {}
        appMetrics['memory_allocated'] = 'Not available'
        appMetrics['cpu_allocated'] = 'Not available'
        appMetrics['disk_allocated'] = 'Not available'
        resources = self.resources
        if self._container is None:
            appMetrics['memory'] = None
            appMetrics['cpu'] = None
            appMetrics['network']= None
            appMetrics['disk'] = None
        else:
            mem = self._container.get_app_memory_usage()
            appMetrics['memory'] = mem
            cpu = self._container.get_app_cpu_usage()
            appMetrics['cpu'] = cpu
            network = self._container.get_app_network_usage()
            appMetrics['network']=network
            disk = self._container.get_app_disk_usage()
            appMetrics['disk'] = disk
            if resources.get("memory") and not isinstance(resources.get("memory"), basestring):
                appMetrics['memory_allocated'] = resources.get("memory") * 1024 #Converting from MB to KB
            if resources.get("cpu") and not isinstance(resources.get("cpu"), basestring):
                appMetrics['cpu_allocated'] = resources.get("cpu")
            if resources.get("disk") and not isinstance(resources.get("disk"), basestring):
                appMetrics['disk_allocated'] = resources.get("disk")
        curAppMetrics = {}
        curAppMetrics['memory'] = {}
        if not appMetrics['memory'] is None:
            curAppMetrics['memory']['current']= appMetrics['memory']
        else:
            curAppMetrics['memory']['current']= 'Not available'
        curAppMetrics['memory']['allocated']= {}
        curAppMetrics['memory']['allocated']['value'] = appMetrics['memory_allocated']
        curAppMetrics['memory']['allocated']['unit'] = 'KB'
        curAppMetrics['memory']['unit']= 'KB'
        curAppMetrics['cpu'] = {}
        if not appMetrics['cpu'] is None:
            curAppMetrics['cpu']['current'] = round(appMetrics['cpu'], 2)
        else:
            curAppMetrics['cpu']['current'] = 'Not available'
        curAppMetrics['cpu']['allocated'] = {}
        curAppMetrics['cpu']['allocated']['value'] = appMetrics['cpu_allocated']
        curAppMetrics['cpu']['allocated']['unit'] = 'SHARES'
        curAppMetrics['cpu']['unit'] = 'percent'
        curAppMetrics['network'] = {}
        if not appMetrics['network'] is None:
            curAppMetrics['network']['current'] = appMetrics['network']
        else:
            curAppMetrics['network']['current'] = 'Not available'
        curAppMetrics['network']['unit'] = 'bytes'
        curAppMetrics['disk'] = {}
        if not appMetrics['disk'] is None:
            curAppMetrics['disk']['current']= round(appMetrics['disk'],2)
        else:
            curAppMetrics['disk']['current']= 'Not available'
        curAppMetrics['disk']['allocated'] = {}
        curAppMetrics['disk']['allocated']['value'] = appMetrics['disk_allocated']
        curAppMetrics['disk']['allocated']['unit'] = 'MB'
        curAppMetrics['disk']['unit'] = 'MB'
        return curAppMetrics

    @property
    def networkInfo(self):
        if self._container:
            nw_info = self._container.get_app_ipaddress_info()
            if nw_info and not self.is_network_assigned:
                count = 0
                for interface, info in nw_info.iteritems():
                    if info.get("ipv4") or info.get("ipv6"):
                        count += 1
                if count == len(nw_info):
                    self.is_network_assigned = True
            return nw_info
        else:
            return "Ip Address is unavailable since Container not created"

    def getLogTail(self, filename, lines):
        if self._container is None:
            raise Exception("Container does not exist. State %s", self.state)
        return self._container.getLogTail(filename, lines)

    def getLogPath(self, filename):
        if self._container is None:
            raise Exception("Container does not exist. State %s", self.state)
        return self._container.getLogPath(filename)

    def getConnectorLogContents(self, fileName):        
        if self._container is None:
            raise Exception("Container does not exist. State %s", self.state)
        return self._container.getLogContents(fileName)

    def logfiles_metadata(self, pattern=".*"):
        """
        Will return the json object consists of file name with metadata.
        metadata will contains all the required info like, how many mtches does that file has,
            last modified, size of the file etc.,
        """
        metadata = {}
        if self._container is None:
            log.info("The container is in DEPLOYED state, so no log files to view!")
            return metadata
        app_logs_list = self._container.getLogsList()
        for file_meta in app_logs_list:
            file_name = file_meta[0]
            file_path = self._container._get_logfile_path(file_name)
            if os.path.isfile(file_path) and Utils.is_textfile(file_path):
                meta = {}
                stat_info = os.stat(file_path)
                meta["size"] = stat_info.st_size
                meta["last_modified"] = time.strftime('%Y-%m-%d %H:%M:%S', time.localtime(stat_info.st_mtime))
                data, rc = grep_pipe("-c", "-e", pattern, file_path)
                if rc == 0:
                    meta["number_of_matches"] = int(data.strip())
                else:
                    meta["number_of_matches"] = 0
                data, rc = wc("-l", file_path)
                if rc == 0:
                    meta["total_lines"] = int(data.split()[0].strip())
                else:
                    meta["total_lines"] = 0
                metadata[os.path.basename(file_path)] = meta
        return metadata

    def logfile_contents(self, filename, pattern, after, before, start_from, seek_direc, num_matches, max_number_of_lines):
        """
        With the applied filters this method will return the contents of the given log file.
        """
        content_obj = {}
        if self._container is None:
            log.info("The container is in DEPLOYED state, so no log files to view!")
            return content_obj
        file_path = self._container._get_logfile_path(filename)
        if not os.path.isfile(file_path):
            log.error("Given file name %s, is not found in app's %s log dir"%(filename, self.app_id))
            raise MandatoryFileMissingError("Given file name %s, is not found in app's %s log dir"%(filename, self.app_id))
        if not Utils.is_textfile(file_path):
            raise IOError("Given file %s invalid file format!"%filename)
        return Utils.filter_file_contents(file_path, pattern, after, before, start_from, seek_direc, num_matches, max_number_of_lines)


    def getRuntimeProperty(self, name):
        if (name.lower() == 'startupready'):             
            return self._connector.runtimeConfig.startupReady
        elif (name.lower() == 'uuid'):             
            return self._connector.runtimeConfig.app_uuid
        else:
            raise Exception("no runtime property found: %s", name)          

    @Utils.synchronized_nonblocking(_connectorWrapper_rlock, timeout=5)
    def setRuntimeProperty(self, name, value):
        if (name.lower() == 'startupready'):
            self._connector.runtimeConfig.startupReady = value
            pass
        else:
            raise Exception("no runtime property found: %s", name)

    def set_data_disk(self, ext_path):
        if os.path.exists(ext_path):
            self._data_disk = ext_path
        else:
            raise Exception("Invalid path provided")

    def get_data_disk(self):
        return self._data_disk

    def is_network_changed(self):
        nw_info = self.networkInfo
        log.info("Network info: %s" % nw_info)
        if self._old_nw_info != nw_info:
            if isinstance(nw_info, dict) and nw_info:
                count = 0
                for interface, info in nw_info.iteritems():
                    if info.get("ipv4") or info.get("ipv6") :
                        count += 1
                if count == len(nw_info):
                    self.is_network_assigned = True
                    self._old_nw_info = copy.deepcopy(nw_info)
                    return True
        return False

    #If Force_entries is passed as True, then eventhough we have already added the static DNS entries,
    # This flag will force this method to perform it again.
    def add_static_dns_entry(self, force_entries = False):
        """
        Check if the static DNS entry present as part of /etc/resolv.conf file,
            if not append the entries to the file.
        In case of all entries are present ib resolv.conf file then will return True otherwise False
        """
        try:
            if not self._static_dns_entries_added or force_entries:
                if self.appType != AppType.VM and self._container and self._container.isRunning():
                    dns_entries = []
                    for interfaces in self.resources.get("network", []):
                        mode = interfaces.get("mode")
                        if mode and mode == "static":
                            if isinstance(interfaces.get("ipv4"), dict) and interfaces["ipv4"].get("dns"):
                                dns_entries.append(interfaces["ipv4"].get("dns"))
                            if isinstance(interfaces.get("ipv6"), dict) and interfaces["ipv6"].get("dns"):
                                dns_entries.append(interfaces["ipv6"].get("dns"))
                        #Check if mode is specified inside the ipv4 or ipv6 section
                        else:
                            if isinstance(interfaces.get("ipv4"), dict) and interfaces.get("ipv4").get("mode") == "static" and interfaces["ipv4"].get("dns"):
                                dns_entries.append(interfaces["ipv4"].get("dns"))
                            if isinstance(interfaces.get("ipv6"), dict) and interfaces.get("ipv6").get("mode") == "static"  and interfaces["ipv6"].get("dns"):
                                dns_entries.append(interfaces["ipv6"].get("dns"))
                    if dns_entries and self._static_dns_tries_on_failure > 0:
                        if self._container.add_static_dns_entry(dns_entries):
                            log.debug("Successfully added the static DNS entries %s"%dns_entries)
                            self._static_dns_entries_added = True
                            self._static_dns_tries_on_failure = STATIC_DNS_MAX_RETRIES
                        else:
                            self._static_dns_tries_on_failure = self._static_dns_tries_on_failure - 1
        except Exception as ex:
            log.exception("Error while adding static dns entries for the app %s. Cause: %s"%(self.app_id, ex.message))

    def serialize(self, detailed=False):
        d = dict()
        for k in self._to_serialize:
            log.debug("Serializing: %s" % k)
            f = getattr(self, k)
            d[k] = f
        if detailed:
            for k in self._detailed:
                f = getattr(self, k)
                d[k] = f
        return d

