'''
Created on Jun 20, 2016

@author: edeschen

Copyright (c) 2016 by Cisco Systems, Inc.
All rights reserved.
'''


import base64
import binascii
import logging
import os
import traceback
from ..utils import utils
from ..utils.commandwrappers import *
from ..utils.utils import Utils
import yaml
import shutil
import tarfile
import tempfile


log = logging.getLogger("runtime.hosting")


class PackageSigning:
    """
    Simple Class to handle Package signing validation
    """
    # Certificate store which contains the trusted certificate
    # loaded from the CA Chain file
    _store = None
    _trusted_certs = []
    # default value if not provided in the configuration file
    _cert_chain_filename = "/opt/caf/ca-chain.cert.pem"
    # marker line for PEM formatted certificate files
    _pem_cert_end_line = "-----END CERTIFICATE-----"
    __singleton = None
    __singleton_init_done = False

    def __new__(cls, *args, **kwargs):
        # Check to see if a __singleton exists already for this class
        # Compare class types instead of just looking for None so
        # that subclasses will create their own __singleton objects
        if cls != type(cls.__singleton):
            #if not cls.__singleton:
            cls.__singleton = super(PackageSigning, cls).__new__(cls, *args, **kwargs)
        return cls.__singleton

    def __init__(self, data=None):
        if not self.__singleton_init_done:
            self.trust_anchor_dir = Utils.getSystemConfigValue('package_validation', 'ca_dir', '/tmp')
            self.trust_anchor_file = "ca-chain.cert.pem"
            self.trust_anchor_package = "trust_anchor.tar.gz"
            if data is None:
                # default it
                data = self._cert_chain_filename
                # try to get value from config
                if self.trust_anchor_file is not None:
                    log.debug("Received package_validation::ca_file from config %s" % self.trust_anchor_file)
                    data = os.path.join(self.trust_anchor_dir, self.trust_anchor_file)

            self.trust_anchor_metadata_file = "info.txt"
            self.trust_anchor_fullpath = data
            self._appsign_init_config = {}
            self._appsign_init_config["enabled"] = Utils.getSystemConfigValue('package_validation', 'check_package_signature', True, "bool")
            repo_folder = Utils.getSystemConfigValue("controller", "repo", "/etc")
            work_folder=None
            if Utils.hasSystemConfigOption("DEFAULT", "caf_work_dir"):            
                work_folder = Utils.getSystemConfigValue("DEFAULT", "caf_work_dir", None)
            if work_folder is None: 
                work_folder = os.path.dirname(repo_folder)

            running_config_repo = os.path.join(work_folder, "running_config")
            self.appsign_runtime_config_file = ".appsign"
            self.appsign_runtime_fullpath = os.path.join(running_config_repo, self.appsign_runtime_config_file)

            self._appsign_runtime_config = {}
            self.load_config()
            self._store = None
            from ..runtime.platformcapabilities import PlatformCapabilities
            pc = PlatformCapabilities.getInstance()
            if pc.trust_anchor_management_enabled:
                self._load_store()
            __singleton_init_done = True



    @property
    def appsign_enabled(self):
        return self._appsign_runtime_config["enabled"]

    def get_config(self):
        return self._appsign_runtime_config

    def set_config(self, val):
        if self.validate_config(val):
            log.debug("about to update the config")
            self._appsign_runtime_config.update(val)
            with open(self.appsign_runtime_fullpath, "w", 0) as f:
                yaml.dump(self._appsign_runtime_config, f, default_flow_style=False)
            log.debug("updated config file")
        else:
            raise Exception("Invalid package signing configuration : %s" % out)


    def validate_and_extract(self, trust_archive, tmpdir):
        try:
            if tarfile.is_tarfile(trust_archive):
                tarfile_expected_contents = [self.trust_anchor_file, self.trust_anchor_metadata_file]
                with tarfile.open(trust_archive) as tar:
                    tarfile_contents = tar.getnames()
                    for content in tarfile_expected_contents:
                        if content not in tarfile_contents:
                            raise Exception("File %s not found, expected contents: %s" % (content,tarfile_expected_contents))
                    Utils.check_for_absolutepaths(tar)
                    rval, errcode = tar_extractall(trust_archive, os.path.join(tmpdir, "extracted_dir"))
                    if errcode != 0:
                        log.error("Failed to extract trust anchor archive file - error: %s", str(rval))
                        raise Exception("Failed to extract trust anchor archive file - error: %s", str(rval))
                # Called just to verify if the certs are valid.
                self._get_certs_from_file(os.path.join(tmpdir, "extracted_dir", self.trust_anchor_file))

                if os.path.isdir(self.trust_anchor_dir):
                    shutil.rmtree(self.trust_anchor_dir)
                shutil.move(os.path.join(tmpdir, "extracted_dir"), self.trust_anchor_dir)
                shutil.move(trust_archive, os.path.join(self.trust_anchor_dir, self.trust_anchor_package))
            else:
                raise Exception("Invalid file format. Import tar archive of trust anchor")
        except Exception as ex:
            log.exception("Invalid file format - %s", str(ex))
            raise Exception("Invalid trust anchor uploaded")
        finally:
            if os.path.isdir(os.path.join(tmpdir, "extracted_dir")):
                shutil.rmtree(os.path.join(tmpdir, "extracted_dir"))

    def get_pkgsign_metadata(self):
        metadata = ""
        pkgsign_metadata_file = os.path.join(self.trust_anchor_dir, self.trust_anchor_metadata_file)
        if os.path.exists(pkgsign_metadata_file):
            try:
                with open(pkgsign_metadata_file) as file_hdlr:
                    metadata = file_hdlr.read()
                return metadata
            except IOError:
                log.error("Failed to read package signing metadata file at %s.", pkgsign_metadata_file)
                log.debug("Stack traceback",  exc_info=True)
                raise IOError
        return metadata

    def validate_config(self, config):
        # check if only allowed keys are passed
        log.debug("Validating the given config %s"% config)
        allowed_keys = self._appsign_runtime_config.keys()
        for key in config.keys():
            if key not in allowed_keys:
                log.debug("Invalid key %s, has been found in new config"%key)
                return False
        return True

    def calculate_sha1sum(self):
        import hashlib
        hash_sha1 = hashlib.sha1()
        trust_anchor_package = os.path.join(self.trust_anchor_dir, self.trust_anchor_package)
        if not os.path.isfile(trust_anchor_package):
            return ""
        with open(trust_anchor_package, "rb") as f:
            for chunk in iter(lambda: f.read(4096), b""):
                hash_sha1.update(chunk)
        return hash_sha1.hexdigest()


    def load_config(self):
        from ..runtime.runtime import RuntimeService
        self.runtime = RuntimeService.getInstance()
        self._appsign_runtime_config = self.runtime.merge_configs(self._appsign_init_config, self.appsign_runtime_fullpath)
        log.debug("app runtime config : %s" % self._appsign_runtime_config)

    def _print_cert_info(certificate):
        print "Subject: %s" % certificate.get_subject()
        print "Issuer: %s" % certificate.get_issuer()
        print "Has expired: %s" % certificate.has_expired()
        print "Version: %s" % certificate.get_version()
        print "Pubkey type: %s" % certificate.get_pubkey().type()
        print "Pubkey bits: %s" % certificate.get_pubkey().bits()
        print "Signature digest: %s" % certificate.get_signature_algorithm()

    def getTrustAnchor(self):
        trust_anchor = ""
        if os.path.exists(self.trust_anchor_fullpath):
            try:
                with open(self.trust_anchor_fullpath) as file_hdlr:
                    trust_anchor = file_hdlr.read()
                return trust_anchor
            except IOError:
                log.error("Failed to read trust anchor content at %s.", self.trust_anchor_fullpath)
                log.debug("Stack traceback",  exc_info=True)
                raise IOError
        return trust_anchor

    def _load_store(self):
        from OpenSSL import crypto
        # Initialize new store
        store = crypto.X509Store()

        log.debug("Loading Certificate store from file: %s", self.trust_anchor_fullpath)
        self._trusted_certs = self._get_certs_from_file(self.trust_anchor_fullpath)
        for cert in self._trusted_certs:
            store.add_cert(cert)

        log.debug("Certificate store loaded with %d certificates", len(self._trusted_certs))
        # Need to delete old self._store or else we might leaking memory
        self._store = store

    def _get_certs_from_file(self, cert_filename):
        from OpenSSL import crypto
        certs = []
        # check if file exists
        if not os.path.isfile(cert_filename):
            return certs

        # open the file and go through each concatenated
        # certificate entries
        try:
            with open(cert_filename, 'r') as f:
                strbuf = ""     # buffer to contain a PEM cert in string

                # read each line until we reach the end of a certificate or the end of the file
                while 1:
                    line = f.readline().strip()
                    if len(line) == 0:
                        # empty line, this is the end of the file
                        break;
                    #print "line: %s" % line
                    log.debug("Certificate line: %s", line)
                    strbuf += line
                    strbuf += '\n'
                    if line == self._pem_cert_end_line:
                        # Found end of the cert, get a cert using the string
                        cert_from_file = crypto.load_certificate(crypto.FILETYPE_PEM, strbuf)
                        certs.append(cert_from_file)
                        #_print_cert_info(cert_from_file)
                        # clear the buffer to start a new certificate buffer
                        strbuf = ""

            return certs
        except Exception as ex:
            log.exception("Invalid trust anchor certificate - %s", str(ex))
            raise Exception("Invalid trust anchor certificate")
        

    def _extract_data_from_pkgcert(self, pkgcert_filename):
        """
        Parse the package.cert file to retrieve
            - cert in PEM format
            - signature
            - digest algorithm
        """
        signature = None
        digest = None
        cert_pem_string = None

        # First line should be the signature
        # The lines following should be the cert in PEM format
        with open(pkgcert_filename, 'r') as f:
            # read the signature first
            line = f.readline().strip()
            if len(line) == 0:
                raise Exception("Found empty line instead of expected signature in %s" % utils.APP_CERTIFICATE_NAME)

            # get the digest algorithm which is the string before (
            digest = line[:line.index('(')]
            log.debug("algo: %s" % digest)

            # signature is the rest of the line (after the space)
            signature = line[line.index(' ')+1:].rstrip('\n')
            log.debug("signature[len=%d]: %s" % (len(signature), signature))

            # we're done with the signature, process the cert
            cert_pem_string = ""

            # next line should be the start of the cert
            cert_pem_string = f.read().rstrip('\n')
            log.debug("cert pem: %s" % cert_pem_string)

        return signature, digest, cert_pem_string

    def _verify_signature(self, cert, signature, signed_fn, digest):
        from OpenSSL import crypto
        """
        Verify that a precomputed signature matches the data
        it was supposedly generated for
        """
        # TODO: should sanitize the length of the signature!
        #print "data: %s" % data

        # The signature is in hex, get the binary version
        log.debug("Getting binary string from hex")
        signature_bin = binascii.unhexlify(signature)

        # Load up the file content in a string
        file_content = ""
        # TODO: there's gotta to be a more pythonic way!
        with open(signed_fn, 'r') as f:
            for i in f:
                file_content += i

        log.debug("Verifying signature...")
        crypto.verify(cert, signature_bin, file_content, digest)

    def _is_cert_trusted(self, cert):
        from OpenSSL import crypto
        """
        Verify if a given cert is trusted using a store
        preloaded with trusted CA chain certificates
        """

        if self._store is None:
            raise Exception("Could not initialize certificate store")

        if not self._trusted_certs:
            raise Exception("No trust anchor found to validate package's certificate")

        # Create a store context and add the cert to verify
        store_ctx = crypto.X509StoreContext(self._store, cert)

        try:
            # Perform the verification
            store_ctx.verify_certificate()
            return True

        except Exception as e:
            raise Exception("Certificate verification failed: %s" % str(e))

        return False

    def verify_package_signature(self, extracted_path):
        from OpenSSL import crypto
        """
        Validate the signature for an extracted package in
        specified location (folder)
        """

        pkgcert_fn = os.path.join(extracted_path,
                                  utils.APP_CERTIFICATE_NAME)
        pkgmf_fn = os.path.join(extracted_path,
                                utils.PACKAGE_MANIFEST_NAME)

        if not os.path.isfile(pkgcert_fn):
            raise Exception("Application package signature validation is enabled. Missing mandatory file %s" % pkgcert_fn)

        if not os.path.isfile(pkgmf_fn):
            raise Exception("Application package signature validation is enabled. Missing mandatory file: %s" % pkgmf_fn)

        # Parse the package.mf to get:
        #   - the signature
        #   - the digest algorithm
        #   - the certificate in PEM format
        signature, digest, cert_str = self._extract_data_from_pkgcert(pkgcert_fn)
        # get a certificate object from the extracted string representation
        cert = crypto.load_certificate(crypto.FILETYPE_PEM, cert_str)

        # First check if the cert is a trusted one
        if not self._is_cert_trusted(cert):
            log.debug("Certificate from package was not trusted")
            raise Exception("Certificate not trusted")

        # Now that we know the cert from the package.cert is
        # trusted, perform signature validation
        self._verify_signature(cert,
                               signature,
                               pkgmf_fn,
                               digest.lower())

    @classmethod
    def getInstance(cls, *args):
        '''
        Returns a singleton instance of the class
        '''
        if not cls.__singleton:
            cls.__singleton = PackageSigning(*args)
        return cls.__singleton

