#! /bin/sh
#
# svcbr_teardown.sh
# Invoked by CAF at removal of a logical network, this script will teardown
# the hosting bridge and the physical or VLAN interface underneath it.
#
# Usage: svcbr_teardown.sh [options]
#     Options:
#     -b or -bridge     - hosting bridge to delete
#     -i or -interface  - physical interface
#     -v or -vlan       - VLAN ID. The script will delete the VLAN interface if VLAN ID is provided
#

SVCBR_IF=""
PHY_IF=""
VLAN_ID=""

LOGGER="/usr/bin/logger -t svcbr_teardown"

# sanity check the input parameters
while [ "$#" != 0 ];
do
    case $1 in
        -b | -bridge )
            SVCBR_IF=$2;
            ;;
        -i | -interface )
            PHY_IF=$2;
            ;;
        -v | -vlan )
            VLAN_ID=$2;
            ;;
        -* )
            ;;
    esac
    shift
done

if [ -z $SVCBR_IF ]; then
    echo "bridge interface is missing!"
    exit 1
fi

# kill the dhcp process if needed
DHCP_PID_FILE=/var/run/udhcpc.${SVCBR_IF}.pid
if [ -e "$DHCP_PID_FILE" ]; then
    kill -9 `cat $DHCP_PID_FILE`
    rm -f $DHCP_PID_FILE
fi

ip link show $SVCBR_IF > /dev/null 2>&1
# delete the bridge svcbr_x, do nothing if the bridge does not exist
if [ $? -eq 0 ]; then
    ip link set $SVCBR_IF down > /dev/null 2>&1
    errmsg=`brctl delbr $SVCBR_IF 2>&1`
    if [ $? -ne 0 ]; then
        echo "fails to delete bridge $SVCBR_IF"
        echo "$errmsg"
        exit 1
    fi
    echo "bridge $SVCBR_IF has been deleted"
    ${LOGGER} "bridge $SVCBR_IF has been deleted"
fi

# delete the VLAN interface if applicable
if [ ! -z "$VLAN_ID" ]; then
    if [ -z "$PHY_IF" ]; then
        echo "physical interface is missing"
        exit 1
    fi
    ip link show ${PHY_IF}.${VLAN_ID} > /dev/null 2>&1
    if [ $? -ne 0 ]; then
        exit
    fi
    ip link delete dev ${PHY_IF}.${VLAN_ID} > /dev/null 2>&1
    if [ $? -ne 0 ]; then
        echo "fails to delete VLAN interface ${PHY_IF}.${VLAN_ID}"
        echo "$errmsg"
        exit 1
    fi
    echo "VLAN interface ${PHY_IF}.${VLAN_ID} has been deleted"
    ${LOGGER} "VLAN interface ${PHY_IF}.${VLAN_ID} has been deleted"
fi
