__author__ = "madawood"

import subprocess
import logging
import os
import yaml
import tarfile
import tempfile
import json
import re
import copy
import shutil
from hashlib import sha256
from appfw.utils.commandwrappers import *

if __package__ is None:
    import sys
    from os import path
    sys.path.append(path.dirname(path.dirname(path.dirname(path.abspath(__file__)))))

from caf_abstractservice import CAFAbstractService, ServiceParams
from appfw.utils.infraexceptions import InvalidConfigError, IntegrityCheckFailedError, MandatoryFileMissingError, LayerDoesNotExistError, LayerDependencyError
from appfw.utils.utils import Utils, LAYER_METADATA_FILE, LAYER_MANIFEST_FILE, LAYER_ARCHIVE_FILE, LAYER_CONTENTS_DIR
from appfw.utils.commandwrappers import tar_extractall, which, cp

LAYER_LIST_FILE = ".layer_list.json"
LAYER_RANKING_FILE = "layer_ranking.json"
BLOCKSIZE = (2**20)
log = logging.getLogger("runtime.hosting")


class LayerRegistry(CAFAbstractService):
    """
    This will handle the all life cycle operations on the layers.
     - Using this can add and delete the layers to CAF registry.
     - Apps needed the layers will use this instance.
    """
    __singleton = None # the one, true Singleton
    __singleton_init_done = False

    def __new__(cls, *args, **kwargs):
        # Check to see if a __singleton exists already for this class
        # Compare class types instead of just looking for None so
        # that subclasses will create their own __singleton objects
        if cls != type(cls.__singleton):
            cls.__singleton = super(LayerRegistry, cls).__new__(cls, *args, **kwargs)
        return cls.__singleton

    def __init__(self, params, base_repo="/tmp", tmpupload_dir="/tmp"):
        """
        :param params: Service param object, contains service name, config needed by this and config file path.
        :param base_repo: In this base repo CAF will create layer_reg repo and store all its layer data in it.
        :param offline_base_repo: In this repo CAF will create a offline dir, where user can upload all layer archives.
                Which are internally picked up by CAF.
        :param tmpupload_dir: For temporary extraction of archives we use this dir.
        """
        super(LayerRegistry, self).__init__()
        log.debug("Initializing the Layer Registry")
        self.name = params.name
        self.config = params.config
        self._config_file = params.config_file
        self.layers = {}
        self._dependent_map = {}
        self.base_repo = base_repo
        self.tmpupload_dir = tmpupload_dir
        self.integrity_handler = sha256
        self.layer_ranking = {}
        log.debug("Layer registry initialized with the config %s"%self.config)


    @property
    def max_layers_per_app(self):
        return self.config.get("max_layers_per_app", 42)

    @property
    def max_allowed_layers(self):
        return self.config.get("max_allowed_layers", 100)

    @property
    def offline_repo(self):
        return self.config.get("offline_repo", "/tmp/offline_layers")

    '''
    Location of the layers provided by platform
    Generally this will RO system.One cannot delete it
    '''
    @property
    def platform_repo(self):
        return self.config.get("platform_repo", "/tmp/platform_layers")

    @property
    def repo(self):
        return os.path.join(self.base_repo, self.config.get("repo", "layers"))

    @property
    def supported_integrity_check_algo(self):
        return self.config.get("supported_integrity_check_algo", "sha256")

    def start(self):
        if self.supported_integrity_check_algo == "SHA256":
            self.integrity_handler = sha256
        else:
            log.error("Supported integrity algo: %s handler is not defined"%self.supported_integrity_check_algo)
            raise ValueError("Supported integrity algo: %s handler is not defined"%self.supported_integrity_check_algo)
        self._load()

    def _calculate_hash(self, file_path):
        hasher = self.integrity_handler()
        if os.path.isfile(file_path):
            with open(file_path, "r") as f:
                while True:
                    data = f.read(BLOCKSIZE)
                    if not data:
                        break
                    hasher.update(data)
            return hasher.hexdigest()
        else:
            log.error("LayerReg: Given path %s is not a file!"%file_path)
            raise ValueError("LayerReg: Given path %s is not a file!"%file_path)

    @classmethod
    def getInstance(cls):
        '''
        Returns a singleton instance of the class
        '''
        if cls.__singleton == None:
            cls.__singleton = LayerRegistry()
        return cls.__singleton

    def _load(self):
        """
        Will load all layer from the registry repo.
        Delete the layer firs from repo, if they are not proper.
        """
        repo = os.path.join(self.base_repo, self.config.get("repo", "layers"))
        if not os.path.exists(repo):
            os.makedirs(repo)
        if not os.path.isdir(self.offline_repo):
            os.makedirs(self.offline_repo)
        try:
            if os.path.isfile(os.path.join(repo, LAYER_RANKING_FILE)):
                with open(os.path.join(repo, LAYER_RANKING_FILE)) as f:
                    self.layer_ranking = json.load(f)
        except Exception as ex:
            log.exception("Error while loading layer ranking file: %s, Cause: %s"%(LAYER_RANKING_FILE, str(ex)))
        log.info("Loading the layers from repo %s"%repo)
        for layer in os.listdir(repo):
            layer_dir = os.path.join(repo, layer.encode("utf-8"))
            try:
                if os.path.isdir(layer_dir):
                    log.debug("Loading the layer dir - %s"%layer_dir)
                    layer_metadata = self._validate_added_layer(layer_dir)
                    layer_id = layer_metadata["layer_id"]
                    self.layers[layer_id] = layer_metadata
                    self.layers[layer_id]["used_by"] = []
                    self.layers[layer_id]["symlinked_to"] = []
            except Exception as ex:
                shutil.rmtree(layer_dir, ignore_errors=True)
                log.exception("Error while loading the layer: %s: cause: %s"%(layer, ex.message))
        self.add_offline_layers()
        #self.add_platform_layers()
        log.debug("Successfully loaded the layers %s from repo!"%self.layers.keys())


    def _validate_added_layer(self, layer_loc):
        """
        :return:
        """
        layer_meta_file = os.path.join(layer_loc, LAYER_METADATA_FILE)
        layer_metadata = {}
        if not os.path.isfile(layer_meta_file):
            log.error("Layer doesn't have metadata file: %s"%LAYER_METADATA_FILE)
            raise MandatoryFileMissingError("Layer doesn't have metadata file: %s"%LAYER_METADATA_FILE)
        with open(layer_meta_file, "r") as f:
            layer_metadata = json.load(f)
        layer_id = layer_metadata.get("layer_id")
        if not layer_id:
            log.error("There is no proper layer id present as part of layer metadata!")
            raise ValueError("There is no proper layer id present as part of layer metadata!")
        if layer_id.strip() != os.path.basename(layer_loc):
            log.error("Layer dir name %s is not the layer id %s"%(os.path.basename(layer_loc), layer_id))
            raise ValueError("Layer dir name %s is not the layer id %s"%(os.path.basename(layer_loc), layer_id))
        if not os.path.isdir(os.path.join(layer_loc, LAYER_CONTENTS_DIR)):
            log.error("For this layer %s contents dir: %s is not found."%(layer_id, LAYER_CONTENTS_DIR))
            raise MandatoryFileMissingError("For this layer %s contents dir: %s is not found."%(layer_id, LAYER_CONTENTS_DIR))
        return layer_metadata


    def get_config(self):
        return self.config

    def add(self, archive):
        """
        Will validate the given layer archive.
        If the archive is valid then will get added to the registry, else appropriate error will be thrown.
        Return: Will return Layer_id, boolean(this will represent whether layer is already existed or newly added)
        """
        log.debug("Adding the layer archive %s to the repo!"%archive)
        if tarfile.is_tarfile(archive):
            temp_dir = tempfile.mkdtemp(dir=self.tmpupload_dir)
            layer_repo_dir = ""
            try:
                with tarfile.open(archive) as tar:
                    Utils.check_for_absolutepaths(tar)
                    log.debug("Extracting the layer to archive %s to the location %s"%(archive, temp_dir))
                    rval, errcode = tar_extractall(archive, temp_dir)
                    if errcode != 0:
                        log.error("Failed to extract docker layer %s - error: %s", (archive, str(rval)))
                        raise Exception("Failed to extract docker layer %s - error: %s", (archive, str(rval)))
                layer_metadata = self.validate_layer(temp_dir)
                layer_id = layer_metadata["layer_id"]
                docker_id = layer_metadata["docker_id"]
                if layer_id in self.layers:
                    log.info("Layer Id %s is already exists as part of registry, so ignoring new payload!"%layer_id)
                    return layer_id, True
                layer_repo_dir = os.path.join(self.repo, layer_id.encode("utf-8"))
                layer_contents = os.path.join(layer_repo_dir, LAYER_CONTENTS_DIR)
                if os.path.isdir(layer_repo_dir):
                    shutil.rmtree(layer_repo_dir, ignore_errors=True)
                os.mkdir(layer_repo_dir)
                os.mkdir(layer_contents)
                shutil.copy2(os.path.join(temp_dir, docker_id, LAYER_ARCHIVE_FILE), layer_contents)
                for f in os.listdir(temp_dir):
                    if f == docker_id:
                        continue
                    shutil.move(os.path.join(temp_dir, f), layer_repo_dir)
                self.layers[layer_id] = layer_metadata
                self.layers[layer_id]["used_by"] = []
                self.layers[layer_id]["symlinked_to"] = []
                if layer_metadata.get("size"):
                    self.layers[layer_id]["size"] = layer_metadata.get("size")
                else:
                    self.layers[layer_id]["size"] = os.path.getsize(archive)
                log.info("Successfully added the layer %s"%layer_id)
                from appfw.runtime.hostingmgmt import HostingManager
                hm = HostingManager.get_instance()
                hasync_service = hm.get_service("hasync-service")
                if hasync_service:
                    hasync_service.sync_caf_data("dir", self.repo)

                return layer_id, False
            except Exception as ex:
                if os.path.isdir(layer_repo_dir):
                    shutil.rmtree(layer_repo_dir, ignore_errors=True)
                log.exception("Error while adding layer archive %s to the registry, cause: %s"%(archive, ex.message))
                raise
            finally:
                if os.path.isdir(temp_dir):
                    shutil.rmtree(temp_dir, ignore_errors=True)
        else:
            log.error("Given archive %s is not a valid layer archive!"%archive)
            raise Exception("Given archive %s is not a valid layer archive!"%archive)

    def add_app_layer(self, layer_dir):
        layer_repo_dir = ""
        layer_id = ""
        try:
            if os.path.islink(os.path.join(layer_dir, LAYER_ARCHIVE_FILE)):
                log.error("Given docker layer %s, contains soft link. Need to repackage the app with IOXCLIENT(Version - greater or equal 1.3.4)!"%os.path.basename(layer_dir))
                raise ValueError("Given docker layer %s, contains soft link. Need to repackage the app with IOXCLIENT(Version - greater or equal 1.3.4)!"%os.path.basename(layer_dir))
            if os.path.isdir(layer_dir):
                if tarfile.is_tarfile(os.path.join(layer_dir, LAYER_ARCHIVE_FILE)):
                    metadata = {}
                    metadata["layer_id"] = self._calculate_hash(os.path.join(layer_dir, LAYER_ARCHIVE_FILE))
                    metadata["size"] = os.path.getsize(os.path.join(layer_dir, LAYER_ARCHIVE_FILE))
                    metadata["docker_id"] = os.path.basename(layer_dir)
                    layer_id = metadata["layer_id"]
                    if layer_id in self.layers:
                        log.info("Layer Id %s is already exists as part of registry, so ignoring new payload!"%layer_id)
                        return layer_id, True
                    layer_repo_dir = os.path.join(self.repo, layer_id.encode("utf-8"))
                    if os.path.isdir(layer_repo_dir):
                        shutil.rmtree(layer_repo_dir, ignore_errors=True)
                    os.mkdir(layer_repo_dir)
                    layer_contents_dir = os.path.join(layer_repo_dir, LAYER_CONTENTS_DIR)
                    os.mkdir(layer_contents_dir)
                    shutil.copy2(os.path.join(layer_dir, LAYER_ARCHIVE_FILE), layer_contents_dir)
                    with open(os.path.join(layer_repo_dir, LAYER_METADATA_FILE), "w", 0) as f:
                        f.write(json.dumps(metadata))
                    self.layers[layer_id] = metadata
                    self.layers[layer_id]["used_by"] = []
                    self.layers[layer_id]["symlinked_to"] = []
                    from appfw.runtime.hostingmgmt import HostingManager
                    hm = HostingManager.get_instance()
                    hasync_service = hm.get_service("hasync-service")
                    if hasync_service:
                        hasync_service.sync_caf_data("dir", self.repo)
                    return layer_id, False
                else:
                    log.error("Given layer dir %s, doesn't have the layer archive %s"%(layer_dir, LAYER_ARCHIVE_FILE))
                    raise Exception("Given layer dir %s, doesn't have the layer archive %s"%(layer_dir, LAYER_ARCHIVE_FILE))
            else:
                log.error("Given layer dir %s, is not a directory!"%layer_dir)
                raise Exception("Given layer dir %s, is not a directory!"%layer_dir)
        except Exception as ex:
            if os.path.isdir(layer_repo_dir):
                shutil.rmtree(layer_repo_dir, ignore_errors=True)
            self.layers.pop(layer_id, "")
            log.exception("Error while adding a APP layer to registry. Cause: %s"%ex.message)
            raise

    def add_offline_layers(self):
        """
        Will list the offline repo dir and loop through them and all them to layer registry.
        Once the package got processed by CAF, it will get deleted(no matter if the package is valid or invalid)
        """
        added_layers = []
        if not os.path.isdir(self.offline_repo):
            return added_layers
        for layer in os.listdir(self.offline_repo):
            if self.is_layer_arch_valid(os.path.join(self.offline_repo, layer)):
                try:
                    self.add(os.path.join(self.offline_repo, layer))
                    added_layers.append(layer)
                except Exception as ex:
                    log.exception("Error while adding the offline layer to registry. Cause: %s"%ex.message)
                finally:
                    if os.path.isfile(os.path.join(self.offline_repo, layer)):
                        log.info("After processing removing the layer archive %s from offline repo!"%os.path.join(self.offline_repo, layer))
                        os.remove(os.path.join(self.offline_repo, layer))
            else:
                log.debug("Given layer path: %s, is not a valid layer"%os.path.join(self.offline_repo, layer))
        if added_layers:
            log.info("Layers added from offline repo are: %s"%added_layers)
        else:
            log.debug("Layers added from offline repo are: %s"%added_layers)
        return added_layers

    def add_platform_layers(self):
        """
        Will list the platform repo dir and loop through them and all them to layer registry.
        Once the package got processed by CAF,it will not be delete since its part of platform image
        """
        added_layers = []
        if not os.path.isdir(self.platform_repo):
            log.info("Platform repo layer dir not available")
            return added_layers
        for layer in os.listdir(self.platform_repo):
            if self.is_layer_arch_valid(os.path.join(self.platform_repo, layer)):
                try:
                    self.add(os.path.join(self.platform_repo, layer))
                    added_layers.append(layer)
                except Exception as ex:
                    log.exception("Error while adding the platform layer to registry. Cause: %s"%ex.message)
            else:
                log.debug("Given platform layer path: %s, is not a valid layer"%os.path.join(self.platform_repo, layer))
        log.info("Layers added from platform repo are: %s"%added_layers)
        return added_layers

    def recover_platform_layers(self, app_id=None):
        # if app_id is not None:
        #     try:
        #         self.destroy_app_layers(app_id)
        #     except Exception as ex:
        #         log.info("Error while destroying app layers")
        if not os.path.isdir(self.platform_repo):
            log.info("Platform layers dir not available")
            return
        deleted_layers = []
        for layer in os.listdir(self.platform_repo):
            if self.is_layer_arch_valid(os.path.join(self.platform_repo, layer)):
                archive = os.path.join(self.platform_repo, layer)
                log.debug("Adding the layer archive %s to the repo!"%archive)
                if tarfile.is_tarfile(archive):
                    temp_dir = tempfile.mkdtemp(dir=self.tmpupload_dir)
                    layer_repo_dir = ""
                    try:
                        with tarfile.open(archive) as tar:
                            Utils.check_for_absolutepaths(tar)
                            log.debug("Extracting the layer to archive %s to the location %s"%(archive, temp_dir))
                            rval, errcode = tar_extractall(archive, temp_dir)
                            if errcode != 0:
                                log.error("Failed to extract docker layer archive %s - error: %s", (archive, str(rval)))
                                raise Exception("Failed to extract docker layer archive %s - error: %s", (archive, str(rval)))
                        layer_metadata = self.validate_layer(temp_dir)
                        layer_id = layer_metadata["layer_id"]
                        docker_id = layer_metadata["docker_id"]
                        if layer_id:
                            log.debug("Deleting the layer %s"%layer_id)
                            '''
                            If platform layer exists in the common registry and is not symlinked , delete and write fresh one
                            Ignore used_by since each container has its own copy
                            If platform layer does not exist in common registry, its ok.
                            While installing the container, platform layers will be copied back
                            '''
                            if layer_id in self.layers.keys():
                                linked_to = self.get_simlinked_to(layer_id)
                                if not linked_to:
                                    if os.path.isdir(os.path.join(self.repo, layer_id.encode("utf-8"))):
                                        shutil.rmtree(os.path.join(self.repo, layer_id.encode("utf-8")), ignore_errors=True)
                                    else:
                                        log.info("Layer dir %s is not found in repo"%os.path.join(self.repo, layer_id))
                                    self.layers.pop(layer_id, "")
                                    deleted_layers.append(layer_id)

                                    layer_repo_dir = os.path.join(self.repo, layer_id.encode("utf-8"))
                                    layer_contents = os.path.join(layer_repo_dir, LAYER_CONTENTS_DIR)
                                    if os.path.isdir(layer_repo_dir):
                                        shutil.rmtree(layer_repo_dir, ignore_errors=True)
                                    os.mkdir(layer_repo_dir)
                                    os.mkdir(layer_contents)
                                    with tarfile.open(os.path.join(temp_dir, docker_id, LAYER_ARCHIVE_FILE)) as tar:
                                        out_put, ret = tar_extractall(os.path.join(temp_dir, docker_id, LAYER_ARCHIVE_FILE), layer_contents)
                                        if ret != 0:
                                            log.error("Extraction of tar file %s using tar binary failed. Reason: %s ret_code: %s"%(os.path.join(temp_dir, docker_id, LAYER_ARCHIVE_FILE), out_put, ret))
                                            Utils.check_for_absolutepaths(tar)
                                            raise Exception("Extraction of tar file %s using tar binary failed. Reason: %s ret_code: %s"%(os.path.join(temp_dir, docker_id, LAYER_ARCHIVE_FILE), out_put, ret))
                                    for f in os.listdir(temp_dir):
                                        if f == docker_id:
                                            continue
                                        shutil.move(os.path.join(temp_dir, f), layer_repo_dir)
                                    self.layers[layer_id] = layer_metadata
                                    self.layers[layer_id]["used_by"] = []
                                    self.layers[layer_id]["symlinked_to"] = []
                                    log.info("Successfully added the layer %s"%layer_id)
                                else:
                                    log.exception("Given layerId %s is used by the apps/services: %s. So will not be get deleted"%(layer_id, linked_to))
                                    continue
                    except Exception as ex:
                        if os.path.isdir(layer_repo_dir):
                            shutil.rmtree(layer_repo_dir, ignore_errors=True)
                        log.exception("Error while adding layer archive %s to the registry, cause: %s"%(archive, ex.message))
                    finally:
                        if os.path.isdir(temp_dir):
                            shutil.rmtree(temp_dir, ignore_errors=True)
                else:
                    log.error("Given archive %s is not a valid layer archive!"%archive)
                    continue

    def is_layer_arch_valid(self, layer_arch):
        """
        This method will check whether the given file is a layer archive or not.
        :return: Boolean - whether the archive is layer archive or not.
        """
        try:
            if not os.path.isdir(layer_arch) and tarfile.is_tarfile(layer_arch) and not str(os.path.basename(layer_arch)).endswith(".part"):
                with tarfile.open(layer_arch) as tar:
                    try:
                        tar.getmember(LAYER_METADATA_FILE)
                    except KeyError:
                        return False
                    return True
            else:
                return False
        except Exception as ex:
            log.info("Error while checking for a valid offline layer. File: %s Cause: %s"%(layer_arch, ex.message))
            return False
    """
    def remove_offline_layers(self):

        Will loop though the offline repo and delete the files which got added.
        :return: Will return the list of file names, which got deleted

        deleted = []
        if not os.path.isdir(self.offline_repo):
            return deleted
        for archive in os.listdir(self.offline_repo):
            if os.path.isfile(os.path.join(self.offline_repo, archive)):
                os.remove(os.path.join(self.offline_repo, archive))
                deleted.append(archive)
    """

    def validate_layer(self, layer_loc):
        """
        Will validate the layer.
        """
        log.debug("Validating the layer!")
        self._check_mandatory_files(layer_loc)
        self._check_integrity(layer_loc)
        layer_json_file = os.path.join(layer_loc, LAYER_METADATA_FILE)
        layer_metadata = {}
        with open(layer_json_file) as f:
            layer_metadata = json.load(f)
            self.validate_layer_metadata(layer_loc, layer_metadata)
        return layer_metadata

    def _check_mandatory_files(self, layer_loc):
        """
        Will check for all mandatory files are present.
         If any file is not preset then MandatoryFileMissingError will be thrown.
        """
        log.debug("Checking for the all mandatory files are present!")
        layer_json_file = os.path.join(layer_loc, LAYER_METADATA_FILE)
        layer_manifest_file = os.path.join(layer_loc, LAYER_MANIFEST_FILE)
        if not os.path.isfile(layer_json_file):
            log.error("Layer doesn't have metadata file: %s"%LAYER_METADATA_FILE)
            raise MandatoryFileMissingError("Layer doesn't have metadata file: %s"%LAYER_METADATA_FILE)
        if not os.path.isfile(layer_manifest_file):
            log.error("Mandatory file %s is missing in layer package!"%LAYER_MANIFEST_FILE)
            raise MandatoryFileMissingError("Mandatory file %s is missing in layer package!"%LAYER_MANIFEST_FILE)

    def _check_integrity(self, layer_loc):
        """
        Will validate the package integrity.
        """
        log.debug("Validating the layer integrity!")
        pattern = re.compile(
            '(?P<hashing_technique>%s)[(](?P<filename>[a-zA-Z0-9-_.\/]+)[)]=\s(?P<shadigest>[A-Fa-f0-9]+)'%self.supported_integrity_check_algo.upper())
        layer_manifest_file = os.path.join(layer_loc, LAYER_MANIFEST_FILE)
        if not os.path.isfile(layer_manifest_file):
            log.error("Mandatory file %s is missing in layer package!"%LAYER_MANIFEST_FILE)
            raise MandatoryFileMissingError("Mandatory file %s is missing in layer package!"%LAYER_MANIFEST_FILE)
        # validating the entries in package manifest
        with open(layer_manifest_file) as f:
            for line in str(f.read()).splitlines():
                # Ignoring the empty and commented lines
                line = line.strip()
                if line == "" or line.startswith("#"):
                    continue
                match = pattern.match(line)
                if match:
                    hashing_technique = match.group("hashing_technique")
                    filename = match.group("filename")
                    shadigest = match.group("shadigest")
                    try:
                        if not os.path.isfile(os.path.join(layer_loc, filename)):
                            log.error("Given file name %s in manifest is not existing"%filename)
                            raise MandatoryFileMissingError("Given file name %s in manifest is not existing"%filename)
                        if shadigest != self._calculate_hash(os.path.join(layer_loc, filename)):
                            log.error("Integrity of the file %s is not matched!"%filename)
                            raise IntegrityCheckFailedError("Integrity of the file %s is not matched!"%filename)
                    except Exception as ex:
                        log.exception("SHA digest check failed for %s, cause: %s" % (filename, ex.message))
                        #raise IntegrityCheckFailedError("SHA digest check failed for %s, cause: %s" % (filename, ex.message))
                        raise
                else:
                    log.error("Invalid digest value")
                    raise IntegrityCheckFailedError("Invalid entry [ %s ] in layer manifest file " % line)

    def validate_layer_metadata(self, layer_loc, layer_json):
        """
        Will validate the data in layer.json file.
        """
        if layer_json.get("layer_id") and layer_json.get("integrity_check_algo") and layer_json.get("docker_id"):
            integrity_algo = layer_json.get("integrity_check_algo")
            if self.supported_integrity_check_algo != integrity_algo:
                log.error("Supported integrity check algo: %s, doesn't match with layer integrity algo: %s"%(self.supported_integrity_check_algo, integrity_algo))
                raise Exception("Supported integrity check algo: %s, doesn't match with layer integrity algo: %s"%(self.supported_integrity_check_algo, integrity_algo))
            layer_id = layer_json.get("layer_id")
            docker_id = layer_json.get("docker_id")
            layer_dir = os.path.join(layer_loc, docker_id)
            layer_arch = os.path.join(layer_loc, docker_id, LAYER_ARCHIVE_FILE)
            if not os.path.isdir(layer_dir):
                log.error("Layer dir %s is not found in layer package"%docker_id)
                raise MandatoryFileMissingError("Layer dir %s is not found in layer package"%docker_id)
            if os.path.islink(layer_arch):
                log.error("Given docker layer %s, contains soft link. Need to repackage the app with IOXCLIENT(Version - greater or equal 1.3.4)!"%os.path.basename(os.path.dirname(layer_arch)))
                raise ValueError("Given docker layer %s, contains soft link. Need to repackage the app with IOXCLIENT(Version - greater or equal 1.3.4)!"%os.path.basename(os.path.dirname(layer_arch)))
            if not tarfile.is_tarfile(layer_arch):
                log.error("Layer archive %s is not valid TAR file"%LAYER_ARCHIVE_FILE)
                raise MandatoryFileMissingError("Layer archive %s is not valid TAR file"%LAYER_ARCHIVE_FILE)
            if layer_id != self._calculate_hash(layer_arch):
                log.error("Integrity check failed for layer archive: %s"%LAYER_ARCHIVE_FILE)
                raise IntegrityCheckFailedError("Integrity check failed for layer archive: %s"%LAYER_ARCHIVE_FILE)
        else:
            log.error("Mandatory fields are missing in layer metadata!")

    def validate_max_limit(self):
        """
        Will validate the maximum number of layers allowed in registry.
        """
        if len(self.layers.keys()) > self.max_allowed_layers:
            return False
        else:
            return True

    def delete(self, layer_id=None):
        """
        If layer id is provided then it will only delete that layer.
        If layer id is not provided then, it will delete all unused layers from registry.
        """
        deleted_layers = []
        offline_deleted_layers = []
        if layer_id:
            log.debug("Deleting the layer %s"%layer_id)
            if layer_id in self.layers.keys():
                used_by = self.get_used_by(layer_id)
                if not used_by:
                    if os.path.isdir(os.path.join(self.repo, layer_id.encode("utf-8"))):
                        shutil.rmtree(os.path.join(self.repo, layer_id.encode("utf-8")), ignore_errors=True)
                    else:
                        log.info("Layer dir %s is not found in repo"%os.path.join(self.repo, layer_id))
                    self.layers.pop(layer_id, "")
                    self.layer_ranking.pop(layer_id, "")
                    deleted_layers.append(layer_id)
                else:
                    log.error("Given layerId %s is used by the apps/services: %s. So will not be get deleted"%(layer_id, used_by))
                    raise LayerDependencyError("Given layerId %s is used by the apps/services: %s. So will not be get deleted"%(layer_id, used_by))
            else:
                log.error("Given layer id %s, is not found"%layer_id)
                raise LayerDoesNotExistError("Given layer id %s, is not found"%layer_id)
        else:
            log.debug("As no layer id is given, so deleting all unused layers!")
            for id in self.layers.keys():
                used_by = self.get_used_by(id)
                if not used_by:
                    if os.path.isdir(os.path.join(self.repo, id.encode("utf-8"))):
                        shutil.rmtree(os.path.join(self.repo, id.encode("utf-8")), ignore_errors=True)
                    else:
                        log.info("Layer dir %s is not found in repo"%os.path.join(self.repo, id))
                    self.layers.pop(id, "")
                    self.layer_ranking.pop(id, "")
                    deleted_layers.append(id)
                else:
                    log.info("Given layerId %s is used by the apps/services: %s. So will not be get deleted"%(id, used_by))
        self._save_layer_ranking()
        log.info("Successfully deleted the layers: %s"%deleted_layers)
        from appfw.runtime.hostingmgmt import HostingManager
        hm = HostingManager.get_instance()
        hasync_service = hm.get_service("hasync-service")
        if hasync_service:
            hasync_service.sync_caf_data("dir", self.repo)

        return deleted_layers, offline_deleted_layers

    def list(self):
        return self.layers.keys()

    def get(self, layer_id=None):
        """
        :param layer_id:
        :return:
        """
        if layer_id:
            if layer_id in self.layers:
                layer_metadata = self.layers[layer_id]
                layer_metadata["rank"] = self.layer_ranking.get(layer_id, -1)
                return layer_metadata
            else:
                return {}
        else:
            metadata = []
            for layer_metadata in self.layers.values():
                layer_metadata["rank"] = self.layer_ranking.get(layer_metadata["layer_id"], -1)
                metadata.append(layer_metadata)
            return metadata

    def get_used_by(self, layer_id):
        """
        Will return the list with apps/services using the given layer id.
        If the layer id not found, then will return the empty list.
        """
        if self.layers.get(layer_id):
            return self.layers[layer_id].get("used_by", [])
        else:
            return []

    def get_simlinked_to(self, layer_id):
        """
        Will return the list with apps/services using the given layer id.
        If the layer id not found, then will return the empty list.
        """
        if self.layers.get(layer_id):
            return self.layers[layer_id].get("symlinked_to", [])
        else:
            return []

    def add_layer_dependency(self, layer_list, dependent_id):
        """
        :param layer_id:
        :param dependent_id:
        :return:
        """
        missing_layers = self.check_layers_exists(layer_list)
        if missing_layers:
            log.debug("Layers %s are missing in registry!"%missing_layers)
            return missing_layers
        if dependent_id in self._dependent_map:
            self._dependent_map[dependent_id].extend(layer_list)
        else:
            self._dependent_map[dependent_id] = layer_list
        log.debug("Dependent map: %s" % str(self._dependent_map[dependent_id]))
        try:
            for layer in layer_list:
                layer_info = self.layers[layer]
                if dependent_id not in layer_info["used_by"]: 
                    layer_info["used_by"].append(dependent_id)
                self.layers[layer] = layer_info
            return []
        except Exception as ex:
            log.exception("Error while adding the app layer %s, in registry!"%ex.message)
            self.remove_dependency(dependent_id)

    def remove_dependency(self, dependent_id):
        """
        Will remove dependency from all layers
        :param layer_id:
        :param dependent_id:
        :return:
        """
        layer_list = self._dependent_map.get(dependent_id, [])
        if layer_list:
            for layer in layer_list:
                if self.layers.get(layer):
                    if dependent_id in self.get_used_by(layer):
                        self.layers[layer]["used_by"].remove(dependent_id)
            self._dependent_map.pop(dependent_id, "")
        else:
            log.debug("There is no layer dependency found for %s"%dependent_id)

    def check_layers_exists(self, layer_list):
        """
        :param layer_list: List of layers needs to be checked
        :return: If there are any missing layers will return that list.
            If all layers are existing then returns the empty list.
        """
        missing_layers = list(set(layer_list) - set(self.layers.keys()))
        if missing_layers:
            log.info("Layers %s are missing in registry"%missing_layers)
            return missing_layers
        else:
            log.debug("All app layers are present as part of registry!")
            return []

    def get_dependent_layers(self, app_id):
        """
        Will returns the list of layers depends on the given app id.
        """
        return self._dependent_map.get(app_id, [])

    def setup_app_layers(self, layers, app_id, app_repo, clean_unused_layers = False, skip_copy_apprepo=False):
        layers_copied_to_apprepo = []
        missing_layers = self.check_layers_exists(layers)
        if missing_layers:
            log.error("Layers %s is not found in registry"%json.dumps(missing_layers).encode("utf8"))
            raise LayerDoesNotExistError("Layers %s is not found in registry"%json.dumps(missing_layers).encode("utf8"))
        try:
            self.add_layer_dependency(layers, app_id)
            self._assign_ranking(layers)
            if clean_unused_layers:
                self.clean_layers(layers=layers)
            for layer in layers:
                layer_info = self.layers.get(layer)
                if layer_info:
                    log.debug("SKIP COPY REPO: %s" % skip_copy_apprepo)
                    if not skip_copy_apprepo:
                        if os.path.exists(os.path.join(app_repo, layer)):
                            if os.path.islink(os.path.join(app_repo, layer)):
                                layer_info["symlinked_to"].append(app_id)
                        else:
                            data, rc = cp("-dpR", os.path.join(self.repo, layer), os.path.join(app_repo, layer))
                            if rc != 0:
                                log.error("Error while copying the files %s to %s, by using CP command. "
                                          "So that going use the internal utils function for copy."%(os.path.join(self.repo, layer), os.path.join(app_repo, layer)))
                                Utils.copytree(os.path.join(self.repo, layer), os.path.join(app_repo, layer), symlinks=True)
                            layers_copied_to_apprepo.append(layer)
                else:
                    log.error("No layer info found as part for a given layer %s"%layer)
                    raise ValueError("No layer info found as part for a given layer %s"%layer)
        except Exception as ex:
            try:
                self.destroy_app_layers(app_id)
            except Exception as ex:
                pass
            if type(ex) == subprocess.CalledProcessError:
                log.error("Error while setting up the layers: %s for the app: %s, cause: %s"%(layers, app_id, ex.output))
                raise Exception("CalledProcessError: " + ex.output)
            else:
                log.error("Error while setting up the layers: %s for the app: %s, cause: %s"%(layers, app_id, ex.message))
                raise ex
        return layers_copied_to_apprepo

    def destroy_app_layers(self, app_id):
        try:
            log.debug("Going to remove the layer dependency of the app %s"%app_id)
            layers_used = self.get_dependent_layers(app_id)
            self.remove_dependency(app_id)
            for layer in layers_used:
                layer_info = self.layers.get(layer)
                if layer_info:
                    if layer_info.get("symlinked_to"):
                        if app_id in layer_info.get("symlinked_to"):
                            layer_info.get("symlinked_to").remove(app_id)
        except Exception as ex:
            log.exception("Error while destroying app dependency of the app %s"%app_id)
            raise ex

    def _assign_ranking(self, ordered_app_layers):
        """
        Depending upon the list of the layers, we will define each layer with ranking.
        Ranking will be define as most base layer as '1' and top most layer will be 'n'.
        """
        log.debug("Assigning the ranking for the layers: %s"%ordered_app_layers)
        count = 1
        for layer in ordered_app_layers:
            if layer in self.layer_ranking:
                #
                if self.layer_ranking[layer] > count:
                    self.layer_ranking[layer] = count
            else:
                self.layer_ranking[layer] = count
            count = count + 1
        self._save_layer_ranking()

    def _save_layer_ranking(self):
        """
        This file will help preserving the ranking of the layers, across the CAF reboots
        """
        with open(os.path.join(self.repo, LAYER_RANKING_FILE), "w", 0) as f:
            f.write(json.dumps(self.layer_ranking))
        from appfw.runtime.hostingmgmt import HostingManager
        hm = HostingManager.get_instance()
        hasync_service = hm.get_service("hasync-service")
        if hasync_service:
            hasync_service.sync_caf_data("file", os.path.join(self.repo, LAYER_RANKING_FILE))

    def clean_layers(self, layers=None, size_req=None):
        """
        Depending on the clean unused layer flag, will clean the ununsed layers depending on the ranking.
        """
        log.debug("Cleaning up the layers to make through app installation incase of disk crunch!")
        if size_req is not None:
            total_space_needed = size_req
        else:
            total_space_needed = self._calculate_space_needed_by_app(layers)
        repo_dir = Utils.getSystemConfigValue("controller", "repo")
        free_disk_space = Utils.get_free_disk_space(repo_dir) * (1024 * 1024)
        if free_disk_space < total_space_needed:
            disk_space_needed = total_space_needed - free_disk_space
            log.debug("Free disk space available: %s and disk space needed by app: %s"%(free_disk_space, disk_space_needed))
            layers = self.get_unused_layers()
            #size_can_freedup = free_disk_space
            size_can_freedup = 0
            layers_can_be_deleted = []
            delete_layers = False
            # Layers are unused and without any ranking
            layers_without_ranking = list(set(layers) - set(self.layer_ranking.keys()))

            #Layers with ranking and unused
            layers_with_ranking = list(set(layers).intersection(self.layer_ranking.keys()))
            for layer in sorted(layers_with_ranking, key=lambda x: self.layer_ranking[x], reverse=True):
                size_can_freedup = size_can_freedup + self.layers[layer].get("size", 0)
                layers_can_be_deleted.append(layer)
                if size_can_freedup >= disk_space_needed:
                    delete_layers = True
                    break
            if not delete_layers:
                for layer in layers_without_ranking:
                    size_can_freedup = size_can_freedup + self.layers[layer].get("size", 0)
                    layers_can_be_deleted.append(layer)
                    if size_can_freedup >= disk_space_needed:
                        delete_layers = True
                        break
            if delete_layers:
                log.debug("Layers going to get deleted are : %s"%layers_can_be_deleted)
                for layer in layers_can_be_deleted:
                    try:
                        self.delete(layer)
                    except Exception as ex:
                        log.exception("Error while freeing up the space by deleting the layers %s, Cause %s"%(layer,str(ex)))
            else:
                log.info("Even if we delete the unused layers app will not get passed disk crunch. So not deleting the layers!")
        else:
            log.info("No need to delete the layers, as there is already disk space available. Free space: %s, Total space needed: %s"%(free_disk_space, total_space_needed))

    def _calculate_space_needed_by_app(self, layers):
        """
        Will calculate the total disk space needed by the layers.
        """
        total_space = 0
        for layer in layers:
            if self.layers.get(layer):
                total_space = total_space + self.layers[layer].get("size", 0)
        return total_space


    def get_unused_layers(self):
        """
        Will return the list of layers not used any app.
        """
        unused_layers = []
        for layer in self.layers.keys():
            if not self.get_used_by(layer):
                unused_layers.append(layer)
        return unused_layers


'''
if "__main__" == __name__:
    config_file = Utils.getLayerRegConfigFile()
    _config = {}
    if os.path.isfile(config_file):
        with open(config_file, "r") as f:
            _config = yaml.load(f)
    else:
        raise Exception("No config file found:%s" % config_file)
    params = {"name": "layer_registry", "config": _config, "config_file": config_file}
    params = ServiceParams(params)
    l_reg = LayerRegistry(params)
    l_reg.start()
    archive_1 = os.path.join("/home/madawood/layer/layers", "08f405d988e4d56c204a995fb7d8cffed4620e297172316beef06fd668b8491e.tar.gz")
    l_reg.add(archive_1)
    #l_reg.add(os.path.join())
    for layer in l_reg.get().keys():
        print "Layer : %s, metadata: %s"%(layer, l_reg.get(layer))
    '''
