#! /bin/ash
#
# Available PD hooks and inputs for docker type application lifecycle
# Environment variables defined in app yaml are also accessible from this PD script
#
# Return codes:
# Only least significant 8 bits of the return code is retained (POSIX convention)
# eg. return code -1 is interpreted as 255
# Non-zero return codes < 127 are reported as errors in CAF log, but CAF operation won't terminate
# Return codes >=128 are considered fatal and will cause CAF to terminate the lifecycle operation
#
export PATH=/bin:/sbin:/usr/bin:/var/wireless:/usr/sbin:/opt/cisco/bin
CAF_REPO=/storage/software/caf/work/repo

. /usr/bin/platformfunc.sh

firmware_download_process() {
    alias=`grep "alias*:" $app_repo_dir/package.yaml | awk -F ": " '{ print $2 }' \
           | sed 's/^[ \t]*//;s/[ \t]*$//'`
    if [ -z "$alias" ] ; then
        return 0
    elif [ "$alias" == "ble" ] || [ "$alias" == "zigbee" ] || [ "$alias" == "iot_radio0" ]; then
        serial_device="/dev/ttyiot0"
        if [ -z $serial_device ] ; then
            echo "Requested serial device does not exist"
            return 255
        else
            local driver=`get_ble_driver`
            process="iox_$app_id"
            #If BLE management is enabled, disable BLE management from CMX and sendout the reason code to wlc.
            device=${serial_device#"/dev/"}
            alias=${alias#"external_"}
            if [  "$alias" != "iot_radio0" ]; then
                /usr/bin/cli_scripts/iot_fw_dnld.sh $alias default $process
            fi
            if [ "$alias" == "zigbee" ] ; then
                echo "Notify eWLC that IOx app is using ZigBee Firmware"
                wcp_write $driver.admin_cfg_cmd 8
            elif [ "$alias" == "ble" ] ; then
                echo "Notify eWLC that IOx app is using BLE Firmware"
                wcp_write $driver.ble_fw_operation 5
            elif [ "$alias" == "iot_radio0" ]; then
                wcp_write $driver.ble_fw_operation 2
                filename="/storage/iot_status_$device.txt"
                sed -i "s/process .*/process $process/g" $filename
                wcp_write $driver.ble_fw_operation 5
            fi
        fi
    fi
    return 0
}

cleanup_process() {
    alias=`grep "alias*:" $app_repo_dir/package.yaml | awk -F ": " '{ print $2 }' \
           | sed 's/^[ \t]*//;s/[ \t]*$//'`
    if [ -z "$alias" ] ; then
        exit 0
    elif [ "$alias" == "ble" ] || [ "$alias" == "zigbee" ] || [ "$alias" == "iot_radio0" ]; then
        serial_device=/dev/ttyiot0
        filename="/storage/iot_status_${serial_device#"/dev/"}.txt"
        local driver=`get_ble_driver`
        if [ -z "$serial_device" ] ; then
            exit 0
        else
            echo "Removing iox_$app_id from $filename"
            if [ -f $filename ]; then
                sed -i "s/iox_$app_id/NA/g" $filename
            fi
            echo " Informing WCP to take over $serial_device"
            wcp_write $driver.ble_fw_operation 5
        fi
        if [ "$alias" == "iot_radio0" ]; then
            default_fw=`cat /storage/iot_status_ttyH0.txt | grep default_fw | awk '{print $2}'`
            if [ "$default_fw" == "false" ];
            then
                echo " Over writing default BLE firmware on to IoT chipset"
                /usr/bin/cli_scripts/iot_fw_dnld.sh ble default test
            fi
        fi
    fi
}

handle_post_activate() {
    alias=""
    firmware_download_process "post-activate"
    ret=$?
    if [ ! -z $alias ];
    then
        echo "Sending ACTIVATED msg to grpc for $appid with alias $alias"
        send_ipc grpc iox_app_status $appid ACTIVATED $alias
    else
        echo "Sending ACTIVATED msg to grpc for $appid"
        send_ipc grpc iox_app_status $appid ACTIVATED
    fi
    exit $ret
}

handle_pre_deploy() {
    exit 0
}

handle_pre_activate() {
    rootfs_mnt=/storage/software/caf/work/repo-lxc/$appid/rootfs_mnt
    data_mnt=/storage/software/caf/work/repo-lxc/lxc-data/$appid

    mountpoint -q $rootfs_mnt && \
        umount $rootfs_mnt ; echo " Unmounting RootFS in pre-activate stage : $rootfs_mnt : Return Val $?"

    mountpoint -q $data_mnt && \
        umount $data_mnt ; echo " Unmounting data mount in pre-activate stage : $data_mnt : Return Val $?"
    exit 0
}

handle_post_deploy() {
    # check alias field for usb, if /dev/ttyUSB0 dosn't exist
    # print error message and retrub error to CAF
    local ret=0
    local serial_device=`python /opt/lxc/scripts/iox_app_get_serial_device.py check-alias $app_repo_dir/package.yaml`
    if [ -z "$serial_device" ] ; then
        ret=0
    else
        for device in $serial_device
        do
            if [ ! -e "$device" ]; then
                echo "Requested USB device not found on AP."
                ret=255
                break
            fi
        done
    fi
    if [ $ret == 0 ]; then
        echo "Sending DEPLOYED msg to grpc for $appid"
        send_ipc grpc iox_app_status $appid DEPLOYED
        #firmware_download_process "post-deploy"
    fi
    exit $ret
}

handle_pre_start() {
    tokenfile=/storage/software/caf/work/repo-lxc/lxc-data/$app_id/.token
    if [ -e $tokenfile ]; then
        echo " Deleting Token file $tokenfile for iox app $app_id"
        rm -rf $tokenfile
    fi
    exit 0
}

handle_post_start() {
    echo "Sending RUNNING msg to grpc for $appid"
    send_ipc grpc iox_app_status $appid RUNNING
    exit 0
}

handle_pre_stop() {
    exit 0
}

handle_post_stop() {
    echo "Sending STOPPED msg to grpc for $appid"
    send_ipc grpc iox_app_status $appid STOPPED
    exit 0
}

handle_pre_deactivate() {
    exit 0
}

handle_post_deactivate() {
    echo "Sending DEACTIVATED msg to grpc for $appid"
    send_ipc grpc iox_app_status $appid DEACTIVATED
    exit 0
}

handle_pre_uninstall() {
    cleanup_process
    exit 0
}

handle_post_uninstall() {
    echo "Sending UNDEPLOYED msg to grpc for $appid"
    send_ipc grpc iox_app_status $appid UNDEPLOYED
    exit 0
}

appid=$2
case "$1" in
    pre-deploy)
        # $2: <app_id>
        # $3: <app_repo_dir>
        # Call function here to handle PD pre-deploy action
        handle_pre_deploy
        ;;
    post-deploy)
        # $2: <app_id>
        # $3: <app_repo_dir>
        # $4: <app_extracted_dir>
        # Call function here to handle PD post-deploy action
        app_id=$2
        app_repo_dir="$CAF_REPO/$2"
        handle_post_deploy
        ;;
    pre-activate)
        # $2: <app_id>
        # Call function here to handle PD pre-activate action
        handle_pre_activate
        ;;
    post-activate)
        # $2: <app_id>
        # Call function here to handle PD post-activate action
        app_id=$2
        app_repo_dir="$CAF_REPO/$2"
        shift 3
        /opt/lxc/scripts/iox-app-port-setup.py activate $app_id "$1"
        handle_post_activate
        ;;
    pre-start)
        # $2: <app_id>
        # Call function here to handle PD pre-start action
        app_id=$2
        app_repo_dir="$CAF_REPO/$2"
        handle_pre_start
        ;;
    post-start)
        # $2: <app_id>
        # Call function here to handle PD post-start action
        handle_post_start
        ;;
    pre-stop)
        # $2: <app_id>
        # Call function here to handle PD pre-stop action
        handle_pre_stop
        ;;
    post-stop)
        # $2: <app_id>
        # Call function here to handle PD post-stop action
        handle_post_stop
        ;;
    pre-deactivate)
        # $2: <app_id>
        # $3: <rootfs_dir>
        # Call function here to handle PD pre-deactivate action
        handle_pre_deactivate
        ;;
    post-deactivate)
        # $2: <app_id>
        # Call function here to handle PD post-deactivate action
        app_id=$2
        app_repo_dir="$CAF_REPO/$2"
        /opt/lxc/scripts/iox-app-port-setup.py deactivate $app_id
        handle_post_deactivate
        ;;
    pre-uninstall)
        # $2: <app_id>
        # Call function here to handle PD pre-uninstall action
        app_id=$2
        app_repo_dir="$CAF_REPO/$2"
        handle_pre_uninstall
        ;;
    post-uninstall)
        # $2: <app_id>
        # Call function here to handle PD post-uninstall action
        app_id=$2
        app_repo_dir="$CAF_REPO/$2"
        handle_post_uninstall
        ;;
    *)
        # Do nothing
        exit 0
	    ;;
esac
exit 0
