#-----------------------------------------------------
#
# Copyright (c) 2012-2014 by cisco Systems, Inc.
# All rights reserved.
#-----------------------------------------------------
__author__ = 'havishwa'

import appfw.overrides.event
import concurrent.futures
import sys

from concurrent.futures._base import CancelledError, TimeoutError
from concurrent.futures._base import CANCELLED, PENDING, FINISHED, RUNNING, CANCELLED_AND_NOTIFIED
from concurrent.futures.thread import _WorkItem

class Future(concurrent.futures.Future):
    """
    Class exposing similar interfaces as concurrent.futures.Future.
    Notable differences are :
        - Uses overrides.even.Event for signalling mechanism
        - Doesn't support call backs on completion of asyn operations
        - Cannot be chained with other future objects to perform concurrent.futures.wait operation

    """

    def __init__(self):
        """Initializes the future. Should not be called by clients."""
        super(Future, self).__init__()

        # Change the signalling mechanism to custom event class
        self._condition = appfw.overrides.event.Event()


    def __del__(self):
        if self._condition:
            del self._condition
            self._condition = None

    def _invoke_callbacks(self):
        # Call backs not supported
        return

    def add_done_callback(self, fn):
        """
        This method is not supported.
        """
        raise NotImplementedError("Callbacks are not supported")


    def result(self, timeout=None):
        """Return the result of the call that the future represents.

        Args:
            timeout: The number of seconds to wait for the result if the future
                isn't done. If None, then there is no limit on the wait time.

        Returns:
            The result of the call that the future represents.

        Raises:
            CancelledError: If the future was cancelled.
            TimeoutError: If the future didn't finish executing before the given
                timeout.
            Exception: If the call raised then that exception will be raised.
        """
        with self._condition:
            if self._state in [CANCELLED, CANCELLED_AND_NOTIFIED]:
                raise CancelledError()
            elif self._state == FINISHED:
                return self.__get_result()

        # Wait is a blocking call on the custom event object. Do it outside the lock
        self._condition.wait(timeout)

        with self._condition:
            if self._state in [CANCELLED, CANCELLED_AND_NOTIFIED]:
                raise CancelledError()
            elif self._state == FINISHED:
                return self.__get_result()
            else:
                raise TimeoutError()

    def exception(self, timeout=None):
        """Return the exception raised by the call that the future represents.

        Args:
            timeout: The number of seconds to wait for the exception if the
                future isn't done. If None, then there is no limit on the wait
                time.

        Returns:
            The exception raised by the call that the future represents or None
            if the call completed without raising.

        Raises:
            CancelledError: If the future was cancelled.
            TimeoutError: If the future didn't finish executing before the given
                timeout.
        """

        with self._condition:
            if self._state in [CANCELLED, CANCELLED_AND_NOTIFIED]:
                raise CancelledError()
            elif self._state == FINISHED:
                return self._exception

        # Wait is a blocking call on the custom event object. Do it outside the lock
        self._condition.wait(timeout)

        with self._condition:
            if self._state in [CANCELLED, CANCELLED_AND_NOTIFIED]:
                raise CancelledError()
            elif self._state == FINISHED:
                return self._exception
            else:
                raise TimeoutError()


class ThreadPoolExecutor(concurrent.futures.ThreadPoolExecutor):
    """
    Overriding concurrent.futures.ThreadPoolExecutor
        - Use the custom futures object defined above in submit method
    """
    def __init__(self, max_workers):
        super(ThreadPoolExecutor, self).__init__(max_workers)


    def submit(self, fn, *args, **kwargs):
        with self._shutdown_lock:
            if self._shutdown:
                raise RuntimeError('cannot schedule new futures after shutdown')

            f = Future()
            w = _WorkItem(f, fn, args, kwargs)

            self._work_queue.put(w)
            self._adjust_thread_count()
            return f