#-----------------------------------------------------
#
# Copyright (c) 2012-2014 by cisco Systems, Inc.
# All rights reserved.
#-----------------------------------------------------
__author__ = 'havishwa'

import logging
from appfw.runtime.runtime import RuntimeService

log = logging.getLogger("runtime.hostingmanager")

class HostingManagerInterface(object):
    """
    Interface specifying apis for managing and interacting with CAF
    """
    def get_runtime_config(self):
        """Return the currently active run time config"""
        pass

    def start_infra(self):
        """
        Start CAF
        """
        pass

    def stop_infra(self, preserveConnectorState=True):
        """
        Stop CAF
        """
        pass

    def get_service(self, service_name):
        """Return service implementation if it exists/supported"""
        pass

    def list_services(self):
        """list all added services"""
        pass

class HostingManager(HostingManagerInterface):
    """
    api implementations
    """
    __singleton = None # the one, true Singleton
    __singleton_init_done = False

    def __new__(cls, *args, **kwargs):
        # Check to see if a __singleton exists already for this class
        # Compare class types instead of just looking for None so
        # that subclasses will create their own __singleton objects
        if cls != type(cls.__singleton):
        #if not cls.__singleton:
            cls.__singleton = super(HostingManager, cls).__new__(cls, *args, **kwargs)
        return cls.__singleton

    def __init__(self, system_config_file, log_config_file=None):
        self._runtime = None
        if not self.__singleton_init_done:
            self._runtime = RuntimeService(system_config_file, log_config_file)
            self.__singleton_init_done = True

    def get_runtime_config(self):
        """Return the currently active run time config"""
        if self._runtime:
            return self._runtime.bootstrapper.getConfig()
        return None

    def get_service(self, service_name):
        """Return service implementation if it exists/supported"""
        if self._runtime:
            rtx = self._runtime.get_runtime_context()
            return rtx.get_service(service_name)
        return None

    def start_infra(self):
        """
        Start CAF
        """
        try:
            if self._runtime:
                self._runtime.start()
        except Exception as ex:
            log.exception("Error while starting the services, cause : %s"%str(ex))
            raise

    def stop_infra(self, preserveConnectorState=True):
        """
        Stop CAF
        """
        try:
            if self._runtime:
                self._runtime.stop(preserveConnectorState)
        except Exception as ex:
            log.exception("Error while stopping the services, cause : %s"%str(ex))
            raise

    def reset(self):
        """
        Resets CAF 
        """
        if self._runtime:
            self._runtime.reset()

    def reload(self, app_restart=False):
        '''
        Reloads CAF
        '''
        if self._runtime:
            self._runtime.reload(app_restart=app_restart)

    def list_services(self):
        if self._runtime:
            rtx = self._runtime.get_runtime_context()
            return rtx.get_all_services()
        return None

    def set_service(self, name, service):
        if self._runtime:
            rtx = self._runtime.get_runtime_context()
            rtx.add_service(name, service)

    @classmethod
    def get_instance(cls):
        '''
        Returns a singleton instance of the class
        '''
        return cls.__singleton
