__author__ = 'hvishwanath'


import subprocess
import logging
import os
import pwd
import grp
import stat
import signal
import time
import pipes

log = logging.getLogger("command_wrapper")

def qemu_img(*args):
    """
    Execute qemu-img command with passed parameters. Return output,returncode.
    :param args:
    :return:
    """
    li = ['qemu-img']
    li.extend(args)

    try:
        log.debug("Executing command : %s" % " ".join(li))
        rval = subprocess.check_output(li, stderr=subprocess.STDOUT)
        return rval, 0
    except subprocess.CalledProcessError as c:
        return c.output, c.returncode
    except Exception as ex:
        log.error("Error executing command : %s" % " ".join(li))
        log.debug("Stack traceback",  exc_info=True)
        return str(ex), -1
    
def tar_create(archivename, filelist):
    """
    Execute tar create command with passed parameters. Return output,returncode.
    :param args:
    :return:
    """
    li = ['tar', '-cf']
    li.append(archivename)
    li.extend(filelist)

    try:
        log.debug("Executing command : %s" % " ".join(li))
        rval = subprocess.check_output(li, stderr=subprocess.STDOUT)
        return rval, 0
    except subprocess.CalledProcessError as c:
        return c.output, c.returncode
    except Exception as ex:
        log.error("Error executing command : %s" % " ".join(li))
        log.debug("Stack traceback",  exc_info=True)
        return str(ex), -1
        
def openssl(*args):
    """
    Execute openssl command with passed parameters. Return output,returncode.
    :param args:
    :return:
    """
    li = ['openssl']
    li.extend(args)

    try:
        log.debug("Executing command : %s" % " ".join(li))
        rval = subprocess.check_output(li, stderr=subprocess.STDOUT)
        return rval, 0
    except subprocess.CalledProcessError as c:
        return c.output, c.returncode
    except Exception as ex:
        log.error("Error executing command : %s" % " ".join(li))
        log.debug("Stack traceback",  exc_info=True)
        return str(ex), -1

def dropbearkey(*args):
    """
    Execute dropbearkey command with passed parameters. Return output,returncode.
    :param args:
    :return:
    """
    li = ['dropbearkey']
    li.extend(args)

    try:
        log.debug("Executing command : %s" % " ".join(li))
        rval = subprocess.check_output(li, stderr=subprocess.STDOUT)
        return rval, 0
    except subprocess.CalledProcessError as c:
        return c.output, c.returncode
    except Exception as ex:
        print ex
        log.exception("Error executing command : %s" % " ".join(li))
        return str(ex), -1

def dropbearconvert(*args):
    """
    Execute dropbearkey command with passed parameters. Return output,returncode.
    :param args:
    :return:
    """
    li = ['dropbearconvert']
    li.extend(args)

    try:
        log.debug("Executing command : %s" % " ".join(li))
        rval = subprocess.check_output(li, stderr=subprocess.STDOUT)
        return rval, 0
    except subprocess.CalledProcessError as c:
        return c.output, c.returncode
    except Exception as ex:
        print ex
        log.exception("Error executing command : %s" % " ".join(li))
        return str(ex), -1
        
def ssh_keygen(*args):
    """
    Execute ip command with passed parameters. Return output,returncode.
    :param args:
    :return:
    """
    li = ['ssh-keygen']
    li.extend(args)

    try:
        log.debug("Executing command : %s" % " ".join(li))
        rval = subprocess.check_output(li, stderr=subprocess.STDOUT)
        return rval, 0
    except subprocess.CalledProcessError as c:
        return c.output, c.returncode
    except Exception as ex:
        log.exception("Error executing command : %s" % " ".join(li))
        return str(ex), -1

def useradd(*args): #pragma: no cover
    """
    Execute ip command with passed parameters. Return output,returncode.
    :param args:
    :return:
    """
    li = ['useradd']
    li.extend(args)

    try:
        log.debug("Executing command : %s" % " ".join(li))
        rval = subprocess.check_output(li, stderr=subprocess.STDOUT)
        return rval, 0
    except subprocess.CalledProcessError as c:
        return c.output, c.returncode
    except Exception as ex:
        log.exception("Error executing command : %s" % " ".join(li))
        return str(ex), -1

def addgroup(*args): #pragma: no cover
    """
    Execute ip command with passed parameters. Return output,returncode.
    :param args:
    :return:
    """
    li = ['addgroup']
    li.extend(args)

    try:
        log.debug("Executing command : %s" % " ".join(li))
        rval = subprocess.check_output(li, stderr=subprocess.STDOUT)
        return rval, 0
    except subprocess.CalledProcessError as c:
        return c.output, c.returncode
    except Exception as ex:
        log.exception("Error executing command : %s" % " ".join(li))
        return str(ex), -1

def usermod(*args): #pragma: no cover
    """
    Execute ip command with passed parameters. Return output,returncode.
    :param args:
    :return:
    """
    li = ['usermod']
    li.extend(args)

    try:
        log.debug("Executing command : %s" % " ".join(li))
        rval = subprocess.check_output(li, stderr=subprocess.STDOUT)
        return rval, 0
    except subprocess.CalledProcessError as c:
        return c.output, c.returncode
    except Exception as ex:
        log.exception("Error executing command : %s" % " ".join(li))
        return str(ex), -1

def ipcmd(*args, **kwargs):
    """
    Execute ip command with passed parameters. Return output,returncode.
    Possible keyword argument in kwargs is netns=value. It specifies the 
    network namespace the command needs to run in. 
    If netns not specified or None, command runs normally.
    :param args:
    :param kwargs:
    :return:
    """
    li = ['ip']
    if kwargs and kwargs.get("netns") != None:
        li = ['ip', 'netns', 'exec', str(kwargs["netns"]), 'ip']
    
    li.extend(args)
    
    try:
        log.debug("Executing command : %s" % " ".join(li))
        rval = subprocess.check_output(li, stderr=subprocess.STDOUT)
        return rval, 0
    except subprocess.CalledProcessError as c:
        return c.output, c.returncode
    except Exception as ex:
        log.exception("Error executing command : %s" % " ".join(li))
        return str(ex), -1

def iptables(*args):
    """
    Execute iptables command with passed parameters. Return output,returncode.
    :param args:
    :return:
    """
    li = ['iptables']
    li.extend(args)

    try:
        log.debug("Executing command : %s" % " ".join(li))
        rval = subprocess.check_output(li, stderr=subprocess.STDOUT)
        return rval, 0
    except subprocess.CalledProcessError as c:
        return c.output, c.returncode
    except Exception as ex:
        log.exception("Error executing command : %s" % " ".join(li))
        return str(ex), -1

def ip6tables(*args):       
    """                    
    Execute iptables command with passed parameters. Return output,returncode.
    :param args:           
    :return:               
    """
    li = ['ip6tables']
    li.extend(args)
            
    try:        
        log.debug("Executing command : %s" % " ".join(li))
        rval = subprocess.check_output(li, stderr=subprocess.STDOUT)
        return rval, 0     
    except subprocess.CalledProcessError as c:
        return c.output, c.returncode
    except Exception as ex:
        log.exception("Error executing command : %s" % " ".join(li))
        return str(ex), -1 


def ifconfig(*args, **kwargs):
    """
    Execute ifconfig command with passed parameters. Return output,returncode.
    Possible keyword argument in kwargs is netns=value. It specifies the 
    network namespace the command needs to run in. 
    If netns not specified or None, command runs normally.
    :param args:
    :return:
    """
    li = ['ifconfig']
    if kwargs and kwargs.get("netns") != None:
        li = ['ip', 'netns', 'exec', str(kwargs["netns"]), 'ifconfig']

    li.extend(args)
    
    try:
        log.debug("Executing command : %s" % " ".join(li))
        rval = subprocess.check_output(li, stderr=subprocess.STDOUT)
        return rval, 0
    except subprocess.CalledProcessError as c:
        return c.output, c.returncode
    except Exception as ex:
        log.exception("Error executing command : %s" % " ".join(li))
        return str(ex), -1


def brctl(*args, **kwargs):
    """
    Execute brctl command with passed parameters. Return output,returncode.
    Possible keyword argument in kwargs is netns=value. It specifies the 
    network namespace the command needs to run in. 
    If netns not specified or None, command runs normally.
    :param args:
    :param kwargs:
    :return:
    """
    li = ['brctl']
    if kwargs and kwargs.get("netns") != None:
        li = ['ip', 'netns', 'exec', str(kwargs["netns"]), 'brctl']
        
    li.extend(args)
    
    try:
        log.debug("Executing command : %s" % " ".join(li))
        rval = subprocess.check_output(li, stderr=subprocess.STDOUT)
        return rval, 0
    except subprocess.CalledProcessError as c:
        return c.output, c.returncode
    except Exception as ex:
        log.exception("Error executing command : %s" % " ".join(li))
        return str(ex), -1


def netstat(*args): #pragma: no cover
    """
    Execute netstat command with passed parameters. Return output,returncode.
    :param args:
    :return:
    """
    li = ['netstat']
    li.extend(args)

    try:
        log.debug("Executing command : %s" % " ".join(li))
        rval = subprocess.check_output(li, stderr=subprocess.STDOUT)
        return rval, 0
    except subprocess.CalledProcessError as c:
        return c.output, c.returncode
    except Exception as ex:
        log.exception("Error executing command : %s" % " ".join(li))
        return str(ex), -1


def execmount(li):
    try:
        # Before mounting, run e2fsck and clean up errors if any
        # Mount command format: mount [-fnrsvw] [-t vfstype] [-o options] device dir

        if "--bind" not in li and "remount" not in li and "aufs" not in li:
            # Run e2fsck only if this is a non bind mount
            device = li[-2]
            log.debug("Running e2fsck on %s", device)
            out, rc = e2fsck((device,))
            if rc != 0 and rc != 1:
                if out.find("Superblock last write time") == -1 and out.find("is in the future") == -1:
                    raise ValueError("Error cleaning up disk(%s): %s" % (device, out))
                else:
                    log.error("Error: provided ext2 is in future:%s"%str(out))
            log.debug("Successfully ran e2fsck : %s", str(out))
        log.debug("Executing command : %s" % " ".join(li))
        rval = subprocess.check_output(li, stderr=subprocess.STDOUT)
        log.debug("Command Execution success")
        return rval, 0
    except subprocess.CalledProcessError as c:
        log.error("Command failed: %s:%s:%s" % (c.cmd, c.returncode, c.output))
        log.exception("Eroor while mounting ext2 :%s" % str(c))
        #Check if we failed bcoz of unavailablity of loop devices
        #then retry after creating them
        if c.returncode == 2 and "loop" in c.output:
            log.info("No loop devices available, will try to allocate more")
            # Retry after creating loop device
            start_no_dev = subprocess.check_output("losetup -a | wc -l", shell=True)
            start_no_dev =  int(start_no_dev)
            log.info("Total loop devices used: " + str(start_no_dev))
            end_no_dev = start_no_dev + 8 
            uid = pwd.getpwnam("root").pw_uid
            gid = grp.getgrnam("disk").gr_gid

            try:
                while start_no_dev < end_no_dev:
                    devname = "/dev/loop"+str(start_no_dev)
                    log.info("creating loop device %s" , devname)
                    os.mknod(devname, 0o600 | stat.S_IFBLK, 
                                os.makedev(7, start_no_dev))
                    start_no_dev += 1
                    os.chown(devname, uid,gid)
                log.debug("Executing command : %s" % " ".join(li))
                rval = subprocess.check_output(li, stderr=subprocess.STDOUT)
                return rval, 0
            except subprocess.CalledProcessError as c:
                return c.output, c.returncode
            except Exception as e:
                log.exception("Error while creating loop device : %s" , str(e))
                return str(e), -1
    except Exception as ex:
        log.exception("Error executing command : %s" % " ".join(li))
        return str(ex), -1

def mountext2(*args):
    """
    Execute mount -t ext2 command with passed parameters. Return output,returncode.
    :param args:
    :return:
    """
    li = ['mount', '-t', 'ext2', '-o', 'noatime,loop']
    li.extend(args)
    return execmount(li)

def  mountext4(*args):
    """
    Execute mount -t extd command with passed parameters. Return output,returncode.
    :param args:
    :return:
    """
    li = ['mount', '-t', 'ext4', '-o', 'noatime,loop']
    li.extend(args)
    return execmount(li)

def mount(arglist=None, *args):
    """
    Execute mount command with passed parameters. Return output,returncode.
    :param arglist: list of argument to paas to mount command
    :return:
    """
    fsck=True
    li = ['mount']
    if arglist:
        if "nofsck" in arglist:
            fsck=False
            arglist.remove("nofsck")
        li.extend(arglist)
    li.extend(args)
    try:
        # Before mounting, run e2fsck and clean up errors if any
        # Mount command format: mount [-fnrsvw] [-t vfstype] [-o options] device dir

        if fsck and "--bind" not in li and "remount" not in li and "aufs" not in li and "overlay" not in li and "vfat" not in li:
            # Run e2fsck only if this is a non bind mount
            device = li[-2]
            log.debug("Running e2fsck on %s", device)
            out, rc = e2fsck((device,))
            if rc != 0 and rc != 1:
                if out.find("Superblock last write time") == -1 and out.find("is in the future") == -1:
                    raise ValueError("Error cleaning up disk(%s): %s" % (device, out))
                else:
                    log.error("Error: provided ext2 is in future:%s"%str(out))

            log.debug("Successfully ran e2fsck : %s", str(out))
        log.debug("Executing command : %s" % " ".join(li))
        rval = subprocess.check_output(li, stderr=subprocess.STDOUT)
        return rval, 0
    except subprocess.CalledProcessError as c:
        log.error("Command failed: %s:%s:%s" % (c.cmd, c.returncode, c.output))
        log.exception("Eroor while mounting ext2 :%s" % str(c))
        #Check if we failed bcoz of unavailablity of loop devices
        #then retry after creating them
        if c.returncode == 2 and "loop" in c.output:
            log.info("No loop devices available, will try to allocate more")
            # Retry after creating loop device
            start_no_dev = subprocess.check_output("losetup -a | wc -l", shell=True)
            start_no_dev =  int(start_no_dev)
            log.info("Total loop devices used: " + str(start_no_dev))
            end_no_dev = start_no_dev + 8
            uid = pwd.getpwnam("root").pw_uid
            gid = grp.getgrnam("disk").gr_gid

            try:
                while start_no_dev < end_no_dev:
                    devname = "/dev/loop"+str(start_no_dev)
                    log.info("creating loop device %s" , devname)
                    os.mknod(devname, 0o600 | stat.S_IFBLK,
                                os.makedev(7, start_no_dev))
                    start_no_dev += 1
                    os.chown(devname, uid,gid)
                log.debug("Executing command : %s" % " ".join(li))
                rval = subprocess.check_output(li, stderr=subprocess.STDOUT)
                return rval, 0
            except subprocess.CalledProcessError as c:
                return c.output, c.returncode
            except Exception as e:
                log.exception("Error while creating loop device : %s" , str(e))
                return str(e), -1
    except Exception as ex:
        log.exception("Error executing command : %s" % " ".join(li))
        return str(ex), -1


def e2fsck(arglist=None, *args):
    """
    Execute fsck check and automatically correct if there are any errors. Return output,returncode.
    :param arglist: list of argument to paas to mount command
    :return:
    """

    # Return values of e2fsck
    # 0 - No Errors
    # 1 - File system corrected
    # 2 - System should be rebooted
    # 4 - File system errors left uncorrected
    # 8 - Operational error
    # 16 - Usage or syntax error
    # 32 - FSCK Cancelled by user requests
    # 128 - Shared library error.

    # Verify if we should run the disk check or not
    from appfw.utils.utils import Utils

    # Repair disks by default:
    repair = Utils.getSystemConfigValue("controller", "repair_disk_errors", default=True, parse_as="bool")
    if not repair:
        log.debug("Disk check is disabled!")
        return "disabled", 0

    li = ['e2fsck', '-f', '-p']
    if arglist:
        li.extend(arglist)
    li.extend(args)

    try:
        log.debug("Executing command : %s" % " ".join(li))
        rval = subprocess.check_output(li, stderr=subprocess.STDOUT)
        return rval, 0
    except subprocess.CalledProcessError as c:
        log.warning("e2fsck could not able to repair errors automatically: %s" % c.output) 
        log.info("e2fsck will try to repair errors manually now.") 
        if c.returncode == 4:
            #Errors were not repaired automatically needs to e2fsck manually
            #https://bbs.archlinux.org/viewtopic.php?id=174133 
            #http://serverfault.com/questions/366863/huge-amounts-of-multiply-claimed-blocks-during-fsck 
            li = ['e2fsck', '-y']
            if arglist:
                li.extend(arglist)
            li.extend(args)
            try:
                log.debug("Executing command : %s" % " ".join(li))
                rval = subprocess.check_output(li, stderr=subprocess.STDOUT)
                return rval, 0
            except subprocess.CalledProcessError as c:
                return c.output, c.returncode
            except Exception as ex:
                log.exception("Error executing command : %s" % " ".join(li))
                return str(ex), -1
        else: 
            return c.output, c.returncode
    except Exception as ex:
        log.exception("Error executing command : %s" % " ".join(li))
        return str(ex), -1


def mount_point(mount_path):
    """
    Returns the mount point of the given path
    """
    cmd = ("df %s | tail -n 1 | awk '{ print $1 }'" % mount_path)
    log.debug("Executing: %s" % cmd)
    try:
        loop_dev = subprocess.check_output(cmd, shell=True)
        return loop_dev, 0
    except subprocess.CalledProcessError as c:
        return c.output, c.returncode
    except Exception as ex:
        log.exception("Error executing command : %s" % cmd)
        return str(ex), -1

def delete_loop_device(loop_dev):
    """
    Delete the loop device
    """
    if loop_dev:
        cmd = "losetup %s" % loop_dev
        log.debug("Executing: %s" % cmd)
        try:
            ro = subprocess.check_output(cmd, shell=True)
            if "loop" in ro:
                try:
                    cmd = "losetup -d %s" % loop_dev
                    log.debug("Executing: %s" % cmd)
                    subprocess.check_output(cmd, shell=True)
                except subprocess.CalledProcessError as c:
                    log.error("Failed to delete: %s" % (loop_dev, c.output))
                    return c.output, c.returncode
                except Exception as ex:
                    log.exception("Error executing command : %s" % cmd)
                    return str(ex), -1
            return None, 0
        except subprocess.CalledProcessError as c:
            log.debug("Loop device %s already deleted:%s" % (loop_dev, c.output)) 
            return None, 0 
        except Exception as ex:
            log.exception("Error executing command : %s" % cmd)
            return str(ex), -1
            
def umount(*args):
    """
    Execute umount command with passed parameters. Return output,returncode.
    :param args:
    :return:
    """
    li = ['umount']
    li.extend(args)
    mount_path = li[-1]
    log.debug("Path to be umounted: %s\n" % mount_path)
    loop_dev, rv = mount_point(mount_path)
    if rv != 0:
        log.error("Failed to get attached mount point for path: %s Error: %s" % (mount_path, loop_dev))
        return rv
    log.debug("Associated mount point:%s" % loop_dev)
    try:
        log.debug("Executing command : %s" % " ".join(li))
        rval = subprocess.check_output(li, stderr=subprocess.STDOUT)
        #Verify that associated loop device should not exists"
        if "loop" in loop_dev:
            ro, rv = delete_loop_device(loop_dev)    
            if rv != 0:
                log.error("Failed to delete loop device: %s Error: %s" % (loop_dev, ro))
                # Just log the error do not return error so caller proceeds
        return rval, 0
    except subprocess.CalledProcessError as c:
        return c.output, c.returncode
    except Exception as ex:
        log.exception("Error executing command : %s" % " ".join(li))
        return str(ex), -1

def dd(arglist=None, *args):
    """
    Execute dd command with passed parameters. Return output,returncode.
    :param arglist: list of argument to pass to dd command
    :return:
    """
    li = ['dd']
    if arglist:
        li.extend(arglist)
    li.extend(args)
    
    try:
        log.debug("Executing command : %s" % " ".join(li))
        rval = subprocess.check_output(li, stderr=subprocess.STDOUT)
        return rval, 0
    except subprocess.CalledProcessError as c:
        return c.output, c.returncode
    except Exception as ex:
        log.exception("Error executing command : %s" % " ".join(li))
        return str(ex), -1

def fallocate(arglist=None, *args):
    """
    Execute dd command with passed parameters. Return output,returncode.
    :param arglist: list of argument to pass to dd command
    :return:
    """
    li = ['fallocate']
    if arglist:
        li.extend(arglist)
    li.extend(args)

    try:
        log.debug("Executing command : %s" % " ".join(li))
        rval = subprocess.check_output(li, stderr=subprocess.STDOUT)
        return rval, 0
    except subprocess.CalledProcessError as c:
        return c.output, c.returncode
    except Exception as ex:
        log.exception("Error executing command : %s" % " ".join(li))
        return str(ex), -1


def mkfs_ext2(arglist=None, *args):
    """
    Execute mkfs.ext2 command with passed parameters. Return output,returncode.
    :param arglist: list of argument to pass to mkfs.ext2 command
    :return:
    """
    li = ['mkfs.ext2']
    if arglist:
        li.extend(arglist)
    li.extend(args)
    
    try:
        log.debug("Executing command : %s" % " ".join(li))
        rval = subprocess.check_output(li, stderr=subprocess.STDOUT)
        return rval, 0
    except subprocess.CalledProcessError as c:
        log.error("Command failed: %s:%s:%s" % (c.cmd, c.returncode, c.output))
        #log.exception("Error while creating ext2 :%s" % str(c))
        #Check if we failed bcoz -E nodiscard option not available 
        #then retry without using it 
        if c.returncode == 1 and ("invalid option" in c.output or "Bad option" in c.output):
            if "-E" in li:
                li.remove("-E")      
            if "nodiscard" in li:
                li.remove("nodiscard")
        try:
            log.debug("Executing command : %s" % " ".join(li))
            rval = subprocess.check_output(li, stderr=subprocess.STDOUT)
            return rval, 0
        except subprocess.CalledProcessError as c:
            return c.output, c.returncode
        except Exception as e:
            log.exception("Error while creating mkfs.ext2 : %s" , str(e))
            return str(e), -1
    except Exception as ex:
        log.exception("Error executing command : %s" % " ".join(li))
        return str(ex), -1

def mkfs_ext4(arglist=None, *args):
    """
    Execute mkfs.ext4 command with passed parameters. Return output,returncode.
    :param arglist: list of argument to pass to mkfs.ext2 command
    :return:
    """
    li = ['mkfs.ext4']
    if arglist:
        li.extend(arglist)
    li.extend(args)
    
    try:
        log.debug("Executing command : %s" % " ".join(li))
        rval = subprocess.check_output(li, stderr=subprocess.STDOUT)
        return rval, 0
    except subprocess.CalledProcessError as c:
        log.error("Command failed: %s:%s:%s" % (c.cmd, c.returncode, c.output))
        #log.exception("Error while creating ext2 :%s" % str(c))
        #Check if we failed bcoz -E nodiscard option not available 
        #then retry without using it 
        if c.returncode == 1 and ("invalid option" in c.output or "Bad option" in c.output):
            if "-E" in li:
                li.remove("-E")      
            if "nodiscard" in li:
                li.remove("nodiscard")
        try:
            log.debug("Executing command : %s" % " ".join(li))
            rval = subprocess.check_output(li, stderr=subprocess.STDOUT)
            return rval, 0
        except subprocess.CalledProcessError as c:
            return c.output, c.returncode
        except Exception as e:
            log.exception("Error while creating mkfs.ext4 : %s" , str(e))
            return str(e), -1
    except Exception as ex:
        log.exception("Error executing command : %s" % " ".join(li))
        return str(ex), -1

def mke2fs(arglist=None, *args):
    """
    Execute mke2fs command with passed parameters. Return output,returncode.
    :param arglist: list of argument to pass to mke2fs command
    :return:
    """
    li = ['mke2fs']
    if arglist:
        li.extend(arglist)
    li.extend(args)
    
    try:
        log.debug("Executing command : %s" % " ".join(li))
        rval = subprocess.check_output(li, stderr=subprocess.STDOUT)
        return rval, 0
    except subprocess.CalledProcessError as c:
        log.error("Command failed: %s:%s:%s" % (c.cmd, c.returncode, c.output))
        return c.output, c.returncode
    except Exception as ex:
        log.exception("Error executing command : %s" % " ".join(li))
        return str(ex), -1

def resize2fs(target, size):
    """
    Execute resize2fs command with passed parameters. Return output,returncode.
    :path - target rootfs file 
    :size - capacity to resize to
    :return:
    """
    li = ['resize2fs']
    li.extend([target])
    li.extend([size])
    try:
        log.debug("Executing command : %s", li)
        rval = subprocess.check_output(li, stderr=subprocess.STDOUT)
        return rval, 0
    except subprocess.CalledProcessError as c:
        log.error("Command failed: %s:%s:%s" % (c.cmd, c.returncode, c.output))
        # Check if error for e2fsck check
	if c.returncode !=0 and c.output.find("Please run \'e2fsck") != -1:
           try:
              device = li[-2]
              log.debug("Running e2fsck on %s", device)
              out, rc = e2fsck((device,))
              if rc != 0 and rc != 1:
                if out.find("Superblock last write time") == -1 and out.find("is in the future") == -1:
                    raise ValueError("Error cleaning up disk(%s): %s" % (device, out))
                else:
                    log.error("Error: provided ext2 is in future:%s"%str(out))
              else:
                log.debug("Successfully ran e2fsck : %s", str(out))
                log.debug("Executing command : %s", li)
                rval = subprocess.check_output(li, stderr=subprocess.STDOUT)
                return rval, 0
           except subprocess.CalledProcessError as c:
                log.error("Error: Command Failed: %s" % c.output)
                return c.output, c.returncode
           except Exception as ex:
                log.exception("Error executing command : %s" % " ".join(li))
                return str(ex), -1
        else:
           log.error("Error: Command Failed: %s" % c.output)
           return c.output, c.returncode
    except Exception as ex:
        log.exception("Error executing command : %s" % " ".join(li))
        return str(ex), -1

def tune2fs(arglist=None, *args):
    """
    Execute virsh command with passed parameters. Return output,returncode.
    :param arglist: list of argument to pass to mkfs.ext2 command
    :return:
    """
    li = ['tune2fs']
    if arglist:
        li.extend(arglist)
    li.extend(args)

    try:
        log.debug("Executing command : %s" % " ".join(li))
        rval = subprocess.check_output(li, stderr=subprocess.STDOUT)
        return rval, 0
    except subprocess.CalledProcessError as c:
        return c.output, c.returncode
    except Exception as ex:
        log.exception("Error executing command : %s" % " ".join(li))
        return str(ex), -1

def virsh(*args):
    """
    Execute virsh command with passed parameters. Return output,returncode.
    :param arglist: list of argument to pass to mkfs.ext2 command
    :return:
    """
    li = ['virsh']
    li.extend(args)
    
    try:
        log.debug("Executing command : %s" % " ".join(li))
        rval = subprocess.check_output(li, stderr=subprocess.STDOUT)
        return rval, 0
    except subprocess.CalledProcessError as c:
        return c.output, c.returncode
    except Exception as ex:
        log.exception("Error executing command : %s" % " ".join(li))
        return str(ex), -1


def adduser(*args):
    """
    Execute virsh command with passed parameters. Return output,returncode.
    :param arglist: list of argument to pass to mkfs.ext2 command
    :return:
    """
    li = ['adduser']
    li.extend(args)
    
    try:
        log.debug("Executing command : %s" % " ".join(li))
        rval = subprocess.check_output(li, stderr=subprocess.STDOUT)
        return rval, 0
    except subprocess.CalledProcessError as c:
        return c.output, c.returncode
    except Exception as ex:
        log.exception("Error executing command : %s" % " ".join(li))
        return str(ex), -1

def deluser(*args):
    """
    Execute virsh command with passed parameters. Return output,returncode.
    :param arglist: list of argument to pass to mkfs.ext2 command
    :return:
    """
    li = ['deluser']
    li.extend(args)
    
    try:
        log.debug("Executing command : %s" % " ".join(li))
        rval = subprocess.check_output(li, stderr=subprocess.STDOUT)
        return rval, 0
    except subprocess.CalledProcessError as c:
        return c.output, c.returncode
    except Exception as ex:
        log.exception("Error executing command : %s" % " ".join(li))
        return str(ex), -1

def arp(*args): #pragma: no cover
    """
    the arp command gives the ouptut for the associated interface
    to derive the ipv4 from the MAC id.
    """
    li = ['arp']
    li.extend(args)

    try:
        log.debug("Executing command : %s" % " ".join(li))
        rval = subprocess.check_output(li, stderr=subprocess.STDOUT)
        return rval, 0
    except subprocess.CalledProcessError as c:
        return c.output, c.returncode
    except Exception as ex:
        log.exception("Error executing command : %s" % " ".join(li))
        return str(ex), -1
        
def call_script_silent(script, *args):
    """
    Calls the given script, doesn't log any errors or exceptions
    """
    if type(script) is list:
        li = script
    else:
        # string
        li = [script]
    li.extend(args)

    try:
        rval = subprocess.check_output(li, stderr=subprocess.STDOUT)
        return rval, 0
    except subprocess.CalledProcessError as c:
        return c.output, c.returncode
    except Exception as ex:
        return str(ex), -1

def restore_signals():
    signals = ('SIGPIPE', 'SIGXFZ', 'SIGXFSZ')
    for sig in signals:
        if hasattr(signal, sig):
            signal.signal(getattr(signal, sig), signal.SIG_DFL)

def call_script(script, *args, **env):
    """
    Calls script with environment variables defined in env (if any) 
    and returns output,returncode.
    """
    
    log.debug("call_script: cmd:%s args:%s env:%s" % (script, args, env))
    if type(script) is list:
        li = script
    else:
        # string
        li = [script]
    li.extend(args)
    
    esc_li = [] 
    for arg in li:
        arg=pipes.quote(arg)
        esc_li.append(arg)
    log.debug("AFTER exec_cmd: %s" % esc_li)
    li = esc_li


    if env:
        li = ' '.join(li)
        env_str = ''
        for key, value in env.items():
            env_str += pipes.quote(str(key)) + '=' + pipes.quote(str(value)) + ' '
        li = env_str + li
        
    try:
        if env:
            log.debug("Executing script : %s" % li)
            rval = subprocess.check_output(li, stderr=subprocess.STDOUT, shell=True, preexec_fn=restore_signals)
        else:
            log.debug("Executing script : %s" % " ".join(li))
            rval = subprocess.check_output(li, stderr=subprocess.STDOUT, preexec_fn=restore_signals)
        return rval, 0
    except subprocess.CalledProcessError as c:
        return c.output, c.returncode
    except Exception as ex:
        log.exception("Error executing script : %s" % " ".join(li))
        return str(ex), -1

def cat(*args, **kwargs):
    """
    Execute cat command with passed parameters. Return output,returncode.
    Possible keyword argument in kwargs is netns=value. It specifies the 
    network namespace the command needs to run in. 
    If netns not specified or None, command runs normally.
    :param arglist: list of argument to pass to cat command
    :return:
    """                 
    li = ['cat']
    if kwargs and kwargs.get("netns") != None:
        li = ['ip', 'netns', 'exec', str(kwargs["netns"]), 'cat']

    li.extend(args) 

    try:
        log.debug("Executing command : %s" % " ".join(li))
        rval = subprocess.check_output(li, stderr=subprocess.STDOUT)
        return rval, 0
    except subprocess.CalledProcessError as c:
        return c.output, c.returncode
    except Exception as ex:
        log.exception("Error executing command : %s" % " ".join(li))
        return str(ex), -1

def nsenter(*args):
    """
    Execute nsenter command with passed parameters. Return output,returncode.
    :param arglist: list of argument to pass to nsenter command
    :return:
    """
    li = ['nsenter']
    li.extend(args)

    try:
        log.debug("Executing command : %s" % " ".join(li))
        rval = subprocess.check_output(li, stderr=subprocess.STDOUT)
        return rval, 0
    except subprocess.CalledProcessError as c:
        return c.output, c.returncode
    except Exception as ex:
        log.exception("Error executing command : %s" % " ".join(li))
        return str(ex), -1

def pgrep(*args):
    """
    Execute pgrep command with passed parameters. Return output,returncode.
    :param arglist: list of argument to pass to nsenter command
    :return:
    """
    li = ['pgrep']
    li.extend(args)

    try:
        log.debug("Executing command : %s" % " ".join(li))
        rval = subprocess.check_output(li, stderr=subprocess.STDOUT)
        return rval, 0
    except subprocess.CalledProcessError as c:
        return c.output, c.returncode
    except Exception as ex:
        log.exception("Error executing command : %s" % " ".join(li))
        return str(ex), -1

def which(program):
    """
    Check the given programme is present and executable in the system.
     If command found returns command else returns None
    """
    fpath, fname = os.path.split(program)
    if fpath:
        if os.path.isfile(program) and os.access(program, os.X_OK):
            return program
    else:
        if os.environ.get("PATH") is None:
            return None
        for path in os.environ["PATH"].split(os.pathsep):
            path = path.strip('"')
            exe_file = os.path.join(path, program)
            if os.path.isfile(exe_file) and os.access(exe_file, os.X_OK):
                return exe_file
    return None

def chsmack(*args):
    """
    Perform Smack-related operations including identifying, adding or removing
    security labels on files. The filesystem being operated on must support
    extended attributes.
    """
    li = ['chsmack']
    li.extend(args)

    # If remove flag is found, remove label from arg list to prevent error
    if "-d" in li:
        if "sec_attr" in li:
            li.remove(sec_attr)
    try:
        #log.debug("Executing command : %s" % " ".join(li))
        rval = subprocess.check_output(li, stderr=subprocess.STDOUT)
        return rval, 0
    except Exception as ex:
        log.exception("Error executing command : %s" % " ".join(li))
        return str(ex), -1

def chown(*args):
    """
    Change standard UNIX ownership on files and folders.
    """
    li = ['chown']
    li.extend(args)

    try:
        log.debug("Executing command : %s" % " ".join(li))
        rval = subprocess.check_output(li, stderr=subprocess.STDOUT)
        return rval, 0
    except Exception as ex:
        log.exception("Error executing command : %s" % " ".join(li))
        return str(ex), -1

def kill(pid, wait_time=30):
    """
    @param pid:
    @param wait_time:
    @return:
    """
    log.debug("Killing the pid %s ,with wait time %s"%(pid, wait_time))
    if os.path.isdir(os.path.join("/proc", pid)):
        os.kill(int(pid), signal.SIGTERM)
        # Wait for 20sec till pid is killed
        count = wait_time
        while True:
            if os.path.isdir(os.path.join("/proc", pid)) and count == 0:
                log.info("Sigterm did not kill the process %s, so passing sigkill" % pid)
                os.kill(int(pid), signal.SIGKILL)
                # wait for couple of seconds for kill to take effect
                time.sleep(2)
                if os.path.isdir(os.path.join("/proc", pid)):
                    log.error("Unable to kill the pid %s"%pid)
                    return False
                break
            elif not os.path.isdir(os.path.join("/proc", pid)):
                break
            time.sleep(1)
            count -= 1
    else:
        log.info("Pid provided %s is not running!"%pid)
    return True


def chk_qemuimg_type(*args):
    """
    Execute virsh command with passed parameters
    :param arglist: full path of the image to check 
    :return: type of image 
    """
    li = ['qemu-img']
    li.extend(['info'])
    li.extend(args)
    
    try:
        log.debug("Executing command : %s" % " ".join(li))
        fformat = 'raw'
        finfo = subprocess.check_output(li, stderr=subprocess.STDOUT)
        fmtstr = finfo.split("format: ", 1)[1]
        if 'qcow2' in fmtstr:
            fformat = 'qcow2' 
        elif 'qcow' in fmtstr:
            fformat = 'qcow'
        return fformat, 0
    except Exception as ex:
        log.exception("Error executing command : %s" % " ".join(li))
        return str(ex), -1

def create_cgroup(group_name, controllers, user=None):
    """
    With the given parameters this method will create the cgroup needed.
    :param group_name: Group name with which Cgroup needs be created
    :param controllers: List of the Cgroup controllers
    :param user: If user is given then Cgroup will be created to bind the user
    :return:
    """
    if which("cgcreate"):
        log.debug("Creating the cgroup with the group name %s"%group_name)
        li = ["cgcreate"]
        if user:
            li.extend(["-a", user])
        li.extend(["-g", ",".join(controllers) + ":" + group_name])
        try:
            log.debug("Executing command : %s" % " ".join(li))
            rval = subprocess.check_output(li, stderr=subprocess.STDOUT)
            return rval, 0
        except Exception as ex:
            log.exception("Error executing command : %s" % " ".join(li))
            return str(ex), -1
    else:
        log.error("Required binary '%s' is not found!"%"cgcreate")
        return "Required binary '%s' is not found!"%"cgcreate", -1


def set_cgroup_control_limit(group_name, controller, control_key, val):
    """
    This method typically sets the limit to cgroup controller,
     in order to make the any processes pin to this group should be bound by this limit.
    """
    if which("cgset"):
        log.debug("Setting the limit for cgroup %s controller %s, attribute %s, value %s"%(group_name, controller, control_key, val))
        li = ["cgset", "-r"]
        li.extend([controller + "." + control_key + "=" + str(val), group_name])
        try:
            log.debug("Executing command : %s" % " ".join(li))
            rval = subprocess.check_output(li, stderr=subprocess.STDOUT)
            return rval, 0
        except Exception as ex:
            log.exception("Error executing command : %s" % " ".join(li))
            return str(ex), -1
    else:
        log.error("Required binary '%s' is not found!" % "cgset")
        return "Required binary '%s' is not found!" % "cgset", -1


def delete_cgroup(group_name, controllers):
    """
    Will delete the cgroup created.
    """
    if which("cgdelete"):
        log.debug("Deleting the cgroup %s"%group_name)
        li = ["cgdelete"]
        li.extend(["-g", ",".join(controllers) + ":" + group_name])
        try:
            log.debug("Executing command : %s" % " ".join(li))
            rval = subprocess.check_output(li, stderr=subprocess.STDOUT)
            return rval, 0
        except Exception as ex:
            log.exception("Error executing command : %s" % " ".join(li))
            return str(ex), -1
    else:
        log.error("Required binary '%s' is not found!"%"cgdelete")
        return "Required binary '%s' is not found!"%"cgdelete", -1

def cgroup_exists(group_name, controllers):
    """
    Will delete the cgroup created.
    """
    if which("lscgroup"):
        log.debug("Deleting the cgroup %s"%group_name)
        li = ["lscgroup"]
        li.extend(["-g", ",".join(controllers) + ":" + group_name])
        try:
            log.debug("Executing command : %s" % " ".join(li))
            rval = subprocess.check_output(li, stderr=subprocess.STDOUT)
            if rval.strip() == "":
                return rval, -1
            return rval, 0
        except Exception as ex:
            log.exception("Error executing command : %s" % " ".join(li))
            return str(ex), -1
    else:
        log.error("Required binary '%s' is not found!"%"lscgroup")
        return "Required binary '%s' is not found!"%"lscgroup", -1

def chroot_cmd(new_root, cmd, *args):
    """
    Perform chroot operations prior to executing the command.
    subprocess.CalledProcessError exception will be thrown if 
    chroot or command executed under chroot encounters error.
    """
    li = ['chroot']
    li.extend([new_root, cmd])
    li.extend(args)

    rval = subprocess.check_output(li, stderr=subprocess.STDOUT)
    return rval, 0

def tar_extractall(tar_file, destination):
    """
    Will extract the tar contents to the specified location.
    """
    if which("tar"):
        log.debug("Extracting the tar %s the destination %s"%(tar_file, destination))
        li = ["tar"]
        #li.extend(["--no-same-owner", "-xvpf", tar_file, "-C", destination])
        li.extend(["-xvpf", tar_file, "-C", destination])
        try:
            if not os.path.exists(destination):
                os.makedirs(destination)
            log.debug("Executing command : %s" % " ".join(li))
            rval = subprocess.check_output(li, stderr=subprocess.STDOUT)
            if rval.strip() == "":
                return rval, 0 
            log.debug("successfully extracted the tar file %s"%tar_file)
            return rval, 0
        except Exception as ex:
            log.exception("Error executing command : %s" % " ".join(li))
            return str(ex), -1
    else:
        log.error("Required binary '%s' is not found!"%"tar")
        return "Required binary '%s' is not found!"%"tar", -1

def tar_extractall_with_exclude(tar_file, destination, excludelist):
    """
    Will extract the tar contents to the specified location excluding given files.
    """
    if which("tar"):
        log.debug("Extracting the tar %s the destination %s"%(tar_file, destination))
        li = ["tar"]
        li.extend(["-xvpf", tar_file, "-C", destination])
        for item in excludelist:
            li.append("--exclude="+item)
        try:
            if not os.path.exists(destination):
                os.makedirs(destination)
            log.debug("Executing command : %s" % " ".join(li))
            rval = subprocess.check_output(li, stderr=subprocess.STDOUT)
            if rval.strip() == "":
                return rval, 0
            log.debug("successfully extracted the tar file %s"%tar_file)
            return rval, 0
        except Exception as ex:
            log.exception("Error executing command : %s" % " ".join(li))
            return str(ex), -1
    else:
        log.error("Required binary '%s' is not found!"%"tar")
        return "Required binary '%s' is not found!"%"tar", -1

def tar_extract_nested_tar_artifiact(tar_file, nested_artifact, destination, compression_type):
    """
    Will extract only the contents of nested artifact contents (like artifact.tar.gz) directly without
    extracting the anyother other outer files.
    """
    if which("tar"):
        log.debug("Extracting the nested artifact %s to the destination %s"%(tar_file, destination))
        cmd1 = ["tar"]
        cmd1.extend(["-xOvpf", tar_file, nested_artifact])

        cmd2 = ["tar"]
        if compression_type == "gz":
            cmd2.append("-zxvpf")
        elif compression_type == "xz":
            cmd2.append("-xJvpf")
        else:
            return "Unknown compression type", -1

        cmd2.extend(["-", "-C", destination])
        try:
            if not os.path.exists(destination):
                os.makedirs(destination)
            log.debug("Executing command : %s, piping to stdoout" % " ".join(cmd1))
            stdout_stream = subprocess.Popen(cmd1, stdout=subprocess.PIPE)
            log.debug("Executing command : %s, reading from piped stdin" % " ".join(cmd1))
            rval = subprocess.check_output(cmd2, stdin=stdout_stream.stdout)
            if rval.strip() == "":
                return rval, 0
            log.debug("successfully extracted the nested tar file %s"%nested_artifact)
            return rval, 0
        except Exception as ex:
            log.exception("Error executing commands: %s \n %s" % (" ".join(cmd1), " ".join(cmd2)))
            if stdout_stream.poll() is None:
                stdout_stream.kill()
            return str(ex), -1
    else:
        log.error("Required binary '%s' is not found!"%"tar")
        return "Required binary '%s' is not found!"%"tar", -1


def grep(*args): #pragma: no cover
    """
    Execute grep command with passed parameters. Return output,returncode.
    :param args:
    :return:
    """
    li = ['grep']
    li.extend(args)
    log.debug("Executing command : %s" % " ".join(li))
    rval = subprocess.check_output(li, stderr=subprocess.STDOUT)
    return rval, 0

def vgs():
    """
    LVM: Check volume group free size 
    """
    li = "vgs --noheadings --options \"vg_free\" --units m" 
    try:
        log.debug("Executing command : %s" % " ".join(li))
        rval = subprocess.check_output(li, shell=True)
        end = rval.find('.')
        rval = int(rval[:end])
        log.debug("Free Volume Group size: %dMB ", rval)
        return rval
    except subprocess.CalledProcessError as c:
        return c.output, c.returncode
    except Exception as ex:
        log.exception("Error executing command : %s" % " ".join(li))
        return str(ex), -1

def lvs():
    """
    LVM: Show all logical volumes  
    """
    li = ['lvs']
    log.debug("Executing command : %s" % " ".join(li))
    rval = subprocess.check_output(li, stderr=subprocess.STDOUT)
    return rval, 0

def lvcreate(name, size, volgrp):
    """
    LVM: Create logical volume
    """
    li = ['lvcreate']
    li.extend(["--name", name])
    li.extend(["--size", size])
    li.extend([volgrp])
    try:
        log.debug("Executing command : %s" % " ".join(li))
        rval = subprocess.check_output(li, stderr=subprocess.STDOUT)
        return rval, 0
    except subprocess.CalledProcessError as c:
        return c.output, c.returncode
    except Exception as ex:
        log.exception("Error executing command : %s" % " ".join(li))
        return str(ex), -1

def lvremove(name, volgrp):
    """
    LVM: Remove logical volume
    """
    lvs()
    li = ['lvremove']
    li.extend(["--force"])
    path = volgrp + '/' + name
    li.extend([path])
    log.debug("Executing command : %s" % " ".join(li))
    rval = subprocess.check_output(li, stderr=subprocess.STDOUT)
    return rval, 0


def grep_pipe(*args):
    """
    Execute wc command with passed parameters. Return output,returncode.
    :param args: All grep command sub commands needs to be passed as list.
    :return: result/exception , return code
    """
    li = ['grep']
    li.extend(args)
    #ignoring arguments in the debug statement to ensure search terms like
    #'Error' don't show up when caf log is grepped for Errors.
    log.debug("Executing grep command")
    try:
        p = subprocess.Popen(li, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
        result, err = p.communicate()
        if p.returncode != 0:
            return err, p.returncode
        else:
            return result, 0
    except Exception as ex:
        log.exception("Error executing command : %s" % " ".join(li))
        return ex.message, -1

def wc(*args):
    """
    Execute wc command with passed parameters. Return output,returncode.
    :param args: All wc command sub commands needs to be passed as list.
    :return: result/exception , return code
    """
    li = ['wc']
    li.extend(args)
    try:
        log.debug("Executing command : %s" % " ".join(li))
        rval = subprocess.check_output(li, stderr=subprocess.STDOUT)
        return rval, 0
    except subprocess.CalledProcessError as c:
        return c.output, c.returncode
    except Exception as ex:
        log.exception("Error executing command : %s" % " ".join(li))
        return str(ex), -1


def file_command(*args):
    """
    Execute file command with passed parameters. Return output,returncode.
    """
    li = ['file']
    li.extend(args)
    try:
        log.debug("Executing command : %s" % " ".join(li))
        rval = subprocess.check_output(li, stderr=subprocess.STDOUT)
        return rval, 0
    except subprocess.CalledProcessError as c:
        return c.output, c.returncode
    except Exception as ex:
        log.exception("Error executing command : %s" % " ".join(li))
        return str(ex), -1


def cp(*args):
    """
    Execute cp command with passed parameters. Return output,returncode.
    """
    li = ['cp']
    li.extend(args)
    try:
        log.debug("Executing command : %s" % " ".join(li))
        rval = subprocess.check_output(li, stderr=subprocess.STDOUT)
        return rval, 0
    except subprocess.CalledProcessError as c:
        if c.output.find("No space left on device") != -1:
            raise c
        return c.output, c.returncode
    except Exception as ex:
        log.exception("Error executing command : %s" % " ".join(li))
        return str(ex), -1

def du(*args):
    """
    Execute cp command with passed parameters. Return output,returncode.
    """
    li = ['du']
    li.extend(args)
    try:
        log.debug("Executing command : %s" % " ".join(li))
        rval = subprocess.check_output(li, stderr=subprocess.STDOUT)
        return rval, 0
    except subprocess.CalledProcessError as c:
        return c.output, c.returncode
    except Exception as ex:
        log.exception("Error executing command : %s" % " ".join(li))
        return str(ex), -1

