#!/usr/bin/env python
#------------------------------------------------------------------
# generate_yang_path_mapping.py -
#
# August 2018
#
#------------------------------------------------------------------

import sys
import argparse
import subprocess
import os
import shlex
import re

def _get_common_longest_match(str1, str2, yang_path, debug):
    common_prefix = ''
    c_len = 0
    str1_nodes = str1.split('/')
    str2_nodes = str2.split('/')
    while str1_nodes[c_len] is '.*' or str1_nodes[c_len] == str2_nodes[c_len]:
        common_prefix += '/' + str1_nodes[c_len]
        c_len += 1
    match = c_len*100/len(str1_nodes)
    if match > 50:
        if debug:
            print("Matched {}% to {} {} {}".format(match, str1, str2, yang_path))
        else:
            print("Matched {}% to {}".format(match, yang_path))


def _run_process(cmd, log, env=None, input=None):
    if env is not None:
        env.update(os.environ.copy())
    DEVNULL = open(os.devnull, 'wb')
    proc = subprocess.Popen(shlex.split(cmd),
                            env=env,
                            stdout=DEVNULL,
                            stderr=subprocess.PIPE)
    # No need for timeout as it is known which processes this spawns (and they
    # don't expect STDIN input)
    (stdout, stderr) = proc.communicate(input=input)
    ret_code = proc.wait()

    return (stderr.decode('ascii'), ret_code)

def _find_sysdb_paths(debug_output, debug):
    # Don't yield the SysDB paths matching any of the following regexes, as
    # they pertain to schemas which exhibit a many to one SysDB path to
    # schema path relationship.
    INVALID_SYSDB_REGEXES = [
        r"^/oper/tty/gl/________/________/.*|^/oper/tty/gl/.*/submode_exit|^/oper/aaa/gl/userdb/vty*"
        ]

    for line in debug_output:
        if debug:
            print(line)
        match = re.match(
            r'SYSDB LIB: (?:CHUNK )?'
            '(GET|SET|DELETE|LIST|DATALIST|FIND\ DATA|FIND_DATA|FIND|GET_ITEMS) .*? '
            '\'/(cfg|oper|action)/([^\']+)\'',
            line)
        if match is not None:
            path = "/{}/{}".format(match.group(2), match.group(3))
            if not any(re.match(regex, path) for regex in INVALID_SYSDB_REGEXES):
                action = match.group(1)
                yield (action, path)

#--------------------------------------------
def _find_yang_from_sysdb_path(action, sysdb_path, mfile, debug, nearest):
    with open(mfile) as search:
        for line in search:
            yang_path,spath = line.split('#')
            match_path = re.sub(r"<[a-zA-Z1-9_-]+?>",".*",spath.strip('\n'))
            match_path = match_path.replace('|','\|')
            if action == 'DATALIST':
                tmp = match_path.rsplit('/', 1)
                if '.*' in tmp[1]:
                    match_path = tmp[0] + '/.*'

            #if debug:
            #    print("match {} with {}".format(match_path, sysdb_path))
            if re.match(match_path, sysdb_path):
                if debug:
                    print("Matched {}->{}".format(match_path, sysdb_path))
                print("Yang xpath: {}".format(yang_path))
            else:
                if nearest:
                    _get_common_longest_match(match_path, sysdb_path, yang_path, debug)

# ----------------------------------------------------------------------------
# Main
# ----------------------------------------------------------------------------

def _get_parser():
    parser = argparse.ArgumentParser(
            description='Find oper yang xpath for given show command.\n'
                        'If meta_file is not passed, it will be generated')
    # hack around argparse bug... (doesn't recognize first added argument)
    parser.add_argument(' ')

    parser.add_argument('show_cmd',
            action='store',
            type=str,
            help='show command')
    parser.add_argument('-m', '--meta_file',
            default="/tmp/delete",
            #type=argparse.FileType('rw'),
            help='file to read from (default=/tmp/delete), '
                 'generated using "mdt_get_gather_paths Cisco-IOS-XR-*oper --xr_path -o /tmp/delete"')
    parser.add_argument('-v', '--verbose',
            default=False,
            action='store_true',
            help='verbose')
    parser.add_argument('-d', '--debug',
            default=False,
            action='store_true',
            help='debug')
    parser.add_argument('-n', '--nearest',
            default=False,
            action='store_true',
            help='print out nearest match, only use if no match found without this flag')
    return parser

if __name__ == '__main__':
    args = _get_parser().parse_args(sys.argv)

    if args.verbose:
        print("Show command: {}".format(args.show_cmd))
    sysdb_output, rc = _run_process(
             "parser_helper '{}'".format(args.show_cmd),
             None,
             env={'SYSDB_LIB_DEBUG_ACCESS': '1'}, input=b"\n")

    if rc:
        print(sysdb_output.split('\n')[-2])

    if not os.path.isfile(args.meta_file):
        subprocess.call(["/pkg/bin/mdt_get_gather_paths", "Cisco-IOS-XR-*oper", "--xr_path", "-o", args.meta_file])
        if not os.path.isfile(args.meta_file):
            print("Error generating meta file {}".format(args.meta_file))
        else:
            if args.verbose:
                print("Generated meta file {}".format(args.meta_file))

    for action, sysdb_path in _find_sysdb_paths(sysdb_output.split('\n'), args.debug):
        if args.verbose:
            print("Action: {}, Sysdb Path: {}".format(action, sysdb_path))
        _find_yang_from_sysdb_path(action, sysdb_path, args.meta_file, args.debug, args.nearest)
