import os
import sys
import shutil
import logging
import logging.handlers
import datetime
import commands
import functools
from utils.inst_utils import *

xr_root = "/opt/cisco/XR/packages/"
#xr_root = "/opt/cisco/calvados/packages/"

def get_card_class_inst(root):
    try:
        card_inst_file = "%s/%s"%(root, "/root/card_instances.txt")
        logger.debug("Will read %s for instance info"%(card_inst_file))
        with open(card_inst_file, "r") as f:
            card, inst = f.readline().strip().split(' ')
            f.close()
            logger.debug("Card instances retrieved: %s %s"%(card,inst))

    except Exception as e:
        logger.error("Unable to retrieve card instance" + str(e))
        sys.exit(-1)
    return card, inst.split(',')

def get_files_from_rpm(cardtype, cardinst, rpmname, root):
    file_list_dict = {}
    try:
        rpm_cmd = "chroot %s rpm -q --queryformat %s %s"%(root, "\'[%{FILENAMES}\\n]\'", rpmname)
        logger.debug("Command to execute %s"%(rpm_cmd))
        try:
            file_list = run_cmd(rpm_cmd, True).split()
        except:
            logger.error("Unable to run cmd %s"%(rpm_cmd))
            sys.exit(-1)
        for f in file_list:
            if os.path.isdir("%s%s"%(root, f)):
                logger.debug ("Skipping directory %s"%("%s%s"%(root, f)))
                continue
            if os.path.isfile("%s/%s"%(root, f)) and xr_root in f:
                pass
                logger.debug("Parsing %s"%(f))
            else:
                continue
            f_consider = f.replace(xr_root, '') 
            f_consider_break = [x.strip() for x in f_consider.split('/') if x] 
            if f_consider_break[0] and rpmname.replace('.x86_64', '') != f_consider_break[0]:
                continue
            if f_consider_break[1] == "all":
                link_file = "%s/%s"%("/pkg",'/'.join(f_consider_break[2:]))
                file_list_dict[link_file] = f
                continue
            if f_consider_break[1] and cardtype.lower() == f_consider_break[1]:
                if f_consider_break[2] and f_consider_break[2] == "instances": 
                    # Expect instance name as a dir
                    card_inst = f_consider_break[3]
                    if card_inst not in cardinst:
                        continue
                    else:
                        link_file = "%s/%s"%("/pkg",'/'.join(f_consider_break[4:]))
                else:
                    link_file = "%s/%s"%("/pkg",'/'.join(f_consider_break[2:]))
                #logger.debug("Consider for symlink %s"%link_file)
                file_list_dict[link_file] = f
                continue

        return file_list_dict
            
    except Exception as e:
        logger.error("Unable to derive list of files from %s with exception %s"%(rpmname, str(e)))
        sys.exit(-1)

'''
    Comparison function determining the precedence given two 
    packages and its metadata. 
    Essentially, it ensures that:
    a. SMU precedes any package.
    b. optional package precedes mandatory package.
    c. PD package precedes PI packages.
'''
def pkg_precedence_fn (rpm1, rpm2):
    if rpm1.packagetype.lower() == 'package' and rpm2.packagetype.lower() == 'smu':
        return 1
    elif rpm1.packagetype.lower() == 'smu' and rpm2.packagetype.lower() == 'package':
        return -1
    if rpm1.pkgpresence == 'optional' and rpm2.pkgpresence == 'optional':
        if rpm1.pkgtype == 'PI' and rpm2.pkgtype == 'PD':
            return 1
        elif rpm1.pkgtype == 'PI' and rpm2.pkgtype == 'PI':
            return 0
        else:
            return -1
    if rpm1.pkgpresence == 'mandatory' and rpm2.pkgpresence == 'mandatory':
        if rpm1.pkgtype == 'PI' and rpm2.pkgtype == 'PD':
            return 1
        elif rpm1.pkgtype == 'PI' and rpm2.pkgtype == 'PI':
            return 0
        else:
            return -1
    if rpm1.pkgpresence == 'optional' and rpm2.pkgpresence == 'mandatory':
        return -1
    else:
        return 1

class Rpm:
    def __init__(self, instance, pkg_root):
        self.name    = None
        self.pkgtype = None
        self.pkgpresence = None
        self.packagetype = None
        cmd = "chroot %s rpm -q --queryformat %s %s"%(pkg_root,
                        "\'%{PKGTYPE} %{PACKAGEPRESENCE} %{PACKAGETYPE}\'", 
                        instance)
        self.name = instance
        self.pkgtype, self.pkgpresence, self.packagetype = run_cmd (cmd).split(' ')
        if self.pkgtype == '(none)':
            self.pkgtype = 'PD'
            if 'iosxr' in instance:
                self.pkgtype = 'PI'
        return

class RpmInst:
    def __init__(self, instance, pkg_root):
        self.rpmname = instance
        self.cardtype = ''
        self.cardinst = []
        self.filelist = {}
        
        self.cardtype, self.cardinst = get_card_class_inst(pkg_root)    
        self.filelist = get_files_from_rpm(self.cardtype, self.cardinst, instance, pkg_root)

    def __str__(self):
        return "card type: %s, card inst: %s"%(self.cardtype, self.cardinst)

def create_links_for_packageObj(rpminst, root_dir):
    for key in rpminst.filelist.keys():
        try:
            dst_file = rpminst.filelist[key]
            link_name = "%s%s"%(root_dir, key)
            logger.debug("Create symlink: %s -> %s"%(link_name, dst_file))
            if os.path.islink(link_name):
                logger.debug ("Duplicate link %s found. Ignoring the same"%(link_name))
                continue
	    dirname = os.path.dirname(link_name)
  	    if not os.path.isdir(dirname):
		os.makedirs(dirname)
            os.symlink(dst_file, link_name)
        except:
            logger.error("Unable to create symlink %s"%(link_name))
            sys.exit(-1)
        
    return 

def get_rpms_from_partition(root_dir):
    rpm_list = []
    if os.path.isdir(root_dir):
        rpm_cmd = "chroot %s rpm -q -g IOS-XR --last | awk '{print $1}'"%(root_dir)
        logger.debug("Command to execute %s"%(rpm_cmd))
        try:
            rpm_list = run_cmd(rpm_cmd).split()
        except:
            logger.error("Unable to run cmd %s"%(rpm_cmd))
            sys.exit(-1)
    return rpm_list

def create_new_ldpath_file (ldpath_file, rpms_installed):
    new_ldpath_list = []
    for rpm in rpms_installed:
        rpm = rpm.replace ('.x86_64','')
        rpm = rpm.replace('.arm','')
        new_ldpath_list.append (rpm)

    from datetime import datetime
    time_stamp = datetime.now().strftime("%b %d %H:%M:%S")
    with open(ldpath_file, 'w') as f:
        f.write("# %s Loadpath\n" % "XR")
        f.write("# Updated on %s\n" % time_stamp)
        f.write("# Updated by %s (initial boot)\n\n" % "orchestrator")
        f.write("%s" % ":".join(new_ldpath_list))
        f.write("\n")

if __name__ == "__main__":
    LOGFILE = "/tmp/inst_hooks.log"
    XR_LOADPATH_FILE="install/instdb/local/xr_ldpath.txt"
    # create logger
    logger = logging.getLogger('install_hook_logger')
    logger.setLevel(logging.DEBUG)
    formatter = logging.Formatter('%(asctime)s::  %(message)s',"%Y-%m-%d %H:%M:%S")

    # Logs to logfile
    fh = logging.handlers.RotatingFileHandler(LOGFILE, maxBytes=(1024*10000), backupCount=2)
    fh.setLevel(logging.DEBUG)
    fh.setFormatter(formatter)
    logger.addHandler(fh)

    cli = sys.argv[:]
    logger.debug("+"*80)
    logger.debug(' '.join(cli))

    if len(sys.argv) < 2:
        logger.error("Insufficient number of arguments to program.")
        sys.exit(-1)

    root_dir = sys.argv[1]
    rpms_installed = get_rpms_from_partition(root_dir)

    RpmInstObjects = {}
    for rpm in rpms_installed:
        logger.debug("Consider %s for file list"%rpm)
        RpmInstObjects[rpm] = RpmInst(rpm, root_dir)

    rpmtypelist = []
    for rpm in rpms_installed:
        rpmtypelist.append (Rpm(rpm, root_dir))

    rpms_sorted = sorted (rpmtypelist, key = functools.cmp_to_key (pkg_precedence_fn))
    rpms_installed_sorted = [x.name for x in rpms_sorted]

    for rpm in rpms_installed_sorted:
        create_links_for_packageObj(RpmInstObjects[rpm], root_dir)

    loadpath_file = os.path.join (root_dir, XR_LOADPATH_FILE)
    ldpath_dir = os.path.dirname(loadpath_file)
    if not os.path.exists(ldpath_dir):
        try:
            os.makedirs(ldpath_dir)
        except:
            pass
    create_new_ldpath_file(loadpath_file, rpms_installed_sorted)
