#!/bin/bash
#
# Copyright (c) 2015-2016, 2018 by Cisco Systems, Inc.
# All rights reserved.
#

# Gather all inputs to this script
# location of host iso
readonly HOSTISO=$1
# absolute path of file to copy (within the rpm)
readonly FILENAME=$2
# destination 
readonly DESTINATION=$3

# constants
readonly RPM=/usr/bin/rpm
readonly RPM2CPIO=/usr/bin/rpm2cpio
readonly LOGFILE=/var/log/v2_extract_log
readonly EXPECTED_ARGS=3
readonly RPMDB_P="/var/lib/rpm/"

# function cleans up mount points and directories
function exitunmount() {
    [ -z "${EXTRACT_RPM_DIR}" ] || umount -R -l ${EXTRACT_RPM_DIR}/proc || true
    [ -z "${EXTRACT_RPM_DIR}" ] || umount -R -l ${EXTRACT_RPM_DIR}/sys  || true
    [ -d "${ISOMNT}" ] && umount ${ISOMNT} && rmdir ${ISOMNT}
    [ -d "${EXTRACT_RPM_DIR}" ] && rm -rf ${EXTRACT_RPM_DIR}
}

function exitclean() {
    exitunmount
    exit 1
}

# function cleans up mount points and directories
function exitclean_success() {
    exitunmount
    exit 0
}


# function executes the rpm command and retry if it fails
function rpm_retry_on_failure() {
    rpmcmd="$1"
    echo "Executing rpmcmd" >> $LOGFILE
    $rpmcmd >> $LOGFILE 2>&1
    if (( $? != 0 )); then
       echo "WARNING: Failed to install the rpm. Retrying.." >> $LOGFILE
       if [ -d "$RPMDB_P" ]; then
          db_path="$RPMDB_P/__db*.*"
       fi
       if [ -z "$db_path" ]; then
          echo "Rpm Db does not exist retrying the operation again" >> $LOGFILE
          $rpmcmd >> $LOGFILE 2>&1
       else
          FUSER_VERSION=1 fuser -s $db_path  
          if (( $? != 0 )); then
             echo "Rpm Db is not in use. Safely clear it" >> $LOGFILE
             rm -rf $db_path
             $rpmcmd >> $LOGFILE 2>&1
          else
             echo "Rpmdb is in use" >> $LOGFILE
             lsof_out=$(lsof -b -S 2  | grep -e $RPMDB_P)
             echo "Processes  using rpmdb:" >> $LOGFILE
             echo "$lsof_out" >> $LOGFILE
             return 1
          fi
       fi  
    else
       echo "Successfully executed rpm command" >> $LOGFILE
       return 0
    fi
} 
# basic error checks

if [ $# -ne $EXPECTED_ARGS ]; then
    echo "Usage: `basename $0` HOSTISO FILENAME OUTPUTDIR"
    echo "        HOSTISO: Absolute path to host ISO" 
    echo "        FILENAME: Absolute path to file in hostos RPM"
    echo "        OUTPUTDIR: Absolute path to copy the extracted file to"
    exit 3
fi

# clean up old logfile
rm -f $LOGFILE

if [ ! -f "${HOSTISO}" ]; then
   echo "${HOSTISO} does not exist" >> $LOGFILE
   exitclean
fi


# mount the host ISO
readonly ISOMNT=$(mktemp -d /tmp/isomnt.XXXXXX)
HDISK="/misc/disk1/"
EXTRACT_RPM_DIR=$(mktemp -d ${HDISK}initrd.XXXXXX)

mount -o loop ${HOSTISO} ${ISOMNT} || exitclean

# extract contents of host os pkg rpm
HOSTOS_RPM_SRC=`ls -1 ${ISOMNT}/rpm/*hostos*.rpm | head -1`
bn=$(basename  $HOSTOS_RPM_SRC)
if [ -n "$HOSTOS_RPM_SRC" ]; then
    cd $EXTRACT_RPM_DIR
    if [ -e "${ISOMNT}/boot/initrd.img" ]; then
       isize=$(ls -lL --block-size=M "${ISOMNT}/boot/initrd.img" | cut -d ' ' -f5 )
       echo "Size of ${ISOMNT}/boot/initrd.img = $isize" >> $LOGFILE  
    
    else
       echo "ERROR: Missing ${ISOMNT}/boot/initrd.img" >> $LOGFILE
       echo "ERROR: Missing initrd file in ${HOSTISO}"
       exitclean
    fi
    df_out="$(df -P --block-size=M ${EXTRACT_RPM_DIR})"
    echo "$df_out" >> $LOGFILE
    avail_space=$(echo "$df_out" |grep % | tail -n1 |  sed 's/\s\+/ /' | \
               awk -F' ' '{ print $(NF-2) }')
    avail_space=$(echo ${avail_space/%?})
    echo "Available space in ${EXTRACT_RPM_DIR}: "$avail_space >> $LOGFILE
    extra_space=$(($avail_space-${isize/%?}))
    if [ "$extra_space" -lt 0 ]; then
       echo "ERROR: Insufficient space in ${HDISK} on sysadmin. Required space for initrd extraction: $isize"
       exitclean
    fi 
    zcat ${ISOMNT}/boot/initrd.img | cpio -id 
    if (( $? != 0 )); then
        exitclean
    else
        echo "initrd extracted in ${EXTRACT_RPM_DIR}" >> $LOGFILE
    fi
    
    rsize=$(ls -lL --block-size=M "$HOSTOS_RPM_SRC" | cut -d ' ' -f5 ) >> $LOGFILE
    extra_space=$(($extra_space-${rsize/%?})) 
    echo "Hostos rpm size: $rsize extra_space: $extra_space" >> $LOGFILE
    if [ "$extra_space" -lt 0 ]; then
       req_space=$((${isize/%?}+${rsize/%?})) 
       echo "ERROR: Insufficient space in ${HDISK} on sysadmin. Required space for rpm: ${req_space}M" 
       exitclean
    fi
 
    mount --rbind /proc ${EXTRACT_RPM_DIR}/proc
    mount --rbind /sys  ${EXTRACT_RPM_DIR}/sys 
    cmd="$RPM --root=$EXTRACT_RPM_DIR -iv --force  --nodeps $HOSTOS_RPM_SRC" 
    echo "CMD: $cmd" >> $LOGFILE
    rpm_retry_on_failure "$cmd"
    if (( $? != 0 )); then
        echo "ERROR: Failed to install hostos rpm $bn." 
        exitclean
    else
        if [ -f "${EXTRACT_RPM_DIR}${FILENAME}" ]; then
            cp ${EXTRACT_RPM_DIR}${FILENAME} $DESTINATION
        else
            echo "${EXTRACT_RPM_DIR}${FILENAME} does not exist" >> $LOGFILE
            echo "ERROR: Failed to find bootstrap config file" >> $LOGFILE
            exitclean
        fi
        #we have succeeded in copying out v2 bootstrap, exit with success
        exitclean_success
    fi
else
    echo "ERROR: Hostos rpms does not exist in the iso ${ISOMNT}" >> $LOGFILE
    echo "ERROR: Hostos rpms are missing in the host iso" 
    exitclean 
fi

