#!/usr/bin/python
#-----------------------------------------------------------------------
# install-functions.py - Generic place holder to put pythonic functions 
#                        related to Install
# 
# This script is independent of install_functions.sh file. Initially, 
# there is no intersection of functions with the functions from file
# install_functions.sh
# 
# In this script, Install class is the main class, whose functions will 
# be invoked from outside of this script (i.e. mainly from C code). 
#
# Copyright (c) 2017-2019 by cisco Systems, Inc.
# All rights reserved.
#-----------------------------------------------------------------------
#


#######################################################
#
# <<<<<<<<<<<<<<<<<<   PRECAUTION   >>>>>>>>>>>>>>>>>>
#                      ----------
#
# Please do not put any 'print' statement if value is 
# not intended to return to the caller of this script
#
# Instead use logger object for printing the debug 
# statements, which can be found in file
# /var/log/install/install_functions_py.log
#
#######################################################


from __future__ import print_function, with_statement
import sys
sys.dont_write_bytecode = True
import os
import re
from subprocess import Popen, PIPE
import logging
import logging.handlers
import datetime as dt
import hashlib
import yaml
from urlparse import urlparse
import shutil
import tarfile
import subprocess
import traceback
sys.path.append(os.path.dirname(os.path.abspath(sys.argv[0])))
from provides_hint import PkgHintsMdata,get_mdata_for_vm
import functools

RPM_BIN='rpm'
PLATFORM_STR1='PLATFORM_EXT'
PLATFORM_STR2='PLATFORM'
BOOTSTRAP_FILE='/etc/rc.d/init.d/calvados_bootstrap.cfg'
CMD_FILE='/proc/cmdline'
LOG_FILE = '/var/log/install/install_functions_py.log'
VIRT_METHOD = "VIRT_METHOD"

def rpm_version_string_cmp (rpm1, rpm2):
    rpm1_name, rpm1_version = re.split(r'(.*)-(.*)-(.*)', rpm1)[1:3]
    rpm2_name, rpm2_version = re.split(r'(.*)-(.*)-(.*)', rpm2)[1:3]
    if rpm1_name == rpm2_name: 
        ilist1 = map(int, rpm1_version.split('.'))
        ilist2 = map(int, rpm2_version.split('.'))
        if ilist1 > ilist2:
            return -1
        elif ilist1 < ilist2:
            return 1
        else:
            return 0
    elif 'CSC' in rpm1_name and 'CSC' in rpm2_name:
        return 0
    elif 'CSC' in rpm1_name and 'CSC' not in rpm2_name:
        return -1
    elif 'CSC' not in rpm1_name and 'CSC' in rpm2_name:
        return 1
    else:
        return 0

def runcmd(cmd, success_ret_values = [0]):
  '''
    Runs cmd command in shell.

    It will print cmd as well as its output to logger.
    If some error occurs then output contains error.
  '''  

  sprc = 0
  process = Popen(cmd, stdout=PIPE, stderr=PIPE, shell=True)
  out, error = process.communicate()
  sprc = process.returncode

  if len(out) > 1 and out[-1] == '\n':
    out = out[:-1]
    
  #for commands like grep, return value will be non-zero even in no-error case
  if sprc == None or sprc not in success_ret_values:
      logger.error('Error in executing CMD="%s" , output="%s", ' \
                   'error="%s"' %(cmd, out, error))
      sys.exit(1)
    
  logger.debug('cmd="%s" output="%s"'%(cmd,out))
  return dict(rc=sprc, output=out)


def get_file_md5(file_name):
  '''
    Get md5 checksum of a file
    
  '''
  
  hash_md5 = hashlib.md5()
  try:
    with open(file_name, 'rb') as file:
      for chunk in iter(lambda: file.read(4096), b''):
        hash_md5.update(chunk)
  except EnvironmentError:
    return ''
  
  return hash_md5.hexdigest()


class MyFormatter(logging.Formatter):
  '''Class to format timestamp in logger to include miliseconds'''
     
  converter=dt.datetime.fromtimestamp

  def formatTime(self, record, datefmt=None):
    ct = self.converter(record.created)
    if datefmt:
      s = ct.strftime(datefmt)
    else:
      t = ct.strftime('%Y-%m-%d %H:%M:%S')
      s = '%s,%03d' % (t, record.msecs)
    return s


class Install:
  '''
    Class containing all pythonic general purpose functions for install
    
     
    Common agruments for most of the functions of this class 
    1. self: object created in __main__
    2. ret(Default value=False): If it is given as True then function will not 
       print anything on stdout, instead values will be returned to the 
       calling function. By default (which is the case when it will be called 
       from C code), all functions will print to stdout and caller has to 
       interpret it. 
  '''

  def __init__(self):
    '''
      Install class initialization.

      Following object variables are initialized:
        platform
        vmtype
        group
        arch
    ''' 

    bootDict = {}
    with open(BOOTSTRAP_FILE) as bootFile:
      for line in bootFile:
        line = line.split('#')[0].strip()
        if not line:
          continue
        (key, value) = line.split('=')
        key = key.strip()
        value = value.strip()

        bootDict[key] = value  

    if PLATFORM_STR1 in bootDict:
      self.platform = bootDict[PLATFORM_STR1]
    elif PLATFORM_STR2 in bootDict:
      self.platform = bootDict[PLATFORM_STR2]
    else:
      logger.error('ERROR: Could not able to get platform from ' 
                   + BOOTSTRAP_FILE)
      sys.exit(1)
    
    if VIRT_METHOD in bootDict:
        self.virt_method = bootDict[VIRT_METHOD]
    else:
        logger.error('ERROR: could not able to get VIRT METHOD from'
                       + BOOTSTRAP_FILE)
        sys.exit(1)

    self.vmtype = ''
    archStr = ''
    if 'VMTYPE' in bootDict:  
      self.vmtype = bootDict['VMTYPE']
    else:
      out_dict = runcmd('/usr/bin/xr_sysctl -n kernel.boot.vmtype')
      if out_dict['rc']:
        logger.error('ERROR: Could not able to get VM Type from '
                      + BOOTSTRAP_FILE + ' or ' + CMD_FILE)
        sys.exit(1)
      else:
        self.vmtype = out_dict['output']


    if self.vmtype == 'xr-vm':
      archStr='XR_SUPPORTED_ARCHS'
      self.group = 'IOS-XR'
    elif self.vmtype == 'sysadmin-vm':
      archStr='CALV_SUPPORTED_ARCHS'
      self.group = 'sysadmin'
    else:
      logger.error('ERROR: Unknown vm-type: ' + self.vmtype)
      sys.exit(1)

    localArch = runcmd("arch")
    localArch = localArch['output'].strip()

    if archStr in bootDict:
      supportedArchs = bootDict[archStr].split(',')
    else:
      supportedArchs = [ 'x86_64' ]
    bFound = False
    for arch in supportedArchs:
      if arch == localArch or arch in localArch :
        localArch = arch
        bFound = True

    if not bFound:
      logger.error('ERRROR: Unknown architecture: ' + localArch)
      sys.exit(1)
    
    self.arch = localArch

    logger.info('PLATFORM: ' + self.platform + ' VM Type: ' + self.vmtype + 
                 ' ARCH: ' + self.arch)

  def run_cmd(self,cmd):
      process = subprocess.Popen(cmd, stdout=subprocess.PIPE,
                                 stderr=subprocess.PIPE, shell=True)
      out, error = process.communicate()
      sprc = process.returncode
      if sprc is None or sprc != 0:
          out = error
          raise RuntimeError("Error CMD=%s returned --->%s" % (cmd, out))
      else:
          logger.debug ("CMD=%s OUTPUT=%s"%(cmd, out))
          pass
      return out.strip()
   
  def invoke_optimised_add_distbn(self,tmp_staging_dir,sdr_name):
      logger.debug("In invoke_optimised_add_distbn")
      logger.debug("Tmp staging dir = %s sdr name =%s", tmp_staging_dir,sdr_name)
      for filename in os.listdir(tmp_staging_dir):
          cmd1 = "file -Lb %s" % (tmp_staging_dir+filename)
          res1 = self.run_cmd(cmd1)
          if "tar archive" in res1:
              cmd = "tar -C %s -xf %s --transform='s/.*\///'" % (tmp_staging_dir,tmp_staging_dir+filename)
              self.run_cmd(cmd)
              cmd2 = "rm -rf %s" % (tmp_staging_dir+filename)
              self.run_cmd(cmd2)

      for filename in os.listdir(tmp_staging_dir):
          cmd1 = "file -Lb %s" % (tmp_staging_dir+filename)
          res1 = self.run_cmd(cmd1)
          if tarfile.is_tarfile(tmp_staging_dir+filename):
              if "ISO" in res1:
                  iso_name = ""
                  iso_version = ""
                  iso_label = ""
                  cmd = "mktemp -d"
                  out_dict = runcmd(cmd)
                  mnt_dir = out_dict['output']
                  cmd = "mount -o loop " + tmp_staging_dir + filename  + " " + mnt_dir
                  runcmd(cmd)
                  iso_info_txt = mnt_dir + "/iso_info.txt"
                  sp_info_txt = mnt_dir + "/sp_info.txt"
                  if os.path.exists(iso_info_txt):
                      cmd = "cat %s | grep 'Name'" %(iso_info_txt)
                      out_dict = runcmd(cmd)
                      if not out_dict['rc']:
                          li = out_dict['output'].split()
                          iso_name = li[1]
                          iso_version = li[3]
                          if "golden" in iso_name:
                              cmd = "/pkg/bin/provides_hint.py -f %s/iosxr_image_mdata.yml -g -a | grep 'GISO Label'" %(mnt_dir)
                              out_dict = runcmd(cmd)
                              if not out_dict['rc']:
                                  iso_label = out_dict['output'].split(":")[1].split()[0]
                              else:
                                  logger.debug("Error:Failed to get GISO Label")

                      iso_full_name = iso_name + "-" + iso_version
                      if iso_label:
                          iso_full_name = iso_full_name + "-" + iso_label + ".iso"
                      else:
                          iso_full_name = iso_full_name + ".iso"
                      os.rename(tmp_staging_dir+filename,tmp_staging_dir+iso_full_name)
                      logger.debug("Renaming %s to %s ",tmp_staging_dir+filename,tmp_staging_dir+iso_full_name)
                      cmd = "umount " + mnt_dir
                      runcmd(cmd)
                      shutil.rmtree(mnt_dir, ignore_errors = True)
                  elif os.path.exists(sp_info_txt):
                      logger.debug("Package is SP. Continue")
                      continue
          elif "RPM" in res1:
              m = re.search(r'(.*/)*(.*)-(.*)-(.*)\.(.*)\.(.*)', filename)
              m.groups()
              if m.groups()[-1] != 'rpm':
                  logger.debug("Inavlid RPM Name: %s\n", filename)
                  print("Invalid RPM Name :" + filename)
                  sys.exit(-1)
          else:
              logger.debug("Pkg %s is an invalid file, removing it from staging location\n", filename)
              os.remove(tmp_staging_dir+filename)

      pkgs_in_stg_dir = os.listdir(tmp_staging_dir)
      logger.debug(pkgs_in_stg_dir)
      pkgs_to_add = ' '.join(pkgs_in_stg_dir)
      logger.debug(pkgs_to_add)
      add_cmd  = "/pkg/bin/sdr_instadd.py -r %s -i %d -p %s -sdr %s"%(tmp_staging_dir,
              -1,pkgs_to_add,sdr_name)
      logger.debug(add_cmd)
      cmd = add_cmd.split(" ")
      logger.debug(add_cmd)
      logger.debug('*'*30)
      try :
          cmd = add_cmd.split(" ")
          add_cmd = [x for x in cmd if x] #Imp else instmgr crashes
          output = self.run_cmd(' '.join(add_cmd))
          logger.debug("Add done")
      except :
          exc_info = sys.exc_info()
          err_str = str(exc_info[1])
          if ("--->" in err_str) and \
              "ERROR" in err_str and \
              err_str.rfind("--->") < err_str.rfind("ERROR"):

              err_str = err_str[err_str.rfind("ERROR")+5:]
              logger.error("Error in ADD phase: %s"%(err_str))
              print("Error in ADD phase: %s"%(err_str))
          else:
              TB = traceback.format_exc()
              logger.debug(TB)
              logger.error("""
              Error: An exception is hit while waiting for optimised add to complete.
              If you hit same error on retries, please collect "show tech install"
              and contact cisco-support.
              """)
          sys.exit(-1)
     
  def _is_smu(self, smu):
    if smu and re.search('CSC[a-z]{2}\d{5}', smu):
      return True
    return False


  def _is_tp_smu(self, smu):
    if self._is_smu(smu) and not smu.startswith(self.platform):
      return True
    return False


  def _is_hostos_smu(self, smu):
    if self._is_smu(smu):
      if "sysadmin-hostos" in smu or "spirit-boot" in smu:
        return True
    return False

  def get_pkgs2consider_list (self, pkglist):
    '''
      Given a list of packages, find all packages to be considered 
      during an SU operation.  
    '''
    pkglist = pkglist.split()
    sorted_pkgs = sorted (pkglist, key = functools.cmp_to_key (rpm_version_string_cmp))
    logger.info ("Sorted package list %s"%(','.join(sorted_pkgs)))
    pkgconsidered = []
    pkgs2install = []
    for pkg in sorted_pkgs:
        if not self._is_smu (pkg):
            pkgs2install.append (pkg)
        else:
            pkg_name = re.split(r'(.*)-(.*)-(.*)\.(.*)', pkg)[1]
            if pkg_name not in pkgconsidered:
                pkgconsidered.append (pkg_name)
                pkgs2install.append (pkg)
                if 'sysadmin-hostos' in pkg:
                    if '.admin' in pkg:
                        host_pkg = pkg.replace ('.admin','.host')
                    elif '.host' in pkg:
                        host_pkg = pkg.replace ('.host','.admin')
                    pkgs2install.append (host_pkg)
                    
    print (','.join(pkgs2install))

  def _strip_arch(self, pkg, bDebugLog=False):
    '''
      Strip architecture from a string or list of a string
    '''
    
    if type(pkg) == list:
      pkg_list = []
    
      for p in pkg:
        p = p.replace('.x86_64','')
        p = p.replace('.arm','')
        pkg_list.append(p)
      
      pkg = pkg_list
      
    elif type(pkg) == str:
      pkg = pkg.replace('.x86_64','')
      pkg = pkg.replace('.arm','')
    
    else:
      logger.error('ERROR: ' + str(type(pkg)) + ' type\'s argument passed to '
                   + '_strip_arch function is not supported')
    
    if bDebugLog:
      logger.debug('Package Name(s) with stripped arch: ' + str(pkg))
    return pkg


  def _initialize_installed_smu_list(self):
    PATH_PREFIX = '/install/instdb/local'
    LDPATH_FILE = ''
    LOGGER_BIN = ''

    if self.vmtype == 'sysadmin-vm':
      LDPATH_FILE = '{}/calvados_ldpath.txt'.format(PATH_PREFIX)
      LOGGER_BIN = '/opt/cisco/calvados/bin/cal_logger'
    elif self.vmtype == 'xr-vm':
      LDPATH_FILE = '{}/xr_ldpath.txt'.format(PATH_PREFIX)
      LOGGER_BIN = '/pkg/bin/logger'
    else: #This is handled in __init__, but just kept here for clarity
      logger.error('Unknown vmtype: {}'.format(self.vmtype))  
      return ([], {}) 

    try:
      with open(LDPATH_FILE) as fLoadpath:
        lines = fLoadpath.readlines()
        new_lines = []
        for x in lines:
          x = x.strip().split('#')[0]
          if x :
            new_lines.append(x)
        lines = new_lines    

        logger.info('{} file Content:'.format(LDPATH_FILE))
        logger.info('{}'.format(str(lines)))

        if len(lines) != 1:
          err_msg = 'Loadpath file {} seems to be corrupted; supersede info ' \
                    'may be wrong.'.format(LDPATH_FILE)
          logger.error(err_msg)
          os.system("{} -s crit {}".format(LOGGER_BIN, err_msg))
          return ([], {})

        installed_smu_list = []   # list of valid SMUs from loadpath file
        installed_smu_tuples = [] # list of (name, version, release) for
                                  # above SMUs
        supersede_dict_temp = {}  # Temporary dict to hold (n, v, r) tuples  
                                  # according to each package SMUs
                                  # {key: first SMU (n, v, r) tuple in loadpath
                                  #       for a given package
                                  #  values: (n, v,r) tuples of remaining SMUs 
        supersede_dict = {}       # {key: top-level SMU rpm name
                                  #  value: all obsoleted SMU rpms of a given 
                                  #         top-level rpm

        for pkg in lines[0].split(':'):
          pkg = self._strip_arch(pkg)

          if not self._is_smu_supersedable(pkg):
            continue

          smu_name_group = re.search(r'(.*)-(.*)-(.*\.CSC[a-z]{2}\d{5}.*)', 
                                     pkg)
          if smu_name_group:
            (name, version, release) = smu_name_group.groups()
            installed_smu_tuples.append((name, version, release)) 
            installed_smu_list.append(pkg)
          else: # Ignoring this loadpth entry
            logger.error('SMU name {} is not in proper format.'.format(pkg))
            logger.error('It could be due to possible corruption '\
                          'of {} file'.format(LDPATH_FILE))

        logger.info('SMUs after filtering: {}'.format(str(installed_smu_list)))

        for (name, version, release) in installed_smu_tuples:
          top_smu_entry = ''
          # checking for existing entries in supersede_dict_temp
          # if same package entry is there then just append it
          for (name_key, ver_key, rel_key) in supersede_dict_temp:
            if name == name_key:
              top_smu_entry = (name_key, ver_key, rel_key)
              break

          # Assumption: First SMU will always be superseding later SMUs in a 
          #             given loadpath
          if top_smu_entry:
            supersede_dict_temp[top_smu_entry].append((name,version,release))
          else:
            supersede_dict_temp[(name,version,release)] = []

        for (name, version, release) in supersede_dict_temp:
          top_smu = '{}-{}-{}'.format(name, version, release)
          supersede_dict[top_smu] = []
          #TODO: can soring be done here based on version?
          for (obs_name, obs_version, obs_release) in \
              supersede_dict_temp[(name, version, release)]:
            obs_smu = '{}-{}-{}'.format(obs_name, obs_version, obs_release)
            supersede_dict[top_smu].append(obs_smu)

        for smu in supersede_dict:
          supersede_dict[smu].sort()

        logger.info('Supersede dict: {}'.format(str(supersede_dict)))

    except:
      logger.exception('Exception occurred in _initialize_installed_smu_list ')
      logger.info('Returning None(s)')
      installed_smu_list = []
      supersede_dict = {}

    return (installed_smu_list, supersede_dict)


  def _is_smu_supersedable(self, smu):
    ret_value = True

    try:
      smu = self._strip_arch(smu)

      if not self._is_smu(smu):
        #logger.debug("Ignoring package {} as it is not SMU".format(smu))
        ret_value = False
      elif self._is_tp_smu(smu):
        #logger.debug("Ignoring SMU {} as it is a TP SMU".format(smu))
        ret_value = False
      elif self._is_hostos_smu(smu):
        #logger.debug("Ignoring SMU {} as it is a hostos SMU".format(smu))
        ret_value = False
      else:
        smu_name_group = re.search(r'(.*)-(.*)-(.*\.CSC[a-z]{2}\d{5})', smu)
        if smu_name_group:
          (name, version, release) = smu_name_group.groups()
          release_group = re.search(r'(.*).(CSC[a-z]{2}\d{5})', release)
          if release_group:
            (rel_ver, ddts) = release_group.groups()
            if rel_ver[-1].isalpha():
              logger.debug("Ignoring SMU {} as it is an engg. SMU".format(smu))
              ret_value = False
          else:
            ret_value = False
        else:
          ret_value = False
    except:
      logger.exception('Exception occurred in _is_smu_supersedable function')
      logger.info('Returning False')
      ret_value = False


    return ret_value  


  def is_smu_superseded(self, smu):
    '''
      Return True if given SMU in_smu has been superseded by any another 
      installed SMU.

    '''

    ret_value = False

    try:
      smu = self._strip_arch(smu)

      if self._is_smu_supersedable(smu):
        (smu_list, supersede_dict) = self._initialize_installed_smu_list()

        if smu in smu_list:
          if smu not in supersede_dict:
            ret_value = True
        else:
          logger.warning("SMU {} is not in installed SMU list".format(smu))
    except:
      logger.exception('Exception occurred in is_smu_superseded')
      ret_value = False

    logger.info('Final print : {}'.format(str(ret_value).upper()))
    print(str(ret_value).upper())


  def are_smus_superseded(self, input_smus):
    '''
      Return a string containing 0/1 indicating supersede detail of SMUs
    ''' 
    
    return_str = ''

    try:
      (smu_list, supersede_dict) = self._initialize_installed_smu_list()

      for smu in input_smus.split(':'):
        smu = self._strip_arch(smu)

        if not self._is_smu_supersedable(smu):
          return_str += '0'

        elif smu in smu_list:
          if smu in supersede_dict:
            return_str += '0'
          else:
            return_str += '1'
        else:
          return_str += '0'
    except:
      logger.exception('Exception occurred in are_smus_superseded function')
      return_list = '0'*len(input_smus)

    logger.info('Final print : {}'.format(return_str))
    print(return_str)


  def get_all_superseded_smus (self, input_smus, ret_option=False): 
    '''
      Return a string containing supsersede info for all input SMUs
      
      Output string format: "2:SUP_SMUa:SUP_SMUb:1:SUP_SMUc" 
      Superseded SMUs in output will be in correspendence to the sequence of  
      SMUs given in input_smus argument
    '''

    return_str = ''
    total_sup_smus = 0

    try:
      (smu_list, supersede_dict) = self._initialize_installed_smu_list()
      # What if input_smus are not same as smu_list??
      # This should be a transient issue during install operation

      for smu in input_smus.split(':'):
        smu = self._strip_arch(smu)

        if not self._is_smu_supersedable(smu):
          return_str += '0:'

        elif smu in smu_list:
          if smu in supersede_dict:
            total_sup_smus += len(supersede_dict[smu])
            return_str += '{}:'.format(len(supersede_dict[smu]))
            for sup_smu in supersede_dict[smu]:
              return_str += '{}:'.format(sup_smu)
          else:
            return_str += '0:'
        else:
           return_str += '0:'
    except:
      logger.exception('Exception occurred in get_all_superseded_smus')
      return_str = '0:'*len(input_smus)

    if ret_option:
      logger.info('Final return : {}'.format(total_sup_smus))
      return total_sup_smus
    else:
      logger.info('Final print : {}'.format(return_str))
      print(return_str)


  def get_all_superseded_smus_count(self, input_smus):
    '''
      Get number of supersiding SMUs all input smus
    ''' 

    total_smu_count = 0

    try:
       total_smu_count = self.get_all_superseded_smus(input_smus, True)
    except:
      logger.exception('Exception occurred in get_all_superseded_smus_count')
      total_smu_count = 0

    logger.info('Final print : {}'.format(total_smu_count))
    print(total_smu_count)      


  def get_username (self, path):
    '''
      to retrieve username from source path in install add and update
    '''
    path = path.replace("\\","")
    path = urlparse(path)
    if(path.username):
        logger.info('Username retrieved: ' + path.username)
        print(path.username)
    else:
        logger.info('Username is not present')

  def get_password (self, path):
    '''
      to retrieve password from source path in install add and update
      Returns "@@$$^^" if password is not present
    '''
    path = path.replace("\\","")
    path = urlparse(path)
    if(path.password):
        print(path.password)
    else:
        logger.info("Password is not present")
        print("@@$$^^")

  def get_server_ip (self, path):
    '''
      to retrieve server ip from source path in install add and update
    '''
    path = path.replace("\\","")
    path = urlparse(path)
    logger.info('Server ip addr retrieved: ' + path.hostname)
    print(path.hostname)

  def get_source_dir (self, path):
    '''
      to retrieve source dir from source path in install add and update
    '''
    path = path.replace("\\","")
    parsed_path = urlparse(path)
    logger.info('Server dir retrieved: ' + parsed_path.path)
    print(parsed_path.path)
    
  def get_path_without_password (self, path):
    '''
      in install add and update, to remove password from path
    '''
    path = path.replace("\\","")
    parsed_path = urlparse(path)
    if(parsed_path.password):
        pwdless_path = path.replace(':'+parsed_path.password,"")
        logger.info('Path without password: ' + pwdless_path)
        print(pwdless_path)
    else:
        logger.info("Password is not present")
        print("@@$$^^")
    
  def get_password_encoded_path (self, path):
    '''
      in install add and update, to remove password from path
    '''
    path = path.replace("\\","")
    parsed_path = urlparse(path)
    if(parsed_path.password):
        encoded_pwd = "*"*len(parsed_path.password)
        encoded_path = path.replace(parsed_path.password,encoded_pwd)
        logger.info('Password Encoded path: ' + encoded_path)
        print(encoded_path)
    else:
        logger.info("Password is not present")
        print("@@$$^^")
    
  def get_modified_path (self, path, is_vrf=None, vrf_name=None, output_file=None):
    '''
      path after removing '\' and
      If VRF interface provided on commandline
      add it to URL, VRF interace is last option in the CLI
      i.e install add source tftp://1.2.3.4/path vrf vrfname
      modify it to the VRF path tftp://1.2.3.4;vrfname

    '''
    path = path.replace("\\","")
    if is_vrf == "vrf" :
       new_path = urlparse(path)
       # local source(harddisk:/) don't need vrf
       if not new_path.scheme or not new_path.netloc:
          path=path
       else:
          if '%' in new_path.netloc:
             # vrf is part of path in add CLI
             path = path
          else:
             new_ip = new_path.netloc+';'+vrf_name
             vrf_path = path.replace(new_path.netloc, new_ip)
             vrf_path_new = vrf_path.replace(":;",";")
             path=vrf_path_new

    if (output_file):
        with open(output_file, 'w+') as outfile:
            outfile.write(path)
    else:
        print(path)

 # get RPM Path from giso_info.txt
  def get_rpm_path_from_giso(self, giso_info):
    """
     Check if RPM_PATH exist in giso_info.txt
     if doesn't exist return NORPMPATH
    """
    for line in open(giso_info, 'r'):
        data = line.find("RPM_PATH:")
        if data != -1:
           return line
    return ("NORPMPATH")

  def is_signed_651_ncs5500_rpm_path(self, line):
     """ 
      check for path like giso/boot/initrd.img/iso/system_image.iso
      in giso_info.txt
     """
     s=line.find("system_image.iso")
     if s != -1:
        if (line.count("boot") == 1):
            return True
     return False

  def is_signed_rpm_path(self, line):
    """
      check for path like giso/boot/initrd.img
      in giso_info.txt
    """
    s = line.find("system_image.iso")
    if s == -1:
       if(line.count("boot") == 1):
          return True
       else:
           return False
    return False

  def is_signed_ncs5500_rpm_path(self, line):
    """ 
     check for path like 
     giso/boot/initrd.img/iso/system_image.iso/boot/initrd.img in
     giso_info.txt return True if exist such path in RPM_PATH
    """
    s = line.find("system_image.iso")
    if s != -1:
      if (line.count("boot") == 2):
          return True
    return False

  def get_giso_rpm_path(self, giso_info):
    """ 
      typedef enum {
          SIGNED_NCS5500_RPM_PATH       = 1,
          SIGNED_651_NCS5500_RPM_PATH   = 2,
          SIGNED_RPM_PATH               = 3,
          DEFAULT_RPM_PATH              = 4,
          /* not golden ISO */
          NO_GISO_ISO_PATH             = 12345,
      } giso_rpm_path_t;

      Checks RPM_PATH in golden iso returns one of the above enum defined 
      based on value RPM_PATH
    """
    path = self.get_rpm_path_from_giso(giso_info)
    result = path.find("NORPMPATH")
    if result == -1:
       if self.is_signed_ncs5500_rpm_path(path):
          print(1)
       elif self.is_signed_651_ncs5500_rpm_path(path): 
          print(2)
       elif self.is_signed_rpm_path(path):
          print(3)
       else:
          print(4)
    else:
        print(4) 

  def check_iso_isgiso(self, path):
      """ 
         check is it golden ISO by looking giso_info.txt
         if giso_info.txt doesn't find returns 
         NO_GISO_ISO_PATH enum defined in caller 
      """
      ret = runcmd("isoinfo -R -l -i " + path + "| grep giso_info.txt",
                    success_ret_values = [0, 1])
      if ret["rc"] == 1:
         print(12345)
      else:
         print(0)

  def get_sp_name (self, iso_mnt_dir):
    '''
       Return the name of the SP from sp_info file
    '''
    with open(os.path.join(iso_mnt_dir, 'sp_info.txt')) as out:
        info = yaml.load(out)
    sp_name = info.get('name')
    if sp_name:
        print(sp_name)
        return(sp_name)
    else:
        logger.info("Name is missing in sp info")
        logger.info(info)
        print("Error: SP info is missing in the iso")

  def get_packages_from_swp_file(self, file_name):
    '''this function will read the .txt swp and returns the list of packages'''
    file_lines = []
    try:
        f = open(file_name, "r")
        for line in f:
          file_lines.append(line)
        f.close()
    except:
        logger.debug("Failed to read swp {}".format(file_name))
        return []

    lines_len = len(file_lines) 
    pkg_list = []
    
    for i in range(lines_len):
        val = file_lines[i].strip()
        val = val.split(":")   
        if len(val) > 1 and (' pkg ' in val[0] or ' SP ' in val[0]):
            val = val[1]
        else:
            continue
        # pkg 0 rpm 0: ncs5500-sysadmin-hostos-7.0.1.105-r70199I.CSCho99999.admin
        #SP 0: ncs5500-sp4-x-7.0.1.99I
        m = re.search(r'(.*/)*(.*)-(.*)-(.*)\.(.*)', val)
        if m :
            pkg_list.append(val)

    return pkg_list 

  def check_if_system_committed(self):
    ''' this function will verify if the active and committed swp .txt'''
    ''' files have the same packages '''
    # get the list of swp profiles
    cmd  = 'ls /install_repo/local/*swprofile*active*txt'
    out_dict = runcmd(cmd)
    if out_dict['rc']:
        print(-1)
    else:
        output = out_dict['output'].splitlines()
        for x in output:
            logger.debug("for file {}".format(x))
            pkg_list_active = self.get_packages_from_swp_file(x)
            logger.debug(pkg_list_active)
            y = x.replace('active','committed')
            pkg_list_committed = self.get_packages_from_swp_file(y)
            logger.debug(pkg_list_committed)
            if set(pkg_list_active) != set(pkg_list_committed):
               print(1)
               break
        else:
            print(0)

  def get_virt_method (self):
    print (self.virt_method)
    return

  def create_symlinks_for_sp(self,sp_name):
    ''' this creates the symlinks for the sp'''
    file_dir = "/misc/disk1/tftpboot/"
    cal_repo = "/install_repo/gl/calvados/"
    xr_repo = "/install_repo/gl/xr/"
    xr_pkg_repo = "/install_repo/gl/instdb/sdr/default-sdr/pkg/"
    cmd = "ln -sf " + file_dir + sp_name +" " + cal_repo + sp_name 
    runcmd(cmd)
    cmd = "ln -sf " + file_dir + sp_name +" " + xr_repo + sp_name 
    runcmd(cmd)
    cmd = "ln -sf " + xr_repo + sp_name +" " + xr_pkg_repo + sp_name 
    runcmd(cmd)

  def copy_sp_file_create_symlinks(self,platform_prefix, ip,sp_name):
    ''' this function will copy sp_file to other rps and create symlinks'''
    file_dir = "/misc/disk1/tftpboot/"
    cmd = platform_prefix  + "scp " + file_dir + sp_name + " " + ip + ":" + file_dir + sp_name
    runcmd(cmd)
    cmd = platform_prefix + "ssh " + ip + " "\
        "/opt/cisco/calvados/bin/install-functions.py create_symlinks_for_sp " + \
        sp_name 
    runcmd(cmd)
        
  def push_sp_to_other_rps(self,sp_file_name):
    ''' this function will push sp file to other rps'''

    if self.virt_method == "vm":
        platform_prefix = "chvrf 0 "
    else:
        platform_prefix =""

    cmd  = platform_prefix + "/opt/cisco/calvados/bin/show_cmd show vm"
    out_dict = runcmd(cmd)
    if out_dict['rc']:
        sys.exit(1)
    else:
        output = out_dict['output'].splitlines()
        rps_vm_ip = []
        for line in output:
            if "Location:" in line:
                location = line.split()[-1]
            elif "sysadmin" in line and "RP" in location:
                ip = re.findall(r'[0-9]+(?:\.[0-9]+){3}', line)
                rps_vm_ip.append(ip)
        if len(rps_vm_ip) > 1:
            for ip in rps_vm_ip:
                self.copy_sp_file_create_symlinks(platform_prefix,ip[0],sp_file_name)

  def create_sp_symlinks(self):
    ''' this function is used by install update'''
    ''' this will create the sp and create the symlink of that using sp_info.txt'''
    file_dir = "/misc/disk1/tftpboot/"
    sp_name = self.get_sp_name(file_dir)
    if sp_name == None:
        logger.error("Error: could not get sp name")
        sys.exit(1)
    cmd = "mv " + file_dir + "sp_info.txt" + " " + file_dir+sp_name   
    runcmd(cmd)
    self.create_symlinks_for_sp(sp_name)
    self.push_sp_to_other_rps(sp_name)        
  
  def get_platform_name(self):
      if self.platform:
        print(self.platform)

  def validate_sp_rpms(self, iso_mnt_dir):
    '''
        Verify the md5sum of rpms within iso with md5 captured in sp_info
    '''
    with open(os.path.join(iso_mnt_dir, 'sp_info.txt')) as out:
        info = yaml.load(out)
    vm_types = ['sysadmin rpms', 'xr rpms', 'host rpms']

    for vm in vm_types:
        vminfo = info[vm]
        for item in vminfo: 
            logger.info(" {} {} {}".format(item['Boxname'], item['Md5sum'],item['Packagepath'], item['name']))
            exp_md5 = item['Md5sum'].strip()
            box_md5 = get_file_md5(os.path.join(iso_mnt_dir, item['Packagepath'].strip()))
            if exp_md5 != box_md5:
                print("Error: Md5 {} of {} doesn't match with md5 withing SP {}".format(
                            exp_md5, os.path.basename(item['Packagepath']), box_md5))
                print("Error: SP is corrupted")
                sys.exit(1)

  def get_iso_rpms(self, iso_dir, iso_file, vm):
      iso_path = os.path.join(iso_dir, iso_file)
      cmd = "mktemp -d"
      out_dict = runcmd(cmd)
      mnt_dir = out_dict['output']
      cmd = "mount -o loop " + iso_path  + " " + mnt_dir
      runcmd(cmd)
      yml_file = os.path.join(mnt_dir,"iosxr_image_mdata.yml")
      logger.debug(yml_file)
      hints = PkgHintsMdata(yml_file)
      data = get_mdata_for_vm(hints, vm)
      if data:
          print(data['rpms in xr ISO'])
      cmd = "umount " + mnt_dir
      runcmd(cmd)
      shutil.rmtree(mnt_dir, ignore_errors = True)

  def read_giso_rpms_mdata (self, iso, vmtype):
    rpms_giso_vm = []
    hint_tool = "/opt/cisco/calvados/bin/provides_hint.py"
    try:
        if os.path.isfile (iso):
            cmd = "mktemp -d"
            out_dict = runcmd(cmd)
            iso_mnt = out_dict['output']
            cmd = "mount -o loop " + iso + " " + iso_mnt
            runcmd(cmd)
            meta_file = os.path.join(iso_mnt, "iosxr_image_mdata.yml")
            if os.path.isfile (meta_file):
                cmd_iso_rpms = "%s -f %s -r -v %s"%(hint_tool, meta_file, vmtype)
                cmd_giso_rpms = "%s -f %s -r -v %s -g"%(hint_tool, meta_file, vmtype)
                out_dict = runcmd (cmd_iso_rpms)
                rpms_iso = out_dict['output'].split()
                out_dict = runcmd (cmd_giso_rpms)
                rpms_giso_all = out_dict['output'].split()
                # Check for cisco and tp packages.
                rpms_giso_cisco_pkgs = [x for x in rpms_giso_all if self.platform in x]
                rpms_giso_tp_smus = [x for x in rpms_giso_all if self._is_tp_smu (x)]
                rpms_giso_all = rpms_giso_cisco_pkgs + rpms_giso_tp_smus
                rpms_giso_vm = list(set(rpms_giso_all) - set(rpms_iso))
                rpms_giso = map(lambda x: x.replace('.rpm', ''), rpms_giso_vm)
                print (' '.join(rpms_giso))
            cmd = "umount " + iso_mnt
            runcmd(cmd)
            shutil.rmtree(iso_mnt, ignore_errors = True)
    except:
        pass
    finally:
        return rpms_giso_vm    

  def get_sp_rpms(self, sp_name):
      sp_dir = "/install/instmgr_cache/"
      runcmd("ls /install/instmgr_cache") 
      cmd = ["cat", os.path.join(sp_dir, sp_name)]
      logger.debug(cmd) 
      p = Popen(cmd, stdout=PIPE, stderr=PIPE)
      result = p.communicate()
      logger.debug(result) 
      if p.returncode:
          return p.returncode, result
      else:
          rpms = [line.split('Boxname:')[1].strip().rpartition('.')[0] for line in result[0].splitlines() if "Boxname:" in line]
          return p.returncode, set(rpms)
   
   
  def get_sp_rpms_to_deactivate(self, sp_string, outfile):
      inputsp_str, activesp_str = sp_string.split('#')[:2]
      inputsp_set = set(inputsp_str.strip(' :').split(':'))
      activesp_set = set(activesp_str.strip(' :').split(':'))
      with open(outfile, 'w') as fd:
          pass
      sp_dict = {}
      for sp in activesp_set:
          failed = 0
          for i in range(3):
              failed, result = self.get_sp_rpms(sp)
              if not failed:
                  sp_dict[sp] = result
                  break
              else:
                  with open(outfile, 'a+') as fd:
                      fd.write(result)
          if failed:
              sys.exit(failed)
      newactivesp_set = activesp_set - inputsp_set
      newactiverpms_set = set()
      for sp in newactivesp_set:
          newactiverpms_set.update(sp_dict[sp])
      knockoff_sps = []
      todeactivate_sps = []
      for sp in inputsp_set:
          sp_dict[sp] = sp_dict[sp] - newactiverpms_set
          if not sp_dict[sp]:
              knockoff_sps.append(sp)
          else:
              todeactivate_sps.append(sp)
      to_deact_rpms_set = set()
      for sp in todeactivate_sps:
          to_deact_rpms_set.update(sp_dict[sp])
      with open(outfile, 'w') as fd:
          print("\n".join(to_deact_rpms_set), file=fd)
      if knockoff_sps:
          print("\n".join(knockoff_sps))

  def parse_pkg_str(self, pkg_name):
      new_pkg=[]
      logger.info("pkg_name:"+ pkg_name)
      pkg_name = pkg_name.strip().split("\n")
      for x in pkg_name:
          x = x.strip()
          new_pkg.append(x.split(" ")[0])

      print(":".join(new_pkg))

  def parse_sup_pkg(self, pkg_str, pkg_type):
      sup_pkg=[]
      sup_by_pkg=[]
      logger.info("pkg_string:"+ pkg_str)
      pkg_str = pkg_str.strip().split("\n")
      for line in pkg_str:
          line = line.strip().split(" ")
          sup_pkg.append(line[0])
          sup_by_pkg.append(line[-1])
      if(pkg_type == "superseded_pkg"):
          sup_pkg_str = ":".join(sup_pkg)
          logger.info("sup_pkg:"+ sup_pkg_str)
          print(sup_pkg_str)
      if(pkg_type == "superseded_by_pkg"):
          sup_by_pkg_str = ":".join(sup_by_pkg)
          logger.info("sup_by_pkg:"+ sup_by_pkg_str)
          print(sup_by_pkg_str)

  def get_status(self,file,op_id):
      success_msg = "Install operation {} finished successfully".format(op_id)
      aborted_msg = "Install operation {} aborted".format(op_id)
      failed_msg  = "Install operation {} failed".format(op_id)

      status = 0 
      fp = open(file)
      line = fp.readline()
      while line:
          if (re.search(success_msg, line)):
              status = 1
              break
          elif ( re.search(aborted_msg, line) or
                 re.search(failed_msg, line) or
                 re.search("install add operation failed",line)):
              status = 2
              break
          elif re.search("There is nothing to add",line):
              line = fp.readline()
              if re.search("There is nothing to activate",line):
                  status = 1
                  break
          line = fp.readline()
      fp.close()
      print(status)

  def get_activateid(self,file):
      fp = open(file)
      line = fp.readline()
      while line:
          if (re.search("Activate operation ID is",line)):
              line = re.split("for",line)[0]
              act_id = line.split()[-1]
              print(act_id)
              break
          line = fp.readline()
      fp.close()

  def get_error(self,file):
      error_banner = "----------------------------------------------------------------------------"
      fp = open(file)
      space_count = 0;
      error_str = []
      error_found = False
      line = fp.readline()
      while line:
          if( re.search("ERROR",line) or
              re.search("Error",line) or
              re.search("aborted",line) or
              re.search("Failed",line) or
              re.search("error",line)):
              if (re.search("The following package",line) or
                  re.search("Following packages",line)):
                  while line:
                      if (re.search(r'Action \d+',line) or
                          re.search(r'Ending operation',line)):
                          error_found = True
                          break
                      else:
                          new_line = line.strip('\n')
                          error_str.append(new_line)
                      line = fp.readline()
                  if(error_found):
                      break
              else:
                  new_line = line.strip('I\n')
                  error_str.append(new_line)
     
          elif (re.search(error_banner,line)):
              next_line = fp.readline()
              while next_line:
                  if(re.search(error_banner,next_line)):
                      error_found = True
                      break
                  else:
                      next_line = next_line.lstrip('D\t').strip('\n')
                      error_str.append(next_line)
                      next_line = fp.readline() 
              if(error_found):
                  break
          line = fp.readline()

      fp.close()
      print("\n".join(error_str))

  def get_input_pkgs(self,file):
      pkg_list = []
      match_found = False
      no_impact = False
      check_no_impact = True
      break_outer_while = False
      input_pkg = False
      fp = open(file)
      line = fp.readline()

      while line:
          if (re.search("Packages added", line) or
               re.search("Package list", line) or
               re.search("Update packages",line) or
               re.search("Adding packages",line) or
               re.search("Skipped downloading",line) or
               re.search("Packages will be activated",line)):
              check_no_impact = False
              match_found = True

          if(check_no_impact):
             if (re.search("It is a NO IMPACT OPERATION",line)):
                 check_no_impact = False
                 no_impact = True

          if(re.search("Packages skipped",line) and no_impact == False):
              check_no_impact = False
              match_found = False
              input_pkg = True

          if(input_pkg):
              next_line = fp.readline()
              while next_line:
                  if(re.search("Packages added", next_line)):
                      line = fp.readline()
                      while line:
                          if (re.search(r'Action \d+',line)):
                              input_pkg = False
                              break
                          else:
                              pkg_name = re.split(r'\d{2}:\d{2}:\d{2}',line)[-1]
                              if pkg_name.find(':'):
                                  pkg_name = pkg_name.split(':')[-1].strip(' \n\t')
                              else:
                                  pkg_name.strip(' \n\t')
                              pkg_list.append(pkg_name)
                              line = fp.readline()

                      if(input_pkg == False):
                          break_outer_while = True
                          break
                  else:
                      pkg_name = re.split(r'\d{2}:\d{2}:\d{2}',next_line)[-1]
                      if pkg_name.find(':'):
                          pkg_name = pkg_name.split(':')[-1].strip(' \n\t')
                      else:
                          pkg_name.strip(' \n\t')
                      pkg_list.append(pkg_name)
                      next_line = fp.readline()


          if(no_impact):
              if re.search("Packages skipped",line):
                  next_line = fp.readline()
                  while next_line:
                      if (re.search(r'Action \d+',next_line)):
                          no_impact = False
                          break
                      else:
                          pkg_name = re.split(r'\d{2}:\d{2}:\d{2}',next_line)[-1]
                          if pkg_name.find(':'):
                             pkg_name = pkg_name.split(':')[-1].strip(' \n\t')
                          else:
                              pkg_name.strip(' \n\t')
                          pkg_list.append(pkg_name)
                          next_line = fp.readline()

          if(match_found) :
              next_line = fp.readline()
              while next_line:
                  if (re.search(r'Action \d+',next_line) or
                      re.search(r'Activating package',next_line) or
                      re.search(r'Following packages',next_line) or
                      re.search(r'Fetching',next_line) or
                      re.search(r'Content of the SP will be',next_line) or
                      re.search(r'There is nothing to add',next_line) or
                      re.search(r'Install add',next_line) or
                      re.search(r'Due to this operation',next_line) or
                      re.search(r'Skipped downloading',next_line)):
                      break_outer_while = True
                      break
                  else:
                      pkg_name = re.split(r'\d{2}:\d{2}:\d{2}',next_line)[-1]
                      if pkg_name.find(':'):
                          pkg_name = pkg_name.split(':')[-1].strip(' \n\t')
                      else:
                          pkg_name.strip(' \n\t')
                      if(len(pkg_name) > 5):
                          pkg_list.append(pkg_name)
                      next_line = fp.readline()
          if(break_outer_while):
              break
          else:
              line = fp.readline()
      fp.close()
      print(":".join(pkg_list))

  def get_noimpact_skipped_pkgs(self,file):
      fp = open(file)
      line = fp.readline()
      pkg_list = []
      while line:
          if(re.search("It is a NO IMPACT OPERATION",line)):
              next_line = fp.readline()
              if re.search("Packages skipped",next_line):
                  new_line = fp.readline()
                  while new_line:
                      if (re.search(r'Action \d+',new_line)):
                          break
                      else:
                          pkg_name = re.split(r'\d{2}:\d{2}:\d{2}',new_line)[-1]
                          if pkg_name.find(':'):
                              pkg_name = pkg_name.split(':')[-1].strip(' \n\t')
                          else:
                              pkg_name.strip(' \n\t')
                          pkg_list.append(pkg_name)
                          new_line = fp.readline()
              else:
                  fp.seek(0,0)
                  new_line = fp.readline()
                  while new_line:
                      if(re.search("Package list",new_line)):
                          next_line = fp.readline()
                          while next_line:
                              if (re.search(r'Action \d+',next_line) or
                                  re.search(r'Following packages are skipped',next_line)):
                                  break
                              else:
                                 pkg_name = re.split(r'\d{2}:\d{2}:\d{2}',next_line)[-1]
                                 if pkg_name.find(':'):
                                     pkg_name = pkg_name.split(':')[-1].strip(' \n\t')
                                 else:
                                     pkg_name.strip(' \n\t')
                                 pkg_list.append(pkg_name)
                                 next_line = fp.readline()
                      new_line = fp.readline()
          line = fp.readline()
      fp.close()
      return(":".join(pkg_list))

  def get_skip_pkgs(self,file):
      fp = open(file)
      line = fp.readline()
      match_found = False
      pkg_list = []
      while line:
          if (re.search("Packages skipped",line) or
              re.search("Skipped packages",line) or
              re.search("Skipped downloading",line) or
              re.search("packages are skipped",line)):
              match_found = True
          if(match_found):
              next_line = fp.readline()
              while next_line:
                  if(re.search("Packages added",next_line) or
                     re.search("Install operation",next_line) or
                     re.search(r'Action \d+',next_line) or
                     re.search(r'Adding packages',next_line) or
                     re.search(r'There is nothing to add, skipping install add operation',next_line) or
                     re.search(r'The prepared software is set to be',next_line) or
                     re.search(r'The software will be deactivated',next_line)):
                      match_found = False
                      break
                  else:
                      pkg_name = re.split(r'\d{2}:\d{2}:\d{2}',next_line)[-1].strip(' ')
                      if(pkg_name.find(':')):
                           pkg_name = pkg_name.split(':')[-1].strip(' \n\t')
                      else:
                          pkg_name = pkg_name.strip(' \n\t')
                      pkg_list.append(pkg_name)
                      next_line = fp.readline()
          line = fp.readline()
      fp.close()
      return(":".join(pkg_list))

  def get_skipped_pkgs(self,file):
      fp = open(file)
      line = fp.readline()
      no_impact = False
      while line:
          if re.search("It is a NO IMPACT OPERATION",line):
              no_impact = True
              break
          line = fp.readline()
      fp.close()
      if(no_impact):
          pkg_str = self.get_noimpact_skipped_pkgs(file)
      else:
          pkg_str = self.get_skip_pkgs(file)
      print(pkg_str)

  def admin_yang_edm_data_parse(self, pkgstrs="", superseded=0):
       '''
         Parse input string to get name,version and release
         from the admin package strings.

         Args:
             pkgstrs: strings of package names
         Returns: 
             outpkgs : formatted strings in form name:version:release#
       '''
       in_packages = pkgstrs.split('\n')
       packages = list(set(in_packages))
       outpkgs = ""
       label = ""
       for pkg in range(0, len(packages)):
           if not packages[pkg].strip():
              continue
           if "[Boot image]" in packages[pkg]:
              if int(superseded) == 1:
                 continue
              bootimg = packages[pkg].strip()
              strlist = bootimg.rsplit('-', 1)
              name = strlist[0]
              versionrel = strlist[1]
              temp = versionrel.split('=', 1)
              version = temp[0].split(' ')[0]
              release = temp[1].replace('.', "")
              logger.info("Name: %s Version: %s Release: %s" %(name, version, release))
              outpkgs = outpkgs + name + ':' + version + ':' + release + '#'

           elif "is  Node" in packages[pkg]:
               node = packages[pkg].strip()
               nodename = node.rsplit('e', 1)[1]
               outpkgs= outpkgs+ nodename+'|'

           elif "Inactive Packages:" in packages[pkg]:
                continue
           elif "Node" in packages[pkg]:
                continue
           elif ("-mini-"  in packages[pkg]) or ("-r" not in packages[pkg]) or\
                ("-golden" in  packages[pkg]):
               try:
                   minip = packages[pkg].strip()
                   label = ""
                   if "-golden" in minip:
                      minip = minip.rsplit('-', 1)
                      mini = minip[0]
                      label = "."+minip[1]
                   else:
                      mini = minip

                   strlist =mini.rsplit('-', 1)
                   name = strlist[0]
                   version = strlist[1]
                   release = "r"+str(version.replace('.', ""))+str(label)
                   outpkgs = outpkgs + name + ':' + version + ':' + release + '#'
               except:
                  pass

           else:
               rpmname = packages[pkg].strip()
               if rpmname:
                  strlist = rpmname.rsplit('-', 1)
                  release = strlist[1]
                  namever = strlist[0].rsplit('-', 1)
                  name = namever[0]
                  version = namever[1]
                  logger.info("Name: %s Version: %s Release: %s" %(name, version, release))
                  outpkgs = outpkgs + name + ':' + version + ':' + release + '#'

       print(outpkgs)



def mask_command(cmd_args):
    proto = ( "sftp", "ftp", "http", "https", "scp")
    cmd = []
    for cmd_arg in cmd_args:
        if not cmd_arg.startswith(proto):
            cmd.append(cmd_arg)
        else:
            try:
                r = urlparse(cmd_arg)
                r2 = r._replace(netloc="{}:{}@{}".format(
                        r.username, "*"*len(r.password), r.hostname))
                cmd.append(r2.geturl())
            except:
                cmd.append(cmd_arg)
    return cmd

if __name__ == '__main__':
  '''
    Generic python script for various install functions.

    First argument should be function name. 
    Rest of the argument should be the arguments for function passed as first 
    argument.
    In main, first logger is intialized and then respective functions are 
    called after instantiating Install class.
  '''

  # create logger
  logger = logging.getLogger('install_functions_logger')
  logger.setLevel(logging.DEBUG)
  formatter = MyFormatter('%(asctime)s  #%(process)d:: %(message)s',
                                '%Y-%m-%d %H:%M:%S.%f')

  # Logs to logfile
  fh = logging.handlers.RotatingFileHandler(LOG_FILE, 
                                            maxBytes=(1024*10000), 
                                            backupCount=2)
  fh.setLevel(logging.DEBUG)
  fh.setFormatter(formatter)
  logger.addHandler(fh)

  logger.debug('+'*80)
  cmd = mask_command(sys.argv)
  logger.debug(' '.join(cmd))

  inst= Install()
  if not hasattr(inst, sys.argv[1]):
    logger.error('No function name ' + sys.argv[1] + ' in ' + sys.argv[0])
    sys.exit(1)

  function = getattr(inst, sys.argv[1])
  if len(sys.argv) > 2:
    args_passed=sys.argv[2:]
    function(*args_passed)
  else:
    function()


