#! python
import sys
sys.dont_write_bytecode = True
import errno
import os
import utils.inst_utils as utils
import utils.iso_params as iso_info
import constants as C
import utils.errors as error
from utils import get_logger

patch_vmtype = {
            C.XR : "xr-vm",
            C.XR_LCP : "xr-vm",
            C.HOST : "hostos",
            C.CALVADOS : "sysadmin-vm"
        }

def prepare_partition_with_iso (dev_mnt, iso_mnt, vmtype, lvindex, repo, rpmlist, plat, device):
    buildtimelist = []
    excl_cards_gl = []
    excl_cards_gl_notp = []
    excl_cards = []
    utils.extract_initrd_to_part (iso_mnt, dev_mnt)
    try:
        import ctypes
        import time
        libc = ctypes.CDLL("libc.so.6")
        libc.sync()
        time.sleep (1)
    except:
        pass
    # Copy iso_info.txt for future references.
    iso_info_file = os.path.join (iso_mnt, C.ISOINFO)
    iso_info_file_dest = os.path.join (dev_mnt, C.ISOINFO)
    if os.path.isfile (iso_info_file):
        import shutil
        shutil.copy (iso_info_file, iso_info_file_dest)
    
    utils.prepenv_mount_sysdirs (dev_mnt, iso_mnt, C.rpm_path[vmtype], repo)
    card = utils.get_install_board_type (vmtype)
    utils.run_pre_install_patches (iso_mnt, dev_mnt, card, C.vm_type[vmtype])
    excl_cards_gl = utils.install_rpm_to_part (dev_mnt, vmtype, "rpm")
    if rpmlist:
        excl_cards_gl_notp, buildtimelist = utils.install_additional_rpms_to_part (dev_mnt, vmtype, "pkgs", plat)
    excl_cards = list(set().union(excl_cards_gl, excl_cards_gl_notp))
    utils.copy_ssh_keys (dev_mnt, vmtype)
    utils.update_symlinks (dev_mnt, vmtype, excl_cards)
    utils.prepenv_umount_sysdirs (dev_mnt)
    utils.prepare_grub_setup (dev_mnt, device, patch_vmtype[vmtype], vmtype)
    utils.chroot_support_mounts (dev_mnt)
    utils.run_post_install_patches (dev_mnt, lvindex, patch_vmtype[vmtype], vmtype)
    utils.chroot_support_umounts (dev_mnt)
    utils.set_owner_group (dev_mnt)
    return buildtimelist

def download_pkg_from_repo (pkg, repo, repoIp, arch, card):
    arch_message = "No such file or directory"
    retry_message = "Connection reset by peer"
    retry_count = 0
    pkg_downloaded = ''
    try:
        if not os.path.isdir (repo):
            utils.run_cmd ("mkdir -p %s"%(repo))
    except:
        msg = "Unable to create temporary repository: %s"%(repo)
        utils.send_err_msg_to_sys_orch(msg)
        raise
    
    cmd = "scp %s:%s %s"%(repoIp, pkg, repo)
    while 1:
        try:
            utils.run_cmd (cmd, True)
            pkg_name = os.path.basename (pkg)
            pkg_downloaded = os.path.join (repo, pkg_name)
        except RuntimeError as e:
            if arch_message in e.message:
                pkg = '%s.%s'%(pkg, arch)
                cmd = "scp %s:%s %s"%(repoIp, pkg, repo)
                continue
            else: 
                if retry_count == 10:
                    raise
                    break
                retry_count += 1
                logger.debug ('Had an ssh disconnect for %s, retrying download'%(pkg))
                import time
                time.sleep(1)
                continue
        else:
            logger.debug ('New package is %s'%(pkg_downloaded))
            break
    # Check if we should update the repo with this package downloaded.
    if not os.path.isfile (pkg):
        pkg_repo = os.path.dirname (pkg)
        if os.path.isdir (pkg_repo):
            if not os.path.isfile (pkg):
                try:
                    import shutil
                    # Copy downloaded package to install repo.
                    shutil.copy (pkg_downloaded, pkg)
                except RuntimeError:
                    # is this ignorable error??
                    #utils.send_err_msg_to_sys_orch(msg)
                    logger.debug ('Unable to copy downloaded package to repo')
    return pkg_downloaded

# TBD: This should be a part of PD utility.
def copy_partfiles_to_repo (dev_mnt, vmtype, version):
    # Check if repo exists on this node.
    import shutil
    if not os.path.isdir (C.CAL_REPO):
        return 

    if vmtype == C.HOST and os.path.isdir (C.MISC_REPO):
        grub_cfg_path = os.path.join (dev_mnt, "boot/grub2", "grub.cfg")
        grub_efi_path = os.path.join (dev_mnt, "boot/grub2", "bootx64.efi")
        grub_cfgrepo_file = os.path.join (C.MISC_REPO, "grub.cfg.tmp")
        grub_efirepo_file = os.path.join (C.MISC_REPO, "grub.efi")
        if os.path.isfile (grub_cfg_path):
            shutil.copy (grub_cfg_path, grub_cfgrepo_file)
        if os.path.isfile (grub_efi_path):
            shutil.copy (grub_efi_path, grub_efirepo_file)
        # Change tftpboot pxe grub parameters and run platform grub update script.
        signstring = ''
        with open (grub_cfg_path) as fp:
            buf = fp.read ()
            signstring = buf[buf.find('signfile'):].split ('\n')[0]
        cmd = "sed -i -e \"/search.*/ d\" "\
            "-e \"s;root=<P.*>;root=/dev/ram install=/dev/sda;\" "\
            "-e \"s;from Disk;from PXE;\" "\
            "-e \"s;quiet;quiet installiso=system_image.iso;\" "\
            "-e \"s;initrd .*;initrd /boot/initrd.img %s;\" %s"%(signstring, grub_cfgrepo_file)
        try:
            utils.run_cmd (cmd)
        except:
            pass
        pxeboot_pd_grub_upd = "/etc/rc.d/init.d/platform_install_update_pxeboot_grub.sh"
        if os.path.isfile (pxeboot_pd_grub_upd):
            cmd = "%s %s"%(pxeboot_pd_grub_upd, grub_cfgrepo_file)
            try:
                utils.run_cmd (cmd)
            except:
                pass    

    if 'arm' in utils.read_bootstrap_supported_arch ():
        if vmtype == C.HOST:
            nbi_file_path = os.path.join (dev_mnt, "opt/cisco/hostos/bin", "ncs5500-sysadmin-image.nbi")
            nbi_repo_file = os.path.join (C.CAL_REPO, "%s-%s"%("ncs5500-sysadmin-image.nbi", version))
            if os.path.isfile (nbi_file_path):
                shutil.copy (nbi_file_path, nbi_repo_file)
    return                   

if __name__ == "__main__":
    mode = sys.argv[1]
    oper_id = sys.argv[2]
    vmtype = sys.argv[3]
    lvindex = sys.argv[4]
    device = sys.argv[5]
    tmp_staging = sys.argv[6]
    repoIp = sys.argv[7]
    repo = sys.argv[8]
    pkglist = [os.path.join(repo, x) for x in sys.argv[9].split(',')]
    iso = ""
    rpmlist = []
    copy_dir = ''
    iso_mnt = ''
    dev_mnt = ''
    cmd_platname = None
    logdir = os.path.join (C.LOGDIR, 'prep_ids', oper_id)

    try:
        import platform
        arch = platform.processor()

        if 'x86_64' in arch:
            arch = 'x86_64'
        elif 'arm' in arch:
            arch = 'arm'

        try:
            os.makedirs (logdir)
        except OSError as e:
            if e.errno != errno.EEXIST:
                print ("Unable to create logdir %s"%(logdir))
                pass

        try:
            logfile = os.path.join (logdir, 'prep_%s'%(vmtype))
            os.environ['MP_LOG_FILE'] = logfile
            logger = get_logger(logfile)
        except:
            print ("Unable to initialize logger for logfile %s"%(logfile))
            del os.environ['MP_LOG_FILE']
            logger = get_logger()
            pass

        if not os.path.isdir (tmp_staging):
            tmp_staging = "/install_repo/tmp"
        if not os.path.isdir (tmp_staging):
            msg = "Staging location %s is not available"%(tmp_staging)
            utils.send_err_msg_to_sys_orch(msg)
            raise Exception ("Staging location not available")
        prep_staging = os.path.join (tmp_staging, "tmpXXX")
        copy_dir = utils.run_cmd ("mktemp -d %s"%(prep_staging))
        card = utils.get_install_board_type (vmtype)
        for pkg in pkglist:
            pkg2consider = download_pkg_from_repo (pkg, copy_dir, repoIp, arch, card)
            if os.path.isfile (pkg2consider):
                file_type = utils.get_file_type (pkg2consider)    
                if file_type == C.ISO:
                    iso = pkg2consider
                elif file_type == C.RPM:
                    rpmlist.append (pkg2consider)    

        iso_mnt = utils.mount_iso (iso)
        offset = utils.get_rootfs_offset (device)
        dev_mnt = utils.mount_device (device, offset)
        plat, host_iso, cal_iso, xr_iso, iso_version = utils.read_iso_info_mdata (iso_mnt)
        try:
            buildtimelist = prepare_partition_with_iso (dev_mnt, iso_mnt, vmtype, lvindex, copy_dir, rpmlist, plat, device)
        except:
            msg = "Error in preparing partition of %s vm"%(vmtype)
            utils.send_err_msg_to_sys_orch(msg)
            raise
        # With partitioning done, copy out any PD specific files needed from created partition to repo, 
        # if applicable.
        copy_partfiles_to_repo (dev_mnt, vmtype, iso_version)
            
        if vmtype != C.HOST:
            iso_info_obj = iso_info.IsoInfo (iso_mnt, C.rpm_path[vmtype], rpmlist, buildtimelist)
            iso_info_obj.create_prepare_swp (iso, lvindex, vmtype, C.LOCALSWP)
        utils.update_prep_lvindex (vmtype, lvindex)
        if vmtype == C.CALVADOS:
            cmd_platname = "chroot %s /opt/cisco/calvados/bin/install-functions.py get_platform_name"%(dev_mnt)
        elif vmtype == C.XR or vmtype == C.XR_LCP:
            cmd_platname = "chroot %s /pkg/bin/install-functions.py get_platform_name"%(dev_mnt)

        if cmd_platname:
            plat2check = ''
            try:
                utils.run_cmd ("mkdir -p %s"%(os.path.join(dev_mnt, "var/log/install")))
                logger.debug ("Sanitize command to run %s"%(cmd_platname))
                try:
                    plat2check = utils.run_cmd (cmd_platname)
                except:
                    utils.partition_remove_corrupt_pyc (dev_mnt)
                    plat2check = utils.run_cmd (cmd_platname)
            except:
                # Log a message and proceed.
                logger.error ("Unable to run santize command. OK to ignore in case of downgrade.")
                pass
            if plat2check == plat:
                logger.info ("Succesfully verified pyc files being intact")
    except:
        import traceback
        exc_info = sys.exc_info()
        TB = traceback.format_exc()
        logger.error(TB)
        sys.exit (-1)
    finally:
        utils.umount_iso (iso_mnt)
        if os.path.ismount (dev_mnt):
            try:
                utils.run_cmd ("umount -Rd %s"%(dev_mnt))
                utils.run_cmd ("rm -rf %s"%(dev_mnt))
            except:
                import ctypes
                import time
                libc = ctypes.CDLL("libc.so.6")
                libc.sync()
                time.sleep(5)
                try:
                    # Auditd starts running on dev_mnt causing umount failures.
                    # Shouldnt matter to us.
                    utils.run_cmd ("umount -Rd %s"%(dev_mnt))
                    utils.run_cmd ("rm -rf %s"%(dev_mnt))
                except:
                    pass
        if os.path.isdir (copy_dir):
            utils.run_cmd ("rm -rf %s"%(copy_dir))
