#!/usr/cisco/bin/python
#-----------------------------------------------------------------------
# install_rel_notes.py - Script used by install to check major changes
#                        conflicts are present between two release notes
# 
# This script takes in two JSON formatted release notes as inputs and 
# checks if there is any major change between the two files. If yes,
# prints an error message and returns a non-zero value to the caller.
# 
# Copyright (c) 2017 by cisco Systems, Inc.
# All rights reserved.
#-----------------------------------------------------------------------
#
import time
import json
import sys

MAJOR_DIFF_MESSAGE = "\nIncompatible sysadmin / sdr versions. In order to proceed with install safely,\n" + \
                      "you must install a compatible version on sysadmin first and then install this\n" + \
                      "version all SDRs.\n\nList of changes are :\n"

MINOR_DIFF_MESSAGE_SA = "\nSome functionality might not work as expected because of minor changes between\n" + \
                      "sysadmin and sdr versions. To ensure all functionality works as expected, please\n" + \
                      "install a compatible version on SDRs.\n\nList of changes are : \n"

MINOR_DIFF_MESSAGE_SDR = "\nSome functionality might not work as expected because of minor changes between\n" + \
                      "sysadmin and sdr versions. To ensure all functionality works as expected, please\n" + \
                      "install a compatible version on Sysadmin.\n\nList of changes are : \n"

LINE_BREAK = "\n-----------------------------\n"

def dumper(obj):
    return obj.__dict__

# Version : Class to keep track of version numbers of the 
# entries in Release Notes. As of now, this class just uses 
# the timestamp at which the entries are added. Note that 
# the same timestamp is also used in the file XR_GuardRails.txt,
# which is added to the bug's enclosure list.
class Version(object):
    def __init__(self):
        self.timestamp = time.strftime("%Y-%m-%dT%H:%M:%S",time.gmtime())
    def __str__(self):
        return self.timestamp

# RevEntry : Class to keep track of the entries in Release Notes.
# It is expected that the SDK-RELEASE-NOTES will be in JSON format.
class RevEntry(object):
    def __init__(self, l):
        if l is not None:
            try:
                self.__dict__ = json.loads(l)
            except ValueError, e:
                print "\nError: JSON error: SDK-RELEASE-NOTES not in JSON format"
                sys.exit(-1)
            tmp = self.ver
            self.ver = Version()
            self.ver.timestamp = tmp['timestamp']
        else:
            self.ver = Version()
            self.ddts = ""
            self.ddts_description = ""
            self.branch_tag = ""
            self.userid = ""
            self.commit_log = ""
            self.major_change = False

    def __str__(self):
        return str(self.ver) + ", " + self.ddts + ", " + self.ddts_description + ", " + self.userid + ", " + \
        self.branch_tag + ", " + self.commit_log + ", " + str(self.major_change)
         
# returns hash table of release notes with ddts as key
def parse_sdk_release_notes(rel):
   entries = {}
   with open(rel, "r") as f:
        for l in f:
            e = RevEntry(l.strip())
            entries[e.ddts] = e 
   return entries

# Returns a list of 2 lists.
# index 0 is the list of entries in e1, but not found in e2
# index 1 is the list of entries in e2, but not found in e1
def compare_rel_notes(e1, e2):
    return [get_diff_list(e1,e2), get_diff_list(e2,e1)]

# returns a list of entries in e1 which are not found in e2
# This is a one direction diff: entries in e2 not found in e1 will not be returned by this function.
def get_diff_list(e1,e2):
    l = []
    for d in e1.keys():
        if d not in e2.keys():
           l.append(e1[d])
    return l

# Helper function to return '\n' separated string of all DDTSses found in the input var 'list'
def get_all_ddts(list):
    rv = ""
    for entry in list:
        rv = rv + str(entry.ddts) + "\n"
    return rv

# Helper function to extract out the top 'k' commit logs from the input var 'list'
# If 'list' does not contain 'k' entries, return the commit logs for the number of
# entries present in the list
def extract_top_k_commit_logs(list,k):
    rv = ""
    len_list = len(list)
    for i in range(k):
        if i >= len_list:
            break
        rv = rv + list[i].commit_log + "\n"
    return rv

if __name__ == "__main__":
    if len(sys.argv) < 4 :
        print "Usage : install_rel_notes.py <SDK-RELEASE-NOTES-1> <SDK-RELEASE-NOTES-2> <is_calvados_upgrade>\n"
        sys.exit(-1)
      
    rel1 = sys.argv[1]
    rel2 = sys.argv[2]
    is_calvados_upgrade = sys.argv[3]
    entries1 = parse_sdk_release_notes(rel1)
    entries2 = parse_sdk_release_notes(rel2)
    difflist = compare_rel_notes(entries1, entries2)
    major_change_list = []

    # list of changes between calvados and XR release notes. This list contains
    # those entries which are found in the calvados Release notes but not found in
    # XR release notes
    minor_change_list_cx = []
    
    # list of changes between XR and calvados release notes. This list contains
    # those entries which are found in the XR Release notes but not found in
    # calvados release notes
    minor_change_list_xc = []

    for e in difflist[0]:
        if e.major_change is True:
            major_change_list.append(e)
        else:
            minor_change_list_cx.append(e)

    for e in difflist[1]:
        if e.major_change is True:
            major_change_list.append(e)
        else:
            minor_change_list_xc.append(e)

    rv = 0
    output_message = ""

    # If major changes are found, print appropriate message and set
    # return value correspondingly
    if len(major_change_list) > 0:
        output_message = MAJOR_DIFF_MESSAGE + LINE_BREAK + get_all_ddts(major_change_list) + get_all_ddts(minor_change_list_xc) + \
                            LINE_BREAK + extract_top_k_commit_logs(major_change_list + minor_change_list_xc,2) + LINE_BREAK
        rv = 1

    # Print the appropriate messages only if there are minor changes in XR which are not present in Calvados. We do not
    # print the warning messages in the other case (calvados has some minor changes not present in XR) because that is
    # an expected / normal situation and we do not want to flood the logs with misleading messages.
    elif len(minor_change_list_xc) > 0:
        if is_calvados_upgrade == 1:
            output_message = MINOR_DIFF_MESSAGE_SA
        else:
            output_message = MINOR_DIFF_MESSAGE_SDR

        output_message = output_message + LINE_BREAK + get_all_ddts(minor_change_list_xc) + \
                            LINE_BREAK + extract_top_k_commit_logs(minor_change_list_xc,2) + LINE_BREAK

    # This output will be captured by the script install-functions.sh and printed in the logs
    print output_message

    sys.exit(rv)
