#! python

import sys
sys.dont_write_bytecode = True
import utils.inst_utils as utils
import constants as C
import os
import traceback
from utils import get_logger
def set_logger():
    if os.environ.get('MP_LOG_FILE'):
        return get_logger(os.environ.get('MP_LOG_FILE'))
    else:
        return get_logger()

sdr_vms = [C.XR, C.XR_LCP]

def BootLvCounter():
    for vm in C.SUPPORTED_VMTYPE:
        if vm in sdr_vms:
            lvfile = os.path.join (C.LOCALREPO, "sdr/default-sdr", "%s_lv_state.txt"%(vm))
        else:
            lvfile = os.path.join (C.LOCALREPO, "%s_lv_state.txt"%(vm))
        yield vm, lvfile

class Qsu_lv_utils:
    def __init__(self):
        self.lv_noremove = []
        self.logger = set_logger()

        for vm_type, lvfile in BootLvCounter():
            if not os.path.isfile (lvfile):
                continue
            cmd = "cat %s"%(lvfile)
            try:
                lv_data = utils.run_cmd(cmd).split()
                if self.validate_lv_data(lv_data):
                    self.lv_noremove.append (os.path.join('/dev', 'panini_vol_grp', '%s_lv%s'%(vm_type, lv_data[1])))
            except RuntimeError as e:
                self.logger.error ('Active and committed validation failed for %s'%(lvfile))
                raise e

    def validate_lv_data (self, lv_data):
        '''
        Check that active and committed are same.
        '''
        active_lv = lv_data[0]
        commit_lv = lv_data[1]
        if active_lv != commit_lv:
            msg = "Active and committed softwares are different. Please retry after performing \"install commit\" operation."
            utils.send_err_msg_to_sys_orch(msg)
            raise RuntimeError("Error: Active and committed LVs are different\n")

        return True

    def remove_unwanted_rootfs (self):
        '''
        Consider removal of only unwanted rootfs partitions in panini_vol_grp.
        '''
        self.logger.debug('LVs which are non-removable %s'%(' '.join(self.lv_noremove)))
        cmd = "lvdisplay -v panini_vol_grp | grep -e 'LV Path' | awk '{print $3}'"
        lvs2consider = utils.run_cmd_on_host (cmd).split()
        for lv in self.lv_noremove:
            if lv in lvs2consider:
                lvs2consider.remove (lv)

        lv_remove = [x for x in lvs2consider if not 'data' in x]
        for lv in lv_remove:
            try:
                utils.detach_volume (lv)
            except:
                pass
            
            try:
                utils.delete_volume (lv)
            except:
                self.logger.info('Error: Unable to remove %s, but proceeding with operation\n'%(lv))    
                pass


def get_disk_free_space(path):
    if not path:
        raise ValueError ("Missing input parameters for space checks")

    if not os.path.exists(path):
        raise IOError ("Path %s doesn't exist" % (path))

    space_data = os.statvfs(path)
    if space_data:
        free_space = space_data.f_frsize*space_data.f_bavail
        #converting to MB, not adding 1 as it is for available free space
        free_space = free_space >> 20
        return free_space

class DiskSpaceCheck:
    def __init__(self, tmp_staging, total_pkg_size=0, initrd=''):
        self.tmp_staging = tmp_staging
        self.total_pkg_size = total_pkg_size
        self.initrd = initrd

    def verify_disk_space(self):
        if self.total_pkg_size == 0:
            return

        staging_free_space = get_disk_free_space(self.tmp_staging)

        if staging_free_space > self.total_pkg_size:
            logger.info("verify_disk_space: Sufficient space to copy packages to staging location")
            logger.info("verify_disk_space: Staging location = %s Free Space = %d MB"%(self.tmp_staging, staging_free_space))
            logger.info("verify_disk_space: Total Package Size = %d MB"%(self.total_pkg_size))        
        else:
            msg = "Insufficient space for staging location %s. Staging Free Space : %d MB, Total Package Size : %d MB"%(self.tmp_staging, staging_free_space, self.total_pkg_size)
            utils.send_err_msg_to_sys_orch(msg)
            raise IOError(msg)

    def verify_initrd_extraction_space(self):
        if self.initrd == '':
            return

        staging_free_space = get_disk_free_space(self.tmp_staging)

        initrd_size = os.path.getsize(self.initrd)
        initrd_size = (initrd_size >> 20) + 1 #converting to MB
        cmd  = "file %s"%(self.initrd)
        out  = utils.run_cmd(cmd)

        uncompress_ratio = 1.3  #considering max initrd compression for estimation
        actual_uncompress_size_cmd = ''

        if 'ASCII' in out:
            uncompress_ratio = 1
        elif 'gzip compressed data' in out:
            actual_uncompress_size_cmd = 'gzip -l %s | awk \' NR > 1  {print $2}\''%(self.initrd)


        est_uncompress_size = initrd_size * uncompress_ratio 

        if staging_free_space > est_uncompress_size:
            logger.info("verify_initrd_extraction_space: Sufficient space to extract initrd %s to staging location"%(self.initrd))
            logger.info("verify_initrd_extraction_space: initrd Size = %d MB, estimated uncompressed initrd size = %d MB"%(initrd_size, est_uncompress_size))
            logger.info("verify_initrd_extraction_space: Staging location = %s Free Space : %d MB"%(self.tmp_staging, staging_free_space))
        else:
            if staging_free_space > initrd_size and actual_uncompress_size_cmd != '':
                act_uncompress_size = utils.run_cmd(actual_uncompress_size_cmd)
                act_uncompress_size = int(act_uncompress_size) * 1.05
                act_uncompress_size = (int(act_uncompress_size) >> 20) + 1

                if staging_free_space > act_uncompress_size:
                    logger.info("verify_initrd_extraction_space: Sufficient space to extract initrd %s to staging location"%(self.initrd))
                    logger.info("verify_initrd_extraction_space: initrd Size = %d MB, actual uncompressed initrd size = %d MB"%(initrd_size, act_uncompress_size))
                    logger.info("verify_initrd_extraction_space: Staging location = %s Free Space : %d MB"%(self.tmp_staging, staging_free_space))
                else:
                    msg = "Insufficient space for %s extraction in staging location %s. Staging Free Space : %d MB, initrd Size : %d MB, Actual uncompressed initrd size : %d MB"%(self.initrd, self.tmp_staging, staging_free_space, initrd_size, act_uncompress_size)
                    utils.send_err_msg_to_sys_orch(msg)
                    raise IOError(msg)
            else:
                msg = "Insufficient space for %s extraction in staging location %s. Staging Free Space : %d MB, initrd Size : %d MB, Estimated uncompressed initrd size : %d MB"%(self.initrd, self.tmp_staging, staging_free_space, initrd_size, est_uncompress_size)
                utils.send_err_msg_to_sys_orch(msg)
                raise IOError(msg)



import argparse

if __name__ == '__main__':

    parser = argparse.ArgumentParser(description='Utility related to disk related handling')
    #sys.argv[1] should always be SYSTEM/NODE which will be used to initialize logger
    parser.add_argument(dest='mode', type=str, help='Either system or node scope')
    parser.add_argument('--lvclean', dest = 'LVCleanup', action='store_true', required=False, default=False, 
                        help = 'Clean LVs which are not needed')
    parser.add_argument('--staging_loc', dest = 'stageLoc', type = str, help = 'Temporary Staging location to copy files')
    parser.add_argument('--pkg_size', dest = 'pkgSize', type = int, help = 'Total size of files to be copied to temp staging location')
    parser.add_argument('--initrd', dest = 'initrd', type = str, help = 'initrd to be extracted in staging location')
   
    args = parser.parse_args()

    if args.stageLoc is not None:
        if (args.pkgSize is None and args.initrd is None) or (args.pkgSize is not None and args.initrd is not None) :
            logger.error("Incorrect arguments passed to lvutils.py")
            logger.error("Arguments: %s"%(str(sys.argv))) 
            raise ValueError("Incorrect arguments passed")

    try:
        logger = set_logger()
        if args.LVCleanup:
            lvobj = Qsu_lv_utils ()
            lvobj.remove_unwanted_rootfs ()
        if args.stageLoc is not None and args.pkgSize is not None:
            dsc = DiskSpaceCheck(args.stageLoc, total_pkg_size=args.pkgSize)
            dsc.verify_disk_space()
        if args.stageLoc is not None and args.initrd is not None:
            dsc = DiskSpaceCheck(args.stageLoc, initrd=args.initrd)
            dsc.verify_initrd_extraction_space()
    except:
        exc_info = sys.exc_info()
        TB = traceback.format_exc()
        logger.error(TB)
        sys.exit (-1)


