#! /usr/bin/env python
# Copyright (c) 2015-2016, 2019 by cisco Systems, Inc.

import re
import sys
import os
import optparse
import commands
import shutil
import logging
import logging.handlers
import traceback
from utils import get_logger
logger = get_logger()

this_path = os.path.dirname(__file__)
gen_rpm_tool = this_path + "/get_rpms.py"
install_repo = "/install_repo/gl/" 
multi_sdr_instdb = "/install_repo/gl/instdb/sdr/" 

def run_cmd(cmd, ret):
    status, text = commands.getstatusoutput(cmd)
    exit_code = status >> 8 # high byte
    if not exit_code:
        if "string" in ret:
            return text.strip()
        else:
            return text.strip().split(' ')
    else:
        logger.error("Unable to run cmd %s"%(cmd))
        sys.exit(-1)

# The RPMs are always in the format <name>-<version>-<release>.<arch>.rpm
# e.g:
# CISCO XR Pkg   : xrv9k-iosxr-infra-2.0.0.0-r60013I.x86_64.rpm
# CISCO XR SMU   : xrv9k-iosxr-infra-2.0.0.1-r60013I.CSCua00003.x86_64.rpm
# CISCO Cal Pkg  : xrv9k-sysadmin-system-6.0.0.13I-r60013I.x86_64.rpm
# CISCO Cal SMU  : xrv9k-sysadmin-system-6.0.0.14-r60013I.CSCcv44444.x86_64.rpm
# <release> field in Calvados/XR PKG are in the form: <LABEL>
# <release> field in Calvados/XR SMU are in the form: <LABEL>.<DDTS>
# Based on the above naming format, find the path of iso.
def parse_input(options, rpms):
    repo = None
    arch = None
    sdr_name = None
    rpm_list = list()
    if rpms:
        rpm = rpms[0]
        m = re.search(r'(.*)-(.*)-(.*)\.(.*)',rpm)
        if m:
            (name, version, release, arch) = m.groups()
            version_m = version
            p = release.split('.')
            label = p[0] if len(p) > 0 and p[0] else None
            vmtype = p[2] if len(p) > 2 and p[2] else None
            ver = label[1:]
            if options.base_pkg:
                version = ''.join(x+'.' for x in version_m.split('.')[:-1])[:-1]
                ver = ver.replace(''.join(x for x in version_m.split('.')[:-1]), '')
                if ver:
                    version = version + '.' + ver

            if options.act_pkg:
                version = ver

            # If package is provided to be extracted, it should be only for XR.
            if options.act_pkg:
                if "sysadmin" in name:
                    logger.error("Given package %s to extract from iso \
                                 not applicable to XR"%(rpm))
                    sys.exit(-1)
                else:
                    if "CSC" in release:
                        version_m = ''.join(x+'.' for x in version_m.split('.')[:-1]) + '0'
                    rpm = "%s-%s-%s.%s"%(name, version_m, label, arch)
                    rpm_list.append(rpm)
                    repo = "xr"  
                    if options.sdr_name:
                        sdr_name = rpms[1]

            if options.base_pkg:
                cmd = gen_rpm_tool + " -b -v host " + ''.join(rpm)
                try:
                    base_pkg = run_cmd(cmd, "list")
                except:
                    exc_info = sys.exc_info()
                    TB = traceback.format_exc()
                    logger.debug(TB)
                    logger.error("Unable to run cmd %s"%(cmd))
                    sys.exit(-1)
                for i in base_pkg:
                    if i: 
                        logger.debug("Input package %s, base package %s"%(rpm,i))
                        arch_b = re.search(r'(.*)\.(.*)',i).groups()[1]
                        if arch == arch_b:
                            rpm = "%s-%s-%s.%s"%(name, version, label, arch)
                            if "sysadmin-hostos" in name and vmtype and vmtype == "host":
                                rpm = "%s-%s-%s.%s.%s"%(name, version, label, "host", arch)
                                repo = "host"
                            elif "sysadmin-hostos" in name:
                                rpm = "%s-%s-%s.%s.%s"%(name, version, label, "admin", arch)
                                repo = "calvados"
                            rpm_list.append(rpm)
                    else:
                        logger.error("Base package could not be retrieved for %s."%(rpm))
                        sys.exit(-1)
        else:
            logger.error("Input rpm %s do not conform to standard naming format"%(rpm))
            sys.exit(-1)
    
    return rpm_list, arch, version, repo, sdr_name

# Find initrd for the base package version
def parse_initrd_list(initrd_list, version):
    logger.debug("Initrd list is %s."%(','.join(initrd_list)))
    for initrd in initrd_list:
        initrd_ver = os.path.basename(initrd).split('-')[-1]
        if version == initrd_ver:
            return initrd
         
    return None

# Ideally we should not depend on iso name.
# But some platforms do not have isoinfo utility.
# This is the approach currently used in install infra as well.
def parse_iso_list(iso_list, version):
    logger.debug("Iso list is %s."%(','.join(iso_list)))
    logger.debug("Version to be retrieved %s."%version)
    for iso in iso_list:
        if "mini" in iso:
            continue
        iso_ver = os.path.basename(iso).split('-')[-1]
        iso_ver = ''.join(x for x in iso_ver.split('.'))
        version = ''.join(x for x in version.split('.'))
        if version == iso_ver:
            logger.debug("Iso which will be considered is %s."%(iso))
            return iso
         
    return None

# In case of Fretta, we do not get the base package for arm arch
# from an iso available under install_repo.
# We need to mount mini iso and extract nbi-initrd.
def copy_from_initrd(rpms, repo, version):
    # Mount and extract initrd from mini iso.
    repo_path = install_repo + "calvados"
    initrd_list = list()
    initrd_img = None

    ret_rpms = rpms[:]
    for rpm in rpms:
        if os.path.isfile(repo_path + "/" + rpm):
            logger.debug("Rpm %s already in repo"%(rpm))
            ret_rpms.remove(rpm)
            break

    rpms = ret_rpms[:]

    if rpms:
        if os.path.isdir(repo_path):
            onlyfiles = [ os.path.join(repo_path,f) for f in os.listdir(repo_path) \
                          if os.path.isfile(os.path.join(repo_path,f)) ]
            for f in onlyfiles:
                if "initrd" in f:
                    initrd_list.append(f)

        initrd_img = parse_initrd_list(initrd_list, version)
        if not initrd_img:
            logger.debug("No valid initrd in install_repo. Check with one in tftpboot.")
            initrd_img = "/misc/disk1/tftpboot/sysadmin-nbi-initrd.img"
        try:
            cwd = os.getcwd()
            initrd_dir = "/misc/disk1/initrd"
            try:
                shutil.rmtree(initrd_dir, ignore_errors = True)
                os.mkdir(initrd_dir, 0755)
                os.chdir(initrd_dir)
                cmd = "zcat " + initrd_img + "| cpio -di"
                run_cmd(cmd, "string")
            except:
                exc_info = sys.exc_info()
                TB = traceback.format_exc()
                logger.debug(TB)
                logger.error("Unable to extract initrd to %s"%(initrd_dir))
                sys.exit(-1)

            os.chdir(cwd)
            rpm_dir = initrd_dir + "/rpm"
            if os.path.isdir(rpm_dir):
                onlyfiles = [ f for f in os.listdir(rpm_dir) \
                          if os.path.isfile(os.path.join(rpm_dir,f)) ]
    
            for rpm in rpms:
                rpm_chk = rpm
                m = re.search(r'(.*)-(.*)-(.*).*',rpm_chk)
                if m:
                    rpm_name = m.groups()[0]
                    rpm_release = m.groups()[2]
                    if 'CSC' in rpm_release:
                        rpm_release = re.sub('.CSC[a-z][a-z]\d{5}','',rpm_release)
                else:
                    logger.error("Unable to parse input rpm %s"%(rpm_chk))
                    sys.exit(-1) 

                initrd_rpm_found = False
                for rpm_src in onlyfiles:
                    m = re.search(r'(.*)-(.*)-(.*).*', rpm_src)
                    if m: 
                        rpm_src_name = m.groups()[0]
                        rpm_src_release = m.groups()[2].replace('.rpm', '')
                    else:
                        logger.error("Unable to parse initrd rpm %s"%(rpm_src))
                        sys.exit(-1) 
                        
                    if rpm_name == rpm_src_name and rpm_release == rpm_src_release:
                        src_file = os.path.join(rpm_dir, rpm_src)
                        dst_file = os.path.join(repo_path, rpm_src.replace('.rpm', ''))
                        logger.debug("Copying %s to %s"%(src_file, dst_file))
                        shutil.copy2(src_file, dst_file)    
                        initrd_rpm_found = True
                        ret_rpms.remove(rpm_chk)
                        break

                if initrd_rpm_found == False:
                    logger.error("Could not find %s in %s"%(rpm, initrd_img))
                    sys.exit(-1)

        finally:
            shutil.rmtree(initrd_dir, ignore_errors = True)
    return ret_rpms
            
def mount_and_copy_rpm(iso_file, rpms, repo, sdr_name):
    if repo != "host":
        repo_path = install_repo + repo
    else:
        repo_path = install_repo + "calvados"

    if sdr_name:
        link_repo = multi_sdr_instdb + sdr_name + "/pkg/" 

    ret_rpms = rpms[:]
    for rpm in rpms:
        if os.path.isfile(repo_path + "/" + rpm):
            logger.debug("Rpm %s already in repo"%(rpm))
            if sdr_name:
                link_repo += rpm
                dst_file = repo_path + "/" + rpm
                if os.path.islink(link_repo):      
                    os.unlink(link_repo)
                try:
                    logger.debug("Create symlink: %s -> %s"%(link_repo,dst_file))
                    os.symlink(dst_file, link_repo)
                except:
                    exc_info = sys.exc_info()
                    TB = traceback.format_exc()
                    logger.debug(TB)
                    logger.error("Failed to create symlink: %s -> %s"%(link_repo,dst_file))
                    sys.exit(-1)
            ret_rpms.remove(rpm)
            break

    rpms = ret_rpms[:]
    if rpms:
        try:
            cmd = "mktemp -d"
            try:
                mnt_dir = run_cmd(cmd, "string")
            except:
                exc_info = sys.exc_info()
                TB = traceback.format_exc()
                logger.debug(TB)
                logger.error("Unable to run cmd %s"%(cmd))
                sys.exit(-1)

            cmd = "mount -o loop " + iso_file  + " " + mnt_dir
            try:
                run_cmd(cmd, "string")
            except:
                exc_info = sys.exc_info()
                TB = traceback.format_exc()
                logger.debug(TB)
                logger.error("Unable to run cmd %s"%(cmd))
                sys.exit(-1)
            if repo != "host":
                rpm_dir_path = mnt_dir + "/rpm/" + repo
            else:
                rpm_dir_path = mnt_dir + "/rpm/"
            if os.path.isdir(rpm_dir_path):
                onlyfiles = [ f for f in os.listdir(rpm_dir_path) \
                          if os.path.isfile(os.path.join(rpm_dir_path,f)) ]
    
                for rpm in rpms:
                    rpm_chk = rpm
                    m = re.search(r'(.*)-(.*)-(.*).*',rpm_chk)
                    if m:
                        rpm_name = m.groups()[0]
                        rpm_release = m.groups()[2]
                        if 'CSC' in rpm_release:
                            rpm_release = re.sub('.CSC[a-z][a-z]\d{5}','',rpm_release)
                    else:
                        logger.error("Unable to parse input rpm %s"%(rpm_chk))
                        sys.exit(-1) 

                    iso_rpm_found = False
                    for rpm_src in onlyfiles:
                        m = re.search(r'(.*)-(.*)-(.*).*', rpm_src)
                        if m: 
                            rpm_src_name = m.groups()[0]
                            rpm_src_release = m.groups()[2].replace('.rpm', '')
                        else:
                            logger.error("Unable to parse iso rpm %s"%(rpm_src))
                            sys.exit(-1) 
                        
                        if rpm_name == rpm_src_name and rpm_release == rpm_src_release:
                            src_file = os.path.join(rpm_dir_path, rpm_src)
                            dst_file = os.path.join(repo_path, rpm_src.replace('.rpm', ''))
                            logger.debug("Copying %s to %s"%(src_file, dst_file))
                            shutil.copy2(src_file, dst_file)    
                            if sdr_name:
                                link_repo += rpm_src.replace('.rpm', '')
                                if os.path.islink(link_repo):      
                                    os.unlink(link_repo)
                                try:
                                    logger.debug("Create symlink: %s -> %s"%(link_repo,dst_file))
                                    os.symlink(dst_file, link_repo)
                                except:
                                    exc_info = sys.exc_info()
                                    TB = traceback.format_exc()
                                    logger.debug(TB)
                                    logger.error("Failed to create symlink: %s -> %s"%(link_repo,dst_file))
                                    sys.exit(-1)
                            iso_rpm_found = True
                            ret_rpms.remove(rpm_chk)
                            break

                    if iso_rpm_found == False:
                        logger.error("Could not find %s in %s"%(rpm, iso_file))
                        sys.exit(-1)
 
        finally:
            cmd = "umount " + mnt_dir
            try:
                run_cmd(cmd, "string")
            except:
                exc_info = sys.exc_info()
                TB = traceback.format_exc()
                logger.debug(TB)
                logger.error("Unable to run cmd %s"%(cmd))
                sys.exit(-1)

            shutil.rmtree(mnt_dir, ignore_errors = True)
    return ret_rpms

# Find all isos in the repo path.
# Check if iso version matches the rpm version.
# Read iso_info.txt for the same.
def extract_from_iso(rpms, repo, version, arch, sdr_name):
    if not rpms:
        return rpms
    iso_list = list()
    if arch != "arm":
        repo_path = install_repo + repo
        if os.path.isdir(repo_path):
            onlyfiles = [ os.path.join(repo_path,f) for f in os.listdir(repo_path) \
                          if os.path.isfile(os.path.join(repo_path,f)) ]
            for f in onlyfiles:
                cmd = "file -b " + f
                try:
                    chk_iso = run_cmd(cmd, "string")
                except:
                    exc_info = sys.exc_info()
                    TB = traceback.format_exc()
                    logger.debug(TB)
                    logger.error("Unable to run cmd %s"%(cmd))
                    sys.exit(-1)
                if "CD-ROM" in chk_iso:
                    iso_list.append(f)

            iso_file = parse_iso_list(iso_list, version)
            if iso_file:
                logger.debug("Iso file obtained %s"%(iso_file))
                if arch != "arm":
                    rpms =  mount_and_copy_rpm(iso_file, rpms, repo, sdr_name)
            else:
                logger.error("No Iso file matching the version of rpm provided")
                sys.exit(-1)
    else:
        rpms = copy_from_initrd(rpms, repo, version)
        
    return rpms

def parsecli():
    oparser = optparse.OptionParser()
    oparser.add_option(
        "-p",
        "--pkg",
        dest="act_pkg",
        action='store_true',
        default=False,
        help='Check for exact package in iso given the name')

    oparser.add_option(
        "-s",
        "--sdr",
        dest="sdr_name",
        action='store_true',
        default=False,
        help='When extracting an XR package, expect SDR name to be given')

    oparser.add_option(
        "-b",
        "--base",
        dest="base_pkg",
        action='store_true',
        default=False,
        help='Get the base package in iso for a given SMU. \
             Currently used for sysadmin-hostos SMU')

    options, args = oparser.parse_args()
    return options, args

def main(options,args):
    rpm, arch, version, repo, sdr_name = parse_input(options, args)
    rpm_rc = list(rpm)
    if arch:
        rpm = extract_from_iso(rpm, repo, version, arch, sdr_name)
        if rpm:
            rpm_rc = None
    return rpm_rc
    
if __name__ == '__main__' :
    LOGFILE = "/var/log/install/inst_hooks.log"
    # create logger
    logger = logging.getLogger('install_hook_logger')
    logger.setLevel(logging.DEBUG)
    formatter = logging.Formatter('%(asctime)s::  %(message)s',"%Y-%m-%d %H:%M:%S")

    # Logs to logfile
    fh = logging.handlers.RotatingFileHandler(LOGFILE, maxBytes=(1024*100))
    fh.setLevel(logging.DEBUG)
    fh.setFormatter(formatter)
    logger.addHandler(fh)

    cli = sys.argv[:]
    logger.debug("+"*80)
    logger.debug(' '.join(cli))
    if len(sys.argv) < 3 or '-h' in sys.argv:
        logger.error("Insufficient number of arguments to program.")
        sys.exit(-1)

    options, args = parsecli()

    print ' '.join(list(main(options,args)))


