#! /usr/bin/env python
# Copyright (c) 2015-2016 by cisco Systems, Inc.


import re
import sys
import os
import optparse
import commands

this_path = os.path.dirname(__file__)
gen_rpm_tool = this_path + "/get_rpms.py"

# Given a list of rpms to remove, check if the list contains
# base packages, in which case, sanitize the input by removing
# base packages from this list.
def sanitize_input_remove(vmtype, rpms):
    base_pkgs = set()
    rpms += find_missing_pkgs(vmtype, rpms)
    input_rpms = ' '.join(rpms)
    if input_rpms:
        # cmd_new = gen_rpm_tool + " -b -v " + vmtype + " " + input_rpms
        cmd_new = gen_rpm_tool + " -b -v host " + input_rpms
        base_pkgs = run_cmd(cmd_new)
        base_pkgs_vmtype = []
        for i in base_pkgs:
            m = re.search(r'(.*)\.(.*)', i)
            name = m.groups()[0]
            arch = m.groups()[1]
            base_pkgs_vmtype.append(name + "." + vmtype + "." + arch + ".rpm")

        for rpm in base_pkgs_vmtype:
            if rpm in rpms:
                rpms.remove(rpm)
    return set(rpms)

def sanitize_input_remove_base(vmtype, input_rpms):
    rpms = set(input_rpms)
    for rpm in input_rpms:
        if "CSC" not in rpm:
            if "hostos" in rpm or "spirit-boot" in rpm:
                if rpm in rpms:
                    rpms.remove(rpm)

    return set(rpms)

def run_cmd(cmd):
    status, text = commands.getstatusoutput(cmd)
    exit_code = status >> 8 # high byte
    if not exit_code:
        return text.strip().split(' ')
    else:
        sys.exit("ERROR: Unable to run cmd " + cmd)

# Create a set of missing packages from the input list.
# This set will be the missing packages in the input list.
def find_missing_pkgs(vmtype, rpms):
    output_set = set()
    input_rpms = ' '.join(rpms)
    if input_rpms:
        cmd = gen_rpm_tool + " -v " + vmtype + " " + input_rpms
        missing_pkgs = run_cmd(cmd)

        # Also need to retrieve the base packages for all
        input_rpms_new = (' '.join(missing_pkgs)).strip()
        if (input_rpms_new):
            cmd_new = gen_rpm_tool + " -b -v " + vmtype + " " + input_rpms_new
            #cmd_new = gen_rpm_tool + " -b -v host " + input_rpms_new
            base_pkgs = run_cmd(cmd_new)
            base_pkgs_vmtype = []
            for i in base_pkgs:
                m = re.search(r'(.*)\.(.*)', i)
                name = m.groups()[0]
                arch = m.groups()[1]
                base_pkgs_vmtype.append(name + "." + vmtype + "." + arch)

            output_set = set(missing_pkgs + base_pkgs_vmtype)

    # Remove the input rpms from this set.
    # They are already available to the add arguments.
    for i in rpms:
        if i:
            i = i.replace(".rpm", "")
            if (i in output_set):
                output_set.remove(i)

    # Append .rpm to each element in output list
    output_set = [x + ".rpm" for x in output_set]
    return output_set 

# Given a list of rpms, 
# return a list of host, calvados and XR rpms
# based on the rpm name.
def sanitize_input(input_rpms):
    pkg_list_xr = []
    pkg_list_calvados = []
    pkg_list_host = []
    pkg_list_base = []
    for i in input_rpms:
        vmtype = None
        if  i.endswith('.rpm'):
            m = re.search(r'(.*/)*(.*)-(.*)-(.*)\.(.*)\.(.*)(\.rpm)',i)
            name = m.groups()[1]
            vmtype = m.groups()[4]

        # Handle hostos SMU differently.
        # Check if hostos in rpm name.
        # IF so, add the correspodning domain SMU to base list for processing later.
        if "hostos" in name:
            pkg_list_base.append(i)
            continue
        if vmtype == "xr":
            pkg_list_xr.append(i)
        elif vmtype == "admin":
            pkg_list_calvados.append(i)
        elif vmtype == "host":
            pkg_list_host.append(i)
        else:
            pkg_list_base.append(i)

    return pkg_list_xr, pkg_list_calvados, pkg_list_host, pkg_list_base

# Validate the input list for any packages which have different
# versioning. In such a scenario ensure that we have both 
# versions in the input list.
def validate_input(vmtype, rpms):
    input_rpms = ' '.join(rpms)
    if input_rpms:
        try:
            cmd = gen_rpm_tool + " -d -v " + str(vmtype) + " " + input_rpms
            diff_list = run_cmd(cmd)
            # if there is no element in diff list, the input is already validated.
            valid = True
            for i in diff_list:
              if i:
                cmd_new = gen_rpm_tool + " -v " + vmtype + " " + i
                rpm_archs = run_cmd(cmd_new)
                # Find all the architectures supported by this rpm i
                for j in rpm_archs:
                    found = False
                    dname = None
                    darch = None
                    if not j.endswith('.rpm'):
                        m = re.search(r'(.*/)*(.*)-(.*)-(.*)\.(.*)',j)
                        dname = m.groups()[1]
                        darch = m.groups()[4]
                    for ip in rpms:
                        if ip.endswith('.rpm'):
                            m = re.search(r'(.*/)*(.*)-(.*)-(.*)\.(.*)(\.rpm)',ip)
                            name = m.groups()[1]
                            arch = m.groups()[4]
                            if name == dname and arch == darch:
                                found = True
                    if found == False:
                        valid = False
                        return i
            if valid == True:
                return None
        except:
            sys.exit("Error: Unable to validate input arguments")

def parsecli():
    oparser = optparse.OptionParser()
    oparser.add_option(
        "-a",
        "--add",
        dest="add_pkg",
        action='store_true',
        default=False,
        help='Get packages to be added given an input list')

    oparser.add_option(
        "-r",
        "--remove",
        dest="rem_pkg",
        action='store_true',
        default=False,
        help='Get the packages to remove given an input list')

    options, args = oparser.parse_args()
    return options, args

def populate_missing_package(vmtype, input_rpms):
    output_rpms = set()
    diff_pkg = validate_input(vmtype, input_rpms)
    if diff_pkg:
        sys.exit("Error: Packages with different versions needs to " + \
                 "be explicitly stated in the input arguments")
    output_rpms = find_missing_pkgs(vmtype, input_rpms)
    return set(output_rpms)

def populate_missing_base_package(input_rpms):
    new_pkgs = set()
    if input_rpms:
        for rpm in input_rpms:
            if "hostos" in rpm:
                cmd = gen_rpm_tool + " -v host " + rpm
                new_pkgs = new_pkgs | set(run_cmd(cmd))
                # Given the hostos SMU, ensure we have the pkg for both domains.
                # During install add, we will not be adding base package for hostos. 
                # It will be a SMU.
                m = re.search(r'(.*/)*(.*)-(.*)-(.*)\.(.*)\.(.*)',rpm)
                name = m.groups()[1]
                version = m.groups()[2]
                release = m.groups()[3]
                m = re.search(r'(.*)\.(CSC.*)\.(.*)', release)
                # Dont care about base package if its not a SMU
                if not m :
                    continue
                label = m.groups()[0] if len(m.groups()) > 0 and m.groups()[0] else None
                ddts  = m.groups()[1] if len(m.groups()) > 1 and m.groups()[1] else None
                vmtype = m.groups()[2] if len(m.groups()) > 2 and m.groups()[2] else None
                if not ddts:
                    continue
                if vmtype:
                    new_vm = "host" if vmtype == "admin" else "admin"
                add_rpm = name + "-" + version + "-" + label + "." + ddts + "." + new_vm
                cmd = gen_rpm_tool + " -v host " + add_rpm
                new_pkgs = new_pkgs | set(run_cmd(cmd))
            
    return set([x + ".rpm" for x in new_pkgs])

def main(options,args):
    output_rpms = set()
    input_rpms = args

    if options.rem_pkg:
        input_rpms = [x + ".rpm" for x in input_rpms if ".rpm" not in x]
    # Split the input arguments based on host, xr, calvados and base packages.
    # The base package list is used to sanitize the output by removing the base
    # packages already given in input list.
    rpm_xr, rpm_calv, rpm_host, rpm_base = sanitize_input(input_rpms)

    if options.add_pkg:
        # Individually find missing package list for XR, Host and Calvados.
        output_rpms = output_rpms | populate_missing_package("xr", rpm_xr)
        output_rpms = output_rpms | populate_missing_package("admin", rpm_calv)
        output_rpms = output_rpms | populate_missing_package("host", rpm_host)
        output_rpms = output_rpms | populate_missing_base_package(rpm_base)
        output_rpms = [x for x in output_rpms if x]
        for rpm in rpm_base:
            if rpm:
                if rpm in output_rpms:
                    output_rpms.remove(rpm) 

    if options.rem_pkg:
        output_rpms = output_rpms | sanitize_input_remove("xr", rpm_xr)
        output_rpms = output_rpms | sanitize_input_remove("admin", rpm_calv)
        output_rpms = output_rpms | sanitize_input_remove("host", rpm_host)
        output_rpms = output_rpms | sanitize_input_remove_base("host", rpm_base)
        output_rpms = [x.replace(".rpm","") for x in output_rpms if x]
        
    return output_rpms

if __name__ == '__main__' :
    options, args = parsecli()
    print ' '.join(list(main(options,args)))


