#! python
import subprocess
import os
import commands
from consts import *
import constants as C
from logger_init import get_logger
import sys
import traceback
import re
import fnmatch

vrf_str = "/opt/cisco/calvados/bin/vrfch.sh CTRL_VRF"
ssh_str = "ssh -q -o ConnectTimeout=1  -o StrictHostKeyChecking=no"
scp_str = "scp -q -o ConnectTimeout=1  -o StrictHostKeyChecking=no"

def set_logger():
    if os.environ.get('MP_LOG_FILE'):
        return get_logger(os.environ.get('MP_LOG_FILE'))
    else:
        return get_logger()

def run_cmd (cmd, skip_logging = False):
    logger = set_logger()
    process = subprocess.Popen(cmd, stdout=subprocess.PIPE, 
                               stderr=subprocess.PIPE, shell=True)
    out, error = process.communicate()
    sprc = process.returncode
    if sprc is None or sprc != 0:
        out = error
        if not skip_logging:
            logger.error ("Error CMD=%s returned --->%s" % (cmd, out))
        raise RuntimeError("Error CMD=%s returned --->%s" % (cmd, out))
    else:
        if not skip_logging:
            logger.debug("CMD=%s OUTPUT=%s"%(cmd, out))
        pass
    return out.strip()

def get_host_ip ():
    s = [x for x in run_cmd("/opt/cisco/calvados/bin/print_bootstrap_var HOST_IP_ADDR")]
    host_ip='.'.join([str(int(x,16)) for x in [''.join(s[i:i+2]) for i in range(0, len(s), 2)]])
    return host_ip

def run_cmd_on_host (cmd):
    host_ip = get_host_ip ()
    cmd = "%s %s %s %s"%(vrf_str, ssh_str, host_ip, cmd)
    return run_cmd (cmd)

def create_lv (vmtype, lvid, size, volgrp):
    cmd = "lvcreate -L %sM -n %s_lv%s %s"%(size, vmtype, lvid, volgrp)
    run_cmd_on_host (cmd)

def attach_lv (source_dev, target_dev):
    cmd = "/opt/cisco/calvados/bin/install-functions.py get_virt_method"
    virt_method = run_cmd (cmd)
    if virt_method == 'vm':
        cmd = "virsh attach-disk sysadmin %s %s"%(source_dev, os.path.basename(target_dev))
    else:
        cmd = "virsh -c lxc:/// attach-disk sysadmin %s %s"%(source_dev, os.path.basename(target_dev))
    target_dev = run_cmd_on_host (cmd)
    return target_dev

def precheck_env ():
    cmd = "/sbin/ldconfig"
    run_cmd (cmd)

def delete_volume (lv):
    cmd = "lvremove -f %s"%(lv)
    run_cmd_on_host (cmd)
    return

def detach_volume (lv):
    logger = set_logger()
    cmd = "/opt/cisco/calvados/bin/install-functions.py get_virt_method"
    import ctypes
    import time
    libc = ctypes.CDLL("libc.so.6")
    libc.sync()
    time.sleep(2)
    virt_method = run_cmd (cmd)
    if virt_method == 'vm':
        cmd = "virsh detach-disk sysadmin %s"%(lv)
    else:
        cmd = "virsh -c lxc:/// detach-disk sysadmin %s"%(lv)
    try:    
        run_cmd_on_host (cmd)
    except:
        logger.info ("Was unable to detach volume %s, will retry detach operation."%(lv))
        libc.sync()
        time.sleep(2)
        run_cmd_on_host (cmd)
    return

def create_and_attach_lv (vmtype, lvid, part_size, rootfsvolgrp, target_dev):
    create_lv (vmtype, lvid, part_size, rootfsvolgrp)
    source_dev = "/dev/%s/%s_lv%s"%(rootfsvolgrp, vmtype, lvid)
    attach_lv (source_dev, target_dev)
    return target_dev
    
def make_fs_device (target_dev, label):
    cmd = "exec 107>&1; source /etc/init.d/disk-functions; check_fs_partition %s %s"%(target_dev, label)
    return run_cmd (cmd)

def mount_device (target_dev, offset = 0):
    import shutil
    import ctypes
    mnt_dir = run_cmd ("mktemp -d")
    #lodev = setup_loop (target_dev)
    cmd = "mount %s %s"%(target_dev, mnt_dir)
    if offset:
        cmd = "%s -o offset=%d"%(cmd, offset)
    run_cmd (cmd)
    for (root, dirs, files) in os.walk(mnt_dir):
        for f in files:
            os.unlink(os.path.join(root, f))
        for d in dirs:
            if os.path.islink (os.path.join(root, d)):
                os.unlink (os.path.join(root, d))
            else:
                shutil.rmtree(os.path.join(root, d))
    libc = ctypes.CDLL("libc.so.6")
    libc.sync()
    return mnt_dir

'''
    Return a dictionary with device as key and vmtype as value.
'''
def get_device_vmtype_attached (targetdev):
    logger = set_logger()
    dev_dict = {}
    try:
        lo_device = setup_loop (targetdev)
        devlabel = get_device_label (lo_device)
        logger.info ("Dev label retrieved %s"%(devlabel))
        for key, value in C.label_vmtype.iteritems():
            if value in devlabel:
                dev_dict[key] = targetdev
    except:
        raise Exception ("Unable to get device dictionary for device %s"%(targetdev))
    finally:
        detach_loop (lo_device)
    return dev_dict

def mount_iso (iso):
    mnt_dir = run_cmd ("mktemp -d")
    cmd = "mount -o loop %s %s"%(iso, mnt_dir)
    run_cmd (cmd)
    return mnt_dir
    
def extract_pd_files_from_initrd (staging, initrd, iso_version):
    try:
        import shutil
        temp_staging = os.path.join (staging, "tmpXXX")
        temp_extract_path = run_cmd ("mktemp -d %s"%(temp_staging))

        if 'arm' in read_bootstrap_supported_arch ():
            files2populate = [ 'iso_name.txt', 'nbi-initrd/*']
            cwd = os.getcwd ()
            os.chdir(temp_extract_path)
            files2extract = ' '.join(files2populate)
            cmd = "zcat -f %s | cpio -idm --no-preserve-owner %s"%(initrd, files2extract)
            run_cmd (cmd)
            os.chdir (cwd)
            nbi, nbi_initrd = read_iso_name_mdata (temp_extract_path)
            if not nbi or not nbi_initrd:
                raise Exception ()
            nbi_initrd_src = os.path.join (temp_extract_path, "nbi-initrd", nbi_initrd)
            nbi_dest = os.path.join (C.CAL_REPO, "%s-%s"%(nbi, iso_version))
            nbi_initrd_dest_cal = os.path.join (C.CAL_REPO, "%s-%s"%(nbi_initrd, iso_version))
            nbi_initrd_dest_host = os.path.join (C.HOST_REPO, "%s-%s"%(nbi_initrd, iso_version))
            if not os.path.isfile (nbi_initrd_dest_cal):
                shutil.copy (nbi_initrd_src, nbi_initrd_dest_cal)
            if not os.path.isfile (nbi_initrd_dest_host):
                shutil.copy (nbi_initrd_src, nbi_initrd_dest_host)
            return os.path.basename(nbi_dest), os.path.basename(nbi_initrd_dest_cal)
        else:
            return None, None
    except:
        logger.error ("Error while copying PD files to repo")
        raise
    finally:
        if os.path.isdir (temp_extract_path):
            run_cmd ("rm -rf %s"%(temp_extract_path))
        
def extract_giso_rpms (staging, iso_mnt, initrd, sdr):
    files2populate = []
    repovmtype = []
    giso_rpm_dir = ''
    temp_extract_path = ''
    sdr_repo = "/install_repo/gl/instdb/sdr/%s/pkg"%(sdr)

    for vmtype in C.SUPPORTED_VMTYPE:
        if vmtype == C.XR_LCP:
            continue
        files2populate.append ("%s_rpms"%(vmtype.lower()))
        if vmtype == C.HOST:
            repovmtype.append (C.CAL_REPO)
        elif vmtype == C.CALVADOS:
            repovmtype.append (C.CAL_REPO)
        elif vmtype == C.XR:
            repovmtype.append (C.XR_REPO)

    try:
        import shutil
        cwd = os.getcwd ()
        giso_rpms = []
        non_optim_giso_index = 4 
        giso_rpm_tool = "/opt/cisco/calvados/bin/install-functions.py"
        giso_file = os.path.join (iso_mnt, "giso_info.txt")
        if os.path.isfile (giso_file):
            cmd_rpm_path = "%s get_giso_rpm_path %s"%(giso_rpm_tool, giso_file)
            path_index = run_cmd (cmd_rpm_path)
            if int(path_index) == non_optim_giso_index:
                # Giso RPMS are packaged at top level iso mount.
                giso_rpm_dir = iso_mnt
            else:
                # Giso RPMS are packaged within the initrd passed to this function.
                temp_staging = os.path.join (staging, "tmpXXX")
                temp_extract_path = run_cmd ("mktemp -d %s"%(temp_staging))
                files2extract = ' '.join ([os.path.join (x, '*') for x in files2populate])    
                os.chdir(temp_extract_path)
                cmd = "zcat -f %s | cpio -idm --no-preserve-owner %s"%(initrd, files2extract)
                run_cmd (cmd)
                os.chdir (cwd)
                giso_rpm_dir = temp_extract_path

        for item, repo in zip(files2populate, repovmtype):
            giso_rpm_path = os.path.join (giso_rpm_dir, item)
            if os.path.isdir (giso_rpm_path):
                giso_rpms = [ f for f in os.listdir(giso_rpm_path) \
                    if os.path.isfile(os.path.join(giso_rpm_path, f)) and ".rpm" in f ]
            for rpm in giso_rpms:
                src_rpm = os.path.join (giso_rpm_path, rpm)
                dest_rpm = os.path.join (repo, rpm.replace ('.rpm', ''))
                if not os.path.isfile (dest_rpm):
                    shutil.copy (src_rpm, dest_rpm)
                if repo == C.XR_REPO:
                    sdr_rpm = os.path.join (sdr_repo, os.path.basename (dest_rpm))
                    if not os.path.exists (sdr_rpm):
                        os.symlink (dest_rpm, sdr_rpm)
    except:
        logger.error ("Error while copying GISO rpm files to repo")
        raise
    finally:
        if os.path.isdir (temp_extract_path):
            run_cmd ("rm -rf %s"%(temp_extract_path))
        

def extract_files_from_initrd (mnt_dir, initrd, files):
    cwd = os.getcwd ()
    os.chdir(mnt_dir)
    files2extract = ' '.join(files)
    cmd = "zcat -f %s | cpio -idm --no-preserve-owner %s"%(initrd, files2extract)
    run_cmd (cmd)
    os.chdir (cwd)

def extract_initrd_to_part (iso_mnt, mnt_dir):
    cwd = os.getcwd ()
    os.chdir(mnt_dir)
    cmd = "zcat -f %s/boot/initrd.img | cpio -idm --no-preserve-owner"%(iso_mnt)
    run_cmd (cmd)
    os.chdir (cwd)


def get_my_cmdline ():
    cmd = "/usr/bin/xr_sysctl -n kernel.boot.cmdline"
    return run_cmd (cmd)
    
''' 
    Update /boot/grub/menu.lst 
    Start with current cmdline in calvados.
    V2 patch scripts will update the grub accordingly for V2 parameters.
'''
def prepare_grub_setup (mnt_dir, device, vmname, vmtype):
    menu_lst_file = os.path.join (mnt_dir, "boot/grub/menu.lst")
    devlabel = ''
    cmdline = get_my_cmdline ()
    if not os.path.isfile (menu_lst_file):
        return

    cmd = "sed -i -e \"s;root=.*;%s;\" %s"%(cmdline, menu_lst_file)
    run_cmd (cmd)
    try:
        lo_device = setup_loop (device)
        devlabel = get_device_label (lo_device)
    except:
        pass
    finally:
        try:
            detach_loop (lo_device)
        except:
            detach_loop (lo_device)
    cmd = "sed -i -e \"s;root=[^ ]* ;root=LABEL=%s ;\" %s"%(devlabel, menu_lst_file)
    run_cmd (cmd)
    cmd = "sed -i -e \"s;vmtype=[^ ]* ;vmtype=%s ;\" %s"%(vmname, menu_lst_file)
    run_cmd (cmd)
    if vmtype == C.XR_LCP:
        cmd = "sed -i -e \"s;boardtype=[^ ]* ;boardtype=LC ;\" %s"%(menu_lst_file)
        run_cmd (cmd)
    grub_conf_file = os.path.join (mnt_dir, "boot/grub/grub.conf" )
    if os.path.isfile (grub_conf_file):
        os.unlink (grub_conf_file)
    cmd = "ln -s ./menu.lst %s"%(grub_conf_file)
    run_cmd (cmd)

    if vmtype != C.HOST:
        cmd = "/usr/bin/grubit.sh %s %s hd0"%(mnt_dir, device)
        run_cmd (cmd)
    cmd = "sed -i -e \"s;root (cd); root (hd0,0);\" %s"%(menu_lst_file)
    run_cmd (cmd)
    if vmtype == C.XR:
        cmd = "sed -i -e \"s;console=ttyS0;console=tty0 console=hvc0 ;\" %s"%(menu_lst_file)
        run_cmd (cmd)
    return

def set_owner_group (dev_mnt):
    cmd_chown = "chown -R root %s"%(dev_mnt)
    cmd_chgrp = "chgrp -R root %s"%(dev_mnt)
    try:
        run_cmd (cmd_chown)
        run_cmd (cmd_chgrp)
    except:
        pass

def chroot_support_mounts (mnt_dir):
    run_cmd ("mkdir -p %s/proc"%(mnt_dir))
    run_cmd ("mkdir -p %s/sys"%(mnt_dir))
    run_cmd ("mount --rbind /proc %s/proc"%(mnt_dir))
    run_cmd ("mount --rbind /sys %s/sys"%(mnt_dir))

def chroot_support_umounts (mnt_dir):
    run_cmd ("umount -R %s"%(os.path.join(mnt_dir, "sys")))
    run_cmd ("umount -R %s"%(os.path.join(mnt_dir, "proc")))

def get_local_board_type (vmtype):
    local_card = ['all']
    card = ''
    if vmtype == XR_LCP:
        card = "lc"
    else:
        cmd = "exec 107>&1; source /etc/init.d/spirit_pd.sh; get_board_type; echo $BOARDTYPE"
        card = run_cmd (cmd)
    local_card.append (card.lower())
    return local_card

def check_iso_is_giso (iso_mnt):
    giso_file = os.path.join (iso_mnt, "giso_info.txt")
    if os.path.isfile (giso_file):
        return True
    return False

def get_giso_type (iso_mnt):
    giso_type = None
    non_optim_giso_index = 4
    giso_rpm_tool = "/opt/cisco/calvados/bin/install-functions.py"
    giso_file = os.path.join (iso_mnt, "giso_info.txt")
    if os.path.isfile (giso_file):
        cmd_rpm_path = "%s get_giso_rpm_path %s"%(giso_rpm_tool, giso_file)
        path_index = run_cmd (cmd_rpm_path)
        if int(path_index) == non_optim_giso_index:
            # Giso RPMS are packaged at top level iso mount.
            giso_type = C.GISO_NONOPTIM
        else:
            # Giso RPMS are packaged within the initrd passed to this function.
            giso_type = C.GISO_OPTIM
    return giso_type
    
def read_giso_rpms_mdata (iso_mnt, vmtype, platform):
    logger = set_logger()
    rpms_giso = []
    hint_tool = "/opt/cisco/calvados/bin/provides_hint.py"
    meta_file = os.path.join(iso_mnt, "iosxr_image_mdata.yml")
    if os.path.isfile (meta_file):
        cmd_iso_rpms = "%s -f %s -r -v %s"%(hint_tool, meta_file, vmtype)
        cmd_giso_rpms = "%s -f %s -r -v %s -g"%(hint_tool, meta_file, vmtype)
        rpms_iso = run_cmd (cmd_iso_rpms).split(' ') 
        rpms_giso_all = run_cmd (cmd_giso_rpms).split(' ') 
        # Check for cisco and tp packages.
        rpms_giso_cisco_pkgs = [x for x in rpms_giso_all if platform in x]
        rpms_giso_tp_smus = [x for x in rpms_giso_all if is_tp_smu (platform, x)]
        rpms_giso_all = rpms_giso_cisco_pkgs + rpms_giso_tp_smus
        rpms_giso = list(set(rpms_giso_all) - set(rpms_iso))
    logger.debug('GISO rpms returned for %s is %s'%(vmtype, ','.join(rpms_giso)))
    return rpms_giso
        

def read_iso_name_mdata (mnt_dir):
    iso_name_path = os.path.join (mnt_dir, "iso_name.txt")
    if os.path.isfile (iso_name_path):
        with open(iso_name_path, 'r') as fp:
            data=fp.read()
            iso_name_dict = {}
            iso_name_dict = dict([(item.partition(':')[0],item.partition(':')[2].strip()) for item in data.splitlines() if not item.strip().startswith('#') and item.strip()])
            return iso_name_dict ['ADMIN_NBI'], iso_name_dict ['ADMIN_NBI_INITRD']
    return None, None

def read_iso_info_mdata (iso_mnt):
    cmd_platname = "/opt/cisco/calvados/bin/install-functions.py get_platform_name"
    iso_info_path = os.path.join (iso_mnt, "iso_info.txt")
    fp = open (iso_info_path, 'r')
    buff = fp.read()
    fp.close()
    plat_name = run_cmd (cmd_platname)
    iso_version = buff[buff.find("Version:"):].split()[1]
    host_iso = os.path.join (host_repo, "host-%s"%(iso_version))
    cal_iso = os.path.join (cal_repo, "%s-sysadmin-%s"%(plat_name, iso_version))
    xr_iso = os.path.join (xr_repo, "%s-xr-%s"%(plat_name, iso_version))
    return plat_name, host_iso, cal_iso, xr_iso, iso_version

def umount_iso (iso_mnt):
    try:
        if os.path.ismount (iso_mnt):
            run_cmd ("umount %s"%(iso_mnt))
            run_cmd ("rm -rf %s"%(iso_mnt))    
    except:
        import ctypes
        import time
        libc = ctypes.CDLL("libc.so.6")
        libc.sync()
        time.sleep(1)
        if os.path.ismount (iso_mnt):
            run_cmd ("umount %s"%(iso_mnt))
            run_cmd ("rm -rf %s"%(iso_mnt))    

def get_device_label (dev):
    return run_cmd ("e2label %s"%(dev))

def check_subpartitions (dev):
    cmd = "fdisk -u -l %s | grep %s1"%(dev, dev)
    return run_cmd (cmd)

def detach_loop (dev):
    return run_cmd ("losetup -d %s"%(dev))

def get_rootfs_offset (dev): 
    logger = set_logger()
    offset = 0
    try:
        check_subpartitions (dev)
        part = 1
        start = ""
        end = ""
        start_cmd = "fdisk -u -l %s 2>/dev/null | grep %s%d | awk '{print $2}'"%(dev, dev, part)
        start = run_cmd (start_cmd)
        if start == "*":
            start_cmd = "fdisk -u -l %s 2>/dev/null | grep %s%d | awk '{print $3}'"%(dev, dev, part)
            start = run_cmd (start_cmd)
        offset = int(start) * 512
        logger.debug('Subpartition exist at offset %d.'%(offset))
    except RuntimeError:
        logger.info('No subpartitions exist. Setup loop device with whole device')
        offset = 0
    return offset

def setup_loop (dev):
    logger = set_logger()
    loop = ''
    try:
        check_subpartitions (dev)
        part = 1
        start = ""
        end = ""
        start_cmd = "fdisk -u -l %s 2>/dev/null | grep %s%d | awk '{print $2}'"%(dev, dev, part)
        start = run_cmd (start_cmd)
        if start == "*":
            start_cmd = "fdisk -u -l %s 2>/dev/null | grep %s%d | awk '{print $3}'"%(dev, dev, part)
            start = run_cmd (start_cmd)
            end_cmd = "fdisk -u -l %s 2>/dev/null | grep %s%d | awk '{print $4}'"%(dev, dev, part)
            end = run_cmd (end_cmd)
        else:
            end_cmd = "fdisk -u -l %s 2>/dev/null | grep %s%d | awk '{print $3}'"%(dev, dev, part)
            end = run_cmd (end_cmd)
        result = int(end) - int(start)
        sizelimit = result * 512
        offset = int(start) * 512
        loop = run_cmd ("losetup --find")
        setup_cmd = "losetup %s %s -o %s --sizelimit %s"%(loop, dev, offset, sizelimit)
    except RuntimeError:
        logger.info('No subpartitions exist. Setup loop device with whole device')
        loop = run_cmd ("losetup --find")
        setup_cmd = "losetup %s %s"%(loop, dev)

    try:
        run_cmd (setup_cmd)
    except:
        import ctypes
        libc = ctypes.CDLL("libc.so.6")
        libc.sync()
        run_cmd (setup_cmd)
    return loop
    
def get_file_type (_file):
    file_type = ""
    cmd = "file -Lb %s"%(_file)
    output = run_cmd (cmd)
    if "CD-ROM" in output:
        file_type = "iso"
    elif "RPM" in output:
        file_type = "rpm"
    else:
        file_type = "unknown"

    if file_type == "iso":
        if "mini" in _file or "golden" in _file:
            file_type = "bundle"

    return file_type

def get_install_board_type (vmtype):
    boardtype = -1
    card = get_local_board_type (vmtype)
    if "rp" in card:
        boardtype = 1
    elif "lc" in card:
        boardtype = 0
    elif "sc" in card:
        boardtype = 2
    elif "fc" in card:
        boardtype = 3
    elif "cc" in card:
        boardtype = 4
    elif "xc" in card:
        boardtype = 14
    return boardtype
    
def prepenv_mount_sysdirs (mnt_dir, iso_mnt, rpm_path, repo):
    run_cmd ("mkdir -p %s"%(os.path.join(mnt_dir, "rpm")))
    run_cmd ("mount --bind %s %s"%(os.path.join(iso_mnt, rpm_path), os.path.join(mnt_dir, "rpm")))
    run_cmd ("mkdir -p %s"%(os.path.join(mnt_dir, "pkgs")))
    run_cmd ("mount --bind %s %s"%(repo, os.path.join(mnt_dir, "pkgs")))

def run_pre_install_patches (iso_mnt, mnt_dir, card_type, vmtype):
    cmd = "%s/etc/init.d/hook_pre_install.sh %s %d %d"%(iso_mnt, mnt_dir, int(card_type), vmtype)
    run_cmd (cmd)

def find_files(directory, pattern):
    for root, dirs, files in os.walk(directory):
        for basename in files:
            if fnmatch.fnmatch(basename, pattern):
                filename = os.path.join(root, basename)
                yield filename

def partition_remove_corrupt_pyc (dev_mnt):
    directory = os.path.join (dev_mnt, "usr/lib64/python2.7")
    pattern = "*.pyc"
    for filename in find_files(directory, pattern):
        os.unlink (filename)        

def calculate_build_time_tppkgs (mnt_dir, rpm_path, rpmlist):
    buildtimelist = []
    rpm_files = [ os.path.join(rpm_path, f) for f in rpmlist ]
    for f in rpm_files:
        cmd = "chroot %s rpm -qp --qf '%s' %s"%(mnt_dir, "%{BUILDTIME}", f)
        buildtime = run_cmd (cmd)
        buildtimelist.append (buildtime)
    return buildtimelist

def create_exclude_path_rpm (mnt_dir, vmtype, rpm_path, rpmlist = None):
    excl_path = []
    excl_cards_gl = []
    buildtimelist = []
    local_card = get_local_board_type (vmtype)
    rpm_dir = os.path.join (mnt_dir, rpm_path)
    if not rpmlist:
        rpm_files = [ os.path.join(rpm_path, f) for f in os.listdir(rpm_dir) \
                    if os.path.isfile(os.path.join(rpm_dir,f)) and ".rpm" in f ]
    else:
        rpm_files = [ os.path.join(rpm_path, f) for f in rpmlist ]

    for f in rpm_files:
        cmd = "chroot %s rpm -qp --qf '%s' %s"%(mnt_dir, "%{PREFIXES} %{SUPPCARDS} %{NAME} %{VERSION} %{RELEASE} %{BUILDTIME}", f)
        prefix, cards, name, version, release, buildtime = run_cmd (cmd).split()
        excl_cards = [f for f in cards.split(',') if f not in local_card]
        excl_cards_gl.extend(set(excl_cards).difference(set(excl_cards_gl)))
        if vmtype == "calvados":
            excl_pkg_path = os.path.join(prefix, "%s-%s-%s"%(name, version, release))
        else:
            excl_pkg_path = prefix
        for card in excl_cards:
            path2excl = "--excludepath=%s/%s"%(excl_pkg_path, card)
            excl_path.append (path2excl)
        buildtimelist.append (buildtime)
    return ' '.join(excl_path), excl_cards_gl, buildtimelist

def install_additional_rpms_to_part (mnt_dir, vmtype, rpm_path, plat):
    logger = set_logger()
    excl_cards_gl = []
    buildtimelist = []
    rpm_dir = os.path.join (mnt_dir, rpm_path)
    rpm_files = [ f for f in os.listdir(rpm_dir) \
                    if get_file_type(os.path.join(rpm_dir, f)) == 'rpm'  ]
    '''
    Take out TP/hostos/spirit-boot packages from rpm_files. They will be installed separately.
    '''
    rpm_files_tp = [x for x in rpm_files if is_tp_smu (plat, x) or 'sysadmin-hostos' in x]
    rpm_files_notp = [x for x in rpm_files if x not in rpm_files_tp]

    if rpm_files_notp:
        excl_path, excl_cards_gl, buildtimelist = create_exclude_path_rpm (mnt_dir, vmtype, rpm_path, rpm_files_notp)
        rpm_cmd_check_install = "D=/ chroot %s rpm -iv --test --replacefiles --ignoresize %s "%(mnt_dir, excl_path)
        rpm_cmd = "D=/ chroot %s rpm -iv --nodeps --replacefiles --ignoresize %s "%(mnt_dir, excl_path)
        for f in rpm_files_notp:
            rpm_cmd += " %s "%(os.path.join(rpm_path, f))
            rpm_cmd_check_install += " %s "%(os.path.join(rpm_path, f))
        try:
            run_cmd (rpm_cmd_check_install)
        except Exception as e:
            msg = str(e)
            substr2check = "error: Failed dependencies:"
            if substr2check in msg:
                msg2log = "Failed dependencies:"
                msg2log += msg.split (substr2check)[1]
                msg = msg2log
            send_err_msg_to_sys_orch(msg)
            raise
        run_cmd (rpm_cmd)

    if rpm_files_tp:
        buildtimelist += calculate_build_time_tppkgs (mnt_dir, rpm_path, rpm_files_tp)
        rpm_cmd_opt = "D=/ chroot %s rpm -Uv --force --nodeps --replacefiles --ignoresize "%(mnt_dir)
        for f in rpm_files_tp:
            if 'sysadmin-hostos' not in f:
                chroot_support_mounts (mnt_dir)    
            rpm_cmd = "%s %s"%(rpm_cmd_opt, os.path.join(rpm_path, f))
            run_cmd (rpm_cmd)
            if 'sysadmin-hostos' not in f:
                chroot_support_umounts (mnt_dir)    
    return excl_cards_gl, buildtimelist

def install_rpm_to_part (mnt_dir, vmtype, rpm_path):
    excl_path = ""
    rpm_dir = os.path.join (mnt_dir, rpm_path)
    rpm_files = [ f for f in os.listdir(rpm_dir) \
                    if os.path.isfile(os.path.join(rpm_dir,f)) and ".rpm" in f ]
    excl_path, excl_cards_gl, buildtimelist = create_exclude_path_rpm (mnt_dir, vmtype, rpm_path)
    rpm_cmd = "D=/ chroot %s rpm -iv --nodeps --replacefiles --ignoresize %s "%(mnt_dir, excl_path)
    for f in rpm_files:
        rpm_cmd += " %s "%(os.path.join("rpm", f))
    try:    
        run_cmd (rpm_cmd)
    except:
        msg = "Error occurred during installation of rpms for %s vm partition"%(vmtype)
        send_err_msg_to_sys_orch(msg)
        raise
    return excl_cards_gl
            
def copy_ssh_keys (mnt_dir, vmtype):
    cmd = ""
    if os.path.isdir (os.path.join(mnt_dir, "root")):
        run_cmd ("chown -R root:root %s"%(os.path.join(mnt_dir, "root")))
        run_cmd ("chmod -R 700 %s"%(os.path.join(mnt_dir, "root")))
    if not os.path.isdir (os.path.join(mnt_dir, "root", ".ssh")):
        os.mkdir (os.path.join(mnt_dir, "root", ".ssh"), 0700)
    if vmtype == "calvados":
        cmd = "cp -raf /root/.ssh/* %s/root/.ssh/;rm -f %s/root/.ssh/known_hosts"%(mnt_dir, mnt_dir)
    elif vmtype == "host":
        host_ip = get_host_ip ()
        cmd = "scp -r %s:/root/.ssh/* %s/root/.ssh/; rm -f %s/root/.ssh/known_hosts"%(host_ip, mnt_dir, mnt_dir)
    run_cmd (cmd)
    
def update_symlinks (mnt_dir, vmtype, excl_cards_gl):
    cmd = ""
    if vmtype == "calvados":
        cmd = "/usr/bin/calv_setup_ldpath.sh %s '%s'"%(mnt_dir, ' '.join(excl_cards_gl))
    elif vmtype == "xr" or vmtype == "xr_lcp":
        cmd = "python /opt/cisco/calvados/bin/create_symlinks.py %s > /dev/null"%(mnt_dir)
    elif vmtype == "host":
        pass
    run_cmd (cmd)

'''
Run any post install patches. Includes call to PD patch functions.
Update self lv index file for sysadmin-vm
'''
def run_post_install_patches (mnt_dir, lvid, vmname, vmtype):
    cmd = "chroot %s /etc/init.d/calvados_patch_lxc_iso.sh / /dev/panini_vol_grp/ %d %s"%(mnt_dir, int(lvid), vmname)
    run_cmd (cmd)
    if vmtype != C.HOST:
        update_index_f = os.path.join(mnt_dir, "install/instdb/local/self_lv_index.txt")
        with open (update_index_f, 'w') as fd:  
            fd.write (str(lvid))  
    return

def prepenv_umount_sysdirs (mnt_dir):
    run_cmd ("umount %s/rpm"%(mnt_dir))
    run_cmd ("umount %s/pkgs"%(mnt_dir))

def read_install_params (iso_mnt):
    import json
    #yaml_f = os.path.join (iso_mnt, "install_params.yml")
    json_f = os.path.join (iso_mnt, "install_params.json")
    if os.path.exists (json_f):
        json_fd = open(json_f, 'r') 
        mdata_dict = json.load(json_fd)
        json_fd.close()
    else:
        raise RuntimeError ("Unable to read %s"%(json_f))
    return mdata_dict, mdata_dict["STAGING"], mdata_dict["SECONDARY_STAGING"]

def get_pd_provided_staging (): 
    loc = '/install/tmp'
    secondary_staging = '/misc/disk1'
    cmd = 'mountpoint %s' % (loc)
    status,output = commands.getstatusoutput(cmd)
    if status:
        staging = '/misc/disk1'
        secondary_staging = '/install_repo'
    else:
        cmd = "df -a %s | awk '{print $1}' | tail -n1" % (loc)
        fs = run_cmd(cmd)
        cmd = "cat /proc/*/mountinfo | sort | uniq | grep '/ /' | grep %s | awk '{print $5}'" % (fs)
        staging = run_cmd(cmd)
        if staging == secondary_staging:
            secondary_staging = '/install_repo/'
    return staging, secondary_staging

def read_bootstrap_install_params ():
    with open("/etc/init.d/calvados_bootstrap.cfg") as cfg:
        data=cfg.read()
        cfg_dict = {}
        bootstrap_keys = ['DISK_XR_BOOT_SIZE', 'DISK_HOST_BOOT_SIZE','DISK_CALVADOS_BOOT_SIZE']
        bootstrap_cfg = dict([(item.partition('=')[0],item.partition('=')[2]) for item in data.splitlines() if not item.strip().startswith('#') and item.strip()])
        required_data = {boot_size: bootstrap_cfg[boot_size] for boot_size in bootstrap_keys}
        install_params = {}
        install_params['DEFAULT-RP'] = required_data
        install_params['STAGING'] = '/install_repo/tmp'
        install_params['STAGING'], secondary_staging = get_pd_provided_staging()
        return install_params, install_params['STAGING'], secondary_staging

def read_bootstrap_supported_arch ():
    with open("/etc/init.d/calvados_bootstrap.cfg") as cfg:
        data=cfg.read()
        cfg_dict = {}
        bootstrap_cfg = dict([(item.partition('=')[0],item.partition('=')[2]) for item in data.splitlines() if not item.strip().startswith('#') and item.strip()])
        return bootstrap_cfg.get('CALV_SUPPORTED_ARCHS', 'x86_64')

def is_cisco_pkg(platform, pkg_name):
    """ Check if the given pkg name is cisco pkg """
    if platform in pkg_name or 'iosxr' in pkg_name:
        return True 

def is_tp_pkg(platform, pkg_name):
    """ Return true if not cisco pkg """
    if not is_cisco_pkg(platform, pkg_name):
        return True 

def is_tp_smu(platform, pkg_name):
    """ Return true if not cisco pkg """
    if not is_cisco_pkg(platform, pkg_name) and \
        re.search('CSC[a-z][a-z]\d{5}',pkg_name):
        return True 

def get_vm_of_tpsmu(release):
    """ Given release field of TP SMU return VM  """
    vmtyoe = 'None'
    vmtype = release.split('.')[-1]
    return vmtype

def get_local_node_ip():
    """ Get local IP of the node """
    import socket
    s = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
    try:
        # doesn't even have to be reachable
        s.connect(('192.0.0.1', 1))
        IP = s.getsockname()[0]
    except:
        IP = '127.0.0.1'
    finally:
        s.close()
    return IP

# TBD
def node_supports_lcp ():
    return False

def update_prep_lvindex (vmtype, lvindex):
    lvfile = ''
    lvfile_dup = ''
    if vmtype == C.HOST:
        lvfile = os.path.join (C.LOCALREPO, "host_lv_state.txt")
    elif vmtype == C.CALVADOS:
        lvfile = os.path.join (C.LOCALREPO, "calvados_lv_state.txt")
    elif vmtype == C.XR:
        lvfile = os.path.join (C.LOCALREPO, "xr_lv_state.txt")
        lvfile_dup = os.path.join (C.LOCALREPO, "sdr/default-sdr", "xr_lv_state.txt")
    elif vmtype == C.XR_LCP:
        lvfile = os.path.join (C.LOCALREPO, "xr_lcp_lv_state.txt")
        lvfile_dup = os.path.join (C.LOCALREPO, "sdr/default-sdr", "xr_lcp_lv_state.txt")

    if not os.path.isfile (lvfile):
        return

    try:
        with open (lvfile, 'r') as fd:
            data = fd.read().split()
            data [2] = str(lvindex)

        with open (lvfile, 'w') as fd:
            fd.write (' '.join(data))
    except IOError:
        pass
    except:
        raise

    if os.path.isfile (lvfile_dup):
        with open (lvfile_dup, 'w') as fd:
            fd.write (' '.join(data))


from multiprocessing.managers import SyncManager
from multiprocessing import Process
class ClientManager (SyncManager): pass
ClientManager.register ('get_error_queue')

def connect_to_server (manager):
    try:
        manager.connect ()
    except:
        raise

def connect_manager(addr):
    logger = set_logger()
    dev_dict = {}
    manager = ClientManager(
            address=addr, authkey=C.INSTAUTHKEY)
    ''' Check if server process is up '''
    connect_proc = Process (target = connect_to_server, args = (manager,))
    connect_proc.start ()
    connect_proc.join (5)
    if connect_proc.is_alive() or connect_proc.exitcode:
        logger.error ("Timedout while connecting to server. Terminating")
        connect_proc.terminate()
        sys.exit (0)
    logger.info ('Server @ %s is up. Proceed with connect here.'%(str(addr)))
    manager.connect()
    return manager

def get_sys_orch_addr():
    sys_orch_ip = '0.0.0.0'
    sys_orch_port = 0

    if len(sys.argv) > 1:
        if sys.argv[1].upper() == "SYSTEM":
            with open (C.SYS_ORCH_ADDR_FILE, 'r') as fd:
                import json
                sys_orch_ip, sys_orch_port = json.load(fd)
                sys_orch_port = sys_orch_port
        elif sys.argv[1].upper() == "NODE":
            import ConfigParser
            config = ConfigParser.ConfigParser()
            config.read (C.NODE_CFG)

            sys_orch_ip = config.get(C.NODESECTION, C.REPOIP)
            sys_orch_port = config.get(C.NODESECTION, C.SYSORCHPORT)

    return sys_orch_ip, int(sys_orch_port)

def send_err_msg_to_sys_orch(msg):
    logger = set_logger()
    try:
        ip, port = get_sys_orch_addr()
        addr = (ip, port)
        manager = connect_manager(addr)
        queue = manager.get_error_queue()
        queue.put((get_local_node_ip(),msg))
        logger.error(msg)
    except:
        logger.error('Error in send_err_msg_to_sys_orch')
        exc_info = sys.exc_info()
        TB = traceback.format_exc()
        logger.error (TB)

