import multiprocessing
import threading
import logging
import sys
import traceback
import os
from logging.handlers import RotatingFileHandler
import time
try:
    import queue
except ImportError:
    import Queue as queue # Python 2.

class MultiProcessHandler(logging.Handler):
    def __init__(self, filename, mode, maxBytes, backupCount):
        logging.Handler.__init__(self)

        self._handler = RotatingFileHandler(filename, mode, maxBytes, backupCount)
        self.queue = multiprocessing.Queue(-1)
        self._is_closed = False
        self._receive_thread = threading.Thread(target=self.receive)
        self._receive_thread.daemon = True
        self._receive_thread.start()

    def setFormatter(self, fmt):
        logging.Handler.setFormatter(self, fmt)
        self._handler.setFormatter(fmt)

    def receive(self):
        while not (self._is_closed and self.queue.empty()):
            try:
                [_name, record] = self.queue.get(timeout=1)
                #print(_name, self.__class__.__name__, record.msg )
                if not record.msg:
                    continue
                self._handler.emit(record)
                #print('received on pid {}'.format(os.getpid()))
            except (KeyboardInterrupt, SystemExit):
                raise
            except EOFError:
                break
            except queue.Empty:
                pass
            except:
                traceback.print_exc(file=sys.stderr)
        self.queue.close()
        self.queue.join_thread()

    def send(self, s):
        self.queue.put_nowait([self.__class__.__name__,s])

    def _format_record(self, record):
        # ensure that exc_info and args have been stringified. Removes any
        # chance of unpickleable things inside and possibly reduces message size
        # sent over the pipe
        if record.args:
            record.msg = record.msg % record.args
            record.args = None
        if record.exc_info:
            dummy = self.format(record)
            record.exc_info = None

        return record

    def emit(self, record):
        try:
            s = self._format_record(record)
            self.send(s)
        except (KeyboardInterrupt, SystemExit):
            raise
        except Exception as e:
            self.handleError(record)

    def close(self):
        if not self._is_closed:
            self._is_closed = True
            time.sleep(1)
            self._receive_thread.join(10.0)  # Waits for receive queue to empty.
            time.sleep(1)
            self._handler.close()
            logging.Handler.close(self)

