#! python

from utils import inst_utils as utils
from utils import errors
import constants as C
import re
import os
import shutil
from logger_init import get_logger
logger = get_logger()

def get_lvindex2prep (oper_id):
    index = -1
    cmd="source /opt/cisco/calvados/bin/install-functions.sh; get_unique_lvid %s"%(oper_id)
    try:
        index = int(utils.run_cmd (cmd))
    except:
        raise errors.PrepareRunCmdFailed ("Unable to run command %s"%(cmd))
    return index
    
def create_and_attach_device (vmtype, partsize, label, targetdev, lvid):
    logger.debug("For attach, got %s %s %s %s"%(vmtype, partsize, label, targetdev))
    lvname = '%s_lv%s'%(vmtype, lvid)
    if vmtype == C.HOST:
        label = '%s%s'%(label, lvid)

    if vmtype == C.HOST:
        cmd = "'source /etc/init.d/disk-functions; create_mkfs_host_boot_lv %s %s %s'" %(partsize, lvname, label)
    elif vmtype == C.CALVADOS:
        cmd = "'source /etc/init.d/disk-functions; create_mkfs_calvados_boot_lv %s %s %s'" %(partsize, lvname, label)
    elif vmtype == C.XR or vmtype == C.XR_LCP:
        cmd = "'source /etc/init.d/disk-functions; create_mkfs_xr_boot_lv %s %s %s'" %(partsize, lvname, label)

    utils.run_cmd_on_host (cmd)
    source_dev = "/dev/%s/%s_lv%s"%("panini_vol_grp", vmtype, lvid)
    utils.attach_lv (source_dev, targetdev)
    targetdev="/dev/%s"%(targetdev)
    cmd = "/opt/cisco/calvados/bin/install-functions.py get_virt_method"
    virt_method = utils.run_cmd (cmd)
    if virt_method == 'vm':
        return {}
    else:
        return {vmtype:targetdev}

def install_pre_check (tmp_staging, total_pkg_size):
    # Check that staging location has enough disk space to dowload packages.
    cmd = 'python /opt/cisco/calvados/bin/lvutils.py %s --lvclean --staging_loc %s --pkg_size %s '%(C.NODE, tmp_staging, total_pkg_size)
    run_script (cmd)
    # Check that /tmp, /var/log and / has enough free space (in M) for operation to proceed.
    cmd = 'python /opt/cisco/calvados/bin/lvutils.py %s --staging_loc %s --pkg_size %s '%(C.NODE, "/tmp", '10')
    run_script (cmd)
    cmd = 'python /opt/cisco/calvados/bin/lvutils.py %s --staging_loc %s --pkg_size %s '%(C.NODE, "/", '100')
    run_script (cmd)
    cmd = 'python /opt/cisco/calvados/bin/lvutils.py %s --staging_loc %s --pkg_size %s '%(C.NODE, "/var/log", '10')
    run_script (cmd)
    return None

def run_script (input_cmd):
    utils.run_cmd (input_cmd)

def validatedevice (vmtype, targetdev, vm_tuple, lvindex):
    label2check = '%s_lv%s'%(vm_tuple[1], lvindex)
    try:
        lo_device = utils.setup_loop (targetdev)
        devlabel = utils.get_device_label (lo_device)
        if devlabel == label2check:
            logger.debug('Label check passed. Return validation as True')
            return 1
        else:
            logger.debug('Label check failed. Return validation as False for label check %s'%(label2check))
            return 0
    except: 
        raise
    finally: 
        logger.debug("About to detach loop device")
        utils.detach_loop (lo_device)

def setup_repo_giso_rpms (iso_mnt, initrd_dir, giso_rpms):
    repo_sync_dict = {}
    non_optim_giso_index = 4
    giso_file = os.path.join (iso_mnt, "giso_info.txt")
    giso_rpm_tool = "/opt/cisco/calvados/bin/install-functions.py"
    if os.path.isfile (giso_file):
        cmd_rpm_path = "%s get_giso_rpm_path %s"%(giso_rpm_tool, giso_file)
        path_index = utils.run_cmd (cmd_rpm_path)
        if int(path_index) == non_optim_giso_index:
            # Giso RPMS are packaged at top level iso mount.
            giso_rpm_dir = iso_mnt
            logger.debug("GISO rpms are packaged at top level mount")
        else:
            # Giso RPMS are packaged within the initrd passed to this function.
            giso_rpm_dir = initrd_dir
            logger.debug("GISO rpms are packaged within initrd")
        for vmtype in C.SUPPORTED_VMTYPE:
            if vmtype == C.XR_LCP:
                continue
            repo = ''
            giso_rpms = []
            if vmtype == C.HOST:
                repo = C.CAL_REPO
            elif vmtype == C.CALVADOS:
                repo = C.CAL_REPO
            elif vmtype == C.XR:
                repo = C.XR_REPO
            giso_rpm_path = os.path.join (giso_rpm_dir, "%s_rpms"%(vmtype.lower()))
            if os.path.isdir (giso_rpm_path):
                giso_rpms = [ f for f in os.listdir(giso_rpm_path) \
                    if os.path.isfile(os.path.join(giso_rpm_path, f)) and ".rpm" in f ]
            for rpm in giso_rpms:
                if not repo_sync_dict.has_key(vmtype):
                    repo_sync_dict[vmtype] = []
                src_rpm = os.path.join (giso_rpm_path, rpm)
                dest_rpm = os.path.join (repo, rpm.replace ('.rpm', ''))
                if not os.path.isfile (dest_rpm):
                    shutil.copy (src_rpm, dest_rpm)            
                repo_sync_dict[vmtype].append (dest_rpm)

    return repo_sync_dict
            

def extract_iso_to_instrepo (plat, host_iso, cal_iso, xr_iso, iso_version, mnt_dir):
    nbi_dest = None
    nbi_initrd_dest_cal = None
    nbi_initrd_dest_host = None
    host_iso_path = os.path.join (mnt_dir, "iso", "host.iso")
    cal_iso_path = os.path.join (mnt_dir, "iso", "%s-sysadmin.iso"%(plat))
    xr_iso_path = os.path.join (mnt_dir, "iso", "%s-xr.iso"%(plat))
    total_size = 0
    if os.path.isfile (host_iso):
	logger.info ("Removed %s from repo"%(host_iso))
	os.unlink(host_iso)
    if os.path.isfile (cal_iso):
	logger.info ("Removed %s from repo"%(cal_iso))
	os.unlink(cal_iso)
    if os.path.isfile (xr_iso):
	logger.info ("Removed %s from repo"%(xr_iso))
	os.unlink(xr_iso)
    if os.path.isfile (host_iso_path):
	size = os.path.getsize (host_iso_path)
	total_size += (size >> 20) + 1
    if os.path.isfile (cal_iso_path):
	size = os.path.getsize (cal_iso_path)
	total_size += (size >> 20) + 1
    if os.path.isfile (xr_iso_path):
	size = os.path.getsize (xr_iso_path)
	total_size += (size >> 20) + 1
    total_size += 100
    cmd = 'python /opt/cisco/calvados/bin/lvutils.py %s --staging_loc %s --pkg_size %s '%(C.SYSTEM, "/install_repo", total_size)
    try:
        utils.run_cmd(cmd)
    except:
	msg = "Insufficient space in install_repo in sysadmin."
	utils.send_err_msg_to_sys_orch(msg)
	raise errors.PrepareInvalidFileType (msg)
    if not os.path.isfile (host_iso):
        shutil.copy (host_iso_path, host_iso)
    if not os.path.isfile (cal_iso):
        shutil.copy (cal_iso_path, cal_iso)
    if not os.path.isfile (xr_iso):
        shutil.copy (xr_iso_path, xr_iso)
    # TBD: This should be a part of PD utility
    if 'arm' in utils.read_bootstrap_supported_arch ():
        nbi, nbi_initrd = utils.read_iso_name_mdata (mnt_dir)
        if not nbi or not nbi_initrd:
            msg = 'NBI metadata not available for arm architecture'
            utils.send_err_msg_to_sys_orch(msg)
            raise errors.PrepareInvalidFileType ("NBI metadata not available during repo preparation")
        nbi_initrd_src = os.path.join (mnt_dir, "nbi-initrd", nbi_initrd)
        nbi_dest = os.path.join (C.CAL_REPO, "%s-%s"%(nbi, iso_version))
        nbi_initrd_dest_cal = os.path.join (C.CAL_REPO, "%s-%s"%(nbi_initrd, iso_version))
        nbi_initrd_dest_host = os.path.join (C.HOST_REPO, "%s-%s"%(nbi_initrd, iso_version))
        # At this stage, only initrd can be copied to repo.
        # NBI image itself can only be copied as part of calvados partition preparation.
        if not os.path.isfile (nbi_initrd_dest_cal):
            shutil.copy (nbi_initrd_src, nbi_initrd_dest_cal)
        if not os.path.isfile (nbi_initrd_dest_host):
            shutil.copy (nbi_initrd_src, nbi_initrd_dest_host)
        return nbi_dest, nbi_initrd_dest_cal, nbi_initrd_dest_host
    return None, None, None

def parse_pkg_list (plat, pkgList):
    host_pkgs = []
    cal_pkgs = []
    xr_pkgs = []
    pkg_considered = ''
    cal_arch = [ 'x86_64' ]
    if 'arm' in utils.read_bootstrap_supported_arch ():
        cal_arch.append ('arm')
    xr_arch = [ 'x86_64' ]

    for pkg in pkgList:
        pkg2consider = str(pkg)
        if not pkg:
            continue
        vm_type = ''
        tp_pkg = False
        if utils.is_tp_pkg (plat, pkg):
            tp_pkg = True

        pkg = pkg.replace (".rpm", '')
        pkg = pkg.replace (".x86_64", '')
        pkg = pkg.replace (".arm", '')
        m = re.search(r'(.*)-(.*)-(.*)',pkg)
        if m:
            (name, version, release) = m.groups()
            if tp_pkg:
                vm_type = release.split('.')[-1]
            elif "-sysadmin-hostos" in name:
                vm_type = release.split('.')[-1]
            elif "sysadmin" in name:
                vm_type = C.CALVADOS
            else:
                vm_type = C.XR
            if vm_type == 'admin':
                vm_type = C.CALVADOS

            if vm_type not in C.SUPPORTED_VMTYPE:
                raise errors.PrepareInvalidFileType ("Invalid VM TYPE")

        else:
            msg = "Invalid RPM file %s in package list"%(pkg)
            utils.send_err_msg_to_sys_orch(msg)
            raise errors.PrepareInvalidFileType ("Invalid RPM in package list")

        pkg2consider = pkg2consider.replace ('.rpm', '')
        if vm_type == C.HOST or vm_type == C.CALVADOS:
            pkg_considered = os.path.join(C.CAL_REPO, pkg2consider)
            archlist = cal_arch
        elif vm_type == C.XR:
            pkg_considered = os.path.join(C.XR_REPO, pkg2consider)
            archlist = xr_arch
    
        if not os.path.isfile (pkg_considered):
            for arch in archlist:
                pkg2check = '%s.%s'%(pkg_considered, arch)
                if os.path.isfile (pkg2check):
                    pkg_considered = pkg2check
                    break    

        if os.path.isfile (pkg_considered):
            cmd = "rpm -qp --qf '%s' %s"%("%{PACKAGEPRESENCE}", pkg_considered)
            pkgpresence = utils.run_cmd (cmd)
            if pkgpresence.lower() == 'mandatory':
                continue
        else:
            msg = "Package %s is not available in repository"%(pkg_considered)
            utils.send_err_msg_to_sys_orch(msg)
            raise errors.PrepareInvalidFileType (msg)
            
        if vm_type == C.HOST:
            host_pkgs.append (os.path.join(C.CAL_REPO, pkg))
        elif vm_type == C.CALVADOS:
            cal_pkgs.append (os.path.join(C.CAL_REPO, pkg))
        elif vm_type == C.XR:
            xr_pkgs.append (os.path.join(C.XR_REPO, pkg))
            
    return host_pkgs, cal_pkgs, xr_pkgs

def create_iso_cfg (staging_loc, cfg, giso_rpms):
    import json
    with open (C.SYS_ORCH_ADDR_FILE, 'r') as fd:
        ip, port = tuple(json.load(fd))
    port = int(port)

    import ConfigParser
    cfg_file = os.path.join (staging_loc, "config.cfg")
    config = ConfigParser.ConfigParser()
    config.add_section (C.NODESECTION)
    config.set (C.NODESECTION, C.ISO, cfg["iso"])
    config.set (C.NODESECTION, C.REPOIP, cfg["repoIP"])
    config.set (C.NODESECTION, C.SYSORCHPORT, port)
    config.set (C.NODESECTION, C.STAGING, cfg["staging"])
    config.set (C.NODESECTION, C.ADMINREPO, "/install_repo/gl/calvados")
    config.set (C.NODESECTION, C.XRREPO, "/install_repo/gl/xr")
    config.set (C.NODESECTION, C.HOSTREPO, "/install_repo/gl/host")
    config.set (C.NODESECTION, C.XR_LABEL, C.label_vmtype[C.XR])
    config.set (C.NODESECTION, C.HOST_LABEL, C.label_vmtype[C.HOST])
    config.set (C.NODESECTION, C.CAL_LABEL, C.label_vmtype[C.CALVADOS])
    config.set (C.NODESECTION, C.XR_LCP_LABEL, C.label_vmtype[C.XR_LCP])
    config.set (C.NODESECTION, C.CALVMS, ','.join(cfg["calVms"]))
    config.set (C.NODESECTION, C.XRVMS, ','.join(cfg["xrVms"]))
    config.set (C.NODESECTION, C.LCXRVMS, ','.join(cfg["xrlcpVms"]))

    host_pkg_list, cal_pkg_list, xr_pkg_list = parse_pkg_list (cfg["plat"], cfg["pkgList"])
    logger.info ("Host packages %s"%(' '.join (host_pkg_list)))
    logger.info ("Calvados packages %s"%(' '.join (cal_pkg_list)))
    logger.info ("Xr packages %s"%(' '.join (xr_pkg_list)))
    for vmtype in C.SUPPORTED_VMTYPE:
        if vmtype == C.XR_LCP:
            continue
        giso_pkg2consider = set(map (lambda x : re.sub (r'.\w+.rpm', '', x), giso_rpms[vmtype]))
        logger.info ("Giso RPM for vmtype %s: %s"%(vmtype, ' '.join(giso_pkg2consider)))
        if giso_rpms[vmtype]:
            if vmtype == C.XR:
                xr_pkg_list += [ os.path.join (C.XR_REPO, x) for x in giso_pkg2consider if x ]
            elif vmtype == C.CALVADOS:
                cal_pkg_list += [ os.path.join (C.CAL_REPO, x) for x in giso_pkg2consider if x ]
            elif vmtype == C.HOST:
                host_pkg_list += [ os.path.join (C.CAL_REPO, x) for x in giso_pkg2consider if x ]
    host_pkg_list.append (cfg["host_iso"])
    cal_pkg_list.append (cfg["cal_iso"])
    xr_pkg_list.append (cfg["xr_iso"])

    host_pkgs = ','.join(host_pkg_list)
    cal_pkgs = ','.join(cal_pkg_list)
    xr_pkgs = ','.join(xr_pkg_list)
    
    for vmtype in C.SUPPORTED_VMTYPE:
        config.add_section ("%s_section"%(vmtype))
        if vmtype == C.HOST:
            config.set ("%s_section"%(vmtype), C.PKGLIST, host_pkgs)
        elif vmtype == C.CALVADOS:
            config.set ("%s_section"%(vmtype), C.PKGLIST, cal_pkgs)
        elif vmtype == C.XR or vmtype == C.XR_LCP:
            config.set ("%s_section"%(vmtype), C.PKGLIST, xr_pkgs)
        else:
            raise errors.PrepareInvalidFileType ("Invalid VM TYPE")

    mdata_dict = cfg["inst_params"]
    for k, v in mdata_dict.iteritems():
        if k == 'STAGING':
            continue
        config.add_section (k) 
        config.set (k, C.HOST_BOOT_SIZE, v["DISK_HOST_BOOT_SIZE"])
        config.set (k, C.CAL_BOOT_SIZE, v["DISK_CALVADOS_BOOT_SIZE"])
        config.set (k, C.XR_BOOT_SIZE, v["DISK_XR_BOOT_SIZE"])

    with open (cfg_file, 'wb') as cfg_fd:
        config.write (cfg_fd)
    return cfg_file, host_pkgs, cal_pkgs, xr_pkgs
   
def append_sizes_to_node_config(cfgfile):
    import ConfigParser
    config = ConfigParser.ConfigParser()
    config.read (cfgfile)

    for vmtype in C.SUPPORTED_VMTYPE:
        if vmtype == C.HOST:
            host_pkgs = config.get ("%s_section"%(vmtype), C.PKGLIST)
        elif vmtype == C.CALVADOS:
            cal_pkgs = config.get ("%s_section"%(vmtype), C.PKGLIST)
        elif vmtype == C.XR or vmtype == C.XR_LCP:
            xr_pkgs = config.get ("%s_section"%(vmtype), C.PKGLIST)
        else:
            raise errors.PrepareInvalidFileType ("Invalid VM TYPE")


    size=0
    total_host_pkg_size = 0
    for pkg in host_pkgs.split(','):
        try:
            size = os.path.getsize(pkg)
        except OSError:
            size = os.path.getsize(pkg+'.x86_64')
        total_host_pkg_size += (size >> 20) + 1

    size=0
    total_cal_pkg_size = 0
    for pkg in cal_pkgs.split(','):
        try:
            size = os.path.getsize(pkg)
        except OSError:
            size = os.path.getsize(pkg+'.x86_64')
        total_cal_pkg_size += (size >> 20) + 1

    size=0
    total_xr_pkg_size = 0
    for pkg in xr_pkgs.split(','):
        try:
            size = os.path.getsize(pkg)
        except OSError:
            size = os.path.getsize(pkg+'.x86_64')
        total_xr_pkg_size += (size >> 20) + 1

    config.set (C.NODESECTION, C.HOSTPKGSIZE, total_host_pkg_size)
    config.set (C.NODESECTION, C.CALPKGSIZE, total_cal_pkg_size)
    config.set (C.NODESECTION, C.XRPKGSIZE, total_xr_pkg_size)

    with open (cfgfile, 'wb') as cfg_fd:
        config.write (cfg_fd)
    return

def sync_repo_rps (sync_repo_dict, rp_ips):
    if not sync_repo_dict:
        return
    for vmtype in C.SUPPORTED_VMTYPE:
        if vmtype == C.XR_LCP:
            continue
        if not sync_repo_dict.has_key(vmtype):
            continue
        repo = ''
        if vmtype == C.HOST:
            repo = C.CAL_REPO
        elif vmtype == C.CALVADOS:
            repo = C.CAL_REPO
        elif vmtype == C.XR:
            repo = C.XR_REPO
        pkglist = [os.path.basename (x) for x in sync_repo_dict[vmtype]]
        copy_pkgs_to_repo (repo, pkglist, rp_ips)
    return

def copy_pkgs_to_repo (src_dir, pkglist, rp_ips):
    my_ip = utils.get_local_node_ip()
    logger.debug ("Local IP is %s"%(my_ip))
    logger.debug ("Packages are to be copied to %s"%(' '.join(rp_ips)))
    sync_ips = []
    try:
        sync_ips = rp_ips
        if my_ip in sync_ips:
            sync_ips.remove(my_ip)
    except ValueError:
        pass
        
    if sync_ips:
        ips = ' '.join(sync_ips)
        source_dir = src_dir
        dest_dir = src_dir
        pkglist = ' '.join(pkglist)
        cmd = ("python /opt/cisco/calvados/bin/utils/install_add_replicate.py -c cp -s %s -d %s -p %s -x %s -a %s -sdr default-sdr"%(source_dir, dest_dir, pkglist, my_ip, ips))
        out = utils.run_cmd(cmd)
        logger.debug("CMD= %s OUT=%s"%(cmd, out))

    return       

def parse_create_iso_cfg (iso, repoIP, pkgList, calVms, xrVms, xrlcpVms):
    initrd_extracted = ''
    mnt_dir = ''
    mnt_dir_internal = ''
    iso_mnt = ''
    iso_mnt_internal = ''
    secondary_staging = None
    giso = False
    giso_config_f = ''
    giso_autorun_f = '' 
    config = {}
    repo_sync_dict = None
    try:
        if os.path.isfile (iso):
            config["iso"] = iso
            config["repoIP"] = repoIP
            config["pkgList"] = pkgList
            config["calVms"] = calVms
            config["xrVms"] = xrVms
            config["xrlcpVms"] = xrlcpVms
            giso_rpms = {}
            for vmtype in C.SUPPORTED_VMTYPE:
                vm = vmtype.lower()
                if vmtype == C.XR_LCP:
                    continue
                if vmtype == C.CALVADOS:
                    vm = "sysadmin"
                giso_rpms[vmtype] = []
            file_type = utils.get_file_type (iso)
            if file_type != "bundle":
                msg = "Invalid file type of %s"%(iso) 
                utils.send_err_msg_to_sys_orch(msg)
                raise errors.PrepareInvalidFileType ("ISO file type is not a bundle")
            iso_mnt = utils.mount_iso (iso)
            plat, host_iso, cal_iso, xr_iso, iso_version = utils.read_iso_info_mdata (iso_mnt)
            # Check if giso_info.txt is present for GISO.
            if os.path.isfile (os.path.join (iso_mnt, "giso_info.txt")):
                giso = True
                for vmtype in C.SUPPORTED_VMTYPE:
                    vm = vmtype.lower()
                    if vmtype == C.XR_LCP:
                        continue
                    if vmtype == C.CALVADOS:
                        vm = "sysadmin"
                    giso_rpms[vmtype] = utils.read_giso_rpms_mdata (iso_mnt, vm, plat)
            config["plat"] = plat
            config["host_iso"] = host_iso
            config["cal_iso"] = cal_iso
            config["xr_iso"] = xr_iso
            try:
                inst_params, tmp_staging, secondary_staging = utils.read_install_params (iso_mnt)
            except:
                inst_params, tmp_staging, secondary_staging = utils.read_bootstrap_install_params ()
            config["inst_params"] = inst_params
            config["staging"] = tmp_staging
            initrd_temp_staging = os.path.join (tmp_staging, "tmpXXX")
            mnt_dir = utils.run_cmd ("mktemp -d %s"%(initrd_temp_staging))
            initrd_extracted = mnt_dir
            utils.run_cmd ('python /opt/cisco/calvados/bin/lvutils.py %s --staging_loc %s --initrd %s '%(C.SYSTEM, tmp_staging, iso_mnt + '/boot/initrd.img' ))

            utils.extract_initrd_to_part (iso_mnt, mnt_dir)
            if os.path.isfile (os.path.join(mnt_dir, "iso", "system_image.iso")):
                # Use secondary staging provided for extration of system_image.iso
                if secondary_staging:
                    initrd_temp_staging = os.path.join (secondary_staging, "tmpXXX")
                mnt_dir_internal = utils.run_cmd ("mktemp -d %s"%(initrd_temp_staging))
                iso_mnt_internal = utils.mount_iso (os.path.join(mnt_dir, "iso", "system_image.iso"))
                utils.run_cmd ('python /opt/cisco/calvados/bin/lvutils.py %s --staging_loc %s --initrd %s '%(C.SYSTEM, secondary_staging, iso_mnt_internal + '/boot/initrd.img' ))
                utils.extract_initrd_to_part (iso_mnt_internal, mnt_dir_internal)
                initrd_extracted = mnt_dir_internal
                utils.umount_iso (iso_mnt_internal)
            nbi, nbiinitrd, nbiinitrd_host = extract_iso_to_instrepo (plat, host_iso, cal_iso, xr_iso, iso_version, initrd_extracted)
            cfgfile, host_pkgs, cal_pkgs, xr_pkgs = create_iso_cfg (tmp_staging, config, giso_rpms)
            if giso:
                giso_config = os.path.join (iso_mnt, "router.cfg")
                if os.path.isfile (giso_config):
                    giso_config_f = os.path.join (C.MISC_REPO, "router.cfg")
                    shutil.copy (giso_config, giso_config_f)   
                giso_autorun = os.path.join (iso_mnt, "autorun")
                if os.path.isfile (giso_autorun):
                    giso_autorun_f = os.path.join (C.MISC_REPO, "autorun")
                    shutil.copy (giso_autorun, giso_autorun_f)
                    
                repo_sync_dict = setup_repo_giso_rpms (iso_mnt, initrd_extracted, giso_rpms)
            append_sizes_to_node_config(cfgfile)    
            utils.umount_iso (iso_mnt)      
            return cfgfile, host_pkgs, cal_pkgs, xr_pkgs, host_iso, cal_iso, xr_iso, nbi, nbiinitrd, nbiinitrd_host, iso_version, giso_config_f, repo_sync_dict, giso_autorun_f 
        else:
            msg = "ISO file %s is not available at specified location"%(iso)
            utils.send_err_msg_to_sys_orch(msg)
            raise errors.PrepareInvalidFileType (msg)
    except:
        import sys
        ttype, value, traceback = sys.exc_info()
        raise ttype, value, traceback        
    finally:
        utils.run_cmd ("rm -rf %s"%(mnt_dir))
        utils.run_cmd ("rm -rf %s"%(mnt_dir_internal))
        utils.umount_iso (iso_mnt)      
        utils.umount_iso (iso_mnt_internal)      
                
def copy_status_update_rp_nodes (rpvms):
    if os.path.isfile (C.UPDATE_STATUS_FILE):    
        for ip in rpvms:
            try:
                cmd = "%s %s %s %s:%s"%(utils.vrf_str, 
                                        utils.scp_str, 
                                        C.UPDATE_STATUS_FILE, ip, 
                                        C.UPDATE_STATUS_FILE)
                utils.run_cmd (cmd)
            except:
                pass

def copy_gisoconfig_xr_rp_nodes (cfg_file, cfg_giso_cfg, rpvms):
    if os.path.isfile (cfg_file):
        for ip in rpvms:
            cmd = "%s %s %s %s:%s"%(utils.vrf_str, utils.scp_str, cfg_file, ip, cfg_giso_cfg)
            utils.run_cmd (cmd)
        os.unlink (cfg_file)
    return

def copy_config_to_cal_nodes (cfg_file, cfg_node_cfg, calvms):
    ret_dict = {}
    for ip in calvms:
        cmd = "%s %s %s %s:%s"%(utils.vrf_str, utils.scp_str, cfg_file, ip, cfg_node_cfg)
        utils.run_cmd (cmd)
        ret_dict [ip] = ip

    return ret_dict
        
def prep_misc_work_items (key, item):
    return {key:item}

def do_misc_work_items (obj, key):
    return obj.fn_pointer[key] ()

