#!/usr/bin/python
#------------------------------------------------------------------
# install_health_tool.py
#
# Copyright (c) 2015-2019 by cisco Systems, Inc.
# All rights reserved.
#------------------------------------------------------------------
#
import subprocess
import re
import sys
import os
import time
from const import *
import logging
import logging.handlers
import traceback
import time
import copy
from threading import Thread
import types
import json

"""
All Global Variables
"""
cal_vms = []  # Cal VMs across all cards
xr_vms = []   # XR VMs across all cards
rps_cal_vms = []  # Cal VMs running across RPs
rps_xr_vms = []  # Cal VMs running across RPs
cards_info = []  # All cards info. like their location and number of XR VMs
lead_rp_cal_vm = None  # Lead RP Cal VM Info
# Dictionary to hold sdr_instmgr XR vm per sdr {sdr_name : vm}
lead_xr_vm_per_sdr = {}
# Dictionary to hold XR VMs info. per sdr {sdr_name: [vm_info1, vm_info2]}
xr_vms_per_sdr = {}
platform_type = None  # Underlying Platform Type
# Prefix for executing cmds on platform for e.g., prefix ="chvrf 0 " for ncs6k
platform_prefix = ""
# Host_IP
host_ip = ""
# list of all the error object which hold the data for any plugin
# verification failed
errors = []
# below is the for running all the test cases for a specific platform
test_cases_cal = []
test_cases_host = []
test_cases_xr = []
rp_test_cases_cal = []
rp_test_cases_xr = []
plugins_running_data = {}
install_progress = False

# list of test_cases
cal_test_cases = ['cal_version',
                  'cal_smus',
                  'cal_local_active_swp',
                  'cal_local_committed_swp',
                  'cal_active_swp_pkg',
                  'xr_active_swp_pkg',
                  'rpmdb',
                  'lv_state_verification',
                  'boot_lvs',
                  'cal_ld_path_verification'
                  ]

host_test_cases = ['host_version',
                   'host_smus',
                   'rpmdb',
                   'grub_cfg',
                   ]

xr_test_cases = ['xr_version',
                 'xr_smus',
                 'rpmdb',
                 'xr_ld_path_verification'
                 ]

rp_cal_test_cases = ['cal_disk_space',
                     'cal_marker_files',
                     'cal_mount_points',
                     'cal_stale_symlinks',
                     'cal_prepared_packages',
                     'cal_master_active_swp',
                     'cal_master_committed_swp',
                     'xr_master_active_swp',
                     'xr_master_committed_swp',
                     'xr_local_active_swp',
                     'xr_local_committed_swp',
                     'cal_tftp_boot_image_version',
                     'cal_image',
                     'system_image_version',
                     'system_image_stale_symlink',
                     'system_image_file',
                     'system_grub_file',
                     'cal_tmp_staging_dir',
                     'system_grub_efi_file',
                     'cal_install_tmp_staging_dir' 
                     ]

rp_xr_test_cases = ['xr_disk_space',
                    'xr_marker_files',
                    'xr_mount_points',
                    'xr_stale_symlinks',
                    'xr_prepared_packages',
                    'xr_tmp_staging_dir'
                    ]
# ALL Commands
CAL_INSTALL_DIR_SPACE_CMD = ''
XR_INSTALL_DIR_SPACE_CMD = ''
CAL_INSTALL_MARKER_FILES = ''
XR_INSTALL_MARKER_FILES = ''
CAL_INSTALL_MOUNT_POINTS = ''
XR_INSTALL_MOUNT_POINTS = ''
CAL_INSTALL_SYMLINKS = ''
XR_INSTALL_SYMLINKS = ''
CAL_SHOW_PREPARED_CMD = ''
XR_SHOW_PREPARED_CMD = ''
CAL_MASTER_ACTIVE_SWP = 'cat /install_repo/gl/instdb/clos-master-swprofile-active.txt | egrep Version'
XR_MASTER_ACTIVE_SWP = 'find \/install_repo\/gl\/instdb\/sdr -name xr-swprofile-active.txt -exec cat {} + | egrep "Version|SDR"'
CAL_MASTER_COMMITTED_SWP = 'cat /install_repo/gl/instdb/clos-master-swprofile-committed.txt | egrep Version'
XR_MASTER_COMMITTED_SWP = 'find \/install_repo\/gl\/instdb\/sdr  -name xr-swprofile-committed.txt -exec cat {} + | egrep "Version|SDR"'
CAL_LOCAL_ACTIVE_SWP = 'cat /install_repo/local/clos-swprofile-active.txt | egrep Version'
XR_LOCAL_ACTIVE_SWP = 'find \/install_repo\/local\/sdr -name xr-swprofile-active.txt -exec cat {} + | egrep "Version|SDR"'
CAL_LOCAL_COMMITTED_SWP = 'cat /install_repo/local/clos-swprofile-committed.txt | egrep Version'
XR_LOCAL_COMMITTED_SWP = 'find \/install_repo\/local\/sdr  -name xr-swprofile-committed.txt -exec cat {} + | egrep "Version|SDR"'
CAL_TFTP_BOOT_IMAGE = 'ls /misc/disk1/tftpboot/{*mini*,*golden*}'
CAL_IMAGE = 'ls  /install_repo/gl/calvados/*sysadmin*'
SYSTEM_IMAGE_SYMLINK_VERSION = 'ls -l /misc/disk1/tftpboot/system_image.iso  | awk \'{print $NF}\''
SYSTEM_IMAGE_STALE_SYMLINK = 'find -L /misc/disk1/tftpboot/system_image.iso -type l'
LV_STATE_VALUES = 'array=($(ls /install_repo/local/*lv*txt)); for i in \"${array[@]}\"; do echo $i; cat $i; echo;  done'
BOOT_LVS_CMD = ''
GRUB_CMD = 'y=`fdisk -l | grep sda | grep EFI | awk \'{print $1}\'`; mkdir -p /tmp/3; mount $y /tmp/3; cat /tmp/3/EFI/boot/grub.cfg; umount /tmp/3; rm -rf /tmp/3;'           
PCI_DISK_CMD = 'pvscan | grep pci_disk'
RPMDB_LIB = 'ls /var/lib/rpm/'
CAL_ACTIVE_SWP_PKG_CMD = "cat /install_repo/local/clos-swprofile-active.txt"
XR_ACTIVE_SWP_PKG_CMD = "cat /install_repo/local/xr-swprofile-active.txt"
CAL_LD_PATH_VERIFICATION_CMD = "python /opt/cisco/calvados/script/install_health_tool_ld_verification.py" 
XR_LD_PATH_VERIFICATION_CMD = "python /pkg/bin/install_health_tool_ld_verification.py"
SYSTEM_IMAGE_FILE = "find /misc/disk1/tftpboot/ -name system_image.iso"
SYSTEM_GRUB_FILE = "find /misc/disk1/tftpboot/ -name grub.cfg"
CAL_TMP_STAGING_DIR = 'find "/install/tmp_staging" -type f -exec echo Found file {} \;'
XR_TMP_STAGING_DIR = 'find "/install/tmp_staging" -type f -exec echo Found file {} \;'
SYSTEM_GRUB_EFI_FILE = "find /misc/disk1/tftpboot/ -name grub.efi"
CAL_INSTALL_TMP_STAGING_DIR = 'find "/install/tmp" -type d -exec echo Found file {} \;'

CAL_MASTER_ACTIVE_SWP_ARM_ADMIN = 'cat /install_repo/gl/instdb/clos-master-swprofile-active.txt | grep "CSC[a-z]\{2\}[0-9]\{5\}" | egrep ".arm $" | egrep "admin" | awk \'{print $NF}\''
CAL_MASTER_ACTIVE_SWP_ARM_HOST = 'cat /install_repo/gl/instdb/clos-master-swprofile-active.txt | grep "CSC[a-z]\{2\}[0-9]\{5\}" | egrep ".arm $" | egrep "host" | awk \'{print $NF}\''

rp_cal_commands = {'cal_disk_space': CAL_INSTALL_DIR_SPACE_CMD,
                   'cal_marker_files': CAL_INSTALL_MARKER_FILES,
                   'cal_mount_points': CAL_INSTALL_MOUNT_POINTS,
                   'cal_stale_symlinks': CAL_INSTALL_SYMLINKS,
                   'cal_prepared_packages': CAL_SHOW_PREPARED_CMD,
                   'cal_master_active_swp': CAL_MASTER_ACTIVE_SWP,
                   'cal_master_committed_swp': CAL_MASTER_COMMITTED_SWP,
                   'xr_master_active_swp': XR_MASTER_ACTIVE_SWP,
                   'xr_master_committed_swp': XR_MASTER_COMMITTED_SWP,
                   'xr_local_active_swp': XR_LOCAL_ACTIVE_SWP,
                   'xr_local_committed_swp': XR_LOCAL_COMMITTED_SWP,
                   'cal_tftp_boot_image_version': CAL_TFTP_BOOT_IMAGE,
                   'cal_image': CAL_IMAGE,
                   'system_image_version': SYSTEM_IMAGE_SYMLINK_VERSION,
                   'system_image_stale_symlink': SYSTEM_IMAGE_STALE_SYMLINK,
                   'system_image_file' : SYSTEM_IMAGE_FILE,
                   'system_grub_file' : SYSTEM_GRUB_FILE,
                   'cal_tmp_staging_dir' : CAL_TMP_STAGING_DIR,
                   'system_grub_efi_file' : SYSTEM_GRUB_EFI_FILE,
                   'cal_install_tmp_staging_dir' : CAL_INSTALL_TMP_STAGING_DIR
                   }

rp_xr_commands = {'xr_disk_space': XR_INSTALL_DIR_SPACE_CMD,
                  'xr_marker_files': XR_INSTALL_MARKER_FILES,
                  'xr_mount_points': XR_INSTALL_MOUNT_POINTS,
                  'xr_stale_symlinks': XR_INSTALL_SYMLINKS,
                  'xr_prepared_packages': XR_SHOW_PREPARED_CMD,
                  'xr_tmp_staging_dir' : XR_TMP_STAGING_DIR
                  }

# Below are the data which needs to be be collected at different VMs
# this is the dictionary where {cmd_name:cmd}
cal_data_commands = {'cal_version': 'rpm -qa | grep hostos',
                     'cal_smus': 'rpm -qa | grep \"CSC[a-z]\{2\}[0-9]\{5\}\"',
                     'cal_local_active_swp': CAL_LOCAL_ACTIVE_SWP,
                     'cal_local_committed_swp': CAL_LOCAL_COMMITTED_SWP,
                     'cal_active_swp_pkg' : CAL_ACTIVE_SWP_PKG_CMD,
                     'xr_active_swp_pkg' : XR_ACTIVE_SWP_PKG_CMD,
                     'cal_ld_path_verification' : CAL_LD_PATH_VERIFICATION_CMD,
                     'rpmdb' : RPMDB_LIB,
                     'lv_state_verification': LV_STATE_VALUES,
                     'boot_lvs' :BOOT_LVS_CMD
                     }

host_data_commands = {'host_version': 'rpm -qa | grep hostos',
                      'host_smus': 'rpm -qa | grep \"CSC[a-z]\{2\}[0-9]\{5\}\"', 
                      'rpmdb' : RPMDB_LIB,
                      'grub_cfg' : GRUB_CMD,
                      }

xr_data_commands = {'xr_version': 'rpm -qa | grep spirit-boot',
                    'xr_smus': 'rpm -q -g "IOS-XR"',
                    'xr_ld_path_verification' : XR_LD_PATH_VERIFICATION_CMD,
                    'rpmdb' :RPMDB_LIB
                    }

# this dictionary is the place holder for all the data across all nodes
nodes_data = {}

# this dictionary will hold the master/ lead data
lead_data = {}

# results data a dictionary {test_case_name : result_output}
test_cases_results = {}

logger = logging.getLogger('Install_health_tool_logger')

"""
All Helper Functions
"""
def compare_smu_lists(data_value1, data_value2, results, count):
    """This is used to comapre smus lists."""
    result = 0
    list1 = data_value1.data_value
    list2 = data_value2.data_value
    if not len(list1) == len(list2):
        result = 0
    else:
        myset = set()
        for x in list1:
            myset.add(os.path.splitext(x)[0])
        for y in list2:
            val = os.path.splitext(y)[0]
            if val not in myset:
                result = 0
                break
            else:
                myset.remove(val)

        if len(myset) > 0:
            result = 0
        else:
            result = 1

    results.insert(count, result_object(data_value1.vm_info, result))


def compare_two_lists(data_value1, data_value2, results, count):
    """This is used to comapre two lists."""
    list1 = data_value1.data_value
    list2 = data_value2.data_value
    result = 0
    if not len(list1) == len(list2):
        val = list(set(list1) ^ set(list2))
        result = 0
    else:
        val = list(set(list1) ^ set(list2))
        if len(val) > 1:
            result = 0
        else:
            result = 1

    results.insert(count, result_object(data_value1.vm_info,
                result,other_info=val))


def compare_swp_lists(data_value1, data_value2, results, count):
    """This is used to comapre two lists."""
    list1 = data_value1.data_value
    list2 = data_value2.data_value
    result = 0
    if not len(list1) == len(list2):
        result = 0
    else:
        dict1 = {}
        result = 1
        for val in list1:
            dict1[val.sdr_name] = val.version
        for val in list2:
            value = dict1.get(val.sdr_name)
            if not value == val.version:
                result = 0

    results.insert(count, result_object(data_value1.vm_info, 1))


def compare_two_items(data_value1, data_value2, results, count):
    """This is used to comapre two item."""
    result = 0
    if data_value1.data_value == data_value2.data_value:
        result = 1
    else:
        result = 0
    results.insert(count, result_object(data_value1.vm_info,
                result,other_info=data_value1.data_value))


def check_item_in_list(data_value1, data_value2, results, count):
    """ This is used to check if an item belongs to list or not """
    list = data_value1.data_value
    item = data_value2.data_value
    result = 0
    if item in list:
        result = 1
    results.insert(count, result_object(data_value1.vm_info, result,other_info=
                item))

def run_on_lead_cal(cmd, list2):
    """ This is used to run the command 'cmd' on the lead """
    ssh_cmd = ("%s ssh -q -T %s" % (platform_prefix, lead_rp_cal_vm.vm_ip))
    sshProcess = subprocess.Popen([ssh_cmd],
                                  stdin=subprocess.PIPE,
                                  stdout=subprocess.PIPE,
                                  stderr=subprocess.PIPE,
                                  shell=True)
    sshProcess.stdin.write("%s\n" % (cmd))
    out, error = sshProcess.communicate()
    if not sshProcess.returncode:
        out_list = list(filter(None, list(out.split('\n'))))
        list2.extend(out_list) 
    sshProcess.stdin.close()
 
def compare_arm_smu_lists(data_value1, cmd, results, count):
    """ This is used to compare SMUs on ARM cards with Master SWP """
    result = 0
    list2 = []
    list1 = data_value1.data_value
    run_on_lead_cal(cmd, list2)
    if not len(list1) == len(list2):
        val = list(set(list1) ^ set(list2))
        result = 0
    else:
        val = list(set(list1) ^ set(list2))
        if len(val) > 1: 
            result = 0
        else:
            result = 1

    results.insert(count, result_object(data_value1.vm_info,
                result,other_info=val)) 

def check_space_usage(data_value, results, count, checkType):
    """This is used to check adequate space for install directories."""
    if checkType == CAL:
        list1 = data_value.data_value
        info = []
        for x in list1:
            if int(
                    x.dir_usage) > CAL_INSTALL_DIR_FOR_SPACE_CHECK.get(
                    x.dir_name):
                info.append(x)

        if len(info) > 0:
            results.insert(count, result_object(data_value.vm_info, 0, info))
        else:
            results.insert(count, result_object(data_value.vm_info, 1))

    elif checkType == XR:
        list1 = data_value.data_value
        info = []
        for x in list1:
            if int(
                    x.dir_usage) > XR_INSTALL_DIR_FOR_SPACE_CHECK.get(
                    x.dir_name):
                info.append(x)

        if len(info) > 0:
            results.insert(count, result_object(data_value.vm_info, 0, info))
        else:
            results.insert(count, result_object(data_value.vm_info, 1))


def check_marker_files(data_value, results, count):
    """This is used to check existance of marker files."""
    list1 = data_value.data_value
    if len(list1) > 0:
        results.insert(
            count,
            result_object(
                data_value.vm_info,
                0,
                list(list1)))
    else:
        results.insert(count, result_object(data_value.vm_info, 1))


def check_mount_points(data_value, results, count, checkType):
    """This is used to verify Install Mount Points ."""
    if checkType == CAL:
        list1 = list(set(CAL_INSTALL_MOUNT_POINTS_CHECK) -
                     set(data_value.data_value))
        if len(list1) > 0:
            results.insert(
                count, result_object(
                    data_value.vm_info, 0, list(list1)))
        else:
            results.insert(count, result_object(data_value.vm_info, 1))

    elif checkType == XR:
        list1 = list(set(XR_INSTALL_MOUNT_POINTS_CHECK) -
                     set(data_value.data_value))
        if len(list1) > 0:
            results.insert(
                count, result_object(
                    data_value.vm_info, 0, list(list1)))
        else:
            results.insert(count, result_object(data_value.vm_info, 1))


def check_stale_symlinks(data_value, results, count):
    """This is used to check existance of stale symlinks."""
    list1 = data_value.data_value
    if len(list1) > 0:
        results.insert(
            count,
            result_object(
                data_value.vm_info,
                0,
                list(list1)))
    else:
        results.insert(count, result_object(data_value.vm_info, 1))


def check_system_image(data_value, results, count):
    """ this will verify existence of system image."""
    data = data_value.data_value
    if data:
        results.insert(count, result_object(data_value.vm_info, 1, data))
    else:
        results.insert(count, result_object(data_value.vm_info, 0, SYSTEM_IMAGE_FILE_CHECK))

def check_system_grub_file(data_value, results, count):
    """ this will verify existence of grub.cfg file."""
    data = data_value.data_value
    if data:
        results.insert(count, result_object(data_value.vm_info, 1, data))
    else:
        results.insert(count, result_object(data_value.vm_info, 0, SYSTEM_GRUB_FILE_CHECK))

def check_cal_tmp_staging_dir(data_value, results, count):
    """ this will verify the existance of file(s) in cal tmp_staging dir."""
    data = data_value.data_value
    if data:
        results.insert(count, result_object(data_value.vm_info, 0, data))
    else:
        results.insert(count, result_object(data_value.vm_info, 1))

def check_cal_install_tmp_staging_dir(data_value, results, count):
    """ this will verify the existance of file(s) in cal install_tmp_staging dir."""
    data = data_value.data_value
    if data:
        results.insert(count, result_object(data_value.vm_info, 1))
    else:
        results.insert(count, result_object(data_value.vm_info, 0, CAL_INSTALL_TMP_STAGING_DIR_CHECK))

def check_xr_tmp_staging_dir(data_value, results, count):
    """ this will verify existence of files(s) in xr tmp_staging dir."""
    data = data_value.data_value
    if data:
        results.insert(count, result_object(data_value.vm_info, 0, data))
    else:
        results.insert(count, result_object(data_value.vm_info, 1))

def check_system_grub_efi_file(data_value, results, count):
    """ this will verify existence of grub.efi file."""
    data = data_value.data_value
    if data:
        results.insert(count, result_object(data_value.vm_info, 1, data))
    else:
        results.insert(count, result_object(data_value.vm_info, 0, SYSTEM_GRUB_EFI_FILE_CHECK))

def verify_lv_states(data_value,results,count):
    """ this will verify the active and committed lv state is same """
    dict1 = {}
    for k,v in (data_value.data_value.items()):
        value = v.split()
        if value[0] != value[1]:
            dict1[k] =value
    if len(dict1) > 0:
        results.insert(
                count, result_object(
                    data_value.vm_info, 0, dict1))
    else:
        results.insert(count, result_object(data_value.vm_info, 1))

def verify_ld_path(data_value,results,count):
    """ this will verify the active and committed lv state is same """
        
    value = data_value.data_value
    if len(value) == 1 and value[0] == 'True':
        results.insert(count, result_object(data_value.vm_info, 1))
    else:
        results.insert(
                count, result_object(
                    data_value.vm_info, 0, value))

def verify_boot_lvs(lv_states_dict,data_value, results, count):
    """ this is will verify the active lvs are same as booted one"""
    list1 = [] 
    for k,v in data_value.data_value.items():
        if int(lv_states_dict[k].split()[0]) != int(v):
            list1.append([k,lv_states_dict[k].split()[0],v])
    if len(list1) > 0:
        results.insert(
            count,
            result_object(
                data_value.vm_info,
                0,
                list(list1)))
    else:
        results.insert(count, result_object(data_value.vm_info, 1))

def verify_grub_cfg(lv_states_dict,data_value,results, count):
    """ this is will verify the active lvs are same as booted one"""
    try:
        val = lv_states_dict["host"].split()
        val = int(val[0])
    except KeyError:
        logger.error("Unable to get the host lv state")
        exit()
    else:
        if val == int(data_value.data_value):
            results.insert(count, result_object(data_value.vm_info, 1))
        else:
            results.insert(count, result_object(data_value.vm_info, 0))

def check_prepared_packages(data_value, results, count):
    """This is used to check existance of prepare packages."""
    list1 = data_value.data_value
    if len(list1) > 0:
        results.insert(
            count,
            result_object(
                data_value.vm_info,
                0,
                list(list1)))
    else:
        results.insert(count, result_object(data_value.vm_info, 1))

def verify_rpmdb_list_empty(data_value, results, count):
    """this is used to check if the rpmdb list is empty or not """
    list1 = data_value.data_value
    if not len(list1) > 0:
        results.insert(
            count,
            result_object(
                data_value.vm_info,
                0,
                list(list1)))
    else:
        results.insert(count, result_object(data_value.vm_info, 1))
        

def check_for_var_log_space():
    """ this will check for the /var/log/ availability """
    cmd = get_commands_output('df -Ph /var/log | tail -1 | awk \'{print $5}\'')
    cmd = cmd.split("%")[0]
    if cmd:
        if int(cmd) > 95:
            print "/var/log usage is more than 95%. Pls free the space at /var/log"
            exit()

def check_cal_install_operation_in_progress():
    """ This is used to check if any cal install operation is in progress"""
    cmd = get_commands_output(
            '%s %s install request' %
            (platform_prefix, CAL_SHOW_CMD))
    if not cmd:
        logger.debug("Failed to show install request")
        exit_with_traceback()
    else:
        if not "No install operation in progress" in cmd:
            logger.error("Cal install operation is in progress.")
            logger.error("Please try after some time.")
            sys.exit("Exiting")

def check_xr_install_operation_in_progress():
    """ This is used to check if any xr install operation is in progress """
    global install_progress

    for vmm in rps_xr_vms:
        cmd = ("%s ssh %s %s install request" %(platform_prefix, vmm.vm_ip, XR_SHOW_CMD))
        output = get_commands_output(cmd)
        if output is None:
            continue

        if "User" in output:
            install_progress = True

        break
    
def get_platform_type():
    """This is used to detect the underlying platform type."""
    cmd = get_commands_output("/usr/bin/xr_sysctl -n kernel.boot.cmdline")
    if cmd:
        global platform_type
        p = re.compile("platform=(.*) ")
        val = p.findall(cmd)
        if val:
            val = val[0].split()[0]
        else:
            logger.error("Error: Unable to get the Platform Information.")
            exit()
        if val == 'panini':
            platform_type = ncs6k
        elif val == 'ncs5k':
            platform_type = ncs5k
        elif val == 'ncs560':
            platform_type = ncs560
        elif val == 'fretta':
            platform_type = ncs5500
        elif val == 'xrv9k':
            platform_type = xrv9k
        elif val == 'zermatt':
            platform_type = zermatt
        elif val == 'ncs1k':
            platform_type = ncs1k
        elif val == 'ncs1004':
            platform_type = ncs1004
        elif val == 'asr9k':
            platform_type = asr9k
        else:
            logger.error("Error: Not a Supported Platform.")
            exit()
    else:
        logger.error("Error: Unable to get the Platform Information.")
        exit()


def get_plugin_running_data():
    '''this is used to collect the test cases which needs to run '''
    with open('/opt/cisco/calvados/script/plugin_data.json') as data_file:
        data = json.load(data_file)

    for i in data["plugins"]:
        val = list(i.keys())
        val = (val[0].encode("ascii"))
        plugins_running_data[val] = i[val].encode("ascii")

    for test_case in cal_test_cases:
        if platform_type in plugins_running_data.get(test_case):
            test_cases_cal.append(test_case)

    for test_case in host_test_cases:
        if platform_type in plugins_running_data.get(test_case):
            test_cases_host.append(test_case)

    for test_case in xr_test_cases:
        if platform_type in plugins_running_data.get(test_case):
            test_cases_xr.append(test_case)

    for test_case in rp_cal_test_cases:
        if platform_type in plugins_running_data.get(test_case):
            rp_test_cases_cal.append(test_case)
    for test_case in rp_xr_test_cases:
        if platform_type in plugins_running_data.get(test_case):
            rp_test_cases_xr.append(test_case)


def generate_cal_space_check_cmd():
    ''' this will generate the cmd for Cal space check, by using the list
        CAL_INSTALL_DIR_FOR_SPACE_CHECK defined in ./const.py'''
    s = 'df -Ph '
    s1 = ' | tail -1 | awk \'{print $5 $6}\''
    my_new_list = []
    global CAL_INSTALL_DIR_SPACE_CMD
    
    try:
        with open('/opt/cisco/calvados/script/disk_usage.json') as f:
            data = json.load(f) 
                                                                                      
        try:
            for k,v in (data[platform_type]["cal"].items()):                                    
                        CAL_INSTALL_DIR_FOR_SPACE_CHECK[k.encode("ascii")] = int(v.encode("ascii"))
        except KeyError:
            pass
    except IOError as e: 
        logger.error("Unable to load file disk_usage")
        exit()
    for x in CAL_INSTALL_DIR_FOR_SPACE_CHECK:
        str1 = s + x + s1
        my_new_list.append(str1)

    CAL_INSTALL_DIR_SPACE_CMD = '\n'.join(map(str, my_new_list))
    rp_cal_commands['cal_disk_space'] = CAL_INSTALL_DIR_SPACE_CMD


def generate_xr_space_check_cmd():
    ''' this will generate the cmd for XR space check, by using the list
        XR_INSTALL_DIR_FOR_SPACE_CHECK defined in ./const.py'''
    s = 'df -Ph '
    s1 = ' | tail -1 | awk \'{print $5 $6}\''
    my_new_list = []
    global XR_INSTALL_DIR_SPACE_CMD
    try:
        with open('/opt/cisco/calvados/script/disk_usage.json') as f:
            data = json.load(f) 
        try:                                                                              
            for k,v in (data[platform_type]["xr"].items()):                                    
                        XR_INSTALL_DIR_FOR_SPACE_CHECK[k.encode("ascii")] = int(v.encode("ascii"))
        except KeyError:
            pass
    except IOError as e: 
        logger.error("Unable to load file disk_usage")
        exit()
    for x in XR_INSTALL_DIR_FOR_SPACE_CHECK:
        str1 = s + x + s1
        my_new_list.append(str1)

    XR_INSTALL_DIR_SPACE_CMD = '\n'.join(map(str, my_new_list))
    rp_xr_commands['xr_disk_space'] = XR_INSTALL_DIR_SPACE_CMD


def generate_cal_marker_files_cmd():
    ''' this will generate the cmd for Cal Marker Files check, by using the list
        CAL_INSTALL_MARKER_FILES_CHECK defined in ./const.py'''
    s = 'ls '
    my_new_list = []
    global CAL_INSTALL_MARKER_FILES

    for x in CAL_INSTALL_MARKER_FILES_CHECK:
        str1 = s + x
        my_new_list.append(str1)

    CAL_INSTALL_MARKER_FILES = '\n'.join(map(str, my_new_list))
    rp_cal_commands['cal_marker_files'] = CAL_INSTALL_MARKER_FILES


def generate_xr_marker_files_cmd():
    ''' this will generate the cmd for XR marker files check, by using the list
        XR_INSTALL_MARKER_FILES_CHECK defined in ./const.py'''
    s = 'ls '
    my_new_list = []
    global XR_INSTALL_MARKER_FILES

    for x in XR_INSTALL_MARKER_FILES_CHECK:
        str1 = s + x
        my_new_list.append(str1)

    XR_INSTALL_MARKER_FILES = '\n'.join(map(str, my_new_list))
    rp_xr_commands['xr_marker_files'] = XR_INSTALL_MARKER_FILES


def generate_cal_mount_points_cmd():
    ''' this will generate the cmd for Cal Mount Points check, by using the list
        CAL_INSTALL_MOUNT_POINTS_CHECK defined in ./const.py'''
    s = 'mount | grep '
    s1 = ' | awk \'{print $3}\''
    my_new_list = []
    global CAL_INSTALL_MOUNT_POINTS

    for x in CAL_INSTALL_MOUNT_POINTS_CHECK:
        str1 = s + x + s1
        my_new_list.append(str1)

    CAL_INSTALL_MOUNT_POINTS = '\n'.join(map(str, my_new_list))
    rp_cal_commands['cal_mount_points'] = CAL_INSTALL_MOUNT_POINTS


def generate_xr_mount_points_cmd():
    ''' this will generate the cmd for XR Mount Points check, by using the list
        XR_INSTALL_MOUNT_POINTS_CHECK defined in ./const.py'''
    s = 'mount | grep '
    s1 = ' | awk \'{print $3}\''
    my_new_list = []
    global XR_INSTALL_MOUNT_POINTS

    for x in XR_INSTALL_MOUNT_POINTS_CHECK:
        str1 = s + x + s1
        my_new_list.append(str1)

    XR_INSTALL_MOUNT_POINTS = '\n'.join(map(str, my_new_list))
    rp_xr_commands['xr_mount_points'] = XR_INSTALL_MOUNT_POINTS


def generate_cal_symlinks_cmd():
    ''' this will generate the cmd for Cal symlinks check, by using the list
        CAL_INSTALL_SYMLINKS_CHECK defined in ./const.py'''
    s = 'find -L '
    s1 = ' -type l'
    my_new_list = []
    global CAL_INSTALL_SYMLINKS

    for x in CAL_INSTALL_SYMLINKS_CHECK:
        str1 = s + x + s1
        my_new_list.append(str1)

    # adding the list of all sdr packages to be checked for stale symlinks
    for sdr in xr_vms_per_sdr:
        dir = '/install_repo/gl/instdb/sdr/' + sdr + '/pkg/'
        str1 = s + dir + s1
        my_new_list.append(str1)

    CAL_INSTALL_SYMLINKS = '\n'.join(map(str, my_new_list))
    rp_cal_commands['cal_stale_symlinks'] = CAL_INSTALL_SYMLINKS


def generate_xr_symlinks_cmd():
    ''' this will generate the cmd for XR symlinks check, by using the list
        XR_INSTALL_SYMLINKS_CHECK defined in ./const.py'''
    s = 'find -L '
    s1 = ' -type l'
    my_new_list = []
    global XR_INSTALL_SYMLINKS

    for x in XR_INSTALL_SYMLINKS_CHECK:
        str1 = s + x + s1
        my_new_list.append(str1)

    XR_INSTALL_SYMLINKS = '\n'.join(map(str, my_new_list))
    rp_xr_commands['xr_stale_symlinks'] = XR_INSTALL_SYMLINKS


def generate_cal_prepared_pkg_cmd():
    ''' this will generate the cmd for Cal Prepared pkg check '''

    global CAL_SHOW_PREPARED_CMD

    CAL_SHOW_PREPARED_CMD = CAL_SHOW_CMD + PREPARE_CMD
    rp_cal_commands['cal_prepared_packages'] = CAL_SHOW_PREPARED_CMD

def generate_cal_boot_lvs():
    ''' this will generate the cmd for boot lvs using lvscan at host'''
    global BOOT_LVS_CMD 

    BOOT_LVS_CMD = platform_prefix + " ssh "+ host_ip +" cat /misc/config/calvados_launch_path.txt"
    cal_data_commands['boot_lvs'] = BOOT_LVS_CMD


def generate_xr_prepared_pkg_cmd():
    ''' this will generate the cmd for XR Prepared pkg check '''

    global XR_SHOW_PREPARED_CMD

    XR_SHOW_PREPARED_CMD = XR_SHOW_CMD + PREPARE_CMD
    rp_xr_commands['xr_prepared_packages'] = XR_SHOW_PREPARED_CMD


def exit_with_traceback():
    '''this is used to exit with tb data'''
    logger.error("Exiting")
    exc_info = sys.exc_info()
    TB = traceback.format_exc()
    logger.debug(TB)
    os._exit(1)


def exit():
    '''this is used to just exit off the script'''
    print("Exiting")
    os._exit(1)


def get_commands_output(cmd):
    '''this is used to execute any command at with retry logic'''
    logger.debug("Executing Command: %s" % (cmd))
    attempts = 1
    while attempts <= 3:
        try:
            res = subprocess.Popen(
                [cmd],
                shell=True,
                stdout=subprocess.PIPE,
                stderr=subprocess.PIPE)
            output, error = res.communicate()
            if output:
                return output
            if error:
                logger.debug(error.strip())
        except OSError as e:
            logger.debug(e.strerror)
        except BaseException:
            logger.error(sys.exc_info()[0])
        logger.debug(
            "Attempt: %s, Failed to get the output of cmd :%s" %
            (attempts, cmd))
        time.sleep(0.005)
        attempts += 1
    return  # Return an empty string if failed to get command o/p after 3 retry


def initialize_nodes_data_set():
    ''' this is used to initialize the data containers for various nodes'''

    nodes_data['admin'] = {}
    nodes_data['host'] = {}

    for sdr in xr_vms_per_sdr:
        nodes_data[sdr] = {}
        for test_case in test_cases_xr:
            nodes_data[sdr][test_case] = []
        for test_case in rp_test_cases_xr:
            nodes_data[sdr][test_case] = []

    for test_case in test_cases_cal:
        nodes_data['admin'][test_case] = []

    for test_case in rp_test_cases_cal:
        nodes_data['admin'][test_case] = []

    for test_case in host_test_cases:
        nodes_data['host'][test_case] = []

    lead_data['admin'] = {}
    lead_data['host'] = {}

    for sdr in xr_vms_per_sdr:
        lead_data[sdr] = {}
        for test_case in test_cases_xr:
            lead_data[sdr][test_case] = []
        for test_case in rp_test_cases_xr:
            lead_data[sdr][test_case] = []

    for test_case in test_cases_cal:
        lead_data['admin'][test_case] = []

    for test_case in rp_test_cases_cal:
        lead_data['admin'][test_case] = []

    for test_case in test_cases_host:
        lead_data['host'][test_case] = []

    test_cases_results['admin'] = {}
    test_cases_results['host'] = {}

    for sdr in xr_vms_per_sdr:
        test_cases_results[sdr] = {}
        for test_case in test_cases_xr:
            test_cases_results[sdr][test_case] = []
        for test_case in rp_test_cases_xr:
            test_cases_results[sdr][test_case] = []

    for test_case in test_cases_cal:
        test_cases_results['admin'][test_case] = []

    for test_case in rp_test_cases_cal:
        test_cases_results['admin'][test_case] = []

    for test_case in test_cases_host:
        test_cases_results['host'][test_case] = []


def get_and_store_cal_result(test_case_name, vm, data, vm_count):
    '''this is used to store the result for the testcase'''
    ''' here you can specify what data you want to store for verification purpose'''
    if test_case_name == cal_test_cases[0]:
        if data:
            data = data[0].split("-")[4].split(".")[0]
            logger.debug(
                "%s at %s: %s" %
                (test_case_name, vm.vm_card_location, data))
            nodes_data.get('admin').get(test_case_name).insert(
                vm_count, data_object(vm, data))
        else:
            logger.debug(
                "Unable to capture %s at: %s" %
                (test_case_name, vm.vm_card_location))
            logger.error("rpmdb is not working at: %s for %s" %
                    (vm.vm_card_location,vm.vm_name))
            exit()

    elif test_case_name == cal_test_cases[1]:
        if data:
            logger.debug(
                "%s at %s: %s" %
                (test_case_name, vm.vm_card_location, data))
            nodes_data.get('admin').get(test_case_name).insert(
                vm_count, data_object(vm, list(data)))
        else:
            nodes_data.get('admin').get(test_case_name).insert(
                vm_count, data_object(vm, list([])))
            logger.debug(
                "Unable to capture %s at: %s" %
                (test_case_name, vm.vm_card_location))

    elif test_case_name == cal_test_cases[2]:
        if data:
            regex = re.compile(".*(Version).*")
            val = [m.group(0) for l in data for m in [regex.search(l)] if m]
            if val:
                val = val[0].split("Version: ")[1]
                nodes_data.get('admin').get(test_case_name).insert(
                    vm_count, data_object(vm, val))
                logger.debug(
                    "%s at %s: %s" %
                    (test_case_name, vm.vm_card_location, val))
            else:
                nodes_data.get('admin').get(test_case_name).insert(
                    vm_count, data_object(vm, ''))
        else:
            nodes_data.get('admin').get(test_case_name).insert(
                    vm_count, data_object(vm, ''))
            logger.debug(
                "Unable to capture %s at: %s" %
                (test_case_name, vm.vm_card_location))

    elif test_case_name == cal_test_cases[3]:
        if data:
            regex = re.compile(".*(Version).*")
            val = [m.group(0) for l in data for m in [regex.search(l)] if m]
            if val:
                val = val[0].split("Version: ")[1]
                nodes_data.get('admin').get(test_case_name).insert(
                    vm_count, data_object(vm, val))
                logger.debug(
                    "%s at %s: %s" %
                    (test_case_name, vm.vm_card_location, val))
            else:
                nodes_data.get('admin').get(test_case_name).insert(
                    vm_count, data_object(vm, ''))
        else:
            nodes_data.get('admin').get(test_case_name).insert(
                    vm_count, data_object(vm, ''))
            logger.debug(
                "Unable to capture %s at: %s" %
                (test_case_name, vm.vm_card_location))

    elif test_case_name == cal_test_cases[4]:
        if data:
                x = list(map((lambda x : x.split()[2]),filter((lambda x :
                                    "[Build" in x),data)))
                logger.debug(
                    "%s at %s: %s" %
                    (test_case_name, vm.vm_card_location, x))
                nodes_data.get('admin').get(test_case_name).insert(
                    vm_count, data_object(vm, list(x)))
        else:        
            nodes_data.get('admin').get(test_case_name).insert(
                vm_count, data_object(vm, []))
            logger.debug(
                "Unable to capture %s at: %s" %
                (test_case_name, vm.vm_card_location))

    elif test_case_name == cal_test_cases[5]:
        if data:
                x = list(map((lambda x : x.split()[2]),filter((lambda x :
                                    "[Build" in x),data)))
                logger.debug(
                    "%s at %s: %s" %
                    (test_case_name, vm.vm_card_location, x))
                nodes_data.get('admin').get(test_case_name).insert(
                    vm_count, data_object(vm, list(x)))
        else:        
            nodes_data.get('admin').get(test_case_name).insert(
                vm_count, data_object(vm, []))
            logger.debug(
                "Unable to capture %s at: %s" %
                (test_case_name, vm.vm_card_location))

    elif test_case_name == cal_test_cases[6]:
        if data:
            logger.debug(
                "%s at %s: %s" %
                (test_case_name, vm.vm_card_location, data))
            nodes_data.get('admin').get(test_case_name).insert(
                vm_count, data_object(vm, list(data)))
        else:
            nodes_data.get('admin').get(test_case_name).insert(
                vm_count, data_object(vm, []))
            logger.debug(
                "Unable to capture %s at: %s" %
                (test_case_name, vm.vm_card_location))

    elif test_case_name == cal_test_cases[7]:
       if data:
         i =0
         dict1 = {}
         try:
            while i < len(data):
                if data[i] == '':
                    i = i+1
                else:
                    key= data[i].split("/")[-1].split(".")[0].split("_")
                    key = "-".join(key[:len(key)-2])
                    if "hdd" in key:
                        i = i+2
                        continue
                    value = data[i+1] 
                    dict1[key] = value 
                    i = i+2
         except IndexError:
            logger.debug(
                "Unable to capture %s at: %s" %
                (test_case_name, vm.vm_card_location))
         else:
            logger.debug(
                    "%s at %s: %s" %
                    (test_case_name, vm.vm_card_location, dict1))
            nodes_data.get('admin').get(test_case_name).insert(
                        vm_count, data_object(vm, dict1))
       else:
            logger.debug(
                "Unable to capture %s at: %s" %
                (test_case_name, vm.vm_card_location))

    elif test_case_name == cal_test_cases[8]:
        if data:
            dict1 = {}
            for x in data:
                if "/" not in x or len(x.split("/")) < 2:
                    continue
                val = x.split("/")[3].split("'")[0]
                val = val.split("_")
                key = "-".join(val[:len(val)-1])
                if "host-data" in key:
                    continue
                value = re.search("\d+",val[-1])
                val = value.group(0) 
                if val:
                    value = val
                else:
                    value = -1 
                dict1[key] = value 
            logger.debug(
                    "%s at %s: %s" %
                    (test_case_name, vm.vm_card_location, dict1))
            nodes_data.get('admin').get(test_case_name).insert(
                        vm_count, data_object(vm, dict1))
        else:
            logger.debug(
                "Unable to capture %s at: %s" %
                (test_case_name, vm.vm_card_location))

    elif test_case_name == "cal_ld_path_verification":
        if data:
            logger.debug(
                    "%s at %s: %s" %
                    (test_case_name, vm.vm_card_location,data ))
            nodes_data.get('admin').get(test_case_name).insert(
                        vm_count, data_object(vm, data))
        else:
            logger.debug(
                "Unable to capture %s at: %s" %
                (test_case_name, vm.vm_card_location))

def get_and_store_cal_master_result(test_case_name, vm, data, vm_count):
    '''this is used to store the result for the testcase for the Lead Cal VM'''
    ''' here you can specify what data you want to store for verification purpose'''
    logger.debug("Storing data for Lead at: %s" % (vm.vm_card_location))
    if test_case_name == cal_test_cases[0]:
        if data:
            data = data[0].split("-")[4].split(".")[0]
            logger.debug(
                "%s at %s: %s" %
                (test_case_name, vm.vm_card_location, data))
            lead_data.get('admin').get(test_case_name).insert(
                vm_count, data_object(vm, data))
        else:
            logger.debug(
                "Unable to capture %s at: %s" %
                (test_case_name, vm.vm_card_location))

    elif test_case_name == cal_test_cases[1]:
        if data:
            logger.debug(
                "%s at %s: %s" %
                (test_case_name, vm.vm_card_location, data))
            lead_data.get('admin').get(test_case_name).insert(
                vm_count, data_object(vm, list(data)))
        else:
            lead_data.get('admin').get(test_case_name).insert(
                vm_count, data_object(vm, []))
            logger.debug(
                "Unable to capture %s at: %s" %
                (test_case_name, vm.vm_card_location))

    elif test_case_name == cal_test_cases[2]:
        if data:
            regex = re.compile(".*(Version).*")
            val = [m.group(0) for l in data for m in [regex.search(l)] if m]
            if val:
                val = val[0].split("Version: ")[1]
                lead_data.get('admin').get(test_case_name).insert(
                    vm_count, data_object(vm, val))
                logger.debug(
                    "%s at %s: %s" %
                    (test_case_name, vm.vm_card_location, val))
            else:
                lead_data.get('admin').get(test_case_name).insert(
                    vm_count, data_object(vm, ''))
        else:
            logger.debug(
                "Unable to capture %s at: %s" %
                (test_case_name, vm.vm_card_location))

    elif test_case_name == cal_test_cases[3]:
        if data:
            regex = re.compile(".*(Version).*")
            val = [m.group(0) for l in data for m in [regex.search(l)] if m]
            if val:
                val = val[0].split("Version: ")[1]
                lead_data.get('admin').get(test_case_name).insert(
                    vm_count, data_object(vm, val))
                logger.debug(
                    "%s at %s: %s" %
                    (test_case_name, vm.vm_card_location, val))
            else:
                lead_data.get('admin').get(test_case_name).insert(
                    vm_count, data_object(vm, ''))
        else:
            logger.debug(
                "Unable to capture %s at: %s" %
                (test_case_name, vm.vm_card_location))

    elif test_case_name == cal_test_cases[4]:
        if data:
                x = list(map((lambda x : x.split()[2]),filter((lambda x :
                                    "[Build" in x),data)))
                logger.debug(
                    "%s at %s: %s" %
                    (test_case_name, vm.vm_card_location, x))
                lead_data.get('admin').get(test_case_name).insert(
                    vm_count, data_object(vm, list(x)))
        else:        
            lead_data.get('admin').get(test_case_name).insert(
                vm_count, data_object(vm, []))
            logger.debug(
                "Unable to capture %s at: %s" %
                (test_case_name, vm.vm_card_location))

    elif test_case_name == cal_test_cases[5]:
        if data:
                x = list(map((lambda x : x.split()[2]),filter((lambda x :
                                    "[Build" in x),data)))
                logger.debug(
                    "%s at %s: %s" %
                    (test_case_name, vm.vm_card_location, x))
                lead_data.get('admin').get(test_case_name).insert(
                    vm_count, data_object(vm, list(x)))
        else:        
            lead_data.get('admin').get(test_case_name).insert(
                vm_count, data_object(vm, []))
            logger.debug(
                "Unable to capture %s at: %s" %
                (test_case_name, vm.vm_card_location))

def parse_cal_output(output, vm, vm_count, test_cases_list, isMaster):
    '''this parses the multi command output and call the function to
    store the output for each individual plugin'''
    config_found = False
    textfile = []
    test_case_count = 0
    for line in output:
        if line == "END\n":
            break
        if re.match("Next command Output", line) and config_found == False:
            config_found = True
        elif re.match("Next command Output", line) and config_found:
            if isMaster:
                get_and_store_cal_master_result(
                    test_cases_list[test_case_count], vm, textfile, vm_count)
            else:
                get_and_store_cal_result(
                    test_cases_list[test_case_count], vm, textfile, vm_count)
            test_case_count = test_case_count + 1
            del textfile[:]
        elif config_found:
            i = line.rstrip()
            textfile.append(i)

    if isMaster:
        get_and_store_cal_master_result(
            test_cases_list[test_case_count], vm, textfile, vm_count)
    else:
        get_and_store_cal_result(
            test_cases_list[test_case_count],
            vm,
            textfile,
            vm_count)


def get_cal_output(vm, vm_count, test_cases_list, isMaster):
    """This function will be used to get the output on a cal vm. """
    if vm:
        cmd = ("%s ssh -q -T %s" % (platform_prefix, vm.vm_ip))
        sshProcess = subprocess.Popen([cmd],
                                      stdin=subprocess.PIPE,
                                      stdout=subprocess.PIPE,
                                      stderr=subprocess.PIPE,
                                      shell=True)

        for test_case in test_cases_list:
            cmd = cal_data_commands.get(test_case)
            sshProcess.stdin.write("echo ""Next command Output"" \n")
            sshProcess.stdin.write("%s\n" % (cmd))
        sshProcess.stdin.close()
        
        parse_cal_output(
            sshProcess.stdout,
            vm,
            vm_count,
            test_cases_list,
            isMaster)


def run_on_all_cal():
    """This function will be used to run all the commands on the cal."""
    logger.info("Collecting sysadmin VMs data")
    if len(test_cases_cal) > 0:
        threads = [None] * len(cal_vms)
        i = 0
        for vm in cal_vms:
            threads[i] = Thread(
                target=get_cal_output, args=(
                    vm, i, test_cases_cal, False))
            threads[i].start()
            i = i + 1
        for t in threads:
            t.join()
    else:
        logger.debug("No Sysadmin Plugin to execute")


def get_and_store_rp_cal_result(test_case_name, vm, data, vm_count):
    '''this is used to store the result for the testcase'''
    ''' here you can specify what data you want to store for verification purpose'''

    if test_case_name == rp_cal_test_cases[0]:
        if data:
            logger.debug(
                "%s at %s: %s" %
                (test_case_name, vm.vm_card_location, data))
            dir_usages = []
            for val in data:
                matchObj = re.match(r'(.*)%(.*)', val, re.M | re.I)
                if matchObj:
                    dir_name = val.split('%')[1]
                    dir_usage = val.split('%')[0]
                    dir_usages.append(install_dir_object(dir_name, dir_usage))
            nodes_data.get('admin').get(test_case_name).insert(
                vm_count, data_object(vm, dir_usages))
        else:
            logger.debug(
                "Unable to capture %s at: %s" %
                (test_case_name, vm.vm_card_location))

    elif test_case_name == rp_cal_test_cases[1]:
        if data:
            logger.debug(
                "%s at %s: %s" %
                (test_case_name, vm.vm_card_location, data))
            marker_files = []
            for val in data:
                if 'No such file or directory' not in val:
                    marker_files.append(val)
            nodes_data.get('admin').get(test_case_name).insert(
                vm_count, data_object(vm, marker_files))
        else:
            logger.debug(
                "Unable to capture %s at: %s" %
                (test_case_name, vm.vm_card_location))

    elif test_case_name == rp_cal_test_cases[2]:
        if data:
            logger.debug(
                "%s at %s: %s" %
                (test_case_name, vm.vm_card_location, data))
            nodes_data.get('admin').get(test_case_name).insert(
                vm_count, data_object(vm, list(data)))
        else:
            logger.debug(
                "Unable to capture %s at: %s" %
                (test_case_name, vm.vm_card_location))

    elif test_case_name == rp_cal_test_cases[3]:
        if data:
            logger.debug(
                "%s at %s: %s" %
                (test_case_name, vm.vm_card_location, data))
            nodes_data.get('admin').get(test_case_name).insert(
                vm_count, data_object(vm, list(data)))
        else:
            logger.debug(
                "Unable to capture %s at: %s" %
                (test_case_name, vm.vm_card_location))

    elif test_case_name == rp_cal_test_cases[4]:
        if data:
            logger.debug(
                "%s at %s: %s" %
                (test_case_name, vm.vm_card_location, data))
            if len(data) > 1 and "No prepared" not in data[0]:
                nodes_data.get('admin').get(test_case_name).insert(
                    vm_count, data_object(vm, list(data)))
            else:
                nodes_data.get('admin').get(test_case_name).insert(
                    vm_count, data_object(vm, []))
        else:
            logger.debug(
                "Unable to capture %s at: %s" %
                (test_case_name, vm.vm_card_location))

    elif test_case_name == rp_cal_test_cases[5]:
        if data:
            regex = re.compile(".*(Version).*")
            val = [m.group(0) for l in data for m in [regex.search(l)] if m]
            if val:
                val = val[0].split("Version: ")[1]
                nodes_data.get('admin').get(test_case_name).insert(
                    vm_count, data_object(vm, val))
                logger.debug(
                    "%s at %s: %s" %
                    (test_case_name, vm.vm_card_location, val))
            else:
                nodes_data.get('admin').get(test_case_name).insert(
                    vm_count, data_object(vm, ''))
        else:
            logger.debug(
                "Unable to capture %s at: %s" %
                (test_case_name, vm.vm_card_location))

    elif test_case_name == rp_cal_test_cases[6]:
        if data:
            regex = re.compile(".*(Version).*")
            val = [m.group(0) for l in data for m in [regex.search(l)] if m]
            if val:
                val = val[0].split("Version: ")[1]
                nodes_data.get('admin').get(test_case_name).insert(
                    vm_count, data_object(vm, val))
                logger.debug(
                    "%s at %s: %s" %
                    (test_case_name, vm.vm_card_location, val))
            else:
                nodes_data.get('admin').get(test_case_name).insert(
                    vm_count, data_object(vm, ''))
        else:
            logger.debug(
                "Unable to capture %s at: %s" %
                (test_case_name, vm.vm_card_location))

    elif test_case_name == rp_cal_test_cases[7]:
        if data:
            data_value = []
            i = 0
            while i < len(data):
                sdr_key = data[i]
                i = i + 1
                version = data[i].split("Version: ")[1]
                i = i + 1
                data_value.append(swp_object(sdr_key, version))
                logger.debug(
                    "%s at %s: %s: %s" %
                    (test_case_name, vm.vm_card_location, sdr_key, version))
            if len(data_value) > 0:
                nodes_data.get('admin').get(test_case_name).insert(
                    vm_count, data_object(vm, data_value))
            else:
                nodes_data.get('admin').get(test_case_name).insert(
                    vm_count, data_object(vm, []))
        else:
            logger.debug(
                "Unable to capture %s at: %s" %
                (test_case_name, vm.vm_card_location))

    elif test_case_name == rp_cal_test_cases[8]:
        if data:
            data_value = []
            i = 0
            while i < len(data):
                sdr_key = data[i]
                i = i + 1
                version = data[i].split("Version: ")[1]
                i = i + 1
                data_value.append(swp_object(sdr_key, version))
                logger.debug(
                    "%s at %s: %s: %s" %
                    (test_case_name, vm.vm_card_location, sdr_key, version))
            if len(data_value) > 0:
                nodes_data.get('admin').get(test_case_name).insert(
                    vm_count, data_object(vm, data_value))
            else:
                nodes_data.get('admin').get(test_case_name).insert(
                    vm_count, data_object(vm, []))
        else:
            logger.debug(
                "Unable to capture %s at: %s" %
                (test_case_name, vm.vm_card_location))

    elif test_case_name == rp_cal_test_cases[9]:
        if data:
            data_value = []
            i = 0
            while i < len(data):
                sdr_key = data[i]
                i = i + 1
                version = data[i].split("Version: ")[1]
                i = i + 1
                data_value.append(swp_object(sdr_key, version))
                logger.debug(
                    "%s at %s: %s: %s" %
                    (test_case_name, vm.vm_card_location, sdr_key, version))
            if len(data_value) > 0:
                nodes_data.get('admin').get(test_case_name).insert(
                    vm_count, data_object(vm, data_value))
            else:
                nodes_data.get('admin').get(test_case_name).insert(
                    vm_count, data_object(vm, []))
        else:
            logger.debug(
                "Unable to capture %s at: %s" %
                (test_case_name, vm.vm_card_location))

    elif test_case_name == rp_cal_test_cases[10]:
        if data:
            data_value = []
            i = 0
            while i < len(data):
                sdr_key = data[i]
                i = i + 1
                version = data[i].split("Version: ")[1]
                i = i + 1
                data_value.append(swp_object(sdr_key, version))
                logger.debug(
                    "%s at %s: %s: %s" %
                    (test_case_name, vm.vm_card_location, sdr_key, version))
            if len(data_value) > 0:
                nodes_data.get('admin').get(test_case_name).insert(
                    vm_count, data_object(vm, data_value))
            else:
                nodes_data.get('admin').get(test_case_name).insert(
                    vm_count, data_object(vm, []))
        else:
            logger.debug(
                "Unable to capture %s at: %s" %
                (test_case_name, vm.vm_card_location))

    elif test_case_name == rp_cal_test_cases[11]:
        if data:
            data_value = []
            for value in data:
                val = value.split('/')[-1]
                if val:
                    matchObj = re.match(r'(.*)x-(.*)', val, re.M | re.I)
                    if matchObj:
                        val1 = val.split('x-')[-1]
                        val1 = 'r' + val1.replace('.', '')
                        val1 = val1.split('-')[0]
                        logger.debug(
                            "%s at %s: %s " %
                            (test_case_name, vm.vm_card_location, val1))
                        data_value.append(val1)
                    else:
                        matchObj = re.match(r'(.*)x64-(.*)', val, re.M | re.I)
                        if matchObj:
                            val1 = val.split('x64-')[-1]
                            val1 = 'r' + val1.replace('.', '')
                            val1 = val1.split('-')[0]
                            logger.debug(
                                "%s at %s: %s " %
                                (test_case_name, vm.vm_card_location, val1))
                            data_value.append(val1)
            nodes_data.get('admin').get(test_case_name).insert(
                vm_count, data_object(vm, data_value))
        else:
            logger.debug(
                "Unable to capture %s at: %s" %
                (test_case_name, vm.vm_card_location))
            nodes_data.get('admin').get(test_case_name).insert(
                vm_count, data_object(vm, []))

    elif test_case_name == rp_cal_test_cases[12]:
        if data:
            data_value = []
            for value in data:
                val = value.split('/')[-1]
                if val:
                    val1 = val.split('sysadmin-')[-1]
                    val1 = 'r' + val1.replace('.', '')
                    data_value.append(val1)
                    logger.debug(
                        "%s at %s: %s " %
                        (test_case_name, vm.vm_card_location, val1))
            nodes_data.get('admin').get(test_case_name).insert(
                vm_count, data_object(vm, data_value))
        else:
            logger.debug(
                "Unable to capture %s at: %s" %
                (test_case_name, vm.vm_card_location))
            nodes_data.get('admin').get(test_case_name).insert(
                vm_count, data_object(vm, []))

    elif test_case_name == rp_cal_test_cases[13]:
        if data:
            val = data[0].split('/')[-1]
            if val:
                matchObj = re.match(r'(.*)x-(.*)', val, re.M | re.I)
                if matchObj:
                    val1 = val.split('x-')[-1]
                    val1 = 'r' + val1.replace('.', '')
                    val1 = val1.split('-')[0]
                    logger.debug(
                        "%s at %s: %s " %
                        (test_case_name, vm.vm_card_location, val1))
                    nodes_data.get('admin').get(test_case_name).insert(
                        vm_count, data_object(vm, val1))
                else:
                    matchObj = re.match(r'(.*)x64-(.*)', val, re.M | re.I)
                    if matchObj:
                        val1 = val.split('x64-')[-1]
                        val1 = 'r' + val1.replace('.', '')
                        val1 = val1.split('-')[0]
                        logger.debug(
                            "%s at %s: %s " %
                            (test_case_name, vm.vm_card_location, val1))
                        nodes_data.get('admin').get(test_case_name).insert(
                            vm_count, data_object(vm, val1))
        else:
            logger.debug(
                "Unable to capture %s at: %s" %
                (test_case_name, vm.vm_card_location))

    elif test_case_name == rp_cal_test_cases[14]:
        if data:
            logger.debug(
                "%s at %s: %s" %
                (test_case_name, vm.vm_card_location, data))
            nodes_data.get('admin').get(test_case_name).insert(
                vm_count, data_object(vm, list(data)))
        else:
            logger.debug(
                "Unable to capture %s at: %s" %
                (test_case_name, vm.vm_card_location))

    elif test_case_name == rp_cal_test_cases[15]:
        if data:
            logger.debug(
                "%s at %s: %s" %
                (test_case_name, vm.vm_card_location, data))
            nodes_data.get('admin').get(test_case_name).insert(
                vm_count, data_object(vm, list(data)))
        else:               
            logger.debug(
                "%s at %s: %s" %
                (test_case_name, vm.vm_card_location, data))
            nodes_data.get('admin').get(test_case_name).insert(
                vm_count, data_object(vm, list(data)))

    elif test_case_name == rp_cal_test_cases[16]:
        if data:
            logger.debug(
                "%s at %s: %s" %
                (test_case_name, vm.vm_card_location, data))
            nodes_data.get('admin').get(test_case_name).insert(
                vm_count, data_object(vm, list(data)))
        else:
            logger.debug(
                "%s at %s: %s" %
                (test_case_name, vm.vm_card_location, data))
            nodes_data.get('admin').get(test_case_name).insert(
                vm_count, data_object(vm, list(data)))

    elif test_case_name == rp_cal_test_cases[17]:
        if data:
            val = data[0].startswith('Found')
            if val:
                logger.debug(
                    "%s at %s: %s" %
                    (test_case_name, vm.vm_card_location, data))
                nodes_data.get('admin').get(test_case_name).insert(
                   vm_count, data_object(vm, list(data)))
            else:
                logger.debug(
                    "Unable to capture %s at: %s" %
                    (test_case_name, vm.vm_card_location))
        else:
            logger.debug(
                "Unable to capture %s at: %s" %
                (test_case_name, vm.vm_card_location))

    elif test_case_name == rp_cal_test_cases[18]:
        if data:
            logger.debug(
                "%s at %s: %s" %
                (test_case_name, vm.vm_card_location, data))
            nodes_data.get('admin').get(test_case_name).insert(
                vm_count, data_object(vm, list(data)))
        else:
            logger.debug(
                "%s at %s: %s" %
                (test_case_name, vm.vm_card_location, data))
            nodes_data.get('admin').get(test_case_name).insert(
                vm_count, data_object(vm, list(data)))

    elif test_case_name == rp_cal_test_cases[19]:
        if data:
            val = data[0].startswith('Found')
            if val:
                logger.debug(
                    "%s at %s: %s" %
                    (test_case_name, vm.vm_card_location, data))
                nodes_data.get('admin').get(test_case_name).insert(
                   vm_count, data_object(vm, list(data)))
            else:
                logger.debug(
                    "%s at %s: %s" %
                    (test_case_name, vm.vm_card_location, data))
                nodes_data.get('admin').get(test_case_name).insert(
                    vm_count, data_object(vm, list(data)))
        else:
            logger.debug(
                "%s at %s: %s" %
                (test_case_name, vm.vm_card_location, data))
            nodes_data.get('admin').get(test_case_name).insert(
                vm_count, data_object(vm, list(data)))


def get_and_store_rp_cal_master_result(test_case_name, vm, data, vm_count):
    '''this is used to store the result for the testcase for the Lead Cal VM'''
    ''' here you can specify what data you want to store for verification purpose'''
    logger.debug("Storing data for Lead at: %s" % (vm.vm_card_location))

    if test_case_name == rp_cal_test_cases[5]:
        if data:
            regex = re.compile(".*(Version).*")
            val = [m.group(0) for l in data for m in [regex.search(l)] if m]
            if val:
                val = val[0].split("Version: ")[1]
                lead_data.get('admin').get(test_case_name).insert(
                    vm_count, data_object(vm, val))
                logger.debug(
                    "%s at %s: %s" %
                    (test_case_name, vm.vm_card_location, val))
            else:
                lead_data.get('admin').get(test_case_name).insert(
                    vm_count, data_object(vm, ''))
        else:
            logger.debug(
                "Unable to capture %s at: %s" %
                (test_case_name, vm.vm_card_location))

    elif test_case_name == rp_cal_test_cases[6]:
        if data:
            regex = re.compile(".*(Version).*")
            val = [m.group(0) for l in data for m in [regex.search(l)] if m]
            if val:
                val = val[0].split("Version: ")[1]
                lead_data.get('admin').get(test_case_name).insert(
                    vm_count, data_object(vm, val))
                logger.debug(
                    "%s at %s: %s" %
                    (test_case_name, vm.vm_card_location, val))
            else:
                lead_data.get('admin').get(test_case_name).insert(
                    vm_count, data_object(vm, ''))
        else:
            logger.debug(
                "Unable to capture %s at: %s" %
                (test_case_name, vm.vm_card_location))

    elif test_case_name == rp_cal_test_cases[7]:
        if data:
            data_value = []
            i = 0
            while i < len(data):
                sdr_key = data[i]
                i = i + 1
                version = data[i].split("Version: ")[1]
                i = i + 1
                data_value.append(swp_object(sdr_key, version))
                logger.debug(
                    "%s at %s: %s: %s" %
                    (test_case_name, vm.vm_card_location, sdr_key, version))
            if len(data_value) > 0:
                lead_data.get('admin').get(test_case_name).insert(
                    vm_count, data_object(vm, data_value))
            else:
                lead_data.get('admin').get(test_case_name).insert(
                    vm_count, data_object(vm, []))
        else:
            logger.debug(
                "Unable to capture %s at: %s" %
                (test_case_name, vm.vm_card_location))

    elif test_case_name == rp_cal_test_cases[8]:
        if data:
            data_value = []
            i = 0
            while i < len(data):
                sdr_key = data[i]
                i = i + 1
                version = data[i].split("Version: ")[1]
                i = i + 1
                data_value.append(swp_object(sdr_key, version))
                logger.debug(
                    "%s at %s: %s: %s" %
                    (test_case_name, vm.vm_card_location, sdr_key, version))
            if len(data_value) > 0:
                lead_data.get('admin').get(test_case_name).insert(
                    vm_count, data_object(vm, data_value))
            else:
                lead_data.get('admin').get(test_case_name).insert(
                    vm_count, data_object(vm, []))
        else:
            logger.debug(
                "Unable to capture %s at: %s" %
                (test_case_name, vm.vm_card_location))

    elif test_case_name == rp_cal_test_cases[9]:
        if data:
            data_value = []
            i = 0
            while i < len(data):
                sdr_key = data[i]
                i = i + 1
                version = data[i].split("Version: ")[1]
                i = i + 1
                data_value.append(swp_object(sdr_key, version))
                logger.debug(
                    "%s at %s: %s: %s" %
                    (test_case_name, vm.vm_card_location, sdr_key, version))
            if len(data_value) > 0:
                lead_data.get('admin').get(test_case_name).insert(
                    vm_count, data_object(vm, data_value))
            else:
                lead_data.get('admin').get(test_case_name).insert(
                    vm_count, data_object(vm, []))
        else:
            logger.debug(
                "Unable to capture %s at: %s" %
                (test_case_name, vm.vm_card_location))

    elif test_case_name == rp_cal_test_cases[10]:
        if data:
            data_value = []
            i = 0
            while i < len(data):
                sdr_key = data[i]
                i = i + 1
                version = data[i].split("Version: ")[1]
                i = i + 1
                data_value.append(swp_object(sdr_key, version))
                logger.debug(
                    "%s at %s: %s: %s" %
                    (test_case_name, vm.vm_card_location, sdr_key, version))
            if len(data_value) > 0:
                lead_data.get('admin').get(test_case_name).insert(
                    vm_count, data_object(vm, data_value))
            else:
                lead_data.get('admin').get(test_case_name).insert(
                    vm_count, data_object(vm, []))
        else:
            logger.debug(
                "Unable to capture %s at: %s" %
                (test_case_name, vm.vm_card_location))


def parse_rp_cal_output(output, vm, vm_count, test_cases_list, isMaster):
    '''this parses the multi command output and call the function to
    store the output for each individual plugin'''
    config_found = False
    textfile = []
    test_case_count = 0
    for line in output:
        if line == "END\n":
            break
        if re.match("Next command Output", line) and config_found == False:
            config_found = True
        elif re.match("Next command Output", line) and config_found:
            if isMaster:
                get_and_store_rp_cal_master_result(
                    test_cases_list[test_case_count], vm, textfile, vm_count)
            else:
                get_and_store_rp_cal_result(
                    test_cases_list[test_case_count], vm, textfile, vm_count)
            test_case_count = test_case_count + 1
            del textfile[:]
        elif config_found:
            i = line.rstrip()
            textfile.append(i)

    if isMaster:
        get_and_store_rp_cal_master_result(
            test_cases_list[test_case_count], vm, textfile, vm_count)
    else:
        get_and_store_rp_cal_result(
            test_cases_list[test_case_count],
            vm,
            textfile,
            vm_count)


def get_rp_cal_output(vm, vm_count, test_cases_list, isMaster):
    """This function will be used to get the output on a RP cal vm. """
    cmd = ("%s ssh -q -T %s" % (platform_prefix, vm.vm_ip))
    sshProcess = subprocess.Popen([cmd],
                                  stdin=subprocess.PIPE,
                                  stdout=subprocess.PIPE,
                                  stderr=subprocess.PIPE,
                                  shell=True)

    for test_case in test_cases_list:
        cmd = rp_cal_commands.get(test_case)
        sshProcess.stdin.write("echo ""Next command Output"" \n")
        sshProcess.stdin.write("%s\n" % (cmd))
    sshProcess.stdin.close()

    parse_rp_cal_output(
        sshProcess.stdout,
        vm,
        vm_count,
        test_cases_list,
        isMaster)


def generate_all_cal_commands_strings():
    """ this function is used to generate all the commands strings """
    generate_cal_space_check_cmd()
    generate_cal_marker_files_cmd()
    generate_cal_mount_points_cmd()
    generate_cal_symlinks_cmd()
    generate_cal_prepared_pkg_cmd()
    generate_cal_boot_lvs()


def generate_all_xr_commands_strings():
    """ this function is used to generate all the commands strings """
    generate_xr_space_check_cmd()
    generate_xr_marker_files_cmd()
    generate_xr_mount_points_cmd()
    generate_xr_symlinks_cmd()
    generate_xr_prepared_pkg_cmd()


def run_on_all_rps_cal():
    """This function will be used to run all the commands on the RPs cal."""
    if len(rp_test_cases_cal) > 0:
        threads = [None] * len(rps_cal_vms)
        i = 0
        for vm in rps_cal_vms:
            threads[i] = Thread(
                target=get_rp_cal_output, args=(
                    vm, i, rp_test_cases_cal, False))
            threads[i].start()
            i = i + 1
        for t in threads:
            t.join()
    else:
        logger.debug("No Sysadmin RP Plugin to execute")


def get_and_store_host_result(test_case_name, vm, data, vm_count):
    '''this is used to store the result for the testcase'''
    ''' here you can specify what data you want to store for verification purpose'''
    if test_case_name == host_test_cases[0]:
        if data:
            data = data[0].split("-")[4].split(".")[0]
            logger.debug(
                "%s at %s: %s" %
                (test_case_name, vm.vm_card_location, data))
            nodes_data.get('host').get(test_case_name).insert(
                vm_count, data_object(vm, data))
        else:
            logger.debug(
                "Unable to capture %s at: %s" %
                (test_case_name, vm.vm_card_location))
            logger.error("rpmdb is not working at: %s for %s" %
                    (vm.vm_card_location,vm.vm_name))
            exit()

    elif test_case_name == host_test_cases[1]:
        if data:
            logger.debug(
                "%s at %s: %s" %
                (test_case_name, vm.vm_card_location, data))
            nodes_data.get('host').get(test_case_name).insert(
                vm_count, data_object(vm, list(data)))
        else:
            logger.debug(
                "Unable to capture %s at: %s" %
                (test_case_name, vm.vm_card_location))

    elif test_case_name == host_test_cases[2]:
        if data:
            logger.debug(
                "%s at %s: %s" %
                (test_case_name, vm.vm_card_location, data))
            nodes_data.get('host').get(test_case_name).insert(
                vm_count, data_object(vm, list(data)))
        else:
            nodes_data.get('host').get(test_case_name).insert(
                vm_count, data_object(vm, []))
            logger.debug(
                "Unable to capture %s at: %s" %
                (test_case_name, vm.vm_card_location))

    elif test_case_name == host_test_cases[3]:
        if data:
            try :
                list2 = list(filter((lambda x:"root=" in x), data ))
                list3 = list(filter((lambda x:"panini_vol_grp" in x),list2[0].split()))
                val = list3[0].split("/")[-1].split("_")
                value = re.search("\d+",val[-1])
                val = value.group(0)                                
                if val:                                                       
                    value = val                                           
                else:                                                           
                    value = -1              
            except IndexError:
                logger.debug(
                    "Unable to capture %s at: %s" %
                        (test_case_name, vm.vm_card_location))
            else:
                logger.debug(
                        "%s at %s: %s" %
                        (test_case_name, vm.vm_card_location, value))
                nodes_data.get('host').get(test_case_name).insert(
                            vm_count, data_object(vm, value))
        else:
            logger.debug(
                    "Unable to capture %s at: %s" %
                        (test_case_name, vm.vm_card_location))


def get_and_store_host_master_result(test_case_name, vm, data, vm_count):
    '''this is used to store the result for the testcase for lead Host VM'''
    ''' here you can specify what data you want to store for verification purpose'''
    logger.debug("Storing data for Lead at: %s" % (vm.vm_card_location))
    if test_case_name == host_test_cases[0]:
        if data:
            data = data[0].split("-")[4].split(".")[0]
            logger.debug(
                "%s at %s: %s" %
                (test_case_name, vm.vm_card_location, data))
            lead_data.get('host').get(test_case_name).insert(
                vm_count, data_object(vm, data))
        else:
            logger.debug(
                "Unable to capture %s at: %s" %
                (test_case_name, vm.vm_card_location))

    elif test_case_name == host_test_cases[1]:
        if data:
            logger.debug(
                "%s at %s: %s" %
                (test_case_name, vm.vm_card_location, data))
            lead_data.get('host').get(test_case_name).insert(
                vm_count, data_object(vm, list(data)))
        else:
            lead_data.get('host').get(test_case_name).insert(
                vm_count, data_object(vm, []))
            logger.debug(
                "Unable to capture %s at: %s" %
                (test_case_name, vm.vm_card_location))


def parse_host_output(output, vm, vm_count, test_cases_list, isMaster):
    '''this parses the multi command output and call the function to
    store the output for each individual plugin'''
    config_found = False
    textfile = []
    test_case_count = 0
    for line in output:
        if line == "END\n":
            break
        if re.match("Next command Output", line) and config_found == False:
            config_found = True
        elif re.match("Next command Output", line) and config_found:
            if isMaster:
                get_and_store_host_master_result(
                    test_cases_list[test_case_count], vm, textfile, vm_count)
            else:
                get_and_store_host_result(
                    test_cases_list[test_case_count], vm, textfile, vm_count)
            test_case_count = test_case_count + 1
            del textfile[:]
        elif config_found:
            i = line.rstrip()
            textfile.append(i)
    if isMaster:
        get_and_store_host_master_result(
            test_cases_list[test_case_count], vm, textfile, vm_count)
    else:
        get_and_store_host_result(
            test_cases_list[test_case_count], vm, textfile, vm_count)


def get_host_output(vm, vm_count, test_cases_list, isMaster):
    """This function will be used to get the output on a cal vm for host data. """
    if vm:
        cmd = ("%s ssh -T -q %s %s ssh -T -q %s" %
               (platform_prefix, vm.vm_ip, platform_prefix, host_ip))
        sshProcess = subprocess.Popen([cmd],
                                      stdin=subprocess.PIPE,
                                      stdout=subprocess.PIPE,
                                      stderr=subprocess.PIPE,
                                      shell=True)

        for test_case in test_cases_list:
            cmd = host_data_commands.get(test_case)
            sshProcess.stdin.write("echo ""Next command Output"" \n")
            sshProcess.stdin.write("%s\n" % (cmd))
        sshProcess.stdin.close()
        
        parse_host_output(
            sshProcess.stdout,
            vm,
            vm_count,
            test_cases_list,
            isMaster)


def run_on_all_host():
    """This function will be used to run all the commands on all cal for host."""
    logger.info("Collecting Host data")
    if len(test_cases_host) > 0:
        threads = [None] * len(cal_vms)
        i = 0
        for vm in cal_vms:
            threads[i] = Thread(
                target=get_host_output, args=(
                    vm, i, test_cases_host, False))
            threads[i].start()
            i = i + 1
        for t in threads:
            t.join()
    else:
        logger.debug("No Host Plugin to execute")


def get_and_store_xr_result(test_case_name, vm, data, vm_count, sdr_name):
    '''this is used to store the result for the testcase for XR VM'''
    ''' here you can specify what data you want to store for verification purpose'''
    if test_case_name == xr_test_cases[0]:
        if data:
            data = data[0].split("-")[-1].split(".")[0]
            logger.debug(
                "%s at %s: %s" %
                (test_case_name, vm.vm_card_location, data))
            nodes_data.get(sdr_name).get(test_case_name).insert(
                vm_count, data_object(vm, data))
        else:
            logger.debug(
                "Unable to capture: %s at %s" %
                (test_case_name, vm.vm_card_location))
            logger.error("rpmdb is not working at: %s for %s" %
                    (vm.vm_card_location,vm.vm_name))
            exit()

    elif test_case_name == xr_test_cases[1]:
        if data:
            logger.debug(
                "%s at %s: %s" %
                (test_case_name, vm.vm_card_location, data))
            nodes_data.get(sdr_name).get(test_case_name).insert(
                vm_count, data_object(vm, list(data)))
        else:
            logger.debug(
                "Unable to capture: %s at %s" %
                (test_case_name, vm.vm_card_location))

    elif test_case_name == xr_test_cases[2]:
        if data:
            logger.debug(
                "%s at %s: %s" %
                (test_case_name, vm.vm_card_location, data))
            nodes_data.get(sdr_name).get(test_case_name).insert(
                vm_count, data_object(vm, list(data)))
        else:
            nodes_data.get(sdr_name).get(test_case_name).insert(
                vm_count, data_object(vm, []))
            logger.debug(
                "Unable to capture %s at: %s" %
                (test_case_name, vm.vm_card_location))

    elif test_case_name == "xr_ld_path_verification":
        if data:
            logger.debug(
                    "%s at %s: %s" %
                    (test_case_name, vm.vm_card_location,data ))
            nodes_data.get(sdr_name).get(test_case_name).insert(
                        vm_count, data_object(vm, data))
        else:
            logger.debug(
                "Unable to capture %s at: %s" %
                (test_case_name, vm.vm_card_location))

def get_and_store_xr_master_result(
        test_case_name,
        vm,
        data,
        vm_count,
        sdr_name):
    '''this is used to store the result for the testcase for lead XR VM per sdr'''
    ''' here you can specify what data you want to store for verification purpose'''
    logger.debug("Storing data for Lead at: %s" % (vm.vm_card_location))
    if test_case_name == xr_test_cases[0]:
        if data:
            data = data[0].split("-")[-1].split(".")[0]
            logger.debug(
                "%s at %s: %s" %
                (test_case_name, vm.vm_card_location, data))
            lead_data.get(sdr_name).get(test_case_name).insert(
                vm_count, data_object(vm, data))
        else:
            logger.debug(
                "Unable to capture: %s at %s" %
                (test_case_name, vm.vm_card_location))

    elif test_case_name == xr_test_cases[1]:
        if data:
            logger.debug(
                "%s at %s: %s" %
                (test_case_name, vm.vm_card_location, data))
            lead_data.get(sdr_name).get(test_case_name).insert(
                vm_count, data_object(vm, list(data)))
        else:
            lead_data.get(sdr_name).get(test_case_name).insert(
                vm_count, data_object(vm, []))
            logger.debug(
                "Unable to capture: %s at %s" %
                (test_case_name, vm.vm_card_location))


def parse_xr_output(output, vm, vm_count, sdr_name, test_cases_list, isMaster):
    '''this parses the multi command output and call the function to
    store the output for each individual plugin'''
    config_found = False
    textfile = []
    test_case_count = 0
    for line in output:
        if line == "END\n":
            break
        if re.match("Next command Output", line) and config_found == False:
            config_found = True
        elif re.match("Next command Output", line) and config_found:
            if isMaster:
                get_and_store_xr_master_result(
                    test_cases_list[test_case_count], vm, textfile, vm_count, sdr_name)
            else:
                get_and_store_xr_result(
                    test_cases_list[test_case_count],
                    vm,
                    textfile,
                    vm_count,
                    sdr_name)
            test_case_count = test_case_count + 1
            del textfile[:]
        elif config_found:
            i = line.rstrip()
            textfile.append(i)
    if isMaster:
        get_and_store_xr_master_result(
            test_cases_list[test_case_count],
            vm,
            textfile,
            vm_count,
            sdr_name)
    else:
        get_and_store_xr_result(
            test_cases_list[test_case_count],
            vm,
            textfile,
            vm_count,
            sdr_name)


def get_xr_output(vm, vm_count, sdr, test_cases_list, isMaster):
    """This function will be used to get the output on a XR vm. """
    if vm:
        cmd = ("%s ssh -T -q %s" % (platform_prefix, vm.vm_ip))
        sshProcess = subprocess.Popen([cmd],
                                      stdin=subprocess.PIPE,
                                      stdout=subprocess.PIPE,
                                      stderr=subprocess.PIPE,
                                      shell=True)

        for test_case in test_cases_list:
            cmd = xr_data_commands.get(test_case)
            sshProcess.stdin.write("echo ""Next command Output"" \n")
            sshProcess.stdin.write("%s\n" % (cmd))
        sshProcess.stdin.close()

        parse_xr_output(
            sshProcess.stdout,
            vm,
            vm_count,
            sdr,
            test_cases_list,
            isMaster)


def run_on_all_xr():
    """This function will be used to run all the commands on the XR VM."""
    logger.info("Collecting XR VMs data")
    if len(test_cases_xr) > 0:
        for sdr in xr_vms_per_sdr:
            threads = [None] * len(xr_vms_per_sdr[sdr])
            i = 0
            for v in xr_vms_per_sdr[sdr]:
                threads[i] = Thread(
                    target=get_xr_output, args=(
                        v, i, sdr, test_cases_xr, False))
                threads[i].start()
                i = i + 1
            for t in threads:
                t.join()
    else:
        logger.debug("No XR Plugin to execute")


def get_and_store_rp_xr_result(test_case_name, vm, data, vm_count, sdr_name):
    '''this is used to store the result for the testcase for XR VM'''
    ''' here you can specify what data you want to store for verification purpose'''
    if test_case_name == rp_xr_test_cases[0]:
        if data:
            logger.debug(
                "%s at %s: %s" %
                (test_case_name, vm.vm_card_location, data))
            dir_usages = []
            for val in data:
                matchObj = re.match(r'(.*)%(.*)', val, re.M | re.I)
                if matchObj:
                    dir_name = val.split('%')[1]
                    dir_usage = val.split('%')[0]
                    dir_usages.append(install_dir_object(dir_name, dir_usage))
            nodes_data.get(sdr_name).get(test_case_name).insert(
                vm_count, data_object(vm, dir_usages))
        else:
            logger.debug(
                "Unable to capture %s at: %s" %
                (test_case_name, vm.vm_card_location))

    elif test_case_name == rp_xr_test_cases[1]:
        if data:
            logger.debug(
                "%s at %s: %s" %
                (test_case_name, vm.vm_card_location, data))
            marker_files = []
            for val in data:
                if 'No such file or directory' not in val:
                    marker_files.append(val)
            nodes_data.get(sdr_name).get(test_case_name).insert(
                vm_count, data_object(vm, marker_files))
        else:
            logger.debug(
                "Unable to capture %s at: %s" %
                (test_case_name, vm.vm_card_location))

    elif test_case_name == rp_xr_test_cases[2]:
        if data:
            logger.debug(
                "%s at %s: %s" %
                (test_case_name, vm.vm_card_location, data))
            nodes_data.get(sdr_name).get(test_case_name).insert(
                vm_count, data_object(vm, list(data)))
        else:
            logger.debug(
                "Unable to capture %s at: %s" %
                (test_case_name, vm.vm_card_location))

    elif test_case_name == rp_xr_test_cases[3]:
        if data:
            logger.debug(
                "%s at %s: %s" %
                (test_case_name, vm.vm_card_location, data))
            nodes_data.get(sdr_name).get(test_case_name).insert(
                vm_count, data_object(vm, list(data)))
        else:
            logger.debug(
                "Unable to capture %s at: %s" %
                (test_case_name, vm.vm_card_location))

    elif test_case_name == rp_xr_test_cases[4]:
        if data:
            logger.debug(
                "%s at %s: %s" %
                (test_case_name, vm.vm_card_location, data))
            if len(data) > 1 and "No prepare" not in data[0]:
                nodes_data.get(sdr_name).get(test_case_name).insert(
                    vm_count, data_object(vm, list(data)))
            else:
                nodes_data.get(sdr_name).get(test_case_name).insert(
                    vm_count, data_object(vm, []))
        else:
            logger.debug(
                "Unable to capture %s at: %s" %
                (test_case_name, vm.vm_card_location))

    elif test_case_name == rp_xr_test_cases[5]:
        if data:
            val = data[0].startswith('Found')
            if val:
                logger.debug(
                    "%s at %s: %s" %
                    (test_case_name, vm.vm_card_location, data))
                nodes_data.get(sdr_name).get(test_case_name).insert(
                    vm_count, data_object(vm, list(data)))
            else:
                logger.debug(
                    "Unable to capture %s at: %s" %
                    (test_case_name, vm.vm_card_location))
        else:
            logger.debug(
                "Unable to capture %s at: %s" %
                (test_case_name, vm.vm_card_location))


def get_and_store_rp_xr_master_result(
        test_case_name,
        vm,
        data,
        vm_count,
        sdr_name):
    '''this is used to store the result for the testcase for lead XR VM per sdr'''
    ''' here you can specify what data you want to store for verification purpose'''
    logger.debug("Storing data for Lead at: %s" % (vm.vm_card_location))


def parse_rp_xr_output(
        output,
        vm,
        vm_count,
        sdr_name,
        test_cases_list,
        isMaster):
    '''this parses the multi command output and call the function to
    store the output for each individual plugin'''
    config_found = False
    textfile = []
    test_case_count = 0
    for line in output:
        if line == "END\n":
            break
        if re.match("Next command Output", line) and config_found == False:
            config_found = True
        elif re.match("Next command Output", line) and config_found:
            if isMaster:
                get_and_store_rp_xr_master_result(
                    test_cases_list[test_case_count], vm, textfile, vm_count, sdr_name)
            else:
                get_and_store_rp_xr_result(
                    test_cases_list[test_case_count],
                    vm,
                    textfile,
                    vm_count,
                    sdr_name)
            test_case_count = test_case_count + 1
            del textfile[:]
        elif config_found:
            i = line.rstrip()
            textfile.append(i)
    if isMaster:
        get_and_store_rp_xr_master_result(
            test_cases_list[test_case_count],
            vm,
            textfile,
            vm_count,
            sdr_name)
    else:
        get_and_store_rp_xr_result(
            test_cases_list[test_case_count],
            vm,
            textfile,
            vm_count,
            sdr_name)


def get_rp_xr_output(vm, vm_count, sdr, test_cases_list, isMaster):
    """This function will be used to get the output on a RP XR vm. """
    cmd = ("%s ssh -T -q %s" % (platform_prefix, vm.vm_ip))
    sshProcess = subprocess.Popen([cmd],
                                  stdin=subprocess.PIPE,
                                  stdout=subprocess.PIPE,
                                  stderr=subprocess.PIPE,
                                  shell=True)

    for test_case in test_cases_list:
        cmd = rp_xr_commands.get(test_case)
        sshProcess.stdin.write("echo ""Next command Output"" \n")
        sshProcess.stdin.write("%s\n" % (cmd))
    sshProcess.stdin.close()

    parse_rp_xr_output(
        sshProcess.stdout,
        vm,
        vm_count,
        sdr,
        test_cases_list,
        isMaster)


def run_on_all_rps_xr():
    """This function will be used to run all the commands on the XR VM."""
    if len(rp_test_cases_xr) > 0:
        threads = [None] * len(rps_xr_vms)
        i = 0
        for v in rps_xr_vms:
            threads[i] = Thread(
                target=get_rp_xr_output, args=(
                    v, i, v.vm_name, rp_test_cases_xr, False))
            threads[i].start()
            i = i + 1

        for t in threads:
            t.join()
    else:
        logger.debug("No XR RP plugin to execute")


def collect_rps_master_data_set():
    '''this is used to collect the data on Lead Cal/ XR VMs'''
    rp_collect_test_case_list = [
        'cal_master_active_swp',
        'cal_master_committed_swp',
        'xr_master_active_swp',
        'xr_master_committed_swp',
        'xr_local_active_swp',
        'xr_local_committed_swp',
    ]
    get_rp_cal_output(lead_rp_cal_vm, 0, rp_collect_test_case_list, True)


def collect_master_data_set():
    '''this is used to collect the data on Lead Cal/ XR VMs'''
    logger.info("Collecting Lead VMs data")
    get_cal_output(lead_rp_cal_vm, 0, test_cases_cal, True)
    get_host_output(lead_rp_cal_vm, 0, test_cases_host, True)
    for sdr in xr_vms_per_sdr:
        get_xr_output(lead_xr_vm_per_sdr.get(sdr), 0, sdr, test_cases_xr, True)


def verify_cal_data(test_case_data):
    '''this is used to verify the data per testcase'''
    for test_case in test_cases_cal:
        if test_case == cal_test_cases[0]:
            i = 0
            threads = [None] * len(test_case_data.get(test_case))
            for data_value in test_case_data.get(test_case):
                threads[i] = Thread(
                    target=compare_two_items,
                    args=(
                        data_value,
                        lead_data.get('admin').get(test_case)[0],
                        test_cases_results['admin'].get(test_case),
                        i,
                    ))
                threads[i].start()
                i = i + 1

            for t in threads:
                t.join()

        elif test_case == cal_test_cases[1]:
            i = 0
            threads = [None] * len(test_case_data.get(test_case))
            for data_value in test_case_data.get(test_case):
               if ((not re.match(re.compile('F./SC.'), data_value.vm_info.vm_card_location)) and 
                  ("FC" in data_value.vm_info.vm_card_location or "SC" in data_value.vm_info.vm_card_location)) :
                   threads[i] = Thread(
                       target=compare_arm_smu_lists,
                       args=(
                           data_value,
                           CAL_MASTER_ACTIVE_SWP_ARM_ADMIN,
                           test_cases_results['admin'].get(test_case),
                           i,  
                       ))  
                   threads[i].start()
                   i = i + 1 
               else : 
                   threads[i] = Thread(
                       target=compare_smu_lists,
                       args=(
                          data_value,
                          lead_data.get('admin').get(test_case)[0],
                          test_cases_results['admin'].get(test_case),
                          i,  
                       ))  
                   threads[i].start()
                   i = i + 1
            for t in threads:
                t.join()

        elif test_case == cal_test_cases[2]:
            i = 0
            threads = [None] * len(test_case_data.get(test_case))
            for data_value in test_case_data.get(test_case):
                threads[i] = Thread(
                    target=compare_two_items,
                    args=(
                        data_value,
                        lead_data.get('admin').get(test_case)[0],
                        test_cases_results['admin'].get(test_case),
                        i,
                    ))
                threads[i].start()
                i = i + 1

            for t in threads:
                t.join()

        elif test_case == cal_test_cases[3]:
            i = 0
            threads = [None] * len(test_case_data.get(test_case))
            for data_value in test_case_data.get(test_case):
                threads[i] = Thread(
                    target=compare_two_items,
                    args=(
                        data_value,
                        lead_data.get('admin').get(test_case)[0],
                        test_cases_results['admin'].get(test_case),
                        i,
                    ))
                threads[i].start()
                i = i + 1

            for t in threads:
                t.join()

        elif test_case == cal_test_cases[4]:
            i = 0
            threads = [None] * len(test_case_data.get(test_case))
            for data_value in test_case_data.get(test_case):
                threads[i] = Thread(
                    target=compare_two_lists,
                    args=(
                        data_value,
                        lead_data.get('admin').get(test_case)[0],
                        test_cases_results['admin'].get(test_case),
                        i,
                    ))
                threads[i].start()
                i = i + 1

            for t in threads:
                t.join()

        elif test_case == cal_test_cases[5]:
            i = 0
            threads = [None] * len(test_case_data.get(test_case))
            for data_value in test_case_data.get(test_case):
                threads[i] = Thread(
                    target=compare_two_lists,
                    args=(
                        data_value,
                        lead_data.get('admin').get(test_case)[0],
                        test_cases_results['admin'].get(test_case),
                        i,
                    ))
                threads[i].start()
                i = i + 1

            for t in threads:
                t.join()

        elif test_case == cal_test_cases[6]:
            i = 0
            threads = [None] * len(test_case_data.get(test_case))
            for data_value in test_case_data.get(test_case):
                threads[i] = Thread(
                    target=verify_rpmdb_list_empty,
                    args=(
                        data_value,
                        test_cases_results['admin'].get(test_case),
                        i,
                    ))
                threads[i].start()
                i = i + 1

            for t in threads:
                t.join()

        elif test_case == cal_test_cases[7]:
            i = 0
            threads = [None] * len(test_case_data.get(test_case))
            for data_value in test_case_data.get(test_case):
                threads[i] = Thread(
                    target=verify_lv_states,
                    args=(data_value,
                          test_cases_results['admin'].get(test_case),
                          i,
                          ))
                threads[i].start()
                i = i + 1

            for t in threads:
                t.join()

        elif test_case == cal_test_cases[8]:
            i = 0
            threads = [None] * len(test_case_data.get(test_case))
            for data_value in test_case_data.get(test_case):
                threads[i] = Thread(
                    target=verify_boot_lvs,
                    args=(test_case_data.get('lv_state_verification')[i].data_value,
                          data_value,
                          test_cases_results['admin'].get(test_case),
                          i,
                          ))
                threads[i].start()
                i = i + 1

            for t in threads:
                t.join()

        elif test_case == 'cal_ld_path_verification' :
            i = 0
            threads = [None] * len(test_case_data.get(test_case))
            for data_value in test_case_data.get(test_case):
                threads[i] = Thread(
                    target=verify_ld_path,
                    args=(data_value,
                          test_cases_results['admin'].get(test_case),
                          i,
                          ))
                threads[i].start()
                i = i + 1

            for t in threads:
                t.join()

    for test_case in rp_test_cases_cal:
        if test_case == rp_cal_test_cases[0]:
            i = 0
            threads = [None] * len(test_case_data.get(test_case))
            for data_value in test_case_data.get(test_case):
                threads[i] = Thread(
                    target=check_space_usage,
                    args=(data_value,
                          test_cases_results['admin'].get(test_case),
                          i, CAL,
                          ))
                threads[i].start()
                i = i + 1

            for t in threads:
                t.join()

        elif test_case == rp_cal_test_cases[1]:
            i = 0
            threads = [None] * len(test_case_data.get(test_case))
            for data_value in test_case_data.get(test_case):
                threads[i] = Thread(
                    target=check_marker_files,
                    args=(data_value,
                          test_cases_results['admin'].get(test_case),
                          i,
                          ))
                threads[i].start()
                i = i + 1

            for t in threads:
                t.join()

        elif test_case == rp_cal_test_cases[2]:
            i = 0
            threads = [None] * len(test_case_data.get(test_case))
            for data_value in test_case_data.get(test_case):
                threads[i] = Thread(
                    target=check_mount_points,
                    args=(data_value,
                          test_cases_results['admin'].get(test_case),
                          i, CAL,
                          ))
                threads[i].start()
                i = i + 1

            for t in threads:
                t.join()

        elif test_case == rp_cal_test_cases[3]:
            i = 0
            threads = [None] * len(test_case_data.get(test_case))
            for data_value in test_case_data.get(test_case):
                threads[i] = Thread(
                    target=check_stale_symlinks,
                    args=(data_value,
                          test_cases_results['admin'].get(test_case),
                          i,
                          ))
                threads[i].start()
                i = i + 1

            for t in threads:
                t.join()

        elif test_case == rp_cal_test_cases[4]:
            i = 0
            threads = [None] * len(test_case_data.get(test_case))
            for data_value in test_case_data.get(test_case):
                threads[i] = Thread(
                    target=check_prepared_packages,
                    args=(data_value,
                          test_cases_results['admin'].get(test_case),
                          i,
                          ))
                threads[i].start()
                i = i + 1

            for t in threads:
                t.join()

        elif test_case == rp_cal_test_cases[5]:
            i = 0
            threads = [None] * len(test_case_data.get(test_case))
            for data_value in test_case_data.get(test_case):
                threads[i] = Thread(
                    target=compare_two_items,
                    args=(
                        data_value,
                        lead_data.get('admin').get(test_case)[0],
                        test_cases_results['admin'].get(test_case),
                        i,
                    ))
                threads[i].start()
                i = i + 1

            for t in threads:
                t.join()

        elif test_case == rp_cal_test_cases[6]:
            i = 0
            threads = [None] * len(test_case_data.get(test_case))
            for data_value in test_case_data.get(test_case):
                threads[i] = Thread(
                    target=compare_two_items,
                    args=(
                        data_value,
                        lead_data.get('admin').get(test_case)[0],
                        test_cases_results['admin'].get(test_case),
                        i,
                    ))
                threads[i].start()
                i = i + 1

            for t in threads:
                t.join()

        elif test_case == rp_cal_test_cases[7]:
            i = 0
            threads = [None] * len(test_case_data.get(test_case))
            for data_value in test_case_data.get(test_case):
                threads[i] = Thread(
                    target=compare_swp_lists,
                    args=(
                        data_value,
                        lead_data.get('admin').get(test_case)[0],
                        test_cases_results['admin'].get(test_case),
                        i,
                    ))
                threads[i].start()
                i = i + 1

            for t in threads:
                t.join()

        elif test_case == rp_cal_test_cases[8]:
            i = 0
            threads = [None] * len(test_case_data.get(test_case))
            for data_value in test_case_data.get(test_case):
                threads[i] = Thread(
                    target=compare_swp_lists,
                    args=(
                        data_value,
                        lead_data.get('admin').get(test_case)[0],
                        test_cases_results['admin'].get(test_case),
                        i,
                    ))
                threads[i].start()
                i = i + 1

            for t in threads:
                t.join()

        elif test_case == rp_cal_test_cases[9]:
            i = 0
            threads = [None] * len(test_case_data.get(test_case))
            for data_value in test_case_data.get(test_case):
                threads[i] = Thread(
                    target=compare_swp_lists,
                    args=(
                        data_value,
                        lead_data.get('admin').get('xr_master_active_swp')[0],
                        test_cases_results['admin'].get(test_case),
                        i,
                    ))
                threads[i].start()
                i = i + 1

            for t in threads:
                t.join()

        elif test_case == rp_cal_test_cases[10]:
            i = 0
            threads = [None] * len(test_case_data.get(test_case))
            for data_value in test_case_data.get(test_case):
                threads[i] = Thread(
                    target=compare_swp_lists,
                    args=(
                        data_value,
                        lead_data.get('admin').get('xr_master_committed_swp')[0],
                        test_cases_results['admin'].get(test_case),
                        i,
                    ))
                threads[i].start()
                i = i + 1

            for t in threads:
                t.join()

        elif test_case == rp_cal_test_cases[11]:
            i = 0
            threads = [None] * len(test_case_data.get(test_case))
            for data_value in test_case_data.get(test_case):
                threads[i] = Thread(
                    target=check_item_in_list,
                    args=(
                        data_value,
                        nodes_data.get('admin').get('cal_version')[0],
                        test_cases_results['admin'].get(test_case),
                        i,
                    ))
                threads[i].start()
                i = i + 1
            for t in threads:
                t.join()

        elif test_case == rp_cal_test_cases[12]:
            i = 0
            threads = [None] * len(test_case_data.get(test_case))
            for data_value in test_case_data.get(test_case):
                threads[i] = Thread(
                    target=check_item_in_list,
                    args=(
                        data_value,
                        nodes_data.get('admin').get('cal_version')[0],
                        test_cases_results['admin'].get(test_case),
                        i,
                    ))
                threads[i].start()
                i = i + 1

            for t in threads:
                t.join()

        elif test_case == rp_cal_test_cases[13]:
            i = 0
            threads = [None] * len(test_case_data.get(test_case))
            for data_value in test_case_data.get(test_case):
                threads[i] = Thread(
                    target=compare_two_items,
                    args=(
                        data_value,
                        nodes_data.get('admin').get('cal_version')[0],
                        test_cases_results['admin'].get(test_case),
                        i,
                    ))
                threads[i].start()
                i = i + 1

            for t in threads:
                t.join()

        elif test_case == rp_cal_test_cases[14]:
            i = 0
            threads = [None] * len(test_case_data.get(test_case))
            for data_value in test_case_data.get(test_case):
                threads[i] = Thread(
                    target=check_stale_symlinks,
                    args=(data_value,
                          test_cases_results['admin'].get(test_case),
                          i,
                          ))
                threads[i].start()
                i = i + 1

            for t in threads:
                t.join()

        elif test_case == rp_cal_test_cases[15]:
            i = 0
            threads = [None] * len(test_case_data.get(test_case))
            for data_value in test_case_data.get(test_case):
                threads[i] = Thread(
                    target=check_system_image,
                    args=(data_value,
                          test_cases_results['admin'].get(test_case),
                          i,
                          ))
                threads[i].start()
                i = i + 1

            for t in threads:
                t.join()

        elif test_case == rp_cal_test_cases[16]:
            i = 0
            threads = [None] * len(test_case_data.get(test_case))
            for data_value in test_case_data.get(test_case):
                threads[i] = Thread(
                    target=check_system_grub_file,
                    args=(data_value,
                          test_cases_results['admin'].get(test_case),
                          i,
                          ))
                threads[i].start()
                i = i + 1

            for t in threads:
                t.join()

        elif test_case == rp_cal_test_cases[17]:
            i = 0
            threads = [None] * len(test_case_data.get(test_case))
            for data_value in test_case_data.get(test_case):
                threads[i] = Thread(
                    target=check_cal_tmp_staging_dir,
                    args=(data_value,
                          test_cases_results['admin'].get(test_case),
                          i,
                          ))
                threads[i].start()
                i = i + 1

            for t in threads:
                t.join()

        elif test_case == rp_cal_test_cases[18]:
            i = 0
            threads = [None] * len(test_case_data.get(test_case))
            for data_value in test_case_data.get(test_case):
                threads[i] = Thread(
                    target=check_system_grub_efi_file,
                    args=(data_value,
                          test_cases_results['admin'].get(test_case),
                          i,
                          ))
                threads[i].start()
                i = i + 1

            for t in threads:
                t.join()

        elif test_case == rp_cal_test_cases[19]:
            i = 0
            threads = [None] * len(test_case_data.get(test_case))
            for data_value in test_case_data.get(test_case):
                threads[i] = Thread(
                    target=check_cal_install_tmp_staging_dir,
                    args=(data_value,
                          test_cases_results['admin'].get(test_case),
                          i,
                          ))
                threads[i].start()
                i = i + 1

            for t in threads:
                t.join()


def verify_host_data(test_case_data):
    '''this is used to verify the data per testcase'''
    for test_case in test_cases_host:
        if test_case == host_test_cases[0]:
            i = 0
            threads = [None] * len(test_case_data.get(test_case))
            for data_value in test_case_data.get(test_case):
                threads[i] = Thread(
                    target=compare_two_items,
                    args=(
                        data_value,
                        lead_data.get('host').get(test_case)[0],
                        test_cases_results['host'].get(test_case),
                        i,
                    ))
                threads[i].start()
                i = i + 1

            for t in threads:
                t.join()

        elif test_case == host_test_cases[1]:
            i = 0
            threads = [None] * len(test_case_data.get(test_case))
            for data_value in test_case_data.get(test_case):
               if ((not re.match(re.compile('F./SC.'), data_value.vm_info.vm_card_location)) and  
                  ("FC" in data_value.vm_info.vm_card_location or "SC" in data_value.vm_info.vm_card_location)) :
                   threads[i] = Thread(
                       target=compare_arm_smu_lists,
                       args=(
                           data_value,
                           CAL_MASTER_ACTIVE_SWP_ARM_HOST,
                           test_cases_results['host'].get(test_case),
                           i,   
                       ))   
                   threads[i].start()
                   i = i + 1  
               else : 
                   threads[i] = Thread(
                       target=compare_smu_lists,
                       args=(
                          data_value,
                          lead_data.get('host').get(test_case)[0],
                          test_cases_results['host'].get(test_case),
                          i,   
                       ))   
                   threads[i].start()
                   i = i + 1
            for t in threads:
                t.join()

        elif test_case == host_test_cases[2]:
            i = 0
            threads = [None] * len(test_case_data.get(test_case))
            for data_value in test_case_data.get(test_case):
                threads[i] = Thread(
                    target=verify_rpmdb_list_empty,
                    args=(
                        data_value,
                        test_cases_results['host'].get(test_case),
                        i,
                    ))
                threads[i].start()
                i = i + 1

            for t in threads:
                t.join()

        elif test_case == host_test_cases[3]:
            i = 0
            threads = [None] * len(test_case_data.get(test_case))
            for data_value in test_case_data.get(test_case):
                threads[i] = Thread(
                    target=verify_grub_cfg,
                    args=(nodes_data.get('admin').get('lv_state_verification')[i].data_value,
                        data_value,
                        test_cases_results['host'].get(test_case),
                        i,
                    ))
                threads[i].start()
                i = i + 1

            for t in threads:
                t.join()

def verify_xr_data(test_case_data, sdr):
    '''this is used to verify the data per testcase'''
    for test_case in test_cases_xr:
        if test_case == xr_test_cases[0]:
            i = 0
            threads = [None] * len(test_case_data.get(test_case))
            for data_value in test_case_data.get(test_case):
                threads[i] = Thread(
                    target=compare_two_items,
                    args=(
                        data_value,
                        lead_data.get(sdr).get(test_case)[0],
                        test_cases_results[sdr].get(test_case),
                        i,
                    ))
                threads[i].start()
                i = i + 1

            for t in threads:
                t.join()

        elif test_case == xr_test_cases[1]:
            i = 0
            threads = [None] * len(test_case_data.get(test_case))
            for data_value in test_case_data.get(test_case):
                threads[i] = Thread(
                    target=compare_smu_lists,
                    args=(
                        data_value,
                        lead_data.get(sdr).get(test_case)[0],
                        test_cases_results[sdr].get(test_case),
                        i,
                    ))
                threads[i].start()
                i = i + 1

            for t in threads:
                t.join()

        elif test_case == xr_test_cases[2]:
            i = 0
            threads = [None] * len(test_case_data.get(test_case))
            for data_value in test_case_data.get(test_case):
                threads[i] = Thread(
                    target=verify_rpmdb_list_empty,
                    args=(
                        data_value,
                        test_cases_results[sdr].get(test_case),
                        i,
                    ))
                threads[i].start()
                i = i + 1

            for t in threads:
                t.join()

    for test_case in rp_test_cases_xr:
        if test_case == rp_xr_test_cases[0]:
            i = 0
            threads = [None] * len(test_case_data.get(test_case))
            for data_value in test_case_data.get(test_case):
                threads[i] = Thread(
                    target=check_space_usage,
                    args=(data_value,
                          test_cases_results[sdr].get(test_case),
                          i, XR,
                          ))
                threads[i].start()
                i = i + 1

            for t in threads:
                t.join()

        elif test_case == rp_xr_test_cases[1]:
            i = 0
            threads = [None] * len(test_case_data.get(test_case))
            for data_value in test_case_data.get(test_case):
                threads[i] = Thread(
                    target=check_marker_files,
                    args=(data_value,
                          test_cases_results[sdr].get(test_case),
                          i,
                          ))
                threads[i].start()
                i = i + 1

            for t in threads:
                t.join()

        elif test_case == rp_xr_test_cases[2]:
            i = 0
            threads = [None] * len(test_case_data.get(test_case))
            for data_value in test_case_data.get(test_case):
                threads[i] = Thread(
                    target=check_mount_points,
                    args=(data_value,
                          test_cases_results[sdr].get(test_case),
                          i, XR,
                          ))
                threads[i].start()
                i = i + 1

            for t in threads:
                t.join()

        elif test_case == rp_xr_test_cases[3]:
            i = 0
            threads = [None] * len(test_case_data.get(test_case))
            for data_value in test_case_data.get(test_case):
                threads[i] = Thread(
                    target=check_stale_symlinks,
                    args=(data_value,
                          test_cases_results[sdr].get(test_case),
                          i,
                          ))
                threads[i].start()
                i = i + 1

            for t in threads:
                t.join()

        elif test_case == rp_xr_test_cases[4]:
            i = 0
            threads = [None] * len(test_case_data.get(test_case))
            for data_value in test_case_data.get(test_case):
                threads[i] = Thread(
                    target = check_prepared_packages,
                    args=(data_value,
                          test_cases_results[sdr].get(test_case),
                          i,
                          ))
                threads[i].start()
                i = i + 1

            for t in threads:
                t.join()

        elif test_case == rp_xr_test_cases[5]:
            i = 0
            threads = [None] * len(test_case_data.get(test_case))
            for data_value in test_case_data.get(test_case):
                threads[i] = Thread(
                    target = check_xr_tmp_staging_dir,
                    args=(data_value,
                          test_cases_results[sdr].get(test_case),
                          i,
                          ))
                threads[i].start()
                i = i + 1

            for t in threads:
                t.join()

def verify_test_case_data():
    '''this is used to verify data for all the test_cases across all nodes'''
    logger.info("Verifying Test Plugins")
    threads = [None] * len(lead_data)
    i = 0
    for node in nodes_data:
        if node == 'admin':
            threads[i] = Thread(
                target=verify_cal_data, args=(
                    nodes_data['admin'],))
            threads[i].start()
        elif node == 'host':
            threads[i] = Thread(
                target=verify_host_data, args=(
                    nodes_data['host'],))
            threads[i].start()
        else:
            threads[i] = Thread(
                target=verify_xr_data, args=(
                    nodes_data[node], node,))
            threads[i].start()
        i = i + 1

    for t in threads:
        t.join()


def verify_cal_host_same_version():
    '''this is used to verify the cal and host have the same version'''
    result = []
    compare_two_items(lead_data.get('admin').get('cal_version')[0],
                      lead_data.get('host').get('host_version')[0], result, 0)
    if not result[0].result_value == 1:
        errors.append(
            error_object(
                'cal_host_same_version',
                lead_data.get('admin').get('cal_version')[0].vm_info,
                []))


def verify_cal_master_active_local_active_swp_same_version():
    '''this is used to verify the cal master active and local active have the same version'''
    result = []
    compare_two_items(lead_data.get('admin').get('cal_master_active_swp')[0],
                      lead_data.get('admin').get('cal_local_active_swp')[0], result, 0)
    if not result[0].result_value == 1:
        errors.append(
            error_object(
                'cal_master_active_local_active',
                lead_data.get('admin').get('cal_master_active_swp')[0].vm_info,
                []))


def verify_cal_master_committed_local_committed_swp_same_version():
    '''this is used to verify the cal master committed and local committed have the same version'''
    result = []
    compare_two_items(lead_data.get('admin').get('cal_master_committed_swp')[0],
                      lead_data.get('admin').get('cal_local_committed_swp')[0], result, 0)
    if not result[0].result_value == 1:
        errors.append(
            error_object(
                'cal_master_committed_local_committed',
                lead_data.get('admin').get(
                    'cal_master_committed_swp')[0].vm_info,
                []))


def verify_xr_master_active_local_active_swp_same_version():
    '''this is used to verify the xr master active and local active have the same version'''
    result = []
    if lead_data.get('admin').get('xr_master_active_swp') and lead_data.get(
            'admin').get('xr_local_active_swp'):
        compare_swp_lists(lead_data.get('admin').get('xr_master_active_swp')[0],
                          lead_data.get('admin').get('xr_local_active_swp')[0], result, 0)
        if not result[0].result_value == 1:
            errors.append(
                error_object(
                    'xr_master_active_local_active',
                    lead_data.get('admin').get(
                        'xr_master_active_swp')[0].vm_info,
                    []))


def verify_xr_master_committed_local_committed_swp_same_version():
    '''this is used to verify the XR master committed and local committed have the same version'''
    result = []
    if lead_data.get('admin').get('xr_master_committed_swp') and lead_data.get(
            'admin').get('xr_local_committed_swp'):
        compare_swp_lists(lead_data.get('admin').get('xr_master_committed_swp')[0],
                          lead_data.get('admin').get('xr_local_committed_swp')[0], result, 0)
        if not result[0].result_value == 1:
            errors.append(
                error_object(
                    'xr_master_committed_local_committed',
                    lead_data.get('admin').get(
                        'xr_master_committed_swp')[0].vm_info,
                    []))


def check_cal_data(results):
    '''this is to check if all the test_cases passed.'''
    for test_case in test_cases_cal:
        logger.info("Verifying Result for:%s" % (test_case))
        for result in results.get(test_case):
            if test_case == 'boot_lvs' and  ( "FC" or "SC" in result.vm_info.vm_card_location ):
                continue
            if test_case == 'cal_active_swp_pkg' and ( "FC" or "SC" in result.vm_info.vm_card_location ):
                continue
            if test_case == 'xr_active_swp_pkg' and ( "FC" or "SC" in result.vm_info.vm_card_location ):
                continue
            if not result.result_value == 1:
                errors.append(
                    error_object(
                        test_case,
                        result.vm_info,
                        result.other_info))

    for test_case in rp_test_cases_cal:
        logger.info("Verifying Result for:%s" % (test_case))
        for result in results.get(test_case):
            if not result.result_value == 1:
                errors.append(
                    error_object(
                        test_case,
                        result.vm_info,
                        result.other_info))

    verify_cal_host_same_version()
    verify_cal_master_active_local_active_swp_same_version()
    verify_cal_master_committed_local_committed_swp_same_version()
    verify_xr_master_active_local_active_swp_same_version()
    verify_xr_master_committed_local_committed_swp_same_version()


def check_host_data(results):
    '''this is to check if all the test_cases passed.'''
    for test_case in test_cases_host:
        logger.info("Verifying Result for:%s" % (test_case))
        for result in results.get(test_case):
            if not result.result_value == 1:
                errors.append(
                    error_object(
                        test_case,
                        result.vm_info,
                        result.other_info))


def check_xr_data(results, sdr):
    '''this is to check if all the test_cases passed.'''
    for test_case in test_cases_xr:
        logger.info("Verifying Result for:%s" % (test_case))
        for result in results.get(test_case):
            if not result.result_value == 1:
                errors.append(
                    error_object(
                        test_case,
                        result.vm_info,
                        result.other_info))

    for test_case in rp_test_cases_xr:
        logger.info("Verifying Result for:%s" % (test_case))
        for result in results.get(test_case):
            if not result.result_value == 1:
                errors.append(
                    error_object(
                        test_case,
                        result.vm_info,
                        result.other_info))


def check_test_case_data():
    '''this is to check if all the test_cases passed'''
    logger.info("Verifying Plugins results")
    threads = [None] * len(lead_data)
    i = 0
    for node in nodes_data:
        if node == 'admin':
            threads[i] = Thread(
                target=check_cal_data, args=(
                    test_cases_results['admin'],))
            threads[i].start()
        elif node == 'host':
            threads[i] = Thread(
                target=check_host_data, args=(
                    test_cases_results['host'],))
            threads[i].start()
        else:
            threads[i] = Thread(
                target=check_xr_data, args=(
                    test_cases_results[node], node,))
            threads[i].start()
        i = i + 1

    for t in threads:
        t.join()


def print_all_error_messages():
    '''this will print all the error messages for plugins'''
    if len(errors) > 0:
        for val in errors:
            logger.info('*' * 80)
            error_handlers[val.test_case_name](val.vm_info, val.other_info)
        logger.info('*' * 80)
    else:
        logger.info('*' * 70)
        logger.info(
            "System is in Consistent State. You can go ahead with next operation.")
        logger.info('*' * 70)


"""
All Global Classes
"""
"""
class Name: data_object
Defines the object for data holding
"""


class data_object():

    def __init__(self, vm_info, data_value):
        self.vm_info = vm_info
        self.data_value = data_value


"""
class Name: result_object
Defines the object for result holding
"""


class result_object():
    def __init__(self, vm_info, result_value, other_info=None):
        self.vm_info = vm_info
        self.result_value = result_value
        self.other_info = other_info


"""
class Name: swp_object
Defines the object for holding swp object
"""


class swp_object():
    def __init__(self, sdr_name, version):
        self.sdr_name = sdr_name
        self.version = version

    def __hash__(self):
        return hash((self.sdr_name, self.version))

    def __eq__(self, other):
        try:
            return (
                self.sdr_name,
                self.version) == (
                other.sdr_name,
                other.version)
        except AttributeError:
            return NotImplemented


"""
class Name: error_object
Defines the object for error holding
"""


class error_object():
    def __init__(self, test_case_name, vm_info, other_info):
        self.test_case_name = test_case_name
        self.vm_info = vm_info
        self.other_info = other_info


"""
class Name: install_dir_object
Defines the object for result holding
"""


class install_dir_object():
    '''this defines the directory and its usage'''

    def __init__(self, dir_name, dir_usage):
        self.dir_name = dir_name
        self.dir_usage = dir_usage


"""
Class Name: vm_info
Defines properties for a purticular VM
Attributes:
i) vm_ip : String (192.1.1.1)
ii) vm_name : String (sysadmin/ default-sdr/ named-sdr)
iii) vm_card_location: String (0/RP0, 0/0, 0/SC)
iv) vm_type : CAL VM: sysadmin, XR VM- xr
v) sdr_name: sdr_name in case of XR VM
"""


class vm_info(object):

    def __init__(
            self,
            vm_ip,
            vm_name,
            vm_card_location,
            vm_type,
            sdr_vm_name="None"):
        """Return a vm_info object."""
        self.vm_ip = vm_ip
        self.vm_name = vm_name
        self.vm_card_location = vm_card_location
        self.vm_type = vm_type
        self.sdr_vm_name = sdr_vm_name


"""
This class defines the strcuture for the Card
It says what is the card location and how many XR VM it holds.
(Multiple in case of MSDR)
Note: Calvados VM count is always One per card.
"""


class card_info(object):

    def __init__(
            self,
            card_location,
            card_type,
            num_xr_vms):
        """Return a card_info object"""
        self.card_location = card_location
        self.card_type = card_type
        self.num_cal_vms = 1
        self.num_xr_vms = num_xr_vms


"""
Class Name: collect_card_info
This is used to collect the info. about all Cards
and update the number of XR VMs per card
"""


class collect_card_info():

    def __init__(self):
        self.collect_card_data()
        self.collect_num_xr_vms_per_card()

    def collect_card_data(self):
        """This function collects the card info. which are up and running"""
        """If any of the card is not UP fully, it raises error"""
        logger.info("Collecting Cards Information")
        cmd = get_commands_output(
            '%s %s platform' %
            (platform_prefix, CAL_SHOW_CMD))
        if not cmd:
            logger.error("Failed to fetch Card Information.")
            exit_with_traceback()
        else:
            for line in cmd.split(os.linesep):
                words = [x for x in line.split()]
                if len(words) > 1:
                    location = words[0]
                    hw_state = words[2]
                    sw_state = words[3]
                    matchObj = re.match(r'(.*)RP(.*)', words[0], re.M | re.I)
                    if matchObj:
                        if hw_state == OPERATIONAL and not sw_state == OPERATIONAL:
                            logger.error(
                                "Error: %s Card is not UP." %
                                (words[0]))
                            exit()
                        elif hw_state == OPERATIONAL and sw_state == OPERATIONAL:
                            cards_info.append(
                                card_info(words[0], CARDTYPE_RP, 0))
                    matchObj = re.match(r'(.*)FC(.*)', words[0], re.M | re.I)
                    if matchObj:
                        if hw_state == OPERATIONAL and not (
                                sw_state == NA or sw_state == OPERATIONAL):
                            logger.error(
                                "Error: %s Card is not UP." %
                                (words[0]))
                            exit()
                        elif hw_state == OPERATIONAL and sw_state == OPERATIONAL:
                            cards_info.append(
                                card_info(words[0], CARDTYPE_FABRIC, 0))
                    matchObj = re.match(r'(.*)SC(.*)', words[0], re.M | re.I)
                    if matchObj:
                        if hw_state == OPERATIONAL and not ( 
                                sw_state == OPERATIONAL or sw_state == NA):
                            logger.error(
                                "Error: %s Card is not UP." %
                                (words[0]))
                            exit()
                        elif hw_state == OPERATIONAL and sw_state == OPERATIONAL:
                            cards_info.append(
                                card_info(words[0], CARDTYPE_SC, 0))
                    matchObj = re.match(
                        r'[0-9]/[0-9](.*)', words[0], re.M | re.I)
                    if matchObj:
                        if hw_state == OPERATIONAL and not (
                                sw_state == NA or sw_state == OPERATIONAL):
                            logger.error(
                                "Error %s Card is not UP." %
                                (words[0]))
                            exit()
                        elif hw_state == OPERATIONAL and sw_state == OPERATIONAL:
                            cards_info.append(
                                card_info(words[0], CARDTYPE_LC, 0))

    def add_count_xr_vms(self, card_location):
        """Update the number of XR VMs for a card"""
        cards_without_xr = ["FC", "SC"]
        if card_location == "all":
            for card in cards_info:
                if not any([x in card.card_location for x in cards_without_xr]):
                    card.num_xr_vms = 1 
        else: 
            for card in cards_info:
                if card.card_location == card_location:
                    card.num_xr_vms = card.num_xr_vms + 1
                    break


    def collect_num_xr_vms_per_card(self):
        """ This is used to collect the number of XR VM Per Card"""
        """ If any of the VM is not in Running State, It raises an error. """
        cmd = get_commands_output(
            '%s %s running-config sdr location' %
            (platform_prefix, CAL_SHOW_CMD))
        if not cmd:
            logger.error("Failed to fetch XR VM Information")
            exit_with_traceback()
        else:
            for line in cmd.split(os.linesep):
                o = re.match("location (.*)",line.strip())
                if o:
                   self.add_count_xr_vms(o.group(1))

"""
Class Name: collect_vm_info
This is used to collect info. about all CAL/ XR
VMs and as well as store data
"""


class collect_vms_info():

    def __init__(self):
        self.collect_sysadmin_vm_info()
        self.collect_xr_vm_info()

    def decrement_vm_count(self, card_location, vm_type):
        """This is used to decrement the VM count for a given Card."""
        for card in cards_info:
            if card.card_location == card_location:
                if vm_type == CAL:
                    card.num_cal_vms = card.num_cal_vms - 1
                    break
                if vm_type == XR:
                    card.num_xr_vms = card.num_xr_vms - 1
                    break

    def collect_sysadmin_vm_info(self):
        """This function is used to collect the Cals VM Info."""
        logger.info("Collecting Sysadmin VMs Information")
        lead_rp_cal_vm_found = 0
        cmd = get_commands_output('%s %s VM' % (platform_prefix, CAL_SHOW_CMD))
        if not cmd:
            logger.error("Failed to fetch VM Information")
            exit_with_traceback()
        else:
            for line in cmd.split(os.linesep):
                if "Location" in line:
                    location = line.split()[1]
                if "running" in line:
                    vm_name = line.split()[0]
                    if "sysadmin" in vm_name:
                        vm_type = "sysadmin"
                        vm_ip = re.findall(r'[0-9]+(?:\.[0-9]+){3}', line)
                        vm = vm_info(vm_ip[0], vm_name, location, vm_type)
                        cal_vms.append(vm)
                        if ("RP" in location or "RSP" in location or "CB" in location) and lead_rp_cal_vm_found == 0:
                            cmd_op = get_commands_output(
                                '%s ssh %s %s' %
                                (platform_prefix, vm_ip[0], CAL_INSTMGR_GREP))
                            if cmd_op:
                                for line1 in cmd_op.split(os.linesep):
                                    if "/opt/cisco/calvados/bin/inst_mgr" in line1:
                                        lead_rp_cal_vm_found = 1
                                        global lead_rp_cal_vm
                                        lead_rp_cal_vm = vm
                        if "RP" in location or "RSP" in location or "CB" in location:
                            rps_cal_vms.append(vm)
                        self.decrement_vm_count(location, CAL)

    def collect_xr_vm_info(self):
        """This function is used to collect the XR VM Info."""
        logger.info("Collecting XR VMs Information")
        cmd = get_commands_output(
            '%s %s sdr' %
            (platform_prefix, CAL_SHOW_CMD))
        if not cmd:
            logger.error("Failed to fetch XR VM Information")
            exit_with_traceback()
        else:
            for line in cmd.split(os.linesep):
                if "SDR" in line:
                    sdr_name = line.split(": ")[1]
                if "RUNNING" in line:
                    location = line.split()[0].rsplit("/", 1)[0]
                    sdr_vm_name = line.split()[0].rsplit("/", 1)[1]
                    vm_type = "xr"
                    vm_ip = re.findall(r'[0-9]+(?:\.[0-9]+){3}', line)
                    vm = vm_info(vm_ip[0], sdr_name, location, vm_type,
                                 sdr_vm_name)
                    xr_vms.append(vm)
                    self.decrement_vm_count(location, XR)
                    if "RP" in location or "RSP" in location or "CB" in location:
                        rps_xr_vms.append(vm)
        self.create_xr_vm_data_per_sdr()

    def get_sdr_instmgr_xrvm_for_sdr(self, vm):
        cmd_op = get_commands_output(
            '%s ssh %s /pkg/bin/placed_show -p sdr_instmgr' %
            (platform_prefix, vm.vm_ip))
        if cmd_op:
            for line1 in cmd_op.split(os.linesep):
                if "sdr_instmgr" in line1:
                    location = line1.split()[3].rsplit("/", 1)[0]
                    if location == vm.vm_card_location:
                        lead_xr_vm_per_sdr[vm.vm_name] = vm
        else: 
              logger.error("System is not fully UP. Please try after some time")
              exit()

    def create_xr_vm_data_per_sdr(self):
        """This function is used to capture dictionary
            {sdr_name:[vm_info1,vm_info2,..]} for XR VMs """
        logger.debug("Capturing XR VM Data per sdr")
        if len(xr_vms) > 0:
            for vm in xr_vms:
                if vm.vm_name not in xr_vms_per_sdr:
                    xr_vms_per_sdr[vm.vm_name] = [vm]
                    if "RP" in vm.vm_card_location or "CB" in vm.vm_card_location or "RSP" in vm.vm_card_location:
                        if vm.vm_name not in lead_xr_vm_per_sdr:
                            self.get_sdr_instmgr_xrvm_for_sdr(vm)

                else:
                    xr_vms_per_sdr[vm.vm_name].append(vm)
                    if "RP" in vm.vm_card_location or "CB" in vm.vm_card_location or "RSP" in vm.vm_card_location:
                        if vm.vm_name not in lead_xr_vm_per_sdr:
                            self.get_sdr_instmgr_xrvm_for_sdr(vm)
        else:
            logger.debug("No XR VMs Info. available")


"""
Class Name: verify_system_state
Defines all the testcases for verifying System is UP and running
"""


class verify_system_state():
    """Note: testcase1: The Card State is already verified while collecting
        the card info. """

    def __init__(self):
        self.verify_vm_running_count()

    def verify_vm_running_count(self):
        logger.info("Verifying all the required VMs are running.")
        for card in cards_info:
            if card.num_cal_vms > 0:
                logger.error(
                    "sysadmin VM is not running at: %s" %
                    (card.card_location))
                exit()
            if card.num_xr_vms > 0:
                logger.error(
                    "One of XR VM is not running at: %s" %
                    (card.card_location))
                exit()
        logger.info("Pass: All required VMs are Running")

        ##################################################################
        #----------------------------------------------------------------#
        #                            MAIN FUNCTION                       #
        #----------------------------------------------------------------#
        ##################################################################


if __name__ == "__main__":

    check_for_var_log_space()
    # create logger
    logger.setLevel(logging.DEBUG)
    formatter = logging.Formatter(
        '%(levelname)s %(message)s')

    formatter1 = logging.Formatter(
        '%(asctime)s:: %(levelname)s %(message)s',
        "%Y-%m-%d %H:%M:%S")

    # Console message
    ch = logging.StreamHandler(sys.stdout)
    ch.setFormatter(formatter)
    ch.setLevel(logging.INFO)
    logger.addHandler(ch)

    # Logs to logfile
    fh = logging.handlers.RotatingFileHandler(LOGFILE, maxBytes=(1024 * 1000),
                                              backupCount=5)
    fh.setLevel(logging.DEBUG)
    fh.setFormatter(formatter1)
    logger.addHandler(fh)

    # Check if log exists and should therefore be rolled
    needRoll = os.path.isfile(LOGFILE)
    if needRoll:
        logger.handlers[1].doRollover()

    start = time.time()
    get_platform_type()
    logger.info ("Platform is: %s " % (platform_type))
    platform_prefix = platform_prefixes.get(platform_type)
    host_ip = hostip.get(platform_type)
    get_plugin_running_data()
    collect_card_info()
    collect_vms_info()
    if not lead_rp_cal_vm:
        logger.error ("System is not fully UP. Please try after some time.!")
        exit()
    else:
        set_const_lead_rp(lead_rp_cal_vm)
    if lead_xr_vm_per_sdr.keys() == 0:
        logger.error ("System is not fully UP. Please try after some time.!")
        exit()
    check_cal_install_operation_in_progress()
    check_xr_install_operation_in_progress()
    if install_progress == True:
        logger.error("XR install operation in progress.")
        logger.error("Please try after some time.")
        sys.exit("Exiting")

    initialize_nodes_data_set()
    verify_system_state()
    generate_all_cal_commands_strings()
    generate_all_xr_commands_strings()
    run_on_all_cal()
    run_on_all_host()
    run_on_all_xr()
    run_on_all_rps_cal()
    run_on_all_rps_xr()
    collect_master_data_set()
    collect_rps_master_data_set()
    verify_test_case_data()
    check_test_case_data()
    print_all_error_messages()
    logger.info("Total time taken: %s seconds." % (time.time() - start))
