#! /bin/bash
#
# Copyright (c) 2012-2015 by Cisco Systems, Inc.
# All Rights Reserved.
#------------------------------------------------------------------
#
# This script interprets user input from the 
#
# show install log
#
# command. Higher-level "user parsing code",
# call it "the parser" for discussion purposes,
# allows two optional 'user arguments' (user-args), eg.
#
# show install log X Y
#
# Two such user-args are modified by the parser into three
# 'script arguments' (script-args) which are passed in here as
#
# --ids X Y
#
# the intention of which was likely to allow a range of
# id's, or possible a list. The parser currently only allows
# two user-args (not more), so it would be a list of only two entries.
# The implementing code (below) insists that the second
# user-arg must be 'verbose' or 'detail', so such a range or
# list functionality is not implemented.
#
# Note that in all cases, 'verbose' or 'detail' mean the same
# thing, and the output is (currently) *always* verbose,
# ie. it is always the same: we simply dump the entire log
# file for indicated id's. In effect, 'verbose' or 'detail'
# has no effect on the output. This is because we check for
# a leading "I" character in a given line for the "verbose"
# case - and all lines in the log files are currently prepended
# with "I", ie. each line is tagged as 'verbose'.
#
# If no user-args are given, all log id's are displayed.
#

if [ ! -d /install_repo/gl/instdb/log ]; then
    echo "No install operation performed"
    exit 0
fi

# remove this temp file in case it is lying around
# (since the following code does exclusively 'append' ops)
rm temp_cut.log 2>/dev/null

if [ $2 -ne 0 -o $2 -eq 0 2>/dev/null ] && [ "$1" = --ids ] ; then 
    #
    # user has entered more than one 'id' value
    # The first value ($2) must be a number, checked by the above
    # (slightly odd) not-equal zero or equal-zero tests.
    # The second value ($3) must be "detail" or "verbose"
    #
    if [ -f /install_repo/gl/instdb/log/install_operation_$2.log ]; then
        while read line1
            do
              first_char=${line1:0:1}
                if [ "${3}" = --detail ] ; then
                   echo ${line1#?} >> temp_cut.log
                elif [ -z "${3}" ] ; then
                 if [ "$first_char" = I ] ; then
                    echo ${line1#?} >> temp_cut.log
                fi

                else
                  echo "syntax error: unknown element"
                  exit 0
               fi
            done < /install_repo/gl/instdb/log/install_operation_$2.log
        # dump this output to user's screen
        cat temp_cut.log
        rm temp_cut.log
   else 
        echo "No log file found for install op $2"
        exit 0
    fi


elif [ -z "${1}" ] || [ "${1}" = --detail ] ; then
    #
    # This the case of:
    # - no user-args (-z "${1}")  OR
    # - a singler user-arg of 'detail' or 'verbose'
    #
    # where in each case, we dump out all available logs.
    # [because, as mentioned, all lines currently begin with "I"]
    #
    # The list of files is in lexical order, as obtained from 'ls -l', so we
    # send it to 'sort' for human-friendly ordering. -t_ sets the column
    # delimiter, and -k3 tells 'sort' to order via the third "column". We use
    # the rigid filename format to our advantage. eg. this raw list
    # 
    # install_operation_1.log
    # install_operation_10.log
    # install_operation_100.log
    # install_operation_2.log
    # install_operation_3.log
    # install_operation_33.log
    # install_operation_4.log
    # install_operation_50.log
    #
    # gets sorted to
    #
    # install_operation_1.log
    # install_operation_2.log
    # install_operation_3.log
    # install_operation_4.log
    # install_operation_10.log
    # install_operation_33.log
    # install_operation_50.log
    # install_operation_100.log
    #
    ls -l --format single-column /install_repo/gl/instdb/log | sort -t_ -k3n > temp_final.txt
    chmod 777 temp_final.txt
    #
    # temp_final.txt now contains a list of files from /install_repo/gl/instdb/log
    # These files should be exclusively install log files of the format
    # install_operation_X.log, where X is a number; ie. there should be
    # no other "junk" files lying around, which is normally the case.
    # The list of files has been sorted into ascending numerical order,
    # for convenient (and acceptable) display.

    # 'line' iterates the list of log filenames in temp_final.txt,
    # 'line1' iterates the text (log) lines within each filename.
    while read line
        do
            if [[ -n "$line" ]] ; then
                while read line1
                    do 
                      first_char=${line1:0:1}
                        if [ "${1}" = --detail ] ; then
                                echo ${line1#?} >> temp_cut.log

                        else
                            if [ "$first_char" = I ] ; then
                                echo ${line1#?} >> temp_cut.log
                            fi
                        fi
                    done < /install_repo/gl/instdb/log/$line
                if [ -f temp_cut.log ]; then
                    # dump this output to user's screen
                    cat temp_cut.log
                    rm temp_cut.log
                fi
            fi
        done <temp_final.txt
    rm temp_final.txt
else
    echo "syntax error: unknown element"
fi
