#!/usr/bin/python
#------------------------------------------------------------------
# recover_grub_file.py
#
# Copyright (c) 2019 by cisco Systems, Inc.
# All rights reserved.
#------------------------------------------------------------------

import subprocess
import logging
from os import path
from const import *

SYSADMIN_SHOW_CMD = " /opt/cisco/calvados/bin/show_cmd show"
HOST_GRUB_FILE =  "/misc/disk1/tftpboot/grub.cfg"
HOST_GRUB_DIR = "/misc/disk1/tftpboot"
TEMP_DIR = "/misc/disk1"
PLATFORM_PREFIX = "/opt/cisco/calvados/bin/vrfch.sh CTRL_VRF"

def get_platform_type():
    """This is used to detect the platform type."""
    logger.debug("Fetching platform type info")
    cmd = "/usr/bin/xr_sysctl -n kernel.boot.cmdline"
    try:
        cmd = run_cmd(cmd)
    except RuntimeError as err:
        error_descp(err)
        sys.exit(0)   
    if cmd:
        p = re.compile("platform=(.*) ")
        val = p.findall(cmd)
        logger.debug("Platform type is %s" %val)
        if val:
            val = val[0].split()[0]
        else:
            logger.error("Unable to get the Platform Information.")
            sys.exit(0)
        if val == 'panini':
            platform = ncs6k
        elif val == 'asr9k':
            platform = asr9k
        else:
            logger.error("Not a Supported Platform.")
            sys.exit(0)
        return platform
    else:
        logger.error("Unable to get the Platform Information.")
        sys.exit(0)

def run_cmd (cmd, skip_logging = False):
    process = subprocess.Popen(cmd, stdout=subprocess.PIPE,
                               stderr=subprocess.PIPE, shell=True)
    out, error = process.communicate()
    sprc = process.returncode
    if sprc is None or sprc != 0:
        out = error
        raise RuntimeError("Error CMD=%s returned --->%s" % (cmd, out))
    else:
        if not skip_logging:
            logger.debug("CMD=%s OUTPUT=%s"%(cmd, out))
        pass
    return out.strip()

def copy_grub_file(host_ip, location, vm_ip):
    try:
        logger.info("started to recover grub.cfg file for sysadmin  %s" %location)
        global HOST_GRUB_FILE
        global HOST_GRUB_DIR

        temp = "cat /boot/grub2/grub.cfg | grep signfile"
        cmd = ("%s ssh -T -q %s %s" %(PLATFORM_PREFIX, host_ip, temp))
        try:
            updStr = run_cmd(cmd)
            if updStr:
                updStr = updStr.split("\n")[0]
                id = updStr.find("signfile")
                new_sf = updStr[id:]
        except RuntimeError as err:
            str_err = str(err)   
            err_type = str_err.split("--->")[1]
            if err_type is "":
                new_sf = None
            else:
                error_descp(err)
                sys.exit(0)
        logger.debug("new_sf is: %s" %new_sf)
        ########copy new .cfg file#######
        updStr = "cp /boot/grub2/grub.cfg /misc/disk1/tftpboot/grub.cfg.tmp"
        cmd = ("%s ssh -T -q %s %s" %( PLATFORM_PREFIX, host_ip, updStr))
        run_cmd(cmd)

        if new_sf is None:
            updStr = "\"sed -i -e \\\"s;search.*;loopback loop system_image.iso;\\\"" +\
            " -e \\\"s;root=<P.*>;root=/dev/ram install=/dev/sda;\\\"" +\
            " -e \\\"s;from Disk;from PXE;\\\" " +\
            " -e \\\"s; /boot/; (loop)/boot/;\\\" %s.tmp\"" %(HOST_GRUB_FILE)
        else:
            updStr = "\"sed -i -e \\\"/search.*/ d\\\" " +\
            " -e \\\"s;root=<P.*>;root=/dev/ram install=/dev/sda;\\\"" +\
            " -e \\\"s;from Disk;from PXE;\\\"" +\
            " -e \\\"s;quiet;quiet installiso=system_image.iso;\\\"" +\
            " -e \\\"s;initrd .*;initrd /boot/initrd.img %s;\\\" %s.tmp\"" %(new_sf, HOST_GRUB_FILE)

        cmd = ("%s ssh -T -q %s %s" %(PLATFORM_PREFIX, host_ip, updStr))
        run_cmd(cmd)

        grub_file =  "/etc/rc.d/init.d/platform_install_update_pxeboot_grub.sh"
        if path.isfile(grub_file):
            updStr = "%s %s.tmp" %(grub_file, HOST_GRUB_FILE)
            cmd = ("%s ssh -T -q %s %s" %(PLATFORM_PREFIX, host_ip, updStr))
            run_cmd(cmd)

        cmd = ("%s scp %s:%s.tmp %s" % (PLATFORM_PREFIX, host_ip, HOST_GRUB_FILE, TEMP_DIR))
        run_cmd(cmd)

        cmd = ("%s scp %s/grub.cfg.tmp %s:%s" %(PLATFORM_PREFIX, TEMP_DIR, vm_ip, HOST_GRUB_FILE))
        run_cmd(cmd)
        
        cmd = ("rm -rf %s/grub.cfg.tmp" %(TEMP_DIR))
        run_cmd(cmd)
   
        logger.info("Successfully recovered grub.cfg file for sysadmin  %s" %location)

    except RuntimeError as err:
        error_descp(err)
        sys.exit(0)  

def grub_exist_func(vm_info):
        logger.debug("Checking grub.cfg file existence")
        temp = "find /misc/disk1/tftpboot -name grub.cfg"
        cmd = ("%s ssh %s %s" %(PLATFORM_PREFIX, vm_info, temp))
        try:
            res = run_cmd(cmd)
        except RuntimeError as err:
            error_descp(err)
            sys.exit(0)  
        if "/misc/disk1" in res:
            logger.debug("grub.cfg file found")
            return True
        logger.debug("grub.cfg file not found")
        return False

def error_descp(err):
    logger.error('*' * 80)
    logger.error(err)
    logger.error('*' * 80)

def sysadmin_vm_info(host_ip):
        logger.debug("Fetching vm info details")
        cmd = ('%s %s VM' % (PLATFORM_PREFIX, SYSADMIN_SHOW_CMD))
        try:
            cmd = run_cmd(cmd)
        except RuntimeError as err:
            error_descp(err)
            sys.exit(0)  
        if not cmd:
            logger.error("Failed to fetch VM Information")
            sys.exit(0)
        else:
            logger.info("Checking for all sysadmin vm")
            for line in cmd.split(os.linesep):
                if "Location" in line:
                    location = line.split()[1]
                if "running" in line:
                    vm_name = line.split()[0]
                    logger.debug("location %s and vm name is %s" %(location,vm_name))
                    if ("sysadmin" in vm_name) and ("RP" in location or "RSP" in location or "CB" in location):
                        vm_ip = re.findall(r'[0-9]+(?:\.[0-9]+){3}', line)
                        vm_ip = " ".join(str(x) for x in vm_ip)
                        logger.debug("vm ip is %s" %vm_ip) 
                        grub_exist = grub_exist_func(vm_ip)
                        logger.info('*' * 80)
                        if grub_exist  == False:
                            logger.info("grub.cfg file does not exist at sysadmin %s" %(location))
                            copy_grub_file(host_ip, location, vm_ip)
                        else:
                            logger.info("grub.cfg file exists at sysadmin  %s" %location)
                            logger.info("Nothing to recover")
                        logger.info('*' * 80)

def get_host_ip():
        logger.debug("Fetching host ip address")
        try:
            s = [x for x in run_cmd("/opt/cisco/calvados/bin/print_bootstrap_var HOST_IP_ADDR")]
            host_ip='.'.join([str(int(x,16)) for x in [''.join(s[i:i+2]) for i in range(0, len(s), 2)]])
            logger.debug("host ip is: %s" %host_ip)   
            return host_ip
        except RuntimeError as err:
            error_descp(err)
            sys.exit(0) 
    
if __name__== "__main__":
   
    # create logger
    logger.setLevel(logging.DEBUG)
    formatter = logging.Formatter('%(levelname)s %(message)s')

    formatter1 = logging.Formatter(
        '%(asctime)s:: %(levelname)s %(message)s',
        "%Y-%m-%d %H:%M:%S")     

    # Console message
    ch = logging.StreamHandler(sys.stdout)
    ch.setFormatter(formatter)
    ch.setLevel(logging.INFO)
    logger.addHandler(ch)

    # Logs to logfile
    fh = logging.handlers.RotatingFileHandler(LOGFILE, maxBytes=(1024 * 1000),
                                              backupCount=5)
    fh.setLevel(logging.DEBUG)
    fh.setFormatter(formatter1)
    logger.addHandler(fh)

    logger.debug("Start executing gurb.cfg recovery tool")
    platform = get_platform_type()
    logger.info("Platform is: %s" %platform)
    host_ip = get_host_ip() 
    sysadmin_vm_info(host_ip)


