#!/bin/bash
#
# This script sets the ROMMON variables and emt information for TFTP BOOT and eUSB BOOT classic XR image
#
# Copyright (c) 2014-2017, 2019 by Cisco Systems, Inc.
# All rights reserved.

source /etc/init.d/set_emt_value_lib.sh 

# Refer all OFFSET's in libasr9kpd:emt.c
BOOT_PROTOCOL_OFF=0x050
IPV4_ADDRESS_OFF=0xa8c
GATEWAY_ADDRESS_OFF=0xa90
SUBNET_MASK_OFF=0xa94
TFTP_SERVER_OFF=0xa98
BOOT_FILENAME_OFF=0x0f0
# Refer BINFO_FLEN_V2 in libasr9kpd:bootinfo.h 
BINFO_FLEN_V2=128

function validate_ip()
{
	local ip=$1
	local hex_ip=""

	if [[ $ip =~ ^[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}$ ]]; then
		TEMP=$IFS
		IFS='.'
		ip=($ip)
		IFS=$TEMP
		if [[ ${ip[0]} -le 255 && ${ip[1]} -le 255 && ${ip[2]} -le 255 && ${ip[3]} -le 255 ]]; then
			# translate the ip to 4 byte hex
			hex_ip=$(printf "%02x" ${ip[0]})$(printf "%02x" ${ip[1]})$(printf "%02x" ${ip[2]})$(printf "%02x" ${ip[3]}) 
		fi
	fi
	echo "$hex_ip"
}

if [[ $# -ne 14 &&  $# -ne 15 ]]; then
	error_exit "Error: Incorrect number of input provided."
fi

while [ $# -gt 0 ]; do
    case "$1" in
        -b|--boot)
            shift
            if [[ "$1" != "tftp" ]]; then
                error_exit "Error: Invalid input."
            fi
            BOOT_TYPE="$1"
            shift
            ;;
        -a|--address)
            shift
            HEX_IPV4_ADDRESS=$(validate_ip $1)
            if [[ HEX_IPV4_ADDRESS == "" ]]; then
                error_exit "Error: Invalid IP address provided:" "$1"
            fi
            IPV4_ADDRESS="$1"
            shift
            ;;
        -g|--gw)
            shift
            HEX_GATEWAY_ADDRESS=$(validate_ip $1)
            if [[ HEX_GATEWAY_ADDRESS == "" ]]; then
                error_exit "Error: Invalid gateway IP address provided:" "$1"
            fi
            GATEWAY_ADDRESS="$1"
            shift
            ;;
        -n|--netmask)
            shift
            HEX_SUBNET_MASK=$(validate_ip $1)
            if [[ HEX_SUBNET_MASK == "" ]]; then
                error_exit "Error: Invalid subnet mask provided:" "$1"
            fi
            SUBNET_MASK="$1"
            shift
            ;;
        -s|--server)
            shift
            HEX_TFTP_SERVER=$(validate_ip $1)
            if [[ HEX_TFTP_SERVER == "" ]]; then
                error_exit "Error: Invalid TFTP server IP provided:" "$1"
            fi
            TFTP_SERVER="$1"
            shift
            ;;
        -p|--path)
            shift
            if [[ ${#1} -le $BINFO_FLEN_V2 ]]; then
                BOOT_FILENAME="$1"
            else
                error_exit "Error: Image path provided exceeds maximum length allow. Please shorten it to $BINFO_FLEN_V2 characters or under."
            fi
            shift
            ;;
        -r|--reboot)
            shift
            REBOOT_AUTOMATICALLY=1
            ;;
        -m|--bootup_slot)
            shift
            shift
            ;;
		*)
			error_exit "Error: Invalid input detected."
			;;
	esac
done

if [[ ${BOOT_TYPE} == "" || ${HEX_IPV4_ADDRESS} == "" || ${HEX_GATEWAY_ADDRESS} == "" || ${HEX_SUBNET_MASK} == "" || ${HEX_TFTP_SERVER} == "" || ${BOOT_FILENAME} == "" ]]; then
	error_exit "Error: Invalid input."
fi
/opt/cisco/calvados/bin/nvram_dump -s MIGRATE=1
echo "Updated the Migration Flag "

/opt/cisco/calvados/bin/nvram_dump -s BOOT=tftp:
echo "Updated the BOOT Parameter Flag "

/opt/cisco/calvados/bin/nvram_dump -s IP_ADDRESS=$IPV4_ADDRESS
echo "Updated IP_ADDRESS to $IPV4_ADDRESS"

/opt/cisco/calvados/bin/nvram_dump -s DEFAULT_GATEWAY=$GATEWAY_ADDRESS
echo "Updated DEFAULT_GATEWAY to $GATEWAY_ADDRESS"

/opt/cisco/calvados/bin/nvram_dump -s IP_SUBNET_MASK=$SUBNET_MASK
echo "Updated IP_SUBNET_MASK to $SUBNET_MASK"

/opt/cisco/calvados/bin/nvram_dump -s TFTP_SERVER=$TFTP_SERVER
echo "Updated TFTP_SERVER to $TFTP_SERVER"

/opt/cisco/calvados/bin/nvram_dump -s TFTP_FILE=$BOOT_FILENAME
echo "Updated TFTP_FILE to $BOOT_FILENAME"

/opt/cisco/calvados/bin/nvram_dump -s RMEMVA=
echo "Unset RMEMVA"
