#!/bin/bash 
#
# pxe-install.sh
# This PI script replaces the old xrnginstall script. 
# The source code resides in calvados WS, which is much more handy.
# The script prepares a host and a calvados volume for the next disk boot. 
# Note: even if the name says pxe, this script is also used for example
# in the case of USB boot.
#
#  With install code (LVM supported)
#    1 Boot partition (300M - HW, 900M - Sim)
#    2 Software repository (2GB)
#    3 LVG - For host, sysadmin-vm, xr lvm's (27.5G)
#    4 for Grub (empty) for EFI bootable (20M)
#
# 2014.05.08 francois Fixed indentation, replaced most tabs with spaces, 
#            simplified script (remove LVM_SUPPORTED and DATA_LV_SUPPORTED).
#            Move PD variables to a PD file. Removed unused capability
#            to prepare XR ISO (remove install_xr_iso & sort_xr_packages).
#
# ---todo---
# -Move some generic functions to separate scripts, callable by both
#  calvados install and xrnginstall. This way, there is only one
#  implementation.
# -search for vconfig: there is some PD vlan related code that needs rework
# -fix install_host_iso so that initrd.img is taken from the hostiso 
# -fix install_host_iso so that RPMs are taken from the hostiso and ALL RPMs are used
# -understand what "legacy" means and let PD specify it
# -selection of disk to be used for install (vda, vdb) should be moved to PD?
#  if so, need to cleanup a few places where hardcoded.
#
# Copyright (c) 2014-2019 by Cisco Systems, Inc.

# Log script callflow
readonly LOGFLOWDIR=/tmp/bakelog
readonly LOGFLOWFILE=`basename $0`.flow.log

. /etc/init.d/logflow
mkdir -p $LOGFLOWDIR
logflow $LOGFLOWDIR/$LOGFLOWFILE

# Keep this code on top of this file.
# Platform hook when pxe_install starts
if [[ $1 == start ]]; then
    if [[ -f /etc/init.d/platform_pxe_start_hook.sh ]]; then
        source /etc/init.d/platform_pxe_start_hook.sh
    fi
fi

if [ -f /etc/rc.d/init.d/pd-functions ]; then
    source /etc/rc.d/init.d/pd-functions
fi
#==============================================================
#                     functions
#==============================================================
function step_log_console {
    local log_msg=${1}
    local time_str=$(date)
    echo "${time_str}: ${log_msg}" >&105
    # Also log console message to file
    echo "${time_str}: ${log_msg}" >&107
    if [ $USE_FINAL -gt 0 ]; then
        sync
    fi
}
readonly -f step_log_console

function verify_md5sum
{
   local path=${1}
   local i_sum
   local o_sum

   i_sum=$(cat ${path}/iso_info.txt 2>/dev/null | grep initrd | awk '{print $3}')
   if [ -z "${i_sum}" ]; then
      return 0
   else
      o_sum=$(md5sum ${path}/boot/initrd.img | awk '{print $1}')
      if [ "${i_sum}" == "${o_sum}" ]; then
         return 0
      else
         return 1
      fi
   fi
}

function run_depmod {
    for d in $1/lib/modules/*;
    do

        if [ -d $d ]; then
            depmod -a -b $1 `basename $d`
        fi
    done
}

function vxr_sim_mode_check {

    #
    # Default to no sim mode.
    #
    VXR_SIM_MODE=0

    if (( $SIMULATION )); then
        VXR_SIM_MODE=1
    fi

    # Allow platform specific tweaking to VXR_SIM_MODE
    #
    declare -F pd_pxe_vxr_sim_mode_check >&107 2>&1
    if [ $? -eq 0 ]
    then
        pd_pxe_vxr_sim_mode_check
    fi
}

function install_sim_tools {
    # we need to untar the binaries we need in the target partition
    step_log_console "SIMULATION PLATFORM: Unpacking tools.."
    mkdir -p ${PARTMNT}/opt/simulation 
    pushd ${PARTMNT}/opt/simulation >&107 2>&1
    bzip2 -dc /opt/simulation/install.tar.bz2 |  tar --no-acls -xf -
    popd >&107 2>&1
}

function remove_sim_tools {
    step_log_file "Remove simulation tools"
    rm -rf ${PARTMNT}/opt/simulation/install*
}

function create_vfat_partition {
    local dev_name=${1} label=${2}

    # Reformat partition with an FAT16 file system
    step_log_file "Create vfat partition ${dev_name}"
    
    mkfs.vfat -n EFS ${dev_name} >&107 2>&1
    sync
}



# This function creates 3 main disk partitions, with size specified by platform variables:
# 1   $DISK_HOST_BOOT_PART_SIZE
# 2   $DISK_HOST_DATA_PART_SIZE--->optional. If not specified, take rest of disk
# 3   $DISK_HOST_EFI_PART_SIZE
# IMP - Please Keep App Host Partition always at last - whenever you create new partition
# App Hosting Volume Creation is designed to take the last partition available in the disk
# to avoid complexity such as: any platform needs partition need not do anything special
# if app hosting partition is at the last and need not worry about legacy/non-legacy Boottype

function create_parted_partition {
    local dev_name=${1}
    local dev=/dev/${dev_name}
    local lvm_part_size_value=${DISK_HOST_DATA_PART_SIZE}
    local rpart=${2}

    step_log_file "Create parted partition ${dev} ${rpart}"

    # zero the first 10000 1024 blocks
    /bin/dd if=/dev/zero of=${dev} bs=1024 count=10000 >&107 2>&1
    parted -s ${dev} mklabel gpt  >&107 2>&1

    case ${PLATFORM} in
        fretta | zermatt | ncs560)
            # Reserve space for debug partition
            if [ "${BOARDTYPE}" == "RP" ]; then
                get_disk_size ${dev}
                if [ -z ${DISK_SIZE} ]; then
                    step_log_console "ERROR! No install boot device found, exiting..."
                    exit 0
                fi
                
                # Reserve space for debug partition
        	DISK_SIZE=$(($DISK_SIZE - $DEBUG_PART_SIZE))
                let second_disk_size=$DISK_SIZE-$DISK_HOST_BOOT_PART_SIZE-$DISK_CALVADOS_REPO_SIZE-$DISK_HOST_EFI_PART_SIZE-$lvm_part_size_value-$DEBUG_PART_SIZE
                #
                # Checking for App Hosting Enablement and configuring Disk size accordingly
                #
                declare -F app_host_with_separate_pv >&107 2>&1 && app_host_with_separate_pv
                if [[ $? -eq 0 ]]; then
                    let second_disk_size=$DISK_SIZE-$DISK_HOST_BOOT_PART_SIZE-$DISK_CALVADOS_REPO_SIZE-$DISK_HOST_EFI_PART_SIZE-$DISK_TP_VOL_PART_SIZE-$lvm_part_size_value
                fi
                step_log_console "Using remaining disk space $second_disk_size MB for secondary disk functions"

            fi
            step_log_console "LVM size ($lvm_part_size_value MB) DEBUG part size ($DEBUG_PART_SIZE MB)"
            step_log_file "LVM size ($lvm_part_size_value MB) DEBUG part size ($DEBUG_PART_SIZE MB)"
            ;;
        asr9k )
            get_disk_size ${dev}
            if [ -z ${DISK_SIZE} ]; then
                step_log_console "ERROR! No install boot device found, exiting..."
                exit 0
            fi
            let lvm_part_size_value=$DISK_SIZE-$DISK_HOST_BOOT_PART_SIZE-$DISK_CALVADOS_REPO_SIZE-$DISK_HOST_EFI_PART_SIZE
            declare -F app_host_with_separate_pv >&107 2>&1 && app_host_with_separate_pv
            if [[ $? -eq 0 ]]; then
                let lvm_part_size_value=${lvm_part_size_value}-$DISK_TP_VOL_PART_SIZE
            fi
            # Reserve space for debug partition
            lvm_part_size_value=$(($lvm_part_size_value - $DEBUG_PART_SIZE))
            step_log_console "LVM size ($lvm_part_size_value MB) DEBUG part size ($DEBUG_PART_SIZE MB)"
            step_log_file "LVM size ($lvm_part_size_value MB) DEBUG part size ($DEBUG_PART_SIZE MB)"
            ;;
        *)
            if [ -z ${lvm_part_size_value} ]; then
                get_disk_size ${dev}
                if [ -z ${DISK_SIZE} ]; then
                    step_log_console "ERROR! No install boot device found, exiting..."
                    # If we cannot find disk, it could be sign of SSD to about
                    # going bad. A reimage retry sometime can recover from this
                    # failure condition, but this is a sign that will tell
                    # user that this failure could become permanent.
                    declare -F pd_notify_hw_fatal_event &>/dev/null && \
                        pd_notify_hw_fatal_event "ERROR! No install boot device found, exiting..."
                    exit 0
                fi
                declare -F pd_pxe_skip_disk_size_check >&107 2>&1
                if [ $? -eq 0 ]; then
                    step_log_console "No max disk size on this platform"
                    pd_pxe_skip_disk_size_check
                elif [ ${DISK_SIZE} -ge 199000 ]; then
                    if [ "${BOARDTYPE}" == "CC" ]; then
                        step_log_console "Ignoring disk size restriction for CC card boot up"
                    else    
                        step_log_console "ERROR! No install boot device found, disk found too big (${DISK_SIZE}MB), exiting..."
                        declare -F pd_notify_hw_fatal_event &>/dev/null && \
                            pd_notify_hw_fatal_event "No install boot device found, disk found too big (${DISK_SIZE}MB), exiting installation"
                    exit 0
                    fi
                fi
                let lvm_part_size_value=$DISK_SIZE-$DISK_HOST_BOOT_PART_SIZE-$DISK_CALVADOS_REPO_SIZE-$DISK_HOST_EFI_PART_SIZE
            fi
            # Reserve space for debug partition
            lvm_part_size_value=$(($lvm_part_size_value - $DEBUG_PART_SIZE))
            step_log_console "LVM size ($lvm_part_size_value MB) DEBUG part size ($DEBUG_PART_SIZE MB)"
            step_log_file "LVM size ($lvm_part_size_value MB) DEBUG part size ($DEBUG_PART_SIZE MB)"
            ;;
    esac

    let beg_part1=0
    let end_part1=0+${DISK_HOST_BOOT_PART_SIZE}
    let beg_part2=${end_part1}
    let end_part2=${end_part1}+${DISK_CALVADOS_REPO_SIZE}
    let beg_part3=${end_part2}
    let end_part3=${end_part2}+${lvm_part_size_value}
    let beg_part4=${end_part3}
    let end_part4=${end_part3}+${DISK_HOST_EFI_PART_SIZE}

    step_log_console "Creating partitions, BOOT=${DISK_HOST_BOOT_PART_SIZE}MB, LVM=${lvm_part_size_value}MB, EFI=${DISK_HOST_EFI_PART_SIZE}MB"

    # Retain recovery partition for ncs6k SC card if it's present. Recovery partition is
    # carved out of partition02, thus contingent on partition02.
    if [ -n "${rpart}" -a "${PLATFORM}" == "panini" -a "${BOARDTYPE}" == "SC" ]; then
       beg_rpart=$((${end_part2}-${DISK_MAIN_RECOVERY_PART_SIZE}))
       end_rpart=${end_part2}
       end_part2=${beg_rpart}
    fi

    parted -s --align=min ${dev} mkpart partition01 ${beg_part1} ${end_part1} >&107 2>&1
    parted -s --align=min ${dev} mkpart partition02 ${beg_part2} ${end_part2} >&107 2>&1
    parted -s --align=min ${dev} mkpart partition03 ${beg_part3} ${end_part3} >&107 2>&1
    parted -s --align=min ${dev} mkpart partition04 ${beg_part4} ${end_part4} >&107 2>&1

    if [ "${PLATFORM}" == "fretta" -o  "${PLATFORM}" == "zermatt" -o "${PLATFORM}" == "ncs560" ] && \
            [ "${BOARDTYPE}" == "RP" ]; then
        let end_part5=${end_part4}+${second_disk_size}
        parted -s --align=min ${dev} mkpart partition05 ${end_part4} ${end_part5} >&107 2>&1
        step_log_console "Created partition of size $second_disk_size for use as secondary disk"
    fi

    if [ "${PLATFORM}" == "ncs1k" ] || [ "${PLATFORM}" == "ncs1001" ]; then
		let end_part5=${end_part4}+${DISK_HOST_GOLDEN_BOOT_SIZE}
		let end_part6=${end_part5}+${DISK_HOST_GOLDEN_EFI_SIZE}

		parted -s --align=min ${dev} mkpart partition05 ${end_part4} ${end_part5} >&107 2>&1
		parted -s --align=min ${dev} mkpart partition06 ${end_part5} ${end_part6} >&107 2>&1
		step_log_console "Created partition Specific to ncs1k"

		check_fs_partition "${dev}5" "${DISK_HOST_GOLDEN_LABEL}"
		check_fs_partition "${dev}6" "${DISK_HOST_GOLDEN_EFI_LABEL}"
    fi

    if [ "${PLATFORM}" == "ncs1004" ]; then
        let end_part5=${end_part4}+${DISK_HOST_GOLDEN_CONFIG_SIZE}
        let end_part6=${end_part5}+${DISK_TP_2ND_VOL_PART_SIZE}

       	parted -s --align=min ${dev} mkpart partitionbk ${end_part4} ${end_part5} >&107 2>&1
    	step_log_console "Created partition Specific to ncs1004 and ncs1k to store golden configs"
       	check_fs_partition "${dev}5" "${DISK_HOST_GOLDEN_CONFIG_LABEL}"
        declare -f dr_create_gdirs >&107 2>&1
        if [ "$?" -eq 0 ]; then
	    	dr_create_gdirs ${DISK_HOST_GOLDEN_CONFIG_LABEL}

	        if [ $DISASTER_RECOVERY == "yes" ]; then
    	       step_log_console "DR baking ISO backup not needed"
	        else
               declare -F pd_dr_iso_callback >&107 2>&1 && pd_dr_iso_callback
            fi
    	fi

        # Additional 20GB space for AppHsosting
       	parted -s --align=min ${dev} mkpart partitiontp2 ${end_part5} ${end_part6} >&107 2>&1
        step_log_console "Created partition size ${DISK_TP_2ND_VOL_PART_SIZE}MB specific to ncs1004 and AppHosting Additional Space"
       	check_fs_partition "${dev}6" "${DISK_TP_2ND_VOL_LABEL}"
        declare -f tp_create_additional_dirs >&107 2>&1
        if [ "$?" -eq 0 ]; then
	    	tp_create_additional_dirs ${DISK_TP_2ND_VOL_LABEL}
    	fi
    fi
   
    #
    # Checking for App Hosting Enablement and configuring Partitioning accordingly
    # Keeping the app-host partition always at the last - calculate Prev from Total
    #
    declare -F app_host_with_separate_pv >&107 2>&1 && app_host_with_separate_pv
    if [[ $? -eq 0 ]]; then
       let end_part_prev=end_part$((PART_NUM_TP -1))
       let end_part_tp=${end_part_prev}+${DISK_TP_VOL_PART_SIZE}
       parted -s --align=min ${dev} mkpart partitiontp ${end_part_prev} ${end_part_tp} >&107 2>&1
       step_log_console "Created Partition of size $DISK_TP_VOL_PART_SIZE for use of App-Volume"
    fi

    # WARNING 
    # Recovery partition was created on the fly when DR feature is enabled. Thus it will be
    # created later than static partitions. Current install implementation requires recovery
    # partition be the 5th partition on ncs6k SC card. Recovery partition does not conflict
    # with AppHosting Enablement for the time being since they are on different cards or on
    # different disks.
    if [ -n "${rpart}" -a "${PLATFORM}" == "panini" -a "${BOARDTYPE}" == "SC" ]; then
        parted -s --align=min ${dev} mkpart partition05 ${beg_rpart} ${end_rpart} >&107 2>&1 
        # set the DR partition to be bootable
        parted -s ${dev} set 5 boot on >&107 2>&1 
        step_log_console "Carved out recovery partition ${dev}${rpart} ${DISK_MAIN_RECOVERY_PART_SIZE}MB"
    fi

   
    if [ "${PLATFORM}" == "fretta" -o "${PLATFORM}" == "zermatt" ] && [ "${DR_EFI_EXISTS}" == "true" ]; then
        declare -F pd_recreate_dr_efi_part &> /dev/null && activate_disk_one_vg && \
            pd_recreate_dr_efi_part "${DISK1_PV}-${DISK1_LV}${D1_HOST_LV}"
    fi

    parted -s ${dev} set $PART_NUM_EFI boot on >&107 2>&1
}

function install_boot_iso {
    local dev_name=${1}
    local dev=/dev/${dev_name}
    local device_part=${2}

    mount -r -o loop /iso/$HOST_ISO ${ISOMNT}
    ret_code=$?
    if [ $ret_code -ne 0 ]; then
      declare -F pd_notify_sw_fatal_event &>/dev/null && \
          pd_notify_sw_fatal_event "Mounting of /iso/$HOST_ISO to ${ISOMNT} failed with error code $ret_code"
      exitclean
    fi

    # First install host OS
    mount -o discard ${dev}${device_part} ${PARTMNT}
    ret_code=$?
    if [ $ret_code -ne 0 ]; then
      declare -F pd_notify_sw_fatal_event &>/dev/null && \
          pd_notify_sw_fatal_event "Mounting of ${dev}${device_part} to ${PARTMNT} failed with error code $ret_code"
      exitclean
    fi
    cd ${PARTMNT}

    if [ -f ${ISOMNT}/boot/initrd.img.host ]; then
      zcat ${ISOMNT}/boot/initrd.img.host | cpio -d -i >&107 2>&1
      ret_code=$?
      if [ $ret_code -ne 0 ]; then
        declare -F pd_notify_error_event &>/dev/null && \
            pd_notify_error_event "zcat of ${ISOMNT}/boot/initrd.img.host returned error code $ret_code"
      fi
    else 
      zcat ${ISOMNT}/boot/initrd.img | cpio -d -i >&107 2>&1
      ret_code=$?
      if [ $ret_code -ne 0 ]; then
        declare -F pd_notify_error_event &>/dev/null && \
            pd_notify_error_event "zcat of ${ISOMNT}/boot/initrd.img returned error code $ret_code"
      fi
    fi

    #cp iso_info.txt to partition for future references
    cp ${ISOMNT}/${ISO_INFO_FILE} ${PARTMNT}

    # delete any of the cached RPM db files that come with the prebuilt RPM db
    # in the initrd.img
    rm -f ./var/lib/rpm/__db.*
    chroot_support=$(check_chroot_support ${ISOMNT})
    if [ "${chroot_support}" == "yes" ] ; then
        #install hostos rpm from the ISO, cleanup __db files so that any references to 
        # --root path in __db is no longer used.
        cp ${ISOMNT}/rpm/*hostos*.rpm ${PARTMNT}
        #bind mount /proc and PD defined FS to ${PARTMNT}
        mount_efi_and_proc ${PARTMNT}
        mount_pd_fs ${PARTMNT}
        chroot ${PARTMNT} rpm -Uvh *hostos*.rpm &> /dev/null
        rm -fr ${PARTMNT}/*hostos*.rpm
        #unmount /proc and other FS passed to ${PARTMNT}
        umount_efi_and_proc ${PARTMNT}
        umount_pd_fs ${PARTMNT}
    fi


    rm -f ./var/lib/rpm/__db.*

    #cp /boot/initrd.img  ${PARTMNT}/boot/
    umount ${ISOMNT}
    ret_code=$?
    if [ $ret_code -ne 0 ]; then
      declare -F pd_notify_sw_fatal_event &>/dev/null && \
          pd_notify_sw_fatal_event "Unmounting of ${ISOMNT} failed with error code $ret_code"
      exitclean
    fi

    # Create SDR_ISO as a marker file if this is LVM mode
    mkdir -p ./iso
    echo "This is only a marker file." > ./iso/$SDR_ISO

    mkdir -p ./iso
    echo "This is only a marker file." > ./iso/data_part.txt

    rm -fr ${PARTMNT}/boot/xr-image.img
    mkdir -p ${PARTMNT}/proc ${PARTMNT}/sys ${PARTMNT}/root/.ssh
    cp ${SSHKEY_DIR}/host/* ${PARTMNT}/root/.ssh/
    mkdir -p ${PARTMNT}/tmp ${PARTMNT}/dev 
    chmod -R 700 ${PARTMNT}/root
    chown -R root:root ${PARTMNT}/*
    cd /

    vxr_sim_mode_check
    if (( $VXR_SIM_MODE )); then
        install_sim_tools
           source ${PARTMNT}/opt/simulation/install/patches/xr_begin.sh
           source ${PARTMNT}/opt/simulation/install/patches/hostOS_begin.sh
	   (
		cd ${PARTMNT}/lib64
	        for i in libz.so.1 libtinfo.so.5; do
		    [ -f $i.* ] && ln -sf $i.* $i
                done
	    )
        sync
        sync
    fi

    step_log_file "Install host grub setup ${PARTMNT} ${PLATFORM}"
    declare -F is_vxr_sim >&107 2>&1
    if [ $? -eq 0 ]; then
        is_vxr_sim
        if [[ $? != 0 ]]; then
            is_sim=1
        else
            is_sim=0;
        fi
    fi

    if [ "${BOARDTYPE}" == "CC" ]; then
          PLATFORM="panini"
    fi

    #fix menu.lst script
    pi_update_menu_lst_boot $BOOTTYPE $grubdir ${PARTMNT}  "1" ${HOST_LV_PART_BOOT} ${PLATFORM} ${dev}${device_part} ${boardtype} ${is_sim}

    if (( $VXR_SIM_MODE )); then
        source ${PARTMNT}/opt/simulation/install/patches/hostOS_end.sh
        remove_sim_tools
    fi

    if [ "$VIRT_METHOD" = "lxc" ]; then
        # Make dir for lxc hooks and create symbolic link to the hook script
        mkdir -p ${PARTMNT}/etc/libvirt/hooks/
        ln -s /etc/init.d/lxc_hook.sh ${PARTMNT}/etc/libvirt/hooks/lxc
        # Allow platform specific patching for HostOS boot partition
        LV_ID=`echo -n ${HOST_LV_NAME} | tail -c -1`
        if [ "${chroot_support}" == "yes" ] ; then
            #bind mount /proc and PD defined FS to ${PARTMNT}
            mount_efi_and_proc ${PARTMNT}
            mount_pd_fs ${PARTMNT}
            chroot ${PARTMNT} /etc/init.d/calvados_patch_lxc_iso.sh "/"  ${LVG_NAME} \
                         ${LV_ID} "hostos"
            #unmount /proc and other FS passed to ${PARTMNT}
            umount_efi_and_proc ${PARTMNT}
            umount_pd_fs ${PARTMNT}
            # Allow platform specific binary copies.
            #
            declare -F pd_pxe_copy_bake_binaries >&107 2>&1
            if [ $? -eq 0 ]
            then
                step_log_console "Copy required binaries for pxe environment"
                pd_pxe_copy_bake_binaries ${PARTMNT} true
            fi
        else 
            /etc/init.d/calvados_patch_lxc_iso.sh ${PARTMNT} ${LVG_NAME} \
                         ${LV_ID} "hostos"
        fi

    else
         # for VM based platforms
        if [ "${PLATFORM}" == "panini" ] #for NCS6K
        then
            step_log_console "pxe_install: install_boot_iso: NCS6K EXR : " ${PLATFORM}

            if mountpoint -q /sys/fs/selinux; then
                step_log_console "selinux already mounted"
            else
                if [ ! -d "/sys/fs/selinux" ]; then
                    mkdir -p /sys/fs/selinux
                fi
                mount -t selinuxfs selinuxfs /sys/fs/selinux
            fi
            /sbin/load_policy

            declare -F pd_lxc_script_for_vm_iso >&107 2>&1
            if [ $? -eq 0 ]
            then
                step_log_console "Patch the Hostos ISO for PD"
                # Allow platform specific patching for HostOS boot partition
                LV_ID=`echo -n ${HOST_LV_NAME} | tail -c -1`
                /etc/init.d/calvados_patch_lxc_iso.sh ${PARTMNT} ${LVG_NAME} \
                             ${LV_ID} "hostos"
            fi
        fi
    fi

    # Allow platform specific tweaking to grub menu.
    #
    declare -F pd_pxe_update_boot_grub >&107 2>&1
    if [ $? -eq 0 ]
    then
        step_log_file "PD: patch ${grubdir}"
        pd_pxe_update_boot_grub ${grubdir}
    else
        step_log_file "PD: no patching of $grubdir needed"
    fi

    #
    # Allow platform specific tweaking to any other things.
    #
    declare -F pd_pxe_install_boot_iso_patch >&107 2>&1
    if [ $? -eq 0 ]
    then
        step_log_file "PD: patch"
        pd_pxe_install_boot_iso_patch
    fi

    run_depmod ${PARTMNT}
    sync
    sync

    SIZE=$(du -hs --exclude="/proc" --exclude="/sys" ${PARTMNT} | awk '{print $1}')
    [ -d "${PARTMNT}" ] && umount ${PARTMNT}
}

function check_parted_fix_disk_part {

    local dev_name=${1}
    local dev=/dev/${dev_name}
    local expected_part=${2}
    #Always wipe off old partition
    local reformat=FALSE

    # Check if reformatting is necessary for ncs1k
    if [ "${PLATFORM}" == "ncs1k"  -a "${PERSIST_ENABLE}" == "TRUE" ] || [ "${PLATFORM}" == "ncs1001"  -a "${PERSIST_ENABLE}" == "TRUE" ] || [ "${PLATFORM}" == "ncs1004"  -a "${PERSIST_ENABLE}" == "TRUE" ] ; then
        #Extract size and label of ncs1k specific partitions 
        label_boot=$(/sbin/e2label "${dev}5")
        get_disk_size "${dev}5"
        persist_boot_size=${DISK_SIZE}

        label_efi=$(/sbin/e2label "${dev}6")
        get_disk_size "${dev}6"
        persist_efi_size=${DISK_SIZE}

        get_disk_size "${dev}1"
        persist_disk1_size=${DISK_SIZE}
        get_disk_size "${dev}2"
        persist_disk2_size=${DISK_SIZE}
        get_disk_size "${dev}3"
        persist_disk3_size=${DISK_SIZE}
        get_disk_size "${dev}4"
        persist_disk4_size=${DISK_SIZE}

        step_log_console "Persist efi size ${persist_efi_size} boot size ${persist_boot_size} DEBUG part size $DEBUG_PART_SIZE"
        step_log_console "Persist efi label ${label_efi} boot label ${label_boot}"
        step_log_console "Disk size ${persist_disk1_size}, ${persist_disk2_size}, ${persist_disk3_size}, ${persist_disk4_size}"

        if [[ ( "${DISK_HOST_GOLDEN_LABEL}" == "${label_boot}" ) && ( "${persist_boot_size}" == "${DISK_HOST_GOLDEN_BOOT_SIZE}" ) && ( "${DISK_HOST_GOLDEN_EFI_LABEL}" == "${label_efi}" )  && ( "${persist_efi_size}" == "${DISK_HOST_GOLDEN_EFI_SIZE}" ) && ( "${persist_disk1_size}" == "${DISK_HOST_BOOT_PART_SIZE}" ) && ( "${persist_disk2_size}" == "${DISK_CALVADOS_REPO_SIZE}" ) && ( "${persist_disk3_size}" == "${DISK_HOST_DATA_PART_SIZE}" ) && ( "${persist_disk4_size}" == "${DISK_HOST_EFI_PART_SIZE}" ) ]]; then
            reformat=TRUE
            step_log_console "Set the reformat tag"
        fi
        # Reserve space for debug partition
        persist_efi_size=$(($persist_efi_size - $DEBUG_PART_SIZE))
        step_log_console "Adjusted persist efi size ${persist_efi_size} DEBUG part size $DEBUG_PART_SIZE"
        DEBUG_PART_SIZE=0
    fi
    if [ "${reformat}" == "TRUE" ]; then
        step_log_console "Reformatting the partition"
        reformat_partition "${dev}1"
        reformat_partition "${dev}2"
        reformat_partition "${dev}3"
        reformat_partition "${dev}4"
    else
        #Check if recovery partition exists.
        rpart=
        declare -F pd_pxe_find_recovery_partition >&107 2>&1
        if [ $? -eq 0 ]; then
            rpart=$(pd_pxe_find_recovery_partition "${dev}" "$DISK_MAIN_RECOVERY_PART_NUM")
            if [ -n "${rpart}" ]; then
               step_log_console "Found recovery partition ${dev_name}${rpart}"
            fi
        fi
        create_parted_partition ${dev_name} ${rpart}
        total_part=$(find_parted_total_partition ${dev_name})
        if [[ "${total_part}" != "${expected_part}" ]]; then
            step_log_console $"Problem with parititon creating. NOT Bailing out"
            declare -F pd_notify_warning_event &>/dev/null && \
                pd_notify_warning_event "Disk partition creation failed in ${dev_name}: total partition created = ${total_part}, expected = ${expected_part}"
        fi
    fi
}
readonly -f check_parted_fix_disk_part


cleanup() {
    # pd_stop_watchdog is necessary only for some platforms
    declare -F pd_stop_watchdog && pd_stop_watchdog

    [ -d "${PARTMNT}" ] && umount ${PARTMNT} && rmdir ${PARTMNT}
    [ -d "${ISOMNT}" ] && umount ${ISOMNT} && rmdir ${ISOMNT}
#    [ -d "${PARTMNT}" ] && rmdir ${PARTMNT}
#    [ -d "${ISOMNT}" ] && rmdir ${ISOMNT}
}

exitclean() {
    step_log_console "Cleaning up to exit..."
    cleanup
    exit 1
}

# Platform supporting notification of SW FATAL event, will handle the reboot
# via RP after processing the SW FATAL event.
# For platform without this support, it will do auto reboot.
exitmd5sum_reboot() {
    declare -F pd_notify_sw_fatal_event &>/dev/null
    local auto_reboot=$?

    if [ $auto_reboot -eq 0 ]; then
        step_log_console "${1} md5sum verification failed"
    else
        step_log_console "${1} md5sum verification failed, cleaning up to reboot..."
    fi

    cleanup
    close_install_log
    
    if [ $auto_reboot -eq 0 ]; then
        pd_notify_sw_fatal_event "${1} md5sum verification failed"
    else
        call_reboot
    fi
}

function install_efi {
    local dev_name=${1}
    local dev=/dev/${dev_name}
    local device_part=${2}
    local root_part=${3}
    local label=${4}

    step_log_file "Install EFI ${dev}${device_part}"
    # First install host OS
    mount -o discard ${dev}${device_part} ${PARTMNT}
    ret_code=$?
    if [ $ret_code -ne 0 ]; then
      declare -F pd_notify_sw_fatal_event &>/dev/null && \
          pd_notify_sw_fatal_event "Mounting of ${dev}${device_part} to ${PARTMNT} failed with error code $ret_code"
      exitclean
    fi
    mkdir -p ${PARTMNT}/EFI.prep/Cisco
    cp -r /boot/grub2/bootx64.efi ${PARTMNT}/EFI.prep/Cisco/grub.efi
    cp -r /boot/grub2/bootx64.efi ${PARTMNT}/EFI.prep/Cisco/bootx64.efi
    cp -r /boot/grub2/grub.cfg ${PARTMNT}/EFI.prep/Cisco/grub.cfg

    declare -F pd_setup_capsule_update_support >&107 2>&1
    if [ $? -eq 0 ]
    then
        step_log_file "Preparing Capsule Update Support"
        pd_setup_capsule_update_support ${PARTMNT}/EFI.prep
    fi
   
    pi_update_efi_grub ${PARTMNT} ${HOST_LV_PART_BOOT} ${PLATFORM} ${BOARDPREFIX}${BOARDTYPE}  ${label} ${SIMULATION}

    # Allow platform specific tweaking to grub menu.
    #
    declare -F pd_pxe_update_efi_grub >&107 2>&1
    if [ $? -eq 0 ]
    then
        step_log_file "PD: patch ${PARTMNT}/EFI.prep/Cisco/"
        pd_pxe_update_efi_grub ${PARTMNT}/EFI.prep/Cisco/
    fi
    mv ${PARTMNT}/EFI.prep ${PARTMNT}/EFI
    cp -r ${PARTMNT}/EFI/Cisco ${PARTMNT}/EFI/boot
    sync
    sync
    [ -d "${PARTMNT}" ] && umount ${PARTMNT}
}

function install_host_iso {
    local dev_name=${1}
    local dev=/dev/${dev_name}
    local device_part=${2}

    source "/etc/init.d/pi_grub_and_menu_lst_update.sh"
    source "/etc/init.d/mount_pd_fs.sh"

    mount -r -o loop /iso/$HOST_ISO ${ISOMNT}
    ret_code=$?
    if [ $ret_code -ne 0 ]; then
      declare -F pd_notify_sw_fatal_event &>/dev/null && \
          pd_notify_sw_fatal_event "Mounting of /iso/$HOST_ISO to ${ISOMNT} failed with error code $ret_code"
      exitclean
    fi

    # First install host OS
    mount -o discard ${dev}${device_part} ${PARTMNT}
    ret_code=$?
    if [ $ret_code -ne 0 ]; then
      declare -F pd_notify_sw_fatal_event &>/dev/null && \
          pd_notify_sw_fatal_event "Mounting of ${dev}${device_part} to ${PARTMNT} failed with error code $ret_code"
      exitclean
    fi
    cd ${PARTMNT}
    if [ -f ${ISOMNT}/boot/initrd.img.host ]; then
      zcat ${ISOMNT}/boot/initrd.img.host | cpio -d -i >&107 2>&1
      ret_code=$?
      if [ $ret_code -ne 0 ]; then
        declare -F pd_notify_error_event &>/dev/null && \
            pd_notify_error_event "zcat of ${ISOMNT}/boot/initrd.img.host returned error code $ret_code"
      fi
    else 
      zcat ${ISOMNT}/boot/initrd.img | cpio -d -i >&107 2>&1
      ret_code=$?
      if [ $ret_code -ne 0 ]; then
        declare -F pd_notify_error_event &>/dev/null && \
            pd_notify_error_event "zcat of ${ISOMNT}/boot/initrd.img returned error code $ret_code"
      fi
    fi
    #cp /boot/initrd.img  ${PARTMNT}/boot/

    # For now, we check iso_info.txt for new packaging architecture.
    # As per this new packaging architecture, we will have a single
    # rpm per package instead of per card type rpm.
    exr_support=$(check_exr_support ${ISOMNT})
    chroot_support=$(check_chroot_support ${ISOMNT})

    #cp iso_info.txt to partition for future references
    cp ${ISOMNT}/${ISO_INFO_FILE} ${PARTMNT}

    # delete any of the cached RPM db files that come with the prebuilt RPM db
    # in the initrd.img
    rm -f ./var/lib/rpm/__db.*

    IsoType=$(check_iso_type ${ISOMNT})

    umount ${ISOMNT}
    ret_code=$?
    if [ $ret_code -ne 0 ]; then
      declare -F pd_notify_sw_fatal_event &>/dev/null && \
          pd_notify_sw_fatal_event "Unmounting of ${ISOMNT} failed with error code $ret_code"
      exitclean
    fi
    
    # Get card instance information and add it in the mount
    declare -f pd_get_card_inst > /dev/null
    if [ "$?" -eq 0 ]; then
        local CARD_INST
        pd_get_card_inst CARD_INST
        echo "$BOARDTYPE $CARD_INST" > $CARD_INST_FILE
        step_log_file $"Using board ${BOARDTYPE} and inst ${CARD_INST}"
    fi
    
     if [ -e $ISO_NAME_FILE ]; then
          cp /$ISO_NAME_FILE ${PARTMNT}/$ISO_NAME_FILE
     fi

    # Create SDR_ISO as a marker file if this is LVM mode
    mkdir -p ./iso
    echo "This is only a marker file." > ./iso/$SDR_ISO

    mkdir -p ./iso
    echo "This is only a marker file." > ./iso/data_part.txt

    # Install HostOs pkg
    mkdir -p rpm/hostos
    if [ "${exr_support}" == "yes" ]; then
        hostrpmfiles=$(ls /rpm/calvados/*hostos*.host.*.rpm 2>&107 | wc -l)
    else    
        hostrpmfiles=$(ls /rpm/calvados/*hostos*.rpm 2>&107 | wc -l)
    fi
    if [ "$hostrpmfiles" != "0" ]; then
      step_log_file "Install HostOs Rpm pkg cnt ${hostrpmfiles}"
      if [ "${exr_support}" == "yes" ]; then
          cp /rpm/calvados/*hostos*.host.*.rpm rpm/hostos
      else
          cp /rpm/calvados/*hostos*.rpm rpm/hostos
      fi
      for file in rpm/hostos/*
      do
         case ${file} in
            *.rpm) 
               STARTRPM=$(date +%s)
               chroot ${PARTMNT} rpm -i --nodeps ${file} >&107 2>&1
               ENDRPM=$(date +%s)
               DIFFRPM=$(( $ENDRPM - $STARTRPM ))
               SIZERPM=$(ls -lh ${file}| awk '{ print $5 }')
               step_log_file $"Install RPM ${file} of size ${SIZERPM} took ${DIFFRPM} sec"
               ;;
         esac
      done
    fi
    rm -rf rpm/hostos >&107 2>&1

    rm -fr ${PARTMNT}/boot/xr-image.img
    mkdir -p ${PARTMNT}/proc ${PARTMNT}/sys ${PARTMNT}/root/.ssh
    cp ${SSHKEY_DIR}/host/* ${PARTMNT}/root/.ssh/
    mkdir -p ${PARTMNT}/tmp ${PARTMNT}/dev 
    chmod -R 700 ${PARTMNT}/root
    chown -R root:root ${PARTMNT}/*
    cd / >&107 2>&1
 
    # Copy the platform board specific install configuration from ramfs 
    # to host boot partition
    cp $TEMP_INSTALL_FILE_NAME ${PARTMNT}/${PD_BOARD_INST_CONF}
    if (( $? != 0 )); then
        step_log_file $"Copying ${PD_BOARD_INST_CONF} failed"
    fi

    vxr_sim_mode_check
    if (( $VXR_SIM_MODE )); then
        install_sim_tools
        source ${PARTMNT}/opt/simulation/install/patches/hostOS_begin.sh
	(
	    cd ${PARTMNT}/lib64
	    for i in libz.so.1 libtinfo.so.5; do
		[ -f $i.* ] && ln -sf $i.* $i
	    done
	)
        sync
        sync
    fi

    cp ${PARTMNT}/boot/grub/menu.lst ${PARTMNT}/boot/grub/menu.lst.tmp
   
    sed -i -e "s;ONBOOT=yes;ONBOOT=no;" ${PARTMNT}/etc/sysconfig/network-scripts/ifcfg-eth0
    sed -i -e 's;^timeout *[0-9]*$;timeout 1;' ${PARTMNT}/boot/grub/menu.lst

    if (( $VXR_SIM_MODE )); then
        source ${PARTMNT}/opt/simulation/install/patches/hostOS_end.sh
        remove_sim_tools
    else
        rm -rf ${PARTMNT}/opt/simulation >&107 2>&1
        rmdir ${PARTMNT}/opt/simulation >&107 2>&1
    fi
 
    if [ "$VIRT_METHOD" = "lxc" ]; then
        # Make dir for lxc hooks and create symbolic link to the hook script
        mkdir -p ${PARTMNT}/etc/libvirt/hooks/
        ln -s /etc/init.d/lxc_hook.sh ${PARTMNT}/etc/libvirt/hooks/lxc
        # Allow platform specific patching for HostOS boot partition
        LV_ID=`echo -n ${HOST_LV_NAME} | tail -c -1`
        if [ "${chroot_support}" == "yes" ] ; then
            #bind mount /proc and PD defined FS to ${PARTMNT}
            mount_efi_and_proc ${PARTMNT}
            mount_pd_fs ${PARTMNT}
            chroot ${PARTMNT} /etc/init.d/calvados_patch_lxc_iso.sh "/" ${LVG_NAME} \
                         ${LV_ID} "hostos"
            #unmount /proc and other FS passed to ${PARTMNT}
            umount_efi_and_proc ${PARTMNT}
            umount_pd_fs ${PARTMNT}
            # Allow platform specific binary copies.
            #
            declare -F pd_pxe_copy_bake_binaries &>/dev/null
            if [ $? -eq 0 ]
            then
                step_log_console "Copy required binaries for pxe environment"
                pd_pxe_copy_bake_binaries ${PARTMNT} true
            fi
            # pd_punch_wd is necessary only for some platforms
            declare -F pd_punch_wd &>/dev/null && pd_punch_wd
        else 
            /etc/init.d/calvados_patch_lxc_iso.sh ${PARTMNT} ${LVG_NAME} \
                         ${LV_ID} "hostos"
        fi
    else
        # Allow non-LXC platform specific patching for HostOS boot partition
        declare -F pd_pxe_install_patch_hostos_boot &>/dev/null
        if [ $? -eq 0 ]; then
            step_log_console "Patch the hostos ISO for PD"

            LV_ID=`echo -n ${HOST_LV_NAME} | tail -c -1`
            if [ "${PLATFORM}" == "panini" ] #for NCS6K
            then
                step_log_console "pxe_install: install_host_iso: NCS6K EXR : " ${PLATFORM}
                /etc/init.d/calvados_patch_lxc_iso.sh ${PARTMNT} ${LVG_NAME} \
                         ${LV_ID} "hostos"
            else
                pd_pxe_install_patch_hostos_boot ${PARTMNT} ${LVG_NAME} \
                    ${LV_ID}
            fi
        fi
    fi

    run_depmod ${PARTMNT}
    sync
    sync
    SIZE=$(du -hs --exclude="/proc" --exclude="/sys" ${PARTMNT} | awk '{print $1}')

    if (( $VXR_SIM_MODE )); then

      step_log_file "Cleaning boot drive"

      mkdir -p /tmp/vda1
      mount /dev/vda1 /tmp/vda1
      cd /tmp/vda1
      rm -rf bin dev init lib iso libexec opt root sys usb
      rm -rf etc lib64 proc sbin usr
      cd /
      sync 
      sync
  
      umount /tmp/vda1
      rm -rf /tmp/vda1
    fi 

    if [ -d  ${HOSTRPMS} ]; then
       if [[ ! ${IsoType} < ${PKG_VER} ]]; then
           step_log_file "Installing additional rpms on Host"
           if [ "${PLATFORM}" == "panini" ]; then
               ${RPM_SEQUENCE_N_INSTALL_SCRIPT} --repo ${HOSTRPMS} --root ${PARTMNT} \
                                    --cardtype ${BOARDTYPE} --plat "ncs6k" 
           else
               ${RPM_SEQUENCE_N_INSTALL_SCRIPT} --repo ${HOSTRPMS} --root ${PARTMNT} \
                                    --cardtype ${BOARDTYPE} --plat ${PLATFORM} 
           fi
           if (( $? != 0 )); then
               step_log_file "Errored Installing additional rpms on Host"
           fi
           cp /root/xtra_rpm_install.log ${PARTMNT}/root/giso_rpm_install.log
           #cp ${GISO_SUMMARY} ${PARTMNT}/root/
       fi
    fi
    if [ -f ${GISO_SUMMARY} ];
    then
           cp ${GISO_SUMMARY} ${PARTMNT}/root/
    fi

    if [ -f ${GISOMNT}${GISO_YML_FILE} ]; then
        cp ${GISOMNT}${GISO_YML_FILE} ${GISO_YML_FILE}
    fi

    [ -d "${PARTMNT}" ] && umount ${PARTMNT}
}



function get_packages_to_be_installed {
    local exr_support=${1}
    shift
    local -a packages=("$@")

    unset packages_to_be_installed
    if [ "${exr_support}" == "yes" ]; then

        for file in ${packages[@]}
        do
            packages_to_be_installed[${#packages_to_be_installed[*]}]=${file}
        done
    else
    for file in ${packages[@]}
    do
        is_rp_package=0
        is_cc_package=0    
        is_lc_package=0
        is_sc_package=0
        is_xc_package=0
        is_fc_package=0
        is_common_package=0

        if [[ ${file} == *.all-* ]] ; then
            is_common_package=1;
        else
            if [[ ${file} == *.rp[-,_]* ]] ; then
                is_rp_package=1;
            fi

            if [[ ${file} == *.cc[-,_]* ]] ; then
                is_cc_package=1;
            fi

            if [[ ${file} == *[.,_]sc* ]] ; then
                is_sc_package=1;
            fi

            if [[ ${file} == *[.,_]lc* ]] ; then
                is_lc_package=1;
            fi

            if [[ ${file} == *[.,_]xc* ]] ; then
                is_xc_package=1;
            fi
            
            if [[ ${file} == *[.,_]fc* ]] ; then
                is_fc_package=1;
            fi
        fi

        if [ "$is_rp_package" -ne "1" -a                                      \
             "$is_cc_package" -ne "1" -a                                      \
             "$is_sc_package" -ne "1" -a                                      \
             "$is_lc_package" -ne "1" -a                                      \
             "$is_xc_package" -ne "1" -a                                      \
             "$is_fc_package" -ne "1" -a                                      \
             "$is_common_package" -ne "1" ]; then
            step_log_file "Error! Pkg ${file} is not applicable to any card"
        fi

        case ${BOARDTYPE} in
            CC)
               if [ "$is_cc_package" -eq "1" -o "$is_common_package" -eq "1" ]; then
                   packages_to_be_installed[${#packages_to_be_installed[*]}]=${file}
               fi
               ;;    
            LC)
               if [ "$is_lc_package" -eq "1" -o "$is_common_package" -eq "1" ]; then
                   packages_to_be_installed[${#packages_to_be_installed[*]}]=${file}
               fi
               ;;
            SC)
               if [ "$is_sc_package" -eq "1" -o "$is_common_package" -eq "1" ]; then
                   packages_to_be_installed[${#packages_to_be_installed[*]}]=${file}
               fi
               ;;
            XC)
               if [ "$is_xc_package" -eq "1" -o "$is_common_package" -eq "1" ]; then
                   packages_to_be_installed[${#packages_to_be_installed[*]}]=${file}
               fi
               ;;
            FC)
               if [ "$is_fc_package" -eq "1" -o "$is_common_package" -eq "1" ]; then
                   packages_to_be_installed[${#packages_to_be_installed[*]}]=${file}
               fi
               ;;
            *)
              # default is RP 
               if [ "$is_rp_package" -eq "1" -o "$is_common_package" -eq "1" ]; then
                   packages_to_be_installed[${#packages_to_be_installed[*]}]=${file}
               fi
               ;;
        esac
    done
    fi
}

function install_calvados_iso {
    local dev_name=${1}
    local dev=/dev/${dev_name}
    local device_part=${2}
    local -a packages

    source "/etc/init.d/pi_grub_and_menu_lst_update.sh"
    source "/etc/init.d/mount_pd_fs.sh"

    # Mount src iso
    step_log_file "Install sysadmin mount src iso ${ISOMNT}"
    mount -r -o loop /iso/$ADMIN_ISO ${ISOMNT}
    ret_code=$?
    if [ $ret_code -ne 0 ]; then
      declare -F pd_notify_sw_fatal_event &>/dev/null && \
          pd_notify_sw_fatal_event "Mounting of /iso/$ADMIN_ISO to ${ISOMNT} failed with error code $ret_code"
      exitclean
    fi
    
    # Mount dest partition
    step_log_file "Install sysadmin mount dest partition ${dev}${device_part}"
    mount -o discard ${dev}${device_part} ${PARTMNT} -o offset=${3},sizelimit=${4}
    ret_code=$?
    if [ $ret_code -ne 0 ]; then
      declare -F pd_notify_sw_fatal_event &>/dev/null && \
          pd_notify_sw_fatal_event "Mounting of ${dev}${device_part} to ${PARTMNT} (offset=${3},sizelimit=${4}) failed with error code $ret_code"
      exitclean
    fi
    # Remove old files, maintained across mkfs
    step_log_file "Install sysadmin clean ${PARTMNT}"
    if [ ! -z ${PARTMNT} ]; then
      rm -fr ${PARTMNT}/* >&107 2>&1
    fi
   

    #if sim, let sysadmin-vm know for host install
    if (( $SIMULATION )); then
        mkdir -p ${PARTMNT}/iso
        echo "This is sim marker file for sysadmin-vm." >${PARTMNT}/iso/sim.txt
    fi

    step_log_file "Install sysadmin copy iso name file"
    if [ -e $ISO_NAME_FILE ]; then
         cp /$ISO_NAME_FILE ${PARTMNT}/$ISO_NAME_FILE
    fi

    # maintain status of data partititon in sysadmin-vm
    if [ -f /iso/data_part.txt ]; then
        mkdir -p ${PARTMNT}/iso
        cp /iso/data_part.txt ${PARTMNT}/iso/data_part.txt
    fi 

    cd ${PARTMNT}
    if [ -d ${ISOMNT}/rpm/calvados ]; then
        step_log_file "Install sysadmin copy rpms"
        mkdir -p ${PARTMNT}/rpm/calvados
        cp ${ISOMNT}/rpm/calvados/* ${PARTMNT}/rpm/calvados/
    fi
    step_log_file "Install sysadmin initrd.img"
    zcat ${ISOMNT}/boot/initrd.img | cpio -d -i >&107 2>&1
    ret_code=$?
    if [ $ret_code -ne 0 ]; then
      declare -F pd_notify_error_event &>/dev/null && \
          pd_notify_error_event "zcat of ${ISOMNT}/boot/initrd.img  returned error code $ret_code"
    fi

    #cp iso_info.txt to partition for future references
    cp ${ISOMNT}/${ISO_INFO_FILE} ${PARTMNT}

    # Get card instance information and add it in the mount
    declare -f pd_get_card_inst >&107 2>&1
    if [ "$?" -eq 0 ]; then
        local CARD_INST
        pd_get_card_inst CARD_INST
        echo "$BOARDTYPE $CARD_INST" > $CARD_INST_FILE
    fi


    # cp -r ${ISOMNT}/* ${PARTMNT}
    
    # delete any of the cached RPM db files that come with the prebuilt RPM db
    # in the initrd.img
    rm -f ./var/lib/rpm/__db.*

    # For now, we check iso_info.txt for new packaging architecture.
    # As per this new packaging architecture, we will have a single
    # rpm per package instead of per card type rpm.
    exr_support=$(check_exr_support ${ISOMNT})
    chroot_support=$(check_chroot_support ${ISOMNT})
    IsoType=$(check_iso_type ${ISOMNT})

    # Obtain board type to decide RP or LC based installation
    vxr_sim_mode_check
    if (( $VXR_SIM_MODE )); then
      install_sim_tools

      if [ -f /opt/simulation/install.tar.bz2 ]; then
          step_log_console "SIM Platform:  Install install.tar.bz2 in sysadmin-vm Partition."
          cp /opt/simulation/install.tar.bz2 ${PARTMNT}/iso/install.tar.bz2
      fi

      step_log_console "SIM: Generating eth1_mac.txt for sysadmin-vm"
     
      /sbin/ifconfig -a | grep eth1 | awk '{print $5}' | sed "s/://g" | sed "s/^/0x/"  > ${PARTMNT}/iso/eth1_mac.txt

    else
      step_log_console "sysadmin-vm: $BOARDTYPE based installation"
    fi
    
    # Obtain the list of packages to be installed
    packages=(rpm/calvados/*)
    get_packages_to_be_installed "${exr_support}" "${packages[@]}"
    
    mount_efi_and_proc ${PARTMNT}
    mount_pd_fs ${PARTMNT}
    for file in ${packages_to_be_installed[@]}
    do
        if [ "${exr_support}" == "yes" ];then
            if [[ ${file} == *thirdparty-libexec* ]] || 
               [[ ${file} == *thirdparty-kernel*  ]] ||
               [[ ${file} == *thirdparty-qemu*    ]] ; then
                step_log_file $"Ignoring Install RPM ${file} of size "
            else
                SUPP_CARDS=()
                excl_card=()
                #SUPPCARDS holds a list of supported cards by this rpm; comma separated.
                supp_card=`chroot ${PARTMNT} rpm -qp --qf '%{SUPPCARDS}\n' ${file}`
                SUPP_CARDS=(${supp_card//,/ })
                board=`echo ${BOARDTYPE} | tr "a-z" "A-Z"`
                for card in ${SUPP_CARDS[@]}
                do
                    card=`echo $card | tr "a-z" "A-Z"`
                    if [ $card != ${BOARDTYPE} -a  $card != ALL ]; then
                        excl_card[${#excl_card[*]}]=`echo $card | tr "A-Z" "a-z"`
                    fi
                done
                declare -a excl
                excl=()
                prefix=`chroot ${PARTMNT} rpm -qp --qf '%{PREFIXES}\n' ${file}`
                pkg_name=`chroot ${PARTMNT} rpm -qp --qf '%{NAME}\n' ${file}`
                version=`chroot ${PARTMNT} rpm -qp --qf '%{VERSION}\n' ${file}`
                release=`chroot ${PARTMNT} rpm -qp --qf '%{RELEASE}\n' ${file}`
                excl_path=${prefix}/${pkg_name}-${version}-${release}
                for card in ${excl_card[@]}
                do
                    excl[${#excl[*]}]=--excludepath=${excl_path}/${card}
                done
                exclude=`echo ${excl[@]}`
                step_log_file $"Install RPM with ${exclude}"
                STARTRPM=$(date +%s)
                chroot ${PARTMNT} rpm -i ${exclude} --nodeps ${file} >&107 2>&1
                ENDRPM=$(date +%s)
                DIFFRPM=$(( $ENDRPM - $STARTRPM ))
                SIZERPM=$(ls -lh ${file}| awk '{ print $5 }')
                step_log_file $"Install RPM ${file} of size ${SIZERPM} took ${DIFFRPM} sec"
            fi
       else
           case ${file} in
               *.rpm) 
                 if [[ ${file} == *thirdparty* ]] ; then
                     STARTRPM=$(date +%s)
                     chroot ${PARTMNT} rpm -i --nodeps ${file} >&107 2>&1
                     ENDRPM=$(date +%s)
                     DIFFRPM=$(( $ENDRPM - $STARTRPM ))
                     SIZERPM=$(ls -lh ${file}| awk '{ print $5 }')
                     step_log_file $"Install RPM ${file} of size ${SIZERPM} took ${DIFFRPM} sec"
                 fi
                 ;;
           esac
        fi
    done

    for file in ${packages_to_be_installed[@]}
    do
       case ${file} in
           *.rpm)
            if [ "${exr_support}" != "yes" ]; then
                if [[ ${file} != *thirdparty* ]] ; then
                     STARTRPM=$(date +%s)
                     chroot ${PARTMNT} rpm -i --nodeps ${file} >&107 2>&1
                     ENDRPM=$(date +%s)
                     DIFFRPM=$(( $ENDRPM - $STARTRPM ))
                     SIZERPM=$(ls -lh ${file}| awk '{ print $5 }')
                     step_log_file $"Install RPM ${file} of size ${SIZERPM} took ${DIFFRPM} sec"
                fi
            fi
            ;;
       esac
    done
    umount_efi_and_proc ${PARTMNT}
    umount_pd_fs ${PARTMNT}

    if [ -d ${CALRPMS} ]; then
       if [[ ! ${IsoType} < ${PKG_VER} ]]; then
           step_log_file "Installing additional rpms on calvados"
           if [ "${PLATFORM}" == "panini" ]; then
               ${RPM_SEQUENCE_N_INSTALL_SCRIPT} --repo ${CALRPMS} --root ${PARTMNT}  \
                                     --cardtype ${BOARDTYPE} --plat "ncs6k" 
           else
               ${RPM_SEQUENCE_N_INSTALL_SCRIPT} --repo ${CALRPMS} --root ${PARTMNT}  \
                                     --cardtype ${BOARDTYPE} --plat ${PLATFORM} 
           fi
           if (( $? != 0 )); then
               step_log_file "Errored Installing additional rpms on calvados"
           fi
           cp /root/xtra_rpm_install.log ${PARTMNT}/root/giso_rpm_install.log
           #cp ${GISO_SUMMARY} ${PARTMNT}/root/
       fi
    fi
    if [ -f ${GISO_SUMMARY} ];
    then
           cp ${GISO_SUMMARY} ${PARTMNT}/root/
    fi

    if [ -f /usr/bin/calv_setup_ldpath.sh ]; then
         step_log_file "Install sysadmin symlink setup"
         STARTRPM=$(date +%s)
         arg="`echo ${excl_card[@]}`"
         /usr/bin/calv_setup_ldpath.sh ${PARTMNT} "$arg" > /tmp/calv_setup_ldpath.log 2>&1
         cp /tmp/calv_setup_ldpath.log ${PARTMNT}/root/
         ENDRPM=$(date +%s)
         DIFFRPM=$(( $ENDRPM - $STARTRPM ))
         step_log_file "sysadmin-vm symlink setup took ${DIFFRPM} sec"
    fi

    step_log_file "Install sysadmin dir setup"
    rm -rf rpm/calvados >&107 2>&1

    mkdir -p ${PARTMNT}/proc ${PARTMNT}/sys ${PARTMNT}/root/.ssh
    cp ${SSHKEY_DIR}/calvados/* ${PARTMNT}/root/.ssh/
    mkdir -p ${PARTMNT}/tmp ${PARTMNT}/dev 
    chmod -R 700 ${PARTMNT}/root
    chown -R root:root ${PARTMNT}/*
    cd / >&107 2>&1

    # Copy the platform board specific install configuration from ramfs 
    # to sysadmin boot partition
    cp $TEMP_INSTALL_FILE_NAME ${PARTMNT}/${PD_BOARD_INST_CONF}
    if (( $? != 0 )); then
        step_log_file $"Copying ${PD_BOARD_INST_CONF} failed"
    fi

    step_log_file "Install sysadmin grub setup ${dev}${device_part}"
    #fix menu.lst script
    pi_update_sysadmin_menu_lst ${PARTMNT} ${dev}${device_part} "1"

    if (( $VXR_SIM_MODE )); then
            source ${PARTMNT}/opt/simulation/install/patches/calvados_begin.sh
            remove_sim_tools
    else
            rm -rf source ${PARTMNT}/opt/simulation >&107 2>&1

            rmdir ${PARTMNT}/opt/simulation >&107 2>&1
    fi
    
    if [ "$VIRT_METHOD" = "lxc" ]; then
        # Allow platform specific patching for calvados boot partition
        LV_ID=`echo -n ${CALVADOS_LV_NAME} | tail -c -1`
        if [ "${chroot_support}" == "yes" ] ; then
            #bind mount /proc and PD defined FS to ${PARTMNT}
            mount_efi_and_proc ${PARTMNT}
            mount_pd_fs ${PARTMNT}
            chroot ${PARTMNT} /etc/init.d/calvados_patch_lxc_iso.sh "/" ${LVG_NAME} \
                         ${LV_ID} "sysadmin-vm"
            #unmount /proc and other FS passed to ${PARTMNT}
            umount_efi_and_proc ${PARTMNT}
            umount_pd_fs ${PARTMNT}
        else
            /etc/init.d/calvados_patch_lxc_iso.sh ${PARTMNT} ${LVG_NAME} \
                         ${LV_ID} "sysadmin-vm"
        fi
    else
        # Allow non-LXC  platform specific patching for SysAdmin VM boot
        #  partition
        declare -F pd_pxe_install_patch_sysadmin_vm_boot &>/dev/null
        if [ $? -eq 0 ]; then
            step_log_console "Patch the SysAdmin VM ISO for PD"

            LV_ID=`echo -n ${CALVADOS_LV_NAME} | tail -c -1`
            if [ "${PLATFORM}" == "panini" ] #for NCS6K
            then
                step_log_console "pxe_install: install_calvados_iso: NCS6K EXR : " ${PLATFORM}
                /etc/init.d/calvados_patch_lxc_iso.sh ${PARTMNT} ${LVG_NAME} \
                          ${LV_ID} "sysadmin-vm"
            else
                pd_pxe_install_patch_sysadmin_vm_boot ${PARTMNT} ${LVG_NAME} \
                    ${LV_ID}
            fi
       fi
    fi

    run_depmod ${PARTMNT}
    sync
    sync
    SIZE=$(du -hs --exclude="/proc" --exclude="/sys" ${PARTMNT} | awk '{print $1}')
    [ -d "${PARTMNT}" ] && umount ${PARTMNT}
    [ -d "${ISOMNT}" ] && umount ${ISOMNT} >&107 2>&1
}


trap exitclean SIGINT SIGTERM
usage() {
    step_log_console $"Usage: $0 {start|stop}"
}

# Copy Host, calvados and XR ISOs to '/iso' dir in repository disk
function copy_xr_iso_to_repository {
    local dev_name=${1}
    local dev=/dev/${dev_name}
    
    # Mount dest partition
    step_log_file "Copy iso to repository ${PARTMNT}"
    mount -o discard ${dev} ${PARTMNT} -o offset=${2},sizelimit=${3}
    ret_code=$?
    if [ $reg_code -ne 0 ]; then
      declare -F pd_notify_sw_fatal_event &>/dev/null && \
          pd_notify_sw_fatal_event "Mounting of ${dev} to ${PARTMNT} failed with error code $ret_code"
      exitclean
    fi
    rm -rf ${PARTMNT}/*
    
    if [ -f /iso/$HOST_ISO ]; then
       step_log_console $"Copying /iso/$HOST_ISO to repository /iso directory"
       mkdir -p ${PARTMNT}/iso
       cp -p  /iso/$HOST_ISO ${PARTMNT}/iso/.
       if [ -d ${HOSTRPMS} ]; then
           mkdir -p ${PARTMNT}${INST_GISO_REPO_CAL}
           cp ${HOSTRPMS}/*rpm ${PARTMNT}${INST_GISO_REPO_CAL}.
           step_log_console "Copy Host rpms to repository"
       fi
    fi

    if [ -f /iso/$ADMIN_ISO ]; then
       step_log_console $"Copying /iso/$ADMIN_ISO to repository /iso directory"
       mkdir -p ${PARTMNT}/iso
       cp -p  /iso/$ADMIN_ISO ${PARTMNT}/iso/.
       if [ -d ${CALRPMS} ]; then
           mkdir -p ${PARTMNT}${INST_GISO_REPO_CAL}
           cp ${CALRPMS}/*rpm ${PARTMNT}${INST_GISO_REPO_CAL}.
           step_log_console "Copy Sysadmin rpms to repository"

           # if Hostos smu is present in the calvados_rpms of GISO
           # then copy base rpm from /rpm/calvados to install_repo
           # hostos smu can't be deactivated so base rpm are required 
           # for re-activating to move to older version  
           hostos_smu_array=( $(find ${CALRPMS} -type f | grep -e ${SYSADMIN_HOSTOS} | cut -d/ -f2) )
           len_hostos_smu_array=${#hostos_smu_array[@]}
           if [[ $len_hostos_smu_array -ge 1 ]]; then
               step_log_file "HostOS smu present in ISO"
               for i in ${hostos_smu_array[@]}; do
                   step_log_file "$i"
               done
               cp /rpm/calvados/*rpm ${PARTMNT}${INST_GISO_REPO_CAL}.
               step_log_console "Copy HostOs rpms to repository"
           fi
       fi
    fi

    if [ -d ${XRRPMS} ]; then
        mkdir -p  ${PARTMNT}${INST_GISO_REPO_XR}
        step_log_console "Copy XR rpms to repository"
        cp ${XRRPMS}/*rpm ${PARTMNT}${INST_GISO_REPO_XR}
    fi
    if [ -f ${GISOSCRIPT} ]; then
        mkdir -p  ${PARTMNT}${INST_GISO_REPO}
        step_log_console "Copy Giso script to repository"
        cp ${GISOSCRIPT} ${PARTMNT}${INST_GISO_REPO}
    fi
    if [ -f ${ZTPINIFILE} ]; then
        mkdir -p  ${PARTMNT}${INST_GISO_REPO}
        step_log_console "Copy Giso ztp ini file to repository"
        cp ${ZTPINIFILE} ${PARTMNT}${INST_GISO_REPO}
    fi
     
    if [ -f ${DEFAULT_ROUTER_CONFIG} ]; then
        mkdir -p ${PARTMNT}${INST_GISO_REPO}
        step_log_console "Copy XR Config to repository"
        cp ${DEFAULT_ROUTER_CONFIG} ${PARTMNT}${INST_GISO_REPO}
    fi

    if [ -f ${GISO_INFO_TXT} ]; then
        mkdir -p ${PARTMNT}${INST_GISO_REPO}
        step_log_console "Copy giso_info.txt to repository"
        cp ${GISO_INFO_TXT} ${PARTMNT}${INST_GISO_REPO}
    fi


    if [ -f ${SP_INFO_TXT} ]; then
        mkdir -p ${PARTMNT}${INST_GISO_REPO}
        step_log_console "Copy sp_info.txt to repository"
        cp ${SP_INFO_TXT} ${PARTMNT}${INST_GISO_REPO}
    fi

    if [ -f /iso/$SDR_ISO ]; then
       step_log_console $"Copying /iso/$SDR_ISO to repository /iso directory"
       mkdir -p ${PARTMNT}/iso
       cp -p  /iso/$SDR_ISO ${PARTMNT}/iso/.
    fi

    run_depmod ${PARTMNT}

    # Call sync to make sure the raw device is in sync with 
    # any bufferring in filesystems.
    sync
    sync
   
    [ -d "${PARTMNT}" ] && umount ${PARTMNT}
}

# Create partitions (sim case)
# First parameter is disk (ex: /dev/vda)
# If LVM size is not specified, calculate it from total disk size. 
# One complication is, fdisk reports total size in MB (1000), but parameters 
# given for creating partitions are in MiB (1024). So, need slight conversion.
function create_part_file_disk {
    local lvm_part_size_value=${DISK_HOST_DATA_PART_SIZE}
    if [ -z ${lvm_part_size_value} ]; then 
        get_disk_size ${1}
        if [ -z ${DISK_SIZE} ]; then
            step_log_console "ERROR! No install boot device found, exiting..."
            # If we cannot find disk, it could be sign of SSD to about
            # going bad. A reimage retry sometime can recover from this
            # failure condition, but this is a sign that will tell
            # user that this failure could become permanent.
            declare -F pd_notify_sw_fatal_event &>/dev/null && \
                pd_notify_hw_fatal_event "No install boot device found for partitioning, exiting installation"
            declare -F pd_notify_hw_fatal_event &>/dev/null && \
                pd_notify_sw_fatal_event "No valid boot device found in ${1}"
            exit 0
        fi
        let disk_size=${DISK_SIZE}*1000000
        let disk_size/=1048576
        let lvm_part_size_value=$disk_size-$DISK_HOST_BOOT_PART_SIZE-$DISK_CALVADOS_REPO_SIZE-$DISK_HOST_EFI_PART_SIZE
        #
        # Checking for App Host Enablement and adjusting Disk Size accordingly
        #       
        declare -F app_host_with_separate_pv >&107 2>&1 && app_host_with_separate_pv
        if [[ $? -eq 0 ]]; then
            let lvm_part_size_value=$disk_size-$DISK_HOST_BOOT_PART_SIZE-$DISK_CALVADOS_REPO_SIZE-$DISK_HOST_EFI_PART_SIZE-$DISK_TP_VOL_PART_SIZE
        fi      
        local lvm_part_size=${lvm_part_size_value}M
        step_log_file "Making LVM as large as possible ($lvm_part_size)"
    fi
    step_log_console "Creating partitions: BOOT=${DISK_HOST_BOOT_PART_SIZE}MB, LVM=${lvm_part_size_value}MB, EFI=${DISK_HOST_EFI_PART_SIZE}MB"

# Checking App-Host Enablement and adjusting Partitioning accordingly
# Always Keep the App Host Partition at the end - when you add additional partitions as well
# App Hosting Volume Creation is designed to take the last partition available in the disk
# to avoid complexity such as: any platform needs partition need not do anything special
# if app hosting partition is at the last and need not worry about legacy/non-legacy Boottype

declare -F app_host_with_separate_pv >&107 2>&1 && app_host_with_separate_pv
if [[ $? -eq 0 ]]; then
# Always Keep the App Host Partition at the end
# Increment the below variable if you add new partitions above App_Host 
PART_NUM_TP=4
echo "

n
p
1

+${DISK_HOST_BOOT_PART_SIZE}M
n
p
2

+${DISK_CALVADOS_REPO_SIZE}M
n
p
3

+${lvm_part_size_value}M
n
p
4

+${DISK_TP_VOL_PART_SIZE}M
n
e


w" > /tmp/fdisk.input
else
echo "

n
p
1

+${DISK_HOST_BOOT_PART_SIZE}M
n
p
2

+${DISK_CALVADOS_REPO_SIZE}M
n
p
3

+${lvm_part_size_value}M
n
e


w" > /tmp/fdisk.input
fi
}
readonly -f create_part_file_disk

function create_ssh_keys {
    step_log_file "Create ssh keys"
    mkdir -p ${SSHKEY_DIR}/host ${SSHKEY_DIR}/calvados 
    ssh-keygen -q -t rsa -f ${SSHKEY_DIR}/calvados/id_rsa -C root@calvados-vm -N '' >&107 2>&1
    ssh-keygen -q -t rsa -f ${SSHKEY_DIR}/host/id_rsa -C root@host -N '' >&107 2>&1
    cat ${SSHKEY_DIR}/calvados/id_rsa.pub >> ${SSHKEY_DIR}/host/authorized_keys
    cat ${SSHKEY_DIR}/calvados/id_rsa.pub >> ${SSHKEY_DIR}/calvados/authorized_keys
    cat ${SSHKEY_DIR}/host/id_rsa.pub >> ${SSHKEY_DIR}/calvados/authorized_keys
}
readonly -f create_ssh_keys

function copy_iso_files {
      declare -F pd_copy_iso_files >&107 2>&1
      if [[ $? == 0 ]]; then
        return
      fi
      # copy iso file if pxe boot is done with stripped initrd image
      ipaddress=`cat /proc/cmdline | grep ipaddress=`
      if [ -n "$ipaddress" ]; then
        #Enable eth-pf1 and configure the interface using DHCP IP
        vlan=`cat /proc/cmdline | sed 's/^.*vlan=//' | cut -d" " -f1`
        if [ "${PLATFORM}" == "panini" ] #for NCS6K
        then
            step_log_console "pxe_install: copy_iso_files: NCS6K EXR : " ${PLATFORM}
            ipaddress=`cat /proc/cmdline | sed 's/^.*ipaddress=//' | cut -d" " -f1`
            netmask=`cat /proc/cmdline | sed 's/^.*mask=//' | cut -d" " -f1`
            gateway=`cat /proc/cmdline | sed 's/^.*gateway=//' | cut -d" " -f1`
        fi

        cd /etc/sysconfig/network-scripts/
        cp /etc/sysconfig/network-scripts/ifcfg-eth0 ./ifcfg-eth-pf1.$vlan
        sed -i -e "s;eth0;eth-pf1.$vlan;" ./ifcfg-eth-pf1.$vlan
        vconfig add eth-pf1 $vlan
        ifup eth-pf1.$vlan >&107 2>&1
        if [ "${PLATFORM}" == "panini" ] #for NCS6K
        then
            step_log_console "pxe_install: Up eth: NCS6K EXR : " ${PLATFORM}
            ifconfig eth-pf1.$vlan $ipaddress netmask $netmask up
            route add default gw $gateway
        fi
        tftpserver=`cat /proc/cmdline | sed 's/^.*tftpserver=//' | cut -d" " -f1`
        installiso=`cat /proc/cmdline | sed 's/^.*installiso=//' | cut -d" " -f1`
        step_log_console "Copying $installiso from tftpserver ${tftpserver}"
        tftp $tftpserver -c get $installiso /$installiso
        # Verify if we we successfully tftp the ISO file
        if [ ! -s /$installiso ]; then
          step_log_console "Error, failed to copy $installiso from tftpserver ${tftpserver}"
          declare -F pd_notify_sw_fatal_event &>/dev/null && \
              pd_notify_sw_fatal_event "Failed to copy $installiso from tftpserver ${tftpserver}"
          exitclean
        fi
        mount -r -o loop /$installiso ${ISOMNT}	    
        ret_code=$?
        if [ $ret_code -ne 0 ]; then
          declare -F pd_notify_sw_fatal_event &>/dev/null && \
              pd_notify_sw_fatal_event "Mounting of /$installiso to ${PARTMNT} failed with error code $ret_code"
          exitclean
        fi
        step_log_console "Mounted $installiso to $ISOMNT"
        verify_md5sum $ISOMNT || exitmd5sum_reboot ${installiso}
        step_log_console "$installiso md5sum verification succeeded"
        md5verified=true
        mkdir -p /installdir
        cd /installdir
        zcat ${ISOMNT}/boot/initrd.img | cpio -d -i >&107 2>&1
        ret_code=$?
        if [ $ret_code -ne 0 ]; then
          declare -F pd_notify_error_event &>/dev/null && \
              pd_notify_error_event "zcat of ${ISOMNT}/boot/initrd.img returned error code $ret_code"
        fi
        mkdir -p /iso
        step_log_console "Copying iso files from ${ISOMNT}"
        cp ./iso_name.txt /
        cp -r ./iso /
        step_log_console "Copying rpm files from ${ISOMNT}"
        cp -r ./rpm /
        cp ./boot/bzImage /boot/
        cp ./boot/initrd.img /boot/
        if [ -d "./boot/certs" ]; then
          cp -r ./boot/certs /boot/
        fi
        if [ -f "./boot/signature.initrd.img" ]; then
          cp -r ./boot/signature.initrd.img /boot/
          cp -r ./boot/grub2 /boot/
        fi

        if [ -d "./${HOSTRPMS}" ]; then
            mkdir -p /${HOSTRPMS}
            cp -fr ./${HOSTRPMS}* /${HOSTRPMS}
        fi
        if [ -d "./${CALRPMS}" ]; then
            mkdir -p /${CALRPMS}
            cp -fr ./${CALRPMS}* /${CALRPMS}
        fi
        if [ -d "./${XRRPMS}" ]; then
            mkdir -p /${XRRPMS}
            cp -fr ./${XRRPMS}* /${XRRPMS}
        fi

        if [ -f "./${GISOSCRIPT}" ]; then
            cp ./${GISOSCRIPT} /${GISOSCRIPT}
        fi
 
        if [ -f "./${ZTPINIFILE}" ]; then
            cp ./${ZTPINIFILE} /${ZTPINIFILE}
        fi

        if [ -f "./${DEFAULT_ROUTER_CONFIG}" ]; then
            cp ./${DEFAULT_ROUTER_CONFIG} /${DEFAULT_ROUTER_CONFIG}
        fi
 
        if [ -f "./${GISO_INFO_TXT}" ]; then
            cp ./${GISO_INFO_TXT} /${GISO_INFO_TXT}
        fi
 
        if [ -f "./${SP_INFO_TXT}" ]; then
           cp ./${SP_INFO_TXT} /${SP_INFO_TXT}
        fi
 
        if [ -f "./${GISO_SUMMARY}" ]; then
           cp ./${GISO_SUMMARY} /${GISO_SUMMARY}
        fi

        rm -rf /installdir
        umount ${ISOMNT}

        set_iso_file_name
        iso_path="/$installiso"
        copy_giso_files ${iso_path}
      fi
}

function create_tftpboot_dir {
    if [ $is_clean_disk1 -eq 1 ]; then
        avail_space=$(df ${mount_dir})
        step_log_console "${avail_space}"           
        step_log_console "Cleaning up ${mount_dir}"           
        rm -rf ${mount_dir}/*
    fi
    mkdir -p ${mount_dir}/tftpboot
    chmod 755 ${mount_dir}/tftpboot
    cd ${mount_dir}/tftpboot
    rm -rf ./*
}

function get_usbdev() {
     
    local dev="sdc"
    local parts=$(grep -c ${dev} /proc/partitions)
    if [ ${parts} -gt 1 ]; then
        # Partitioned disk. Return first partition
        echo "/dev/${dev}1"
    else
        # Unpartitioned disk. Return base disk
        echo "/dev/${dev}"
    fi
    return
}

function install_pxeboot_files {
    local mount_dir=/misc/disk1
    local dev=/dev/${1}
    local usbdev=$(get_usbdev)

    if [ "${PLATFORM}" == "fretta" ]; then
        # For Fretta, mount internal PXE dir on Host secondary disk LV
        local label=${DISK1_PV}-${DISK1_LV}${D1_HOST_LV}
        usbdev=${2}1
    else
        local label=${DISK1_PV}-${DISK1_LV}${D1_CVDS_LV1}
    fi
    local device=/dev/mapper/$label

    # Check if it is a PCI device
    local pci_disk_check=$(udevadm info -q path -n ${dev} 2>&107 \
                         | grep pci | grep -v usb)

    # Check usb device
    local usb_disk_check=$(udevadm info -q path -n ${usbdev} 2>&107 \
                             | grep pci | grep usb)

    if [ ! -z "$pci_disk_check" ]; then
      STARTPXE=$(date +%s)
      # Activate volume group on disk1
      activate_disk_one_vg

      fsck -fy ${device} >&107 2>&1
      /sbin/tune2fs -c 0 -i 0 ${device} >&107 2>&1
      mkdir -p ${mount_dir}
      rm -rf ${mount_dir}/*
      mount -o discard ${device} ${mount_dir} >&107 2>&1
      ipaddress=`cat /proc/cmdline | grep ipaddress=`
      is_clean_disk1=0  
      create_tftpboot_dir 
      isofile=system_image.iso
      if [ -L "$isofile" -a -z "$INSTREIMAGE" ]; then
          rm "`readlink -f $isofile`"
          rm -f $isofile
      fi
      iso_source=
      if [ -f /"$isofile" ]; then
          if [ "${IS_ISO_REQUIRED}" == "TRUE" ]; then
              cp /"$isofile" .
              iso_source="Golden ISO"
          fi
      elif [ -f /iso/system_image.iso ]; then
          if [ "${IS_ISO_REQUIRED}" == "TRUE" ]; then
              cp /iso/system_image.iso .
              if [ $? -ne 0 ]; then
                  step_log_console "Could not copy system_image.iso to ${mount_dir}"
                  is_clean_disk1=1  
                  create_tftpboot_dir 
                  step_log_console "Retrying to copy system_image.iso to ${mount_dir}"
                  cp /iso/system_image.iso .
              fi
          fi
          iso_source="PACKED"
      elif [ -n "$RECOVERYBOOT" ]; then
        declare -F pd_get_recovery_image &>/dev/null && pd_get_recovery_image "${BOARDTYPE}"
        iso_source="Recovery"
      elif [ -n "$ipaddress" ]; then
        tftpfile=`cat /proc/cmdline | sed 's/^.*tftpfile=//' | cut -d" " -f1`
        tftpserver=`cat /proc/cmdline|sed 's/^.*tftpserver=//'|cut -d" " -f1`
        if [ -f /$installiso ]; then
          cp /$installiso $isofile
        else
          ipaddress=`cat /proc/cmdline | sed 's/^.*ipaddress=//' | cut -d" " -f1`
          netmask=`cat /proc/cmdline | sed 's/^.*mask=//' | cut -d" " -f1`
          gateway=`cat /proc/cmdline | sed 's/^.*gateway=//' | cut -d" " -f1`
          tftpfile=`cat /proc/cmdline | sed 's/^.*tftpfile=//' | cut -d" " -f1`
          vlan=`cat /proc/cmdline | sed 's/^.*vlan=//' | cut -d" " -f1`
          vconfig add eth-pf1 $vlan
          ifconfig eth-pf1.$vlan $ipaddress netmask $netmask up
          route add default gw $gateway
          step_log_console "Copying $tftpfile from tftpserver ${tftpserver}"
          tftp $tftpserver -c get $tftpfile $isofile
          # Verify if we we successfully tftp the ISO file
          if [ ! -s $isofile ]; then
               step_log_console "Error, failed to copy $tftpfile from tftpserver ${tftpserver}"
              declare -F pd_notify_sw_fatal_event &>/dev/null && \
              pd_notify_sw_fatal_event "Failed to copy $tftpfile from tftpserver ${tftpserver}"
          fi
        fi
        iso_source="tftpserver ${tftpserver}"
      else
        if [ "$BOARDTYPE" == "CC" -a "$bootmedia" == "disk" ]; then
          #this is inst_reimage CVM case for CC
          iso_source="DISK"
          chmod 544 $isofile
        elif [ ! -z "$usb_disk_check" ]; then
          usbmnt=$(mktemp -d /tmp/usbdev.XXXXXX)
          mount $usbdev $usbmnt >&107 2>&1
          if [ -f "$usbmnt/EFI/boot/grub.cfg" ]; then
            usbgrub=$usbmnt/EFI/boot/grub.cfg
          elif [ -f "$usbmnt/boot/grub.cfg" ]; then
            usbgrub=$usbmnt/boot/grub.cfg
          fi
          if [ -n "$usbgrub" ]; then
            echo "Found grub file $usbgrub" >&107
            usbiso=$(grep loopback $usbgrub | head -1 | awk '{print $3}' | sed 's/\r$//')
            echo "Looking for $usbiso on $usbmnt" >&107
            if [ -f "$usbmnt$usbiso" ]; then
              usbiso=$usbmnt$usbiso
              echo "Found iso image at $usbiso" >&107
            else
              usbiso=
              echo "Cannot find iso image from grub.cfg" >&107
            fi
          fi
          if [ -z "$usbiso" ]; then
            if [ -f "$(echo $usbmnt/boot/*.iso*)" ]; then
              usbiso=$(echo $usbmnt/boot/*.iso*)
              echo "Found iso image at $usbiso" >&107
            elif [ -f "$(echo $usbmnt/*.iso*)" ]; then
              usbiso=$(echo $usbmnt/*.iso*)
              echo "Found iso image at $usbiso" >&107
            fi
          fi

          if [ -n "$usbiso" -a -f "$usbiso" ]; then
            cp $usbiso $isofile
          else
            step_log_console "Install ISO image not found"
            ( cd $usbmnt && ls -laR . ) >&107
            ( while true; do
                 cd $usbmnt && ls -laR .
                 printf "\nInstall aborted due to USB faiure\n"
                 printf "Cold reboot to retry USB reimage. If not working, replace USB to try again!\n"
                 sleep 10
              done
            ) &
            disown $!
            exit
          fi
          umount $usbmnt
          rmdir $usbmnt
          iso_source="USB"
        else
          #this is inst_reimage case
          iso_source="DISK"
          chmod 544 $isofile
        fi
      fi
     
      #Copy sp_info.txt file to /misc/disk1/tftpboot(.)
      if [ -f $SP_INFO_TXT ]; then
          cp $SP_INFO_TXT .
      fi
      
      cd ${mount_dir}/tftpboot
      if [ "${PLATFORM}" != "fretta" -a "${PLATFORM}" != "zermatt" ]; then
        # CSCuv58958 has added iPXE for fretta LC.
        # With this change, we don't require following files in /misc/disk1/tftpboot
        cp -r /boot/grub2/bootx64.efi grub.efi
        cp -r /boot/grub2/grub.cfg grub.cfg
      fi

      declare -F pd_update_grub_files >&107 2>&1 && \
          pd_update_grub_files ./

      if [ "${PLATFORM}" == "fretta" -a "${BOARDTYPE}" == "RP" ]; then
          cp -f /nbi-initrd/ncs5500-sysadmin-nbi-initrd.img ./sysadmin-nbi-initrd.img
          cp -f /opt/cisco/hostos/bin/ncs5500-sysadmin-image.nbi sc.nbi
          ln -f sc.nbi fm.nbi
          cp /etc/init.d/ipxe-lc.efi .
          cp /etc/init.d/ipxe-lc.ipxe.slot26 .
          cp /etc/init.d/ipxe-lc.ipxe.slot27 .
      fi

      if [ -f $isofile -a -z "$md5verified" ]; then
         mount -r -o loop $isofile $ISOMNT
         verify_md5sum $ISOMNT || exitmd5sum_reboot "tftpboot ${isofile}"
         umount $ISOMNT
         step_log_console "tftpboot $isofile md5sum verfication succeeded"
      fi

      if [ -z "$RECOVERYBOOT" ]; then
          declare -F pd_pxe_update_pxeboot_iso >&107 2>&1 && \
              pd_pxe_update_pxeboot_iso ./
      fi

      declare -F pd_pxe_update_pxeboot_grub >&107 2>&1 && \
          pd_pxe_update_pxeboot_grub ./

      if [ -n "$iso_source" ]; then
          mkdir -p boot
          cp /boot/bzImage boot/
          cp /boot/initrd.img boot/
          if [ -d "/boot/certs" ]; then
            cp -r /boot/certs boot/
          fi
          if [ -f "/boot/signature.initrd.img" ]; then
            cp -r /boot/signature.initrd.img boot/
          fi

          sed -i -e "/search.*/d" \
            -e "s;root=<P.*>;root=/dev/ram install=/dev/sda;" \
            -e "s;quiet;quiet installiso=$isofile;" \
            -e "s;recoveryboot ;;" \
            -e "s;from Disk;from PXE;" ./grub.cfg
          ENDPXE=$(date +%s)
          DIFFPXE=$(( $ENDPXE - $STARTPXE ))
          step_log_console "Copying Pxeboot files from ${iso_source} took ${DIFFPXE} seconds"
      fi
      cd /
      sync

      # remove the apprepo directory any if exists
      if [ "$BOARDTYPE" == "RP" ]; then
          if [ -d "${mount_dir}/apprepo-dont-delete" ]; then
              rm -fr "${mount_dir}/apprepo-dont-delete"
            fi
      fi

      umount ${mount_dir}
    else 
      # This is needed for debuggability 
      if [ "$BOARDTYPE" == "RP" ]; then
        ipaddress=`cat /proc/cmdline | grep ipaddress=`
        if [ -n "$ipaddress" ]; then
          tftpserver=`cat /proc/cmdline | sed 's/^.*tftpserver=//' | cut -d" " -f1`
          step_log_file "No disk1 to copy Pxeboot files from tftpserver ${tftpserver}"
          declare -F pd_notify_error_event &>/dev/null && \
            pd_notify_error_event "No disk1 to copy Pxeboot files from tftpserver ${tftpserver}"
        fi
      fi
    fi
}


function dhclient_setup {
cat <<EOF > /etc/dhclient-exit-hooks
#!/bin/bash
if [[ \${new_filename} = *"http"* ]]; then
  url=\${new_filename}
  echo Downloading file: \${url}
  /usr/bin/wget -q \${url} -O $1
elif [ -n "\${new_dhcp_server_identifier}" ]; then
  echo Downloading file: \${new_filename}
  tftp_path=\`echo \${new_filename} | sed "s/tftp:\/\///g" | sed "s/^[^\/]*\///g"\`
  echo tftp -v -m binary \${new_dhcp_server_identifier} -c get \${tftp_path}
  /usr/bin/tftp -v -m binary \${new_dhcp_server_identifier} -c get \${tftp_path} $1
fi
EOF
echo "timeout 20;" >> /etc/dhcp/dhclient.conf
}

function dhclient_conf {
  echo "send vendor-class-identifier \"cisco\";" >> /etc/dhcp/dhclient.conf;
  echo "send user-class \"iPXE\";" >> /etc/dhcp/dhclient.conf;
# TBD: Get SerialNumber of the box and store it in config
  echo "send dhcp-client-identifier CHANGE-ME-;" >> /etc/dhcp/dhclient.conf;
}

function copy_giso_files {
    local GISOMNT=/tmp/tmp_gisomnt
    local system_image=$1

    if [ -e $system_image ]; then
       mkdir -p ${GISOMNT}
       step_log_console "Starting of copy_giso_files function"
       mount -o loop ${system_image} ${GISOMNT} > /dev/null 2>&1
       if [[ ! $? -eq 0 ]]; then
           step_log_console "Error: Failed to mount ${system_image}"
           return
       fi
       verify_md5sum $GISOMNT || exitmd5sum_reboot ${system_image}
    else 
      step_log_console " giso files will be copied from /"
      GISOMNT="/"
    fi

    if [ -d ${GISOMNT}${HOSTRPMS} ]; then
        mkdir ${HOSTRPMS}
        cp -fr ${GISOMNT}${HOSTRPMS}* ${HOSTRPMS}
    fi

    if [ -d ${GISOMNT}${CALRPMS} ]; then
        mkdir ${CALRPMS}
        cp -fr ${GISOMNT}${CALRPMS}* ${CALRPMS}
    fi

    if [ -d ${GISOMNT}${XRRPMS} ]; then
        mkdir ${XRRPMS}
        cp -fr ${GISOMNT}${XRRPMS}* ${XRRPMS}
    fi

    if [ -f ${GISOMNT}${GISOSCRIPT} ]; then
        cp ${GISOMNT}${GISOSCRIPT} ${GISOSCRIPT}
    fi

    if [ -f ${GISOMNT}${ZTPINIFILE} ]; then
        cp ${GISOMNT}${ZTPINIFILE} ${ZTPINIFILE}
    fi

    if [ -f ${GISOMNT}${DEFAULT_ROUTER_CONFIG}  ]; then
        cp ${GISOMNT}${DEFAULT_ROUTER_CONFIG} ${DEFAULT_ROUTER_CONFIG}
    fi

    if [ -f ${GISOMNT}${GISO_INFO_TXT} ]; then
        cp ${GISOMNT}${GISO_INFO_TXT} ${GISO_INFO_TXT}
    fi

    if [ -f ${GISOMNT}${SP_INFO_TXT} ]; then
        cp ${GISOMNT}${SP_INFO_TXT} ${SP_INFO_TXT}
    fi
       
    if [ -f ${GISOMNT}${GISO_SUMMARY} ]; then
        cp ${GISOMNT}${GISO_SUMMARY} ${GISO_SUMMARY}
    fi
    if [ -e $system_image ]; then
        umount -l ${GISOMNT}
    fi
    step_log_console "Ending of copy_giso_files function"
}

function download_giso {
    local dev_mnt="/tmp/dev_mnt"
    local raw_boot_dev=""
    local boot_dev=""
    local ipxe=""
    local tftp=""
    local giso_disk=""
    local system_image="system_image.iso"

    mount -r -o loop /iso/$HOST_ISO ${ISOMNT} || exitclean
    local GIsoType=$(check_iso_type ${ISOMNT}) 
    umount ${ISOMNT} || exitclean
    step_log_console "Golden ISO type = $GIsoType and PKG_VER = $PKG_VER"

    if [[ ${GIsoType} < ${PKG_VER} ]]; then
        step_log_console "Detected non eXR platform. Exiting download_giso"
        return
    fi

    # fretta pd function
    declare -f pd_get_boot_mode >&107 2>&1
    if [ "$?" -eq 0 ]; then
        boot_dev=$(pd_get_boot_mode)
        if [ -z "${boot_dev}" ]; then
            step_log_console "Couldn't find boot id(iPXE/USB). Returning from download_giso"
            return
        else
            ipxe=$boot_dev
            declare -f pd_get_giso_disk >&107 2>&1
            if [ "$?" -eq 0 ]; then
                giso_disk=$(pd_get_giso_disk)
                ipxe="$(echo $boot_dev | grep PXE )"
                tftp="$(echo $boot_dev | grep TFTP )"
            fi

        fi
    else 
        #   Mount efivarfs. Run efibootmgr to findout if current boot is iPXE/USB
        mount -t efivarfs efivarfs /sys/firmware/efi/efivars > /dev/null 2>&1 
        local boot_id="$(efibootmgr | grep "BootCurrent:" | cut -d" " -f2 )"
        
        if [ -z "${boot_id}" ]; then
            step_log_console "Couldn't find boot id(iPXE/USB). Returning from download_giso"
            return 
        fi

        raw_boot_dev="$( efibootmgr | grep Boot${boot_id} | cut -d':' -f2 | cut -d' ' -f2- )"
        ipxe="$(echo $raw_boot_dev | grep PXE )"
        boot_dev="${raw_boot_dev// /_}"

        #Disk can be USB or local disk
        local dev_pattern="$( echo $raw_boot_dev | cut -d'_' -f1 | cut -d' ' -f1 | cut -d'/' -f1 | cut -d'\' -f1)"
        giso_disk="$(ls /dev/disk/by-id/ -l | grep ${dev_pattern} | cut -f2 -d">" | grep 1 | cut -d" " -f2)"
    fi 

    step_log_console "Current Boot: ${boot_dev} "

    if [ -n "${ipxe}" ]; then
        if [ ! -f /${system_image} ]; then
            # fretta pd
            if [ "${ipxe}" == "IPXE" ]; then
                declare -F pd_int_pxe_download_iso >&107 2>&1 && \
                    pd_int_pxe_download_iso ./
            else                     
                if [ "$PLATFORM" == "ncs5k" ] ; then
                 if [ -d /${HOSTRPMS} -o -d /${CALRPMS} -o -d /${XRRPMS} ]; then
                     step_log_console "Additional RPMS are embedded in initrd" 
                     return
                 fi
                 if [ -f /${DEFAULT_ROUTER_CONFIG} ]; then
                     step_log_console "router config is embeded in initrd" 
                     return
                 fi
                fi
                

                dhclient_setup /${system_image}
                dhclient_conf

                step_log_console "Loading igb driver"
                declare -F pd_modprobe_igb >& /dev/null && pd_modprobe_igb

                local eths=( $(ifconfig -a | grep "Link\ " | cut -d' ' -f1 ) )
                for eth in "${eths[@]}";
                do
                    ifconfig ${eth} up
                done
                sleep 5
                # Fetch ISO from PXE server.
                # Try on each interface.
                local dhc_try=0
                while [ ! \( -f /${system_image} -a `cat /${system_image} | wc -c` -gt 0 \) ]; do
                    for eth in "${eths[@]}"
                    do
                        step_log_console "Running dhclient over ${eth}"      
                        dhclient -1 ${eth}
                        if [ -f /${system_image} -a `cat /${system_image} | wc -c` -gt 0 ]; then
                            break 
                        fi
                    done

                    dhc_try=$((dhc_try+1))
                    if [ ${dhc_try} -gt 10 ]; then
                        step_log_console "Exceeded Max try. Failed to fetch ISO from PXE server" 
                        break
                    fi
                done
            fi
        fi
    elif [ -n "${tftp}" -a "${boot_dev}" == "TFTP" ]; then
        if [ ! -f /${system_image} ]; then
            declare -F pd_tftp_download_iso >&107 2>&1 && \
                pd_tftp_download_iso /${system_image}
        fi
    elif [ -n "${giso_disk}" ]; then
        mkdir -p ${dev_mnt}
        mount  -o loop "/dev/disk/by-id/${giso_disk}" ${dev_mnt}
        cp /dev/disk/by-id/${giso_disk} /${system_image}
    fi
    
    step_log_console "Going to copy GISO ISO files..."
    if [ -n "${giso_disk}" -o -n "${ipxe}" -o -n "{tftp}" ]; then
        copy_giso_files ${system_image}
        step_log_console "Copied GISO ISO files..."
        if [ ! -z ${giso_disk} ]; then
           rm -fr ${dev_mnt}
        fi
    fi
}

function download_iso {
    local ipaddress=`cat /proc/cmdline | sed 's/^.*ipaddress=//' | cut -d" " -f1`
    local netmask=`cat /proc/cmdline | sed 's/^.*mask=//' | cut -d" " -f1`
    local gateway=`cat /proc/cmdline | sed 's/^.*gateway=//' | cut -d" " -f1`
    local tftpfile=`cat /proc/cmdline | sed 's/^.*tftpfile=//' | cut -d" " -f1`
    local tftpserver=`cat /proc/cmdline|sed 's/^.*tftpserver=//'|cut -d" " -f1`
    local vlan=`cat /proc/cmdline | sed 's/^.*vlan=//' | cut -d" " -f1`
    local isofile=system_image.iso
    vconfig add eth-pf1 $vlan
    ifconfig eth-pf1.$vlan $ipaddress netmask $netmask up
    route add default gw $gateway
    step_log_console "Copying $tftpfile from tftpserver ${tftpserver}"
    tftp $tftpserver -c get $tftpfile $isofile
    # Verify if we we successfully tftp the ISO file
    if [ ! -s $isofile ]; then
        step_log_console "Error, failed to copy $tftpfile from tftpserver ${tftpserver}"
        declare -F pd_notify_sw_fatal_event &>/dev/null && \
        pd_notify_sw_fatal_event "Failed to copy $tftpfile from tftpserver ${tftpserver}"
    else
        copy_giso_files ${isofile}
    fi
}


#==============================================================
#                     Main script code
#==============================================================

#---Basic PI variables---
readonly PD_DYNAMIC="/etc/init.d/spirit_pd.sh"
readonly PI_GRUB_SCRIPT="/etc/init.d/pi_grub_and_menu_lst_update.sh"
readonly MOUNT_PD_SCRIPT="/etc/init.d/mount_pd_fs.sh"
readonly LOGGING_SCRIPT="/etc/init.d/logging_functions.sh"

# TEMP_INSTALL_FILE_NAME: This file would be in the ramfs at this stage,
# but would be copied to the host and calvados boot paritions after they
# are prepared. The file this copied will be used by install software
# during steady state install operation.
readonly TEMP_INSTALL_FILE_NAME="/tmp/pxe_install_temp.cfg"
readonly LOGDIR=/var/log
readonly LOGFILE0=host-install0.log
readonly LOGFILE=host-install.log
readonly LOGDIR2=/tmp/bakelog
readonly LOGFILE2=sysadmin-install.log
readonly THIS_SCRIPT=pxe_install.sh
readonly CARD_INST_FILE="./root/card_instances.txt"
readonly HOSTRPMS=/host_rpms/
readonly CALRPMS=/calvados_rpms/
readonly XRRPMS=/xr_rpms/
readonly GISOSCRIPT=/autorun
readonly ZTPINIFILE=/ztp.ini
readonly RPM_SEQUENCE_N_INSTALL_SCRIPT=/etc/init.d/sequence_n_install_rpms.py
readonly INST_GISO_REPO=/giso_repo/
readonly INST_GISO_REPO_XR=${INST_GISO_REPO}/xr_repo/
readonly INST_GISO_REPO_CAL=${INST_GISO_REPO}/cal_repo/
readonly INST_GISO_BOOT_FILE=${INST_GISO_REPO}/giso_boot.txt
readonly DEFAULT_ROUTER_CONFIG=/router.cfg
readonly GISO_SUMMARY=/giso_summary.txt
readonly GISO_YML_FILE=/iosxr_image_mdata.yml
readonly GISO_INFO_TXT=/giso_info.txt
readonly SP_INFO_TXT=/sp_info.txt
readonly SYSADMIN_HOSTOS="sysadmin-hostos"
USE_FINAL=0

#-------get logging functions--------------
if [ -f $LOGGING_SCRIPT ]; then
    step_log_console "Sourcing $LOGGING_SCRIPT PI script"
    . $LOGGING_SCRIPT
else
    step_log_console "Error, could not find $LOGGING_SCRIPT file which contains console/file logging functions."
    step_log_console "Exiting $THIS_SCRIPT installation script."
    declare -F pd_notify_sw_fatal_event &>/dev/null && \
        pd_notify_sw_fatal_event "Error, could not find $LOGGING_SCRIPT file which contains PD parameters for installation."
    exit 0
fi
redirect_output $LOGDIR2/$LOGFILE

# Source function library.
. /etc/init.d/functions
. /etc/init.d/disk-functions
. /etc/init.d/spirit-functions
. /etc/init.d/lv-functions

if [ -f /etc/init.d/pd-functions ]; then
    . /etc/init.d/pd-functions
    declare -F replace_calv_boot_strap_cfg >&107 2>&1 && replace_calv_boot_strap_cfg
    declare -F platform_log >&107 2>&1 && platform_log "PXE install"
fi

# Show that this script is starting
step_log_file "Starting to execute $THIS_SCRIPT"


#-------get PD variables--------------
if [ -f $PD_DYNAMIC ]; then
    step_log_file "Sourcing $PD_DYNAMIC PD script"
    . $PD_DYNAMIC
else
    step_log_console "Error, could not find $PD_DYNAMIC file which contains PD parameters for installation."
    step_log_console "Exiting $THIS_SCRIPT installation script."
    declare -F pd_notify_sw_fatal_event &>/dev/null && \
        pd_notify_sw_fatal_event "Error, could not find $PD_DYNAMIC file which contains PD parameters for installation."
    exit 0
fi

#-------get PI grub update functions--------------
if [ -f $PI_GRUB_SCRIPT ]; then
    step_log_console "Sourcing $PI_GRUB_SCRIPT PI script"
    . $PI_GRUB_SCRIPT
else
    step_log_console "Error, could not find $PI_GRUB_SCRIPT file which contains PD parameters for installation."
    step_log_console "Exiting $THIS_SCRIPT installation script."
    declare -F pd_notify_sw_fatal_event &>/dev/null && \
        pd_notify_sw_fatal_event "Error, could not find $PI_GRUB_SCRIPT file which contains PD parameters for installation."
    exit 0
fi

#-------get PI bind mount functions--------------
if [ -f $MOUNT_PD_SCRIPT ]; then
    step_log_console "Sourcing $MOUNT_PD_SCRIPT PI script"
    . $MOUNT_PD_SCRIPT
else
    step_log_console "Error, could not find $MOUNT_PD_SCRIPT file which contains PD parameters for installation."
    step_log_console "Exiting $THIS_SCRIPT installation script."
    declare -F pd_notify_sw_fatal_event &>/dev/null && \
        pd_notify_sw_fatal_event "Error, could not find $MOUNT_PD_SCRIPT file which contains PD parameters for installation."
    exit 0
fi
on_sim

# Get board type from PD function
get_board_type
step_log_file "Got board type $BOARDTYPE"


# Retrieves platform bootstrap variable and customizes install 
# specific variables based on the board type.

if [ -f $INSTALL_PARAMS_OVERRIDE ]; then
    readonly PI_CTYPE=UNKNOWN
    source $INSTALL_PARAMS_OVERRIDE $PD_BOOTSTRAP  \
           1 $BOARDTYPE $PI_CTYPE $TEMP_INSTALL_FILE_NAME
    if (( $? != 0 )); then
        step_log_file "Cannot read platform bootstrap cfg"
        declare -F pd_notify_sw_fatal_event &>/dev/null && \
            pd_notify_sw_fatal_event "Cannot read platform bootstrap cfg"
        exit 1 
    fi
fi

step_log_file "Simulation=$SIMULATION"

#---More PI variables, having some dependency on above PD definitions---

# Name of logical volume group. As in '/dev/panini_vol_grp/calvados_lv0'.
# It is unfortunate to have 'panini' in it. But now we have to live with it.
# We could rename to something more generic like 'volume_grp', but
# this would affect husdh, calvados install code, and would affect upgrade
# compatibility.
function set_lvm_param {
    LVG_NAME=panini_vol_grp
    PCI_DISK_NAME=pci_disk1

    CALVADOS_LV_NAME=calvados_lv0
    CALVADOS_LV_PART=${LVG_NAME}/${CALVADOS_LV_NAME}
    CALVADOS_LV_PART_BOOT=${LVG_NAME}/${CALVADOS_LV_NAME}

    CALVADOS_LV_NAME_DATA=calvados_data_lv0 
    CALVADOS_LV_PART_DATA=${LVG_NAME}/${CALVADOS_LV_NAME_DATA}

    CALVADOS_LV_NAME_REPO=calvados_repo_lv0 
    CALVADOS_LV_PART_REPO=${LVG_NAME}/${CALVADOS_LV_NAME_REPO}

    HOST_LV_NAME=host_lv0
    HOST_LV_PART_BOOT=${LVG_NAME}/${HOST_LV_NAME}
    HOST_LV_NAME_FULL="/dev/${LVG_NAME}/${HOST_LV_NAME}"

    HOST_LV_NAME_DATA_SCRATCH=host_data_scratch_lv0
    HOST_LV_NAME_DATA_CONFIG=host_data_config_lv0
    HOST_LV_NAME_DATA_LOG=host_data_log_lv0

    HOST_LV_NAME_DATA_SCRATCH_FULL=/dev/${LVG_NAME}/${HOST_LV_NAME_DATA_SCRATCH}
    HOST_LV_NAME_DATA_CONFIG_FULL=/dev/${LVG_NAME}/${HOST_LV_NAME_DATA_CONFIG}
    HOST_LV_NAME_DATA_LOG_FULL=/dev/${LVG_NAME}/${HOST_LV_NAME_DATA_LOG}

    DISK_SSD_INSTALL_REPO_NAME=install_repo_lv
    DISK_SSD_INSTALL_REPO_LVNAME=${PCI_DISK_NAME}/${DISK_SSD_INSTALL_REPO_NAME}

    PART_NUM_HOST=1
    PART_NUM_REP=2
    PART_NUM_PV=3
    PART_NUM_EFI=4
    TOTAL_PART_NUM=4

    if [ "${PLATFORM}" == "fretta" -o "${PLATFORM}" == "zermatt" -o "${PLATFORM}" == "ncs560" ] && \
        [ "${BOARDTYPE}" == "RP" ]; then
        let TOTAL_PART_NUM=TOTAL_PART_NUM+1
        PART_NUM_SECDISK=$TOTAL_PART_NUM
    fi

    if [ "${PLATFORM}" == "ncs1k" ] || [ "${PLATFORM}" == "ncs1001" ]; then
		PART_NUM_PERSIST=5
        TOTAL_PART_NUM=6
    fi

    if [ "${PLATFORM}" == "ncs1004" ]; then
        let TOTAL_PART_NUM=TOTAL_PART_NUM+2
    fi

    # Always Keep the App-Hosting Partition as the last Partition
    declare -F app_host_with_separate_pv >&107 2>&1 && app_host_with_separate_pv
    if [[ $? -eq 0 ]]; then
        let TOTAL_PART_NUM=TOTAL_PART_NUM+1
        PART_NUM_TP=$TOTAL_PART_NUM
    fi 

    if [ "${PLATFORM}" == "fretta" -o "${PLATFORM}" == "zermatt" ] && [ "${DR_EFI_EXISTS}" == "true" ]; then
        let TOTAL_PART_NUM=TOTAL_PART_NUM+1
        PART_NUM_DR_EFI=$TOTAL_PART_NUM
    fi   
    
    TOTAL_SUB_PART_DATA=3
}

function set_iso_file_name {
    # if the iso name file exists, read the names for the ISO's from it
    readonly ISO_NAME_FILE=/iso_name.txt
    if [ -e $ISO_NAME_FILE ]; then
       HOST_ISO=`cat   $ISO_NAME_FILE | grep  HOST_ISO:  | awk '{print $2}'`
       ADMIN_ISO=`cat  $ISO_NAME_FILE | grep ADMIN_ISO:  | awk '{print $2}'`
       SDR_ISO=`cat    $ISO_NAME_FILE | grep SDR_ISO:    | awk '{print $2}'`
       BUNDLE_ISO=`cat $ISO_NAME_FILE | grep BUNDLE_ISO: | awk '{print $2}'`
    else
        HOST_ISO=host.iso
        ADMIN_ISO=calvados.iso
        SDR_ISO=xr.iso
        BUNDLE_ISO=panini-mini-x.iso
    fi 
}

# PLATFORM is set in spirit_pd.sh by sourcing calvados_bootstrap.cfg 
if [ -z "$PLATFORM" ]; then
      step_log_console "No platform specified. Exiting."
      declare -F pd_notify_sw_fatal_event &>/dev/null && \
          pd_notify_sw_fatal_event "No platform specified. Exiting."
      exit 1
else
    if [ "${PLATFORM}" == "zermatt" ]; then
        IS_ISO_REQUIRED=FALSE
        step_log_file "This platform does not require ISO to be stored under tftpboot"
    else   
        IS_ISO_REQUIRED=TRUE
        step_log_file "This platform requires ISO to be stored under tftpboot"
    fi
fi

# Only read this once.
cmdline=$(cat /proc/cmdline)
echo "cmdline = $cmdline"
# Check recovery boot
RECOVERYBOOT=`echo $cmdline | grep "recoveryboot"`
# Check install reimage (biv)
INSTREIMAGE=`echo $cmdline | grep "instreimage"`

AUTOREBOOT=yes
if strstr "$cmdline" noautoreboot ; then
    AUTOREBOOT=no
fi

PXEEXIT=no
if strstr "$cmdline" pxeexit ; then
    PXEEXIT=yes
fi
PXEDEBUG=no
if strstr "$cmdline" pxedebug ; then
    PXEDEBUG=yes
fi

if strstr "$cmdline" giso_boot ; then
    GISOBOOT=yes
fi

if strstr "$cmdline" disaster_recovery ; then
    DISASTER_RECOVERY=yes
fi

function call_reboot {
  declare -F pd_pxe_call_reboot >/dev/null
  if [ $? -eq 0 ]; then
    pd_pxe_call_reboot
  elif (( $SIMULATION )); then
    echo "SIMULATION: Please login and shutdown, and restart simulation."
  elif [ "$PLATFORM" == "ncs1k" ] || [ "$PLATFORM" == "ncs1001" ] || [ "$PLATFORM" == "ncs1004" ]; then
    echo "Rebooting system after installation ..."
    sync
    #Reboot
    /usr/bin/perl /etc/rc.d/init.d/reset_fpgas.pl all > /dev/null 
    sync
    /sbin/reboot -f
  elif [ "$PLATFORM" == "ncs5k" ] ; then
        echo "Automatic rebooting system after installation ..."
        # Set panic timeout in case if kernel panics
        /sbin/sysctl -w kernel.panic=2 >/dev/null
        echo "Syncing file system"
        sync
        # reboot system
        echo b > /proc/sysrq-trigger >/dev/null
        sleep 2
        /sbin/reboot -f
  elif [ "$PLATFORM" == "asr9k" ] ; then
    if strstr "$cmdline" noautoreboot ; then
        echo "Please login and restart."
    else
        sync
        sleep 3
        declare -F asr9k_reboot && asr9k_reboot
    fi
  elif [ "$PLATFORM" == "iosxrv" ] ; then
      # Eject CDROM before rebooting
      echo "Eject CDROM and reboot system after installation ..."
      /usr/bin/eject /dev/cdrom
      sleep 2
      /sbin/reboot -f
  elif [ "$BOARDTYPE" == "CC" ]; then
      echo "Automatic rebooting CC after installation...."
      /sbin/reboot -f
  else
    declare -F pd_reset_bios_boot_mode && pd_reset_bios_boot_mode
    if [ "$AUTOREBOOT" == "yes" ]; then
        # pd_notify_img_install_reset is necessary only for some platforms
        # that need to track HOST events
        declare -F pd_notify_img_install_reset && pd_notify_img_install_reset
        echo "Automatic rebooting system after installation ..."
        # Set panic timeout in case if kernel panics
        /sbin/sysctl -w kernel.panic=2 >/dev/null
        # reboot system
        echo b > /proc/sysrq-trigger >/dev/null
        sleep 2
        if [ "${PLATFORM}" == "ncs560" -a "${BOARDTYPE}" == "RP" ]; then
            local IOFPGA_VENDOR_NCS560=1137
            local IOFPGA_DEVICE_NCS560=60 
            local PCI_BASE_ADDR_IOFPGA_NCS560=0x`lspci -vd $IOFPGA_VENDOR_NCS560:$IOFPGA_DEVICE_NCS560 | grep "Memory at" | cut -d' ' -f3`
            #Offset value to reset the board
            local iofpga_offset_addr=0x818
            local reg_addr=$(printf 0x%08X $(($PCI_BASE_ADDR_IOFPGA_NCS560 + $iofpga_offset_addr)))
            local val=1
            step_log_console "pcimemwrite $reg_addr $val"
            #Triggering board reset by writing value into AdmStatusCntl register
            /usr/bin/logger -t pcimemwrite "$(pcimemwrite $reg_addr 4 $val)"
        else 
            /sbin/reboot -f
        fi
    else
        # pd_notify_img_install_no_reset is necessary only for some platforms
        # that need to track HOST events
        declare -F pd_notify_img_install_no_reset && \
            pd_notify_img_install_no_reset
    fi
  fi
}

function collect_archive_interrupted_bake_logs {
    # Check if the disk /dev/sda exist and is partitioned or carved out.
    step_log_console "In collect_archive_interrupted_bake_logs function...."
    local primary_disk="/dev/${MAIN_DISK[0]}"
    fdisk -l ${primary_disk} &>/dev/null
    rc=$?
    if [[ $rc -eq 1 ]]; then
        num_disk_partition=0
        
    else
    num_disk_partition=$(fdisk -l ${primary_disk} | grep "^${primary_disk}" | wc -l 2>/dev/null)
    fi

    if [[ $num_disk_partition -gt 1 ]]; then
        # The card is baked previously and check if the previous bake operation 
        # is terminated abnormally or not.
        declare -F pd_check_if_debug_part_needed >&107 2>&1
        if [[ $? -eq 0 ]]; then
            step_log_console "Debug partition exist and is getting mounted.."
            get_dp_offset ${primary_disk} 
            dploop=$(losetup --find)
            if [ -z "${dploop}" ]; then
                step_log_console "Failed to create loopback device"
                return 1
            fi 
            losetup --offset ${dpoffset} --sizelimit ${dpsizeb} ${dploop} ${primary_disk}
            mount ${dploop} ${LOGFLOWDIR}
            if [[ $? -ne 0 ]]; then
                step_log_console "Failed to mount ${dploop} debug partition"
                return 1
            fi
            step_log_console "Debug partition mounted on $LOGFLOWDIR" 
            
            # with in debug partition identify, if the previous baking is aborted or not
            num_log_files=`ls -l ${LOGFLOWDIR}/*.log | wc -l`
            step_log_console "Found $num_log_files log files in debug partition" 
            if [[ $num_log_files -gt 0 ]]; then
                step_log_file "Below are interrupted bake logs"
                step_log_file "`ls -l ${LOGFLOWDIR}/*.log`"

                index=`cat ${LOGFLOWDIR}/index`
                if [ $index == 8 ] || [ -z "${index}" ]; then
                    index=0
                fi
                bakedir=${LOGFLOWDIR}/interrupt-bake.${index}
                step_log_console "copying interrupted logs into directory $bakedir" 
                mkdir -p ${bakedir}
                rm -rf ${bakedir}/*
                mv ${LOGFLOWDIR}/*.log ${bakedir}/
                tar cvfz ${LOGFLOWDIR}/interrupt-bake.${index}.tgz ${bakedir}/* >&107 2>&1
                rm -rf ${bakedir}
                index=`expr $index + 1`
                echo $index > ${LOGFLOWDIR}/index
            else
                step_log_console "Previous bake was successful"
            fi
            umount ${LOGFLOWDIR}
        else
            step_log_file "debug partition not supported"
        fi
    elif [[ $num_disk_partition -eq 0 && $rc -eq 0 ]]; then
        # This is first time bake on the card, the disks are not formatted.
        step_log_console "No, previous installation detected on the system"
    else
        step_log_console "The disk ${primary_disk} is not detected on this card"
    fi
}

CMD_PLATFORM=`echo $cmdline | sed 's/^.*platform=//' | cut -d" " -f1`
if [ "$CMD_PLATFORM" != "$PLATFORM" ]; then
   step_log_file "Platform in cmd line = $CMD_PLATFORM. Using $PLATFORM instead"
fi

#If platform hw has some way of discovering itself then OVERRIDE_PLATFORM is set to it
OVERRIDE_PLATFORM=$(override_platform_type ${PLATFORM}  $SIMULATION)
if [ "$OVERRIDE_PLATFORM" != "$PLATFORM" ]; then
    step_log_console "Wrong image for platform $OVERRIDE_PLATFORM. Exiting."
    declare -F pd_notify_sw_fatal_event &>/dev/null && \
        pd_notify_sw_fatal_event "Wrong image for platform $OVERRIDE_PLATFORM. Exiting."
    exit 1
fi

step_log_console "Preparing disk for PLATFORM=$PLATFORM:"

platform=$(cat /proc/cmdline | awk -F "platform=" ' { print $2 }' | cut -d " "  -f1)
if [ "$platform" == 'ncs5k' ]; then
    if [ -f /proc/scsi/scsi ]; then
     DISK_CNT=$(grep -c Vendor /proc/scsi/scsi )
    else
     DISK_CNT=0
    fi
    if [ $DISK_CNT -gt 1 ]
    then
     DISK_CHK="/dev/sdb"
    else
     DISK_CHK="/dev/sda"
    fi
    echo "Number of disks detected: $DISK_CNT: Disk Check for: $DISK_CHK "

    cnt=60;
    while [ $cnt -gt 0 ] ;  do
     if [ -b  "$DISK_CHK" ]; then
      break;
     fi;
     cnt=$(($cnt-1));
     echo "Waiting for $DISK_CHK device..($cnt)" ;
     sleep 1;
    done
    [ $cnt -lt 1 ] && echo "Error $DISK_CHK not found"
fi

ONIEINSTALL=`cat /proc/cmdline | grep onieinstall`
if [ -n "$ONIEINSTALL" ]; then
    ONIE_INSTALL_SCRIPT=/etc/init.d/onie_install.sh
    if [ -f $ONIE_INSTALL_SCRIPT ]; then
        echo "Passing control to $ONIE_INSTALL_SCRIPT $1"
        $ONIE_INSTALL_SCRIPT $1
        exit 0;
    else
        echo "$ONIE_INSTALL_SCRIPT not found"
        exit 0;
    fi
fi


if [ "${PLATFORM}" == "fretta" ]; then
    if [ "$BOARDTYPE" == "XC" -o "$BOARDTYPE" == "FC" ]; then
        BOOTTYPE="memboot"
    fi
fi

# if BOOTTYPE is memboot, pass control to pxe_install_memboot.sh
MEMBOOT_INSTALL_SCRIPT=/etc/init.d/pxe_install_memboot.sh
if [ -f $MEMBOOT_INSTALL_SCRIPT -a "$BOOTTYPE" == "memboot" ]; then 
    echo "Passing control to $MEMBOOT_INSTALL_SCRIPT $1"
    $MEMBOOT_INSTALL_SCRIPT $1
    exit 0;
fi

# Set boot type. Sim is legacy.
BOOTTYPE=`echo $cmdline | sed 's/^.*boot_type=//' | cut -d" " -f1`
if (( $SIMULATION )); then
    BOOTTYPE="legacy"
fi

declare -F pd_pxe_set_boottype >&107 2>&1 && pd_pxe_set_boottype

PARTMNT=$(mktemp -d /tmp/partmnt.XXXXXX)
ISOMNT=$(mktemp -d /tmp/isomnt.XXXXXX)
SSHKEY_DIR=$(mktemp -d /tmp/sshkeys.XXXXXX)
rootdir=${PARTMNT}
bootdir=${rootdir}/boot
grubdir=${bootdir}/grub
grub_shell=/sbin/grub

# This is where things start to happen
# reset the WD timer if being started from bios
declare -F reset_watchdog >&107 2>&1 && reset_watchdog
collect_archive_interrupted_bake_logs

case "$1" in
    start)
      if [ "$PXEEXIT" == "yes" -a ! -f /tmp/pxestart ]; then
        echo Exiting install script due to pxeexit setting
        echo To resume, create an empty /tmp/pxestart file and restart install
        # pd_notify_img_install_exit is necessary only for some platforms
        # that need to track HOST events
        declare -F pd_notify_img_install_exit && pd_notify_img_install_exit
        exit 0
      fi

      INSTALLDEV=`cat /proc/cmdline | sed 's/^.*install=//' | cut -d" " -f1`
      # Get the actual disk device names in case MAIN_DISK & SECONDARY_DISK contain symlinks
      sym_disk=${MAIN_DISK[0]}
      disk=$(get_disk_devices ${sym_disk})

      #Assumming the 2nd PCI disk is in /dev/sdb
      sym_second_disk=${SECONDARY_DISK[@]}
      second_disk=$(get_disk_devices ${sym_second_disk})
      #---Init 'disk' and 'second_disk' variables---
      platform=$(cat /proc/cmdline | awk -F "platform=" ' { print $2 }' | cut -d " "  -f1)
      if [ "$platform" == 'ncs5k' ]; then
          if [ $(lspci -mm | grep Broadcom | grep -ic -e b960 -e b965 -e b970) -ne 0 ]; then
              echo 'KERNEL=="sd*", ATTRS{vendor}=="ATA", SYMLINK+="ieusb%n"' > /etc/udev/rules.d/60-perssistent-storage-skywarp-eusb.rules;
          elif [ $(lspci -mm | grep Broadcom | grep -ic -e b867) -ne 0 ]; then
              echo 'KERNEL=="sd*", ATTRS{devpath}=="1.1", SYMLINK+="ieusb%n"' > /etc/udev/rules.d/60-perssistent-storage-skywarp-eusb.rules;
          else
              echo 'KERNEL=="sd*", ATTRS{vendor}=="SanDisk ", SYMLINK+="ieusb%n"' > /etc/udev/rules.d/60-perssistent-storage-skywarp-eusb.rules;
          fi

          #remove blkid rule which cause superblock errors for skywarp
          sed  -e "/sr.*.builtin.*.blkid/d" -i /lib/udev/rules.d/60-persistent-storage.rules
          /sbin/start_udev;

          #Seen instances where internal disk init takes longer due to external disk
          #which leads to udev rules missing  out internal disk. So check and retry
          RETRY_CNT=30
          curr_retry_cnt=0
          if [ ! -b /dev/$disk ]; then
              step_log_console "No install device $INSTALLDEV found.  Retrying.."
              while [ $curr_retry_cnt -lt $RETRY_CNT ]
              do
                # Try re-running the udev rules
                /sbin/start_udev
                sleep 1
                if [ -b /dev/$disk ]; then
                    step_log_console  "Disk detected. Resuming with install.."
                break;
                fi
                step_log_console  "Retrying..($curr_retry_cnt/$RETRY_CNT)"
                curr_retry_cnt=`expr $curr_retry_cnt + 1`
              done
              if [ $curr_retry_cnt == $RETRY_CNT ]; then
                step_log_console  "No Install Disk found. Exiting."
                declare -F pd_notify_hw_fatal_event &>/dev/null && \
                         pd_notify_hw_fatal_event "No Install Disk found. Exiting."
                exit ;
              fi
          fi
      fi

      if [ "$PLATFORM" == "ncs1k" ]; then
         echo "ncs1k udev activated "
         #mkdir -p /dev/.udev/rules.d/
         echo 'KERNELS=="ata2", SYMLINK+="mb_disk%n"' >> /etc/udev/rules.d/ncs1k-internal-sd.rules;
         #echo 'udev_rules="/dev/.udev/rules.d/"' >> /etc/udev/udev.conf;
         #remove blkid rule which cause superblock errors for skywarp
         sed  -e "/sr.*.builtin.*.blkid/d" -i /lib/udev/rules.d/60-persistent-storage.rules
         /sbin/start_udev;
         ls -l /dev/mb_disk
      fi
      if [ "$PLATFORM" == "ncs1001" ]; then
         echo "ncs1001 udev activated "
         #mkdir -p /dev/.udev/rules.d/
         echo 'KERNELS=="ata2", SYMLINK+="mb_disk%n"' >> /etc/udev/rules.d/ncs1001-internal-sd.rules;
         #echo 'udev_rules="/dev/.udev/rules.d/"' >> /etc/udev/udev.conf;
         #remove blkid rule which cause superblock errors for skywarp
         sed  -e "/sr.*.builtin.*.blkid/d" -i /lib/udev/rules.d/60-persistent-storage.rules
         /sbin/start_udev;
         ls -l /dev/mb_disk
      fi
      if [ "$PLATFORM" == "ncs1004" ]; then
         echo "ncs1004 udev activated "
         # mkdir -p /dev/.udev/rules.d/
         echo 'KERNELS=="ata5", SYMLINK+="cpu_disk%n"' >> /etc/udev/rules.d/ncs1004-internal-sd.rules;
         echo 'KERNELS=="ata12", SYMLINK+="mb_disk%n"' >> /etc/udev/rules.d/ncs1004-internal-sd.rules;
         # echo 'udev_rules="/dev/.udev/rules.d/"' >> /etc/udev/udev.conf;
         # remove blkid rule which cause superblock errors for skywarp
         sed  -e "/sr.*.builtin.*.blkid/d" -i /lib/udev/rules.d/60-persistent-storage.rules
         /sbin/start_udev;
         ls -l /dev/mb_disk
         ls -l /dev/cpu_disk
      fi
      if [ "$PLATFORM" == "asr9k" ]; then
         declare -f pd_pre_start_udev >&107 2>&1
         if [ "$?" -eq 0 ]; then
            pd_pre_start_udev >&107 2>&1
            step_log_console "asr9k-x64 udev activated "
            udevadm control --reload-rules
            udevadm trigger --type=subsystems --action=add
            udevadm trigger --type=devices --action=add
            udevadm trigger --sysname-match=dm-* --action=change
            udevadm settle
         fi
      fi


      #runtime check for secondary disk; applicable only for scapa platform 
      declare -F platform_get_second_disk && platform_get_second_disk

      if [ "${PLATFORM}" == "panini" -o "${PLATFORM}" == "scapa" ]; then
        if [ "${second_disk}" == "sda" ]; then
          echo "ERROR! No install boot device found, exiting..."
          # If we cannot find disk, it could be sign of SSD to about
          # going bad. A reimage retry sometime can recover from this
          # failure condition, but this is a sign that will tell
          # user that this failure could become permanent.
          declare -F pd_notify_hw_fatal_event &>/dev/null && \
              pd_notify_hw_fatal_event "No install boot device found, exiting installation"
          exit 0
        fi

        # Check if disk1 is partitioned
        if [ -n "${second_disk}" ]; then
          rpart=
          declare -F pd_pxe_find_recovery_partition >&107 2>&1
          if [ $? -eq 0 ]; then
             rpart=$(pd_pxe_find_recovery_partition "/dev/${second_disk}" "$DISK_ONE_RECOVERY_PART_NUM")
          fi
          if [ -n "${rpart}" ]; then
             vg2part=$DISK_ONE_VG_PART
             step_log_console "Found recovery partition ${second_disk}${rpart}"
          fi
        fi 
      fi

      # Check for Disaster Recovery EFI partition
      declare -F pd_check_for_dr_efi_part &> /dev/null && pd_check_for_dr_efi_part

      if (( $SIMULATION )); then
        if [ -b /dev/vda ]; then
          DISK_HOST_BOOT_PART_SIZE=900
          disk=vda
        fi
        if [ -b /dev/vdb ]; then
          second_disk=('vdb')
        fi
      fi #  $SIMULATION

        #
        # If the SCSI disk specified is not present, fallback and see if 
        # a corresponding virtio disk exists.
        #
        if [ ! -b /dev/$disk ]; then
            vdisk=`echo $disk | sed 's/sd/vd/g'`
            if [ -b /dev/$vdisk ]; then
                disk=$vdisk
            fi
        fi

        if [ ! -b /dev/$second_disk ]; then
            vsecond_disk=`echo $second_disk | sed 's/sd/vd/g'`
            if [ -b /dev/$vsecond_disk ]; then
                second_disk=$vsecond_disk
                step_log_console "Overwriting 2nd disk with $second_disk"
            fi
        fi

      # Make sure secondary disk exists whereever it is supposed to exist
      if [ ! -b /dev/$second_disk ]; then 
          second_disk=""
          declare -F pd_platform_has_second_disk >&107 2>&1
          if [ $? -eq 0 ]; then
              pd_platform_has_second_disk; 
              if [ $? -eq 0 ]; then
                  step_log_console "Secondary disk is not present. Exiting."
                  declare -F pd_notify_hw_fatal_event &>/dev/null && \
                      pd_notify_hw_fatal_event "Secondary disk is not present. Exiting."
                  exit
              fi
          else
              step_log_console "Secondary disk is not present"
          fi
      fi

      # pd_punch_watchdog is necessary only for some platforms
      declare -F pd_punch_watchdog && pd_punch_watchdog

      # pd_notify_img_install_started is necessary only for some platforms
      # that need to track HOST events
      declare -F pd_notify_img_install_started && pd_notify_img_install_started

      if [ ! -z $second_disk ]; then
          #pd_pxe_second_disk_check is needed only for some platforms 
          declare -F pd_pxe_second_disk_check >&107 2>&1 && pd_pxe_second_disk_check $vg2part
      fi

      #=======================Prepare 3rd disk=============================
      declare -F platform_third_disc_format >&107 2>&1 && platform_third_disc_format

      # make sure install disk exists before carrying on. 
      if [ ! -b /dev/$disk ]; then
          step_log_console "No install device $INSTALLDEV found.  Unable to install.  Exiting."
          declare -F pd_notify_sw_fatal_event &>/dev/null && \
              pd_notify_sw_fatal_event "No install device $INSTALLDEV found. Unable to install"
          exit 0
      fi

      step_log_console $"Installer will install image on ${disk}"

      #Check and report disk failures
      if [ "$SIMULATION" == 0 ]; then
        check_for_disk_errors $disk $second_disk
      fi

      set_lvm_param

      # Wipe out EFI to prevent potentially immature boot
      step_log_console "Removing old boot partition"
      efimnt=$(mktemp -d /tmp/efi.XXXXXX)
      mkdir -p ${efimnt}
      mount /dev/${disk}${PART_NUM_EFI} ${efimnt} >>/tmp/log 2>&1
      rm -rf ${efimnt}/*
      sync
      umount ${efimnt}
      rm -rf ${efimnt}

      # Create debug partition(dp), then relocate all logs to dp
      if [[ "$SIMULATION" == 0 ]]; then
         declare -F pd_check_if_debug_part_needed >&107 2>&1
         if [[ $? -eq 0 ]];then
            # Carve out debug partition
            step_log_console "Preparing debug partition ${PLATFORM}"
            if [ -z "${DEBUG_PART_SIZE}" ]; then
               DEBUG_PART_SIZE=${DEBUG_PART_SIZE_DEFAULT}
            fi
            get_dp_offset /dev/${disk}
            prepare_dploop /dev/${disk}
            if [ $? -eq 0 ]; then
               dpflag=TRUE
               logflow ${LOGFLOWDIR}/${LOGFLOWFILE}
               unredirect_output
               redirect_output $LOGDIR2/$LOGFILE
            fi
         fi
      fi
      if [ -z "$dpflag" ]; then
         DEBUG_PART_SIZE=0
      fi

      # Remove old volumes
      step_log_console "Removing old volumes"
      remove_old_volumes

      # This is done by slave-RP or LC nodes
      # installiso is set by grub.cfg, find 'sed' and 'installiso' in this file
      installiso=`cat /proc/cmdline | grep installiso=`
      if [ -n "$installiso" ]; then
        copy_iso_files
      fi

      # at this juncture, calvados and xr iso must be available
      if [ ! -f /iso/${ADMIN_ISO} -o ! -f /iso/${SDR_ISO} ]; then
         declare -F pd_copy_iso_files >&107 2>&1 && pd_copy_iso_files
      fi

      set_iso_file_name

      if [ "$GISOBOOT" == "yes" ]; then
          # This will download the giso image
          declare -f pd_download_giso >&107 2>&1
          if [ "$?" -eq 0 ]; then
              pd_download_giso
          else
              download_giso
          fi
          step_log_console "Going to copy GISO files..."
          copy_giso_files system_image.iso
          step_log_console "Copied Golden ISO files..."
      fi
      

      ipaddress=`cat /proc/cmdline | grep ipaddress=`
      if [ "${PLATFORM}" == "panini" -a "${BOARDTYPE}" == "RP" -a -z "${installiso}" -a -n "${ipaddress}" ]; then
          step_log_console "Downloading whole ISO !!!!!!!!!"
          step_log_console "PLATFORM = ${PLATFORM} BOARDTYPE = ${BOARDTYPE} installiso = ${installiso}"
          download_iso
          step_log_console "Downloaded whole ISO !!!!!!!!!"
      fi

      if [ "$BOARDTYPE" == "CC" ]; then
          step_log_console "Downloading CC ISO..."
          declare -F pd_pxe_download_system_iso >&107 2>&1 
          if [ $? -eq 0 ]; then
              bootmedia=$(pd_pxe_download_system_iso $BOARDTYPE)
          fi    
      fi

      #fretta lc
      declare -f pd_lc_int_pxe >&107 2>&1
      if [ "$?" -eq 0 ]; then
          pd_lc_int_pxe
          if [ -e system_image.iso ]; then
              copy_giso_files system_image.iso
          fi
      fi

      mount -r -o loop /iso/${HOST_ISO} $ISOMNT
      verify_md5sum $ISOMNT || exitmd5sum_reboot ${HOST_ISO}
      umount $ISOMNT
      mount -r -o loop /iso/${ADMIN_ISO} $ISOMNT
      verify_md5sum $ISOMNT || exitmd5sum_reboot ${ADMIN_ISO}
      umount $ISOMNT
      mount -r -o loop /iso/${SDR_ISO} $ISOMNT
      verify_md5sum $ISOMNT || exitmd5sum_reboot ${SDR_ISO}
      umount $ISOMNT
      step_log_console "Md5sum verification succeeded"

      #================= Boss-Hogg specific modifications ========================
	  if [ $DISASTER_RECOVERY == "yes" ]; then
 	     step_log_console "---------------------------------------"
 	     step_log_console "-------Disaster recovery baking--------"
 	     step_log_console "---------------------------------------"
         declare -f dr_copy_giso >&107 2>&1
         if [ "$?" -eq 0 ]; then
	        dr_copy_giso ${HOSTRPMS} ${CALRPMS} ${XRRPMS}
	     fi
	  else
	     step_log_console $"---Starting to prepare mb_disk---"
	     declare -f mb_disk_create_partitions >&107 2>&1
	     if [ "$?" -eq 0 ]; then
		    mb_disk_create_partitions 
	     fi
	  fi

      #=======================Prepare sda/vda ===============================
      step_log_console
      if [ "${sym_disk}" != "${disk}" ]; then
         step_log_console $"---Starting to prepare ${sym_disk} [${disk}]---"
      else
         step_log_console $"---Starting to prepare ${disk}---"
      fi

      # Setup the ssh keys 
      create_ssh_keys 
      

      # Host OS part. Main disk 
      # Create parted based partition or fdisk based partition based on platform
      START=$(date +%s)
      if [ "$BOOTTYPE" != "legacy" ]; then
          check_parted_fix_disk_part ${disk} "${TOTAL_PART_NUM}"
      else
        create_part_file_disk /dev/$disk
        check_fix_disk_part ${disk} "${TOTAL_PART_NUM}" "/tmp/fdisk.input"
        mark_part_bootable ${disk} "${disk}${PART_NUM_HOST}"
      fi
      END=$(date +%s)
      DIFF=$(( $END - $START ))
      step_log_console $"Partition creation on /dev/${disk} took ${DIFF} seconds"

      START=$(date +%s)
      check_fs_partition "/dev/${disk}${PART_NUM_HOST}" "SimHostOs"
      END=$(date +%s)
      DIFF=$(( $END - $START ))
      step_log_console $"File system creation on /dev/${disk}${PART_NUM_HOST} took ${DIFF} seconds"

      # Provision disk one on the main disk for RP on Fretta platform
      if [ "${PLATFORM}" == "fretta" -a "${BOARDTYPE}" == "RP" ]; then
        provision_disk_one ${disk}${PART_NUM_SECDISK}
        install_pxeboot_files $disk
      fi
      if [ "${PLATFORM}" == "zermatt" -a "${BOARDTYPE}" == "RP" ]; then
        provision_disk_one ${disk}${PART_NUM_SECDISK}
        install_pxeboot_files $disk
      fi
      if [ "${PLATFORM}" == "ncs560" -a "${BOARDTYPE}" == "RP" ]; then
          step_log_console "Set boot mode as Boot from Disk after installation"
          /bin/mount -t efivarfs efivarfs /sys/firmware/efi/efivars
          pxe_reimage=$(/usr/sbin/efibootmgr | grep OS |\
          awk '{print $1}' | awk -F '*' '{print $1}'| sed -e s/Boot//)
          #Cmd for Setting Bootorder
          /usr/sbin/efibootmgr -n $pxe_reimage
          /bin/umount efivarfs

          provision_disk_one ${disk}${PART_NUM_SECDISK}
          install_pxeboot_files $disk
      fi

      step_log_console $"Install boot image on /dev/${disk}${PART_NUM_HOST}"
      START=$(date +%s)
      install_boot_iso "${disk}" "${PART_NUM_HOST}"
      END=$(date +%s)
      DIFF=$(( $END - $START ))
      step_log_console $"Installing host image size of ${SIZE} took ${DIFF} seconds"

      create_sub_part_file_disk_boot
      create_sub_part_file_disk_data

      #=======================Prepare host volumes=============================
      step_log_console
      step_log_console $'---Starting to prepare host logical volume---'

      # check for additional disks for the panini_vol_grp volume group
      for i in "${disk[@]}"
      do
         if [ ${i} != ${disk[0]} ]; then
            addtl_disks="${addtl_disks} /dev/$i"
         fi
      done

      # create physical volumes
      step_log_file $"pvcreate -ff -y /dev/${disk}${PART_NUM_PV} ${addtl_disks}"
      pvcreate -ff -y /dev/${disk}${PART_NUM_PV} ${addtl_disks} >&107 2>&1
     
      # Check for App Host Enablement and trigger App Host Volume Creation Job accordingly
      declare -F app_host_with_separate_pv >&107 2>&1 && app_host_with_separate_pv
      if [[ $? -eq 0 ]]; then
            declare -F platform_log_console >&107 2>&1 && \
            platform_log_console "Create app volume on $disk"

            . /etc/init.d/app_volume.sh
            app_volume_mount $disk $PART_NUM_TP
      fi

      # Create volume group 
      step_log_file $"vgcreate ${LVG_NAME} /dev/${disk}${PART_NUM_PV} ${addtl_disks}"
      vgcreate ${LVG_NAME} /dev/${disk}${PART_NUM_PV} ${addtl_disks} >&107 2>&1

      # Create host boot,data, config and misc logical volumes
      create_host_boot_and_data_lvs

      install_host_iso "${HOST_LV_PART_BOOT}" ""

      # Save accumulated log to persistent disk & from on, save direct to disk
      save_install_log "${HOST_LV_NAME_DATA_LOG_FULL}"

      #=======================Prepare calvados volumes=========================
      START=$(date +%s)
      create_sysadmin_boot_and_data_lv
      END=$(date +%s)
      DIFF=$(( $END - $START ))
      step_log_console $"File system creation on /dev/${CALVADOS_LV_PART} took ${DIFF} seconds"
      loop_dev=$(setup_loop_offset "/dev/${CALVADOS_LV_PART}" "p1")
      partcal_offset=$(find_fs_offset "/dev/${CALVADOS_LV_PART}" "p1")
      patcalo_szlmt=$(find_fs_sizelimit "/dev/${CALVADOS_LV_PART}" "p1")
      step_log_console $"Install sysadmin-vm image on /dev/${CALVADOS_LV_PART}"
      START=$(date +%s)
      install_calvados_iso  "${CALVADOS_LV_PART}" "" ${partcal_offset} ${patcalo_szlmt}
      #  sleep needed to sync to occur
      sync
      END=$(date +%s)
      DIFF=$(( $END - $START ))
      step_log_console $"Installing sysadmin-vm image size of ${SIZE} took ${DIFF} seconds"
      sleep 2
      losetup -d ${loop_dev}

      # Create 18+ GB Install repository in disk1 as LV for panini platform
      if [ "$BOARDTYPE" == "RP" -o "$BOARDTYPE" == "CC" ] && [[ "$SIMULATION" == 0 ]]; then
        declare -F pd_create_install_repo_in_disk >&107 2>&1
        if [ $? -eq 0 ]
        then
            pd_create_install_repo_in_disk
        fi
      fi
      #=======================Prepare repository===============================
      create_and_prep_repository $disk $PART_NUM_REP
      repo_offset=$(find_fs_offset "/dev/${disk}${PART_NUM_REP}" "p1")
      repo_szlmt=$(find_fs_sizelimit "/dev/${disk}${PART_NUM_REP}" "p1")
      START=$(date +%s)
      copy_xr_iso_to_repository  "${disk}${PART_NUM_REP}" ${repo_offset} ${repo_szlmt}
      END=$(date +%s)
      DIFF=$(( $END - $START ))
      step_log_console $"Copying all ISOs to repository took ${DIFF} seconds"

      #================= Additional LVs ========================
      # If some platforms/boards require additional LVs to be created on
      # the main LVG - panini_vol_grp. Platforms can choose to specify 
      # additional LVs using variables - 
      # NUM_EXTRA_PARTS - number of additional LVs needed.
      # EXTRA_PART_SIZEx - Size of additional LVx. 
      # EXTRA_PART_NAMEx - Name of additional LVx.
      #   where x represents the extra LV number capped to NUM_EXTRA_PARTS

      if [[ "$NUM_EXTRA_PARTS" -gt 0 ]]; then
          for (( i=1; i<=$NUM_EXTRA_PARTS; i++ )); do
              PART_SIZE="EXTRA_PART_SIZE${i}"
              PART_NAME="EXTRA_PART_NAME${i}"
              FS_VOL_LBL="EXTRA_PART_VOL_LBL${i}"
              PART_NAME_FULL=/dev/${LVG_NAME}/${!PART_NAME}
              step_log_file $"Creating ${PART_NAME_FULL} on card"
              step_log_file $"lvcreate -L ${!PART_SIZE} -n ${!PART_NAME} ${LVG_NAME}"
              lvcreate -L ${!PART_SIZE} -n ${!PART_NAME} ${LVG_NAME} >&107 2>&1
              check_fs_partition "${PART_NAME_FULL}" "${!FS_VOL_LBL}"
              clean_partition "${PART_NAME_FULL}" 
          done
      fi 

      if [ ! -z "$second_disk" ]; then
         step_log_console "-----Found secondary disk: ${second_disk} -----"
         if [ "${BOARDTYPE}" == "RP" -o "${BOARDTYPE}" == "CC" ]; then

              #=======================Prepare 2nd disk=============================
              # Configure 2nd disk before installing sysadmin-vm and XR
              modprobe dm-mod
              for i in ${second_disk[@]}; do
                  if [ "${i}" != "${second_disk[0]}" ]; then
                     s_disks="${s_disks} ${i}"
                  else
                     s_disks="${s_disks} ${i}${vg2part}"
                  fi
              done
              step_log_console
              step_log_console "---Starting to prepare secondary disks:${s_disks}---"
              provision_disk_one "${s_disks}"

              # In case of GISO pxe boot system_image.sio already downloaded
              # If non GISO pxe boot Wait for background iso creation to complete
              if [ ! -e system_image.iso ]; then                                       
                  declare -F pd_wait_for_iso >&107 2>&1 && pd_wait_for_iso
              fi

              install_pxeboot_files ${second_disk[0]}
          fi
          
	fi

      #================= EFI install ========================
      if [ "$BOOTTYPE" != "legacy" ]; then
          step_log_console $"Install EFI on /dev/${disk}${PART_NUM_EFI}"
          create_vfat_partition "/dev/${disk}${PART_NUM_EFI}" 
          install_efi "${disk}" "${PART_NUM_EFI}" "1" "HostOs"
          # Allow platform specific tweaking to efi boot order.
          #    
          declare -F pd_pxe_update_efi_bootorder >&107 2>&1
          if [ $? -eq 0 ]
          then 
              step_log_file "PD: Setting ${disk}${PART_NUM_EFI} as efi bootoption"
              pd_pxe_update_efi_bootorder "${disk}" "${PART_NUM_EFI}" "$BOARDTYPE" 
          fi
      fi

      # Delete temporary ssh keys dir
      rm -fr ${SSHKEY_DIR}

      #================= Skywarp Specific Initrd modification ===
      platform=$(cat /proc/cmdline | awk -F "platform=" ' { print $2 }' | cut -d " "  -f1)
      if [ "$platform" == 'ncs5k' ]; then
          DISK="/dev/ieusb"
          mkdir -p /tmp1 && mount $DISK"1" /tmp1 && \
          cp /tmp1/boot/initrd.img /initrd.img.orig && \
          cp /tmp1/boot/initrd.img / && cd / && mv initrd.img initrd.img.gz && \
          gunzip initrd.img.gz && rm -rf /init.tmp && mkdir -p /init.tmp &&  \
          cd /init.tmp &&  cpio -id < ../initrd.img && \
	  if [ $(cat  /proc/cpuinfo  | grep processor | wc -l) == 4 ] ; then \
           sed -i -e "s#modprobe virtio_blk.*#modprobe virtio_blk; \n \
              sleep 5; if [ -f /proc/scsi/scsi ] ; then  cat /proc/scsi/scsi ; DISK_CNT=\$( grep -c Vendor /proc/scsi/scsi ); else DISK_CNT=0; fi ; \n \
              if [ \$DISK_CNT -gt 1 ]; then DISK_CHK=\"/dev/sdb\"; else DISK_CHK=\"/dev/sda\"; fi ; \n \
              echo \"Number of disks detected: \$DISK_CNT: Disk Check for: \$DISK_CHK \" \n \
              cnt=0; \n mkdir -p /dev/.udev/rules.d ; \n while [ \$cnt -lt 60 ] ; \n do \n \ if [ -b \$DISK_CHK ]  ; then \n \
              echo \'KERNEL==\"sd*\", ATTRS{vendor}==\"SanDisk \", SYMLINK+=\"ieusb%n\"\' > /dev/.udev/rules.d/skywarp-internal-eusb.rules; \n \
              echo \'udev_rules=\"/dev/.udev/rules.d/\"\' >> /etc/udev/udev.conf; \n   /sbin/start_udev; \n \
              echo \"Detected \$DISK_CHK\"; \n  break; \n fi; \n sleep 1;  \n cnt=\$((\$cnt+1)); \n \
              echo \"Waiting for \$DISK_CHK device..(\$cnt/60)\"; \n  done \n \
              if [ \$cnt -eq 60 ];  then  echo \"Error.Device \$DISK_CHK not detected\"; fi #"  \
              -e "s#\(export PATH.*$\)#\1\nmodprobe usb-storage#"  ./init ; \
          else  \
           sed -i -e "s#modprobe virtio_blk.*#modprobe virtio_blk; \n \
              sleep 5; if [ -f /proc/scsi/scsi ] ; then  cat /proc/scsi/scsi ; DISK_CNT=\$( grep -c Vendor /proc/scsi/scsi ); else DISK_CNT=0; fi ; \n \
              if [ \$DISK_CNT -gt 1 ]; then DISK_CHK=\"/dev/sdb\"; else DISK_CHK=\"/dev/sda\"; fi ; \n \
              echo \"Number of disks detected: \$DISK_CNT: Disk Check for: \$DISK_CHK \" \n \
              cnt=0; \n mkdir -p /dev/.udev/rules.d ; \n while [ \$cnt -lt 60 ] ; \n do \n \ if [ -b \$DISK_CHK ]  ; then \n \
              echo \'KERNEL==\"sd*\", ATTRS{devpath}==\"1.1\", SYMLINK+=\"ieusb%n\"\' > /dev/.udev/rules.d/skywarp-internal-eusb.rules; \n \
              echo \'udev_rules=\"/dev/.udev/rules.d/\"\' >> /etc/udev/udev.conf; \n    \
              echo \"Detected \$DISK_CHK\"; \n  break; \n fi; \n sleep 1;  \n cnt=\$((\$cnt+1)); \n \
              echo \"Waiting for \$DISK_CHK device..(\$cnt/60)\"; \n  done \n \
              if [ \$cnt -eq 60 ];  then  echo \"Error.Device \$DISK_CHK not detected\"; fi #"  \
              -e "s#\(export PATH.*$\)#\1\nmodprobe usb-storage#"  ./init ; \
          fi  && \
          find . | cpio --create --format='newc' > /newinitrd && \
          cd / && gzip newinitrd && \
          mv newinitrd.gz /tmp1/boot/initrd.img && \
          umount /tmp1
      fi


      #================= prepare for reboot ========================

      # pd_notify_img_install_done is necessary only for some platforms
      # that need to track HOST events
      declare -F pd_notify_img_install_done && pd_notify_img_install_done

      # Install finished, Save install log before system reboot
      step_log_console $"Install finished on ${disk}"

      #================= prepare for reboot ========================

      close_install_log

      call_reboot
        ;;

    stop)
        exit 0
        ;;

    restart | reload | force-reload | status | condrestart | try-restart)
        usage
        exit 3
        ;;

    *)
        usage
        exit 2
        ;;
esac
