#!/bin/bash
# Author : Mahesh Singh
# Copyright (c) 2018 by Cisco Systems, Inc.
# All rights reserved.

trap exitclean SIGINT SIGTERM
usage() {
    step_log_console $"Usage: $0 {start|stop}"
}

function step_log_console {
    local log_msg=${1}
    local time_str=$(date)
    echo "${time_str}: ${log_msg}" >&105
    # Also log console message to file
    echo "${time_str}: ${log_msg}" >&107
    if [ $USE_FINAL -gt 0 ]; then
        sync
    fi
}
readonly -f step_log_console

readonly CARD_INST_FILE="./root/card_instances.txt"
# Log script callflow
readonly LOGFLOWDIR=/tmp/bakelog
readonly LOGFLOWFILE=`basename $0`.flow.log
. /etc/init.d/logflow
mkdir -p $LOGFLOWDIR
logflow $LOGFLOWDIR/$LOGFLOWFILE

# Keep this code on top of this file.
# Platform hook when pxe_install starts
if [[ $1 == start ]]; then
    if [[ -f /etc/init.d/platform_pxe_start_hook.sh ]]; then
        source /etc/init.d/platform_pxe_start_hook.sh
    fi
fi  

if [ -f /etc/rc.d/init.d/pd-functions ]; then
    source /etc/rc.d/init.d/pd-functions 
fi

# First parameter is disk (ex: /dev/vda)
# If LVM size is not specified, calculate it from total disk size. 
# One complication is, fdisk reports total size in MB (1000), but parameters 
# given for creating partitions are in MiB (1024). So, need slight conversion.
function create_part_file_disk {
    local lvm_part_size_value=${DISK_HOST_DATA_PART_SIZE}
    local obfl_flash_size=64
    get_disk_size ${1}
    if [ -z ${DISK_SIZE} ]; then
        step_log_console "ERROR! No install boot device found, exiting..."
        # If we cannot find disk, it could be sign of SSD to about
        # going bad. A reimage retry sometime can recover from this
        # failure condition, but this is a sign that will tell
        # user that this failure could become permanent.
        declare -F pd_notify_sw_fatal_event &>/dev/null && \
            pd_notify_hw_fatal_event "No install boot device found for partitioning, exiting installation"
        declare -F pd_notify_hw_fatal_event &>/dev/null && \
            pd_notify_sw_fatal_event "No valid boot device found in ${1}"
        exit 0
    fi
    let disk_size=${DISK_SIZE}*1000000
    let disk_size/=1048576
    let disk_size=$disk_size-$lvm_part_size_value-$DISK_HOST_DATA_PART_SIZE-$DEBUG_PART_SIZE-$obfl_flash_size
    step_log_console "Creating partitions: BOOT=${DISK_HOST_BOOT_PART_SIZE}MB, LVM=${lvm_part_size_value}MB"
    step_log_console "Creating partitions: OBFL=${obfl_flash_size}MB"
    declare -F pd_check_if_obfl_on_disk >&107 2>&1                          
    if [[ $? -eq 0 ]];then                                                  
       obfl_on_disk=$(pd_check_if_obfl_on_disk)                             
    fi 
#below code is intentionally unalligned.
if [ "x${obfl_on_disk}" == "xyes" ]; then 
echo "

n
p
1

+${DISK_HOST_BOOT_PART_SIZE}M
n
p
2

+${DISK_CALVADOS_REPO_SIZE}M
n
p
3

+${lvm_part_size_value}M
n
e
4

+${disk_size}M
n

+${obfl_flash_size}M

w" > /tmp/fdisk.input
else
echo "

n
p
1

+${DISK_HOST_BOOT_PART_SIZE}M
n
p
2

+${DISK_CALVADOS_REPO_SIZE}M
n
p
3

+${lvm_part_size_value}M
n
e
4

w" > /tmp/fdisk.input
 
fi
}
readonly -f create_part_file_disk

function create_obfl_part_file {
echo "

n
p



w" > /tmp/obfl_fdisk.input
}
readonly -f create_obfl_part_file

cleanup() {
    # pd_stop_watchdog is necessary only for some platforms
    declare -F pd_stop_watchdog && pd_stop_watchdog

    [ -d "${PARTMNT}" ] && umount ${PARTMNT} && rmdir ${PARTMNT}
}

exitclean() {
    step_log_console "Cleaning up to exit..."
    cleanup
    exit 1
}

function create_ssh_keys {
    step_log_file "Create ssh keys"
    mkdir -p ${SSHKEY_DIR}/hostos ${SSHKEY_DIR}/calvados 
    ssh-keygen -q -t rsa -f ${SSHKEY_DIR}/calvados/id_rsa -C root@calvados-vm -N '' >&107 2>&1
    ssh-keygen -q -t rsa -f ${SSHKEY_DIR}/hostos/id_rsa -C root@host -N '' >&107 2>&1
    cat ${SSHKEY_DIR}/calvados/id_rsa.pub >> ${SSHKEY_DIR}/hostos/authorized_keys
    cat ${SSHKEY_DIR}/calvados/id_rsa.pub >> ${SSHKEY_DIR}/calvados/authorized_keys
    cat ${SSHKEY_DIR}/hostos/id_rsa.pub >> ${SSHKEY_DIR}/calvados/authorized_keys
}
readonly -f create_ssh_keys


function copy_nbi_to_boot_partition {
    dev_name=${1}
    mount ${dev_name} ${PARTMNT}
    if [ $? -ne 0 ]; then
        step_log_console "failed to mount ${dev_name} to ${PARTMNT}"
        return 1
    fi
    mkdir -p ${PARTMNT}/boot/
    #uboot starts diskboot by loading sda1/boot/sflc-auto.img    
    cp /fm.nbi ${PARTMNT}/boot/sflc-auto.img
    umount ${PARTMNT}
    sync
}
#==============================================================
#                     Main script code
#==============================================================

#---Basic PI variables---
readonly PD_DYNAMIC="/etc/init.d/spirit_pd.sh"
readonly PI_GRUB_SCRIPT="/etc/init.d/pi_grub_and_menu_lst_update.sh"
readonly MOUNT_PD_SCRIPT="/etc/init.d/mount_pd_fs.sh"
readonly NBI_DISK_BAKE_SCRIPT="/etc/init.d/nbi_disk_bake.sh"
readonly LOGGING_SCRIPT="/etc/init.d/logging_functions.sh"

# TEMP_INSTALL_FILE_NAME: This file would be in the ramfs at this stage,
# but would be copied to the host and calvados boot paritions after they
# are prepared. The file this copied will be used by install software
# during steady state install operation.
readonly TEMP_INSTALL_FILE_NAME="/tmp/pxe_install_temp.cfg"
readonly LOGDIR=/var/log
readonly LOGFILE0=host-install0.log
readonly LOGFILE=host-install.log
readonly LOGDIR2=/tmp/bakelog
readonly LOGFILE2=sysadmin-install.log
readonly THIS_SCRIPT=pxe_install_arm_diskboot.sh
USE_FINAL=0

#-------get logging functions--------------
if [ -f $LOGGING_SCRIPT ]; then
    step_log_console "Sourcing $LOGGING_SCRIPT PI script"
    . $LOGGING_SCRIPT
else
    step_log_console "Error, could not find $LOGGING_SCRIPT file which contains console/file logging functions."
    step_log_console "Exiting $THIS_SCRIPT installation script."
    declare -F pd_notify_sw_fatal_event &>/dev/null && \
        pd_notify_sw_fatal_event "Error, could not find $LOGGING_SCRIPT file which contains PD parameters for installation."
    exit 0
fi
redirect_output $LOGDIR2/$LOGFILE

# Source function library.
. /etc/init.d/functions
. /etc/init.d/disk-functions
. /etc/init.d/spirit-functions
. /etc/init.d/pd-functions
. /etc/init.d/lv-functions

# Show that this script is starting
step_log_file "Starting to execute $THIS_SCRIPT"

#-------get PD variables--------------
if [ -f $PD_DYNAMIC ]; then
    step_log_file "Sourcing $PD_DYNAMIC PD script"
    . $PD_DYNAMIC
else
    step_log_console "Error, could not find $PD_DYNAMIC file which contains PD parameters for installation."
    step_log_console "Exiting $THIS_SCRIPT installation script."
    declare -F pd_notify_sw_fatal_event &>/dev/null && \
        pd_notify_sw_fatal_event "Error, could not find $PD_DYNAMIC file which contains PD parameters for installation."
    exit 0
fi

#-------get PI bootloader update functions--------------
if [ -f $PI_GRUB_SCRIPT ]; then
    step_log_console "Sourcing $PI_GRUB_SCRIPT PI script"
    . $PI_GRUB_SCRIPT
else
    step_log_console "Error, could not find $PI_GRUB_SCRIPT file which contains PD parameters for installation."
    step_log_console "Exiting $THIS_SCRIPT installation script."
    declare -F pd_notify_sw_fatal_event &>/dev/null && \
        pd_notify_sw_fatal_event "Error, could not find $PI_GRUB_SCRIPT file which contains PD parameters for installation."
    #FIXME
    #exit 0
fi

#-------get NBI DISK BAKE functions--------------
if [ -f $NBI_DISK_BAKE_SCRIPT ]; then
    step_log_console "Sourcing $NBI_DISK_BAKE_SCRIPT PI script"
    . $NBI_DISK_BAKE_SCRIPT
else
    step_log_console "Error, could not find $NBI_DISK_BAKE_SCRIPT file which contains PD parameters for installation."
    step_log_console "Exiting $THIS_SCRIPT installation script."
    declare -F pd_notify_sw_fatal_event &>/dev/null && \
        pd_notify_sw_fatal_event "Error, could not find $NBI_DISK_BAKE_SCRIPT file which contains PD parameters for installation."
    exit 0
fi

#-------get PI bind mount functions--------------
if [ -f $MOUNT_PD_SCRIPT ]; then
    step_log_console "Sourcing $MOUNT_PD_SCRIPT PI script"
    . $MOUNT_PD_SCRIPT
else
    step_log_console "Error, could not find $MOUNT_PD_SCRIPT file which contains PD parameters for installation."
    step_log_console "Exiting $THIS_SCRIPT installation script."
    declare -F pd_notify_sw_fatal_event &>/dev/null && \
        pd_notify_sw_fatal_event "Error, could not find $MOUNT_PD_SCRIPT file which contains PD parameters for installation."
    #FIXME
    #exit 0
fi
on_sim

# Get board type from PD function
get_board_type
step_log_file "Got board type $BOARDTYPE"

# Retrieves platform bootstrap variable and customizes install 
# specific variables based on the board type.

if [ -f $INSTALL_PARAMS_OVERRIDE ]; then
    readonly PI_CTYPE=UNKNOWN
    source $INSTALL_PARAMS_OVERRIDE $PD_BOOTSTRAP  \
           1 $BOARDTYPE $PI_CTYPE $TEMP_INSTALL_FILE_NAME
    if (( $? != 0 )); then
        step_log_file "Cannot read platform bootstrap cfg"
        declare -F pd_notify_sw_fatal_event &>/dev/null && \
            pd_notify_sw_fatal_event "Cannot read platform bootstrap cfg"
        exit 1 
    fi
fi

step_log_file "Simulation=$SIMULATION"

#Function to setup logical volume and volume group names
function set_lvm_param {
    LVG_NAME=panini_vol_grp
    PCI_DISK_NAME=pci_disk1

    CALVADOS_LV_NAME=calvados_lv0
    CALVADOS_LV_PART=${LVG_NAME}/${CALVADOS_LV_NAME}
    CALVADOS_LV_PART_BOOT=${LVG_NAME}/${CALVADOS_LV_NAME}

    CALVADOS_LV_NAME_DATA=calvados_data_lv0 
    CALVADOS_LV_PART_DATA=${LVG_NAME}/${CALVADOS_LV_NAME_DATA}

    HOST_LV_NAME=host_lv0
    HOST_LV_PART_BOOT=${LVG_NAME}/${HOST_LV_NAME}
    HOST_LV_NAME_FULL="/dev/${LVG_NAME}/${HOST_LV_NAME}"

    HOST_LV_NAME_DATA_SCRATCH=host_data_scratch_lv0
    HOST_LV_NAME_DATA_CONFIG=host_data_config_lv0
    HOST_LV_NAME_DATA_LOG=host_data_log_lv0

    HOST_LV_NAME_DATA_SCRATCH_FULL=/dev/${LVG_NAME}/${HOST_LV_NAME_DATA_SCRATCH}
    HOST_LV_NAME_DATA_CONFIG_FULL=/dev/${LVG_NAME}/${HOST_LV_NAME_DATA_CONFIG}
    HOST_LV_NAME_DATA_LOG_FULL=/dev/${LVG_NAME}/${HOST_LV_NAME_DATA_LOG}

    PART_NUM_HOST=1
    PART_NUM_REP=2
    PART_NUM_PV=3
    PART_NUM_EXTENDED=4
    PART_NUM_OBFL=5
    TOTAL_PART_NUM=5

    TOTAL_SUB_PART_DATA=3
}

# PLATFORM is set in spirit_pd.sh by sourcing calvados_bootstrap.cfg 
if [ -z "$PLATFORM" ]; then
      step_log_console "No platform specified. Exiting."
      declare -F pd_notify_sw_fatal_event &>/dev/null && \
          pd_notify_sw_fatal_event "No platform specified. Exiting."
      exit 1
fi

# Only read this once.
cmdline=$(cat /proc/cmdline)
echo "cmdline = $cmdline"
# Check recovery boot
RECOVERYBOOT=`echo $cmdline | grep "recoveryboot"`
# Check install reimage (biv)

AUTOREBOOT=yes
if strstr "$cmdline" noautoreboot ; then
    AUTOREBOOT=no
fi

PXEEXIT=no
if strstr "$cmdline" pxeexit ; then
    PXEEXIT=yes
fi

function call_reboot {
  declare -F pd_pxe_call_reboot >/dev/null
  if [ $? -eq 0 ]; then
    pd_pxe_call_reboot
  elif (( $SIMULATION )); then
    echo "SIMULATION: Please login and shutdown, and restart simulation."
  else
    declare -F pd_reset_bios_boot_mode && pd_reset_bios_boot_mode
    if [ "$AUTOREBOOT" == "yes" ]; then
        # pd_notify_img_install_reset is necessary only for some platforms
        # that need to track HOST events
        declare -F pd_notify_img_install_reset && pd_notify_img_install_reset
        echo "Automatic rebooting system after installation ..."
        # Set panic timeout in case if kernel panics
        /sbin/sysctl -w kernel.panic=2 >/dev/null
        # reboot system
        echo b > /proc/sysrq-trigger >/dev/null
        sleep 2
        /sbin/reboot -f
    else
        # pd_notify_img_install_no_reset is necessary only for some platforms
        # that need to track HOST events
        declare -F pd_notify_img_install_no_reset && \
            pd_notify_img_install_no_reset
    fi
  fi
}

function create_obfl_partition {
    declare -F pd_check_if_obfl_on_disk >&107 2>&1 && \
        obfl_on_disk=$(pd_check_if_obfl_on_disk)
    if [ "x${obfl_on_disk}" == "xyes" ]; then
        retcode=1
        loop_dev=$(setup_loop_offset /dev/"${OBFL_DEV}" "p1")
        if [ -n "${loop_dev}" ]; then
            check_fs_sb ${loop_dev}
            retcode=$?
        fi
        #if retcode is 0, partition already exists.
        #we don't need to do anything.
        #in nonzero cases, either partition didn't exist or is corrupt.
        #we need to reformat it.
        if [ "${retcode}" -ne 0 ]; then
            # Carve out obfl partition
            create_obfl_part_file
            create_fdisk_partition "${OBFL_DEV}" "/tmp/obfl_fdisk.input"
            sync
            loop_dev=$(setup_loop_offset /dev/"${OBFL_DEV}" "p1")
            if [ -z "${loop_dev}" ]; then                                              
                step_log_console "Cannot setup loop device for ${disk}${PART_NUM_REP}"   
            fi                                 
            check_fs_partition "${loop_dev}" "obfl"
            losetup -d "${loop_dev}"
        fi
    fi
}

CMD_PLATFORM=`echo $cmdline | sed 's/^.*platform=//' | cut -d" " -f1`
if [ "$CMD_PLATFORM" != "$PLATFORM" ]; then
   step_log_file "Platform in cmd line = $CMD_PLATFORM. Using $PLATFORM instead"
fi

#If platform hw has some way of discovering itself then OVERRIDE_PLATFORM is set to it
OVERRIDE_PLATFORM=$(override_platform_type ${PLATFORM}  $SIMULATION)
if [ "$OVERRIDE_PLATFORM" != "$PLATFORM" ]; then
    step_log_console "Wrong image for platform $OVERRIDE_PLATFORM. Exiting."
    declare -F pd_notify_sw_fatal_event &>/dev/null && \
        pd_notify_sw_fatal_event "Wrong image for platform $OVERRIDE_PLATFORM. Exiting."
    exit 1
fi

step_log_console "Preparing disk for PLATFORM=$PLATFORM:"

PARTMNT=$(mktemp -d /tmp/partmnt.XXXXXX)
SSHKEY_DIR=$(mktemp -d /tmp/sshkeys.XXXXXX)

# This is where things start to happen
case "$1" in
    start)
        # Get the actual disk device names in case MAIN_DISK & SECONDARY_DISK contain symlinks
        sym_disk=${MAIN_DISK[0]}
        disk=$(get_disk_devices ${sym_disk})

        # pd_punch_watchdog is necessary only for some platforms
        declare -F pd_punch_watchdog && pd_punch_watchdog

        # pd_notify_img_install_started is necessary only for some platforms
        # that need to track HOST events
        declare -F pd_notify_img_install_started && pd_notify_img_install_started

        # make sure install disk exists before carrying on. 
        if [ ! -b /dev/$disk ]; then
            step_log_console "No install device /dev/$disk found.  Unable to install.  Exiting."
            declare -F pd_notify_sw_fatal_event &>/dev/null && \
                pd_notify_sw_fatal_event "No install device /dev/$disk found. Unable to install"
            exit 0
        fi
        step_log_console $"Installer will install image on ${disk}"

        check_for_disk_errors
        #set LV, VG and PARTNUM values
        set_lvm_param

        # Create debug partition(dp), then relocate all logs to dp
        declare -F pd_check_if_debug_part_needed >&107 2>&1
        if [[ $? -eq 0 ]];then
          # Carve out debug partition
          step_log_console "Preparing debug partition ${PLATFORM}"
          if [ -z "${DEBUG_PART_SIZE}" ]; then
             DEBUG_PART_SIZE=${DEBUG_PART_SIZE_DEFAULT}
          fi
          get_dp_offset /dev/${disk}
          prepare_dploop /dev/${disk}
          if [ $? -eq 0 ]; then
             dpflag=TRUE
             logflow ${LOGFLOWDIR}/${LOGFLOWFILE}
             unredirect_output
             redirect_output $LOGDIR2/$LOGFILE
          fi
        fi
        if [ -z "$dpflag" ]; then
           DEBUG_PART_SIZE=0
        fi

        # Remove old volumes
        step_log_console "Removing old volumes"
        remove_old_volumes

        #=======================Prepare sda/vda ===============================
        step_log_console
        if [ "${sym_disk}" != "${disk}" ]; then
           step_log_console $"---Starting to prepare ${sym_disk} [${disk}]---"
        else
           step_log_console $"---Starting to prepare ${disk}---"
        fi

        # Setup the ssh keys 
        create_ssh_keys 
        #get boot type
        declare -F pd_pxe_set_boottype &> /dev/null
        if [ $? -eq 0 ]; then
            pd_pxe_set_boottype
        else  
            BOOTTYPE="legacy"
        fi

        # Create fdisk based partition 
        START=$(date +%s)
        create_part_file_disk /dev/$disk
        check_fix_disk_part ${disk} "${TOTAL_PART_NUM}" "/tmp/fdisk.input"
        mark_part_bootable ${disk} "${disk}${PART_NUM_HOST}"
        END=$(date +%s)
        DIFF=$(( $END - $START ))
        step_log_console "Partition creation on /dev/${disk} took ${DIFF} seconds"

        # Create OBFL partition if it's needed on disk. 
        # currently only augustus cards have it.
        create_obfl_partition
        if [ $? -eq 0 ]; then
          step_log_console "successfully created obfl partition"
        else
          step_log_console "problem with creating obfl partition"
        fi
        
        #create/format/prepare boot partition  
        START=$(date +%s)
        check_fs_partition_ext2 "/dev/${disk}${PART_NUM_HOST}" "SimHostOs"
        if [ $? -ne 0 ]; then
            step_log_console "problem with formatting /dev/${disk}${PART_NUM_HOST}"
        fi
        copy_nbi_to_boot_partition "/dev/${disk}${PART_NUM_HOST}"
        if [ $? -ne 0 ]; then
            step_log_console "problem in copying nbi to boot partition"
        fi
        END=$(date +%s)
        DIFF=$(( $END - $START ))
        step_log_console $"File system creation on /dev/${disk}${PART_NUM_HOST} took ${DIFF} seconds"

        create_sub_part_file_disk_boot
        create_sub_part_file_disk_data

        # create physical volumes
        step_log_file $"pvcreate -ff -y /dev/${disk}${PART_NUM_PV} ${addtl_disks}"
        pvcreate -ff -y /dev/${disk}${PART_NUM_PV} ${addtl_disks} >&107 2>&1

        # Create volume group 
        step_log_file $"vgcreate ${LVG_NAME} /dev/${disk}${PART_NUM_PV} ${addtl_disks}"
        vgcreate ${LVG_NAME} /dev/${disk}${PART_NUM_PV} ${addtl_disks} >&107 2>&1

        #=======================Prepare host volumes=============================
        step_log_console $'---Starting to prepare host logical volume---'
        create_host_boot_and_data_lvs
        check_fs_partition "${HOST_LV_NAME_FULL}" "HostOs"
        install_host "${HOST_LV_NAME_FULL}" "/"

        # Save accumulated log to persistent disk & from on, save direct to disk
        save_install_log "${HOST_LV_NAME_DATA_LOG_FULL}"

        #=======================Prepare calvados volumes=========================
        create_sysadmin_boot_and_data_lv
        step_log_console $"File system creation on /dev/${CALVADOS_LV_PART} took ${DIFF} seconds"
        loop_dev=$(setup_loop_offset "/dev/${CALVADOS_LV_PART}" "p1")
        partcal_offset=$(find_fs_offset "/dev/${CALVADOS_LV_PART}" "p1")
        step_log_console $"Install sysadmin-vm image on /dev/${CALVADOS_LV_PART}"

        install_calvados "/dev/${CALVADOS_LV_PART}" "/" "${partcal_offset}" 

        #=======================Prepare repository===============================
        create_and_prep_repository $disk $PART_NUM_REP

        #================= prepare for reboot ========================

        # pd_notify_img_install_done is necessary only for some platforms
        # that need to track HOST events
        declare -F pd_notify_img_install_done && pd_notify_img_install_done

        # Install finished, Save install log before system reboot
        step_log_console $"Install finished on ${disk}"
        step_log_console $"rebooting now"
        close_install_log
        call_reboot
        ;;

    stop)
        exit 0
        ;;

    restart | reload | force-reload | status | condrestart | try-restart)
        usage
        exit 3
        ;;

    *)
        usage
        exit 2
        ;;
esac
