#!/bin/bash

# SSD firmware update functions
#
# This script defines functions
# for SSD firmware upgrade/downgrade.
#
# Copyright (c) 2017-2021 by Cisco Systems, Inc.
# All rights reserved.
#

source /etc/init.d/spirit_pd.sh
source /etc/rc.d/init.d/pd-functions

HDPARM=/sbin/hdparm
HDPARM_IDENTIFY="${HDPARM} -i "
HDPARM_IDENTIFY_DETAIL="${HDPARM} -I "
HDPARM_FWUPDATE_CMD="${HDPARM} --fwdownload-mode7"
HDPARM_FWUPDATE_CMD_PARAMS=" --yes-i-know-what-i-am-doing --please-destroy-my-drive"
SSD_BIN_FILE_PATH="/opt/cisco/calvados/bin/"
declare -A ssd_table
ssd_table_count=0


#
#            SSD FW Upgrade Table
#------------------------------------------------
#    Vendor    |    From      |       To        |
#------------------------------------------------
#    SMART     | FW962 (702k) |  FW1091 (705a)  |
#------------------------------------------------
#    SMART     |    FW1026    |  FW1092 (705b)  |
#------------------------------------------------
#    Unigen    |  S5FAM017    |     S5FAM031    |
#------------------------------------------------
#    Unigen    |  S5FAM018    |     S5FAM031    |
#------------------------------------------------
#    Unigen    |  S5FAM022    |     S5FAM031    |
#------------------------------------------------
#    Intel     |  N2010112    |     N2010121    |
#------------------------------------------------
#    Micron    |  D0MU027     |     D0MU051     |
#------------------------------------------------
#    Micron    |  M0MU020     |     M0MU031     |
#------------------------------------------------
#    Micron    |  D0MU037     |     D0MU051     |
#------------------------------------------------
#    Intel     |  XC311102    |     XC311120    |
#------------------------------------------------
#    Micron    |  D0MU051     |     D0MU075     |
#------------------------------------------------
#    Micron    |  D0MU071     |     D0MU075     |
#------------------------------------------------
#
function init_ssd_upgrade_table () 
{
    # Upgrade from FW962(7.02k) to FW1091(7.05a) 
    ssd_table[0,0]="Ver7.02k FW962"
    ssd_table[0,1]="ssdfw_SMART_Ver7.05a_v7.05.bin"
    let "ssd_table_count = $ssd_table_count + 1"
    
    # Upgrade from FW1026 to FW1092(7.05b) 
    ssd_table[1,0]="FW1026"
    ssd_table[1,1]="ssdfw_SMART_Ver7.05b_v7.05.bin"
    let "ssd_table_count = $ssd_table_count + 1"
    
    # Upgrade from S5FAM017, S5FAM018, S5FAMU22 to S5FAM031 
    #ssd_table[2,0]="S5FAM017 S5FAM018 S5FAMU22"
    #ssd_table[2,1]="ssd_Unigen_S5FAM031_v3.10.bin"
    #let "ssd_table_count = $ssd_table_count + 1"
    
    # Intel SSD 3520
    # Upgrade from N2010112 to N2010121
    ssd_table[2,0]="N2010112"
    ssd_table[2,1]="ssdfw_INTEL_N2010121_v1.21.bin"
    let "ssd_table_count = $ssd_table_count + 1"

    # Micron SSD 5100 M.2
    # Upgrade from D0MU027 to D0MU051
    ssd_table[3,0]="D0MU027"
    ssd_table[3,1]="ssdfw_MICRON_D0MU051_v5.01.bin"
    let "ssd_table_count = $ssd_table_count + 1"

    # Micron SSD 1100 M.2
    # Upgrade from M0MU020 to M0MU031
    ssd_table[4,0]="M0MU020"
    ssd_table[4,1]="ssdfw_MICRON_M0MU031_v3.01.bin"
    let "ssd_table_count = $ssd_table_count + 1"

    # Micron SSD 5100 M.2
    # Upgrade from D0MU037 to D0MU051
    ssd_table[5,0]="D0MU037"
    ssd_table[5,1]="ssdfw_MICRON_D0MU051_v5.01.bin"
    let "ssd_table_count = $ssd_table_count + 1"

    # Intel SSD D3-S4510
    # Upgrade from XC311102 to XC311120
    ssd_table[6,0]="XC311102"
    ssd_table[6,1]="ssdfw_INTEL_XC311120_v1.12.bin"
    let "ssd_table_count = $ssd_table_count + 1"

    # Micron SSD 5100 M.2
    # Upgrade from D0MU051 to D0MU075
    ssd_table[7,0]="D0MU051"
    ssd_table[7,1]="ssdfw_MICRON_D0MU075_v7.05.bin"
    let "ssd_table_count = $ssd_table_count + 1"

    # Micron SSD 5100 M.2
    # Upgrade from D0MU071 to D0MU075
    ssd_table[8,0]="D0MU071"
    ssd_table[8,1]="ssdfw_MICRON_D0MU075_v7.05.bin"
    let "ssd_table_count = $ssd_table_count + 1"
}

#
#            SSD FW Downgrade Table
#-----------------------------------------------------
#    Vendor    |    From       |         To          |
#-----------------------------------------------------
#    SMART     | FW1091 (705a) | 705y -> FW962 (702k)|
#-----------------------------------------------------
#    SMART     |    FW1092     |       FW1026        |
#-----------------------------------------------------
#    Unigen    |   S5FAM031    |       S5FAM017      |
#-----------------------------------------------------
#
function init_ssd_downgrade_table () 
{
    # Downgrade from FW1091 (7.05a) to FW962 (7.02k)
    ssd_table[0,0]="Ver7.05a FW1091"
    ssd_table[0,1]="ssdfw_SMART_Ver7.05y_v7.05.bin"
    ssd_table[0,2]="ssdfw_SMART_Ver7.02k_v7.02.bin"
    let "ssd_table_count = $ssd_table_count + 1"
    
    # Downgrade from FW1092 (7.05b) to FW1026
    ssd_table[1,0]="Ver7.05b FW1092"
    ssd_table[1,1]="ssdfw_SMART_FW1026_v7.02.bin"
    let "ssd_table_count = $ssd_table_count + 1"
    
    # Downgrade from S5FAM031 to S5FAM017
    ssd_table[2,0]="S5FAM031"
    ssd_table[2,1]="ssdfw_Unigen_S5FAM017_v1.70.bin"
    let "ssd_table_count = $ssd_table_count + 1"
}

function get_fw_file_by_fw_ver () 
{
    local fw=$1
    local fw_pos=$2
    local tbl_fw
    local tbl_fw_bin=""
    local dev
    
    let "fw_pos = $fw_pos + 1"

    for (( dev=0; dev<$ssd_table_count; dev++ ))
    do
        for tbl_fw in ${ssd_table[$dev,0]}
        do
            if [[ "${tbl_fw}" == "${fw}" ]]; then
                tbl_fw_bin=${ssd_table[$dev,$fw_pos]}
                break
            fi
        done
   done
   
   echo ${tbl_fw_bin}
}

function check_ssd_fw_change_required () 
{
    local fw=$1
    local tbl_fw
    local dev
    
    for (( dev=0; dev<$ssd_table_count; dev++ ))
    do
        for tbl_fw in ${ssd_table[$dev,0]}
        do
            if [[ "${tbl_fw}" == "${fw}" ]]; then
                return 0 
            fi
        done
    done
    
    return 1 
}

function get_ssd_type () 
{
    local fw=$1
    local tbl_fw
    local dev
    local vendor="Unknown"
    
    for (( dev=0; dev<$ssd_table_count; dev++ ))
    do
        for tbl_fw in ${ssd_table[$dev,0]}
        do
            if [[ "${tbl_fw}" == "${fw}" ]]; then
                vendor=$(${ssd_table[$dev,1]} | awk  -F',' '{ print $2 ; }')
            fi
        done
    done
    
    return $vendor
}


function ssd_trim_whitepspaces () {
    local var="$*"
    
    # Remove front whitespaces
    var="${var#"${var%%[![:space:]]*}"}"
    
    # Remove trail whitespaces
    var="${var%"${var##*[![:space:]]}"}"   
    
    echo -n "$var"
}


# Parse hdparm -i <dev> | grep Model
# SMART Example: Model=SMART SATA SGSLM32GEBCCTHD01, FwRev=FW1091, SerialNo=STP1724028J
# Unigen Example: Model=UGBA1TPH32H0S2-PNN-CTF, FwRev=S5FAM017, SerialNo=11000255120
# field 2 is Model
# field 3 is FwRev
function get_ssd_model () 
{
    local model
    local dev=$1

    model=$(${HDPARM_IDENTIFY} ${dev} | grep Model | awk  -F',' '{ print $1 ; }' | awk  -F'=' '{ print $2 ; }')

    echo ${model}
}

# Parse hdparm -I <dev> | grep Firmware
function get_ssd_fw_ver () 
{
    local fw_ver
    local dev=$1
    
    fw_ver=$(${HDPARM_IDENTIFY_DETAIL} ${dev} | grep  Firmware | awk  -F':' '{ print $2 ; }')
    
    fw_ver=$(ssd_trim_whitepspaces ${fw_ver})

    echo ${fw_ver}
}

function get_ssd_drives ()
{
    local drives
    
    if [[ "${BOARDTYPE}" == "RP" ]]; then
        drives="/dev/sda /dev/sdb"
    else     
        drives="/dev/sda"
    fi

    echo ${drives}
}

function do_hdparm ()
{
    local fw_path=$1
    local dev=$2
    local fw_ver=$3
    local new_fw_ver
    local rc;

    ${HDPARM_FWUPDATE_CMD} ${fw_path} ${HDPARM_FWUPDATE_CMD_PARAMS} ${dev} > /dev/null 2>&1
    rc=$?
    if [ $rc != 0 ]; then
        return $rc
    fi

    sleep 1
    new_fw_ver=$(get_ssd_fw_ver ${dev})
    if [ ${fw_ver} == ${new_fw_ver} ]; then
        return 1
    fi
    echo "$(date): ${dev} fw ${oper} from ${fw_ver} to ${new_fw_ver} success"
    return 0
}

function ssd_update_fw () 
{
    local dev=$1
    local oper=$2
    local model
    local fw_ver
    local new_fw_ver=$3
    local fw_path
    local second_fw
    local fw
    local rc
    
    model=$(get_ssd_model ${dev})
    fw_ver=$(get_ssd_fw_ver ${dev})
    if [ ${new_fw_ver} != "all" ]; then
        if [ ${fw_ver} != ${new_fw_ver} ]; then
            return 0
        fi
    fi
    check_ssd_fw_change_required ${fw_ver}
    if [ $? -ne 0 ]; then
        echo "$(date): Skip ${oper}, ${dev} fw ${fw_ver} upto date"
        return 0
    fi
    
    fw=$(get_fw_file_by_fw_ver ${fw_ver} 0) 
    fw_path="${SSD_BIN_FILE_PATH}${fw}"
    
    if [ -z "${fw_path}" ] || [ ! -f ${fw_path} ]; then
        echo "$(date): ${dev} ${oper} fw file ${fw_path} not found"
        return 1
    fi

    do_hdparm ${fw_path} ${dev} ${fw_ver}
    rc=$?
    if [ $rc != 0 ]; then
        return 1
    else 
        if [[ ${oper} == "downgrade" ]]; then
            second_fw=$(get_fw_file_by_fw_ver ${fw_ver} 1) 
            fw_path="${SSD_BIN_FILE_PATH}${second_fw}"
            if [[ -z "${second_fw}" ]]; then
                echo "$(date): ${dev} level 2 ${oper} not required ${second_fw}"
                return 0
            fi
        
            sleep 0.5 
        
            fw_ver=$(get_ssd_fw_ver ${dev})
            echo "$(date): ${dev} level 2 fw ${oper} using ${second_fw}"
            do_hdparm ${fw_path} ${dev} ${fw_ver}
            rc=$?
            if [ $rc != 0 ]; then
                echo "$(date): ${dev} level 2 hdparm failed with error ${rc}!!"
                return 1
            else
                new_fw_ver=$(get_ssd_fw_ver ${dev})
                echo "$(date): ${dev} level 2 fw ${oper} from ${fw_ver} to ${new_fw_ver} success"
            fi
        fi
    fi

    return 0
}


function ssd_update_fw_exec () 
{
    local oper=$1
    local firmware=$2
    local drive
    local drives=$(get_ssd_drives)
    local upg_rc
    local rc=0
    local install
    local retry_cnt=900
    
    if [[ ${oper} == "upgrade" ]]; then
        init_ssd_upgrade_table
    else    
        init_ssd_downgrade_table
    fi

    install=$(cat /proc/cmdline | grep install=)
    if [ -n "${install}" ]; then
        retry_cnt=3
    fi

    for drive in ${drives}
    do
        upg_rc=1
        for (( retry = 1; retry <= $retry_cnt; retry++ ))
        do
            ssd_update_fw ${drive} ${oper} ${firmware} 
            if [ $? -eq 0 ]; then
                echo "$(date): ${drive} fw ${oper} successful in $retry retries"
                upg_rc=0
                break
            fi
            sleep 0.5
        done
        if [ $upg_rc -eq 1 ]; then
            echo "$(date): ${drive} fw ${oper} failed. Total retry attempts:$retry"
            rc=1
        fi
    done 

    return $rc
}
function ssd_update_fw_all
{
    ssd_update_fw_exec "upgrade" "FW1026"
    ssd_update_fw_exec "upgrade" "N2010112"
    ssd_update_fw_exec "upgrade" "D0MU027"
    ssd_update_fw_exec "upgrade" "D0MU037"
    ssd_update_fw_exec "upgrade" "XC311102"
    ssd_update_fw_exec "upgrade" "D0MU051"
    ssd_update_fw_exec "upgrade" "D0MU071"
}

