#!/bin/bash
#
# This script init sm15 fabric asic
#
# Copyright (c) 2014-2017 by Cisco Systems, Inc.
# All rights reserved.

fab_log_file=/var/log/fab.log

source /etc/init.d/pd-functions
source /etc/init.d/spirit_pd.sh
myvmtype=`cat /proc/cmdline | sed 's/^.*vmtype=//' | cut -d" " -f1`
if [ "$myvmtype" == "hostos" ]; then
    myshowpcie="/sbin/dopcie"
else
    myshowpcie="/pkg/sbin/dopcie_static"
fi

platform_is_virtvm
rc=$?
if [[ $rc == "1" ]]; then
    VIRT_METHOD=vm
else
    VIRT_METHOD=lxc
fi


# args : offset and length
ipu_read() {
    local ipu_base=${IPU_BASE}
    IPU_BASE_d=`printf "%d" $ipu_base`
    IPU_OFFSET_d=`printf "%d" $1`
    IPU_ADDR_d=`expr $IPU_BASE_d + $IPU_OFFSET_d`
    IPU_ADDR_x=`printf "%x" $IPU_ADDR_d`
    IPU_ADDR_LEN_x=`printf "%x" $2`
    pcimemread $IPU_ADDR_x $IPU_ADDR_LEN_x | grep -v PCI
}

# args : offset, length and value
ipu_write() {
    local ipu_base=${IPU_BASE}
    IPU_BASE_d=`printf "%d" $ipu_base`
    IPU_OFFSET_d=`printf "%d" $1`
    IPU_ADDR_d=`expr $IPU_BASE_d + $IPU_OFFSET_d`
    IPU_ADDR_x=`printf "%x" $IPU_ADDR_d`
    IPU_ADDR_LEN_x=`printf "%x" $2`
    IPU_ADDR_VAL_x=`printf "%x" $3`
    pcimemwrite $IPU_ADDR_x $IPU_ADDR_LEN_x $IPU_ADDR_VAL_x | grep -v PCI | grep -v "Write at Address"
}

# args : offset 32bit wide 
ipu_r32() {
    ipu_read $1 4 | awk '{print $3}'
}

ipu_r32_poll() {
    local addr=$1
    local expected=$2
    local secs=$3
    local tries
    local rval
    sleep 1 # min sleep
    let "secs = $secs - 1"
    let "tries = $secs * 4" # poll the rest at 0.25s interval
    while [  "$tries" != "0" ]; do
        sleep 0.25
        rval=$( ipu_r32 $addr )
        if [[ "$rval" == "$expected" ]]; then
            echo $rval
            return
        fi
        let "tries = $tries - 1"
    done
    echo $rval
}

# args : offset 32bit wide and 32 bit value
ipu_w32() {
    ipu_write $1 4 $2 
}

ipu_mbox_offset() {
    get_board_type
    local MBOXOFF=$1

    if [ "$BOARDTYPE" == "RP" ]; then
        echo $(printf "0x%x" $(($(get_ipu_block_offset MBOX 1) + 0x100 + $MBOXOFF)))
    else
        echo $(printf "0x%x" $(($(get_ipu_block_offset MBOX 1) + 0x120 + $MBOXOFF)))
    fi
}

mbox_reset () {
    local mbox_offset=`ipu_mbox_offset $4`
    local ipu_base=${IPU_BASE}
    local IPU_BASE_d=`printf "%d" $ipu_base`
    local IPU_OFFSET_d=`printf "%d" $mbox_offset`
    local MBOX_PADDR_d=`expr $IPU_BASE_d + $IPU_OFFSET_d`
    local MBOX_PADDR_x=`printf "%x" $MBOX_PADDR_d`

    echo "$myshowpcie integrated $1 $2 $3 $MBOX_PADDR_x $5 $6" >> $fab_log_file
    kmsg_log "$myshowpcie integrated $1 $2 $3 $MBOX_PADDR_x $5 $6"

    PCIE_DEBUG=/misc/scratch $myshowpcie integrated $1 $2 $3 $MBOX_PADDR_x $5 $6
    local rc=$?

    echo "$myshowpcie integrated $1 $2 $3 $MBOX_PADDR_x $5 $6 rc=$rc" >> $fab_log_file
    kmsg_log "$myshowpcie integrated $1 $2 $3 $MBOX_PADDR_x $5 $6 rc=$rc"
}

fabric_device_rescan() 
{
   echo $1 $2 $3 >> $fab_log_file
   if [ "$VIRT_METHOD" == "vm" ]
   then
       kmsg_log "Bring device online $1 $2 $3"
       $myshowpcie make_online $1 $2 $3
       if [[ $? != 0 ]]; then
           kmsg_log "$1 $2 $3  device online failed $?"
           echo "$1 $2 $3  showpcie make_online error" >> $fab_log_file
           return
       fi
       kmsg_log "$1 $2 $3  device online done"
   else
       kmsg_log "Rescan device $1 $2 $3"
       $myshowpcie sysfs_rescan_device $1 $2 $3
       if [[ $? != 0 ]]; then
           kmsg_log "$1 $2 $3  showpcie rescan device error $?"
           echo "$1 $2 $3  showpcie rescan device error" >> $fab_log_file
           return
       fi
   fi
}

fabric_device_remove()
{
   echo $1 $2 $3 >> $fab_log_file
   if [ "$VIRT_METHOD" == "vm" ]
   then
       kmsg_log "Bring device offline $1 $2 $3"
       $myshowpcie make_offline $1 $2 $3
       if [[ $? != 0 ]]; then
           kmsg_log "$1 $2 $3  device offline failed $?"
           echo "$1 $2 $3  showpcie make_offline error" >> $fab_log_file
           return
       fi
       kmsg_log "$1 $2 $3  device offline done"
       sleep 2
   else
       kmsg_log "Remove device $1 $2 $3"
       $myshowpcie sysfs_remove_ep $1 $2 $3
       if [[ $? != 0 ]]; then
           kmsg_log "$1 $2 $3  showpcie remove device error $?"
           echo "$1 $2 $3  showpcie remove device error" >> $fab_log_file
           return
       fi
   fi
}

sm15_pci_bdf() {
    lspci -d1137:00ca | awk '{print $1}'
}

sm15_find() {
   local index=0
 
   if [ "$VIRT_METHOD" == "vm" ]
   then 
       fabric_device_rescan 1137 ca $index 
       return 1
   fi

   local sm15bdf=`sm15_pci_bdf`
   local retry=1
   while  [ "$sm15bdf" == "" ]
   do
        fabric_device_rescan 1137 ca $index
        sleep 1
        sm15bdf=`sm15_pci_bdf`
        retry=`expr $retry + 1`
        if [[ $retry -eq 10 ]]
        then
            return 0
        fi
   done
   return 1

}

sm15_init() {
    local SM15_MBOXOFF=0x20
    if [ "$VIRT_METHOD" == "vm" ]
    then
        mbox_reset 1137 ca 0 $SM15_MBOXOFF 0x00000901 3
        return
    fi

    local mbox_offset=`ipu_mbox_offset $SM15_MBOXOFF`

    # remove    
    local status=`ipu_r32 $mbox_offset`
    fabric_device_remove 1137 ca 0
    # reset by issuing command on mailbox
    # ARM-status 00, SM15 instance 00, SM15 device 09, op init 01
    ipu_w32 $mbox_offset 0x00000901

    status=$( ipu_r32_poll $mbox_offset "80000901" 3 )

    echo "SM15 reset status $status" >> $fab_log_file
    if [ "$status" != "80000901" ]
    then 
        echo "SM15 reset failed" >> $fab_log_file
        kmsg_log "SM15 reset failed status = $status"
    else
        echo "SM15 reset ok" >> $fab_log_file
        kmsg_log "SM15 reset ok"
    fi   
    sleep 2
    sm15_find
    sm15_ret=$?
    if [[ $sm15_ret -eq 1 ]]
    then  
        echo "SM15 found ok" >> $fab_log_file
        kmsg_log "SM15 found ok"
    else
        echo "SM15 not found " >> $fab_log_file
        kmsg_log "SM15 not found"
    fi
}

lgt_pci_bdf() {
    lspci -d1137:003b | awk '{print $1}'
}

lgt_find() {
   local index=0

   if [ "$VIRT_METHOD" == "vm" ]
   then
       fabric_device_rescan 1137 3b $index
       return 1
   fi
   
   local lgtbdf=`lgt_pci_bdf`
   local retry=1
   while  [ "$lgtbdf" == "" ]
   do
        fabric_device_rescan 1137 3b $index
        sleep 1
        lgtbdf=`lgt_pci_bdf`
        retry=`expr $retry + 1`
        if [[ $retry -eq 10 ]]
        then
            return 0
        fi
   done
   return 1

}


lgt_init() {
    local LGT_MBOXOFF=0x40
    if [ "$VIRT_METHOD" == "vm" ]
    then
        mbox_reset 1137 3b 0 $LGT_MBOXOFF 0x00000a01 3
        if [[ $? != 0 ]]; then
            kmsg_log "LGT mbox reset failed - Retry"
            mbox_reset 1137 3b 0 $LGT_MBOXOFF 0x00000a01 3
        fi
        return
    fi

    local mbox_offset=`ipu_mbox_offset $LGT_MBOXOFF`

    # remove    
    local status=`ipu_r32 $mbox_offset`
    fabric_device_remove 1137 3b 0
    # reset by issuing command on mailbox
    # ARM-status 00, LGT instance 00, LGT device 0a, op init 01
    ipu_w32 $mbox_offset 0x00000a01

    status=$( ipu_r32_poll $mbox_offset "80000a01" 3 )

    echo "LGT reset status $status" >> $fab_log_file
    if [ "$status" != "80000a01" ]
    then 
        echo "LGT reset failed status=$status" >> $fab_log_file
        kmsg_log "LGT reset failed status = $status"
    else
        echo "LGT reset ok" >> $fab_log_file
        kmsg_log "LGT reset ok"
    fi    
    sleep 2 
    lgt_find
    lgt_ret=$?
    if [[ $lgt_ret -eq 1 ]]
    then
        echo "LGT found ok" >> $fab_log_file
        kmsg_log "LGT found ok"
    else
        echo "LGT not found " >> $fab_log_file
        kmsg_log "LGT not found"
    fi

}

tsh_init()
{
    local data=$(get_card_specific_value FIA0_OFF)      
    local offset=0x10000
    local TSH_MBOXOFF=0x60

    num_fia=$(get_card_specific_value NUM_FIA)
    echo "number of fia $num_fia" >> $fab_log_file
    kmsg_log "Total FIAs $num_fia"

    if [ "$VIRT_METHOD" == "vm" ]
    then
        for ((i=0; i<$num_fia; i++)); do
            mbox_reset 1137 82 $i $TSH_MBOXOFF $data 3
            data=$(printf "0x%x" $(($data+$offset)))
        done
        return
    fi

    local devices=$(lspci -nn | grep "1137:0082" | awk '{print $1}')
    local mbox_offset=`ipu_mbox_offset $TSH_MBOXOFF`

    for ((i=0; i<$num_fia; i++)); do
        fabric_device_remove 1137 82 $i 
    done
    for ((i=0; i<$num_fia; i++)); do
        local status=`ipu_r32 $mbox_offset`
        local expected_result=$(printf "%x" $(($data+0x80000000)))

        # reset by issuing command on mailbox
        # ARM-status 00, TSH instance 00, TSH device 03, op init 01
        echo " offset $mbox_offset data $data" >> $fab_log_file
        ipu_w32 $mbox_offset $data 

        status=$( ipu_r32_poll $mbox_offset expected_result 3 )

        echo "TSH reset status $status" >> $fab_log_file
        if [ "$status" != "$expected_result" ]
        then
            echo "TSH reset failed $status $expected_result" >> $fab_log_file
            kmsg_log "TSH $i reset failed status = $status"
        else
            echo "TSH reset success $status $expected_result" >> $fab_log_file
            kmsg_log "TSH $i reset ok"
        fi
        data=$(printf "0x%x" $(($data+$offset)))
    done
    sleep 2
    for ((i=0; i<$num_fia; i++)); do
        fabric_device_rescan 1137 82 $i 
    done 
}

fabric_init() {
    get_board_type

    echo "`date` fabric init for $BOARDTYPE $VIRT_METHOD" > $fab_log_file

    set_ipu_base
    if [[ $? != 0 ]]; then
        echo Cannot find IPU base in fabric_init >> $fab_log_file
        echo Skipping fabric_init >> $fab_log_file
        return
    else
        echo IPU BASE: ${IPU_BASE} >> $fab_log_file
    fi

    if [ "$BOARDTYPE" == "RP" ]; then
        lgt_init
    else
        sm15_init
        lgt_init
        tsh_init
    fi

    sleep 2

}

