#!/bin/bash
# 
# lxc - libvirt hook script
#
#
# This script is used for setting up container resources and this is
# PI/PD seperated.
# lxc_hook.sh
# |__ ${WS}/calvados/<platform>_pkg/boot/scripts/Pd-lxchook-functions.sh
#
# Input arguments expected: 4
# $1 : Name of the LXC, used in the XML.
#      eg: sysadmin/default-sdr--1
# $2 : Operation of the LXC.
#      (a) prepare
#      (b) start
#      (c) init_start (Cisco added)
#      (d) started (Cisco modified)
#      (e) init_shutdown (Cisco added)
#      (f) init_destroy (Cisco added)
#      (g) stopped
#      (h) release
#      (i) reconnect
# $3 : Sub-operation of the LXC mentioned in $2
#      (begin/end)
# $4 : One additional argument.
#      For operation init_start/started/init_shutdown/init_destroy, 
#      we get the PID of /sbin/init of container.
#
# Copyright (c) 2014-2017 by Cisco Systems, Inc.
#

PROCESS_NAME="LXC_HOOK"
######################################################################
# DANGER: This log file is used by fretta sanity to measure boot time.
# Ensure that it does not get overwritten by any subsequently sourced
# scripts.
######################################################################
LXC_HOOK_LOG_FILE="/var/log/lxc_hook_script.log"

source /etc/init.d/spirit-functions

timestamp=$( date +"%b %d %T" )

CGROUP_MNT="/dev/cgroup"
VIRT_GP="machine"
VIRT_PROC="libvirt-lxc"

if [ -f /etc/init.d/pd-functions ]; then
    source /etc/init.d/pd-functions
fi

if [ -f /etc/init.d/pd-lxchook-functions.sh ]; then
    source /etc/init.d/pd-lxchook-functions.sh
fi

if [ -f /etc/rc.d/init.d/app_hosting_lxc_hook.sh ]; then
    source /etc/rc.d/init.d/app_hosting_lxc_hook.sh
fi

if [ -f /etc/init.d/app-hosting-functions ]; then
    source /etc/init.d/app-hosting-functions
fi

if [ -f /etc/init.d/spirit_log.sh ]; then
    source /etc/init.d/spirit_log.sh
fi

#
# Wrapper function to log messages
#
function log_msg()
{
    declare -F platform_log &>/dev/null && platform_log "$@"
    printf "%s %s\n" "$timestamp" "$@" >> $LXC_HOOK_LOG_FILE
}
readonly -f log_msg

#
# Wrapper function to log errors. In addition to the existing logfile,
# we also redirect to syslog also.
#
function log_err()
{
    declare -F platform_log &>/dev/null && platform_log "$@"
    printf "%s %s\n" "$timestamp" "$@" >> $LXC_HOOK_LOG_FILE
    logger -t $PROCESS_NAME -p 3 "$@"
}
readonly -f log_err

#
# Wrapper function to execute shell commands
#
function execute_cmd()
{
    $@ >>$LXC_HOOK_LOG_FILE 2>&1
}
readonly -f execute_cmd

#
# Identifies the XR rootfs and mounts the same.
# This script can also perform, pre-launch setup, if required.
#
function prelaunch_setup()
{
    local vm_name=$1

    /usr/bin/xr_pre_launch.sh "$vm_name"
}
readonly -f prelaunch_setup

#
# cleanup_bind_mounts
#
# For every LXC, we start, the kernel create a copy of the mount tree of the 
# HOSTOS for the libvirt_lxc controller process.
# If sysadmin is started first and then XR is started:
#           Sysadmin's bind mount on hostOS is copied to XR's mount tree.
# If XR is already running and sysadmin is restarted:
#           XR's bind mount on hostOS is copied to sysadmin's mount tree.
#
# What this basically does is, increments the refcnt for the /dev/loop device 
# and the mount directory. So, if we shutdown sysadmin, the /dev/loop 
# associated with the bindmount is not released, since a refcnt is maintained 
# for that bindmount in XR's mount tree.
#
# How do we solve this?
# When a LXC is started, we look for all other active LXC's and parse for the
# <filesystem> tag, those represent the bind mounts. We unmount them specifically in the
# context of this LXC, only if it's a mount point
# NOTE: This will not affect the HOSTOS.
#
# !!!!! This function should be executed, only for "init_start" !!!!!
#
function cleanup_bind_mounts()
{
    local vm_name=$1
    
    for files in `ls /var/run/libvirt/lxc/*.xml | grep -v $vm_name`
    do
        for line in `cat $files | grep -A 4 "<filesystem" | \
                     grep "<source" | cut -d"'" -f2`
        do
            cat /proc/mounts | awk '{ print $2 }' | \
                grep "$line" 2>&1 >/dev/null
            if [[ $? -eq 0 ]]; then
                # Recursively unmount the bind mounts  
                log_msg "cleanup_bind_mounts: umounting $line for $vm_name"
                umount "$line" -d -R
            fi
        done
    done

    return 0
}
readonly -f cleanup_bind_mounts

#
# Move kimctrl netdevice to XR LXC 
#
function lxc_set_kimctrl_intf_hook()
{
    local vm_name=$1
    local vm_op=$2
    local vm_subop=$3
    local vm_initpid=$4
    local kim_dev="kimctrl"

    #
    # We only need kim where we have app hosting.
    #
    netstack_is_enabled "$vm_name"
    if [[ $? -ne 0 ]]; then
        log_msg "Move of $kim_dev not needed on VM $vm_name"
        true
        return
    fi 

    #
    # move 'kimctrl' netdevice, which is created by lcndklm.ko, to
    # XR container.
    #
    log_msg "Moving $kim_dev netdevice to XR LXC for VM $vm_name"
    CMD="ip link set $kim_dev netns $vm_initpid"
    printf "%s CMD: %s\n" "$timestamp" "$CMD" >> $LXC_HOOK_LOG_FILE
    platform_log_exec "$CMD"
    if [[ $? -ne 0 ]]; then
        log_msg "Failed CMD: $CMD"
        #
        # check if netdevice exists
        #
        CMD="ifconfig $kim_dev"
        printf "%s CMD: %s\n" "$timestamp" "$CMD" >> $LXC_HOOK_LOG_FILE
        platform_log_exec "$CMD"
        if [[ $? -ne 0 ]]; then
            log_msg "$kim_dev netdevice not found"
        else
            log_msg "$kim_dev netdevice is present"
        fi
    fi

    return 0
}
readonly -f lxc_set_kimctrl_intf_hook

#
# setup_sched_runtime
#
# At present, we set the system defaults for XR LXC.
# It inherits the values from the parent cgroup.
#
function setup_sched_runtime
{
    local vm_name=$1
    local ret_code=0
    local source_file
    local target_file

    #
    # Setup parent cgroup first, since this setting is hierarchical.
    # (i,e) Child should always have a value that is a subset of the
    #       parent.
    #
    source_file="$CGROUP_MNT/cpu/cpu.rt_runtime_us"
    target_file="$CGROUP_MNT/cpu/$VIRT_GP/cpu.rt_runtime_us"
    cat "$source_file" > "$target_file"
    ret_code=$?
    if [[ "$ret_code" -ne 0 ]]; then
        log_err "$source_file to $target_file"
        log_err "setup_sched_runtime: Failed to set rt_runtime for machine"
        return $ret_code
    fi

    #
    # Setup for the VM now
    #
    source_file="$CGROUP_MNT/cpu/cpu.rt_runtime_us"
    target_file="$CGROUP_MNT/cpu/$VIRT_GP/$vm_name.$VIRT_PROC/cpu.rt_runtime_us"

    declare -F pd_handle_sched_runtime &>/dev/null
    ret_code=$?
    if [[ "$ret_code" -ne 0 ]]; then
         # PI handling as PD function is not defined
        cat "$source_file" > "$target_file"
    else
        pd_handle_sched_runtime $vm_name $source_file $target_file
    fi    
    ret_code=$?

    if [[ "$ret_code" -ne 0 ]]; then
        log_err "$source_file to $target_file"
        log_err "setup_sched_runtime: Failed to set rt_runtime for VM $vm_name"
        return $ret_code
    fi

    log_msg "setup_sched_runtime: Set rt_runtime for VM $vm_name"

    return $ret_code
}

#
# handle_start
#
function handle_start()
{
    local vm_name=$1
    local vm_op=$2
    local vm_subop=$3
    local vm_initpid=$4
    local func_exit_code=0

    log_msg "Calling start setup for LXC $vm_name"

    declare -F pd_handle_start &> /dev/null
    if [[ $? -eq 0 ]]; then
        pd_handle_start $vm_name $vm_op $vm_subop $vm_initpid
        func_exit_code=$?
    fi

    if [[ $func_exit_code -ne 0 ]]; then
        log_err "Failed to handle start for VM $vm_name"
    fi

    return $func_exit_code
}
readonly -f handle_start

#
# handle_init_start
#
function handle_init_start()
{
    local vm_name=$1
    local vm_op=$2
    local vm_subop=$3
    local vm_initpid=$4
    local func_exit_code=0
    local devallow_retval=0
    
    log_msg "Providing device permissions for LXC"
    /etc/init.d/dev-allow $vm_name
    devallow_retval=$?

    declare -F app_host_is_enabled &>/dev/null && app_host_is_enabled
    if [[ $? -ne 0 ]]; then
        # In case of LCs and on platforms which do not support app hosting
        # return if the dev-allow failed
        if [[ $devallow_retval -ne 0 ]]; then
            log_err "dev-allow failed for VM $vm_name"
            return $devallow_retval
        fi
    else
        # On RPs which have platform support for app hosting
        # return immediately if dev-allow failed for system LXCs and 
        # continue for third party LXCs
        case "$vm_name" in
        sysadmin | default-sdr--*)
            if [[ $devallow_retval -ne 0 ]]; then
                log_err "dev-allow failed for VM $vm_name"
                return $devallow_retval
            fi
            ;;
        *)
            ;;
        esac
    fi

    cleanup_bind_mounts "$vm_name"
    if [[ $? -ne 0 ]]; then
        log_err "cleanup_bind_mounts: Failed for VM $vm_name, continuing... "
    fi

    case "$vm_name" in
    sysadmin)
        lxc_set_sysadmin_intf_hook $vm_name $vm_op $vm_subop $vm_initpid
        func_exit_code=$?
        ;;
 
    default-sdr--*)
        lxc_set_kimctrl_intf_hook $vm_name $vm_op $vm_subop $vm_initpid

        if [[ "$func_exit_code" -eq 0 ]]; then
            setup_sched_runtime $vm_name $vm_op $vm_subop $vm_initpid
            func_exit_code=$?
        fi

        if [[ "$func_exit_code" -eq 0 ]]; then
            lxc_set_sdr_intf_hook $vm_name $vm_op $vm_subop $vm_initpid
            func_exit_code=$?
        fi
        ;;
 
    *)
        #Nothing to do
        ;;
    esac

    if [[ "$func_exit_code" -ne 0 ]]; then
        log_err "Failed to handle init_start for VM $vm_name"
        return $func_exit_code
    fi
  
    declare -F pd_handle_init_start &> /dev/null
    if [[ $? -eq 0 ]]; then
        pd_handle_init_start $vm_name $vm_op $vm_subop $vm_initpid
        func_exit_code=$?
    fi

    if [[ "$func_exit_code" -ne 0 ]]; then
        log_err "PD failed to handle init_start for VM $vm_name"
        return $func_exit_code
    fi
  
    return $func_exit_code
}
readonly -f handle_init_start

#
# handle_started
#
function handle_started()
{
    local vm_name=$1
    local vm_op=$2
    local vm_subop=$3
    local vm_initpid=$4
    local func_exit_code=0

    /opt/cisco/hostos/bin/lxc_cleanup_helper_static -n "$vm_name" \
                                    -p "$vm_initpid" -s "$vm_op"
    func_exit_code=$?
    if [[ "$func_exit_code" -ne 0 ]]; then
        log_err "Failed to make NETNS persistent for VM $vm_name"
        # return $func_exit_code
        return 0
    fi
    log_msg "NETNS persistent for VM $vm_name"

    local ns_file=/var/run/netns/$vm_name.libvirt
    touch "${ns_file}"
    mount -o bind /proc/$vm_initpid/ns/net "${ns_file}"

    return $func_exit_code
}

#
# handle_init_shutdown
#
function handle_init_shutdown()
{
    local vm_name=$1
    local vm_op=$2
    local vm_subop=$3
    local vm_initpid=$4
    local func_exit_code=0

    log_msg "Nothing to do for VM $vm_name"
    declare -F pd_pci_cleanup && pd_pci_cleanup $vm_name

    declare -F handle_destroy_mv_interfaces &> /dev/null
    if [[ $? -eq 0 ]]; then
        handle_destroy_mv_interfaces $vm_name $vm_initpid
        func_exit_code=$?
    fi
        
    return $func_exit_code
}
readonly -f handle_init_shutdown

#
# handle_init_destroy
#
function handle_init_destroy()
{
    local vm_name=$1
    local vm_op=$2
    local vm_subop=$3
    local vm_initpid=$4
    local func_exit_code=0

    declare -F pd_pci_cleanup && pd_pci_cleanup $vm_name

    declare -F handle_destroy_mv_interfaces &> /dev/null
    if [[ $? -eq 0 ]]; then
        handle_destroy_mv_interfaces $vm_name $vm_initpid
        func_exit_code=$?
    fi

    return $func_exit_code
}
readonly -f handle_init_destroy

#
# handle_stopped
#
function handle_stopped()
{
    local vm_name=$1
    local vm_op=$2
    local vm_subop=$3
    local vm_initpid=$4
    local func_exit_code=0

    #
    # IGNORE ERRORS, while stopping LXC.
    #

    log_msg "Unmounting rootfs for LXC"
    pd_unmount_lxc_rootfs "$vm_name"

    declare -F lxc_app_hosting_hook &> /dev/null
    if [[ $? -eq 0 ]]; then
        log_msg "Calling app hosting stopped hook"
        lxc_app_hosting_hook $vm_name $vm_op $vm_subop $vm_initpid
    fi

    /opt/cisco/hostos/bin/lxc_cleanup_helper_static -n "$vm_name" -s "$vm_op"
    func_exit_code=$?
    if [[ "$func_exit_code" -ne 0 ]]; then
        log_err "Failed to reclaim physical interfaces for VM $vm_name. Cont..."
    fi
    log_msg "Reclaimed physical interfaces for VM $vm_name, if present."

    case "$vm_name" in
    sysadmin)
        lxc_reclaim_sysadmin_intf_hook $vm_name $vm_op $vm_subop $vm_initpid
        func_exit_code=$?
        ;;

    default-sdr--*)
        lxc_reclaim_sdr_intf_hook $vm_name $vm_op $vm_subop $vm_initpid
        func_exit_code=$?
        ;;

    *)
        #Nothing to do
        ;;
    esac

    if [[ "$func_exit_code" -ne 0 ]]; then 
        log_err "Failed to handle stopped for VM $vm_name and continuing..."
    fi

    return 0
}
readonly -f handle_stopped

#
# handle_prepare
#
function handle_prepare()
{
    local vm_name="$1"
    local ns_file="/var/run/netns/$vm_name.libvirt"
    local rc

    if [ -f "$ns_file" ] && ! rm "$ns_file"; then
	log_msg "Leftover network namespace $ns_file from previous run.  Cleaning up."
	reclaim_network_interfaces $vm_name
	umount "$ns_file"
	rm "$ns_file"
    fi

    log_msg "Calling prelaunch setup for  LXC"
    pd_prelaunch_setup "$vm_name"
    rc=$?
    if [[ $rc -ne 0 ]]; then
        log_err "Failed to execute prelaunch setup for VM $vm_name"
        return $rc
    fi

    return 0
}
readonly -f handle_prepare

function reclaim_network_interfaces()
{
    local vm_name=$1
    local func_exit_code=0

    case "$vm_name" in
    default-sdr--*)
        kim_dev="kimctrl"
        log_msg "Moving $kim_dev netdevice to initial namespace ..."
        CMD="ip netns exec $vm_name.libvirt ip link set $kim_dev netns 1"
        printf "%s CMD: %s\n" "$timestamp" "$CMD" >> $LXC_HOOK_LOG_FILE
        platform_log_exec "$CMD"
    esac

    if declare -F handle_release_mv_interfaces &> /dev/null; then
        handle_release_mv_interfaces $vm_name 
        func_exit_code=$?
    fi

    return $func_exit_code
}
readonly -f reclaim_network_interfaces

#
# handle_release
#
function handle_release()
{
    local vm_name=$1
    local vm_op=$2
    local vm_subop=$3
    local vm_initpid=$4
    local func_exit_code=0

    log_msg "handle release"

    declare -F pd_handle_release &>/dev/null
    func_exit_code=$?
    if [[ "$func_exit_code" -eq 1 ]]; then
        log_err "PD did not define release for VM $vm_name; no work to do"
    else
      pd_handle_release $vm_name $vm_op $vm_subop $vm_initpid
      func_exit_code=$?
      if [[ "$func_exit_code" -ne 0 ]]; then
        log_err "PD failed to handle release for VM $vm_name"
      fi
    fi

    reclaim_network_interfaces $vm_name

    local ns_file="/var/run/netns/$vm_name.libvirt"
    umount "${ns_file}"
    rm "${ns_file}"

    return $func_exit_code
}
readonly -f handle_release

#
# Parses the XML, finds the rootfs mount point and unmounts it.
#
function unmount_lxc_rootfs()
{
    local vm_name=$1

    #
    # Sample XML below for reference:
    # <filesystem type='mount' accessmode='passthrough'>
    #    <source dir='/lxc_rootfs/panini_vol_grp-xr_lv0'/>
    #     <target dir='/'/>
    # </filesystem>
    local mnt_path=`cat /var/run/libvirt/lxc/"$vm_name".xml | \
                    grep "[\"']/lxc_rootfs" | cut -d"'" -f2`
    if [[ -n "$mnt_path" ]]; then
        log_msg "Found mount point: $mnt_path and unmounting it for $vm_name."

        #
        # Previously if we had mounted the host /var/run/netns into XR, 
        # we now need to unmount it.
        #
        declare -F app_host_unmount_vrfs &> /dev/null
        if [[ $? -eq 0 ]]; then
            platform_log "VRF: Calling unmount VRFs hook"
            app_host_unmount_vrfs $vm_name $mnt_path
            if [[ $? -ne 0 ]]; then
                platform_log_error "VRF: Calling unmount VRFs hook failed"
            fi
        fi

        umount "$mnt_path" -d
        if [[ $? -ne 0 ]]; then
            log_err "Failed to unmount $mnt_path for $vm_name."
        else 
            log_msg "Removing directory: $mnt_path for $vm_name."
            rmdir $mnt_path
        fi
    else 
        log_err "Failed to find mount point for $vm_name on cleanup."
    fi
}
readonly -f unmount_lxc_rootfs

#
# handle_main
#
function handle_main()
{
    local vm_name=$1
    local vm_op=$2
    local vm_subop=$3
    local vm_initpid=$4

    local ret_code=0

    case "$vm_op" in
    prepare)
        handle_prepare "$vm_name" "$vm_op" "$vm_subop" "$vm_initpid"
        ret_code=$?
        ;;

    start)
        handle_start "$vm_name" "$vm_op" "$vm_subop" "$vm_initpid"
        ret_code=$?
        ;;

    started)
        handle_started "$vm_name" "$vm_op" "$vm_subop" "$vm_initpid"
        ret_code=$?
        ;;

    init_start)
        handle_init_start "$vm_name" "$vm_op" "$vm_subop" "$vm_initpid"
        ret_code=$?
        ;;

    init_shutdown)
        handle_init_shutdown "$vm_name" "$vm_op" "$vm_subop" "$vm_initpid"
        ret_code=$?
        ;;

    init_destroy)
        handle_init_destroy "$vm_name" "$vm_op" "$vm_subop" "$vm_initpid"
        ret_code=$?
        ;;

    stopped)
        handle_stopped "$vm_name" "$vm_op" "$vm_subop" "$vm_initpid"
        ret_code=$?
        ;;

    release)
       handle_release "$vm_name" "$vm_op" "$vm_subop" "$vm_initpid"
       ret_code=$?
       ;;
    *)
        #Nothing to do
        ;;
    esac

    return "$ret_code"
}

#
# Main starts here
#
declare -F platform_log &>/dev/null && platform_log "start"

#
# Verify the input arguments to the script.
#
if [[ "$#" -ne 4 ]]
then
    printf "%s Error: Calling script %s with illegal arguments\n" \
           "$timestamp" "$0" >> $LXC_HOOK_LOG_FILE
    exit 1
fi    

printf "%s %s %s %s %s %s\n" "$timestamp" "$0" "$1" "$2" "$3" "$4" >> $LXC_HOOK_LOG_FILE

exit_code=0
handle_main $1 $2 $3 $4
exit_code=$?

declare -F platform_log &>/dev/null && platform_log "done"

exit "$exit_code"
