#!/bin/bash
#
# calvados/asr9k_pkg/boot/scripts/spirit_pd.sh
# This script provides PD variables to other scripts.
#
# Copyright (c) 2014-2017 by Cisco Systems, Inc.
# All rights reserved.

BOOTSTRAP_FILE="/etc/init.d/calvados_bootstrap.cfg"
PD_FUNCS="/etc/rc.d/init.d/pd-functions"
if [ -f $BOOTSTRAP_FILE ]; then
    source $BOOTSTRAP_FILE
fi
if [ -f /etc/init.d/update_mac_v2.sh ]; then
    source /etc/init.d/update_mac_v2.sh
fi
if [ -f /etc/rc.d/init.d/rp_resync.sh ]; then
    source /etc/rc.d/init.d/rp_resync.sh
fi

function is_vxr_sim() 
{
    if [[ ! -z $VXR_SIM ]]; then
        return $VXR_SIM
    fi

    local cmdline=`cat /proc/cmdline`
    local CMD_SIM=`echo $cmdline | sed 's/^.*__vxr=//' | cut -d" " -f1`
    if [ "$CMD_SIM" == "true" ]
    then
        VXR_SIM=1
    else
        VXR_SIM=0
    fi
    return $VXR_SIM
}

function on_sim()
{
    # The use of $SIMULATION is legacy
    # $SIMULATION retained below as a one time setting flag
    if [ "${SIMULATION}" == "" ]; 
    then
        #
        # Perform one time actions
        # 
        readonly SIMULATION=0
        if [ -f /etc/init.d/xrnginstall-asr9k-pd.sh ]; then
            source /etc/init.d/xrnginstall-asr9k-pd.sh
        fi
        readonly PLATFORM_HOST_BRIDGE=1
        readonly TMPFS_SIZE=3072m
        readonly SAVEFS_SIZE=1024m
    fi

    return $SIMULATION
}

function get_4B_off {

    local hw_ct

    if [[ -z ${HW_CARDTYPE} ]]; then
        kmsg_log "get_card_specific_value $1: HW_CARDTYPE not set"
        return
    fi
    let "hw_ct = ${HW_CARDTYPE} & 0x3f"

    hw_ct=$( printf "%02x" ${hw_ct} )

    case "$hw_ct" in
    0[b]) # All LS cards
       echo 0
       ;;
    *)  # Everything else
       echo 0x100000
       ;;
    esac

}

function get_ipu_nvram_off {
    INFO_ROM_DS_OFF=0x30
    INFO_ROM_NV_OFF=0xE8
    EMT_MODE_OFF=0x20

    if [[ ! -z $NVRAM_OFF ]]; then
        return
    fi
    source $PD_FUNCS

    set_ipu_base
    if [[ $? != 0 ]]; then
        kmsg_log "get_ipu_nvram_off: Cannot determine IPU NVRAM base"
        return
    fi
    INFO_ROM_DS_VAL=$(($IPU_BASE + $INFO_ROM_DS_OFF))
    INFO_ROM_CT_OFF=$( pcimemread `printf "%x" $INFO_ROM_DS_VAL` 0x4 | grep -v PCI | awk '{ print $3 ; }' | sed -e 's/^/0x/' )

    HW_CARDTYPE_ADDR=$(($IPU_BASE + $INFO_ROM_CT_OFF))
    HW_CARDTYPE=$( pcimemread `printf "%x" $HW_CARDTYPE_ADDR` 0x4 | grep -v PCI | awk '{ print $3 ; }' | sed -e 's/^/0x/' )
    if (( $HW_CARDTYPE & 0x20 )) ; then BOARDTYPE="RP"; else BOARDTYPE="LC"; fi
    if (( $HW_CARDTYPE & 0x20 )) ; then EMT_OFF=0x7F000; else EMT_OFF=0x1F000; fi

    INFO_ROM_NV_ADDR=$(( $IPU_BASE + $INFO_ROM_NV_OFF ))
    INFO_ROM_NV_VAL=$( pcimemread `printf "%x" $INFO_ROM_NV_ADDR` 0x4 | grep -v PCI | awk '{ print $3 ; }' | sed -e 's/^0/0x0/' )

    NVRAM_4B_OFF=$(get_4B_off)

    NVRAM_OFF=$(( $INFO_ROM_NV_VAL + $NVRAM_4B_OFF ))

    INFO_ROM_DB_OFF=$((INFO_ROM_CT_OFF + 4))
    DB_CARDTYPE_ADDR=$(($IPU_BASE + $INFO_ROM_DB_OFF))
    DB_CARDTYPE=$( pcimemread `printf "%x" $DB_CARDTYPE_ADDR` 0x4 | grep -v PCI | awk '{ print $3 ; }' | sed -e 's/^/0x/' )

    EMT_MODE_PADDR=$(($IPU_BASE + $NVRAM_OFF + $EMT_OFF + $EMT_MODE_OFF))
    EMT_MODE=$( pcimemread `printf "%x" $EMT_MODE_PADDR` 0x10 | grep -v PCI | awk '{ print $3 ; }' | sed -e 's/^000000/0x/' )

}

function display_memory_details
{
    echo Memory details
    dmidecode --type 17 | egrep "Part|Size|Manuf|Serial" | grep -iv No | \ 
                           sed -e "s; Number;;" | paste - - - - | \
                           sed -e "s;\t; ;g" | sed -e "s;^ ;;"
}

function display_image_details
{
    if [[ ! -f /etc/show_version.txt ]]; then
        return
    fi
    local WORKSPACE=$( grep Workspace /etc/show_version.txt | awk '{ print $3 }' )
    local BUILDNODE=$( grep Host /etc/show_version.txt | awk '{ print $3 }' )
    local EFR=$( grep "Efr Info" -A 1 /etc/show_version.txt | tail -1 | awk '{ print $2 }' )
    local LINEUP=$( grep "Efr Info" -A 1 /etc/show_version.txt | tail -1 | awk '{ print $1 }' )
    local BUILDDATE=$( grep Date /etc/show_version.txt | sed -e "s;Date: ;;" )
    if [[ $1 == "1" ]]; then
        echo Image built $BUILDDATE
        echo Build location $BUILDNODE:$WORKSPACE
        echo Lineup $LINEUP:$EFR
    else
        step_log_console "Image built $BUILDDATE"
        step_log_console "Build location $BUILDNODE:$WORKSPACE"
        step_log_console "Lineup $LINEUP:$EFR"
    fi
}

function check_cardtype
{
    local cardtype=$PD_CARDTYPE
    local cardclass
    let "cardclass = ${cardtype} >> 16"
    cardclass=$( printf "%x" ${cardclass} )

    # ASR9K RSP's have 0x10 in upper half-word of card type
    if [[ ${BOARDTYPE} == "RP" && ${cardclass} == "10" ]]; then
        return 0
    fi

    # ASR9K X86 LC's have 0x3D in upper half-word of card type
    if [[ ${BOARDTYPE} == "LC" && ${cardclass} == "3d" ]]; then
        return 0
    fi

    # ASR9K X86 LC's have 0x3F in upper half-word of card type
    if [[ ${BOARDTYPE} == "LC" && ${cardclass} == "3f" ]]; then
        return 0
    fi

    return 1
}

function check_chassistype
{
    local chassistype=$1
    local chassisclass
    let "chassisclass = ${chassistype} >> 16"
    chassisclass=$( printf "%x" ${chassisclass} )
    if [[ ${chassisclass} == "ef" ]]; then
        return 0
    else
        return 1
    fi
}


function get_board_type_ipu {
    PD_CARDTYPE_OFF=0x3C
    get_ipu_nvram_off

    PD_CARDTYPE_PADDR=$(($IPU_BASE + $NVRAM_OFF + $EMT_OFF + $PD_CARDTYPE_OFF))
    PD_CARDTYPE=$( pcimemread `printf "%x" $PD_CARDTYPE_PADDR` 0x10 | grep -v PCI | awk '{ print $3 ; }' | sed -e 's/^00/0x/' )
}

function get_board_type_cmd {
    if [[ -f /root/cmdline ]]; then
        PROC_CMDLINE=$(cat /root/cmdline)
    else
        PROC_CMDLINE=$(cat /proc/cmdline)
    fi
    for x in ${PROC_CMDLINE} ; do
        case "$x" in
            boardtype=*)
            BOARDTYPE="${x#boardtype=}"
            ;;
            cardtype=*)
            PD_CARDTYPE="${x#cardtype=}"
            ;;
            iputype=*)
            HW_CARDTYPE="${x#iputype=}"
            ;;
        esac
    done
}

function get_board_type {
    # first read from /proc/cmdline
    if [ -z ${BOARDTYPE} ] || [ -z ${PD_CARDTYPE} ]; then
        get_board_type_cmd
    fi
    # failing which read from IPU
    if [ -z ${BOARDTYPE} ] || [ -z ${PD_CARDTYPE} ]; then
        get_board_type_ipu
    fi
}

function get_chassis_details_ipu {
    CHASSIS_TYPE_OFF=0x48
    CHASSIS_SERIAL_OFF=0x70
    CHASSIS_PID_OFF=0x90

    get_ipu_nvram_off

    CHASSIS_TYPE_PADDR=$(($IPU_BASE + $NVRAM_OFF + $EMT_OFF + $CHASSIS_TYPE_OFF))
    CHASSIS_TYPE=$( pcimemread `printf "%x" $CHASSIS_TYPE_PADDR` 0x4 | grep -v PCI | awk '{ print $3 ; }' | sed -e 's/00ef/0xef/' )

    CHASSIS_SERIAL_PADDR=$(($IPU_BASE + $NVRAM_OFF + $EMT_OFF + $CHASSIS_SERIAL_OFF))
    CHASSIS_SERIAL=$( pcimemread `printf "%x" $CHASSIS_SERIAL_PADDR` 0x10 | grep -v PCI | awk '{ print $7 ; }' | tr -d "." )

    CHASSIS_PID_PADDR=$(($IPU_BASE + $NVRAM_OFF + $EMT_OFF + $CHASSIS_PID_OFF))
    CHASSIS_PID=$( pcimemread `printf "%x" $CHASSIS_PID_PADDR` 0x10 | grep -v PCI | awk '{ print $7 ; }' | tr -d "." )
}

function get_chassis_details_cmd {
    PROC_CMDLINE=$(cat /proc/cmdline)
    for x in ${PROC_CMDLINE} ; do
        case "$x" in
            chassis_serial=*)
            CMD_CHASSIS_SERIAL="${x#chassis_serial=}"
            ;;
            chassis_type=*)
            CMD_CHASSIS_TYPE="${x#chassis_type=}"
            ;;
            chassis_pid=*)
            CMD_CHASSIS_PID="${x#chassis_pid=}"
            ;;
        esac
    done
}

function override_platform_type {
   echo asr9k
   return
}

function get_cctrl_board_type {
    echo ${BOARDTYPE}
}

function get_card_specific_value {
    local hw_ct
    local db_ct

    if [[ -z ${HW_CARDTYPE} ]]; then
        get_ipu_nvram_off
    fi

    if [[ -z ${HW_CARDTYPE} ]]; then
        kmsg_log "get_card_specific_value $1: HW_CARDTYPE not set"
        return
    fi
    let "hw_ct = ${HW_CARDTYPE} & 0x3f"

    hw_ct=$( printf "%02x" ${hw_ct} )

    case "$hw_ct" in
    2[012345])  # RSP4/RP2 all
        case "$1" in
        LGT_GPIO_RESET_OFFSET)
            echo 0xfe8
            ;;
        LGT_GPIO_SET_OFFSET)
            echo 0xfe4
            ;;
        OIR_RESET_SUPPORT)
            echo 1 # true
            ;;
        NUM_FIA)
            echo 1
            ;;
        FIA0_OFF)
            echo 0x301
            ;;
        PXE_ETHIF)
            if [[ "$hw_ct" == "21" || "$hw_ct" == "25" ]]; then
                echo eth4   # RP2
            else
                echo eth5   # RSP4
            fi
            return
            ;;
        PCI_RP_8749)
            if [[ "$hw_ct" == "21" ]]; then
                echo 8086:0e02   # RP2 Proto
            elif [[ "$hw_ct" == "25" ]]; then
                echo 8086:0e08   # RP2 Prod
            else
                echo 8086:0e02   # RSP4/RSP4S
            fi
            return
            ;;
        MSIGRP0_COUNT)
            echo 24
            return
            ;;
        MSIGRP1_COUNT)
            echo 0
            return
            ;;
        MSIGRP2_COUNT)
            echo 0
            return
            ;;
        BIGPHYSAREA_HOST)
            echo 12M
            return
            ;;
        BIGPHYSAREA_CALVADOS)
            echo 112M
            return
            ;;
        BIGPHYSAREA_XR)
            echo 12M
            return
            ;;
        DMAINFO_HOST)
            echo "mvdma0=1M,0 mvdma1=1M,1M mvdma2=1M,2M" 
            return
            ;;
        DMAINFO_CALVADOS)
            echo "mvdma0=1M,0 mvdma1=1M,1M mvdma2=1M,2M" 
            return
            ;;
        DMAINFO_XR)
            echo "mvdma0=1M,0 mvdma1=1M,1M mvdma2=1M,2M" 
            return
            ;;
        esac
        ;;
    03)         # Octane
        case "$1" in
        LGT_GPIO_RESET_OFFSET)
            echo 0x23e8
            ;;
        LGT_GPIO_SET_OFFSET)
            echo 0x23e4
            ;;
        OIR_RESET_SUPPORT)
            echo 1 # true
            ;;
        PXE_ETHIF)
            echo eth2
            ;;
        NUM_FIA)
            if [[ -z ${DB_CARDTYPE} ]]; then
                get_ipu_nvram_off
            fi
            let "db_ct = $DB_CARDTYPE & 0xF"
            if [ $db_ct -eq 5 -o $db_ct -eq 3 ]; then # Laserbeak
                echo 2
            else
                echo 4
            fi
            ;;
         FIA0_OFF)
            if [[ -z ${DB_CARDTYPE} ]]; then
                get_ipu_nvram_off
            fi
            let "db_ct = $DB_CARDTYPE & 0xF"
            if [ $db_ct -eq 5 -o $db_ct -eq 3 ]; then # Laserbeak
                echo 0x10301 
            else
                echo 0x301
            fi
            ;;
        PCI_RP_8749)
            echo 8086:0e02
            ;;
        MSIGRP0_COUNT)
            echo 24
            ;;
        MSIGRP1_COUNT)
            echo 0
            ;;
        MSIGRP2_COUNT)
            echo 0
            ;;
        BIGPHYSAREA_HOST)
            echo 12M
            return
            ;;
        BIGPHYSAREA_CALVADOS)
            echo 12M
            return
            ;;
        BIGPHYSAREA_XR)
            echo 12M
            return
            ;;
        DMAINFO_HOST)
            echo "mvdma0=1M,0 mvdma1=1M,1M mvdma2=1M,2M" 
            return
            ;;
        DMAINFO_CALVADOS)
            echo "mvdma0=1M,0 mvdma1=1M,1M mvdma2=1M,2M" 
            return
            ;;
        DMAINFO_XR)
            echo "mvdma0=1M,0 mvdma1=1M,1M mvdma2=1M,2M" 
            return
            ;;
        esac
        ;;

    0b)         # LS LC
        case "$1" in
        LGT_GPIO_RESET_OFFSET)
            echo 0x23e8 #FIXME
            ;;
        LGT_GPIO_SET_OFFSET)
            echo 0x23e4 #FIXME
            ;;
        OIR_RESET_SUPPORT)
            echo 1 # true
            ;;
        PXE_ETHIF)
            echo eth0
            ;;
        NUM_FIA)
            echo 6 
            ;;
        FIA0_OFF)
            echo 0x301
            ;;
        MSIGRP0_COUNT)
            echo 24
            ;;
        MSIGRP1_COUNT)
            echo 0
            ;;
        MSIGRP2_COUNT)
            echo 0
            ;;
        BIGPHYSAREA_HOST)
            echo 112M
            return
            ;;
        BIGPHYSAREA_CALVADOS)
            echo 112M
            return
            ;;
        BIGPHYSAREA_XR)
            echo 112M
            return
            ;;
        DMAINFO_HOST)
            echo "mvdma0=1M,0 mvdma1=1M,1M mvdma2=1M,2M" 
            return
            ;;
        DMAINFO_CALVADOS)
            echo "mvdma0=1M,0 mvdma1=1M,1M mvdma2=1M,2M" 
            return
            ;;
        DMAINFO_XR)
            echo "mvdma0=1M,0 mvdma1=1M,1M mvdma2=1M,2M" 
            return
            ;;
        esac
        ;;

    04)         # Sureshot
        case "$1" in
        LGT_GPIO_RESET_OFFSET)
            echo 0x1c88
            ;;
        LGT_GPIO_SET_OFFSET)
            echo 0x1c84
            ;;
        OIR_RESET_SUPPORT)
            echo 1 # true
            ;;
        PXE_ETHIF)
            echo eth0
            ;;
        NUM_FIA)
            if [[ -z ${DB_CARDTYPE} ]]; then
                get_ipu_nvram_off
            fi
            let "db_ct = $DB_CARDTYPE & 0xF"

            if [ $db_ct -eq 3 ]; then
                echo 2
            else
                echo 1
            fi
            ;;
        FIA0_OFF)
            echo 0x301
            ;;
        PCI_RP_8749)
            echo 8086:0e09
            ;;
        MSIGRP0_COUNT)
            echo 24
            ;;
        MSIGRP1_COUNT)
            echo 0
            ;;
        MSIGRP2_COUNT)
            echo 0
            ;;
        8749_UNUSED_PORTS)
            # PCI ports connected to AMCC PHY
            # for Slingshot/Cliffjumper MPAs
            echo "04 0a 0d 15"
            ;;
        BIGPHYSAREA_HOST)
            echo 12M
            return
            ;;
        BIGPHYSAREA_CALVADOS)
            echo 12M
            return
            ;;
        BIGPHYSAREA_XR)
            echo 12M
            return
            ;;
        DMAINFO_HOST)
            echo "mvdma0=1M,0 mvdma1=1M,1M mvdma2=1M,2M" 
            return
            ;;
        DMAINFO_CALVADOS)
            echo "mvdma0=1M,0 mvdma1=1M,1M mvdma2=1M,2M" 
            return
            ;;
        DMAINFO_XR)
            echo "mvdma0=1M,0 mvdma1=1M,1M mvdma2=1M,2M" 
            return
            ;;
        esac
        ;;
    05)         # Skyhammer
        case "$1" in
        LGT_GPIO_RESET_OFFSET)
            echo 0x18a8
            ;;
        LGT_GPIO_SET_OFFSET)
            echo 0x18a4
            ;;
        OIR_RESET_SUPPORT)
            echo 1 # true
            ;;
        PXE_ETHIF)
            echo eth2
            ;;
        NUM_FIA)
            echo 6
            ;;
        FIA0_OFF)
            echo 0x301
            ;;
        PCI_RP_8749)
            echo 8086:6f02
            ;;
        MSIGRP0_COUNT)
            echo 24
            ;;
        MSIGRP1_COUNT)
            echo 0
            ;;
        MSIGRP2_COUNT)
            echo 0
            ;;
        BIGPHYSAREA_HOST)
            echo 12M
            return
            ;;
        BIGPHYSAREA_CALVADOS)
            echo 12M
            return
            ;;
        BIGPHYSAREA_XR)
            echo 12M
            return
            ;;
        DMAINFO_HOST)
            echo "mvdma0=1M,0 mvdma1=1M,1M mvdma2=1M,2M" 
            return
            ;;
        DMAINFO_CALVADOS)
            echo "mvdma0=1M,0 mvdma1=1M,1M mvdma2=1M,2M" 
            return
            ;;
        DMAINFO_XR)
            echo "mvdma0=1M,0 mvdma1=1M,1M mvdma2=1M,2M" 
            return
            ;;
        esac
        ;;
     *)
        kmsg_log "Unknown IPU_TYPE: $HW_CARDTYPE for $1"
        echo 0x0000
        ;;
    esac
}

function reset_lgt_pcie {
    local vm_type=$1
    local virt_method=$2

    get_board_type
    source $PD_FUNCS

    if [[ -z ${IPU_BASE} ]]; then
        kmsg_log "No IPU base - skipping lightning asic reset"
        return
    fi

    if [ "$BOARDTYPE" == "RP" ]; then
        local gpio_base=$(get_ipu_block_offset GPIO 3)
        if [[ -z ${gpio_base} ]]; then
            kmsg_log "Cannot find GPIO base - skipping RP lightning asic reset"
            return
        fi
    else
        local gpio_base=$(get_ipu_block_offset GPIO 2)
        if [[ -z ${gpio_base} ]]; then
            kmsg_log "Cannot find GPIO base - skipping LC lightning asic reset"
            return
        fi
    fi
    local LGT_GPIO_RESET_OFFSET=$(printf "0x%x" $(($gpio_base + $( get_card_specific_value LGT_GPIO_RESET_OFFSET ))))
    local LGT_GPIO_SET_OFFSET=$(printf "0x%x" $(($gpio_base + $( get_card_specific_value LGT_GPIO_SET_OFFSET ))))
    local LGT_RESET=$( printf "0x%x" $(( $IPU_BASE + $LGT_GPIO_RESET_OFFSET )))
    local LGT_SET=$( printf "0x%x" $(( $IPU_BASE + $LGT_GPIO_SET_OFFSET )))

    pcimemwrite $LGT_RESET 4 0x10 > /dev/null 2>&1
    pcimemwrite $LGT_SET 4 0x10  > /dev/null 2>&1
    sleep 0.5
    echo "1" > /sys/bus/pci/rescan
    kmsg_log "Lightning reset done"
}

update_host_counters() {
    # Obtain host counter address in EMT
    showemt addr | grep host_reboot_count > /tmp/emt_addr_offset
    source /tmp/emt_addr_offset

    # Increment host counter in EMT
    if [ -n "${host_reboot_count}" ]; then 
        CURR_CNT=$(pcimemread `printf '%x' $host_reboot_count` 4 | grep -v PCI | awk '{ print $3 ;}' | sed -e 's/^0/0x/')
        count=`expr $(($CURR_CNT)) + 1`
        incr=`printf '%x' $count`
        pcimemwrite `printf '%x' $host_reboot_count` 4 $incr > /dev/null 2>&1
    fi

    # Rommon clears the XR and Calvados LXC Respawn counters on load

    rm /tmp/emt_addr_offset
}

host_specific_configuration () {

  local VM_TYPE=$1
  local VIRT_METHOD=$2

  modprobe 8021q

  if [[ $VIRT_METHOD == "vm" ]]; then
      echo 1 > /sys/module/kvm/parameters/allow_unsafe_assigned_interrupts
      modprobe uio_pci_proxy stub=1 >/dev/null 2>&1
  else
      modprobe uio_pci_proxy >/dev/null 2>&1
  fi
  local dmainfo=$( get_card_specific_value "DMAINFO_HOST" )
  modprobe uio_dma_proxy dmainfo=\"$dmainfo\"
  sleep 0.5

  a9k_pcie_proxy_enable $VM_TYPE $VIRT_METHOD
  a9k_show_media

  is_vxr_sim
  if [[ $? != 0 ]]; then
      # IPU
      #before that unbind from pcieport driver (happens in VXR) 
      loc=$(lspci -nn | grep 87a0 | awk '{print $1}')
      echo "0000:$loc" > /sys/bus/pci/drivers/pcieport/unbind
  fi

  # mask PLX PCI unsupported request PCI express Device Control
  setpci -d 10b5:8749 CAP_EXP+8.l=0:8

  # set PCI completion timeout in X86 RP (connected to 8749) to 16-55ms
  # PCI Express Capability Offset 28 - DEVCTRL2 bits 3:0
  # 5: 16-55ms
  # 9: 260-900ms
  local RP_DEV=$( get_card_specific_value PCI_RP_8749 )
  if [[ "$RP_DEV" != "0x0000" ]]; then
    setpci -d $RP_DEV CAP_EXP+28.l=5:f
  fi

  # Enable IPU Bus Mastership
  set_ipu_bdf
  if [[ $? != 0 ]]; then
      echo Warning: Cannot set IPU bus mastership
  else
      setpci -s $IPU_BDF COMMAND=4:4
  fi

  # Obtain IPU_BASE for future use
  set_ipu_base
  if [[ $? != 0 ]]; then
      echo Warning: Cannot determine IPU base
  fi

  a9k_devices_enable $VM_TYPE $VIRT_METHOD
  lspci > /misc/scratch/pcie/lspci.log
}

 
# Change first two partitions to qnx4 (types 4d and 4e)
# File systems are not created
# Only for testing
function eusb_to_qnx4
{
sfdisk --force -N1 /dev/eusb << EOF
,,4d
EOF
sfdisk --force -N2 /dev/eusb << EOF
,,4e
EOF
}

# If either of first two partitions are qnx4, set them to ${extfs} (83)
# Partitions sizes remain intact
# New ${extfs} file-systems are created
# All qnx4 f/s data in these two partitions is lost
# Modify the DOS partition type from 11 to 83 (Linux)
# this allows the clean reversal of the node to Classic
# The dos partition content is retained intact
extfs=ext2
function eusb_to_extfs
{
    local isqnx=`fdisk -l /dev/eusb 2> /dev/null | grep eusb1 | grep 4d | wc -l`
    local isnotextfs=`file -sL /dev/eusb1 | grep -v ${extfs} | wc -l`
    if [ "$isqnx" != "0" -o "$isnotextfs" != "0" ]; then
        sfdisk --force -N1 /dev/eusb << EOF
,,L
EOF

        /sbin/mkfs.${extfs} -L eusb -b 4096 -vv /dev/eusb1 > /dev/null 2>&1
        /sbin/tune2fs -c 0 -i 0 /dev/eusb1 > /dev/null 2>&1
    fi

    isqnx=`fdisk -l /dev/eusb 2> /dev/null | grep eusb2 | grep 4e | wc -l`
    isnotextfs=`file -sL /dev/eusb2 | grep -v ${extfs} | wc -l`
    if [ "$isqnx" != "0" -o "$isnotextfs" != "0" ]; then
        sfdisk --force -N2 /dev/eusb << EOF
,,L
EOF

        /sbin/mkfs.${extfs} -L eusba -b 4096 -vv /dev/eusb2 > /dev/null 2>&1
        /sbin/tune2fs -c 0 -i 0 /dev/eusb2 > /dev/null 2>&1
    fi
    local islnx=`fdisk -l /dev/eusb 2> /dev/null | grep eusb3 | grep Linux | wc -l`
    if [ "$islnx" == "0" ]; then
        sfdisk --force -N3 /dev/eusb << EOF
,,L
EOF
    fi

}

# Convert first three partitions to FLEXR model, create fresh file systems
# First two partitions are ${extfs} fs (Linux type 83)
# Third partition is DOS fs (type 11)
# Space for tiny Fourth partition is set aside. However, not created
# Reboot may be required
# All data is lost
function eusb_format
{
    umount /dev/eusb1 > /dev/null 2>&1
    umount /dev/eusb2 > /dev/null 2>&1
    umount /dev/eusb3 > /dev/null 2>&1

sfdisk --force /dev/eusb << EOF
1,500,L
501,100,L
601,400,L
EOF

    /sbin/mkfs.${extfs} -L eusb -b 4096 -vv /dev/eusb1 > /dev/null 2>&1
    /sbin/tune2fs -c 0 -i 0 /dev/eusb1 > /dev/null 2>&1

    /sbin/mkfs.${extfs} -L eusba -b 4096 -vv /dev/eusb2 > /dev/null 2>&1
    /sbin/tune2fs -c 0 -i 0 /dev/eusb2 > /dev/null 2>&1

    /sbin/mkfs.vfat -n eusbb -vv /dev/eusb3 > /dev/null
}

function is_ata ()
{
      local is_ata_dev=$(udevadm info --query=path --name=$1 | grep ata | wc -l)
      if [[ ${is_ata_dev} != "0" ]]; then
        return 1
      else
        return 0
      fi
}

function eusb_mount_one
{
    local part=$1
    local mnt_pt=$2
    local do_fsck=$3
    if [[ -b /dev/${part} ]]; then
        if [[ ! -d /${mnt_pt} ]]; then
            mkdir /${mnt_pt}
        fi
        if [[ ${do_fsck} == "1" ]]; then
            fsck -aT /dev/${part} >> /tmp/eusb.log 2>&1
        fi
        kmsg_exec "mount -o sync /dev/${part} /${mnt_pt}"
    fi
}

# /dev/eusb* paths have been created by udev 90 script
# Mount eusb partitions in hostos
# Update lxc eusb pass thru path
function eusb_mount_host
{
    eusb_mount_one eusb1 eusb 1
    eusb_mount_one eusb2 eusba 1
    eusb_mount_one eusb3 eusbb 1

    # remove old sdX paths in devlist files
    sed -i -e "/sd/d" /etc/sysconfig/Devlist_calvados.conf
    sed -i -e "/sd/d" /etc/sysconfig/Devlist_xr.conf

    # update with applicable sdX in devlist files
    is_ata /dev/sda
    if [[ $? == 1 ]]; then
        local is_sdc=$(udevadm info --query=all -n /dev/sdc --attribute-walk | grep devpath | egrep "1.1|1.5" | wc -l)
        if [[ ${is_sdc} != "0" ]]; then
            echo "/dev/sdc*" >> /etc/sysconfig/Devlist_calvados.conf
            echo "/dev/sdc*" >> /etc/sysconfig/Devlist_xr.conf
        else
            echo "/dev/sdd*" >> /etc/sysconfig/Devlist_calvados.conf
            echo "/dev/sdd*" >> /etc/sysconfig/Devlist_xr.conf
        fi
        return
    fi

    echo "/dev/sda*" >> /etc/sysconfig/Devlist_calvados.conf
    echo "/dev/sda*" >> /etc/sysconfig/Devlist_xr.conf
    return
}

# Create /dev/usb* paths via symlinks in lxc
# Mount front Panel usb partitions in lxc
function usb_mount_lxc
{
    if [[ ! -b /dev/sdc && ! -b /dev/sdd ]]; then
        # LC or nousb case
        return
    fi

    # devpath 1.3 is front panel eusb

    local is_sdd=$(udevadm info --query=all -n /dev/sdd --attribute-walk 2> /dev/null | grep devpath | egrep "1.3" | wc -l)
    local is_sdc=$(udevadm info --query=all -n /dev/sdc --attribute-walk 2> /dev/null | grep devpath | egrep "1.3" | wc -l)
    if [[ ${is_sdd} != "0" ]]; then
        (cd /dev; ln -sf sdd usb)
        (cd /dev; ln -sf sdd1 usb1)
        (cd /dev; ln -sf sdd2 usb2)
        (cd /dev; ln -sf sdd3 usb3)
    elif [[ ${is_sdc} != "0" ]]; then
        (cd /dev; ln -sf sdc usb)
        (cd /dev; ln -sf sdc1 usb1)
        (cd /dev; ln -sf sdc2 usb2)
        (cd /dev; ln -sf sdc3 usb3)
    else
        echo "Front Panel USB not found."
        return
    fi

    # Note, we only look at the three partitions on eusb
    # the fourth partition is used as a reboot history area in classic.
    # The fourth partition area is left intact.
    if [[ "$1" == "bake" ]]; then
        eusb_mount_one usb1 usb 0
    else 
        eusb_mount_one usb1 usb 0
        eusb_mount_one usb2 usba 0
        eusb_mount_one usb3 usbb 0
    fi
}

# Create /dev/eusb* paths via symlinks in lxc
# Mount eusb partitions in lxc
function eusb_mount_lxc
{
    if [[ ! -b /dev/sdc && ! -b /dev/sdd ]]; then
        # LC or nousb case
        return
    fi

    # devpath 1.1 (P2) or 1.5 (P0/P1) is internal eusb
    # if sdc is either 1.1 or 1.5, then eusb is sdc, else eusb is sdd

    local is_sda=$(udevadm info --query=all -n /dev/sda --attribute-walk 2> /dev/null | grep devpath | egrep "1.1|1.5" | wc -l)
    local is_sdc=$(udevadm info --query=all -n /dev/sdc --attribute-walk 2> /dev/null | grep devpath | egrep "1.1|1.5" | wc -l)
    if [[ ${is_sda} != "0" ]]; then
        (cd /dev; ln -sf sda eusb)
        (cd /dev; ln -sf sda1 eusb1)
        (cd /dev; ln -sf sda2 eusb2)
        (cd /dev; ln -sf sda3 eusb3)
    elif [[ ${is_sdc} != "0" ]]; then
        (cd /dev; ln -sf sdc eusb)
        (cd /dev; ln -sf sdc1 eusb1)
        (cd /dev; ln -sf sdc2 eusb2)
        (cd /dev; ln -sf sdc3 eusb3)
    else
        (cd /dev; ln -sf sdd eusb)
        (cd /dev; ln -sf sdd1 eusb1)
        (cd /dev; ln -sf sdd2 eusb2)
        (cd /dev; ln -sf sdd3 eusb3)
    fi

    # Note, we only look at the three partitions on eusb
    # the fourth partition is used as a reboot history area in classic.
    # The fourth partition area is left intact.
    if [[ "$1" == "bake" ]]; then
        eusb_mount_one eusb3 eusbb 0
    elif [[ "$1" == "zapdisk" ]]; then
        eusb_format
    else 
        eusb_mount_one eusb1 eusb 0
        eusb_mount_one eusb2 eusba 0
        eusb_mount_one eusb3 eusbb 0
    fi
}

SAVEFS_PATH=
function save_sys_files {
    local retval

    local mnt_path=$1
    local dir_name=$2

    if [[ "$BOARDTYPE" != "LC" ]]; then
        return
    fi

    if [[ -z ${SAVEFS_PATH} ]]; then
        return
    fi

    modprobe dm-mod
    sleep 1

    step_log_console "Saving ${dir_name} files"
    bash /etc/init.d/raw_mount quiet rw ${mnt_path}
    retval=$?
    if [[ $retval != 0 ]]; then
        step_log_console "Unable to mount ${mnt_path}"
        return
    fi
    mnt_path=$( echo /${mnt_path}_lv* )

    if [[ ! -d /${mnt_path}/${dir_name} ]]; then
        step_log_console "No ${dir_name} files to save"
        umount /${mnt_path}
        return
    fi

    # np files get large, bzip tar np files
    if [[ "${dir_name}" == "np" ]]; then
        local curdir=`pwd`
        cd /${mnt_path}
        tar cjf /${SAVEFS_PATH}/${dir_name}.tar ${dir_name}
        retval=$?
        cd $curdir
    elif [[ "${dir_name}" == "pcie" ]]; then
        local curdir=`pwd`
        cd /${mnt_path}
        tar cjf /${SAVEFS_PATH}/${dir_name}.tar pcie*
        retval=$?
        cd $curdir
    else
        cp -pr /${mnt_path}/${dir_name} /${SAVEFS_PATH}
        retval=$?
    fi
    if [[ $retval != 0 ]]; then
        step_log_console "${dir_name} files save failed"
    fi

    umount /${mnt_path}
    retval=$?
    if [[ $retval != 0 ]]; then
        step_log_console "${dir_name} save umount failed"
    fi
    return
}

function restore_sys_files {
    local retval

    local mnt_path=$1
    local dir_name=$2

    if [[ "$BOARDTYPE" != "LC" ]]; then
        return
    fi

    if [[ -z ${SAVEFS_PATH} ]]; then
        return
    fi

    step_log_console "Restoring ${dir_name} files"
    if [[ ! -d /${SAVEFS_PATH}/${dir_name} && ! -f /${SAVEFS_PATH}/${dir_name}.tar ]]; then
        step_log_console "No ${dir_name} files to restore"
        return
    fi

    bash /etc/init.d/raw_mount quiet rw ${mnt_path}
    retval=$?
    if [[ $retval != 0 ]]; then
        step_log_console "Unable to mount ${mnt_path}"
        return
    fi
    mnt_path=$( echo /${mnt_path}_lv* )

    # np (xr) files are temporarily saved in (calvados)
    # this is because xr f/s is nor prepared during bake
    # update_misc_Scratch_np restores the np files
    if [[ "${dir_name}" == "np" ]]; then
        cp /${SAVEFS_PATH}/${dir_name}.tar /${mnt_path}
    elif [[ "${dir_name}" == "pcie" ]]; then
        cp /${SAVEFS_PATH}/${dir_name}.tar /${mnt_path}
    else
        rm -rf /${mnt_path}/${dir_name}
        cp -pr /${SAVEFS_PATH}/${dir_name} /${mnt_path}
    fi
    retval=$?
    if [[ $retval != 0 ]]; then
        step_log_console "${dir_name} files restore failed"
    fi

    umount /${mnt_path}
    retval=$?
    if [[ $retval != 0 ]]; then
        step_log_console "${dir_name} restore umount failed"
    fi
    return
}

function mount_savefs {
    if [[ "$BOARDTYPE" != "LC" ]]; then
        return
    fi

    SAVEFS_PATH=$(mktemp -d /tmp/isomnt.XXXXXX)
    if [[ -z ${SAVEFS_PATH} ]]; then
        return
    fi

    mount -t tmpfs -o size=${SAVEFS_SIZE} tmpfs ${SAVEFS_PATH}
    if [[ $? != 0 ]]; then
        step_log_console "Cannot mount tmpfs for saving old logs"
        SAVEFS_PATH=
        return
    fi

}

function umount_savefs {
    if [[ "$BOARDTYPE" != "LC" ]]; then
        return
    fi

    if [[ -z ${SAVEFS_PATH} ]]; then
        return
    fi

    umount ${SAVEFS_PATH}
}

function pd_create_tpa_volume {
    local disk1_pv=${1}
    local retval

    if [[ "$BOARDTYPE" != "RP" ]]; then
        return
    fi

    . /etc/init.d/app_volume_config.conf
    local TP_LV_A9K_NAME_FULL="/dev/pci_disk1/${TP_LV_NAME}"

    if [[ -b $TP_LV_A9K_NAME_FULL ]]; then
        return
    fi

    step_log_console "Creating App Hosting LV $TP_LV_NAME on $disk1_pv"
    lvcreate -L $DISK_TP_VOL_PART_SIZE  -n $TP_LV_NAME $disk1_pv >&107 2>&1
    retval=$?
    if [ $retval != 0 ]; then
        step_log_console "App Hosting LV creation failed"
        return
    fi
    format_partition "${TP_LV_A9K_NAME_FULL}" "AppHost"
}

function pd_remove_tpa_volume {
    local disk1_pv=${1}
    lvremove -f /dev/${disk1_pv}/app_lv0  >&107 2>&1
}

function pd_tp_lv_name_full {
    . /etc/init.d/app_volume_config.conf
    local TP_LV_A9K_NAME_FULL="/dev/pci_disk1/${TP_LV_NAME}"
    echo ${TP_LV_A9K_NAME_FULL}
}

function pd_create_separate_pv {
    false
    return
}

function check_zapdisk {

    local zapdisk=$(cat /proc/cmdline | grep zapdisk | wc -l)
    if [ "$zapdisk" == "1" ]; then
        return 1
    else
        return 0
    fi
}

# Remove non-DISK1_LV LVs
function pd_remove_old_volumes {
    local disk1_lv=${1}     # ssd_disk1
    local disk1_pv=${2}     # pci_disk1
    local _rc

    
    if [ -f /etc/init.d/ssd_upgrade.sh ]; then
        source /etc/init.d/ssd_upgrade.sh
        ssd_update_fw_exec "upgrade" "all"
    fi   
    check_zapdisk
    local _zapdisk=$?
    if [ "$_zapdisk" == "1" ]; then
        pv_list=$(pvdisplay | grep Name | paste - - | awk '{ print $3 }')
        vg_list=$(vgscan | grep Found | awk '{print $4}') | tr -d '"'
    else
        pv_list=$(pvdisplay | grep Name | paste - - | grep -v ${disk1_pv} | awk '{ print $3 }')
        vg_list=$(vgscan | grep -v ${disk1_pv}| grep Found | awk '{print $4}' | tr -d '"')
    fi

    if [[ -z $pv_list ]]; then
        step_log_console "Volume-Cleaning no PVs"
        return
    fi

    lv_list=$(lvdisplay | grep Name | paste - - | sed -e "s; *LV Name *;/;" -e "s; *VG Name *;;" | grep -v ${disk1_lv} | awk '{print $2 $1}')

    mount_savefs
    save_sys_files admin_var_log obfl
    save_sys_files xr_misc_scratch np
    save_sys_files xr_misc_scratch pcie

    step_log_console "Volume-Cleaning"

    for _vol in $lv_list; do
        step_log_file "Unmounting Volume $_vol"
        umount -f $_vol >> /tmp/log 2>&1
        _rc=$?
        if [[ $_rc != 0 ]]; then
            step_log_file "Unmounting $_vol returns $_rc"
        fi
        step_log_file "Changing Volume $_vol"
        lvchange -an $_vol >&107 2>&1
        step_log_console "Removing $_vol"
        yes | lvremove -f $_vol >&107 2>&1
        _rc=$?
        if [[ $_rc != 0 ]]; then
            step_log_file "lvremove $_vol returns $_rc"
        fi
    done

    for _vg_vol in $vg_list; do
        step_log_console "Removing Logical Volume $_vg_vol"
        vgchange -an $_vg_vol >&107 2>&1
        yes | vgremove -f $_vg_vol >&107 2>&1
        _rc=$?
        if [[ $_rc != 0 ]]; then
            step_log_console "vgremove $_vg_vol returns $_rc"
        fi
    done

    for _pv_vol in $pv_list; do
        step_log_console "Removing Physical Volume $_pv_vol"
        yes | pvremove -ff -y $_pv_vol >&107 2>&1
        _rc=$?
        if [[ $_rc != 0 ]]; then
            step_log_console "pvremove $_pv_vol returns $_rc"
        fi
        dd if=/dev/zero of=$_pv_vol bs=512 count=255 >&107 2>&1
    done

    for _part in `awk '/dm-/{print $4}' /proc/partitions`; do
        yes | pvremove -ff -y $_part >&107 2>&1
        dd if=/dev/zero of=$_part bs=512 count=255 >&107 2>&1
    done


    step_log_console "Volume-Cleaning Completed"

    pd_create_tpa_volume $disk1_pv
}

# filesystem check DISK1_LV LVs
function pd_check_fs_secondary_vols {
    local disk1_pv=${1}
    local disk1_lv=${2}
    local retval=0

    # unlike PI code, additionally filter for DISK1_LV volumes
    # above pd_remove_old_volumes, removes any panini_vol_grp volumes
    # so no need to check for panini_vol_grp volumes here

    lvchange -a y /dev/${disk1_pv} > /dev/null 2>&1
    ssd_lvnames=`lvdisplay ${disk1_pv} | grep "LV Path" | awk '{print $3}' | grep ${disk1_lv}`
    for lv in ${ssd_lvnames[@]}; do
        check_fs_sb $lv
        retval=$?
        if [ $retval != 0 ]; then
            step_log_console "Filesystem for $lv is unexpected!!"
            step_log_console "Will re-format ${disk1_pv} volumes"
            break
        else
            step_log_console "$lv ok"
        fi
    done
    lvchange -a n /dev/${disk1_pv} > /dev/null 2>&1
    return $retval
}

#
# Cook up the sysctl to make kernel panic on all scenarios
#
function configure_panics
{
    /sbin/sysctl -w kernel.softlockup_panic=1
    /sbin/sysctl -w kernel.panic_on_oops=1
    /sbin/sysctl -w kernel.panic_on_stackoverflow=1
    /sbin/sysctl -w kernel.hung_task_panic=1
}

function pci_disable_8749_unused_ports
{
    local ports=$( get_card_specific_value 8749_UNUSED_PORTS)

    #Perform only for cards that report unused ports
    if [ ! -z "$ports" ]; then
        for port in $ports; do
            bdf=$( lspci -n | grep 8749 | grep "$port\.0" | awk '{print $1}' )
            setpci -s $bdf CAP_EXP+10.l=10:10
        done
    fi

}

#
# Initialize PCI during Host bootup
#
function init_pci
{
    source $PD_FUNCS
    pci_disable_8717_down
    pci_disable_8749_unused_ports
}

function rotate_pcie_logs {
    local log_base=$1
    local i
    local j

    mkdir /tmp/pcie > /dev/null 2>&1

    if [[ ! -d ${log_base} ]]; then
        return
    fi

    update_misc_scratch_xr

    if [[ ! -d ${log_base}/pcie ]]; then
        mkdir ${log_base}/pcie
        return
    fi

    # if pcie dir is empty, nothing to rotate
    if [[ "$(ls -A ${log_base}/pcie 2> /dev/null)" == "" ]]; then
        return
    fi

    # rotate up to 10 prior pcie log dirs
    for i in {10..1}; do
        if [[ ! -d ${log_base}/pcie.$i ]]; then
            continue
        fi
        j=$(( i + 1 ))
        if [[ -d ${log_base}/pcie.$j ]]; then
            rm -rf ${log_base}/pcie.$j
        fi
        mv ${log_base}/pcie.$i ${log_base}/pcie.$j
    done
    if [[ -d ${log_base}/pcie ]]; then
        # compress the logs
        local cwd=$( pwd )
        cd ${log_base}/pcie; tar --remove-files -czf pcie.tgz *log
        cd $cwd
        mv ${log_base}/pcie ${log_base}/pcie.1
    fi

    # fresh pcie dir
    mkdir ${log_base}/pcie
}

function pd_notify_sysadmin_vm_started() {
   kmsg_log "pd_notify_sysadmin_vm_started"
}

function create_recovery_file() {
    local recovery_marker=/$1/recovery.txt
    if [[ -f $recovery_marker ]]; then
        return
    fi
    echo "Do not delete this file" > $recovery_marker
    echo "This file indicates the presence of a recovery iso image" >> $recovery_marker
    if [[ -f /etc/show_version.txt ]]; then
        cat /etc/show_version.txt >> $recovery_marker
    fi
}

function copy_recovery_files() {
    # This check if to check if eusbb is mounted to /dev/sdc3 or not
    # if not then we should not continue 
    local var=$(cat /proc/mounts | grep eusbb | wc -l)
    if [ "$var" != 1 ]; then
        echo "eusbb mount not found..returning"
        rm -rf /misc/disk1/tftpboot/eusbb
        return
    fi
    
    if [[ -f /eusbb/recovery.txt ]]; then
        rm -rf /misc/disk1/tftpboot/eusbb
        return
    fi
        
    echo "Copying recovery files"
    rm -rf /eusbb/boot
    mkdir /eusbb/boot
    rm -rf /eusbb/EFI
    mkdir /eusbb/EFI
    mkdir /eusbb/EFI/boot

    mv /misc/disk1/tftpboot/eusbb/*.cfg /eusbb/
    mv /misc/disk1/tftpboot/eusbb/*.txt /eusbb/

    mywd=${pwd}

    cd /eusbb
    tar -cvf /var/log/migration.tar *.cfg *.txt

    if [[ -d /misc/disk1/tftpboot/eusbb ]]; then
        cd /misc/disk1/tftpboot/eusbb/
        find * -print | grep -v ".iso" | cpio -pd /eusbb/
        rm -rf /misc/disk1/tftpboot/eusbb
    fi
    cd ${mywd}

    cp -r /misc/disk1/tftpboot/boot/* /eusbb/boot
    cp /misc/disk1/tftpboot/grub.efi /eusbb/EFI/boot
    
    cat > /eusbb/EFI/boot/grub.cfg <<%%
set default=0
terminal_input console
terminal_output console

set timeout=5
set iso_filename=system_image.iso

menuentry "ASR9K Host OS - ISO (\$iso_filename)" {
        echo "Booting from embedded USB"
        echo "Loading Kernel..."
        linux (hd0,msdos3)/boot/bzImage root=/dev/ram install=/dev/sda pci=hpmemsize=0M,hpiosize=0M platform=asr9k console=ttyS0,115200 eusbboot=\$iso_filename intel_idle.max_cstate=0 processor.max_cstate=1 quiet
        echo "Loading initrd..."
        initrd (hd0,msdos3)/boot/initrd.img signfile=(hd0,msdos3)/boot/signature.initrd.img
}
%%
    cp -L /misc/disk1/tftpboot/system_image.iso /eusbb/system_image.iso
    # Set recovery marker
    create_recovery_file /eusbb
}

#Update lxc respawn count
# arg1 : vm name
function update_virt_respawn_count()
{
    local issu_id=$2

    case "$1" in
    xr-vm)
        # VKG_NGISSU_PROTO - Check the logic with issu id 0
        if [ $issu_id -eq 0 ]; then
            LXC_COUNTER_ADDR=$($myshowemt addr | grep xr_lxc_respawn_count | cut -d= -f 2)
        else
            LXC_COUNTER_ADDR=$($myshowemt addr | grep xr2_lxc_respawn_count | cut -d= -f 2)
        fi
        ;;
    sysadmin-vm)
        LXC_COUNTER_ADDR=$($myshowemt addr | grep cal_lxc_respawn_count | cut -d= -f 2)
    ;;

    *)
        kmsg_log "invalid virt type" $1
        # do not get unknown address
        return
        ;;
    esac

    kmsg_log "update_virt_respawn_count: $1"

    CURR_CNT=$(pcimemread `printf '%x' $LXC_COUNTER_ADDR` 4 | grep -v PCI | awk '{ print $3 ;}' | sed -e 's/^00/0x/')

    count=`expr $(($CURR_CNT)) + 1`
    incr=`printf '%x' $count`
    pcimemwrite `printf '%x' $LXC_COUNTER_ADDR` 4 $incr > /dev/null 2>&1
}


#Get lxc respawn count
# arg1 : vm name
function get_virt_respawn_count()
{
    local issu_id=$2

    case "$1" in
    xr-vm)
        if [ $issu_id -eq 0 ]; then
            LXC_COUNTER_ADDR=$($myshowemt addr | grep xr_lxc_respawn_count | cut -d= -f 2)
        else
            LXC_COUNTER_ADDR=$($myshowemt addr | grep xr2_lxc_respawn_count | cut -d= -f 2)
        fi
        ;;
    sysadmin-vm)
        LXC_COUNTER_ADDR=$($myshowemt addr | grep cal_lxc_respawn_count | cut -d= -f 2)
    ;;

    *)
        kmsg_log "invalid virt type" $1
        # do not get unknown address
        return
        ;;
    esac

    kmsg_log "get_virt_respawn_count: $1"

    CURR_CNT=$(pcimemread `printf '%x' $LXC_COUNTER_ADDR` 4 | grep -v PCI | awk '{ print $3 ;}' | sed -e 's/^00/0x/')
    count=$(($CURR_CNT))

    return $count
}

# initialize fabric on XR vm
# arg1 : vm name
function virt_fabric_init()
{
    local issu_id=$2
    local issu_role=$3    

    case "$1" in
    xr-vm)
        ;;

    *)
        kmsg_log "virt_fabric_init: invalid lxc $1"
        return
        ;;
    esac

    get_virt_respawn_count $1 $issu_id
    xr_respawn_cnt=$?

    if [ $xr_respawn_cnt -eq 1 ]; then
        kmsg_log "virt_fabric_init: first boot init skipped"
        return
    fi

    # ISSU role primary is 1
    if [ $issu_role -ne 1 ]; then
        kmsg_log "virt_fabric_init: skipped for issu role $issu_role"
        return
    fi

    kmsg_log "virt_fabric_init $1 issu role $issu_role"

    source /etc/init.d/fabric.sh

    is_vxr_sim
    if [[ $? == 0 ]]; then
        fabric_init
    fi
}


function configure_sysadmin_eobc
{
    if [ -L /sys/class/net/eth-vnic1 ]; then
        # Rename vnic1 as vf1
        kmsg_exec "ifconfig eth-vnic1 down"
        kmsg_exec "ip link set eth-vnic1 name eth-vf1"
        kmsg_exec "ifconfig eth-vf1 mtu 9700"
    fi
}

A9K_PD_CRIT_PATH="/root"

function wait_for_ipu() {
    local npass
    let "npass = $1 * 4"
    
    for  (( i = 0; i < $npass; i++ )); do
        if [[ -f /tmp/ipu_in ]]; then
            rm -rf /tmp/ipu_in
            break
        fi
        sleep 0.25
    done

    if [[ $i == $npass ]]; then
        kmsg_log "Failed to hot-plug IPU"
    else 
        kmsg_log "IPU hot-plug seen after $i tries" 
    fi
}

function platform_vm_specific_configuration() {
   local VM_TYPE=$1
   local VIRT_METHOD=$2
   local vdev_id
   local DMAINFO_WHERE

   modprobe uio_pci_proxy >/dev/null 2>&1
   if [ "$VM_TYPE" == "sysadmin-vm" ]; then
    DMAINFO_WHERE="DMAINFO_CALVADOS"
   else
    DMAINFO_WHERE="DMAINFO_XR"
   fi
   local dmainfo=$( get_card_specific_value $DMAINFO_WHERE )
   modprobe uio_dma_proxy dmainfo=\"$dmainfo\"
   sleep 1

   a9k_pcie_proxy_enable $VM_TYPE $VIRT_METHOD

   # Bring-in HP devices and enable interfaces
   if [ "$VM_TYPE" == "sysadmin-vm" ]; then
       modprobe ixgbevf
       ifconfig eth1 down
       ip link set eth1 name eth-vf1
       ifconfig eth-vf1 mtu 9700
       modprobe igba9k
       ifconfig eth1 down
       ip link set eth1 name eth-mgmt2
       ifconfig eth-mgmt2 mtu 1500
       initctl start proxy_attach_cal
       PCIE_DEBUG=/misc/scratch /opt/cisco/calvados/bin/msixd_static \
                                -v 0x1137 -d 0x2345 -n 64 \
                                -f $A9K_PD_CRIT_PATH/msivec0 &
       
       # Device Passthrough takes some time, stall further access to IPU
       wait_for_ipu 8
                          
   else
       initctl start proxy_attach_ipu
       # Device Passthrough takes some time, stall further access to IPU
       wait_for_ipu 8

       # XR v1->2346(ALL), v2->2347(LC only)
       local issu_role=`xr_bootstrap_field_value ISSU_ROLE`
       local issu_id=`xr_bootstrap_field_value ISSU_ID`
       if [[ $issu_id -eq 1 ]] && [[ "${BOARDTYPE}" == "LC" ]]; then
           vdev_id=0x2347
       else
           vdev_id=0x2346
       fi

       update_virt_respawn_count $VM_TYPE $issu_id
       virt_fabric_init $VM_TYPE $issu_id $issu_role

       initctl start proxy_attach_xr
       local msigrp_count=$( get_card_specific_value MSIGRP0_COUNT )
       PCIE_DEBUG=/misc/scratch /pkg/sbin/msixd_static \
                                -v 0x1137 -d $vdev_id -n $msigrp_count \
                                -f $A9K_PD_CRIT_PATH/msivec0 &

       msigrp_count=$( get_card_specific_value MSIGRP1_COUNT )
       if [[ $msigrp_count != "0" ]]; then
           let "vdev_id = $vdev_id + 2"
           vdev_id=$( printf "0x%x" $vdev_id )
           PCIE_DEBUG=/misc/scratch /pkg/sbin/msixd_static \
                                -v 0x1137 -d $vdev_id -n $msigrp_count \
                                -f $A9K_PD_CRIT_PATH/msivec1 &
       fi

       msigrp_count=$( get_card_specific_value MSIGRP2_COUNT )
       if [[ $msigrp_count != "0" ]]; then
           let "vdev_id = $vdev_id + 2"
           vdev_id=$( printf "0x%x" $vdev_id )
           PCIE_DEBUG=/misc/scratch /pkg/sbin/msixd_static \
                                -v 0x1137 -d $vdev_id -n $msigrp_count \
                                -f $A9K_PD_CRIT_PATH/msivec2 &
       fi

       rmmod igb
   fi
   lspci > /misc/scratch/pcie/lspci.log
}

function platform_specific_startup_clean() {
   FILE=$1
   TMP_FILE="${FILE}.tmp"

   sed -n '/mandatory/!p' $FILE | sed -n '/respawn/!p' > $TMP_FILE
   echo "respawn: OFF" >> $TMP_FILE
   cp -f $TMP_FILE $FILE
   rm -f $TMP_FILE
}


function platform_specific_create_stub_dll() {
    DLL=$1
    if [ ! -e $DLL ] ; then
        kmsg_log "Stubbing $DLL"
        ln -s /lib64/libc-2.20.so $DLL
    fi
}

function platform_vxr_sim_cleanup() {
    BRD_TYPE=`cat /proc/cmdline | sed 's/^.*boardtype=//' | cut -d" " -f1`
    kmsg_log "platform_vxr_sim_cleanup start for type $BRD_TYPE"
    # this is common for both RP and LC. How about FC ?
    find /opt/cisco/XR/packages/ -name 'fab*startup' | xargs rm
    find /opt/cisco/XR/packages/ -name 'fia*startup' | xargs rm
    if [ "$BRD_TYPE" == "RP" ]; then
       #
       # There is no plan to emulate this HW on sim, so removal is
       # fine.
       #
       find /opt/cisco/XR/packages/ -name 'zl30160_server.*' | xargs rm
       find /opt/cisco/XR/packages/ -name 'rsptiming.*' | xargs rm
       find /opt/cisco/XR/packages/ -name 'pcie_flexr.*' | xargs rm
       find /opt/cisco/XR/packages/ -name 'ether_ctrl_msg_server.*' | xargs rm
       find /opt/cisco/XR/packages/ -name 'icpe_satmgr.*' | xargs rm
       find /opt/cisco/XR/packages/ -name 'vkg_mgid_prgm.*' | xargs rm
    fi
    if [ "$BRD_TYPE" == "LC" ]; then
       #
       # There is no plan to emulate this HW on sim, so removal is
       # fine.
       #
       find /opt/cisco/XR/packages/ -name 'prm_server_to.*' | xargs rm
       find /opt/cisco/XR/packages/ -name 'ether_ctrl_msg_client.*' | xargs rm
       find /opt/cisco/XR/packages/ -name 'pcie_flexr.*' | xargs rm
       find /opt/cisco/XR/packages/ -name 'vkg_mgid_prgm.*' | xargs rm
       find /opt/cisco/XR/packages/ -name 'pfm_node_lc.*' | xargs rm
       find /opt/cisco/XR/packages/ -name 'fpd_fpga_agent.*' | xargs rm
       find /opt/cisco/XR/packages/ -name 'zllc.*' | xargs rm
       find /opt/cisco/XR/packages/ -name 'ipv6_ma.*' | xargs rm
       find /opt/cisco/XR/packages/ -name 'ipv6_io.*' | xargs rm
       find /opt/cisco/XR/packages/ -name 'mpls_io.*' | xargs rm
       find /opt/cisco/XR/packages/ -name 'mpls_io_ea.*' | xargs rm
       find /opt/cisco/XR/packages/ -name 'ipv6_nd.*' | xargs rm

       find /opt/cisco/XR/packages/ -name 'libspp_cfm_offload.*' | xargs rm
       find /opt/cisco/XR/packages/ -name 'libspp_frag.*' | xargs rm
       find /opt/cisco/XR/packages/ -name 'libspp_ptp_offload.*' | xargs rm

       find /pkg/lib/spp_plugins/   -name 'libspp_cfm_offload.*' | xargs rm
       find /pkg/lib/spp_plugins/   -name 'libspp_frag.*' | xargs rm
       find /pkg/lib/spp_plugins/   -name 'libspp_ptp_offload.*' | xargs rm

       #
       # Stub some libraries
       # Plans need to be made to eventually fix these libraries.
       #
       platform_specific_create_stub_dll /pkg/lib/lib_gcp_pbr_ea_hal.so
       platform_specific_create_stub_dll /pkg/lib/lib_pbrkm_hal.so
       platform_specific_create_stub_dll /pkg/lib/lib_vservice_hal.so
       platform_specific_create_stub_dll /pkg/lib/libetna_lda_api.so
       platform_specific_create_stub_dll /pkg/lib/libfrenzyflash.so
       platform_specific_create_stub_dll /pkg/lib/libvkg_l2fib_hal_mac_cache.so

       #
       # This will remove mandatory, and turn respawn off, until we can further debug
       # These processes
       # Creates a MY_TMP_SCRIPT file, which makes the necessary function calls
       #

       MY_PWD=`pwd`
       cd /pkg/startup/

       ls | awk '{print "platform_specific_startup_clean " $1}' > MY_TMP_SCRIPT
       chmod a+x MY_TMP_SCRIPT
       source ./MY_TMP_SCRIPT
       rm ./MY_TMP_SCRIPT

       cd /pkg/init.d/

       ls *.init | awk '{print "platform_specific_startup_clean " $1}' > MY_TMP_SCRIPT
       chmod a+x MY_TMP_SCRIPT
       source ./MY_TMP_SCRIPT
       rm ./MY_TMP_SCRIPT

       cd $MY_PWD
    fi
    kmsg_log "platform_vxr_sim_cleanup end for type $BRD_TYPE"
}

function platform_specific_configuration() {
   local VM_TYPE=$1
   local VIRT_METHOD=$2
   local PLATFORM=$3

   is_vxr_sim
   if [[ $? != 0 ]]; then
       platform_vxr_sim_cleanup
   fi

   set_var_log_links
    
   # Must match xr_startup_envs.sh:export PCIE_DEBUG=/misc/scratch
   rotate_pcie_logs /misc/scratch

   if [ "$VM_TYPE" != "hostos" ] && [ $VIRT_METHOD == "vm" ]; then
       platform_vm_specific_configuration $VM_TYPE $VIRT_METHOD
   fi 

   get_board_type
   kmsg_log "platform_specific_configuration $VM_TYPE $VIRT_METHOD"

   echo 1 > $A9K_PD_CRIT_PATH/pcie_scan_lock
   
   if [ "$VM_TYPE" == "hostos" ]; then
       set_proc_sys_net
       init_pci
       if [[ "${BOARDTYPE}" == "LC" ]]; then
           grub_cmdline_resync $1 $BOARDTYPE
       fi
   fi

   if [ $VIRT_METHOD == "vm" ] && [ $VM_TYPE == "sysadmin-vm" ] && [ $BOARDTYPE == "LC" ]; then
       # Canbus driver expects ttyS1 to talk to CBC
       ln -sf /dev/ttyS2 /dev/ttyS1
   fi

   # eusbb_size implies two settings
   # In calvados:
   # min partition size of /eusbb, above which recovery images will be copied
   # In host:
   # min partition size of /eusbb, below which the drive will be reformatted
   local eusbb_size=1500000

   if [ "$VM_TYPE" != "hostos" ]; then
       eusb_mount_lxc "all"
       if [ "$VM_TYPE" == "sysadmin-vm" ]; then
           IPU_BASE=${IPU_BASE} bash /etc/init.d/genhosts.sh / admin
            configure_sysadmin_eobc

            if [[ $VIRT_METHOD == "vm" ]]; then
                eusb_mount_one "disk/by-label/eusba" eusba 1
                eusb_mount_one "disk/by-label/eusbb" eusbb 1
            fi

            if [[ -d /eusbb ]]; then
                if [[ $VIRT_METHOD == "vm" ]]; then
                    local val=$( fdisk -s /dev/disk/by-label/eusbb )
                else
                    local val=$( fdisk -s /dev/eusb3 )
                fi
 
                if [[ $val -gt $eusbb_size ]]; then
                   echo "Copying recovery files"
                   copy_recovery_files &
                fi
            fi
            is_vxr_sim
            if [[ $? != 0 ]]; then
                find /opt/cisco/calvados/etc/startup/ -name 'esd_asr9k.startup' |
                xargs rm
            fi
       else
           # Already called for VMs. Dont call again
           # LXC flow will log discovered devices
           if [[ $VIRT_METHOD == "lxc" ]]; then
               a9k_pcie_proxy_enable $VM_TYPE $VIRT_METHOD
           fi
           IPU_BASE=${IPU_BASE} bash /etc/init.d/genhosts.sh / xr 

           if [[ $VIRT_METHOD == "vm" ]]; then
               # Mount eusb in XR VM
               eusb_mount_one "disk/by-label/eusb" eusb 1

               # Mount Host App Volume.
               if [[ -b /dev/disk/by-label/AppHost ]]; then
                   if [[ ! -d /misc/app_host ]]; then
                       mkdir /misc/app_host
                   fi
                   kmsg_exec "mount -o sync /dev/disk/by-label/AppHost /misc/app_host"
               fi
           fi
           ln -s /eusb /eusbb
           update_eusb_xr

           if [[ ! -f //misc/config/config/running/alternate_cfg/router.cfg ]]; then
                mkdir -p /misc/config/config/running/alternate_cfg/
                if [[ -f /eusbb/cXR_xr_plane_converted_eXR.cfg ]]; then
                    cp -rf /eusbb/cXR_xr_plane_converted_eXR.cfg //misc/config/config/running/alternate_cfg/router.cfg
                    echo "Restoring the Config with Converted Config."
                elif [[ -f /eusbb/cXR_xr_plane.cfg ]]; then
                    cp -rf /eusbb/cXR_xr_plane.cfg //misc/config/config/running/alternate_cfg/router.cfg
                    echo "Restoring the Config."
                fi

                if [[ "$?" != "0" ]]; then
                    step_log_console "Unable to copy the config file"
                    echo "Unable to copy the config file"
                fi
                mkdir /eusbb/backup_config
                mv /eusbb/cXR_xr_*.cfg /eusbb/backup_config
                if [[ "$?" != "0" ]]; then
                    step_log_console "Unable to move the config file"
                    echo "Unable to move the config file"
                fi
           fi

       fi
       grub_cmdline_resync $1 $BOARDTYPE
       return
   fi

   display_image_details 1
   echo "${BOARDTYPE}: ${PD_CARDTYPE}, IPU: ${HW_CARDTYPE}"
   check_cardtype
   if [[ "$?" != "0" ]]; then
        echo "Warning: Invalid Card Type ${PD_CARDTYPE} detected"
   fi

   if [[ "$BOARDTYPE" == "RP" ]]; then
        ifconfig eth1 up promisc
   fi

   if [[ -b /dev/eusb ]]; then
        local do_format=0
        if [[ -b /dev/eusb3 ]];
        then
            local val=$( fdisk -s /dev/eusb3 )
            if [[ $val -lt $eusbb_size ]]; then
                do_format=1
            fi
        fi
        if [[ "$do_format" == "1" ]]; then
            eusb_format
        else
            eusb_to_extfs
        fi
        if [[ $VIRT_METHOD == "lxc" ]]; then 
            eusb_mount_host
        fi
   fi

   host_specific_configuration $VM_TYPE $VIRT_METHOD

   if [[ "$BOARDTYPE" == "RP" ]]; then
       grub_cmdline_resync $1 $BOARDTYPE
   fi

   return
}

function clean_disk_internal()
{
    local days=$1
    local disk1mnt=$2
    while true; do
        usage=$(df -kh ${disk1mnt} | awk 'FNR == 2{print $5 }' | sed 's/[^0-9]*//g')
        if [[ ${usage} -gt 50 && ${days} -ge 0 ]]; then
            find ${disk1mnt}/*core* -mmin +$((1440*${days})) -exec rm {} \; > /dev/null 2>&1
            find ${disk1mnt}/*.tgz -mmin +$((1440*${days})) -exec rm {} \; > /dev/null 2>&1
            find ${disk1mnt}/*.kdump -mmin +$((1440*${days})) -exec rm {} \; > /dev/null 2>&1
            days=$(($days-5))
        else 
           break
        fi
    done
}

function pd_pxe_second_disk_check()
{
    local device
    local disk1mnt
    local usage
    local days
    local list_str
    
    if [ ! -z ${SIMULATION} ]; then
        
        activate_disk_one_vg >&107 2>&1
        
        device=/dev/mapper/pci_disk1-ssd_disk1_calvados_1
        disk1mnt=$(mktemp -d /tmp/disk1dev.XXXXXX) 
        
        mount ${device} ${disk1mnt} >&107 2>&1
        usage=$(df -kh ${disk1mnt} | awk 'FNR == 2{print $5 }' | sed 's/[^0-9]*//g')
        days=$((100-usage))
        
        clean_disk_internal ${days} ${disk1mnt}
        
        usage=$(df -kh ${disk1mnt} | awk 'FNR == 2{print $5 }' | sed 's/[^0-9]*//g')
        if [[ ${usage} -gt 80 ]]; then
            list_str=$(ls -alhS ${disk1mnt} | grep ^- |  head -5 | awk '{ printf "%d. %-60s\t%-20s\n", NR, $9, $5 }')
            step_log_console "Secondary Disk Usage = ${usage}%. Following are top 5 files occupying highest disk space:"
            while read -r line; do
                step_log_console "${line}"
            done <<< "${list_str}"
            step_log_console "Please review and delete unused files otherwise router may not install successfully"
        fi
        [ -d "${disk1mnt}" ] && umount ${disk1mnt} && rmdir ${disk1mnt}
    fi
}

function strip_replace()
{
    local file=$3

    sed -i -e "s;${1}[^ ]*;;"             $file # strip
    sed -i -e "s;console=;${2} console=;" $file # replace
    sed -i -e "s; [ *]; ;"                $file # white space
}

function update_bigphysarea() {
    local GRUB_FILE=$1
    local BIGPHYSAREA_WHERE=$2

    local bigphysarea=$( get_card_specific_value $BIGPHYSAREA_WHERE )
    kmsg_log "update_bigphysarea $bigphysarea"
    strip_replace "bigphysarea" "bigphysarea=$bigphysarea" $GRUB_FILE
}

function asr9k_update_grub() {
    local GRUB_FILE=$1
    local BIGPHYSAREA_WHERE=$2

    strip_replace "quiet" "quiet" ${GRUB_FILE}

    if [ $VIRT_METHOD == "vm" ]; then
        strip_replace "virtvm" "virtvm" ${GRUB_FILE}
    else
        strip_replace "virtlxc" "virtlxc" ${GRUB_FILE}
    fi

    update_bigphysarea ${GRUB_FILE} $BIGPHYSAREA_WHERE

    is_vxr_sim
    # Don't treat LC special under VXR.
    if [[ "${BOARDTYPE}" == "LC"  && $? == 0 ]]; then
        get_chassis_details_cmd
        if [[ ! -z ${CMD_CHASSIS_TYPE} ]]; then
            strip_replace "chassis_type" "chassis_type=${CMD_CHASSIS_TYPE}"  \
                ${GRUB_FILE}
        fi
        if [[ ! -z ${CMD_CHASSIS_SERIAL} ]]; then
            strip_replace "chassis_serial" "chassis_serial=${CMD_CHASSIS_SERIAL}" \
                ${GRUB_FILE}
        fi
        if [[ ! -z ${CMD_CHASSIS_PID} ]]; then
            strip_replace "chassis_pid" "chassis_pid=${CMD_CHASSIS_PID}" \
                ${GRUB_FILE}
        fi
    else
        get_chassis_details_ipu
        if [[ ! -z ${CHASSIS_TYPE} ]]; then
            strip_replace "chassis_type" "chassis_type=${CHASSIS_TYPE}" \
                ${GRUB_FILE}
        fi
        if [[ ! -z ${CHASSIS_SERIAL} ]]; then
            strip_replace "chassis_serial" "chassis_serial=${CHASSIS_SERIAL}" \
                ${GRUB_FILE}
        fi
        if [[ ! -z ${CHASSIS_PID} ]]; then
            strip_replace "chassis_pid" "chassis_pid=${CHASSIS_PID}" \
                ${GRUB_FILE}
        fi
    fi
    # Un-comment once CSCvg74562 is fixed 
    # sed -i -e "s;aer=off;aer=on aer=logonly;" ${GRUB_FILE}

    # intel_idle.max_cstate=0 disables the intel_idle driver,
    # processor.max_cstate=1 enforces CPU to fully operational state
    strip_replace "intel_idle" "intel_idle.max_cstate=0" ${GRUB_FILE}
    strip_replace "processor.max_cstate" "processor.max_cstate=1" ${GRUB_FILE}

    sed -i -e "s; cgroup_disable=memory;;"                               \
           -e "s; installiso=system_image.iso;;"                         \
           -e "s;terminal_output serial;terminal_output console;"        \
           -e "s; nointremap msireserve=133-182:0x2345,183-232:0x2346;;" \
           -e "/ISSU/,\$d"                                               \
           -e "s;ttyS1;ttyS0;"                                           \
                ${GRUB_FILE}

    return;
}

function pd_pxe_update_pxeboot_grub() {
    local input_dir=$1
    local is_pxeboot=`grep "root=/dev/ram" /proc/cmdline | wc -l`
    if [ "$is_pxeboot" != "0" ]; then
        # PXE Bake Enviroment
        local grubcfg=${input_dir}/grub.cfg
    else
        local grubcfg=$1
    fi

    strip_replace "platform=" "platform=asr9k" ${grubcfg}
    asr9k_update_grub ${grubcfg} "BIGPHYSAREA_HOST"
    

    # Propagate cmdline settings
    local pxedebug=$(cat /proc/cmdline | grep pxedebug | wc -l)
    if [ "$pxedebug" == "1" ]; then
        strip_replace "pxedebug" "pxedebug" ${grubcfg}    
    fi

    local hostconsole=$(cat /proc/cmdline | grep hostconsole | wc -l)
    if [ "$hostconsole" == "1" ]; then
        strip_replace "hostconsole" "hostconsole" ${grubcfg}    
    fi

    local zapdisk=$(cat /proc/cmdline | grep zapdisk | wc -l)
    if [ "$zapdisk" == "1" ]; then
        strip_replace "zapdisk" "zapdisk" ${grubcfg}
    fi

    if [ "$is_pxeboot" != "0" ]; then
        chmod 777 -R .

        rm -rf ${input_dir}/eusbb
        mkdir -p ${input_dir}/eusbb

        # Migrating via pxeboot, save and restore /harddiskb: content (like eusb migration)
        if [[ -d /eusbb ]]; then
            if [ ! -f /eusbb/recovery.txt ]; then
                step_log_console "Taking Backup from eusb."
                cp -rf `find /eusbb/ -mindepth 1 -maxdepth 1 | \grep -v "efi\|EFI\|BOOT\|boot"` ${input_dir}/eusbb
            fi 
        fi
    fi
}

function pd_pxe_update_efi_grub () {
    local input_dir=$1
    local grubcfg=${input_dir}/grub.cfg
    local chassistype
    local is_pxeboot=`grep "root=/dev/ram" /proc/cmdline | wc -l`

    asr9k_update_grub ${grubcfg} "BIGPHYSAREA_HOST"
    if [[ ! -z ${PD_CARDTYPE} ]]; then
        strip_replace "cardtype" "cardtype=${PD_CARDTYPE}" ${grubcfg}            
    fi

    strip_replace "iputype" "iputype=${HW_CARDTYPE}" ${grubcfg}
    strip_replace "quiet" "quiet" ${grubcfg}

    is_vxr_sim
    # Don't treat LC special under VXR.
    if [[ "${BOARDTYPE}" == "LC"  && $? == 0 ]]; then
        chassistype=${CMD_CHASSIS_TYPE}
    else
        chassistype=${CHASSIS_TYPE}
    fi

    step_log_console    "${BOARDTYPE}: $PD_CARDTYPE, IPU: ${HW_CARDTYPE}, CHASSIS: ${chassistype}"
    check_chassistype ${chassistype}
    if [[ "$?" != "0" ]]; then
        step_log_console "${BOARDTYPE}: $PD_CARDTYPE, IPU: ${HW_CARDTYPE}, CHASSIS: ${chassistype}"
        step_log_console "Invalid Chassis Type ${chassistype} detected. Exiting install"
        exit 1
    fi

    if [ "$is_pxeboot" != "0" ]; then
        step_log_console "Setting disk boot mode"
        EMT_MODE_OFF=0x20
        EMT_MODE_PADDR=$(($IPU_BASE + $NVRAM_OFF + $EMT_OFF + $EMT_MODE_OFF))
        pcimemwrite `printf "%x" $EMT_MODE_PADDR` 4 2 >/dev/null 2>&1
    fi

    strip_replace "intremap" "intremap=off" ${grubcfg}

    restore_sys_files admin_var_log obfl
    # Note: we really need to copy np files to xr_var_log
    # However, xr_var_log is not available yet
    # Instead, save now to admin_var_log and retrieve later
    # The retrieval is done in pd-functions:update_misc_scratch_np
    restore_sys_files admin_var_log np 
    restore_sys_files admin_var_log pcie 
    umount_savefs

    if [[ "${BOARDTYPE}" == "LC" ]]; then
        return;
    fi

    local hostconsole=$(cat /proc/cmdline | grep hostconsole | wc -l)
    if [ "$hostconsole" == "1" ]; then
        strip_replace "hostconsole" "hostconsole" ${grubcfg}
    fi
}

function pd_get_disk_size {
    local dev=${1}
    max=$(fdisk -l ${dev} 2>/dev/null | grep "Disk /dev"  | cut -d ',' -f2)
    max=${max%bytes}
    let PD_DISK_SIZE=max/1000000
}

function pd_board_specific_override()
{
    local boardtype=$1
    local ata_cnt=0
    local sd_count=0

#   Inventory media
    is_vxr_sim
    if [[ $? != 0 ]]; then
         if [[ -b /dev/vda ]]; then
             MAIN_DISK=('vda')
             SECONDARY_DISK=('vdb')
             return 1
         fi
    fi
   
    if [[ -b /dev/sda ]]; then
        sd_cnt=$(( sd_cnt + 1 ))
        is_ata /dev/sda
        if [[ $? == 1 ]]; then
            ata_cnt=$(( ata_cnt + 1 ))
        else
            # eUSB assigned sda, adjust MAIN and SECONDARY
            MAIN_DISK=('sdb')
            SECONDARY_DISK=('sdc')
            step_log_console Adjusted MAIN_DISK: ${MAIN_DISK[0]}
            step_log_console Adjusted SECONDARY_DISK: ${SECONDARY_DISK[0]}
        fi
    fi
    if [[ -b /dev/sdb ]]; then
        sd_cnt=$(( sd_cnt + 1 ))
        is_ata /dev/sdb
        if [[ $? == 1 ]]; then
            ata_cnt=$(( ata_cnt + 1 ))
        fi
    fi
    if [[ -b /dev/sdc ]]; then
        sd_cnt=$(( sd_cnt + 1 ))
        is_ata /dev/sdc
        if [[ $? == 1 ]]; then
            ata_cnt=$(( ata_cnt + 1 ))
        fi
    fi

    if [[ $sd_cnt == 1 ]]; then
        # LC case
        if [[ $ata_cnt != 1 ]]; then
            step_log_console "Unexpected LC device count sd=$sd_cnt ata=$ata_cnt"
            exitclean
        fi
    elif [[ $sd_cnt == 2 ]]; then
        # nousb case
        if [[ $ata_cnt != 2 ]]; then
            step_log_console "Unexpected RP device count sd=$sd_cnt ata=$ata_cnt"
            exitclean
        fi
    elif [[ $sd_cnt == 3 ]]; then
        # normal RSP/RP case
        if [[ $ata_cnt != 2 ]]; then
            step_log_console "Unexpected RP device count sd=$sd_cnt ata=$ata_cnt"
            exitclean
        fi
    else
        step_log_console "Unexpected media count sd=$sd_cnt ata=$ata_cnt"
        exitclean
    fi

#   Dial down some disk sizes when sda is a 16GB drive
    pd_dev=/dev/${MAIN_DISK[0]}
    if [ -b ${pd_dev} ]; then
        pd_get_disk_size ${pd_dev}
        if [ ${PD_DISK_SIZE} -le 20000 ]; then
# Host adjusted size = 2400
            DISK_HOST_BOOT_PART_SIZE=900            

            DISK_HOST_BOOT_SIZE=1000
            DISK_HOST_SCRATCH_SIZE=500

# Calvados adjusted size = 5000
            DISK_CALVADOS_BOOT_SIZE=1250
            DISK_CALVADOS_REPO_SIZE=1500
            DISK_CALVADOS_DATA_SIZE=1500
            DISK_CALVADOS_SCRATCH_SIZE=750

# XR adjusted size = 4000
            DISK_XR_BOOT_SIZE=1750
            DISK_XR_DATA_SIZE=1500
            DISK_XR_SCRATCH_SIZE=750
        fi
    fi
}
# populates the variable DEVVDB1 & VDB1P1 
# $1 should be DEVVDB1
# $2 should be VDB1P1
function xrnginstall_platform_set_vdb
{
    if [[ $# -ne 2 ]]; then
        echo "$FUNCNAME: Wrong argument list"
        return 1
    fi

    kmsg_log "xrnginstall_platform_set_vdb"

    eval $1="/dev/vdb1"
    eval $2="vdb1p1"
}

# Load the platform specific kernel modules here
function platform_load_kernel_module () {
    return
}

# Copy Iso file from Front-Panel/Internal USB.
function copy_iso_files_from_usb ()
{
    local usbpath=$1
    local systemiso=$2
    if [[ ! -d $usbpath ]]; then
        step_log_console "Unable to access $usbpath"
        exitclean
    fi

    PWD=`pwd`
    cd /

    step_log_file "Mounting ram disks"
    RD0=$(mktemp -d /tmp/isomnt.XXXXXX)
    RD1=$(mktemp -d /tmp/isomnt.XXXXXX)
    HD0=$(mktemp -d /tmp/harddisk.XXXXXX)
    mount -t tmpfs -o size=${TMPFS_SIZE} tmpfs ${RD0} || exitclean
    mount -t tmpfs -o size=${TMPFS_SIZE} tmpfs ${RD1} || exitclean

    local tar=`grep eusbboottar /proc/cmdline | wc -l`

    if [ "$tar" != "0" ]; then
        local systemtar=`cat /proc/cmdline | sed 's/^.*eusbboottar=//' | cut -d" " -f1`
        mount -t qnx4 /dev/sdc1 $HD0

        if [[ ! -f  $HD0/$systemtar ]]; then
            umount $HD0
            mount -t qnx4 /dev/sdd1 $HD0
        fi
        if [[ ! -f  $HD0/$systemtar ]]; then
            step_log_console "Cannot locate $systemtar file in USB device."
            exitclean
        fi

        systemiso=`tar -tf $HD0/$systemtar | grep '\.iso$'`
        step_log_console "Extracting $systemiso"
        step_log_file "Extracting $systemiso from  $usbpath/$systemtar"
        tar -xvf "$HD0/$systemtar" -C "${RD1}" "$systemiso" > /dev/null 2>&1
        if [ "$?" != "0" ] ; then
            step_log_console "Unable to extact from $usbpath/$systemtar"
            exitclean
        fi
        umount $HD0
    else
        step_log_console "Copying $usbpath/$systemiso"
        cp "$usbpath/$systemiso" "${RD1}/$systemiso"

        if [ "$?" != "0" ] ; then
            step_log_console "Unable to copy $usbpath/$systemiso"
            exitclean
        fi
    fi

    step_log_console "Extracting iso files from ${systemiso}"
    mount -r -o loop ${RD1}/$systemiso ${ISOMNT} || exitclean
    cd ${RD0}
    zcat ${ISOMNT}/boot/initrd.img | cpio -idmu >&107 2>&1
    copy_giso_rpms ${ISOMNT}
    umount ${ISOMNT}

    # Stage iso files
    cp iso_name.txt /
    mkdir -p /iso
    ln -sf ${RD0}/iso/asr9k-sysadmin.iso /iso
    ln -sf ${RD0}/iso/asr9k-xr.iso /iso
    ln -sf ${RD0}/iso/host.iso /iso
    ln -sf ${RD0}/rpm /rpm
    ln -sf ${RD1}/$systemiso /iso/system_image.iso

    ln -sf ${RD0}/boot/initrd.img /boot/
    if [ -f "${RD0}/boot/signature.initrd.img" ]; then
        cp -f ${RD0}/boot/signature.initrd.img /boot/
    fi


    cd $PWD
}

function pd_modprobe_pf_driver
{
    local ethif=$1
    local rc
    local try=0

    pd_modprobe_pf_driver_internal $ethif
    until [ $? -eq 0 ]; do
        if [[ $try == 3 ]]; then
            echo     "$(date): Failed pd modprobe"
            kmsg_log "Failed pd modprobe"
            return 1
        fi
        let "try = ${try} + 1"
        dmesg | grep -i "The EEPROM Checksum Is Not Valid"
        if [[ $? -eq 0 ]]; then
            echo     "$(date): Reprogram MAC devices"
            kmsg_log "Reprogram MAC devices"
            if [[ $try == 1 ]]; then
                source /etc/init.d/update_mac.sh
            else
                check_mac_programming
            fi
        else
            return 1
        fi
        sleep 1
        echo     "$(date): Retry starting PF devices"
        kmsg_log "Retry starting PF devices"
        pd_modprobe_pf_driver_internal $ethif
    done
    return 0
}

function eth_waitfor
{
    local try=0
    test -L /sys/class/net/$1
    until [ $? -eq 0 ]; do
        if [[ $try == 10 ]]; then
            echo "$(date): Network interface $1 not found after $try seconds"
            kmsg_log "Network interface $1 not found after $try seconds"
            return 1
        fi
        let "try = ${try} + 1"
        sleep 1
        test -L /sys/class/net/$1
    done
    return 0
}

function pd_modprobe_pf_driver_internal
{
    modprobe ixgbe
    eth_waitfor $1
    rc1=$?
    if [[ $rc1 != 0 ]]; then
        modprobe -r ixgbe
        return $rc1
    else
        return 0
    fi

    kmsg_log "pd_modprobe_pf_driver_internal(): rc $rc1"
}

function configure_pxe_lan () {
    # remove all udev defined interfaces
    local ixgbe=`lsmod | grep ixgbe | wc -l`
    if [ "$ixgbe" != "0" ]; then
        rmmod ixgbe
    fi
    rm -f /lib/udev/rules.d/40*

    ETHIF=$(get_card_specific_value PXE_ETHIF)
    pd_modprobe_pf_driver $ETHIF
    if [[ "$?" != "0" ]]; then
        kmsg_log "PXE LAN Configuration Failed"
        return 1
    fi

    # Configure VLAN with jumbo frames
    vconfig add ${ETHIF} 3073 > /dev/null 2>&1
    ifconfig ${ETHIF} up
    ifconfig ${ETHIF}.3073 up
    sleep 1
    step_log_console "Obtaining DHCP address"
    dhclient -1 ${ETHIF}.3073 || exitclean
}

function copy_scp ()
{
    return 1
}

function copy_iso_files_from_active_pxe ()
{
    step_log_console "Golden ISO install"
    return 1
}

function copy_giso_rpms ()
{
    readonly HOSTRPMS=/host_rpms/
    readonly CALRPMS=/calvados_rpms/
    readonly XRRPMS=/xr_rpms/
    readonly DEFAULT_ROUTER_CONFIG=/router.cfg
    readonly GISO_SUMMARY=/giso_summary.txt
    readonly GISO_INFO_TXT=/giso_info.txt

    GISOMNT=$1
    step_log_console "Copying Golden ISO files from ${GISOMNT}"
    if [ -d ${GISOMNT}${HOSTRPMS} ]; then
        mkdir ${HOSTRPMS}
        cp -fr ${GISOMNT}${HOSTRPMS}* ${HOSTRPMS}
        step_log_file "Copied Golden ISO host rpms from ${GISOMNT}${HOSTRPMS}"
    fi

    if [ -d ${GISOMNT}${CALRPMS} ]; then
        mkdir ${CALRPMS}
        cp -fr ${GISOMNT}${CALRPMS}* ${CALRPMS}
        step_log_file "Copied Golden ISO calvados rpms from ${GISOMNT}${CALRPMS}"
    fi

    if [ -d ${GISOMNT}${XRRPMS} ]; then
        mkdir ${XRRPMS}
        cp -fr ${GISOMNT}${XRRPMS}* ${XRRPMS}
        step_log_file "Copied Golden ISO xr rpms from ${GISOMNT}${XRRPMS}"
    fi

    if [ -f ${GISOMNT}${DEFAULT_ROUTER_CONFIG}  ]; then
        cp ${GISOMNT}${DEFAULT_ROUTER_CONFIG} ${DEFAULT_ROUTER_CONFIG}
        step_log_file "Copied xr config from ${GISOMNT}${DEFAULT_ROUTER_CONFIG}"
    fi

    if [ -f ${GISOMNT}${GISO_INFO_TXT} ]; then
        cp ${GISOMNT}${GISO_INFO_TXT} ${GISO_INFO_TXT}
        step_log_file "Copied giso_info.txt from ${GISOMNT}${GISO_INFO_TXT}"
    fi
       
    if [ -f ${GISOMNT}${GISO_SUMMARY} ]; then
        cp ${GISOMNT}${GISO_SUMMARY} ${GISO_SUMMARY}
        step_log_file "Copied giso_summary from ${GISOMNT}${GISO_SUMMARY}"
    fi
    step_log_console "Copied Golden ISO files from ${GISOMNT}"
}

function copy_iso_files_from_active_rp ()
{
    if [ -z "${tftpserver}" ]; then
        mkdir -p /var/lib/dhclient
        configure_pxe_lan
        if [[ "$?" != "0" ]]; then
            exitclean
        fi
        tftpserver=$(grep dhcp-server-identifier /var/lib/dhcp/dhclient.leases | head -1 | sed -e "s;^.*192;192;" -e "s;\;$;;")
    fi

    if [ -z "${tftpserver}" ]; then
        step_log_console "Cannot determine tftp server address"
        exitclean
    fi

    PWD=`pwd`
    cd /

    step_log_file "Mounting ram disks"
    RD0=$(mktemp -d /tmp/isomnt.XXXXXX)
    RD1=$(mktemp -d /tmp/isomnt.XXXXXX)
    mount -t tmpfs -o size=${TMPFS_SIZE} tmpfs ${RD0} || exitclean
    mount -t tmpfs -o size=${TMPFS_SIZE} tmpfs ${RD1} || exitclean

    local systemiso=`cat /proc/cmdline | sed 's/^.*installiso=//' | cut -d" " -f1`
    step_log_console "Copying $systemiso using scp from server ${tftpserver}"
    copy_scp "/misc/disk1/tftpboot/$systemiso" "${RD1}/$systemiso"
    if [ "$?" != "0" ] ; then
        step_log_console "Copying $systemiso using tftp from server ${tftpserver}"
        tftp -m binary $tftpserver -c get $systemiso ${RD1}/$systemiso
        if [ "$?" != "0" ] ; then
            step_log_console "Unable to copy $systemiso from server ${tftpserver}"
            exitclean
        fi
    fi

    step_log_console "Extracting iso files from ${systemiso}"
    mount -r -o loop ${RD1}/$systemiso ${ISOMNT} || exitclean
    cd ${RD0}
    zcat ${ISOMNT}/boot/initrd.img | cpio -idmu >&107 2>&1
    copy_giso_rpms ${ISOMNT} 
    umount ${ISOMNT}

    # Stage iso files
    cp iso_name.txt /
    mkdir -p /iso
    ln -sf ${RD0}/iso/asr9k-sysadmin.iso /iso
    ln -sf ${RD0}/iso/asr9k-xr.iso /iso
    ln -sf ${RD0}/iso/host.iso /iso
    ln -sf ${RD0}/rpm /rpm
    ln -sf ${RD1}/system_image.iso /iso
 
    ln -sf ${RD0}/boot/initrd.img /boot/
    if [ -f "${RD0}/boot/signature.initrd.img" ]; then
        cp -f ${RD0}/boot/signature.initrd.img /boot/
    fi

    cd $PWD
}

function compress_iso () {
    local rd0=$1
    local rd1=$2

    (cat ${rd1}/cpio.out | gzip > ${rd0}/boot/initrd.img)
    if [ "$?" != "0" ] ; then
        local time_str=$(date)
        echo "${time_str}: Background Compression Error" 
        exit 1
    fi
    local md5
    md5=$(md5sum ${rd0}/boot/initrd.img)
    if [ "$?" != "0" ] ; then
        local time_str=$(date)
        echo "${time_str}: Error Creating md5sum of Initrd.img"
    else
        echo "Initrd: initrd.img $(echo ${md5} | cut -d' ' -f 1)" >> ${rd0}/iso_info.txt
    fi 
    rm -f ${rd1}/cpio.out

    (cd ${rd0}; mkisofs -quiet -R -b boot/grub/stage2_eltorito -no-emul-boot -boot-load-size 4 -boot-info-table -o ${rd1}/system_image.iso . > /tmp/mkisofs.out 2>&1 )
    if [ "$?" != "0" ] ; then
        local time_str=$(date)
        echo "${time_str}: ISO Creation Error" 
        exit 1
    fi
    ln -s ${rd1}/system_image.iso /iso/system_image.iso || exitclean
}

function adjust_calvados_misc_disk1_label
{
    if [[ "${BOARDTYPE}" == "LC" ]]; then
        return;
    fi

    # If thirdparty reformatted disk due to space issue
    # and used the wrong misc disk1 label, adjust here
    if [[ ! -b /dev/mapper/pci_disk1-ssd_disk1_calvados_1 ]]; then
        return;
    fi
    local LABEL=$( e2label /dev/mapper/pci_disk1-ssd_disk1_calvados_1 )
    if [[ "$LABEL" != "cd_1_DISK_ONE" ]]; then
        e2label /dev/mapper/pci_disk1-ssd_disk1_calvados_1 cd_1_DISK_ONE
        if [[ "$?" != "0" ]]; then
            step_log_console "Unable to adjust Calvados /misc/disk1 label"
            echo "Exiting install"
            exit 1
        else
            step_log_console "Adjusted Calvados /misc/disk1 label"
        fi
    fi
}

function update_bootenv_virtmethod() {
    local emtcheck=$1

    platform_is_virtvm $emtcheck
    local rc=$?
    if [[ $rc == "1" ]]; then
        export VIRT_METHOD=vm
    else
        export VIRT_METHOD=lxc
    fi
    readonly VIRT_METHOD
    step_log_console "Installing with VIRT_METHOD $VIRT_METHOD"
    update_bootstrap /
}

function pd_copy_iso_files () {
    
    local eusbb_size=1500000
    rm -f /dev/ecu
    check_mac_programming
    a9k_show_media
    echo "$(date): Starting watchdog punch"
    bash /etc/init.d/pd_wd.sh &

    adjust_calvados_misc_disk1_label

    display_image_details 0
    step_log_console "${BOARDTYPE}: $PD_CARDTYPE, IPU: ${HW_CARDTYPE}"
    check_cardtype
    if [[ "$?" != "0" ]]; then
        step_log_console "Invalid Card Type ${PD_CARDTYPE} detected. Exiting install"
        exit 1
    fi

    local eusbboot=`grep eusbboot /proc/cmdline | wc -l`
    if [ "$eusbboot" != "0" ]; then
        # if "eusbboot=" is set, this is an eUSB boot
        # copy system_image.iso and iso files from eUSB
        eusb_mount_lxc "bake"
        local systemiso=`cat /proc/cmdline | sed 's/^.*eusbboot=//' | cut -d" " -f1`
        local eusbpath="/eusbb"
        copy_iso_files_from_usb $eusbpath $systemiso

        # update virtmethod, checking /proc/cmdline and emt area
        update_bootenv_virtmethod 1

        return
    fi

    local usbboot=`grep usbimage /proc/cmdline | wc -l`
    if [ "$usbboot" != "0" ]; then
        # if "eusbboot=" is set, this is an eUSB boot
        # copy system_image.iso and iso files from eUSB
        usb_mount_lxc "bake"
        local systemiso=`cat /proc/cmdline | sed 's/^.*usbimage=//' | cut -d" " -f1`
        local usbpath="/usb"
        copy_iso_files_from_usb $usbpath $systemiso
        return
    fi

    local eusbmount=`grep eusbmount /proc/cmdline | wc -l`
    if [ "$eusbmount" != "0" ]; then
        eusb_mount_lxc "bake"
    fi

    local val=$( fdisk -s /dev/eusb3 )
    if [[ $val -lt $eusbb_size ]]; then
        eusb_mount_lxc "bake"
    fi

    local zapdisk=`grep zapdisk /proc/cmdline | wc -l`
    if [ "$zapdisk" != "0" ]; then
        eusb_mount_lxc "zapdisk"
    fi

    local gisoboot=`grep giso_boot /proc/cmdline | wc -l`
    if [ "$gisoboot" != "0" ]; then
        # active RP
        copy_iso_files_from_active_pxe

        # update virtmethod, checking /proc/cmdline but not emt area
        update_bootenv_virtmethod 0

        return
    fi

    local systemiso=`grep installiso /proc/cmdline | wc -l`
    if [ "$systemiso" != "0" ]; then
        # Standby RP and LC path:
        # if "installiso=" is set, this is internal PXE
        # copy system_image.iso and iso files from active RP
        copy_iso_files_from_active_rp

        # update virtmethod, checking /proc/cmdline but not emt area
        update_bootenv_virtmethod 0

        return
    fi

    # Active RP path: reconstruct system_image.iso
    PWD=`pwd`

    step_log_file "Mounting ram disks"
    cd /
    RD0=$(mktemp -d /tmp/isomnt.XXXXXX)
    RD1=$(mktemp -d /tmp/isomnt.XXXXXX)
    mount -t tmpfs -o size=${TMPFS_SIZE} tmpfs ${RD0} || exitclean
    mount -t tmpfs -o size=${TMPFS_SIZE} tmpfs ${RD1} || exitclean

    step_log_console "Staging Root Filesystem"
    mkdir ${RD0}/boot
    if [ -f /iso/host.iso ]; then
        mkdir /host
        mount -r -o loop /iso/host.iso /host
        if [ -f /host/iso_info.txt ]; then
            grep -e PKG_FORMAT_VER -e Version /host/iso_info.txt | \
               sed -e "s;host          ;asr9k-mini-x64;" -e "s;Type:.*$;Type: 1;" > ${RD0}/iso_info.txt
        fi
        HOSTOS_RP_RPM_SRC=`ls -1 /host/rpm/*hostos*.rpm | head -1`
        if [ -n "$HOSTOS_RP_RPM_SRC" ]; then
            rpm --quiet -iv --nodeps --ignoresize --replacepkgs $HOSTOS_RP_RPM_SRC > /dev/null 2>&1
            if [[ "$?" != "0" ]]; then
                echo Unable to extract $HOSTOS_RP_RPM_SRC rpm
            fi
        else
            step_log_file "Error, could not find  /host/rpm/$HOST_ISO file"
        fi

        umount /host
        rm -rf /host
    fi
    cp /boot/bzImage ${RD0}/boot/
    cp -pr /boot/grub/ ${RD0}/boot/
    cp -pr /boot/grub2/ ${RD0}/boot/
    if [ -d "/boot/certs" ]; then
        cp -pr /boot/certs ${RD0}/boot/
    fi

    # update virtmethod, checking /proc/cmdline and emt area
    update_bootenv_virtmethod 1

    (find . -mount -print |  cpio --quiet -o -H newc -O ${RD1}/cpio.out) || exitclean

    step_log_console "Creating ISO in Background"
    compress_iso $RD0 $RD1 &
    CPID=$!

    cd ${PWD}
}

function pd_wait_for_iso () {
    if [[ "${BOARDTYPE}" == "LC" ]]; then
        return;
    fi

    if [ -L /iso/system_image.iso ]; then
        return
    fi
    step_log_console "Waiting for Compression completion $CPID"
    wait $CPID
    sleep 1
    if [ ! -L /iso/system_image.iso ]; then
        step_log_console "ISO Creation Error"
        exit 1
    fi
    step_log_console "ISO Creation Completed"
    return
}

#
# Called in Kdump path when the board needs to be reset at the end
#
function pd_board_reset {
    echo "Resetting board through CBC"
    . $PD_FUNCS
    set_ipu_base
    X86_CTRL_KEY=$(($IPU_BASE + 0x628))
    X86_CTRL=$(($IPU_BASE + 0x62c))
    pcimemwrite `printf 0x%x $X86_CTRL_KEY` 4 1
    pcimemwrite `printf 0x%x $X86_CTRL` 4 2
    sleep 1
}


#
# Usage : pd_get_card_inst_from_card_type card_type card_inst
function pd_get_card_inst_from_card_type () {
    local pd_card_inst

    if [ -z ${PD_CARDTYPE} ]; then
        get_board_type
    fi

    function smart_echo {
        echo "`date` $1" >> /misc/scratch/pd_card_inst_log
    }
        
    smart_echo "Starting get_card_inst"
    get_ci=$(grep -iw ${PD_CARDTYPE} /etc/init.d/asr9k_ci_list | head -1 | awk '{print $2 " " $3}')
    smart_echo $get_ci
    local pd_card_inst=$(echo $get_ci | cut -d "," -f1 | awk '{print $2}')
    local remaining=$(echo $get_ci | cut -d " " -f2)
    local pd_card_group=$(/usr/bin/python -c "a='$remaining'; print ','.join(a.split(',')[1:])")
    smart_echo "$pd_card_inst,$pd_card_group"

    eval "$2=$(echo $pd_card_inst)"
    eval "$3=$(echo $pd_card_group)"
}

function pd_platform_has_second_disk {

    case "$BOARDTYPE" in
        "RP")
            return 0 # exit status 0 = true
            ;;
        "LC")
            return 1 # exist status 1 = false
            ;;
        *)
            return 1 # exist status 1 = false
            ;;
    esac
}

## Kdump hooks start ##

function platform_copy_required_kernel_crash_modules() {
    local dest=$1

    # Copying all required modules
    cp /lib/modules/3*/kernel/drivers/md/dm-mod.ko $dest
    cp /lib/modules/3*/kernel/drivers/md/dm-mod.ko $dest
    cp /lib/modules/3*/kernel/crypto/crc32.ko  $dest
    cp /lib/modules/3*/kernel/crypto/crc32c.ko  $dest
    cp /lib/modules/3*/kernel/crypto/crct10dif_common.ko  $dest
    cp /lib/modules/3*/kernel/crypto/crct10dif_generic.ko  $dest
    cp /lib/modules/3*/kernel/fs/ext3/ext3.ko  $dest
    cp /lib/modules/3*/kernel/fs/ext4/ext4.ko  $dest
    cp /lib/modules/3*/kernel/fs/jbd/jbd.ko  $dest
    cp /lib/modules/3*/kernel/fs/jbd2/jbd2.ko  $dest
    cp /lib/modules/3*/kernel/drivers/ata/*  $dest
    cp /lib/modules/3*/kernel/drivers/ata/ata_piix.ko  $dest
    cp /lib/modules/3*/kernel/drivers/ata/ahci.ko  $dest
    cp /lib/modules/3*/kernel/drivers/ata/libahci.ko  $dest
    cp /lib/modules/3*/kernel/lib/asn1_decoder.ko  $dest
    cp /lib/modules/3*/kernel/lib/crc-t10dif.ko  $dest
    cp /lib/modules/3*/kernel/lib/libcrc32c.ko  $dest
    cp /lib/modules/3*/kernel/drivers/scsi/sd_mod.ko $dest
    cp /lib/modules/3*/updates/drivers/net/ethernet/intel/ixgbe/ixgbe.ko $dest
    cp /lib/modules/3*/kernel/net/8021q/8021q.ko $dest
    cp /lib/modules/3*/kernel/net/802/* $dest
    cp /lib/modules/3*/kernel/net/llc/* $dest
    cp /lib/modules/3*/kernel/drivers/uio/* $dest
}

function platform_load_required_kernel_crash_modules() {

    insmod /lib*/modules/dm-mod.ko >/dev/null 2>/dev/null
    insmod /lib*/modules/jbd.ko >/dev/null 2>/dev/null
    insmod /lib*/modules/jbd2.ko >/dev/null 2>/dev/null
    insmod /lib*/modules/ext3.ko >/dev/null 2>/dev/null
    insmod /lib*/modules/ext4.ko >/dev/null 2>/dev/null
    insmod /lib*/modules/crct10dif_common.ko >/dev/null 2>/dev/null
    insmod /lib*/modules/crct10dif_generic.ko >/dev/null 2>/dev/null
    insmod /lib*/modules/crc-t10dif.ko >/dev/null 2>/dev/null
    insmod /lib*/modules/sd_mod.ko >/dev/null 2>/dev/null
    insmod /lib*/modules/libahci.ko >/dev/null 2>/dev/null
    insmod /lib*/modules/ahci.ko >/dev/null 2>/dev/null
    insmod /lib*/modules/ixgbe.ko >/dev/null 2>/dev/null
    insmod /lib*/modules/llc.ko  >/dev/null 2>/dev/null
    insmod /lib*/modules/stp.ko  >/dev/null 2>/dev/null
    insmod /lib*/modules/garp.ko  >/dev/null 2>/dev/null
    insmod /lib*/modules/mrp.ko  >/dev/null 2>/dev/null
    insmod /lib*/modules/8021q.ko  >/dev/null 2>/dev/null
    insmod /lib*/modules/uio_pci_proxy.ko >/dev/null 2>/dev/null

    echo "10b5 87a0" > /sys/bus/pci/drivers/pci_proxy/new_id
}

function punch_wd
{
    . $PD_FUNCS
    set_ipu_base
    IPU_WD_ADDR=$(($IPU_BASE + 0x630))

    local punch=0
    # Punch for max 6 mins, this is a fail-safe path that we can reload if
    # dumping really fails. And 6 mins is twice as long as we require
    for i in {1..360}; do
        pcimemwrite `printf 0x%x $IPU_WD_ADDR` 4 $punch >> /dev/null
        punch=$(($punch ^ 1))
        sleep 1
    done
}

function punch_mastership
{
    . $PD_FUNCS
    set_ipu_base
    get_board_type
    if [ "${BOARDTYPE}" == "RP" ]; then
        IPU_MAST_OFF=$(($IPU_BASE + $(get_ipu_block_offset RP_FAB 1) + 0x3c))
        pcimemwrite `printf "0x%x" $IPU_MAST_OFF` 4 0x1
        echo "Punching mastership"
    fi
}

KDUMP_CODE=0x1
KDUMP_SEND_COUNT=30

function punch_kdump_notification
{
    local slot=$1

    echo "Starting punch kdump for slot $slot"
    /sbin/sock_send 192.255.255.255 $slot $KDUMP_CODE $KDUMP_SEND_COUNT
}

function pd_notify_host_kdump_started
{
    echo "KDUMP started"
    punch_wd &
}

function pd_notify_host_kdump_ended
{
    echo "KDUMP ended"
}

function get_pd_slot
{
    . /etc/rc.d/init.d/ipu-functions.sh
    PSLOT=$(pcimemread `printf "0x%x" $PSLOT_PADDR` 4 | grep -v PCI | awk '{print $3}' | sed -e 's/^000000//')
    echo 0x$PSLOT
}

function platform_configuration
{
    . $PD_FUNCS
    get_board_type
    local slot=$(get_pd_slot)
    local intf=eth5
    if [ "${BOARDTYPE}" == "LC" ]; then
        intf=eth1
    fi

    echo "Starting for $slot"
    /sbin/ifconfig $intf up
    if [ "$?" != "0" ]; then
        echo "$intf not available trying again with reset"
        rmmod ixgbe
        insmod /lib*/modules/ixgbe.ko
        /sbin/ifconfig $intf up
    fi
    /sbin/vconfig add $intf 3073
    /sbin/ifconfig $intf.3073 192.0.$(($slot << 2)).1/8 up
    echo "Done with configurations ${BOARDTYPE}"

    punch_kdump_notification $slot
    punch_mastership
}

function platform_copy_bins_libs
{
    local crash_root=$1
   
    cp /sbin/vconfig $crash_root/sbin/
    cp /bin/dmesg $crash_root/sbin/
    cp /sbin/modprobe $crash_root/sbin/
    cp /bin/cp $crash_root/sbin/
    cp /usr/bin/tee $crash_root/sbin/
    cp /bin/lsmod $crash_root/sbin/
    cp /sbin/ifconfig $crash_root/sbin/
    cp /sbin/route $crash_root/sbin/
    cp /sbin/rmmod $crash_root/sbin/
    cp /sbin/sock_send $crash_root/sbin/
    cp /sbin/showipu $crash_root/sbin/
    cp /sbin/showemt $crash_root/sbin/
    cp /etc/rc.d/init.d/ipu-functions.sh $crash_root/etc/rc.d/init.d/
}

function platform_binaries_set
{
    local set=('vconfig' 'dmesg' 'modprobe' 'lsmod' 'cp' 'tee' 'sock_send' 'showipu' 'showemt')
    echo ${set[@]}
}

function pd_set_kernel_panic_triggers
{
    configure_panics
}

function pd_update_grub_files
{
    local input_dir=$1

    # In case of standby RP or usb, update grub file.
    # On active RP, there will already be secured grub file (and during
    # bake, /boot/grub2-secure/bootx64.efi" will not be available).
    if [[ -d /boot/grub2-secure/ ]]; then
        cp /boot/grub2-secure/bootx64.efi ${input_dir}/grub.efi
        cp /boot/grub2-secure/grub.cfg ${input_dir}/.
        step_log_console "Updating to secure grub EFI"
    fi

}


function pd_modprobe_igb
{
    modprobe igba9k 
}

# Find boot mode(ipxe/usb)
RP_EMT_MODE_PXEBOOT=0x01
RP_EMT_MODE_USBBOOT=0x04
RP_EMT_MODE_EUSBBOOT=0x05
RP_EMT_MODE_EXTERNAL_PXE_DHCP=0x07
RP_EMT_MODE_EXTERNAL_PXE_RMON=0x08

function pd_get_boot_mode () 
{
    get_ipu_nvram_off

    local boot_dev=""
    if [ "${BOARDTYPE}" == "RP" ]; then

        if [ $EMT_MODE == $RP_EMT_MODE_PXEBOOT ]; then
            local NVRAM_BASE_PADDR=$(($IPU_BASE+$NVRAM_OFF+$INFO_ROM_NV_OFF))
            local NVRAM_DATA=$( pcimemread `printf "%x" $NVRAM_BASE_PADDR` 0x300| grep -v PCI | awk '{for (i=7; i<=NF-1; i++)printf("%s", $i); printf("%s",$NF);}')
            local DHCP_BOOT=$(echo $NVRAM_DATA| grep -c "IPXE_PREF.0")
            if [ $DHCP_BOOT == 1 ]; then
                boot_dev="iPXE"
            else
                boot_dev="TFTP"
            fi
        elif [ $EMT_MODE == $RP_EMT_MODE_EXTERNAL_PXE_DHCP ]; then
            boot_dev="iPXE"
        elif [ $EMT_MODE == $RP_EMT_MODE_EXTERNAL_PXE_RMON ]; then
            boot_dev="TFTP"
        elif [ $EMT_MODE == $RP_EMT_MODE_USBBOOT ]; then
            boot_dev="USB"
        elif [ $EMT_MODE == $RP_EMT_MODE_EUSBBOOT ]; then
            boot_dev="EUSB"
        fi
        echo $boot_dev
    fi
}

function pd_tftp_download_iso () 
{
    get_ipu_nvram_off
    
    local image_name=$1
    step_log_console "Downloading GISO image via tftp."

    local IPV4_ADDRESS_OFF=0xA8C
    local GATEWAY_ADDRESS_OFF=0xA90
    local SUBNET_MASK_OFF=0xA94
    local SERVER_URL_OFF=0xAA8

    local IPV4_ADDRESS_PADDR=$(($IPU_BASE + $EMT_OFF + $NVRAM_OFF + $IPV4_ADDRESS_OFF))
    local GATEWAY_ADDRESS_PADDR=$(($IPU_BASE + $EMT_OFF + $NVRAM_OFF + $GATEWAY_ADDRESS_OFF))
    local SUBNET_MASK_PADDR=$(($IPU_BASE + $EMT_OFF + $NVRAM_OFF + $SUBNET_MASK_OFF))
    local SERVER_URL_PADDR=$(($IPU_BASE + $EMT_OFF + $NVRAM_OFF + $SERVER_URL_OFF))


    local IPV4_ADDRESS=$( pcimemread `printf "%x" $IPV4_ADDRESS_PADDR` 0x10 | grep -v PCI | awk '{ print $3 ; }' | sed -e 's/^000000/0x/' )
    local GATEWAY_ADDRESS=$( pcimemread `printf "%x" $GATEWAY_ADDRESS_PADDR` 0x10 | grep -v PCI | awk '{ print $3 ; }' | sed -e 's/^000000/0x/' )
    local SUBNET_MASK=$( pcimemread `printf "%x" $SUBNET_MASK_PADDR` 0x10 | grep -v PCI | awk '{ print $3 ; }' | sed -e 's/^000000/0x/' )
    local SERVER_URL_DATA=$( pcimemread `printf "%x" $SERVER_URL_PADDR` 0x100| grep -v PCI | awk '{for (i=7; i<=NF-1; i++)printf("%s", $i); printf("%s",$NF);}')
    local SERVER_URL=$(echo $SERVER_URL_DATA |sed   -r 's/\.{2,}//')

    local SERVER_ADDR=$( echo $SERVER_URL | awk -F '/' '{print $3}')
    local FILE_NAME=$( echo $SERVER_URL | awk -F '/' '{for (i=4; i<=NF-1; i++)printf("/%s", $i); printf("/%s",$NF);}')
    
    modprobe igba9k

    local eths=( $(ifconfig -a |  grep "Link\ " | grep eth | cut -d' ' -f1 ) )
    for eth in "${eths[@]}";
    do
        ifconfig $eth 0x$IPV4_ADDRESS netmask 0x$SUBNET_MASK up
        route add default gw 0x$GATEWAY_ADDRESS $eth
        ping -c 2 $SERVER_ADDR >> /tmp/varmount 2>&1
        if (( $? != 0 )); then
            ifconfig $eth 0x0 down
            continue
        fi
        tftp -v -m binary $SERVER_ADDR -c get $FILE_NAME $image_name  >> /tmp/varmount 2>&1
        if (( $? != 0 )); then
            step_log_console "No response from Image download" 
            ifconfig $eth 0x0 down
        else 
            step_log_console "Image downloaded " 
            break;
        fi
    done

}
function pd_get_giso_disk () 
{
    #   Mount efivarfs. Run efibootmgr to findout if current boot is iPXE/USB
    mount -t efivarfs efivarfs /sys/firmware/efi/efivars > /dev/null 2>&1 
    local boot_id="$(efibootmgr | grep "BootCurrent:" | cut -d" " -f2 )"
        
    if [ -z "${boot_id}" ]; then
        step_log_console "Couldn't find giso_disk for USB."
            return 
    fi

    local raw_boot_dev="$( efibootmgr | grep Boot${boot_id} | cut -d':' -f2 | cut -d' ' -f2- )"

    #Disk can be USB or local disk
    local dev_pattern="$( echo $raw_boot_dev | cut -d'_' -f1 | cut -d' ' -f1 | cut -d'/' -f1 | cut -d'\' -f1)"
    local giso_disk="$(ls /dev/disk/by-id/ -l | grep ${dev_pattern} | cut -f2 -d">" | grep 1 | cut -d" " -f2)"
    echo $giso_disk

}

## Kdump hooks end ##

# Note:
# Up to this point, the scripts in this file are bash functions
# These functions are invoked from the PI layers
# The following functions are invoked on every instantiation of spirit-pd.sh
# These functions cache their results, so repeated calls have little overhead
on_sim
get_board_type
