#!/bin/bash
# update_mac_v2.sh script.
# This script is used for automatically programming the MAC
# addresses in the NIC EEPROMs.
#
# Copyright (c) 2015-2017 by Cisco Systems, Inc.
# All rights reserved.
#

source /etc/rc.d/init.d/pd-functions

function ixgbe_eeprom_wrenable
{
    if [[ "${BOARDTYPE}" != "RP" ]]; then
        return;
    fi

    set_ipu_base
    local rc=$?
    if [[ $rc != 0 ]]; then
        echo "$(date): Cannot find IPU base - skipping 10G EEPROM wrenable, rc=$rc"
        return
    fi

    # write enable each Niantic instance via optimus gpio register
    GPIO_OFFSET=$(get_ipu_block_offset GPIO 3)
    GPIO_BASE_DEC=$(( $IPU_BASE + $GPIO_OFFSET ))
    GPIO_BASE_HEX=0x$(printf "%x" $GPIO_BASE_DEC )
    NI2_EEPROM_WP_L_O=0x1660
    NI1_EEPROM_WP_L_O=0x1680
    NI0_EEPROM_WP_L_O=0x16a0
    for WP_OFF_HEX in $NI0_EEPROM_WP_L_O $NI1_EEPROM_WP_L_O $NI2_EEPROM_WP_L_O
    do
        WP_BASE_DEC=$(( ${GPIO_BASE_HEX} + ${WP_OFF_HEX} ))
        WP_BASE_HEX=$( printf "%x" $WP_BASE_DEC )
        pcimemwrite $WP_BASE_HEX 4 0x01000010 > /dev/null 2>&1
    done

    # Turn off BP0/BP1 in each Niantic port instance
    for i in 0 1 2 3 4 5; do
        /sbin/ixgbe_eeprom -i $i
        rc=$?
        if [[ $rc != 0 ]]; then
            echo "$(date): ixgbe_eeprom $i exit status $rc"
        fi
    done
}

UPDATE_NEEDED=
REWRITE_FORCE=
UPDATE_FILES=1

function update_eeupdate_files ()
{
    local MAC_HOSTOS_ALL="/opt/cisco/calvados/"

    if [[ $UPDATE_FILES == 0 ]]; then
        return
    fi

    cp ${MAC_HOSTOS_ALL}/bin/eeupdate64e /sbin
    cp ${MAC_HOSTOS_ALL}/bin/showipu /sbin
    cp ${MAC_HOSTOS_ALL}/bin/ixgbe_eeprom /sbin


    cp ${MAC_HOSTOS_ALL}/usr/lib64/libasr9kpd.so /usr/lib64

    local lib_files="lib4arg.so libosc.so libsafec.so libctrace.so libcalvados_infra.so"
    for i in ${lib_files}
    do
      (cd /usr/lib64; ln -s libasr9kpd.so $i)
    done
}

function remove_eeupdate_files ()
{
    if [[ $UPDATE_FILES == 0 ]]; then
        return
    fi

    rm -f /sbin/eeupdate64e
    rm -f /sbin/showipu
    rm -f /sbin/ixgbe_eeprom

    rm -f /usr/lib64/libasr9kpd.so

    local lib_files="lib4arg.so libosc.so libsafec.so libctrace.so libcalvados_infra.so"
    for i in ${lib_files}
    do
      rm -f /usr/lib64/$i
    done
}


function mac_update_needed ()
{
    get_board_type_ipu

    MAC_BASE_LO_OFF=0x570
    MAC_BASE_HI_OFF=0x574
    MAC_UPD_OFF=0x57C

    MAC_UPD_PADDR=$(($IPU_BASE + $NVRAM_OFF + $EMT_OFF + $MAC_UPD_OFF))
    UPD=$( pcimemread `printf "%x" $MAC_UPD_PADDR` 0x4 | grep -v PCI | awk '{ print $3 ; }' )
    upd=$(expr $UPD + 0)

    if [[ $upd = 0 ]]; then
        local FORCE
        if [[ ! -z ${REWRITE_FORCE} ]]; then
            FORCE="1"
        else
            FORCE=$( grep rewritemac /proc/cmdline | wc -l ) 
        fi 
        if [[ ${FORCE} == "0" ]]; then
            echo "$(date): MAC addresses are already programmed"
            return
        else
            REWRITE_FORCE=1
            echo "$(date): MAC addresses will be re-programmed" 
        fi
    fi
    MAC_BASE_LO_PADDR=$(($IPU_BASE + $NVRAM_OFF + $EMT_OFF + $MAC_BASE_LO_OFF))
    MAC_BASE_HI_PADDR=$(($IPU_BASE + $NVRAM_OFF + $EMT_OFF + $MAC_BASE_HI_OFF))

    LO4=$( pcimemread `printf "%x" $MAC_BASE_LO_PADDR` 0x4 | grep -v PCI | awk '{ print $3 ; }' )
    HI2=$( pcimemread `printf "%x" $MAC_BASE_HI_PADDR` 0x4 | grep -v PCI | awk '{ print $3 ; }' )

    # Don't reuse invalid MAC address cookie data
    if [[ $LO4 == "00000000" && $HI2 == "00000000" ]]; then
        echo -n "$(date): WARN: MAC address cookie is zeroed out. "
        echo "Not programming NICs."
    else
        UPDATE_NEEDED=1
    fi

    return
}

mac_address_array=()

# Read the file in parameter and fill the array named "mac_address_array"
getArray() {
    i=0
    while read line # Read a line
    do
        mac_address_array[i]=$line # Put it into the mac_address_array
        i=$(($i + 1))
    done < $1
}

function gather_mac_addresses ()
{
    # Gather eeupdate64e /MAC_DUMP /ALL output to mac_checker.txt
    chmod 777 /sbin/eeupdate64e
    /sbin/eeupdate64e /MAC_DUMP /ALL | sed -n '/LAN/ p' | sed 's/^.*is //g' | sed 's/\.//' > /tmp/mac_checker.txt
    local rc=${PIPESTATUS[0]}
    if [[ $rc != 0 ]]; then
        echo "$(date): eeupdate MAC_DUMP failed, rc=$rc."
    fi
    sleep 1
    getArray "/tmp/mac_checker.txt"
}

function program_mac_addresses()
{
    local FIRST_NIC=1
    # Sorting NICs by vendorid:deviceid (-k5)
    # Based on assumption that 1G NIC has lower sort result vs. 10G NIC.
    NIC_INSTANCES=$(   eeupdate64e -all 2> /dev/null| grep 8086 | sort -k 5 | awk '{ print $1-1 }' )
    local rc=${PIPESTATUS[0]}
    if [[ $rc != 0 ]]; then
        echo "$(date): NIC instances get failed in eeupdate -all, rc=$rc."
    fi

    gather_mac_addresses

    local l=$LO4
    local h=$HI2

    local H=${l:6:2}${l:4:2}${l:2:2}
    local L=0x${l:0:2}${h:6:2}${h:4:2}

    let j=0
    for i in $NIC_INSTANCES 
        do
        NIC_PROGRAMMED=$( printf "%s" ${mac_address_array[$i]} )
        LI=$((${L} + ${j}))
        LO=$( printf "%06x" $LI )
        MAC=${H}${LO}
        MAC_TO_PROGRAM=$( printf "%s" $MAC )
        NIC_PROGRAMMED="$(tr [A-Z] [a-z] <<< "$NIC_PROGRAMMED")"
        IND=$((${i} + 1))
        if [[ $FIRST_NIC == "1" ]]; then
            FIRST_NIC=0
            ixgbe_eeprom_wrenable
        fi
        echo "$(date): /sbin/eeupdate64e /NIC=${IND} /MAC=$MAC"
        /sbin/eeupdate64e /NIC=${IND} /MAC=$MAC >> /tmp/mac_program.txt  2>&1 
        rc=$?
        if [[ $rc != 0 ]]; then
            echo "$(date): NIC=${IND} not programmed correctly, rc=$rc."
        fi
        sleep 1
        j=${j}+1
    done
}

function check_mac_programming ()
{
    mac_update_needed
    if [[ -z ${UPDATE_NEEDED} ]]; then
        return
    fi

    update_eeupdate_files
    program_mac_addresses
    remove_eeupdate_files
}
