#!/bin/bash
#
# Script to collect process core and logs in a lxc
#
# Dec 2015, Rajesh Ranga
#
# Copyright (c) 2015-2016 by Cisco Systems, Inc.
# All rights reserved.
#

VM_NAME=$1

if [ -z ${VM_NAME} ]; then
	echo "Usage: lxc_dump.sh sysadmin/default-sdr--n"
    exit
fi

source /etc/rc.d/init.d/spirit_pd.sh
if [ ${VM_NAME} == "sysadmin" ]; then
    pd_punch_watchdog
fi

if [ "$(type -t pd_get_my_int_nodeid_str)" = "function" ]; then
    SLOT=`pd_get_my_int_nodeid_str`
else
    SLOT=RP
fi

lxc_core_dir="/misc/scratch/${VM_NAME}.`date +"%Y%m%d-%H%M%S"`.core.${SLOT}.lxcdump"
pcore_dir=${lxc_core_dir}/core_files
mkdir -p ${pcore_dir}

#Execute command in name space VM_NAME
function run_namespace_cmd
{
    local time=`date +"%Y-%m-%d-%H:%M:%S:%N"`
    echo $time: "$*" >> ${lxc_core_dir}/lxc_dump.log
    virsh lxc-enter-namespace ${VM_NAME} --noseclabel "$@" 2>>${lxc_core_dir}/lxc_dump.log
}

# Freeze lxc
prev_freezer_state=`virsh domstate ${VM_NAME}`
if [ "${prev_freezer_state}" != "paused" ]; then
    if ! virsh suspend ${VM_NAME}; then
        # Failed to freeze
        exit 1
    fi
fi

# Stop all the process in the lxc (all process in lxc have same pid ns as init)
INITPID=$(echo $(ps h --ppid $(virsh domid ${VM_NAME}) -o pid))
PIDS=$( ls -l /proc/*/ns/pid | grep -F $(readlink /proc/$INITPID/ns/pid) | cut -f3 -d/ )
kill -STOP ${PIDS[@]}

# Thaw with everything stopped to allow core collection
virsh resume ${VM_NAME}

# Collect the process list in the lxc namespace
run_namespace_cmd /bin/ps h -e -o pid,comm > ${lxc_core_dir}/pslist.txt

# delete entry for ps that collects the process list and empty lines
sed -i '/ ps$/d' ${lxc_core_dir}/pslist.txt
sed -i '/^$/d' ${lxc_core_dir}/pslist.txt

#Collect ctrace logs and decode files
ctrace_buffer_dir=/var/log/ctrace
if [ ${VM_NAME} == "sysadmin" ]; then
    ctrace_decode_dir=/opt/cisco/calvados/etc
else
    ctrace_decode_dir=/pkg/ctrace
fi
run_namespace_cmd /bin/tar -cPf - ${ctrace_buffer_dir} > ${lxc_core_dir}/ctrace_buffer.tar
run_namespace_cmd /bin/tar -hcPf - ${ctrace_decode_dir} > ${lxc_core_dir}/ctrace_decode.tar

#Collect syslog
run_namespace_cmd /bin/cat /var/log/syslog > ${lxc_core_dir}/syslog
if [ ${VM_NAME} != "sysadmin" ]; then
    run_namespace_cmd /bin/cat /dev/shmem/sysmgr.log > ${lxc_core_dir}/sysmgr.log
    run_namespace_cmd /bin/cat /dev/shm/tmp/syslog.local.log > ${lxc_core_dir}/syslog.local.log
fi

time=$(date +"%Y%m%d-%H%M%S")
if [ ${VM_NAME} == "sysadmin" ]; then
    node=`virsh lxc-enter-namespace ${VM_NAME} --noseclabel /bin/hostname`
else
    node=`virsh lxc-enter-namespace ${VM_NAME} --noseclabel /pkg/bin/hostname`
fi

## Gather all cores
while read line; do 
    pid=`echo $line | awk '{print $1}'`
    comm=`echo $line | awk '{print $2}'`
    core_file=${comm}_${pid}.by.lxc_dump.${time}.${node}.core.lz4
    run_namespace_cmd /usr/bin/ccore -o /dev/fd/1 ${pid} | lz4 -c > ${pcore_dir}/${core_file}
done < ${lxc_core_dir}/pslist.txt 

virsh suspend ${VM_NAME}

#Allow process to continue
kill -CONT ${PIDS[@]}

if [ ${prev_freezer_state} != paused ]; then
    virsh resume ${VM_NAME}
fi

function nfs_mount_core_dir
{
    slot_num=$(my_slot_id)
    act_rp_slot=$(pd_get_active_rp_slot)
    echo "active rp is on slot: ${act_rp_slot}" >> $LOG
    rp_nfs_dir="/misc/scratch/core"
    mount -t nfs 127.1.1.${act_rp_slot}:${rp_nfs_dir} ${COREDIR} -o nolock 2>/dev/null
    if [ $? != "0" ] ; then
        echo "nfs mount of rp failed on eth0" >> $LOG
        return 1
    fi
    return 0
}

ARCH=`arch`
# For membooted cards, tar and copy to active RP
if  [[ "$ARCH" =~ "arm" ]]; then
    if [ $HB_MISS_COUNT -eq $HB_MISS_COUNT_MAX ] ; then
        COREDIR="/RP"
        mkdir -p ${COREDIR}
        nfs_mount_core_dir

        if [ "$?" == "0" ]; then
            tar --remove-files -cPf - ${lxc_core_dir} | lz4 -c > ${lxc_core_dir}.tar.lz4
            mv ${lxc_core_dir}.tar.lz4 ${COREDIR}
            umount -l ${COREDIR}
        fi
    fi
else
    tar --remove-files -cPf - ${lxc_core_dir} | lz4 -c > ${lxc_core_dir}.tar.lz4
    mv ${lxc_core_dir}.tar.lz4 /misc/scratch/core
fi
